/*
 * libdpkg - Debian packaging suite library routines
 * pkg-spec.c - routines to handle package specifiers
 *
 * Copyright © 2011 Linaro Limited
 * Copyright © 2011 Raphaël Hertzog <hertzog@debian.org>
 * Copyright © 2011 Guillem Jover <guillem@debian.org>
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>
#include <compat.h>

#include <stdlib.h>
#include <fnmatch.h>
#include <string.h>

#include <dpkg/i18n.h>
#include <dpkg/dpkg.h>
#include <dpkg/dpkg-db.h>
#include <dpkg/arch.h>
#include <dpkg/pkg-spec.h>

static void
pkg_spec_blank(struct pkg_spec *ps)
{
	ps->name = NULL;
	ps->arch = NULL;

	ps->name_is_pattern = false;
	ps->arch_is_pattern = false;
}

static void
pkg_spec_iter_blank(struct pkg_spec *ps)
{
	ps->pkg_iter = NULL;
	ps->pkg_next = NULL;
}

void
pkg_spec_init(struct pkg_spec *ps, enum pkg_spec_flags flags)
{
	ps->flags = flags;

	pkg_spec_blank(ps);
	pkg_spec_iter_blank(ps);
}

const char *
pkg_spec_is_illegal(struct pkg_spec *ps)
{
	static char msg[1024];
	const char *emsg;

	if (!ps->name_is_pattern &&
	    (emsg = pkg_name_is_illegal(ps->name))) {
		snprintf(msg, 1024, _("package name in specifier '%s%s%s' is "
		                      "illegal: %s"), ps->name,
		         (ps->arch->type != arch_none) ? ":" : "",
		         ps->arch->name, emsg);
		return msg;
	}

	if (!ps->arch_is_pattern && ps->arch->type == arch_illegal) {
		emsg = dpkg_arch_name_is_illegal(ps->arch->name);
		snprintf(msg, 1024, _("architecture name in specifier '%s%s%s' "
		                      "is illegal: %s"), ps->name,
		         (ps->arch->type != arch_none) ? ":" : "",
		         ps->arch->name, emsg);
		return msg;
	}

	return NULL;
}

bool
pkg_spec_is_pattern(struct pkg_spec *ps)
{
	if (ps->name_is_pattern || ps->arch_is_pattern)
		return true;
	if ((ps->flags & psf_def_wildcard) && ps->arch->type == arch_none)
		return true;
	return false;
}

static const char *
pkg_spec_do_checks(struct pkg_spec *ps)
{
	ps->name_is_pattern = false;
	ps->arch_is_pattern = false;

	/* Detect if we have patterns and/or illegal names. */
	if ((ps->flags & psf_patterns) && strpbrk(ps->name, "*[?\\"))
		ps->name_is_pattern = true;

	if ((ps->flags & psf_patterns) && strpbrk(ps->arch->name, "*[?\\"))
		ps->arch_is_pattern = true;

	return pkg_spec_is_illegal(ps);
}

const char *
pkg_spec_set(struct pkg_spec *ps, const char *pkgname, const char *archname)
{
	ps->name = m_strdup(pkgname);
	ps->arch = dpkg_arch_find(archname);

	return pkg_spec_do_checks(ps);
}

const char *
pkg_spec_parse(struct pkg_spec *ps, const char *str)
{
	char *archname;

	archname = strchr(str, ':');
	if (archname == NULL)
		ps->name = m_strdup(str);
	else {
		ps->name = m_strndup(str, archname - str);
		archname++;
	}
	ps->arch = dpkg_arch_find(archname);

	return pkg_spec_do_checks(ps);
}

static bool
pkg_spec_match_arch(struct pkg_spec *ps, const struct dpkg_arch *arch)
{
	if (ps->arch_is_pattern)
		return (fnmatch(ps->arch->name, arch->name, 0) == 0);
	else if (ps->arch->type != arch_none) /* !arch_is_pattern */
		return (ps->arch == arch);

	/* No arch specified. */
	switch (ps->flags & psf_def_mask) {
	case psf_def_native:
		return (arch->type == arch_native || arch->type == arch_all ||
		        arch->type == arch_none);
	case psf_def_wildcard:
		return true;
	default:
		internerr("multiple/unknown/no psf_def_* configured in pkg_spec");
	}
}

static bool
pkg_spec_match_pkgname(struct pkg_spec *ps, const char *name)
{
	if (ps->name_is_pattern)
		return (fnmatch(ps->name, name, 0) == 0);
	else
		return (strcmp(ps->name, name) == 0);
}

static bool
pkg_spec_match_flags(struct pkg_spec *ps, struct pkginfo *pkg)
{
	if ((ps->flags & psf_skip_not_installed) &&
	    pkg->status == stat_notinstalled)
		return false;
	if ((ps->flags & psf_skip_config_files) && pkg &&
	    pkg->status == stat_configfiles)
		return false;
	return true;
}

bool
pkg_spec_match_pkg(struct pkg_spec *ps, struct pkginfo *pkg,
                   struct pkgbin *pkgbin)
{
	return (pkg_spec_match_flags(ps, pkg) &&
	        pkg_spec_match_arch(ps, pkgbin->arch) &&
	        pkg_spec_match_pkgname(ps, pkg->set->name));
}

struct pkginfo *
pkg_spec_find_pkg(struct pkg_spec *ps)
{
	struct pkginfo *pkg;

	if (ps->flags & (psf_patterns | psf_def_wildcard))
		internerr("pkg_spec_find_pkg() incompatible with patterns");

	pkg = pkg_db_find_pkg(ps->name, ps->arch);
	if (pkg_spec_match_flags(ps, pkg))
		return pkg;

	return NULL;
}

struct pkginfo *
pkg_spec_parse_pkg(enum pkg_spec_flags flags, const char *str)
{
	struct pkg_spec ps;
	struct pkginfo *pkg;
	const char *emsg;

	pkg_spec_init(&ps, flags);

	emsg = pkg_spec_parse(&ps, str);
	if (emsg)
		ohshit("%s", emsg);

	pkg = pkg_spec_find_pkg(&ps);

	pkg_spec_destroy(&ps);

	return pkg;
}

void
pkg_spec_iter_init(struct pkg_spec *ps)
{
	if (ps->name_is_pattern)
		ps->pkg_iter = pkg_db_iter_new();
	else
		ps->pkg_next = &pkg_db_find_set(ps->name)->pkg;
}

struct pkginfo *
pkg_spec_iter_next_pkg(struct pkg_spec *ps)
{
	struct pkginfo *pkg;
	struct pkgset *set;

loop_arch:
	/* Loop over all architectures and return the matching ones. */
	while (ps->pkg_next) {
		pkg = ps->pkg_next;
		ps->pkg_next = pkg->arch_next;
		if (pkg_spec_match_flags(ps, pkg) &&
		    pkg_spec_match_arch(ps, pkg->installed.arch))
			return pkg;
	}

	/* Iterate over other package sets when needed. */
	if (!ps->pkg_iter)
		return NULL;

	while ((set = pkg_db_iter_next_set(ps->pkg_iter))) {
		if (!pkg_spec_match_pkgname(ps, set->name))
			continue;
		ps->pkg_next = &set->pkg;
		goto loop_arch;
	}

	return NULL;
}

void
pkg_spec_iter_destroy(struct pkg_spec *ps)
{
	pkg_db_iter_free(ps->pkg_iter);
	pkg_spec_iter_blank(ps);
}

void
pkg_spec_destroy(struct pkg_spec *ps)
{
	free(ps->name);
	pkg_spec_blank(ps);
	pkg_spec_iter_destroy(ps);
}
