﻿/* ****************************************************************************
 *
 * Copyright (c) Microsoft Corporation. 
 *
 * This source code is subject to terms and conditions of the Microsoft Public License. A 
 * copy of the license can be found in the License.html file at the root of this distribution. If 
 * you cannot locate the  Microsoft Public License, please send an email to 
 * ironruby@microsoft.com. By using this source code in any fashion, you are agreeing to be bound 
 * by the terms of the Microsoft Public License.
 *
 * You must not remove this notice, or any other, from this software.
 *
 *
 * ***************************************************************************/

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Dynamic;
using System.Linq.Expressions;
using System.Reflection;
using System.Threading;
using IronRuby.Builtins;
using IronRuby.Compiler;
using Microsoft.Scripting.Actions;
using Microsoft.Scripting.Utils;
using Ast = System.Linq.Expressions.Expression;
using AstUtils = Microsoft.Scripting.Ast.Utils;
using System.Collections;
using Microsoft.Scripting.Generation;

namespace IronRuby.Runtime.Calls {
    public sealed class MetaObjectBuilder {
        // RubyContext the site binder is bound to or null if it is unbound.
        private readonly RubyContext _siteContext;

        private Expression _condition;
        private BindingRestrictions/*!*/ _restrictions;
        private Expression _result;
        private List<ParameterExpression> _temps;
        private bool _error;
        private bool _treatRestrictionsAsConditions;

        internal MetaObjectBuilder(RubyMetaBinder/*!*/ binder, DynamicMetaObject/*!*/[]/*!*/ arguments)
            : this(binder.Context, (DynamicMetaObject)null, arguments) {
        }

        internal MetaObjectBuilder(IInteropBinder/*!*/ binder, DynamicMetaObject target, params DynamicMetaObject/*!*/[]/*!*/ arguments)
            : this(binder.Context, target, arguments) {
        }

        internal MetaObjectBuilder(DynamicMetaObject target, params DynamicMetaObject/*!*/[]/*!*/ arguments)
            : this((RubyContext)null, target, arguments) {
        }

        private MetaObjectBuilder(RubyContext siteContext, DynamicMetaObject target, DynamicMetaObject/*!*/[]/*!*/ arguments) {
            var restrictions = BindingRestrictions.Combine(arguments);
            if (target != null) {
                restrictions = target.Restrictions.Merge(restrictions);
            }

            _restrictions = restrictions;
            _siteContext = siteContext;
        }

        public bool Error {
            get { return _error; }
        }

        public Expression Result {
            get { return _result; }
            set { _result = value; }
        }

        public ParameterExpression BfcVariable { get; set; }

        /// <summary>
        /// A rule builder sets this up if the resulting rule is required to be wrapped in a non-local control flow handler.
        /// This delegate must be called exactly once (<see cref="BuildControlFlow"/> method).
        /// </summary>
        public Action<MetaObjectBuilder, CallArguments> ControlFlowBuilder { get; set; }

        public bool TreatRestrictionsAsConditions {
            get { return _treatRestrictionsAsConditions; }
            set { _treatRestrictionsAsConditions = value; }
        }

        internal DynamicMetaObject/*!*/ CreateMetaObject(DynamicMetaObjectBinder/*!*/ action) {
            return CreateMetaObject(action, action.ReturnType);
        }

        internal DynamicMetaObject/*!*/ CreateMetaObject(DynamicMetaObjectBinder/*!*/ action, Type/*!*/ returnType) {
            Debug.Assert(ControlFlowBuilder == null, "Control flow required but not built");

            var expr = _error ? Ast.Throw(_result, returnType) : AstUtils.Convert(_result, returnType);

            if (_condition != null) {
                var deferral = action.GetUpdateExpression(returnType);
                expr = Ast.Condition(_condition, expr, deferral);
            }

            if (_temps != null) {
                expr = Ast.Block(_temps, expr);
            }

            RubyBinder.DumpRule(action, _restrictions, expr);
            return new DynamicMetaObject(expr, _restrictions);
        }

        public ParameterExpression/*!*/ GetTemporary(Type/*!*/ type, string/*!*/ name) {
            return AddTemporary(Ast.Variable(type, name));
        }

        private ParameterExpression/*!*/ AddTemporary(ParameterExpression/*!*/ variable) {
            if (_temps == null) {
                _temps = new List<ParameterExpression>();
            }

            _temps.Add(variable);
            return variable;
        }

        public void BuildControlFlow(CallArguments/*!*/ args) {
            if (ControlFlowBuilder != null) {
                ControlFlowBuilder(this, args);
                ControlFlowBuilder = null;
            }
        }

        #region Result

        public void SetError(Expression/*!*/ expression) {
            Assert.NotNull(expression);
            Debug.Assert(!_error, "Error already set");

            _result = expression;
            _error = true;
        }

        public void SetWrongNumberOfArgumentsError(int actual, int expected) {
            SetError(Methods.MakeWrongNumberOfArgumentsError.OpCall(AstUtils.Constant(actual), AstUtils.Constant(expected)));
        }

        public void SetMetaResult(DynamicMetaObject/*!*/ metaResult, CallArguments/*!*/ args) {
            // TODO: 
            // Should NormalizeArguments return a struct that provides us an information whether to treat particular argument's restrictions as conditions?
            // The splatted array is stored in a local. Therefore we cannot apply restrictions on it.
            SetMetaResult(metaResult, args.Signature.HasSplattedArgument);
        }

        public void SetMetaResult(DynamicMetaObject/*!*/ metaResult, bool treatRestrictionsAsConditions) {
            _result = metaResult.Expression;
            if (treatRestrictionsAsConditions || _treatRestrictionsAsConditions) {
                AddCondition(metaResult.Restrictions.ToExpression());
            } else {
                Add(metaResult.Restrictions);
            }
        }

        #endregion

        #region Restrictions

        public void AddObjectTypeRestriction(object value, Expression/*!*/ expression) {
            if (value == null) {
                AddRestriction(Ast.Equal(expression, AstUtils.Constant(null)));
            } else {
                AddTypeRestriction(value.GetType(), expression);
            }
        }

        public void AddTypeRestriction(Type/*!*/ type, Expression/*!*/ expression) {
            if (_treatRestrictionsAsConditions) {
                AddCondition(Ast.TypeEqual(expression, type));
            } else if (expression.Type != type || !type.IsSealed) {
                Add(BindingRestrictions.GetTypeRestriction(expression, type));
            }
        }

        public void AddRestriction(Expression/*!*/ restriction) {
            if (_treatRestrictionsAsConditions) {
                AddCondition(restriction);
            } else {
                Add(BindingRestrictions.GetExpressionRestriction(restriction));
            }
        }

        public void AddRestriction(BindingRestrictions/*!*/ restriction) {
            if (_treatRestrictionsAsConditions) {
                AddCondition(restriction.ToExpression());
            } else {
                Add(restriction);
            }
        }

        private void Add(BindingRestrictions/*!*/ restriction) {
            Debug.Assert(!_treatRestrictionsAsConditions);
            _restrictions = _restrictions.Merge(restriction);
        }

        #endregion

        #region Conditions

        public void AddCondition(Expression/*!*/ condition) {
            Assert.NotNull(condition);
            _condition = (_condition != null) ? Ast.AndAlso(_condition, condition) : condition;
        }

        public static Expression/*!*/ GetObjectTypeTestExpression(object value, Expression/*!*/ expression) {
            if (value == null) {
                return Ast.Equal(expression, AstUtils.Constant(null));
            } else {
                return MakeTypeTestExpression(value.GetType(), expression);
            }
        }

        public static Expression MakeTypeTestExpression(Type t, Expression expr) {
            // we must always check for non-sealed types explicitly - otherwise we end up
            // doing fast-path behavior on a subtype which overrides behavior that wasn't
            // present for the base type.
            //TODO there's a question about nulls here
            if (CompilerHelpers.IsSealed(t) && t == expr.Type) {
                if (t.IsValueType) {
                    return AstUtils.Constant(true);
                }
                return Ast.NotEqual(expr, AstUtils.Constant(null));
            }

            return Ast.AndAlso(
                Ast.NotEqual(
                    expr,
                    AstUtils.Constant(null)),
                Ast.Equal(
                    Ast.Call(
                        AstUtils.Convert(expr, typeof(object)),
                        typeof(object).GetMethod("GetType")
                    ),
                    AstUtils.Constant(t)
                )
            );
        }


        public void AddObjectTypeCondition(object value, Expression/*!*/ expression) {
            AddCondition(GetObjectTypeTestExpression(value, expression));
        }

        #endregion

        #region Tests

        public void AddTargetTypeTest(object target, RubyClass/*!*/ targetClass, Expression/*!*/ targetParameter, DynamicMetaObject/*!*/ metaContext,
            IEnumerable<string>/*!*/ resolvedNames) {

            // no changes to the module's class hierarchy while building the test:
            targetClass.Context.RequiresClassHierarchyLock();

            // initialization changes the version number, so ensure that the module is initialized:
            targetClass.InitializeMethodsNoLock();

            var context = (RubyContext)metaContext.Value;

            if (target is IRubyObject) {
                Type type = target.GetType();
                AddTypeRestriction(type, targetParameter);
            
                // Ruby objects (get the method directly to prevent interface dispatch):
                MethodInfo classGetter = type.GetMethod(Methods.IRubyObject_get_ImmediateClass.Name, BindingFlags.Public | BindingFlags.Instance);
                if (type.IsVisible && classGetter != null && classGetter.ReturnType == typeof(RubyClass)) {
                    AddCondition(
                        // (#{type})target.ImmediateClass.Version.Value == #{immediateClass.Version}
                        Ast.Equal(
                            Ast.Field(
                                Ast.Field(
                                    Ast.Call(Ast.Convert(targetParameter, type), classGetter), 
                                    Fields.RubyClass_Version
                                ),
                                Fields.VersionHandle_Value
                            ),
                            AstUtils.Constant(targetClass.Version.Value)
                        )
                    );
                    return;
                }

                // TODO: explicit iface-implementation
                throw new NotSupportedException("Type implementing IRubyObject should be visible and have ImmediateClass getter");
            }

            AddRuntimeTest(metaContext);

            // singleton nil:
            if (target == null) {
                AddRestriction(Ast.Equal(targetParameter, AstUtils.Constant(null)));
                AddVersionTest(context.NilClass);
                return;
            }

            // singletons true, false:
            if (target is bool) {
                AddRestriction(Ast.AndAlso(
                    Ast.TypeIs(targetParameter, typeof(bool)),
                    Ast.Equal(Ast.Convert(targetParameter, typeof(bool)), AstUtils.Constant(target))
                ));

                AddVersionTest((bool)target ? context.TrueClass : context.FalseClass);
                return;
            }

            var nominalClass = targetClass.NominalClass;

            Debug.Assert(!nominalClass.IsSingletonClass);
            Debug.Assert(!nominalClass.IsRubyClass);

            // Do we need a singleton check?
            if (nominalClass.ClrSingletonMethods == null ||
                CollectionUtils.TrueForAll(resolvedNames, (methodName) => !nominalClass.ClrSingletonMethods.ContainsKey(methodName))) {

                // no: there is no singleton subclass of target class that defines any method being called:
                AddTypeRestriction(target.GetType(), targetParameter);
                AddVersionTest(targetClass);

            } else if (targetClass.IsSingletonClass) {

                // yes: check whether the incoming object is a singleton and the singleton has the right version:
                AddTypeRestriction(target.GetType(), targetParameter);
                AddCondition(Methods.IsClrSingletonRuleValid.OpCall(
                    metaContext.Expression,
                    targetParameter,
                    AstUtils.Constant(targetClass.Version.Value)
                ));

            } else {

                // yes: check whether the incoming object is NOT a singleton and the class has the right version:
                AddTypeRestriction(target.GetType(), targetParameter);
                AddCondition(Methods.IsClrNonSingletonRuleValid.OpCall(
                    metaContext.Expression, 
                    targetParameter,
                    Ast.Constant(targetClass.Version),
                    AstUtils.Constant(targetClass.Version.Value)
                ));
            }
        }

        private void AddRuntimeTest(DynamicMetaObject/*!*/ metaContext) {
            Assert.NotNull(metaContext);

            // check for runtime (note that the module's runtime could be different from the call-site runtime):
            if (_siteContext == null) {
                // TODO: use holder
                AddRestriction(Ast.Equal(metaContext.Expression, AstUtils.Constant(metaContext.Value)));
            } else if (_siteContext != metaContext.Value) {
                throw new InvalidOperationException("Runtime-bound site called from a different runtime");
            }
        }

        internal void AddVersionTest(RubyClass/*!*/ cls) {
            cls.Context.RequiresClassHierarchyLock();

            // check for module version (do not burn a module reference to the rule):
            AddCondition(Ast.Equal(Ast.Field(AstUtils.Constant(cls.Version), Fields.VersionHandle_Value), AstUtils.Constant(cls.Version.Value)));
        }

        internal bool AddSplattedArgumentTest(object value, Expression/*!*/ expression, out int listLength, out ParameterExpression/*!*/ listVariable) {
            if (value == null) {
                AddRestriction(Ast.Equal(expression, AstUtils.Constant(null)));
            } else {
                // test exact type:
                AddTypeRestriction(value.GetType(), expression);

                IList list = value as IList;
                if (list != null) {
                    Type type = typeof(IList);
                    listLength = list.Count;
                    listVariable = GetTemporary(type, "#list");
                    AddCondition(Ast.Equal(
                        Ast.Property(Ast.Assign(listVariable, Ast.Convert(expression, type)), typeof(ICollection).GetProperty("Count")),
                        AstUtils.Constant(list.Count))
                    );
                    return true;
                }
            }

            listLength = -1;
            listVariable = null;
            return false;
        }

        #endregion
    }
}
