/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.digikam.org
 *
 * Date        : 2009-12-23
 * Description : Autodetect binary program and version
 *
 * Copyright (C) 2009-2010 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include "binaryiface.h"

// Qt includes

#include <QProcess>
#include <QMessageBox>

// KDE includes

#include <kapplication.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kdebug.h>
#include <klocale.h>
#include <kvbox.h>

namespace KIPIPlugins
{

    BinaryIface::BinaryIface(const QString& binaryName, const QString& minimalVersion, const QString& header,
                             const int headerLine, const QString& projectName, const QString& url,
                             const QString& pluginName, const QStringList& args)
    : m_headerStarts(header), m_headerLine(headerLine), m_minimalVersion(minimalVersion),
      m_configGroup(pluginName + " Settings"), m_binaryBaseName(goodBaseName(binaryName)),
      m_binaryArguments(args), m_projectName(projectName), m_url(KUrl(url)),
      m_isFound(false), m_developmentVersion(false),
      m_version(""), m_pathDir("")
{
}

BinaryIface::~BinaryIface()
{
}

QString BinaryIface::version() const
{
    return m_version;
}

bool BinaryIface::versionIsRight() const
{
    QRegExp reg("^(\\d*[.]\\d*)");
    version().indexOf(reg);
    float floatVersion = reg.capturedTexts()[0].toFloat();
    return (!version().isNull()
         && isFound()
         && floatVersion >= minimalVersion().toFloat());
}

QString BinaryIface::findHeader(const QStringList& output, const QString& header) const
{
    foreach(const QString& s, output)
    {
        if (s.startsWith(header))
            return s;
    }
    return QString();
}

bool BinaryIface::parseHeader(const QString& output)
{
    QString firstLine = output.section('\n', m_headerLine, m_headerLine);
    kDebug() << path() << " help header line: \n" << firstLine;
    if (firstLine.startsWith(m_headerStarts))
    {
        QString version = firstLine.remove(0, m_headerStarts.length());
        if (version.startsWith("Pre-Release "))
        {
            version.remove("Pre-Release ");            // Special case with Hugin beta.
            m_developmentVersion = true;
        }
        setVersion(version);
        return true;
    }
    return false;
}

void BinaryIface::setVersion(QString& version)
{
    QRegExp versionRegExp("\\d*(\\.\\d+)*");
    version.indexOf(versionRegExp);
    m_version = versionRegExp.capturedTexts()[0];
}

void BinaryIface::slotNavigateAndCheck()
{
    KUrl start;
    if (isValid() && m_pathDir != "")
    {
        start = KUrl(m_pathDir);
    }
    else
    {
#if defined Q_WS_MAC
        start = KUrl(QString("/Applications/"));
#elif defined Q_WS_WIN
        start = KUrl(QString("C:/Program Files/"));
#else
        start = KUrl(QString("/usr/bin/"));
#endif
    }
    QString f = KFileDialog::getOpenFileName(start,
                                             QString(m_binaryBaseName),
                                             0,
                                             QString(i18n("Navigate to %1", m_binaryBaseName)));
    QString dir = KUrl(f).directory();
    m_searchPaths << dir;
    if (checkDir(dir))
    {
        emit signalSearchDirectoryAdded(dir);
    }
}

void BinaryIface::slotAddPossibleSearchDirectory(const QString& dir)
{
    if (!isValid())
    {
        m_searchPaths << dir;
        checkDir(dir);
    }
    else
    {
        m_searchPaths << dir;
    }
}

void BinaryIface::slotAddSearchDirectory(const QString& dir)
{
    m_searchPaths << dir;
    checkDir(dir);       // Forces the use of that directory
}

QString BinaryIface::readConfig()
{
    KConfig config("kipirc");
    KConfigGroup group = config.group(m_configGroup);
    return group.readPathEntry(QString("%1Binary").arg(m_binaryBaseName), "");
}

void BinaryIface::writeConfig()
{
    KConfig config("kipirc");
    KConfigGroup group = config.group(m_configGroup);
    group.writePathEntry(QString("%1Binary").arg(m_binaryBaseName), m_pathDir);
}

QString BinaryIface::path(const QString& dir) const
{
    if (dir.isEmpty())
    {
        return baseName();
    }
    return QString("%1%2%3").arg(dir).arg('/').arg(baseName());
}

void BinaryIface::setup()
{
    QString previous_dir = readConfig();
    m_searchPaths << previous_dir;
    checkDir(previous_dir);
    if (previous_dir != "" && !isValid())
    {
        m_searchPaths << "";
        checkDir("");
    }
}

bool BinaryIface::checkDir(const QString& possibleDir)
{
    bool ret = false;
    QString possiblePath = path(possibleDir);

    kDebug() << "Testing " << possiblePath << "...";
    QProcess process;
    process.setProcessChannelMode(QProcess::MergedChannels);
    process.start(possiblePath, m_binaryArguments);
    process.waitForFinished();

    if (process.error() != QProcess::FailedToStart)
    {
        m_isFound = true;

        QString stdOut(process.readAllStandardOutput());
        if (parseHeader(stdOut))
        {
            m_pathDir = possibleDir;
            writeConfig();

            kDebug() << "Found " << path() << " version: " << version();
            ret = true;
        }
        else
        {
            // TODO: do something if the version is not right or not found
        }
    }
    emit signalBinaryValid();
    return ret;
}

bool BinaryIface::recheckDirectories()
{
    if (isValid())
    {
        // No need for recheck if it is already valid...
        return true;
    }
    foreach(QString dir, m_searchPaths)
    {
        checkDir(dir);
        if (isValid())
        {
            return true;
        }
    }
    return false;
}

}  // namespace KIPIPlugins
