# Copyright 2010 Canonical Ltd.
#
# This file is part of desktopcouch.
#
# desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Manuel de la pena <manuel.delapena@canonical.com>

"""Tests for the Ubuntu One pairing."""

import ubuntu_sso

from mocker import MockerTestCase, ANY

from desktopcouch.application.plugins import ubuntuone_pairing as uone
from desktopcouch.application.plugins.ubuntuone_pairing import (
    pair_with_ubuntuone, U1_PAIR_RECORD, MAP_JS)


class TestUbuntonePairing(MockerTestCase):
    """Tests for Ubuntu One Pairing."""

    def setUp(self):
        """setup each test"""
        super(TestUbuntonePairing, self).setUp()
        # set a number of mocks that are used in all tests
        self.couchdb = self.mocker.mock()
        self.couchdb_port = self.mocker.mock()
        self.put_static_paired_service = self.mocker.mock()
        self.database_class = self.mocker.mock()
        self.blocking_semaphores = self.mocker.mock()

    def test_pair_with_ubuntuone_no_view(self):
        """Test that when the view is not present it is indeed created."""
        # plugin_init adds name to blocking semaphores, but we remove it.
        self.couchdb.view_exists(U1_PAIR_RECORD, U1_PAIR_RECORD)
        self.mocker.result(False)
        # we are interested in the fact that the view is created
        self.couchdb.add_view(
            U1_PAIR_RECORD, MAP_JS, design_doc=U1_PAIR_RECORD)
        # added to ensure that the method carries on normally
        self.couchdb.execute_view(U1_PAIR_RECORD, U1_PAIR_RECORD)
        self.mocker.result([])
        self.put_static_paired_service(None, 'ubuntuone', ctx=None, uri=ANY)
        self.blocking_semaphores.discard(ANY)

        self.mocker.replay()
        pair_with_ubuntuone(self.couchdb_port, self.blocking_semaphores,
                            management_db=self.couchdb,
                            db_class=self.database_class,
                            put_service_fn=self.put_static_paired_service)
        self.mocker.verify()

    def test_pair_with_ubuntuone_no_record(self):
        """Ensure pairing is not done when there are no records ."""
        # execute the steps when no records are returned by the view
        # plugin_init adds name to blocking semaphores, but we remove it.
        self.couchdb.view_exists(U1_PAIR_RECORD, U1_PAIR_RECORD)
        self.mocker.result(True)
        self.couchdb.execute_view(U1_PAIR_RECORD, U1_PAIR_RECORD)
        self.mocker.result([])
        self.put_static_paired_service(None, 'ubuntuone', ctx=None, uri=ANY)
        self.blocking_semaphores.discard(ANY)

        self.mocker.replay()
        pair_with_ubuntuone(self.couchdb_port, self.blocking_semaphores,
                            management_db=self.couchdb,
                            db_class=self.database_class,
                            put_service_fn=self.put_static_paired_service)

        self.mocker.verify()

    def test_pair_with_ubuntuone_user_deleted_record(self):
        """Ensure pairing is not done when the user explicitly removed it."""
        # create a mock object that will be the result from the view
        row = self.mocker.mock()
        # plugin_init adds name to blocking semaphores, but we remove it.
        # execute the steps to show that the user deleted the record
        self.couchdb.view_exists(U1_PAIR_RECORD, U1_PAIR_RECORD)
        self.mocker.result(True)
        self.couchdb.execute_view(U1_PAIR_RECORD, U1_PAIR_RECORD)
        self.mocker.result([row])
        # FIXME does this do anything?
        _ = row.value
        self.mocker.result(1)
        self.blocking_semaphores.discard(ANY)

        self.mocker.replay()
        pair_with_ubuntuone(self.couchdb_port, self.blocking_semaphores,
                            management_db=self.couchdb)
        self.mocker.verify()

    def test_pair_with_ubuntuone_record_present(self):
        """Ensure pairing is not done when the record is already present."""
        # create a mock object that will be the result from the view
        row = self.mocker.mock()
        # plugin_init adds name to blocking semaphores, but we remove it.
        # execute the steps to show that the user deleted the record
        self.couchdb.view_exists(U1_PAIR_RECORD, U1_PAIR_RECORD)
        self.mocker.result(True)
        self.couchdb.execute_view(U1_PAIR_RECORD, U1_PAIR_RECORD)
        self.mocker.result([row])
        # FIXME does this do anything?
        _ = row.value
        self.mocker.result(0)
        self.blocking_semaphores.discard(ANY)

        self.mocker.replay()
        pair_with_ubuntuone(self.couchdb_port, self.blocking_semaphores,
                            management_db=self.couchdb)
        self.mocker.verify()


class TestUbuntuOnePlugin(MockerTestCase):
    """Test the signal handling in the plug-in."""

    def setUp(self):
        super(TestUbuntuOnePlugin, self).setUp()
        self.couchdb_port = self.mocker.mock()
        self.blocking_semaphores = self.mocker.mock()
        self.called = False

    def test_got_new_credentials_other(self):
        """Check that pairing is not called."""
        def fail_if_called(db=None):
            """Fail if we get called."""
            self.called = True

        saved_pair = uone.pair_with_ubuntuone
        uone.pair_with_ubuntuone = fail_if_called
        uone.got_new_credentials(self.couchdb_port, self.blocking_semaphores,
                                 'Unknown App', {})
        self.assertFalse(self.called, 'pair_with_ubuntuone was not expected.')
        self.mocker.replay()
        uone.pair_with_ubuntuone = saved_pair

    def test_got_new_credentials(self):
        """Check that pairing is called for Ubuntu One."""
        def pass_if_called(db=None, semaphores=None):
            """Check that pair_with_ubuntuone was called."""
            self.called = True

        saved_pair = uone.pair_with_ubuntuone
        uone.pair_with_ubuntuone = pass_if_called
        uone.got_new_credentials(self.couchdb_port, self.blocking_semaphores,
                                 uone.APP_NAME, {})
        self.assertTrue(self.called, 'pair_with_ubuntuone was not called.')
        self.mocker.replay()
        uone.pair_with_ubuntuone = saved_pair

    def test_listen_to_dbus(self):
        """Test that listening to credentails works."""
        session_class = self.mocker.replace("dbus.SessionBus")
        iface_class = self.mocker.replace("dbus.Interface")

        session_class()
        bus = self.mocker.mock()
        self.mocker.result(bus)

        iface = ubuntu_sso.DBUS_CREDENTIALS_IFACE
        bus.add_signal_receiver(handler_function=ANY,
                                signal_name='CredentialsNotFound',
                                dbus_interface=iface)
        bus.add_signal_receiver(handler_function=ANY,
                                signal_name='CredentialsFound',
                                dbus_interface=iface)

        bus.get_object(ubuntu_sso.DBUS_BUS_NAME,
                       ubuntu_sso.DBUS_CREDENTIALS_PATH,
                       follow_name_owner_changes=True)
        an_obj = self.mocker.mock()
        self.mocker.result(an_obj)

        iface_class(an_obj, dbus_interface=iface)
        sso_backend = self.mocker.mock()
        self.mocker.result(sso_backend)

        sso_backend.find_credentials(uone.APP_NAME, {})

        self.mocker.replay()

        uone.listen_to_dbus(self.couchdb_port, self.blocking_semaphores)
