# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: RateMeasure.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Simple measurement of the download rate.

@type FACTOR: C{float}
@var FACTOR: the factor to use to adjust for TCP overhead

"""

from clock import clock

FACTOR = 0.999

class RateMeasure:
    """Simple measurement of the download rate.
    
    @type last: C{float}
    @ivar last: the last time the rate was updated
    @type time: C{float}
    @ivar time: the amount of time that has elapsed since data first came in
    @type got: C{float}
    @ivar got: the amount of data that's come in so far
    @type remaining: C{float}
    @ivar remaining: the number of seconds remaining from the last calculation
    @type broke: C{boolean}
    @ivar broke: unused
    @type got_anything: C{boolean}
    @ivar got_anything: whether any data has been received yet
    @type last_checked: C{float}
    @ivar last_checked: the last time the finishing time was calculated
    @type rate: C{int}
    @ivar rate: the last calculated download rate
    @type lastten: C{boolean}
    @ivar lastten: whether the download is in the last ten seconds
    
    """
    
    def __init__(self):
        """Initialize the instance."""
        self.last = None
        self.time = 1.0
        self.got = 0.0
        self.remaining = None
        self.broke = False
        self.got_anything = False
        self.last_checked = None
        self.rate = 0
        self.lastten = False

    def data_came_in(self, amount):
        """Add new data received to the rate.
        
        @type amount: C{int}
        @param amount: the amount of data received
        
        """
        
        if not self.got_anything:
            self.got_anything = True
            self.last = clock()
            return
        self.update(amount)

    def data_rejected(self, amount):
        """Add new data received to the rate.
        
        @type amount: C{int}
        @param amount: the amount of data received
        
        """
        
        pass

    def get_time_left(self, left):
        """Calculate the amount of time left to complete the download.
        
        @type left: C{long}
        @param left: the amount of data still to be downloaded
        @rtype: C{float}
        @return: the number of seconds left in the download
            (or None if no data has been received)
        
        """
        
        t = clock()
        if not self.got_anything:
            return None
        if t - self.last > 15:
            self.update(0)
        try:
            remaining = left/self.rate
            if not self.lastten and remaining <= 10:
                self.lastten = True
            if self.lastten:
                return remaining
            delta = max(remaining/20,2)
            if self.remaining is None:
                self.remaining = remaining
            elif abs(self.remaining-remaining) > delta:
                self.remaining = remaining
            else:
                self.remaining -= t - self.last_checked
        except ZeroDivisionError:
            self.remaining = None
        if self.remaining is not None and self.remaining < 0.1:
            self.remaining = 0.1
        self.last_checked = t
        return self.remaining

    def update(self, amount):
        """Update the rate with new received data.
        
        @type amount: C{int}
        @param amount: the amount of data received
        
        """
        
        t = clock()
        t1 = int(t)
        l1 = int(self.last)
        for i in xrange(l1,t1):
            self.time *= FACTOR
            self.got *= FACTOR
        self.got += amount
        if t - self.last < 20:
            self.time += t - self.last
        self.last = t
        try:
            self.rate = self.got / self.time
        except ZeroDivisionError:
            pass
