/*
 *  publichubswidget.h
 *  ui
 *
 *  Created by Mikael Gransell on 3/21/06.
 *
 */

#include <boost/shared_ptr.hpp>

#include <QWidget>
#include <QAbstractTableModel>

#include "rpctypes.h"
#include "ui_publichublist.h"
#include "backendconnection.h"

class PublicHubs : public QWidget
{
	Q_OBJECT
public:
	PublicHubs( const BackendConnectionPtr& backendConn,
				QWidget* parent = NULL );
	~PublicHubs() {	}
	
public slots:
	//! Called when a new list of public hubs are received from the backend
	void hubList( const QList<rpc_types::HubEntry>& hubs );
	void on_filerEdit_textChanged();
	void on_connectButton_pressed();
	void on_refreshButton_pressed();
	void on_hubList_doubleClicked();

signals:
	void close();
	
private:
	class PublicHubsModel;

	Ui::PublicHubForm ui;
	/// To send rpc commands to the backend
	BackendConnectionPtr backendConnection;
	/// Keep track of the data that is in the list
	boost::shared_ptr<PublicHubsModel> model;
	
	/**
	 * Class that holds the data displayed in the public hub list.
	 */
	class PublicHubsModel : public QAbstractTableModel
	{
	public:
		PublicHubsModel(QObject* parent);
		~PublicHubsModel()
		{}
		int rowCount ( const QModelIndex& ) const
		{
			//return hubEntries.size();
			return shownEntries.size();
		}
		int columnCount ( const QModelIndex& ) const
		{
			return 6;
		}
		QVariant data ( const QModelIndex & index, int role = Qt::DisplayRole ) const;
		QVariant headerData ( int section, Qt::Orientation orientation, int role = Qt::DisplayRole ) const;
		const QString getUrl(int row) const;
		void sort(int col,Qt::SortOrder order = Qt::AscendingOrder);
		
		class Entry 
		{
		public:
			Entry( const Entry& rhs ) 
				: data( rhs.getData() )
			{}
			
			Entry( const rpc_types::HubEntry& rawData )
				: data( rawData )
			{}
			
			bool operator<(const PublicHubsModel::Entry& e) const
			{
				rpc_types::HubEntry data1 = data;
				rpc_types::HubEntry data2 = e.getData();
				
				if(sortOrder!=Qt::AscendingOrder) {
					std::swap(data1, data2);
				}
				
				switch(sortCol)
				{
					case 1:
						return data2.server < data2.server;
					case 2:
						return data1.description < data2.description;
					case 3:
						return data1.country < data2.country;
					case 4:
						return data1.minShare < data2.minShare;
					case 5:
						return data1.users < data2.users;
					case 0:
					default:
						return data1.name < data2.name;
				}
			}
			
			/// Return the data that this list entry represents as raw backend data
			const rpc_types::HubEntry& getData() const { return data; }
			/// The collumn that the list is sorted on
			static int sortCol;
			/// Ascending or descending sort order
			static Qt::SortOrder sortOrder;
		
		private:
			/// The actual data stored as is from the backend
			rpc_types::HubEntry data;
		};
		
		//const PublicHubsModel::Entry& entry(int row) const {
		//	if(row < 0 || row>=hubEntries.size()) throw "index out of bounds";
		//	return hubEntries.at(row);
		//}
		
		void setHubs( const QList<rpc_types::HubEntry>& );
		void applyFilter(const QString& pattern);
		
	private:
			QList<Entry> hubEntries;
			QList<int> shownEntries;
			QString filterPattern;
	};
};


