/* Copyright (C) 2007 db4objects Inc. http://www.db4o.com */
using System;
using System.IO;
using System.Collections.Generic;

using Db4objects.Db4o;
using Db4objects.Db4o.Query;
using Db4objects.Db4o.Ext;
using Db4objects.Db4o.Config;
using System.Collections;


namespace Db4objects.Db4odoc.BuiltInTranslators
{
    class BuiltInTranslatorsExample
    {
        private const string Db4oFileName = "reference.db4o";

        private static IObjectContainer _container = null;

        public static void Main(string[] args)
        {
            TestTNull();
            TestSerializable();
        }
        // end Main

        private static IObjectContainer Database()
        {
            if (_container == null)
            {
                try
                {
                    _container = Db4oFactory.OpenFile(Db4oFileName);
                }
                catch (DatabaseFileLockedException ex)
                {
                    System.Console.WriteLine(ex.Message);
                }
            }
            return _container;
        }

        // end Database

        private static IObjectContainer Database(IConfiguration configuration)
        {
            if (_container == null)
            {
                try
                {
                    _container = Db4oFactory.OpenFile(configuration, Db4oFileName);
                }
                catch (DatabaseFileLockedException ex)
                {
                    System.Console.WriteLine(ex.Message);
                }
            }
            return _container;
        }

        // end Database

        private static void CloseDatabase()
        {
            if (_container != null)
            {
                _container.Close();
                _container = null;
            }
        }

        // end CloseDatabase


        public static void SaveNotStorable()
        {
            File.Delete(Db4oFileName);
            IConfiguration configuration = Db4oFactory.NewConfiguration();
            // Configure NotStorable class with TNull translator to prevent its storage
            configuration.ObjectClass(typeof(NotStorable)).Translate(new TNull());
            IObjectContainer container = Database(configuration);
            if (container != null)
            {
                try
                {
                    NotStorable ns =  new NotStorable("test1");
                    container.Set(ns);
                    ns = new NotStorable("test2");
                    container.Set(ns);
                }
                catch (Db4oException ex)
                {
                    System.Console.WriteLine("Db4o Exception: " + ex.Message);
                }
                catch (Exception ex)
                {
                    System.Console.WriteLine("System Exception: " + ex.Message);
                }
                finally
                {
                    CloseDatabase();
                }
            }
        }
        // end SaveNotStorable

        public static void TestTNull()
        {
            SaveNotStorable();
            IObjectContainer container = Database();
            if (container != null)
            {
                try
                {
                    // Trying to retrieve
                    IList<NotStorable> result = container.Query<NotStorable>();
                    // As the class is configured with TNull, the data should be null
                    ListResult(result);
                }
                catch (Db4oException ex)
                {
                    System.Console.WriteLine("Db4o Exception: " + ex.Message);
                }
                catch (Exception ex)
                {
                    System.Console.WriteLine("System Exception: " + ex.Message);
                }
                finally
                {
                    CloseDatabase();
                }
            }
        }
        // end TestTNull

        public static void TestSerializable()
        {
            SaveSerializable();
            IConfiguration configuration = Db4oFactory.NewConfiguration();
            // configure class as serializable to retrieve correctly
            configuration.ObjectClass(typeof(Pilot)).Translate(new TSerializable());
            IObjectContainer container = Database(configuration);
            if (container != null)
            {
                try
                {
                    System.Console.WriteLine("Retrieving pilots by name:");
                    IQuery query = container.Query();
                    query.Constrain(typeof(Pilot));
                    query.Descend("_name").Constrain("Test Pilot 1");
                    IObjectSet resultByName = query.Execute();
                    ListResult(resultByName);
                    
                    System.Console.WriteLine("Retrieve all pilot objects:");
                    IList<Pilot> result = container.Query<Pilot>();
                    ListResult(result);
                }
                catch (Db4oException ex)
                {
                    System.Console.WriteLine("Db4o Exception: " + ex.Message);
                }
                catch (Exception ex)
                {
                    System.Console.WriteLine("System Exception: " + ex.Message);
                }
                finally
                {
                    CloseDatabase();
                }
            }
        }
        // end TestSerializable

        public static void SaveSerializable()
        {
            File.Delete(Db4oFileName);
            IConfiguration configuration = Db4oFactory.NewConfiguration();
            // configure class as serializable
            configuration.ObjectClass(typeof(Pilot)).Translate(new TSerializable());
            IObjectContainer container = Database(configuration);
            if (container != null)
            {
                try
                {
                    Pilot pilot = new Pilot("Test Pilot 1", 99);
                    container.Set(pilot);
                    pilot = new Pilot("Test Pilot 2", 100);
                    container.Set(pilot);
                }
                catch (Db4oException ex)
                {
                    System.Console.WriteLine("Db4o Exception: " + ex.Message);
                }
                catch (Exception ex)
                {
                    System.Console.WriteLine("System Exception: " + ex.Message);
                }
                finally
                {
                    CloseDatabase();
                }
            }

        }
        // end SaveSerializable


        private static void ListResult(IObjectSet result)
        {
            System.Console.WriteLine(result.Count);
            for (int i = 0; i < result.Count; i++)
            {
                System.Console.WriteLine(result[i]);
            }
        }

        // end ListResult

        private static void ListResult<T>(IList<T> result)
        {
            System.Console.WriteLine(result.Count);
            for (int i = 0; i < result.Count; i++)
            {
                System.Console.WriteLine(result[i]);
            }
        }

        // end ListResult
    }
}
