// Cyphesis Online RPG Server and AI Engine
// Copyright (C) 2008 Alistair Riddoch
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

// $Id: Identified.cpp,v 1.3 2008-01-28 23:48:31 alriddoch Exp $

#include "Identified.h"

#include "compose.hpp"
#include "log.h"

#include <Atlas/Objects/Anonymous.h>
#include <Atlas/Objects/Operation.h>
#include <Atlas/Objects/SmartPtr.h>

Identified::Identified(const std::string & id, long intId) : m_id(id),
                                                             m_intId(intId)
{
}

/// \brief Report an Error.
///
/// The error reported is noted in the log, and an error operation is
/// genereated.
/// @param op The operation that caused the error.
/// @param errstring A message describing the error.
/// @param res The resulting error operation is returned here.
/// @param to The error operation should be directed to this ID.
void Identified::error(const Operation & op,
                       const std::string & errstring,
                       OpVector & res,
                       const std::string & to) const
{
    Atlas::Objects::Operation::Error e;

    log(NOTICE, String::compose("ERROR generated by %1 with message [%2]",
                                getId(), errstring));

    std::vector<Atlas::Objects::Root> & args = e->modifyArgs();

    Atlas::Objects::Entity::Anonymous arg1;
    arg1->setAttr("message", errstring);
    args.push_back(arg1);
    args.push_back(op);

    if (!to.empty()) {
        if (!op->isDefaultSerialno()) {
            e->setRefno(op->getSerialno());
        }
        e->setTo(to);
    }

    res.push_back(e);
}

/// \brief Report an Error to a client.
///
/// The error reported generates an error operation.
/// This is used instead of error() when an event occurs which is of no
/// interest to the server admin, or world builder, and should only be
/// be reported to the client. It stops the logs from getting filled
/// with reports of authentication failures, and other similar occurences.
/// @param op The operation that caused the error.
/// @param errstring A message describing the error.
/// @param res The resulting error operation is returned here.
/// @param to The error operation should be directed to this ID.
void Identified::clientError(const Operation & op,
                             const std::string & errstring,
                             OpVector & res,
                             const std::string & to) const
{
    Atlas::Objects::Operation::Error e;

    std::vector<Atlas::Objects::Root> & args = e->modifyArgs();

    Atlas::Objects::Entity::Anonymous arg1;
    arg1->setAttr("message", errstring);
    args.push_back(arg1);
    args.push_back(op);

    if (!to.empty()) {
        if (!op->isDefaultSerialno()) {
            e->setRefno(op->getSerialno());
        }
        e->setTo(to);
    }

    res.push_back(e);
}

/// \brief Dummy protected constructor without args.
///
/// Because Identified is virtually inherited, it's constructor is called
/// directly by actual constructor, not from here. This is added as a
/// convenience so that the ID arguments don't have to be explicitly passed
/// to this constructor as well. We have to pass something to the Identified
/// constructor here, but it has in fact already been called.
IdentifiedRouter::IdentifiedRouter() : Identified("", -1)
{
     assert(!getId().empty());
     assert(getIntId() != -1);
}

IdentifiedRouter::~IdentifiedRouter()
{
}

/// \brief Copy the attribute values of this object to an Atlas Message
void IdentifiedRouter::addToMessage(Atlas::Message::MapType & omap) const
{
    omap["objtype"] = "obj";
    omap["id"] = getId();
}

/// \brief Copy the attribute values of this object to an Atlas Entity
void IdentifiedRouter::addToEntity(const Atlas::Objects::Entity::RootEntity & ent) const
{
    ent->setObjtype("obj");
    ent->setId(getId());
}
