// 
// Copyright (c) 2006-2010, Benjamin Kaufmann
// 
// This file is part of Clasp. See http://www.cs.uni-potsdam.de/clasp/ 
// 
// Clasp is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// Clasp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Clasp; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
#ifndef CLASP_CLASP_OPTIONS_H_INCLUDED
#define CLASP_CLASP_OPTIONS_H_INCLUDED

#ifdef _MSC_VER
#pragma warning (disable : 4200) // nonstandard extension used : zero-sized array
#pragma once
#endif

#include <string>
#include <utility>
#include <program_opts/app_options.h>
#include <clasp/clasp_facade.h>
#include <clasp/solver.h>
#include <iosfwd>

namespace Clasp {

/////////////////////////////////////////////////////////////////////////////////////////
// Option groups - Mapping between command-line options and libclasp objects
/////////////////////////////////////////////////////////////////////////////////////////
// Function for mapping positional options
bool parsePositional(const std::string& s, std::string& out);
struct StringToEnum {
	const char* str; // string value
	int         ev;  // corresponding enum value
};
typedef StringToEnum EnumMap;

// Group "Basic Options"
struct BasicOptions {
	BasicOptions();
	typedef std::pair<int, int> QPair;
	typedef ProgramOptions::StringSeq StringSeq;
	static bool mapIfs(const std::string& s, char& ifs);
	void initOptions(ProgramOptions::OptionContext& root);
	StringSeq   input;  // list of input files - only first used!
	int         verbose;// verbosity level
	int         timeout;// timeout in seconds (default: none=-1)
	QPair       quiet;  // configure printing of models and optimization values
	int         stats;  // print statistics
	int         outf;   // output format
	bool        maxSat; // interpret dimacs as MaxSAT problem
	char        ifs;    // field separator
	enum OutputFormat {
		out_def     = 0,
		out_comp    = 1,
		out_json    = 2
	};
};

// Group "Clasp - General Options"
// Options of this group are mapped to ClaspConfig::api
// and ClaspConfig::enumerate
struct GeneralOptions {
	explicit GeneralOptions(ClaspConfig* c = 0) : config(c) {}
	static bool parseEnumMode(const std::string& s, GlobalOptions::EnumMode& m); 
	static bool mapMyVec(GeneralOptions*, const std::string&, const std::vector<int>* v);
	static bool mapLearnExplicit(GeneralOptions*, const std::string& k, const bool* b);
	void initOptions(ProgramOptions::OptionContext& root);
	bool validateOptions(const ProgramOptions::ParsedOptions& parsed, ProgramOptions::Messages&);
	std::vector<int>  myVec;
	ClaspConfig*      config;
	static const EnumMap enumModes[];
	static const EnumMap extRules[];
};

// Groups "Clasp - Search Options" and "Clasp - Lookback Options"
// Options of these groups are mapped to ClaspConfig::solve 
// and ClaspConfig::solver
struct SearchOptions {
	explicit SearchOptions(LocalOptions* l);
	void initOptions(ProgramOptions::OptionContext& root);
	bool validateOptions(const ProgramOptions::ParsedOptions& parsed, ProgramOptions::Messages&);
	// value parsing 
	static bool parseHeuristic(const std::string& s, std::string&);
	static bool parseSchedule(const std::string& s, ScheduleStrategy& sched);
	// value mapping
	static bool mapCustom(SearchOptions*  , const std::string& opt, const std::string& v);
	static bool mapNumber(SearchOptions*  , const std::string& opt, const double* v);
	static bool mapRandOpts(SearchOptions*, const std::string& opt, const std::string& v);
	LocalOptions*      local;
	SolverStrategies*  solverOpts() const;
	double             aNumber;
	static const EnumMap lookTypes[];
	static const EnumMap anteTypes[];
	static const EnumMap loopTypes[];
};
	
#define CLASP_DEFAULT_PORTFOLIO_SIZE 9
extern const char* portfolio_g;

// Combines all groups and drives initialization/validation 
// of command-line options.
class ClaspOptions {
public:
	ClaspOptions() : genTemplate(false), satPreDefault(true), config(0) {}
	BasicOptions basic;
	std::string  portfolio;
	bool         genTemplate;
	bool         satPreDefault;
	void         initOptions(ProgramOptions::OptionContext& root, ClaspConfig& c);
	bool         validateOptions(const ProgramOptions::ParsedOptions& parsed, ProgramOptions::Messages&);
	void         applyDefaults(Input::Format f);
	const char*  getInputDefaults(Input::Format f) const;
	static bool  mapThreadOpts(ClaspOptions*, const std::string& k, const std::string& v);
private:
	bool parsePortfolio(std::istream& in, uint32 max, std::string& mem) const;
	bool populateThreadConfigs(const char* port, const ProgramOptions::ParsedOptions& parsed, ProgramOptions::Messages&);
	bool parseConfig(const char* p, uint32 id, const ProgramOptions::ParsedOptions& parsed, ProgramOptions::Messages&);
	void initThreadOptions(ProgramOptions::OptionContext& root);
	ClaspConfig*                  config;
	std::auto_ptr<GeneralOptions> mode;
	std::auto_ptr<SearchOptions>  search;
};
}
#endif

