# Copyright (c) 2011 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

# There's a couple key GYP variables that control how FFmpeg is built:
#   ffmpeg_branding
#     Controls whether we build the Chromium or Google Chrome version of
#     FFmpeg.  The Google Chrome version contains additional codecs.
#     Typical values are Chromium, Chrome, ChromiumOS, and ChromeOS.
#   use_system_ffmpeg
#     When set to non-zero will build Chromium against the system FFmpeg
#     headers via pkg-config.  When Chromium is launched it will assume that
#     FFmpeg is present in the system library path.  Default value is 0.
#   build_ffmpegsumo
#     When set to zero will build Chromium against the patched ffmpegsumo
#     headers, but not build ffmpegsumo itself.  Users are expected to build
#     and provide their own version of ffmpegsumo.  Default value is 1.
#  use_system_vpx
#     When set to non-zero, will build Chromium against the system libvpx.
#     libvpx doesn't currently supply a pkg-config file, so we assume that
#     -lvpx is sufficient.
#
# TODO(ihf): Define these variables and use them.
# LIBAVCODEC_VERSION_MAJOR=53
# LIBAVFORMAT_VERSION_MAJOR=53
# LIBAVUTIL_VERSION_MAJOR=51

{
  'target_defaults': {
    'conditions': [
      ['OS!="linux" and OS!="freebsd" and OS!="openbsd" and OS!="solaris"', {
        'sources/': [['exclude', '/linux/']]
      }],
      ['OS!="mac"', {'sources/': [['exclude', '/mac/']]}],
      ['OS!="win"', {'sources/': [['exclude', '/win/']]}],
    ],
    'variables': {
      # Since we are not often debugging FFmpeg, and performance is
      # unacceptable without optimization, freeze the optimizations to -O2.
      # If someone really wants -O1 , they can change these in their checkout.
      # If you want -O0, see the Gotchas in README.Chromium for why that
      # won't work.
      'debug_optimize': '2',
      'mac_debug_optimization': '2',
    },
  },
  'variables': {
    # Allow overridding the selection of which FFmpeg binaries to copy via an
    # environment variable.  Affects the ffmpeg_binaries target.

    'conditions': [
      ['armv7==1 and arm_neon==1', {
        # Need a separate config for arm+neon vs arm
        'ffmpeg_config%': 'arm-neon',
      }, {
        'ffmpeg_config%': '<(target_arch)',
      }],
      ['OS=="mac" or OS=="win"', {
        'os_config%': '<(OS)',
      }, {  # all other Unix OS's use the linux config
        'os_config%': 'linux',
      }],
      ['chromeos==1', {
        'ffmpeg_branding%': '<(branding)OS',
      }, {  # otherwise, assume Chrome/Chromium.
        'ffmpeg_branding%': '<(branding)',
      }],
      # libvpx location.
      # TODO(scherkus): libvpx_hack_dir is a hack to make -L work on linux.
      ['OS=="mac" or OS=="win"', {
        'libvpx_hack_dir': '../libvpx',
      }],
    ],

    'ffmpeg_variant%': '<(target_arch)',

    'use_system_ffmpeg%': 0,
    'use_system_yasm%': 0,
    'build_ffmpegsumo%': 1,
    'use_system_vpx%': 0,

    # Locations for generated artifacts.
    'shared_generated_dir': '<(SHARED_INTERMEDIATE_DIR)/third_party/ffmpeg',
    'asm_library': 'ffmpegasm',
  },
  'conditions': [
    ['OS!="win" and use_system_ffmpeg==0 and build_ffmpegsumo!=0', {
      'targets': [
        {
          'target_name': 'ffmpegsumo',
          'type': 'shared_library',
          'sources': [
            'patched-ffmpeg/libavcodec/allcodecs.c',
            'patched-ffmpeg/libavcodec/audioconvert.c',
            'patched-ffmpeg/libavcodec/avfft.c',
            'patched-ffmpeg/libavcodec/avpacket.c',
            'patched-ffmpeg/libavcodec/bitstream.c',
            'patched-ffmpeg/libavcodec/bitstream_filter.c',
            'patched-ffmpeg/libavcodec/dsputil.c',
            'patched-ffmpeg/libavcodec/faandct.c',     # sep 09
            'patched-ffmpeg/libavcodec/faanidct.c',
            'patched-ffmpeg/libavcodec/fft_fixed.c',   # jun 11
            'patched-ffmpeg/libavcodec/fft_float.c',   # jun 11
            'patched-ffmpeg/libavcodec/fmtconvert.c',  # jun 11
            'patched-ffmpeg/libavcodec/golomb.c',
            'patched-ffmpeg/libavcodec/imgconvert.c',
            'patched-ffmpeg/libavcodec/jfdctfst.c',    # sep 09
            'patched-ffmpeg/libavcodec/jfdctint.c',    # sep 09
            'patched-ffmpeg/libavcodec/jrevdct.c',
            'patched-ffmpeg/libavcodec/mdct_fixed.c',  # jun 11
            'patched-ffmpeg/libavcodec/mdct_float.c',  # jun 11
            'patched-ffmpeg/libavcodec/mpeg12data.c',
            'patched-ffmpeg/libavcodec/mpeg4audio.c',
            'patched-ffmpeg/libavcodec/mpegaudiodata.c', # jun 11
            'patched-ffmpeg/libavcodec/options.c',
            'patched-ffmpeg/libavcodec/parser.c',
            'patched-ffmpeg/libavcodec/pcm.c',
            'patched-ffmpeg/libavcodec/pthread.c',
            'patched-ffmpeg/libavcodec/raw.c',
            'patched-ffmpeg/libavcodec/rawdec.c',      # jun 11
            'patched-ffmpeg/libavcodec/rdft.c',
            'patched-ffmpeg/libavcodec/simple_idct.c',
            'patched-ffmpeg/libavcodec/utils.c',
            'patched-ffmpeg/libavcodec/vorbis.c',
            'patched-ffmpeg/libavcodec/vorbis_data.c',
            'patched-ffmpeg/libavcodec/vorbisdec.c',   # jun 11
            'patched-ffmpeg/libavcodec/vp3.c',
            'patched-ffmpeg/libavcodec/vp3dsp.c',
            'patched-ffmpeg/libavcodec/xiph.c',
            'patched-ffmpeg/libavformat/allformats.c',
            'patched-ffmpeg/libavformat/avio.c',
            'patched-ffmpeg/libavformat/aviobuf.c',
            'patched-ffmpeg/libavformat/cutils.c',
            'patched-ffmpeg/libavformat/id3v1.c',
            'patched-ffmpeg/libavformat/id3v2.c',
            'patched-ffmpeg/libavformat/isom.c',
            'patched-ffmpeg/libavformat/matroska.c',
            'patched-ffmpeg/libavformat/matroskadec.c',
            'patched-ffmpeg/libavformat/metadata.c',
            'patched-ffmpeg/libavformat/oggdec.c',
            'patched-ffmpeg/libavformat/oggparseogm.c',
            'patched-ffmpeg/libavformat/oggparsecelt.c',      # jun 11
            'patched-ffmpeg/libavformat/oggparseskeleton.c',  # new
            'patched-ffmpeg/libavformat/oggparsetheora.c',
            'patched-ffmpeg/libavformat/oggparsevorbis.c',
            'patched-ffmpeg/libavformat/options.c',
            'patched-ffmpeg/libavformat/pcm.c',        # sep 09
            'patched-ffmpeg/libavformat/riff.c',
            'patched-ffmpeg/libavformat/rm.c',
            'patched-ffmpeg/libavformat/rmdec.c',
            'patched-ffmpeg/libavformat/seek.c',
            'patched-ffmpeg/libavformat/utils.c',
            'patched-ffmpeg/libavformat/vorbiscomment.c',
            'patched-ffmpeg/libavformat/wav.c',
            'patched-ffmpeg/libavutil/audioconvert.c', # jun 11
            'patched-ffmpeg/libavutil/avstring.c',
            'patched-ffmpeg/libavutil/base64.c',
            'patched-ffmpeg/libavutil/cpu.c',          # sep 09
            'patched-ffmpeg/libavutil/crc.c',
            'patched-ffmpeg/libavutil/dict.c',         # jul 11
            'patched-ffmpeg/libavutil/eval.c',
            'patched-ffmpeg/libavutil/imgutils.c',     # jun 11
            'patched-ffmpeg/libavutil/intfloat_readwrite.c',
            'patched-ffmpeg/libavutil/inverse.c',      # aug 10
            'patched-ffmpeg/libavutil/log.c',
            'patched-ffmpeg/libavutil/lzo.c',
            'patched-ffmpeg/libavutil/mathematics.c',
            'patched-ffmpeg/libavutil/md5.c',
            'patched-ffmpeg/libavutil/mem.c',
            'patched-ffmpeg/libavutil/opt.c',          # oct 05
            'patched-ffmpeg/libavutil/parseutils.c',   # jun 11
            'patched-ffmpeg/libavutil/pixdesc.c',
            'patched-ffmpeg/libavutil/random_seed.c',  # jun 11
            'patched-ffmpeg/libavutil/rational.c',
            'patched-ffmpeg/libavutil/samplefmt.c',    # jun 11
            'patched-ffmpeg/libavutil/utils.c',        # jun 11
            'config/<(ffmpeg_branding)/<(os_config)/<(ffmpeg_config)/config.h',
            'config/libavutil/avconfig.h',
          ],
          'include_dirs': [
            'config/<(ffmpeg_branding)/<(os_config)/<(ffmpeg_config)',
            'patched-ffmpeg',
            'config',
          ],
          'defines': [
            'HAVE_AV_CONFIG_H',
            '_POSIX_C_SOURCE=200112',
            '_XOPEN_SOURCE=600',
          ],
          'cflags': [
            '-fomit-frame-pointer',
          ],
          'conditions': [
            ['ffmpeg_branding=="Chrome" or ffmpeg_branding=="ChromeOS"', {
              'sources': [
                'patched-ffmpeg/libavcodec/aacadtsdec.c',  #new
                'patched-ffmpeg/libavcodec/aacdec.c',
                'patched-ffmpeg/libavcodec/aacps.c',  # new
                'patched-ffmpeg/libavcodec/aacsbr.c',
                'patched-ffmpeg/libavcodec/aactab.c',
                'patched-ffmpeg/libavcodec/cabac.c',
                'patched-ffmpeg/libavcodec/dct.c',                # jun 11
                'patched-ffmpeg/libavcodec/dct32_fixed.c',        # jun 11
                'patched-ffmpeg/libavcodec/dct32_float.c',        # jun 11
                'patched-ffmpeg/libavcodec/error_resilience.c',
                'patched-ffmpeg/libavcodec/h264.c',
                'patched-ffmpeg/libavcodec/h264_cabac.c',
                'patched-ffmpeg/libavcodec/h264_cavlc.c',
                'patched-ffmpeg/libavcodec/h264_direct.c',
                'patched-ffmpeg/libavcodec/h264_loopfilter.c',
                'patched-ffmpeg/libavcodec/h264_mp4toannexb_bsf.c',
                'patched-ffmpeg/libavcodec/h264_ps.c',
                'patched-ffmpeg/libavcodec/h264_refs.c',
                'patched-ffmpeg/libavcodec/h264_sei.c',
                'patched-ffmpeg/libavcodec/h264dsp.c',
                'patched-ffmpeg/libavcodec/h264idct.c',
                'patched-ffmpeg/libavcodec/h264pred.c',
                'patched-ffmpeg/libavcodec/kbdwin.c',             # jun 11
                'patched-ffmpeg/libavcodec/mpegaudio.c',
                'patched-ffmpeg/libavcodec/mpegaudio_parser.c',
                'patched-ffmpeg/libavcodec/mpegaudiodata.c',
                'patched-ffmpeg/libavcodec/mpegaudiodec.c',
                'patched-ffmpeg/libavcodec/mpegaudiodsp.c',       # jun 11
                'patched-ffmpeg/libavcodec/mpegaudiodsp_fixed.c', # jun 11
                'patched-ffmpeg/libavcodec/mpegaudiodsp_float.c', # jun 11
                'patched-ffmpeg/libavcodec/mpegaudiodecheader.c',
                'patched-ffmpeg/libavcodec/mpegvideo.c',
                'patched-ffmpeg/libavcodec/sinewin.c',            # jun 11
                'patched-ffmpeg/libavformat/mov.c',
                'patched-ffmpeg/libavformat/mp3dec.c',            # jun 11
                'patched-ffmpeg/libavformat/mp3enc.c',            # jun 11
              ],
            }],  # ffmpeg_branding
            ['ffmpeg_branding=="ChromiumOS" or ffmpeg_branding=="ChromeOS"', {
              'sources': [
                # TODO(ihf): let alpha sort these files when he does his merge
                'patched-ffmpeg/libavcodec/flac.c',               # jul 11
                'patched-ffmpeg/libavcodec/flacdata.c',           # jul 11
                'patched-ffmpeg/libavcodec/flacdec.c',            # jul 11
                'patched-ffmpeg/libavcodec/flac_parser.c',        # jul 11
                # BUG(ihf): Seems like a bug in ffmpeg that we need 
                #           h264pred.c for vp8.
                'patched-ffmpeg/libavcodec/h264pred.c',           # jul 11
                'patched-ffmpeg/libavcodec/vp56rac.c',            # jul 11
                'patched-ffmpeg/libavcodec/vp8.c',                # jul 11
                'patched-ffmpeg/libavcodec/vp8_parser.c',         # jul 11
                'patched-ffmpeg/libavcodec/vp8dsp.c',             # jul 11
                'patched-ffmpeg/libavformat/flacdec.c',           # jul 11
                'patched-ffmpeg/libavformat/rawdec.c',            # jul 11
                'patched-ffmpeg/libavformat/oggparseflac.c',      # jul 11
                'patched-ffmpeg/libavutil/fifo.c',                # jul 11
              ],
            }],  # ffmpeg_branding
            ['ffmpeg_branding=="ChromeOS"', {
              'sources': [
                'patched-ffmpeg/libavcodec/mpeg4video.c',         # jun 11
                'patched-ffmpeg/libavcodec/mpeg4videodec.c',      # jun 11
                'patched-ffmpeg/libavcodec/mpeg4video_parser.c',  # jun 11
                'patched-ffmpeg/libavcodec/h263.c',               # jun 11
                'patched-ffmpeg/libavcodec/h263_parser.c',        # jun 11
                'patched-ffmpeg/libavcodec/h263dec.c',            # jun 11
                'patched-ffmpeg/libavcodec/intelh263dec.c',       # jun 11
                'patched-ffmpeg/libavcodec/ituh263dec.c',         # jun 11
                'patched-ffmpeg/libavformat/avi.c',               # jun 11
                'patched-ffmpeg/libavformat/avidec.c',            # jun 11
                # TODO(ihf): let alpha sort these files when he does his merge
                'patched-ffmpeg/libavcodec/acelp_filters.c',      # jul 11
                'patched-ffmpeg/libavcodec/acelp_pitch_delay.c',  # jul 11
                'patched-ffmpeg/libavcodec/acelp_vectors.c',      # jul 11
                'patched-ffmpeg/libavcodec/amrnbdec.c',           # jul 11
                'patched-ffmpeg/libavcodec/amrwbdec.c',           # jul 11
                'patched-ffmpeg/libavcodec/celp_filters.c',       # jul 11
                'patched-ffmpeg/libavcodec/celp_math.c',          # jul 11
                'patched-ffmpeg/libavcodec/lsp.c',                # jul 11
                'patched-ffmpeg/libavformat/amr.c',               # jul 11
                'patched-ffmpeg/libavutil/lfg.c',                 # jul 11
              ],
            }],  # ffmpeg_branding
            ['use_system_yasm==0', {
              'dependencies': [
                '../yasm/yasm.gyp:yasm#host',
              ],
            }],
            ['target_arch=="ia32" or target_arch=="x64"', {
              'sources': [
                'patched-ffmpeg/libavcodec/libvpxdec.c',
                'patched-ffmpeg/libavcodec/libvpxenc.c',            # sep 09
                'patched-ffmpeg/libavcodec/x86/dsputil_mmx.c',
                'patched-ffmpeg/libavcodec/x86/dsputilenc_mmx.c',   # sep 09
                'patched-ffmpeg/libavcodec/x86/fdct_mmx.c',
                'patched-ffmpeg/libavcodec/x86/fft.c',
                'patched-ffmpeg/libavcodec/x86/fft_sse.c',
                'patched-ffmpeg/libavcodec/x86/fmtconvert_mmx.c',   # jun 11
                'patched-ffmpeg/libavcodec/x86/h264_intrapred_init.c',  # sep 09
                'patched-ffmpeg/libavcodec/x86/h264dsp_mmx.c',      # sep 09
                'patched-ffmpeg/libavcodec/x86/idct_mmx_xvid.c',
                'patched-ffmpeg/libavcodec/x86/idct_sse2_xvid.c',
                'patched-ffmpeg/libavcodec/x86/motion_est_mmx.c',   # sep 09
                'patched-ffmpeg/libavcodec/x86/simple_idct_mmx.c',
                'patched-ffmpeg/libavutil/x86/cpu.c',  # sep 09
                # The FFmpeg yasm files.
                'patched-ffmpeg/libavcodec/x86/dct32_sse.asm',      # jun 11
                'patched-ffmpeg/libavcodec/x86/deinterlace.asm',    # aug 10
                'patched-ffmpeg/libavcodec/x86/dsputil_yasm.asm',
                'patched-ffmpeg/libavcodec/x86/dsputilenc_yasm.asm',# oct 05
                'patched-ffmpeg/libavcodec/x86/fft_mmx.asm',
                'patched-ffmpeg/libavcodec/x86/fmtconvert.asm',     # jun 11
                'patched-ffmpeg/libavcodec/x86/h264_chromamc.asm',  # sep 09
                'patched-ffmpeg/libavcodec/x86/h264_chromamc_10bit.asm', # jul 11
                'patched-ffmpeg/libavcodec/x86/h264_deblock.asm',   # sep 09
                'patched-ffmpeg/libavcodec/x86/h264_deblock_10bit.asm', # jun 11
                'patched-ffmpeg/libavcodec/x86/h264_idct.asm',      # oct 05
                'patched-ffmpeg/libavcodec/x86/h264_idct_10bit.asm',# jun 11
                'patched-ffmpeg/libavcodec/x86/h264_intrapred.asm', # new
                'patched-ffmpeg/libavcodec/x86/h264_intrapred_10bit.asm', # jun 11
                'patched-ffmpeg/libavcodec/x86/h264_qpel_10bit.asm',# jul 11
                'patched-ffmpeg/libavcodec/x86/h264_weight.asm',    # sep 09
                'patched-ffmpeg/libavcodec/x86/h264_weight_10bit.asm', # jul 11
                'patched-ffmpeg/libavcodec/x86/vc1dsp_yasm.asm',    # new
                'patched-ffmpeg/libavcodec/x86/vp3dsp.asm',         # sep 09
                'patched-ffmpeg/libavcodec/x86/vp8dsp.asm',         # jul 11
              ],
            }],
            ['(target_arch=="ia32" or target_arch=="x64") and '
             '(ffmpeg_branding=="Chrome" or ffmpeg_branding=="ChromeOS")', {
              'sources': [
                'patched-ffmpeg/libavcodec/x86/mpegaudiodec_mmx.c',  # jun 11
                'patched-ffmpeg/libavcodec/x86/mpegvideo_mmx.c',
              ],
            }],
            ['(target_arch=="ia32" or target_arch=="x64") and '
             '(ffmpeg_branding=="ChromeOS" or ffmpeg_branding=="ChromiumOS")', {
              'sources': [
                'patched-ffmpeg/libavcodec/x86/vp8dsp-init.c',       # jul 11
              ],
            }],
            ['(target_arch=="ia32" or target_arch=="x64") and ffmpeg_branding=="ChromeOS"', {
              'sources': [
                'patched-ffmpeg/libavcodec/x86/vc1dsp_mmx.c',
              ],
            }],
            ['target_arch=="ia32"', {
              'cflags!': [
                # Turn off valgrind build option that breaks ffmpeg builds.
                # Allows config.h HAVE_EBP_AVAILABLE 1 and HAVE_EBX_AVAILABLE 1
                '-fno-omit-frame-pointer',
              ],
            }],  # target_arch=="ia32"
            ['target_arch=="x64"', {
              # x64 requires PIC for shared libraries. This is opposite
              # of ia32 where due to a slew of inline assembly using ebx,
              # FFmpeg CANNOT be built with PIC.
              'defines': [
                'PIC',
              ],
              'cflags': [
                '-fPIC',
              ],
            }],  # target_arch=="x64"
            ['target_arch=="arm"', {
              'defines': [
                'PIC',
              ],
              # TODO(ihf): The chromium buildbot adds -mthumb to all CFLAGS.
              # It breaks the build, so remove it here. Can be removed if
              # http://code.google.com/p/chromium/issues/detail?id=90473
              # is fixed.
              'cflags!': [
                '-mthumb',
              ],
              'cflags': [
                '-fPIC',
                '-marm',                                             # jul 11
                '-march=armv7-a',                                    # jul 11
                '-mtune=cortex-a8',                                  # jul 11
                '-mfloat-abi=softfp',                                # jul 11
              ],
              'conditions': [
                ['arm_neon==0', {
                  'cflags': [
                    '-mfpu=vfp',                                     # jul 11
                  ],
                }],
                ['arm_neon==1', {
                  'cflags': [
                    '-mfpu=neon',                                    # jul 11
                  ],
                }],
              ],
            }],
            # BUG(IHF): GYP syntax check does not like 2 conditions inside of 
            # target_arch=="arm", so split it into two
            ['target_arch=="arm"', {
              'sources': [
                'patched-ffmpeg/libavcodec/arm/dsputil_arm.S',
                'patched-ffmpeg/libavcodec/arm/dsputil_armv6.S',
                'patched-ffmpeg/libavcodec/arm/dsputil_init_arm.c',
                'patched-ffmpeg/libavcodec/arm/dsputil_init_armv5te.c',
                'patched-ffmpeg/libavcodec/arm/dsputil_init_armv6.c',
                'patched-ffmpeg/libavcodec/arm/dsputil_init_vfp.c',
                'patched-ffmpeg/libavcodec/arm/dsputil_vfp.S',
                'patched-ffmpeg/libavcodec/arm/fft_fixed_init_arm.c',  # jun 11
                'patched-ffmpeg/libavcodec/arm/fft_init_arm.c',
                'patched-ffmpeg/libavcodec/arm/fmtconvert_init_arm.c', # jun 11
                'patched-ffmpeg/libavcodec/arm/fmtconvert_vfp.S',      # jun 11
                # BUG(ihf): Seems like a bug in ffmpeg that we need
                #           h264pred_init_arm.c to enable vp8.
                'patched-ffmpeg/libavcodec/arm/h264pred_init_arm.c',
                'patched-ffmpeg/libavcodec/arm/jrevdct_arm.S',
                'patched-ffmpeg/libavcodec/arm/mpegaudiodsp_init_arm.c', # jul 11
                'patched-ffmpeg/libavcodec/arm/mpegaudiodsp_fixed_armv6.S', # jul 11
                'patched-ffmpeg/libavcodec/arm/mpegvideo_arm.c',         # jul 11
                'patched-ffmpeg/libavcodec/arm/mpegvideo_armv5te.c',     # jul 11
                'patched-ffmpeg/libavcodec/arm/mpegvideo_armv5te_s.S',   # jul 11
                'patched-ffmpeg/libavcodec/arm/simple_idct_arm.S',
                'patched-ffmpeg/libavcodec/arm/simple_idct_armv5te.S',
                'patched-ffmpeg/libavcodec/arm/simple_idct_armv6.S',
                'patched-ffmpeg/libavutil/arm/cpu.c',
              ],
              # TODO(scherkus): Temporary until libvpx compiles on ARM.
              'sources!': [
                'patched-ffmpeg/libavcodec/libvpxdec.c',
              ],
              'conditions': [
                ['arm_neon==1', {
                  'sources': [
                    'patched-ffmpeg/libavcodec/arm/dsputil_init_neon.c',
                    'patched-ffmpeg/libavcodec/arm/dsputil_neon.S',
                    'patched-ffmpeg/libavcodec/arm/fft_fixed_neon.S',  # jun 11
                    'patched-ffmpeg/libavcodec/arm/fft_neon.S',
                    'patched-ffmpeg/libavcodec/arm/fmtconvert_neon.S', # jun 11
                    'patched-ffmpeg/libavcodec/arm/int_neon.S',
                    'patched-ffmpeg/libavcodec/arm/mdct_fixed_neon.S', # jun 11
                    'patched-ffmpeg/libavcodec/arm/mdct_neon.S',
                    'patched-ffmpeg/libavcodec/arm/mpegvideo_neon.S',  # oct 10
                    'patched-ffmpeg/libavcodec/arm/rdft_neon.S',
                    'patched-ffmpeg/libavcodec/arm/simple_idct_neon.S',
                    'patched-ffmpeg/libavcodec/arm/vp3dsp_neon.S',
                  ],
                }],
              ],
            }],  # target_arch=="arm"
            ['target_arch=="arm" and (ffmpeg_branding=="Chrome" or ffmpeg_branding=="ChromeOS")', {
              'sources': [
                'patched-ffmpeg/libavcodec/arm/h264dsp_init_arm.c',
                'patched-ffmpeg/libavcodec/arm/h264pred_init_arm.c',
                'patched-ffmpeg/libavcodec/arm/mpegvideo_arm.c',
                'patched-ffmpeg/libavcodec/arm/mpegvideo_armv5te.c',
                'patched-ffmpeg/libavcodec/arm/mpegvideo_armv5te_s.S',
              ],
              'conditions': [
                ['arm_neon==1', {
                  'sources': [
                    'patched-ffmpeg/libavcodec/arm/h264dsp_neon.S',
                    'patched-ffmpeg/libavcodec/arm/h264idct_neon.S',
                    'patched-ffmpeg/libavcodec/arm/h264pred_neon.S',
                  ],
                }],
              ],
            }],
            ['target_arch=="arm" and ffmpeg_branding=="ChromeOS"', {
              'sources': [
                'patched-ffmpeg/libavcodec/h264_mp4toannexb_bsf.c',
               ],
            }],
            ['target_arch=="arm" and '
             '(ffmpeg_branding=="ChromeOS" or ffmpeg_branding=="ChromiumOS")', {
              'sources': [
                'patched-ffmpeg/libavcodec/arm/vp8_armv6.S',           # jul 11
                'patched-ffmpeg/libavcodec/arm/vp8dsp_init_arm.c',     # jul 11
               ],
               'conditions': [
                 ['arm_neon==1', {
                   'sources': [
                    'patched-ffmpeg/libavcodec/arm/vp8dsp_neon.S',     # jul 11
                  ],
                }],
              ],
            }],  # target_arch=="arm" and ffmpeg_branding=="C*OS" 
            ['OS=="linux" or OS=="freebsd" or OS=="openbsd" or OS=="solaris"', {
              'defines': [
                '_ISOC99_SOURCE',
                '_LARGEFILE_SOURCE',
                # BUG(ihf): ffmpeg compiles with this define. But according to
                # ajwong: I wouldn't change _FILE_OFFSET_BITS.  That's a scary change
                # cause it affects the default length of off_t, and fpos_t,
                # which can cause strange problems if the loading code doesn't
                # have it set and you start passing FILE*s or file descriptors
                # between symbol contexts.
                # '_FILE_OFFSET_BITS=64',
              ],
              'cflags': [
                '-std=c99',
                '-pthread',
                '-fno-math-errno',
                '-fno-signed-zeros',
                '-fno-tree-vectorize',
                '-fomit-frame-pointer',
                # Don't warn about libavformat using its own deprecated APIs.
                '-Wno-deprecated-declarations',
              ],
              'cflags!': [
                # Ensure the symbols are exported.
                #
                # TODO(ajwong): Fix common.gypi to only add this flag for
                # _type != shared_library.
                '-fvisibility=hidden',
              ],
              'conditions': [
                ['use_system_vpx==0 and target_arch!="arm"', {
                  'dependencies': [
                    '../libvpx/libvpx.gyp:libvpx',
                  ]
                }]
              ],
              'link_settings': {
                'ldflags': [
                  '-Wl,-Bsymbolic',
                  '-L<(shared_generated_dir)',
                ],
                'libraries': [
                  '-lz',
                ],
                'conditions': [
                  ['use_system_vpx==1 and target_arch!="arm"',
                   {
                     # Using libvpx provided by the system.
                     'libraries': [
                       '-lvpx',
                     ],
                   }],
                ],
              },
              'variables': {
                'obj_format': 'elf',
                'conditions': [
                  [ 'use_system_yasm==1', {
                    'yasm_path': '<!(which yasm)',
                  }, {
                    'yasm_path': '<(PRODUCT_DIR)/yasm',
                  }],
                  [ 'target_arch=="ia32"', {
                    'yasm_flags': [
                      '-DARCH_X86_32',
                      # The next two lines are needed to enable AVX assembly.
                      '-Iconfig/<(ffmpeg_branding)/<(os_config)/<(ffmpeg_config)',
                      '-Pconfig.asm',
                      '-m', 'x86',
                    ],
                  }, {
                    'yasm_flags': [
                      '-DARCH_X86_64',
                      '-m', 'amd64',
                      '-DPIC',
                      # The next two lines are needed to enable AVX assembly.
                      '-Iconfig/<(ffmpeg_branding)/<(os_config)/<(ffmpeg_config)',
                      '-Pconfig.asm',
                    ],
                  }],
                ],
              },
            }],  # OS=="linux" or OS=="freebsd" or OS=="openbsd" or OS=="solaris"
            ['OS=="mac"', {
              'conditions': [
                ['mac_breakpad==1', {
                  'variables': {
                    # A real .dSYM is needed for dump_syms to operate on.
                    'mac_real_dsym': 1,
                  },
                }],
              ],
              'dependencies': [
                '../libvpx/libvpx.gyp:libvpx',
              ],
              'link_settings': {
                'libraries': [
                  '$(SDKROOT)/usr/lib/libz.dylib',
                ],
              },
              'xcode_settings': {
                'GCC_SYMBOLS_PRIVATE_EXTERN': 'NO',  # No -fvisibility=hidden
                'GCC_DYNAMIC_NO_PIC': 'YES',         # -mdynamic-no-pic
                                                     # (equiv -fno-PIC)
                'DYLIB_INSTALL_NAME_BASE': '@loader_path',
                'LIBRARY_SEARCH_PATHS': [
                  '<(shared_generated_dir)'
                ],
                'OTHER_LDFLAGS': [
                  # This is needed because FFmpeg cannot be built as PIC, and
                  # thus we need to instruct the linker to allow relocations
                  # for read-only segments for this target to be able to
                  # generated the shared library on Mac.
                  #
                  # This makes Mark sad, but he's okay with it since it is
                  # isolated to this module. When Mark finds this in the
                  # future, and has forgotten this conversation, this comment
                  # should remind him that the world is still nice and
                  # butterflies still exist...as do rainbows, sunshine,
                  # tulips, etc., etc...but not kittens. Those went away
                  # with this flag.
                  '-Wl,-read_only_relocs,suppress',
                  '-L<(libvpx_hack_dir)/lib/mac/ia32'
                ],
              },
              'variables': {
                'obj_format': 'macho',
                'yasm_flags': [ '-DPREFIX' ],
                'conditions': [
                  [ 'use_system_yasm==1', {
                    'yasm_path': '<!(which yasm)',
                  }, {
                    'yasm_path': '<(PRODUCT_DIR)/yasm',
                  }],
                  [ 'target_arch=="ia32"', {
                    'yasm_flags': [
                      '-DARCH_X86_32',
                      # The next two lines are needed to enable AVX assembly.
                      '-Iconfig/<(ffmpeg_branding)/<(os_config)/<(ffmpeg_config)',
                      '-Pconfig.asm',
                      '-m', 'x86',
                    ],
                  }, {
                    'yasm_flags': [
                      '-DARCH_X86_64',
                      '-m', 'amd64',
                      '-DPIC',
                      # The next two lines are needed to enable AVX assembly.
                      '-Iconfig/<(ffmpeg_branding)/<(os_config)/<(ffmpeg_config)',
                      '-Pconfig.asm',
                    ],
                  }],
                ],
              },
            }],  # OS=="mac"
            ['use_system_vpx==0', {
                'include_dirs': [
                  # TODO(fischman): when both mac & linux are building from
                  # libvpx source replace this hackery with a simple target
                  # dependency on libvpx and let that export its headers.
                  '../libvpx/include',
                ],
              }, {
                # Using libvpx provided by the system.
                'include_dirs': [
                  '/usr/include/vpx',
                ],
              }
            ],
          ],
          'rules': [
            {
              'rule_name': 'assemble',
              'extension': 'asm',
              'inputs': [ '<(yasm_path)', ],
              'outputs': [
                '<(shared_generated_dir)/<(RULE_INPUT_ROOT).o',
              ],
              'action': [
                '<(yasm_path)',
                '-f', '<(obj_format)',
                '<@(yasm_flags)',
                '-I', 'patched-ffmpeg/libavcodec/x86/',
                '-o', '<(shared_generated_dir)/<(RULE_INPUT_ROOT).o',
                '<(RULE_INPUT_PATH)',
              ],
              'process_outputs_as_sources': 1,
              'message': 'Build ffmpeg yasm build <(RULE_INPUT_PATH).',
            },
          ],
        },
        {
          # A target shim that allows putting a dependency on ffmpegsumo
          # without pulling it into the link line.
          #
          # We use an "executable" taget without any sources to break the
          # link line relationship to ffmpegsumo.
          #
          # Most people will want to depend on this target instead of on
          # ffmpegsumo directly since ffmpegsumo is meant to be
          # used via dlopen() in chrome.
          'target_name': 'ffmpegsumo_nolink',
          'type': 'executable',
          'sources': [ 'dummy_nolink.cc' ],
          'dependencies': [
            'ffmpegsumo',
          ],
          'conditions': [
            ['OS=="linux" or OS=="freebsd" or OS=="openbsd" or OS=="solaris"', {
              'copies': [
                {
                  # On Make and Scons builds, the library does not end up in
                  # the PRODUCT_DIR.
                  #
                  # http://code.google.com/p/gyp/issues/detail?id=57
                  #
                  # TODO(ajwong): Fix gyp, fix the world.
                  'destination': '<(PRODUCT_DIR)',
                  'files': ['<(SHARED_LIB_DIR)/<(SHARED_LIB_PREFIX)ffmpegsumo<(SHARED_LIB_SUFFIX)'],
                },
              ],
            }],
          ],
        },
      ],
    }],
  ],  # conditions
  'targets': [
    {
      # Determine whether we should export libvpx symbols.
      'variables': {
        'generate_stubs_script': '../../tools/generate_stubs/generate_stubs.py',
        'extra_header': 'ffmpeg_stub_headers.fragment',
        'conditions': [
          ['target_arch=="arm" or use_system_ffmpeg==1 or use_system_vpx == 1', {
            'sig_files': [
              # Note that these must be listed in dependency order.
              # (i.e. if A depends on B, then B must be listed before A.)
              'avutil-51.sigs',
              'avcodec-53.novpx_sigs',
              'avformat-53.sigs',
            ],
          }, {  # otherwise we export libvpx symbols.
            'sig_files': [
              # Note that these must be listed in dependency order.
              # (i.e. if A depends on B, then B must be listed before A.)
              'avutil-51.sigs',
              'avcodec-53.sigs',
              'avformat-53.sigs',
            ],
          }],
        ],
      },

      'target_name': 'ffmpeg',
      'msvs_guid': 'D7A94F58-576A-45D9-A45F-EB87C63ABBB0',
      'sources': [
        # Hacks to introduce C99 types into Visual Studio.
        'include/win/inttypes.h',
        'include/win/stdint.h',

        # Files needed for stub generation rules.
        '<@(sig_files)',
        '<(extra_header)'
      ],
      'defines': [
        '__STDC_CONSTANT_MACROS',  # FFmpeg uses INT64_C.
      ],
      'hard_dependency': 1,

      # Do not fear the massive conditional blocks!  They do the following:
      #   1) Use the Window stub generator on Windows
      #   2) Else, use the POSIX stub generator on non-Windows
      #     a) Use system includes when use_system_ffmpeg!=0
      #     b) Else, use our local copy in patched-ffmpeg
      'conditions': [
        ['OS=="win"', {
          'variables': {
            'outfile_type': 'windows_lib',
            'output_dir': '<(PRODUCT_DIR)/lib',
            'intermediate_dir': '<(INTERMEDIATE_DIR)',
            # TODO(scherkus): Change Windows DEPS directory so we don't need
            # this conditional.
            'conditions': [
              [ 'ffmpeg_branding=="Chrome"', {
                'ffmpeg_bin_dir': 'chrome/<(OS)/<(ffmpeg_variant)',
              }, {  # else ffmpeg_branding!="Chrome", assume chromium.
                'ffmpeg_bin_dir': 'chromium/<(OS)/<(ffmpeg_variant)',
              }],
            ],
          },
          'type': 'none',
          'sources!': [
            '<(extra_header)',
          ],
          'direct_dependent_settings': {
            'include_dirs': [
              'include/win',
              'config',
              'patched-ffmpeg',
            ],
            'link_settings': {
              'libraries': [
                '<(output_dir)/avcodec-53.lib',
                '<(output_dir)/avformat-53.lib',
                '<(output_dir)/avutil-51.lib',
              ],
              'msvs_settings': {
                'VCLinkerTool': {
                  'DelayLoadDLLs': [
                    'avcodec-53.dll',
                    'avformat-53.dll',
                    'avutil-51.dll',
                  ],
                },
              },
            },
          },
          'rules': [
            {
              # TODO(hclam): Remove this when the novpx hack is unneeded.
              'rule_name': 'generate_libs_novpx',
              'extension': 'novpx_sigs',
              'inputs': [
                '<(generate_stubs_script)',
                '<@(sig_files)',
              ],
              'outputs': [
                '<(output_dir)/<(RULE_INPUT_ROOT).lib',
              ],
              'action': ['python', '<(generate_stubs_script)',
                         '-i', '<(intermediate_dir)',
                         '-o', '<(output_dir)',
                         '-t', '<(outfile_type)',
                         '<@(RULE_INPUT_PATH)',
              ],
              'message': 'Generating FFmpeg import libraries.',
            },
            {
              'rule_name': 'generate_libs',
              'extension': 'sigs',
              'inputs': [
                '<(generate_stubs_script)',
                '<@(sig_files)',
              ],
              'outputs': [
                '<(output_dir)/<(RULE_INPUT_ROOT).lib',
              ],
              'action': ['python', '<(generate_stubs_script)',
                         '-i', '<(intermediate_dir)',
                         '-o', '<(output_dir)',
                         '-t', '<(outfile_type)',
                         '<@(RULE_INPUT_PATH)',
              ],
              'message': 'Generating FFmpeg import libraries.',
            },
          ],

          # Copy prebuilt binaries to build directory.
          'dependencies': ['../../build/win/system.gyp:cygwin'],
          'copies': [{
            'destination': '<(PRODUCT_DIR)/',
            'files': [
              'binaries/<(ffmpeg_bin_dir)/avcodec-53.dll',
              'binaries/<(ffmpeg_bin_dir)/avformat-53.dll',
              'binaries/<(ffmpeg_bin_dir)/avutil-51.dll',
            ],
          }],
        }, {  # else OS!="win", use POSIX stub generator
          'variables': {
            'outfile_type': 'posix_stubs',
            'stubs_filename_root': 'ffmpeg_stubs',
            'project_path': 'third_party/ffmpeg',
            'intermediate_dir': '<(INTERMEDIATE_DIR)',
            'output_root': '<(SHARED_INTERMEDIATE_DIR)/ffmpeg',
          },
          'type': '<(library)',
          'include_dirs': [
            '<(output_root)',
            '../..',  # The chromium 'src' directory.
          ],
          'direct_dependent_settings': {
            'defines': [
              '__STDC_CONSTANT_MACROS',  # FFmpeg uses INT64_C.
            ],
            'include_dirs': [
              '<(output_root)',
              '../..',  # The chromium 'src' directory.
            ],
          },
          'actions': [
            {
              'action_name': 'generate_stubs',
              'inputs': [
                '<(generate_stubs_script)',
                '<(extra_header)',
                '<@(sig_files)',
              ],
              'outputs': [
                '<(intermediate_dir)/<(stubs_filename_root).cc',
                '<(output_root)/<(project_path)/<(stubs_filename_root).h',
              ],
              'action': ['python',
                         '<(generate_stubs_script)',
                         '-i', '<(intermediate_dir)',
                         '-o', '<(output_root)/<(project_path)',
                         '-t', '<(outfile_type)',
                         '-e', '<(extra_header)',
                         '-s', '<(stubs_filename_root)',
                         '-p', '<(project_path)',
                         '<@(_inputs)',
              ],
              'process_outputs_as_sources': 1,
              'message': 'Generating FFmpeg stubs for dynamic loading.',
            },
          ],

          'conditions': [
            # Linux/Solaris need libdl for dlopen() and friends.
            ['OS=="linux" or OS=="solaris"', {
              'link_settings': {
                'libraries': [
                  '-ldl',
                ],
              },
            }],

            ['use_system_vpx==0', {
              'include_dirs': [
                # TODO(fischman): when both mac & linux are building from
                # libvpx source replace this hackery with a simple target
                # dependency on libvpx and let that export its headers.
                '../libvpx/include',
              ],
            }, { # use_system_vpx != 0
              'defines': [
                'USE_SYSTEM_VPX',
              ],
              'direct_dependent_settings': {
                'defines': [
                  'USE_SYSTEM_VPX',
                ],
                'link_settings': {
                  'libraries': [
                    '-lvpx',
                  ],
                },
              },
            }],

            # Add pkg-config result to include path when use_system_ffmpeg!=0
            ['use_system_ffmpeg!=0', {
              'cflags': [
                '<!@(pkg-config --cflags libavcodec libavformat libavutil)',
              ],
              'direct_dependent_settings': {
                'cflags': [
                  '<!@(pkg-config --cflags libavcodec libavformat libavutil)',
                ],
              },
            }, {  # else use_system_ffmpeg==0, add local copy to include path
              'include_dirs': [
                'config',
                'patched-ffmpeg',
              ],
              'direct_dependent_settings': {
                'include_dirs': [
                  'config',
                  'patched-ffmpeg',
                ],
              },
              'conditions': [
                ['build_ffmpegsumo!=0', {
                  'dependencies': [
                    'ffmpegsumo_nolink',
                  ],
                }],
              ],
            }],
          ],  # conditions
        }],
      ],  # conditions
    },
  ],  # targets
}

# Local Variables:
# tab-width:2
# indent-tabs-mode:nil
# End:
# vim: set expandtab tabstop=2 shiftwidth=2:
