/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#include <gtk/gtk.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <arpa/inet.h>
#include <string.h>
#include <netdb.h>
#include <sys/stat.h>
#include <errno.h>
#include <pwd.h>
#include "cheops.h"

int option_automap=1;
int option_remember_positions=1;
int option_port_scan = 1;
int option_trim = 1;
int option_queso = 1;
int option_reverse_dns = 1;
int option_add_routes=0;
int option_use_icmp=0;
int option_use_visible_area=1;
int option_update_release=1;
int option_remember_hosts=0;
int option_remember_connects=0;
int option_remember_aliases=0;
int option_need_logfile=1;
int option_send_email=0;
int option_immediate_retry=0;
int option_goto_failure=1;
int option_verbose_display=0;
int connection_timeout=30;

#ifdef SNMP
int option_snmp=1;
int option_snmp_verbose=0;
GtkWidget *usesnmp;
GtkWidget *snmpverbose;
#endif

char xterm[256]="xterm";
char logfile[256]="";
char email[256]="";

static GtkWidget *options=NULL;
static GtkWidget *verbose;
static GtkWidget *automap;
static GtkWidget *remember;
static GtkWidget *portscan;
static GtkWidget *trim;
static GtkWidget *queso;
static GtkWidget *revdns;
static GtkWidget *addroutes;
static GtkWidget *useicmp;
static GtkWidget *visible;
static GtkWidget *release;
static GtkWidget *savehosts;
static GtkWidget *saveconnects;
static GtkWidget *savealiases;
static GtkWidget *xt=NULL;
static GtkWidget *needlogfile;
static GtkWidget *logfilew=NULL;
static GtkWidget *emailw=NULL;
static GtkWidget *sendemail;
static GtkWidget *clearw;
static GtkWidget *iretry;
static GtkWidget *connectt=NULL;
static GtkWidget *gotofailure;

static GtkWidget *clist;

static int modified_services=0;

void generic_set_status(GtkWidget *bar, char *s)
{
	char *tmp;
	gtk_label_get(GTK_LABEL(GTK_STATUSBAR(bar)->label), &tmp);
	if (strcmp(tmp, s)) {
		gtk_statusbar_pop(GTK_STATUSBAR(bar), 1);
		gtk_statusbar_push(GTK_STATUSBAR(bar), 1, s);
	}
	while(gtk_events_pending())
		gtk_main_iteration();
}

static void cancel_options()
{
	char *c;
	if (!options) 
		return;
	c = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(xt)->entry));
	strncpy(xterm, c, sizeof(xterm));
	c = gtk_entry_get_text(GTK_ENTRY(logfilew));
	if (strcmp(c, logfile) || !option_need_logfile) {
		close_logfile();
		strncpy(logfile, c, sizeof(logfile));
	}
	c = gtk_entry_get_text(GTK_ENTRY(emailw));
	if (c && strlen(c))
		strncpy(email,c, sizeof(email));
	connection_timeout = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(connectt));
	gtk_widget_destroy(options);
	connectt = NULL;
	options=NULL;
	logfilew=NULL;
	emailw=NULL;
	xt=NULL;
	connectt=NULL;
	show_hide_verbose(0);
}

void make_logfile()
{
	char *c;
	if ((c = getenv("HOME"))) {
		snprintf(logfile, sizeof(logfile), "%s/.cheops/eventlog",c);
	}
}

void make_email()
{
	struct passwd *pw;
	if ((pw = getpwuid(getuid()))) {
		snprintf(email, sizeof(email), "%s",pw->pw_name);
	}
}

struct add_modify {
	struct service *svc;
	GtkWidget *window;
	GtkWidget *name;
	GtkWidget *port;
	GtkWidget *format;
	GtkWidget *type;
	GtkWidget *status;
};

static void do_cancel(GtkWidget *w)
{
	struct add_modify *am = (struct add_modify *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	gtk_widget_destroy(am->window);
	g_free(am);
}

static void build_clist_svcs(GtkWidget *w)
{
	char *args[4];
	char tmp[20];
	int res;
	struct service *s;
	s = users;
	gtk_clist_freeze(GTK_CLIST(w));
	gtk_clist_clear(GTK_CLIST(w));
	while(s) {
		g_snprintf(tmp, sizeof(tmp), "%d", s->port);
		args[0]=s->name;
		args[1]=tmp;
		args[2]=s->format;
		args[3]=str_plugin(s->connect, TYPE_SERVICE, SERVICE_CONNECT);
		res = gtk_clist_append(GTK_CLIST(w), args);
		gtk_clist_set_row_data(GTK_CLIST(w), res, s);
		s=s->next;
	}
	gtk_clist_thaw(GTK_CLIST(w));
}

static void do_ok(GtkWidget *w)
{
	struct add_modify *am = (struct add_modify *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	char *name;
	char *port;
	char *format;
	char *type;
	void *callback;
	int flags=0;
	name = gtk_entry_get_text(GTK_ENTRY(am->name));
	port = gtk_entry_get_text(GTK_ENTRY(am->port));
	format = gtk_entry_get_text(GTK_ENTRY(am->format));
	type = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(am->type)->entry));
	if (!strlen(name) || !strlen(port) || !strlen(format) || !strlen(type)) {
		generic_set_status(am->status, "Name, port, format, and plugin must be specified");
		return;
	}
	if (am->svc) 
		unregister_service(am->svc);
	if (strstr(format,"%u"))
		flags |= FLAG_USERNAME;
	
	callback = get_plugin(type, TYPE_SERVICE, SERVICE_CONNECT);
	if (!callback) {
		generic_set_status(am->status, "No such plugin");
		return;
	}
	register_service(name, atoi(port), format, 
			callback,
			flags);
	build_clist_svcs(clist);	
	gtk_widget_destroy(am->window);
	modified_services=1;
	g_free(am);
}

static void make_add_modify(struct service *svc)
{
	struct add_modify *am;
	GtkWidget *label;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *ok;
	GtkWidget *cancel;
	vbox = gtk_vbox_new(FALSE, 5);
	am = g_new0(struct add_modify, 1);
	am->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	am->svc = svc;
	am->status = gtk_statusbar_new();
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	label = gtk_label_new("Name");
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	am->name = gtk_entry_new();
	if (svc)
		gtk_entry_set_text(GTK_ENTRY(am->name), svc->name);
	gtk_widget_show(am->name);
	gtk_box_pack_end(GTK_BOX(hbox), am->name, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	label = gtk_label_new("Port");
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	am->port = gtk_entry_new();
	if (svc) {
		char ports[20];
		g_snprintf(ports, sizeof(ports), "%d", svc->port);
		gtk_entry_set_text(GTK_ENTRY(am->port), ports);
	}
	gtk_widget_show(am->port);
	gtk_box_pack_end(GTK_BOX(hbox), am->port, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), am->status, FALSE, FALSE, 5);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	label = gtk_label_new("Command\n(%x=xterm\n%u=username\n%i=ip\n%h=hostname\n%p=port)");
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	am->format = gtk_entry_new();
	if (svc)
		gtk_entry_set_text(GTK_ENTRY(am->format), svc->format);
	gtk_widget_show(am->format);
	gtk_box_pack_end(GTK_BOX(hbox), am->format, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	label = gtk_label_new("Plugin");
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	am->type = gtk_combo_new();
	gtk_combo_set_popdown_strings(GTK_COMBO(am->type), list_plugin(TYPE_SERVICE));
	if (svc)
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(am->type)->entry), 
		str_plugin(svc->connect, TYPE_SERVICE, SERVICE_CONNECT));
	gtk_widget_show(am->type);
	gtk_box_pack_end(GTK_BOX(hbox), am->type, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	ok = gtk_button_new_with_label("   OK   ");
	cancel = gtk_button_new_with_label("Cancel");
	
	gtk_object_set_user_data(GTK_OBJECT(ok), am);
	gtk_object_set_user_data(GTK_OBJECT(cancel), am);
	gtk_object_set_user_data(GTK_OBJECT(am->window), am);
	
	gtk_signal_connect(GTK_OBJECT(cancel), "clicked", GTK_SIGNAL_FUNC(do_cancel), NULL);
	gtk_signal_connect(GTK_OBJECT(am->window), "delete_event", GTK_SIGNAL_FUNC(do_cancel), NULL);
	gtk_signal_connect(GTK_OBJECT(ok), "clicked", GTK_SIGNAL_FUNC(do_ok), NULL);
	
	gtk_widget_show(ok);
	gtk_widget_show(cancel);
	gtk_box_pack_end(GTK_BOX(hbox), cancel, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(hbox), ok, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);

	gtk_widget_show(vbox);	
	gtk_container_add(GTK_CONTAINER(am->window), vbox);
	gtk_container_border_width(GTK_CONTAINER(am->window), 10);
	gtk_window_set_title(GTK_WINDOW(am->window), svc ? "Edit Service" : "Add Service");
	gtk_widget_realize(am->window);
	fix_icon(am->window->window);
	gtk_widget_show(am->window);
}

static void do_add()
{
	make_add_modify(NULL);
}

static struct service *get_selection()
{
	GList *l;
	int r;
	l = g_list_first(GTK_CLIST(clist)->selection);
	if (!l)
		return NULL;
	r = GPOINTER_TO_INT(l->data);
	if (r < 0)
		return NULL;
	return (struct service *)gtk_clist_get_row_data(GTK_CLIST(clist), r);
}

static void do_edit()
{
	struct service *svc;
	if ((svc = get_selection()))
		make_add_modify(svc);
}

static void do_delete()
{
	struct service *svc;
	if ((svc = get_selection())) {
		unregister_service(svc);
		build_clist_svcs(clist);
		modified_services=1;
	}
}


static void change_int(GtkWidget *w, int *data)
{
	*data = !*data;
}

#define PREFS_LINK(a,b) { \
	gtk_widget_show(a); \
	if (b) \
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(a), TRUE); \
	else \
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(a), FALSE); \
	gtk_signal_connect(GTK_OBJECT(a), "clicked", GTK_SIGNAL_FUNC(change_int), &b); \
	gtk_box_pack_start(GTK_BOX(w), a, FALSE, FALSE, 0); \
} 

#define PREFS_LINK2(a,b,c) { \
	gtk_widget_show(a); \
	if (b) \
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(a), TRUE); \
	else \
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(a), FALSE); \
	gtk_signal_connect(GTK_OBJECT(a), "clicked", GTK_SIGNAL_FUNC(c), &b); \
} 

#define CHECK_OPT(a,b) { \
	if (!strcasecmp(opt,a)) \
		b = atoi(val); \
}
	

int load_options()
{
	char buf[256];
	char *h;
	char *opt, *val;
	FILE *f;
	if (!(h = getenv("HOME")))
		return -1;
	check_old_config(".cheops-options",".cheops/options");
	g_snprintf(buf, sizeof(buf), "%s/.cheops/options", getenv("HOME"));
	if (!(f=fopen(buf, "r")))
		return -1;
	while(!feof(f)) {
		fgets(buf, sizeof(buf), f);
		if (!feof(f)) {
			opt = strchr(buf, '[');
			if (!opt)
				continue;
			opt++;
			val = strchr(opt, ' ');
			if (!val)
				continue;
			*val='\0';
			val++;
			h = strchr(val,']');
			if (!h)
				continue;
			*h='\0';
			if (!strcasecmp(opt, "xterm"))
				strncpy(xterm, val, sizeof(xterm));
			if (!strcasecmp(opt, "logfile"))
				strncpy(logfile, val, sizeof(logfile));
			if (!strcasecmp(opt, "email"))
				strncpy(email, val, sizeof(logfile));
			CHECK_OPT("automap",option_automap);
			CHECK_OPT("remember", option_remember_positions);
			CHECK_OPT("portscan", option_port_scan);
			CHECK_OPT("trim",option_trim);
			CHECK_OPT("queso", option_queso);
			CHECK_OPT("revdns", option_reverse_dns);
			CHECK_OPT("addroutes", option_add_routes); 			
			CHECK_OPT("useicmp", option_use_icmp);
			CHECK_OPT("release", option_update_release);
			CHECK_OPT("savehosts", option_remember_hosts);
			CHECK_OPT("saveconnects", option_remember_connects);
			CHECK_OPT("savealiases", option_remember_aliases);
			CHECK_OPT("needlogfile", option_need_logfile);
			CHECK_OPT("sendemail", option_send_email);
			CHECK_OPT("ctimeout", connection_timeout);
			CHECK_OPT("iretry", option_immediate_retry);
			CHECK_OPT("verbose", option_verbose_display);
			CHECK_OPT("gotofailure", option_goto_failure);
#ifdef SNMP
			CHECK_OPT("usesnmp", option_snmp);
			CHECK_OPT("snmpverbose", option_snmp_verbose);
#endif
		}
	}
	fclose(f);
	return 0;
}

int save_options()
{
	char buf[256];
	char *h;
	FILE *f;
	char *c;
	if (xt) {
		c = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(xt)->entry));
		strncpy(xterm, c, sizeof(xterm));
	}
	if (logfilew) {
		c = gtk_entry_get_text(GTK_ENTRY(logfilew));
		if (strcmp(c, logfile) || !option_need_logfile) {
			close_logfile();
			strncpy(logfile, c, sizeof(logfile));
		}
	}
	if (emailw) {
		c = gtk_entry_get_text(GTK_ENTRY(emailw));
		if (c && strlen(c))
			strncpy(email,c, sizeof(email));
	}
	if (connectt) 
		connection_timeout = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(connectt));
	
	if (!(h = getenv("HOME")))
		return -1;
	g_snprintf(buf, sizeof(buf), "%s/.cheops/options", getenv("HOME"));
	if (!(f=fopen(buf, "w")))
		return -1;
	fprintf(f,"[automap %d]\n", option_automap);
	fprintf(f,"[remember %d]\n", option_remember_positions);
	fprintf(f,"[portscan %d]\n", option_port_scan);
	fprintf(f,"[trim %d]\n", option_trim);
	fprintf(f,"[queso %d]\n", option_queso);
	fprintf(f,"[revdns %d]\n",option_reverse_dns);
	fprintf(f,"[addroutes %d]\n",option_add_routes);
	fprintf(f,"[useicmp %d]\n", option_use_icmp);
	fprintf(f,"[visible %d]\n", option_use_visible_area);
	fprintf(f,"[release %d]\n", option_update_release);
	fprintf(f,"[xterm %s]\n", xterm);
	fprintf(f,"[savehosts %d]\n", option_remember_hosts);
	fprintf(f,"[saveconnects %d]\n", option_remember_connects);
	fprintf(f,"[savealiases %d]\n", option_remember_aliases);
	fprintf(f,"[needlogfile %d]\n", option_need_logfile);
	fprintf(f,"[sendemail %d]\n", option_send_email);
	fprintf(f,"[ctimeout %d]\n", connection_timeout);
	fprintf(f,"[iretry %d]\n", option_immediate_retry);
	fprintf(f,"[gotofailure %d]\n", option_goto_failure);
	fprintf(f,"[verbose %d]\n", option_verbose_display);
#ifdef SNMP
	fprintf(f,"[usesnmp %d]\n", option_snmp);
	fprintf(f,"[snmpverbose %d]\n", option_snmp_verbose);
#endif
	if (!strlen(logfile))
		make_logfile();
	fprintf(f,"[logfile %s]\n", logfile);
	if (!strlen(email))
		make_email();
	fprintf(f,"[email %s]\n", email);
	fclose(f);
	if (options) {
		gtk_widget_destroy(options);
		options=NULL;
		logfilew=NULL;
		xt=NULL;
		emailw=NULL;
		connectt=NULL;
	}
	if (modified_services)
		save_services();
	modified_services = 0;
	show_hide_verbose(0);
	return 0;
}


static void make_prefs(GtkWidget *w)
{
	verbose = gtk_check_button_new_with_label("Show more information in main window");
	automap = gtk_check_button_new_with_label("Automatically map all hosts I add to a mapped viewspace");
	trim = gtk_check_button_new_with_label("Trim hostnames when displaying them");
	queso = gtk_check_button_new_with_label("Determine the operating system of new hosts (queso)");
	portscan = gtk_check_button_new_with_label("Determine what services a host supports (half-scan)");
	revdns = gtk_check_button_new_with_label("Lookup hosts I add with reverse DNS");	
	addroutes = gtk_check_button_new_with_label("When I map a host, add all hosts on the map to the viewspace");
	useicmp = gtk_check_button_new_with_label("Use ICMP instead of UDP for mapping");
	visible = gtk_check_button_new_with_label("Fill width in the area I see only");
	release = gtk_check_button_new_with_label("Only update the map drawing when I stop moving an object");

	PREFS_LINK(verbose, option_verbose_display);
	PREFS_LINK(automap, option_automap);
	PREFS_LINK(portscan, option_port_scan);
	PREFS_LINK(trim, option_trim);
	PREFS_LINK(queso, option_queso);
	PREFS_LINK(revdns, option_reverse_dns);
	PREFS_LINK(addroutes, option_add_routes);
	PREFS_LINK(useicmp, option_use_icmp);
	PREFS_LINK(visible, option_use_visible_area);
	PREFS_LINK(release, option_update_release);
}

static void make_memory(GtkWidget *w)
{
	remember = gtk_check_button_new_with_label("Remember where I put all the hosts that I've moved");
	savehosts = gtk_check_button_new_with_label("Remember all the hosts I've discovered");
	saveconnects = gtk_check_button_new_with_label("Remember all the connections between hosts");
	savealiases = gtk_check_button_new_with_label("Remember all aliases for a host");
	PREFS_LINK(remember, option_remember_positions);
	PREFS_LINK(savehosts, option_remember_hosts);
	PREFS_LINK(saveconnects, option_remember_connects);
	PREFS_LINK(savealiases, option_remember_aliases);
}

#ifdef SNMP
static void make_snmp(GtkWidget *w)
{
	usesnmp = gtk_check_button_new_with_label("Use SNMP when discovering hosts");
	snmpverbose = gtk_check_button_new_with_label("Display SNMP errors");
	PREFS_LINK(usesnmp, option_snmp);
	PREFS_LINK(snmpverbose, option_snmp_verbose);
}
#endif

static void make_mons(GtkWidget *w)
{
	GtkWidget *label;
	GtkWidget *hbox;
	GtkObject *adj;
	adj = gtk_adjustment_new((double)connection_timeout, 1.0, 50.0, 1.0, 1.0, 1.0);
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	label = gtk_label_new("Give a service this number of seconds to connect:");
	gtk_widget_show(label);
	connectt = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1.0, 0);
	gtk_widget_show(connectt);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), connectt, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(w), hbox, FALSE, FALSE, 5);
	iretry = gtk_check_button_new_with_label("Immediately retry services that time out");
	gotofailure = gtk_check_button_new_with_label("Switch to viewspace with most recent failure");
	PREFS_LINK(iretry, option_immediate_retry);
	PREFS_LINK(gotofailure, option_goto_failure);
}


static void make_helpers(GtkWidget *w)
{
	GtkWidget *label;
	GList *strs;
	strs = g_list_append(NULL, "xterm");
	strs = g_list_append(strs, "kvt");
	strs = g_list_append(strs, "Eterm");
	strs = g_list_append(strs, "rxvt");
	xt = gtk_combo_new();
	gtk_combo_set_popdown_strings(GTK_COMBO(xt), strs);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(xt)->entry), xterm);
	gtk_entry_select_region(GTK_ENTRY(GTK_COMBO(xt)->entry), 0, -1);
	gtk_widget_show(xt);
	label = gtk_label_new("Terminal Program");
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(w), label, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(w), xt, FALSE, FALSE, 5);
}

static void check_need_logfile(GtkWidget *w, int *data)
{
	int tmp;
	if (data)
		*data = !*data;
	tmp = GTK_TOGGLE_BUTTON(needlogfile)->active;
	gtk_widget_set_sensitive(logfilew, tmp);
	gtk_widget_set_sensitive(clearw, tmp);
		
}

static void check_send_email(GtkWidget *w, int *data)
{
	int tmp;
	if (data)
		*data = !*data;
	tmp = GTK_TOGGLE_BUTTON(sendemail)->active;
	gtk_widget_set_sensitive(emailw, tmp);
		
}

static void make_logs(GtkWidget *w)
{
	GtkWidget *hbox;
	GtkWidget *bbox;
	needlogfile = gtk_check_button_new_with_label("Logfile: ");
	sendemail = gtk_check_button_new_with_label("Send e-mail to: ");
	PREFS_LINK2(needlogfile, option_need_logfile, check_need_logfile);
	PREFS_LINK2(sendemail, option_send_email, check_send_email);
	if (!strlen(logfile))
		make_logfile();
	if (!strlen(email))
		make_email();
	logfilew = gtk_entry_new();
	gtk_widget_show(logfilew);
	emailw = gtk_entry_new();
	gtk_widget_show(emailw);

	bbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(bbox);
	clearw = gtk_button_new_with_label("Clear Log");
	gtk_signal_connect(GTK_OBJECT(clearw), "clicked", GTK_SIGNAL_FUNC(clear_logfile), NULL);
	gtk_widget_show(clearw);
	gtk_box_pack_end(GTK_BOX(bbox), clearw, FALSE, FALSE, 5);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(hbox), sendemail, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), emailw, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(w), hbox, FALSE, FALSE, 5);
	gtk_entry_set_text(GTK_ENTRY(emailw), email);
	gtk_entry_select_region(GTK_ENTRY(emailw), 0, -1);

	
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(hbox), needlogfile, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), logfilew, TRUE, TRUE, 5);
	gtk_entry_set_text(GTK_ENTRY(logfilew), logfile);
	gtk_entry_select_region(GTK_ENTRY(logfilew), 0, -1);
	gtk_box_pack_start(GTK_BOX(w), hbox, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(w), bbox, FALSE, FALSE, 5);
	check_need_logfile(NULL, NULL);
	check_send_email(NULL, NULL);
}

void make_services(GtkWidget *w)
{
	GtkWidget *hbox;
	GtkWidget *add;
	GtkWidget *edit;
	GtkWidget *delete;
#if (GTK_MINOR_VERSION > 2) || ((GTK_MICRO_VERSION > 4)	&& (GTK_MINOR_VERSION > 0))
	GtkWidget *sw;
#endif
	static char *titles[] = { "Name", "Port", "Command", "Type" };
	clist = gtk_clist_new_with_titles(4, titles);
#if ((GTK_MINOR_VERSION > 0) && (GTK_MICRO_VERSION > 2))  || (GTK_MAJOR_VERSION > 1)
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 2, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 3, TRUE);
#else
	gtk_clist_set_column_width(GTK_CLIST(clist), 0, 80);
	gtk_clist_set_column_width(GTK_CLIST(clist), 1, 30);
	gtk_clist_set_column_width(GTK_CLIST(clist), 2, 400);
	gtk_clist_set_column_width(GTK_CLIST(clist), 3, 40);
#endif
	build_clist_svcs(clist);
	gtk_widget_show(clist);
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	add = gtk_button_new_with_label("  Add...  ");
	edit = gtk_button_new_with_label("  Edit...  ");
	delete = gtk_button_new_with_label("  Delete  ");
	
	gtk_signal_connect(GTK_OBJECT(add), "clicked", GTK_SIGNAL_FUNC(do_add), NULL);
	gtk_signal_connect(GTK_OBJECT(edit), "clicked", GTK_SIGNAL_FUNC(do_edit), NULL);
	gtk_signal_connect(GTK_OBJECT(delete), "clicked", GTK_SIGNAL_FUNC(do_delete), NULL);
	
	gtk_widget_show(add);
	gtk_widget_show(edit);
	gtk_widget_show(delete);
	gtk_box_pack_end(GTK_BOX(hbox), delete, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(hbox), edit, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(hbox), add, FALSE, FALSE, 5);
#if (GTK_MINOR_VERSION > 2) || ((GTK_MICRO_VERSION > 4)	&& (GTK_MINOR_VERSION > 0))
	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(sw);
	gtk_container_add(GTK_CONTAINER(sw), clist);
	gtk_box_pack_start(GTK_BOX(w), sw, TRUE, TRUE, 5);
#else
	gtk_box_pack_start(GTK_BOX(w), clist, TRUE, TRUE, 5);
#endif
	gtk_box_pack_start(GTK_BOX(w), hbox, FALSE, FALSE, 5);	
	
}

void show_options()
{
	GtkWidget *bbox;
	GtkWidget *vbox;
	GtkWidget *features;
	GtkWidget *memory;
	GtkWidget *helpers;
	GtkWidget *services;
#ifdef SNMP
	GtkWidget *snmp;
#endif
	GtkWidget *notebook;
	GtkWidget *label;
	GtkWidget *sep;
	GtkWidget *close;
	GtkWidget *save;
	GtkWidget *logs;
	GtkWidget *mons;
	
	if (options) {
		gtk_widget_show(options);
		return;
	}
	/* Misc */
	sep = gtk_hseparator_new();
	gtk_widget_show(sep);

	/* Window and containers */
	options = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(options), "Cheops Options");

	notebook = gtk_notebook_new();
	gtk_widget_show(notebook);
	
	features = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(features);

	memory = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(memory);
	
	helpers = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(helpers);
	
	services = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(services);
	
#ifdef SNMP	
	snmp = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(snmp);
#endif
	mons = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(mons);

	logs = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(logs);

	make_prefs(features);
	
	make_memory(memory);
	
	make_helpers(helpers);
	
	make_services(services);

#ifdef SNMP	
	make_snmp(snmp);
#endif
	make_mons(mons);
	
	make_logs(logs);
	
	label = gtk_label_new("Features");
	gtk_widget_show(label);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), features, label);
	
	label = gtk_label_new("Memory");
	gtk_widget_show(label);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), memory, label);
	
	label = gtk_label_new("Helpers");
	gtk_widget_show(label);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), helpers, label);

	label = gtk_label_new("Services");
	gtk_widget_show(label);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), services, label);

#ifdef SNMP
	label = gtk_label_new("SNMP");
	gtk_widget_show(label);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), snmp, label);
#endif
	label = gtk_label_new("Monitoring");
	gtk_widget_show(label);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), mons, label);

	label = gtk_label_new("Event Log");
	gtk_widget_show(label);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), logs, label);

	vbox = gtk_vbox_new(FALSE, 5);
	bbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(bbox);
	gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 5); 
	gtk_box_pack_start(GTK_BOX(vbox), sep, FALSE, FALSE, 5); 
	gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 5); 
	gtk_widget_show(vbox);
	

	/* Buttons */
	close = gtk_button_new_with_label("  Close  ");
	save = gtk_button_new_with_label("  Save  ");
	gtk_widget_show(close);
	gtk_widget_show(save);
	
	
	gtk_box_pack_end(GTK_BOX(bbox),close, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(bbox),save, FALSE, FALSE, 5);
	
	/* Signal handlers */
	gtk_signal_connect(GTK_OBJECT(save), "clicked", GTK_SIGNAL_FUNC(save_options), NULL);
	gtk_signal_connect(GTK_OBJECT(close), "clicked", GTK_SIGNAL_FUNC(cancel_options), NULL);
	gtk_signal_connect(GTK_OBJECT(options), "delete_event", GTK_SIGNAL_FUNC(cancel_options), NULL);
	
	gtk_container_add(GTK_CONTAINER(options), vbox);
	gtk_container_border_width(GTK_CONTAINER(options), 10);
	gtk_widget_grab_focus(close); 
	
	gtk_widget_show(options);	
	
}

int file_exists(char *s)
{
	struct stat buf;
	if (stat(s, &buf) < 0)
		return 0;
	if (S_ISREG(buf.st_mode)) {
		return 1;
	}
	return 0;
	
}

char *find_file(char *s)
{
	static char *paths[] = {
#ifdef DEFAULT_PATH
		DEFAULT_PATH,
#endif
		"/usr/share/cheops",
		"/usr/local/share/cheops",
		"/usr/lib/cheops",
		"/usr/local/lib/cheops",
		"/etc/cheops",
		"/usr/local/etc/cheops",
		"pixmaps",
		".",
		NULL
	};
	static char buf[256];
	static char *e = NULL;
	int x;

        if (s[0]=='/' && file_exists(s))
                return s;
	if (!e) 
		e = getenv("CHEOPS_HOME");
	if (e) {
		g_snprintf(buf, sizeof(buf), "%s/%s", e, s);
		if (file_exists(buf))
			return buf;
	}
	for (x=0; paths[x]; x++) {
		g_snprintf(buf, sizeof(buf), "%s/%s", paths[x], s);
		if (file_exists(buf))
			return buf;
	}
	fprintf(stderr, "Critical: Unable to find file '%s'...  Expect a crash soon!\n",s);
	return NULL;
}

void check_old_config(char *old, char *new)
{
	/* This procedure is to get people from older cheops's
	   with individual files to the new cheops with a .cheops
	   directory */
	char buf[256];
	char buf2[256];
	char *c;
	
	if (!(c=getenv("HOME")))
		return;
	snprintf(buf, sizeof(buf), "%s/.cheops", c);
	mkdir(buf, 0700);
	snprintf(buf, sizeof(buf), "%s/.cheops/plugins", c);
	mkdir(buf, 0700);
	snprintf(buf, sizeof(buf), "%s/%s", c, old);
	if (file_exists(buf)) {
		snprintf(buf2, sizeof(buf2), "%s/%s", c, new);
		fprintf(stderr, "Warning: moving file from old location %s to new location %s\n",
				 buf,buf2);
		if (rename(buf, buf2)) 
			fprintf(stderr, "Move failed: %s\n", g_strerror(errno));
	}
}
