/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _GNU_SOURCE /* strndup */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

/* open() */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "cdw_config.h"
#include "cdw_widgets.h"
#include "cdw_file_picker.h"
#include "cdw_string.h"
#include "cdw_processwin.h"
#include "cdw_read_disc.h"
#include "cdw_debug.h"
#include "gettext.h"
#include "cdw_logging.h"
#include "cdw_main_window.h"
#include "cdw_read_disc_info.h"

/* variable holding cdw configuration */
extern cdw_config_t global_config;

/**
   \brief Maximum lenght of name of file created when copying tracks to hard disc

   30 chars for core of output file name, track number and file name
   extension not included; 30 is an arbitrary size, but it seems to be ok;
   the value does not include ending '\0' char */
#define CDW_READ_DICS_TRACK_CORE_NAME_LEN 30



static struct strings {
	const char *log_string;
	const char *window_title;
	const char *window_message;
} strings_table[] = {
	{ /* 2TRANS: this is message printed to log file: a header for any further messages related to ripping audio CD */
		gettext_noop("Copying audio tracks:\n"),
		/* 2TRANS: this is title of dialog window, it is related to ripping audio CD */
		gettext_noop("Copy audio CD"),
		/* 2TRANS: this is message in dialog window: ripping audio cd is in progress */
		gettext_noop("Copying audio CD...") },

	{ /* 2TRANS: this is message printed to log file: a header for any further messages related to ripping data CD */
		gettext_noop("Copying data CD tracks:\n"),
		/* 2TRANS: this is title of dialog window, it is related to ripping data CD */
		gettext_noop("Copy data CD"),
		/* 2TRANS: this is message in dialog window: ripping data CD is in progress */
		gettext_noop("Copying data CD...") },

	{ /* 2TRANS: this is message printed to log file: a header for any further messages related to ripping data DVD */
		gettext_noop("Copying data DVD track:\n"),
		/* 2TRANS: this is title of dialog window, it is related to ripping data DVD */
		gettext_noop("Copy data DVD"),
		/* 2TRANS: this is message in dialog window: ripping data DVD is in progress */
		gettext_noop("Copying data DVD...")
	} };




static char *cdw_read_disc_get_audio_file_name(void);
void cdw_read_disc_update_track_in_processwin(track_t t, track_t n_tracks);
char *cdw_read_disc_get_audio_track_fullpath(track_t t, const char *filename);
cdw_rv_t cdw_read_disc_actual_reading(track_t t, const char *fullpath);
cdw_rv_t cdw_read_disc_read_audio_disc_prepare(char **audio_file_name);
cdw_rv_t cdw_read_disc_read_data_disc_prepare(void);
cdw_rv_t cdw_read_disc_read_audio_tracks(const char *filename);
cdw_rv_t cdw_read_disc_read_data_track(const char *fullpath);

/**
   \brief Copy tracks from optical disc to separate files

   Copy all tracks from audio CD or data CD/DVD to separate files in
   selected directory.

   You have to get disc meta information before calling this
   function - it does not read meta information itself. The function
   does not check for disc in drive and correct disc type either.

   Track files are written to existing directory pointed by user.

   \return CDW_OK when operation finished, most probably successfully
   \return CDW_NO when some preconditions were not met
   \return CDW_CANCEL if user cancelled operation
   \return CDW_MEM_ERROR if malloc failed
   \return CDW_GEN_ERROR if some other error occurred
*/
cdw_rv_t cdw_read_disc(void)
{
	cdw_rv_t crv = cdw_read_disc_check_preconditions();
	if (crv != CDW_OK) { /* conditions for reading disc are not met */
		return CDW_NO;
	}

	char *audio_file_name = (char *) NULL;
	cdw_disc_t *current_disc = cdw_disc_get();
	cdw_vdm ("INFO: disc type = \"%s\"\n", current_disc->type_label);
	int ind = 0;
	/* this opens cdio disc for getting some more meta info and for reading */
	crv = cdw_cdio_open_disc();
	if (current_disc->type == CDW_CD_AUDIO) {
		ind = 0;
		crv = cdw_read_disc_read_audio_disc_prepare(&audio_file_name);
	} else {
		if (current_disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
			ind = 1;
		} else {
			ind = 2;
		}
		crv = cdw_read_disc_read_data_disc_prepare();
	}
	/* this refreshes UI after file picker and before processwin */
	cdw_main_ui_main_window_wrefresh();

	if (crv != CDW_OK) {
		cdw_cdio_close_disc();
		return crv;
	}

	cdw_logging_write(_(strings_table[ind].log_string));
	cdw_processwin_create(_(strings_table[ind].window_title),
			      _(strings_table[ind].window_message), true);


	if (crv != CDW_OK) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window: disc
				      is in drive, but cdw cannot open it using a
				      library API */
				   _("Cannot open disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		cdw_main_ui_main_window_wrefresh();
		if (audio_file_name != (char *) NULL) {
			free(audio_file_name);
			audio_file_name = (char *) NULL;
		}
		cdw_cdio_close_disc();
		return CDW_ERROR;
	}


	/* here we finally read tracks */
	if (current_disc->type == CDW_CD_AUDIO) {
		crv = cdw_read_disc_read_audio_tracks(audio_file_name);
		free(audio_file_name);
		audio_file_name = (char *) NULL;
	} else {
		crv = cdw_read_disc_read_data_track(global_config.iso_image_full_path);
	}

	cdw_cdio_close_disc();

	/* when cdw is run in login console and there are some disc read
	   errors, OS prints some error messages to the console that
	   overwrite cdw ui; call to cdw_main_ui_main_window_wrefresh()
	   conveniently fixes it */
	if (crv == CDW_OK) {
		/* 2TRANS: this is message in dialog window: reading
		   a disc is finished. I'm not adding "successfully",
		   because this is not 100% sure. */
		cdw_processwin_destroy(_("Finished."), true);
		cdw_main_ui_main_window_wrefresh();
		return CDW_OK;
	} else {
		/* 2TRANS: this is message in dialog window: reading
		   a disc is finished, but with errors */
		cdw_processwin_destroy(_("Finished. Some errors occurred."), true);
		cdw_main_ui_main_window_wrefresh();
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Failed to read disc content. Consult log file ('L' key in main window) for further info."),
				   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		return CDW_NO;
	}
}





cdw_rv_t cdw_read_disc_read_audio_disc_prepare(char **audio_file_name)
{
	/* 2TRANS: this is title of dialog window in which user has
	   to enter path to valid dir */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("Output directory"),
					     /* 2TRANS: this is label in dialog
						window below it is input field */
					     _("Enter path to existing output directory"),
					     &(global_config.audiodir),
					     CDW_FS_DIR, R_OK|W_OK|X_OK, CDW_FS_EXISTING);
	cdw_main_ui_main_window_wrefresh();
	if (crv != CDW_OK) {
		/* user failed to provide correct path when
		   asked X times, or pressed ESCAPE */
		return CDW_CANCEL;
	}

	*audio_file_name = cdw_read_disc_get_audio_file_name();
	if (*audio_file_name == (char *) NULL) {
		return CDW_CANCEL;
	}

	return CDW_OK;

}




cdw_rv_t cdw_read_disc_read_data_disc_prepare(void)
{
	track_t n_tracks = cdw_cdio_get_number_of_tracks();
	if (n_tracks > 1) {
		/* 2TRANS: this is title of dialog window */
	        cdw_rv_t crv = cdw_buttons_dialog(_("Warning"),
						  /* 2TRANS: this is message in dialog window */
						  _("You are attempting to read multi-track disc. cdw can't read second and following tracks, so only first track will be read. Continue?"),
						  CDW_BUTTONS_OK_CANCEL, CDW_COLORS_WARNING);
		cdw_main_ui_main_window_wrefresh();
		if (crv != CDW_OK) {
			cdw_vdm ("INFO: returning after multi-track prompt\n");
			return crv;
		}
	}

	/* 2TRANS: this is title of dialog window in which user has
	   to enter path to valid dir */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("Output file"),
					     /* 2TRANS: this is label in dialog
						window below it is input field */
					     _("Select new or existing image file"),
					     &(global_config.iso_image_full_path),
					     CDW_FS_FILE, R_OK|W_OK,
					     CDW_FS_NEW | CDW_FS_EXISTING);
	if (crv != CDW_OK) {
		cdw_vdm ("INFO: file picker returns !CDW_OK\n")
		return crv;
	} else {
		return CDW_OK;
	}
}


/**
   \brief Function that encapsulates reading tracks in loop

   Function with no more than code in for () loop that goes through all
   tracks and reads them one by one. All checks have to be done before
   calling this function.

   \p filename is only core of file names - the function will append
   track number to this string to form names of all tracks read.

   \param disc - cdio disc structure, using which function will read tracks
   \param filename - core of name of files for all tracks

   \return CDW_OK on success (all tracks were read correctly)
   \return CDW_ERROR if at least one track was not read correctly
*/
cdw_rv_t cdw_read_disc_read_audio_tracks(const char *filename)
{
	track_t first_track = cdw_cdio_get_first_track();
	track_t last_track = cdw_cdio_get_last_track();
	track_t n_tracks = cdw_cdio_get_number_of_tracks();

	int status = 0;
	for (track_t t = first_track; t <= last_track; t++) {
		cdw_vdm ("INFO: copying track %d out of %d\n", t, n_tracks);
		cdw_read_disc_update_track_in_processwin(t, n_tracks);

		char *fullpath = cdw_read_disc_get_audio_track_fullpath(t, filename);
		if (fullpath == (char *) NULL) {
			status--;
			continue;
		}

		cdw_rv_t crv = cdw_read_disc_actual_reading(t, fullpath);
		if (crv != CDW_OK) {
			status--;
		}
		free(fullpath);
		fullpath = (char *) NULL;

		/* under these circumstances:
		   - cdw works on real console
		   - cdw reads data CD
		   - there are read errors at the end of track
		   OS will print error messages to the console, which will
		   disrupt cdw UI; these lines refresh the UI; sleep time
		   was selected by experimenting on my old 686 machine */
		sleep(1);
		cdw_main_ui_main_window_wrefresh();
		cdw_processwin_wrefresh();

	} /* for */

	if (status == 0) {
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: returning CDW_ERROR after reading errors\n");
		return CDW_ERROR;
	}
}






cdw_rv_t cdw_read_disc_read_data_track(const char *fullpath)
{
	track_t first_track = cdw_cdio_get_first_track();
	track_t n_tracks = cdw_cdio_get_number_of_tracks();

	cdw_vdm ("INFO: copying first track %d (out of %d tracks)\n", first_track, n_tracks);
	cdw_read_disc_update_track_in_processwin(first_track, n_tracks);

	cdw_rv_t retval = cdw_read_disc_actual_reading(first_track, fullpath);

	/* under these circumstances:
	   - cdw works on real console
	   - cdw reads data CD
	   - there are read errors at the end of track
	   OS will print error messages to the console, which will
	   disrupt cdw UI; these lines refresh the UI; sleep time
	   was selected by experimenting on my old 686 machine */
	sleep(1);
	cdw_main_ui_main_window_wrefresh();
	cdw_processwin_wrefresh();

	if (retval != CDW_OK) {
		cdw_vdm ("ERROR: returning CDW_ERROR after reading errors\n");
	}
	return retval;
}



void cdw_read_disc_update_track_in_processwin(track_t t, track_t n_tracks)
{
	char track_info[PROCESSWIN_MAX_RTEXT_LEN + 1];
	/* 2TRANS: this is message in dialog window:
	   first %d number of track being currently ripped,
	   second %d is total number of tracks */
	snprintf(track_info, PROCESSWIN_MAX_RTEXT_LEN + 1, _("Copying track %d/%d"),
		 t, n_tracks);
	cdw_processwin_display_main_info(track_info);

	return;
}



char *cdw_read_disc_get_audio_track_fullpath(track_t t, const char *filename)
{
	/* 7 is length of _%2.2d.raw; there can be 99 audio tracks
	   on audio cd, so place for 2 digits will be enough */
	char ending[7 + 1];
	snprintf(ending, 7 + 1, "_%02d.raw", t);

	char *fullpath = cdw_string_concat(global_config.audiodir, "/", filename, ending, (char *) NULL);
	cdw_vdm ("INFO: output raw file name: \"%s\"\n\n", fullpath);

	return fullpath;
}




cdw_rv_t cdw_read_disc_actual_reading(track_t t, const char *fullpath)
{
	cdw_rv_t retval = CDW_OK;
	int fd = open(fullpath, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
	int e = errno;
	if (fd == -1) {
		cdw_vdm ("ERROR: failed to open output file for writing, error = \"%s\"\n", strerror(e));
		cdw_vdm ("ERROR: fullpath \"%s\"\n", fullpath);
			/* 2TRANS: this is string written to log file when
			   program attempted to read track from CD, but failed
			   to open file, to which the track was supposed to
			   be written; 'skipping' means 'not performing read
			   of the track and going to next track' */
		cdw_logging_write(_("Failed to open output file for track %d, skipping...\n"), t);
	} else {
		/* 2TRANS: this is string written to log file when
		   program starts to read one track from CD disc;
		   "%d" is number of track */
		cdw_logging_write(_("Track %d: "), t);
		int rv = cdw_cdio_copy_sectors_to_file(t, fd, -1);

		/* can't write "success"/"failure" in these branches
		   of conditions, since success of cdw_cdio_copy_sectors_to_file()
		   does not mean that resulting iso image will be
		   correct - it may turn out that this is quite the
		   opposite: in tests on multisession discs first
		   track was read with errors (few last sectors could
		   not be read correctly), but iso file of first
		   track could be open and viewed; for the same disc
		   reading of tracks 2 and 3 by cdw_cdio* was
		   successful, but resulting image files were not
		   readable by isoinfo, so opening them and viewing
		   their content in any file manager was not possible */
		if (rv == 0) {
			/* don't set success = true, this might
			   overwrite previous failures */

			/* 2TRANS: this is string written to log file when
			   reading one track from CD disc is finished;
			   it is written after "Track %d: " string */
			cdw_logging_write(_("done\n"));
		} else {
			/* if any error will occur for any track,
			   then mark whole operation as unsuccessful */
			retval = CDW_ERROR;
			cdw_vdm ("ERROR: track read error for track %d\n", t);

			/* 2TRANS: this is string written to log file when
			   reading one track from CD disc is finished;
			   it is written after "Track %d: " string */
			cdw_logging_write(_("failed: there were some problems while reading track\n"));
			cdw_cdio_copy_print_debug_on_error(rv);
		}
		close(fd);
	}
	return retval;
}




/**
   \brief Check if process of reading tracks of CD can be performed

   Do all necessary checks needed before performing operation of
   copying tracks from CD to hard disc. These checks include:
   \li checking for disc in drive
   \li checking if disc is not blank
   \li checking if cdw can read tracks from disc
   \li checking if disc type and action selected by user match (e.g. if
       user pressed 'copy audio cd' and there is audio cd in drive

   Task specification:
   task.id must be set to CDW_TASK_READ_DISC or CDW_TASK_CHECK_MD5SUM.
   task.id can't be set to CDW_TASK_CHECK_MD5SUM

   \return CDW_OK if all necessary preconditions are met
   \return CDW_NO if any of preconditions is not met
*/
cdw_rv_t cdw_read_disc_check_preconditions(void)
{
	cdw_rv_t crv = cdw_read_disc_info();
	if (crv != CDW_OK) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot get metainformation from disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		cdw_main_ui_main_window_wrefresh();
		return CDW_NO;
	}

	cdw_disc_t *current_disc = cdw_disc_get();

	/* obviously we can't read any user data from blank disc;
	   check if disc is empty _before_ calling cdw_disc_supported_for_ripping(),
	   because the function will return false for empty disc, even "rippable";
	   TODO: on the other hand this code informs that "disc is empty" even for
	   non-rippable discs, which should also be avoided */
	if (current_disc->state_empty == CDW_TRUE) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Information"),
				   /* 2TRANS: this message in dialog window */
				   _("Disc in drive is empty, can't read any data from the disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		cdw_main_ui_main_window_wrefresh();

		return CDW_NO;
	} else if (current_disc->state_empty == CDW_UNKNOWN) {
		/* 2TRANS: this is title of dialog window */
		crv = cdw_buttons_dialog(_("Information"),
					 /* 2TRANS: this message in dialog window */
					 _("Disc in drive may be empty and reading from disc may fail. Continue?"),
					 CDW_BUTTONS_OK_CANCEL, CDW_COLORS_DIALOG);
		cdw_main_ui_main_window_wrefresh();
		if (crv != CDW_OK) {
			return CDW_NO;
		}
	} else {
		;
	}

	const char *message = cdw_disc_check_if_is_readable(current_disc);
	if (message != (char *) NULL) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Information"),
				   message,
				   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		cdw_main_ui_main_window_wrefresh();
		return CDW_NO;
	}

	return CDW_OK;
}




char *cdw_read_disc_get_audio_file_name(void)
{
	/* 2TRANS: this is main part of name of audio tracks ripped from
	   cd, no more than TRACK_CORE_NAME_LEN = 30 chars */
	char *filename = strndup(_("track"), CDW_READ_DICS_TRACK_CORE_NAME_LEN);
	if (filename == (char *) NULL) {
		return (char *) NULL;
	}

	cdw_safe_input_data_t data;

	data.attempts_max = 3; /* how many attempts? */
	/* 2TRANS: this is title of dialog window where user have to
	   enter core of track file names */
	data.window_title = _("File name");   /* title of input window and error dialogs */
	/* 2TRANS: this is label in dialog window, below it is an input field */
	data.prompt_message = _("Enter core of track file names (no more than 30 characters).");  /* message in input window */
	data.input_type = CDW_NCURSES_INPUT_NONE;
	data.chars_max = CDW_READ_DICS_TRACK_CORE_NAME_LEN;    /* maximal length of string */
	data.buffer = &filename;
	/* error message displayed at the end in case of failure */
	/* 2TRANS: this is error message displayed in dialog window
	   when getting a file name from user failed */
	data.error_message = _("Can't get valid file name, aborting.");

	cdw_rv_t crv = cdw_safe_input_dialog(&data);
	cdw_main_ui_main_window_wrefresh();
	if (crv == CDW_OK) {
		return filename;
	} else {
		free(filename);
		filename = (char *) NULL;
		if (crv == CDW_CANCEL) {
			;
		} else {
			cdw_vdm ("ERROR: can't get correct file name\n");
		}
		return (char *) NULL;
	}

}
