/*
 * HopEditor.cpp is part of Brewtarget, and is Copyright Philip G. Lee
 * (rocketman768@gmail.com), 2009-2011.
 *
 * Brewtarget is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * Brewtarget is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QtGui>
#include <iostream>
#include <string>
#include <QIcon>
#include "HopEditor.h"
#include "database.h"
#include "config.h"
#include "unit.h"
#include "brewtarget.h"

HopEditor::HopEditor( QWidget* /*parent*/ )
{
   setupUi(this);

   setWindowIcon(QIcon(SMALLHOP));
   
   connect( buttonBox, SIGNAL( accepted() ), this, SLOT( save() ));
   connect( buttonBox, SIGNAL( rejected() ), this, SLOT( clearAndClose() ));

   obsHop = 0;
}

void HopEditor::setHop( Hop* h )
{
   if( h && h != obsHop )
   {
      obsHop = h;
      setObserved(h);
      showChanges();
   }
}

void HopEditor::save()
{
   Hop *h = obsHop;

   if( h == 0 )
   {
      setVisible(false);
      return;
   }

   // Need to disable notification since every "set" method will cause a "showChanges" that
   // will revert any changes made.
   h->disableNotification();

   h->setName(lineEdit_name->text());
   h->setAlpha_pct(lineEdit_alpha->text().toDouble());
   h->setAmount_kg(Brewtarget::weightQStringToSI(lineEdit_amount->text()));
   h->setUse(static_cast<Hop::Use>(comboBox_use->currentIndex()));
   h->setTime_min(Brewtarget::timeQStringToSI(lineEdit_time->text()));
   h->setType(static_cast<Hop::Type>(comboBox_type->currentIndex()));
   h->setForm(static_cast<Hop::Form>(comboBox_form->currentIndex()));
   h->setBeta_pct(lineEdit_beta->text().toDouble());
   h->setHsi_pct(lineEdit_HSI->text().toDouble());
   h->setOrigin(lineEdit_origin->text());
   h->setHumulene_pct(lineEdit_humulene->text().toDouble());
   h->setCaryophyllene_pct(lineEdit_caryophyllene->text().toDouble());
   h->setCohumulone_pct(lineEdit_cohumulone->text().toDouble());
   h->setMyrcene_pct(lineEdit_myrcene->text().toDouble());

   h->setSubstitutes(textEdit_substitutes->toPlainText());
   h->setNotes(textEdit_notes->toPlainText());

   h->reenableNotification();
   h->forceNotify();

   Database::getDatabase()->resortHops(); // If the name changed, need to resort.

   setVisible(false);
}

void HopEditor::clearAndClose()
{
   if( obsHop )
   {
      obsHop->removeObserver(this);
      obsHop = 0;
   }
   setVisible(false); // Hide the window.
}

void HopEditor::notify(Observable* notifier, QVariant info)
{
   if( notifier == obsHop )
      showChanges();
}

void HopEditor::showChanges()
{
   Hop* h = obsHop;
   if( h == 0 )
      return;

   lineEdit_name->setText(h->getName());
   lineEdit_name->setCursorPosition(0);
   lineEdit_alpha->setText(Brewtarget::displayAmount(h->getAlpha_pct(), 0));
   lineEdit_amount->setText(Brewtarget::displayAmount(h->getAmount_kg(), Units::kilograms));
   comboBox_use->setCurrentIndex(obsHop->getUse());
   lineEdit_time->setText(Brewtarget::displayAmount(h->getTime_min(), Units::minutes));
   comboBox_type->setCurrentIndex(obsHop->getType());
   comboBox_form->setCurrentIndex(obsHop->getForm());
   lineEdit_beta->setText(Brewtarget::displayAmount(h->getBeta_pct(), 0));
   lineEdit_HSI->setText(Brewtarget::displayAmount(h->getHsi_pct(), 0));
   lineEdit_origin->setText(h->getOrigin());
   lineEdit_origin->setCursorPosition(0);
   lineEdit_humulene->setText(Brewtarget::displayAmount(h->getHumulene_pct(), 0));
   lineEdit_caryophyllene->setText(Brewtarget::displayAmount(h->getCaryophyllene_pct(), 0));
   lineEdit_cohumulone->setText(Brewtarget::displayAmount(h->getCohumulone_pct(), 0));
   lineEdit_myrcene->setText(Brewtarget::displayAmount(h->getMyrcene_pct(), 0));
   textEdit_substitutes->setPlainText(h->getSubstitutes());
   textEdit_notes->setPlainText(h->getNotes());
}
