/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "shell.h"
#include "gwrappers.h"
#include "directories.h"
#include "utilities.h"
#include "gtkwrappers.h"
#include "progresswindow.h"


bool program_is_running (const ustring & commandline)
// Returns true if the program given on "commandline" is running.
{
  return (programs_running_count (commandline) > 0);
}


int programs_running_count (const ustring & commandline)
// Returns how many times the program given on "commandline" is running.
{
  // Run the process status program.
  FILE *fd;
  fd = popen ("ps ax", "r");
  if (!fd) {
    return false;
  }
  // Read the results of it.
  int i = 0;
  char buf[1024];
  while (fgets (buf, sizeof (buf), fd) != NULL) {
    ustring line = buf;
    if (line.find (commandline) != string::npos) {
      i++;
    }
  }
  int result;
  result = pclose (fd);
  // Return the count.
  return i;
}


bool program_is_running_basic (const ustring & program)
// Returns true if the program given running.
{
  return (programs_running_count_basic (program) > 0);
}

int programs_running_count_basic (const ustring & program)
// Returns how many times the program given is running.
{
  // Run the process status program.
  FILE *fd;
  fd = popen ("ps -e", "r");
  if (!fd) {
    return false;
  }
  // Read the results of it.
  int i = 0;
  char buf[1024];
  while (fgets (buf, sizeof (buf), fd) != NULL) {
    ustring line = buf;
    size_t position = line.rfind (" ");
    line.erase (0, position);
    line = trim (line);
    if (line == program) {
      i++;
    }
  }
  int result;
  result = pclose (fd);
  // Return the count.
  return i;
}


int run_shell_progress (const ustring& commandline, const ustring& programname, 
                        bool gui, const ustring& message, 
                        bool output_to_logfile, int max_logfile_size)
/*
Some shell processes take a long time.
This function runs that shell process, and updates the progressbar also,
so that the user has some feedback and sees things moving forward.
The size of the logfile must not exceeded "max_logfile_size". If it does,
the process will be interrupted.
*/
{
  // Return value to indicate success or not.
  int returnvalue = -1;
  // Check logfile size.
  bool size_exceeded = false;
  // Progress information.
  ProgressWindow * progresswindow = NULL;
  if (gui) progresswindow = new ProgressWindow (message, true);
  // Make a logfile.
  ustring logfile;
  logfile = gw_build_filename (directories_get_temp(), "shell.log");
  unlink (logfile.c_str());
  // Start the process in the background.
  ustring command (commandline);
  // Route output to the logfile.
  if (output_to_logfile) {
    command.append (" >");
    command.append (shell_quote_space (logfile));
    command.append ("2>&1");
  }
  command.append (" &");
  if (system (command.c_str()) >= 0) {
    // Indicate zero for no error.
    returnvalue = 0;
    // Wait a second for shell process to start.
    g_usleep (1000000);
    // Wait for the shell process to finish.
    while (program_is_running (programname)) {
      if (gui) progresswindow->pulse ();
      // Check the size of the logfile is stil within limits.
      if (output_to_logfile) {
        struct stat statbuf;
        stat (logfile.c_str(), &statbuf);
        if (statbuf.st_size > max_logfile_size)
          size_exceeded = true;
      }
      // Check whether cancel.
      bool cancel = false;
      if (gui) 
        if (progresswindow->cancel) 
          cancel = true;
      if (cancel || size_exceeded) {
        // Kill process.
        ustring command;
        command = "killall -HUP ";
        command.append (programname);
        system (command.c_str());
        returnvalue = -1;
        break;
      }
      g_usleep (1000000);
    }
  }
  // Give message if logsize was exceeded.
  if (size_exceeded && gui)
    gtkw_dialog_error (NULL, "Process produced a logfile which became too large. Interrupted");
  // Read the logfile made by the shell command.
  if (output_to_logfile) {
    ReadText rt (logfile, true, false);
    for (unsigned int i = 0; i < rt.lines.size(); i++) {
      // Add to program's logfile.
      gw_message (rt.lines[i]);
    }
  }
  // Finish.
  if (progresswindow) delete progresswindow;
  return returnvalue;  
}


ustring shell_quote_space (const ustring& filename)
// Puts quotes and spaces around a filename, making it fit for the shell.
// Example: /home/user/John Johnson/.bibledit/projects/test/Genesis
// becomes: '/home/user/John Johnson/.bibledit/projects/test/Genesis'
// with an extra space prefixed and suffixed.
// Unix does not allow spaces in the user names, but Cygwin does.
// This function was introduced to address this situation, but has a wider
// use too.
{
  ustring quotedname;
  quotedname = " '" + filename + "' ";
  return quotedname;
}


bool copy_file (const ustring& from, const ustring& to)
// Copies a file. Returns true if succeeded.
{
  ustring command = "cp";
  command.append (shell_quote_space (from));
  command.append (shell_quote_space (to));
  return (system (command.c_str()) == 0);
}


ustring shell_clean_filename (const ustring& filename)
// Replace characters like ' and / occur in the filename with _.
{
  ustring cleanfile (filename);
  replace_text (cleanfile, "'", "_");
  replace_text (cleanfile, "/", "_");
  return cleanfile;
}
