/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include <libgen.h>
#include <glib.h>
#include <config.h>
#include "constants.h"
#include "odtutils.h"
#include "date_time_utils.h"
#include "fonts.h"
#include "gwrappers.h"
#include "generalconfig.h"
#include "projectconfig.h"
#include "color.h"


void odt_paragraph_style (vector<ustring>& lines, const ustring& name,
                          const ustring& fontname,
                          double fontsize, int lineheight,
                          const ustring& italic, const ustring& bold, 
                          const ustring& underline, const ustring& smallcaps,
                          ustring justification,
                          double spacebefore, double spaceafter,
                          double leftmargin, double rightmargin,
                          double firstlineindent, bool spancolumns,
                          bool startpage)
{
  // Variables.
  ustring line;
  ustring s;

  // Style properties.
  line = "<style:style style:name=\"xx\" style:family=\"paragraph\" style:parent-style-name=\"Standard\">";
  replace_text (line, "xx", name);
  lines.push_back (line);

  // Paragraph properties.  
  line = "<style:paragraph-properties";
  if (lineheight != 100) {
    s = " fo:line-height=\"xxx%\"";
    replace_text (s, "xxx", convert_to_string (lineheight));
    line.append (s);
  }
  if (!justification.empty()) {
    s = " fo:text-align=\"xx\" style:justify-single-word=\"false\"";
    if (justification == LEFT) {
      replace_text (s, "xx", "start");
    } else if (justification == RIGHT) {
      replace_text (s, "xx", "end");
    } else {
      replace_text (s, "xx", justification);
    }
    line.append (s);
  }
  if (spacebefore != 0) {
    s = " fo:margin-top=\"xxmm\"";
    replace_text (s, "xx", convert_to_string (spacebefore));
    line.append (s);
  }
  if (spaceafter != 0) {
    s = " fo:margin-bottom=\"xxmm\"";
    replace_text (s, "xx", convert_to_string (spaceafter));
    line.append (s);
  }
  if (leftmargin != 0) {
    s = " fo:margin-left=\"xxmm\"";
    replace_text (s, "xx", convert_to_string (leftmargin));
    line.append (s);
  }
  if (rightmargin != 0) {
    s = " fo:margin-right=\"xxmm\"";
    replace_text (s, "xx", convert_to_string (rightmargin));
    line.append (s);
  }
  if (firstlineindent != 0) {
    s = " fo:text-indent=\"xxmm\"";
    replace_text (s, "xx", convert_to_string (firstlineindent));
    line.append (s);
  }
  // Starting odd pages is not supported in OpenDocument 1.0.
  if (startpage) {
    line.append (" fo:break-before=\"page\"");
  }
  // Spanning of columns is not implemented because that interferes with how
  // easily the text can be edited.
  line.append ("/>");
  lines.push_back (line);
  
  // Text properties.
  line = "<style:text-properties";
  {
    s = " fo:font-size=\"xxpt\" style:font-size-asian=\"xxpt\" style:font-size-complex=\"xxpt\"";
    replace_text (s, "xx", convert_to_string (fontsize));
    line.append (s);
  }
  if (!fontname.empty()) {
    s = " style:font-name=\"xx\"";
    replace_text (s, "xx", fontname);
    line.append (s);
  }
  if (italic == ON) {
    line.append (" fo:font-style=\"italic\" style:font-style-asian=\"italic\" style:font-style-complex=\"italic\"");
  }
  if (bold == ON) {
    line.append (" fo:font-weight=\"bold\" style:font-weight-asian=\"bold\" style:font-weight-complex=\"bold\"");
  }
  if (underline == ON) {
    line.append (" style:text-underline-style=\"solid\" style:text-underline-width=\"auto\" style:text-underline-color=\"font-color\"");
  }
  if (smallcaps == ON) {
    line.append (" fo:font-variant=\"small-caps\"");
  }
  line.append ("/>");
  lines.push_back (line);

  // Close style  
  lines.push_back ("</style:style>");
}


void odt_span_style (vector<ustring>& lines, const ustring& name,
                     const ustring& fontname,
                     double fontpercentage, const ustring& italic, const ustring& bold, 
                     const ustring& underline, const ustring& smallcaps, bool superscript,
                     unsigned int color)
{
  // Variables.
  ustring styleproperties;
  ustring textproperties;
  ustring line;
  
  styleproperties = "<style:style style:name=\"";
  styleproperties.append (name);
  styleproperties.append ("\" style:family=\"text\"");
  textproperties.append ("<style:text-properties");

  if (!fontname.empty()) {
    line = " style:font-name=\"xx\"";
    replace_text (line, "xx", fontname);
    textproperties.append (line);
  }
  
  line = odt_span_fontpercentage (fontpercentage);
  if (!line.empty()) textproperties.append (line);
  textproperties.append (odt_span_italic (italic));
  textproperties.append (odt_span_bold (bold));
  textproperties.append (odt_span_underline (underline));
  textproperties.append (odt_span_smallcaps (smallcaps));
  textproperties.append (odt_span_superscript (superscript));
  textproperties.append (color_odt (color));

  styleproperties.append (">");
  textproperties.append ("/>");

  lines.push_back (styleproperties);  
  lines.push_back (textproperties);  
  lines.push_back ("</style:style>");
}


OdtTextParagraph::OdtTextParagraph (vector<ustring> * lines, const ustring& stylename)
// OpenDocument Text Text Paragraph.
// If no stylename is given, it takes the "Standard" style.
{
  // Save pointer, init variables.
  mylines = lines;
  mynewline = false;
  
  // Build the line.
  myline = "<text:p text:style-name=\"";
  if (stylename.empty())
    myline.append ("Standard");
  else
    myline.append (stylename);
  myline.append ("\">");
  
  // Line length.
  linelength = myline.length();
}


OdtTextParagraph::~OdtTextParagraph ()
{
  // Close the paragraph.
  myline.append ("</text:p>");
  
  // Store the line.
  mylines->push_back (myline);
}


void OdtTextParagraph::newline ()
{
  mynewline = true;
}


void OdtTextParagraph::plaintext (const ustring& line)
{
  if (mynewline) myline.append (" ");
  mynewline = false;
  if (linelength != myline.length()) myline.append (" ");
  myline.append (line);
}


void OdtTextParagraph::spannedtext (const ustring& line, const ustring& stylename)
{
  if (mynewline) myline.append (" ");
  mynewline = false;
  myline.append ("<text:span text:style-name=\"");
  myline.append (stylename);
  myline.append ("\">");
  myline.append (line);
  myline.append ("</text:span>");
}


ustring odt_styles_xml_filename (const ustring& directory)
{
  return gw_build_filename (directory, "styles.xml");
}


ustring odt_content_xml_filename (const ustring& directory)
{
  return gw_build_filename (directory, "content.xml");
}


void odt_create_page_master (const ustring& directory)
// Writes the page layout to the right file in the directory given.
{
  GeneralConfiguration genconfig (0);
  ReadText rt (odt_styles_xml_filename (directory), true, false);
  for (unsigned int i = 0; i < rt.lines.size(); i++) {
    replace_text (rt.lines[i], "21pw", convert_to_string (genconfig.paper_width()));
    replace_text (rt.lines[i], "29.7ph", convert_to_string (genconfig.paper_height()));
    replace_text (rt.lines[i], "2.0mt", convert_to_string (genconfig.paper_top_margin()));
    replace_text (rt.lines[i], "2.0mb", convert_to_string (genconfig.paper_bottom_margin()));
    replace_text (rt.lines[i], "2.0ml", convert_to_string (genconfig.paper_left_margin()));
    replace_text (rt.lines[i], "2.0mr", convert_to_string (genconfig.paper_right_margin()));
  }
  write_lines (odt_styles_xml_filename (directory), rt.lines);
}


void odt_set_font (const ustring& directory, const ustring& fontname)
// Writes the font to the right files in the directory given.
{
  // Save the font in the styles file.
  ReadText rt1 (odt_styles_xml_filename (directory), true, false);
  for (unsigned int i = 0; i < rt1.lines.size(); i++) {
    replace_text (rt1.lines[i], "Bitstream", fontname);
  }
  write_lines (odt_styles_xml_filename (directory), rt1.lines);
  // Save the font in the content file.
  ReadText rt2 (odt_content_xml_filename (directory), true, false);
  for (unsigned int i = 0; i < rt2.lines.size(); i++) {
    replace_text (rt2.lines[i], "Bitstream", fontname);
  }
  write_lines (odt_content_xml_filename (directory), rt2.lines);
}

void odt_insert_content (const ustring& directory, const vector <ustring>& text, const vector <ustring>& styles)
{
  vector <ustring> odtlines;
  ReadText rt (odt_content_xml_filename (directory), true, false);
  for (unsigned int i = 0; i < rt.lines.size(); i++) {
    if (rt.lines[i].find ("</office:text>") == 0) {
      for (unsigned int i2 = 0; i2 < text.size(); i2++) {
        odtlines.push_back (text[i2]);
      }
    }
    if (rt.lines[i].find ("</office:automatic-styles>") == 0) {
      for (unsigned int i2 = 0; i2 < styles.size(); i2++) {
        odtlines.push_back (styles[i2]);
      }
    }
    odtlines.push_back (rt.lines[i]);
  }
  write_lines (odt_content_xml_filename (directory), odtlines);  
}


void odt_insert_styles (const ustring& directory, const vector <ustring>& styles)
{
  vector <ustring> odtlines;
  ReadText rt (odt_styles_xml_filename (directory), true, false);
  for (unsigned int i = 0; i < rt.lines.size(); i++) {
    if (rt.lines[i].find ("</office:styles>") == 0) {
      for (unsigned int i2 = 0; i2 < styles.size(); i2++) {
        odtlines.push_back (styles[i2]);
      }
    }
    odtlines.push_back (rt.lines[i]);
  }
  write_lines (odt_styles_xml_filename (directory), odtlines);  
}


ustring odt_span_fontpercentage (double fontpercentage) 
{
  if (fontpercentage == 100) return "";
  ustring line = " fo:font-size=\"xx%\"";
  replace_text (line, "xx", convert_to_string (fontpercentage));
  return line;
}


ustring odt_span_italic (const ustring& italic)
{
  ustring line = " fo:font-style=\"xx\" style:font-style-asian=\"xx\" style:font-style-complex=\"xx\"";
  if ((italic == ON) || (italic == TOGGLE)) replace_text (line, "xx", "italic");
  else replace_text (line, "xx", "normal");
  // Note that ON and TOGGlE both switch italic on 
  // and OFF and INHERIT switch italic off. 
  // Improvements could be made here.
  // Bold, underline, etc., work the same.
  return line;
}


ustring odt_span_bold (const ustring& bold)
{
  ustring line = " fo:font-weight=\"xx\" style:font-weight-asian=\"xx\" style:font-weight-complex=\"xx\"";
  if ((bold == ON) || (bold == TOGGLE)) replace_text (line, "xx", "bold");
  else replace_text (line, "xx", "normal");
  return line;
}


ustring odt_span_underline (const ustring& underline)
{
  ustring line = " style:text-underline-style=\"xx\" style:text-underline-width=\"auto\" style:text-underline-color=\"font-color\"";
  if ((underline == ON) || (underline == TOGGLE)) replace_text (line, "xx", "solid");
  else replace_text (line, "xx", "none");
  return line;
}


ustring odt_span_smallcaps (const ustring& smallcaps)
{
  ustring line = " fo:font-variant=\"xx\"";
  if ((smallcaps == ON) || (smallcaps == TOGGLE)) replace_text (line, "xx", "small-caps");
  else replace_text (line, "xx", "normal");
  return line;
}


ustring odt_span_superscript (bool superscript)
{
  ustring line;
  if (superscript) 
    line = " style:text-position=\"super 58%\"";
  return line;
}
