/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "generalconfig.h"
#include "directories.h"
#include "gwrappers.h"
#include "sqlite_reader.h"
#include "utilities.h"
#include "vacuum.h"
#include "date_time_utils.h"
#include "shell.h"


void configuration_vacuum ()
{
  vacuum_database (configuration_filename (), 0);
}


ustring configuration_filename ()
{
  return gw_build_filename (directories_get_configuration (), "configuration.sqlite.2");
}


void upgrade_configuration ()
{
  // Nothing to do just now.
}


/*
A test is now going on to see if the cause of irregular crashes of Bibledit, 
when used a lot, is, that GeneralConfiguration is always left open in mainwindow.
It is now always closed, and we'll see if that gets rid of the crashes.
*/


GeneralConfiguration::GeneralConfiguration (int dummy)
{
  error = NULL;
  ustring filename = configuration_filename ();
  try
  {
    if (!g_file_test (filename.c_str(), G_FILE_TEST_IS_REGULAR))
      create (filename);
    rc = sqlite3_open (filename.c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
}


GeneralConfiguration::~GeneralConfiguration ()
{
  // Close connection.
  sqlite3_close (db);
}


void GeneralConfiguration::create (const ustring& filename)
// Creates a new table for storing the configuration.
// One tables is created for boolean, string and integer values.
// This is for easier management.
// A sequence field is added for storing lists of things.
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open (filename.c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    char * sql;
    sql = g_strdup_printf ("create table data (key text, value text, sequence integer);");
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    g_free (sql);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);
}


ustring GeneralConfiguration::value_get (const ustring& key, const ustring& standard)
{
  ustring value = standard;
  try
  {
    char * sql;
    sqlite3_busy_timeout (db, 1000);
    SqliteReader sqlitereader (0);
    sql = g_strdup_printf ("select value from data where key = '%s';", key.c_str());
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if ((sqlitereader.ustring0.size() > 0)) {
      value = sqlitereader.ustring0[0];
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  return value;
}


void GeneralConfiguration::value_set (const ustring& key, const ustring& value)
{
  try
  {
    char * sql;
    sql = g_strdup_printf ("delete from data where key = '%s';", key.c_str());
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    sql = g_strdup_printf ("insert into data values ('%s', '%s', 0)", key.c_str(), double_apostrophy (value).c_str());
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    g_free (sql);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
}


vector<ustring> GeneralConfiguration::valuelist_get (const ustring& key)
{
  vector<ustring> value;
  vector<unsigned int> sequence;
  try
  {
    char * sql;
    sqlite3_busy_timeout (db, 1000);
    SqliteReader sqlitereader (0);
    sql = g_strdup_printf ("select value, sequence from data where key = '%s';", key.c_str());
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    value.clear ();
    value.assign (sqlitereader.ustring0.begin(), sqlitereader.ustring0.end());
    for (unsigned int i = 0; i < sqlitereader.ustring1.size(); i++) {
      sequence.push_back (convert_to_int (sqlitereader.ustring1[i]));
    }
    quick_sort (sequence, value, 0, sequence.size());
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  return value;
}


void GeneralConfiguration::valuelist_set (const ustring& key, const vector<ustring>& value)
{
  try
  {
    char * sql;
    sql = g_strdup_printf ("delete from data where key = '%s';", key.c_str());
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    for (unsigned int i = 0; i < value.size(); i++) {
      sql = g_strdup_printf ("insert into data values ('%s', '%s', %d)", key.c_str(), value[i].c_str(), i);
      rc = sqlite3_exec (db, sql, NULL, NULL, &error);
      g_free (sql);
      if (rc) {
        throw runtime_error (sqlite3_errmsg(db));
      }
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
}


bool GeneralConfiguration::bool_get (const ustring& id, bool standard)
{
  return convert_to_bool (value_get (id, convert_to_string (standard)));
}


void GeneralConfiguration::bool_set (const ustring& id, bool value)
{
  value_set (id, convert_to_string (value));
}


int GeneralConfiguration::int_get (const ustring& id, int standard)
{
  return convert_to_int (value_get (id, convert_to_string (standard)));
}


void GeneralConfiguration::int_set (const ustring& id, int value)
{
  value_set (id, convert_to_string (value));
}


ustring GeneralConfiguration::string_get (const ustring& id, const ustring& standard)
{
  return value_get (id, standard);
}


void GeneralConfiguration::string_set (const ustring& id, const ustring& value)
{
  value_set (id, value);
}


double GeneralConfiguration::double_get (const ustring& id, double standard)
{
  return convert_to_double (value_get (id, convert_to_string (standard)));  
}


void GeneralConfiguration::double_set (const ustring& id, double value)
{
  value_set (id, convert_to_string (value));
}


vector<bool> GeneralConfiguration::vector_bool_get (const ustring& id)
{
  vector<ustring> value = valuelist_get (id);
  vector<bool> value2;
  for (unsigned int i = 0; i < value.size(); i++)
    value2.push_back (convert_to_bool (value[i]));
  return value2;
}


void GeneralConfiguration::vector_bool_set (const ustring& id, const vector<bool>& value)
{
  vector<ustring> value2;
  for (unsigned int i = 0; i < value.size(); i++)
    value2.push_back (convert_to_string (value[i]));
  valuelist_set (id, value2);
}


vector<ustring> GeneralConfiguration::vector_string_get (const ustring& id)
{
  return valuelist_get (id);
}


void GeneralConfiguration::vector_string_set (const ustring& id, const vector<ustring>& value)
{
  valuelist_set (id, value);
}


vector<int> GeneralConfiguration::vector_int_get (const ustring& id)
{
  vector<ustring> value = valuelist_get (id);
  vector<int> value2;
  for (unsigned int i = 0; i < value.size(); i++)
    value2.push_back (convert_to_int (value[i]));
  return value2;
}


void GeneralConfiguration::vector_int_set (const ustring& id, const vector<int>& value)
{
  vector<ustring> value2;
  for (unsigned int i = 0; i < value.size(); i++)
    value2.push_back (convert_to_string (value[i]));
  valuelist_set (id, value2);
}


ustring GeneralConfiguration::synchronize_project_on_open_key ()
/*
Whether Bibledit will synchronize a project when opening it.
*/
{
  return "sync_on_open";
}

bool GeneralConfiguration::synchronize_project_on_open ()
{
  return bool_get (synchronize_project_on_open_key(), true);
}

void GeneralConfiguration::synchronize_project_on_open_set (bool value)
{
  bool_set (synchronize_project_on_open_key(), value);
}


ustring GeneralConfiguration::synchronize_project_on_close_key ()
/*
Whether Bibledit will synchronize a project when closing it.
*/
{
  return "sync_on_close";
}

bool GeneralConfiguration::synchronize_project_on_close ()
{
  return bool_get (synchronize_project_on_close_key(), true);
}

void GeneralConfiguration::synchronize_project_on_close_set (bool value)
{
  bool_set (synchronize_project_on_close_key(), value);
}


ustring GeneralConfiguration::synchronize_project_regularly_key ()
/*
Whether Bibledit will synchronize a project regularly, while it is open.
*/
{
  return "sync_regularly";
}
  
bool GeneralConfiguration::synchronize_project_regularly ()
{
  return bool_get (synchronize_project_regularly_key(), false);
}

void GeneralConfiguration::synchronize_project_regularly_set (bool value)
{
  bool_set (synchronize_project_regularly_key(), value);
}


ustring GeneralConfiguration::synchronize_project_minutes_key ()
/*
If Bibledit synchronizes a project regularly, this gives the interval in minutes, 
e.g. it will synchronize a project every n minutes.
*/
{
  return "sync_minutes";
}

int GeneralConfiguration::synchronize_project_minutes ()
{
  return int_get (synchronize_project_minutes_key(), 10);
}

void GeneralConfiguration::synchronize_project_minutes_set (int value)
{
  int_set (synchronize_project_minutes_key(), value);
}


ustring GeneralConfiguration::synchronize_on_startup_key ()
/*
Whether all projects get synchronized when Bibledit starts.
*/
{
  return "sync_startup";
}

bool GeneralConfiguration::synchronize_on_startup ()
{
  return bool_get (synchronize_on_startup_key(), false);
}

void GeneralConfiguration::synchronize_on_startup_set (bool value)
{
  bool_set (synchronize_on_startup_key(), value);
}


ustring GeneralConfiguration::synchronize_on_shutdown_key ()
/*
Whether all projects get synchronized when Bibledit shuts down.
*/
{
  return "sync_shutdown";
}

bool GeneralConfiguration::synchronize_on_shutdown ()
{
  return bool_get (synchronize_on_shutdown_key (), true);
}

void GeneralConfiguration::synchronize_on_shutdown_set (bool value)
{
  bool_set (synchronize_on_shutdown_key(), value);
}


ustring GeneralConfiguration::startup_time_key ()
/*
Stores the internal time when Bibledit started, in seconds.
*/
{
  return "startup_time";
}

int GeneralConfiguration::startup_time ()
{
  return int_get (startup_time_key(), 0);
}

void GeneralConfiguration::startup_time_set (int value)
{
  int_set (startup_time_key(), value);
}


ustring GeneralConfiguration::screen_width_key ()
/*
The width of the screen, in pixels.
*/
{
  return "screen_width";
}


int GeneralConfiguration::screen_width ()
{
  return int_get (screen_width_key(), 0);
}


void GeneralConfiguration::screen_width_set (int value)
{
  int_set (screen_width_key(), value);
}


ustring GeneralConfiguration::screen_height_key ()
/*
The height of the screen, in pixels.
*/
{
  return "screen_height";
}


int GeneralConfiguration::screen_height ()
{
  return int_get (screen_height_key(), 0);
}


void GeneralConfiguration::screen_height_set (int value)
{
  int_set (screen_height_key(), value);
}


ustring GeneralConfiguration::window_maximized_key ()
/*
Whether the main window is maximized.
*/
{
  return "window_maximized";
}

bool GeneralConfiguration::window_maximized ()
{
  return bool_get (window_maximized_key (), false);
}

void GeneralConfiguration::window_maximized_set (bool value)
{
  bool_set (window_maximized_key(), value);
}


ustring GeneralConfiguration::window_width_key ()
/*
The width of the main window, in pixels.
*/
{
  return "window_width";
}


int GeneralConfiguration::window_width ()
{
  return int_get (window_width_key(), 0);
}


void GeneralConfiguration::window_width_set (int value)
{
  int_set (window_width_key(), value);
}


ustring GeneralConfiguration::window_height_key ()
/*
The height of the main window, in pixels.
*/
{
  return "window_height";
}


int GeneralConfiguration::window_height ()
{
  return int_get (window_height_key(), 0);
}


void GeneralConfiguration::window_height_set (int value)
{
  int_set (window_height_key(), value);
}


ustring GeneralConfiguration::window_x_position_key ()
/*
The position of the left of the main window, in x-direction, in pixels.
*/
{
  return "window_x_pos";
}


int GeneralConfiguration::window_x_position ()
{
  return int_get (window_x_position_key(), 0);
}


void GeneralConfiguration::window_x_position_set (int value)
{
  int_set (window_x_position_key(), value);
}


ustring GeneralConfiguration::window_y_position_key ()
/*
The position of the top of the main window, in y-direction, in pixels.
*/
{
  return "window_y_pos";
}


int GeneralConfiguration::window_y_position ()
{
  return int_get (window_y_position_key(), 0);
}


void GeneralConfiguration::window_y_position_set (int value)
{
  int_set (window_y_position_key(), value);
}


ustring GeneralConfiguration::hpane_position_key ()
/*
The position, in pixels, of the grip of the main horizontal pane.
It affects the size of the Tools Area.
*/
{
  return "hpane_pos";
}


int GeneralConfiguration::hpane_position ()
{
  return int_get (hpane_position_key(), 0);
}


void GeneralConfiguration::hpane_position_set (int value)
{
  int_set (hpane_position_key(), value);
}


ustring GeneralConfiguration::vpane_editor_position_key ()
/*
The position, in pixels, of the grip of the editor's vertical pane.
It affects the size of the Text Area.
*/
{
  return "vpane_edit_pos";
}


int GeneralConfiguration::vpane_editor_position ()
{
  return int_get (vpane_editor_position_key(), 0);
}


void GeneralConfiguration::vpane_editor_position_set (int value)
{
  int_set (vpane_editor_position_key(), value);
}


ustring GeneralConfiguration::vpane_biblenotes_position_key ()
/*
The position, in pixels, of the grip of the Biblenotes's vertical pane.
It affects the size of the Bible Notes Area.
*/
{
  return "vpane_bibnote_pos";
}


int GeneralConfiguration::vpane_biblenotes_position ()
{
  return int_get (vpane_biblenotes_position_key(), 0);
}


void GeneralConfiguration::vpane_biblenotes_position_set (int value)
{
  int_set (vpane_biblenotes_position_key(), value);
}


ustring GeneralConfiguration::vpane_quick_references_position_key ()
/*
The position, in pixels, of the grip of the quick reference's vertical pane.
It affects the size of the Quick Reference Area.
*/
{
  return "vpane_qref_pos";
}


int GeneralConfiguration::vpane_quick_references_position ()
{
  return int_get (vpane_quick_references_position_key(), 0);
}


void GeneralConfiguration::vpane_quick_references_position_set (int value)
{
  int_set (vpane_quick_references_position_key(), value);
}


ustring GeneralConfiguration::tools_area_left_key ()
/*
Whether the Tools Area is at the left.
*/
{
  return "tools_area_left";
}

bool GeneralConfiguration::tools_area_left ()
{
  return bool_get (tools_area_left_key(), true);
}

void GeneralConfiguration::tools_area_left_set (bool value)
{
  bool_set (tools_area_left_key(), value);
}


ustring GeneralConfiguration::edit_note_standard_text_one_key ()
/*
Standard text 1 that can be inserted in a note.
*/
{
  return "note_std_txt1";
}


ustring GeneralConfiguration::edit_note_standard_text_one ()
{
  return string_get (edit_note_standard_text_one_key (), "");
}


void GeneralConfiguration::edit_note_standard_text_one_set (ustring value)
{
  string_set (edit_note_standard_text_one_key (), value);
}


ustring GeneralConfiguration::edit_note_standard_text_two_key ()
/*
Standard text 2 that can be inserted in a note.
*/
{
  return "note_std_txt2";
}


ustring GeneralConfiguration::edit_note_standard_text_two ()
{
  return string_get (edit_note_standard_text_two_key (), "");
}


void GeneralConfiguration::edit_note_standard_text_two_set (ustring value)
{
  string_set (edit_note_standard_text_two_key (), value);
}


ustring GeneralConfiguration::edit_note_standard_text_three_key ()
/*
Standard text 3 that can be inserted in a note.
*/
{
  return "note_std_txt3";
}


ustring GeneralConfiguration::edit_note_standard_text_three ()
{
  return string_get (edit_note_standard_text_three_key (), "");
}


void GeneralConfiguration::edit_note_standard_text_three_set (ustring value)
{
  string_set (edit_note_standard_text_three_key (), value);
}


ustring GeneralConfiguration::edit_note_standard_text_four_key ()
/*
Standard text 4 that can be inserted in a note.
*/
{
  return "note_std_txt4";
}


ustring GeneralConfiguration::edit_note_standard_text_four ()
{
  return string_get (edit_note_standard_text_four_key (), "");
}


void GeneralConfiguration::edit_note_standard_text_four_set (ustring value)
{
  string_set (edit_note_standard_text_four_key (), value);
}


ustring GeneralConfiguration::project_key ()
/*
The project that is opened.
*/
{
  return "project";
}


ustring GeneralConfiguration::project ()
{
  return string_get (project_key (), "");
}


void GeneralConfiguration::project_set (ustring value)
{
  string_set (project_key (), value);
}


ustring GeneralConfiguration::book_key ()
/*
The book that is opened.
*/
{
  return "book";
}


unsigned int GeneralConfiguration::book ()
{
  return int_get (book_key (), 0);
}


void GeneralConfiguration::book_set (unsigned int value)
{
  int_set (book_key (), value);
}


ustring GeneralConfiguration::chapter_key ()
/*
The chapter that is opened.
*/
{
  return "chapter";
}


ustring GeneralConfiguration::chapter ()
{
  return string_get (chapter_key (), "");
}


void GeneralConfiguration::chapter_set (ustring value)
{
  string_set (chapter_key (), value);
}


ustring GeneralConfiguration::verse_key ()
/*
The verse that is opened.
*/
{
  return "verse";
}


ustring GeneralConfiguration::verse ()
{
  return string_get (verse_key (), "");
}


void GeneralConfiguration::verse_set (ustring value)
{
  string_set (verse_key (), value);
}


ustring GeneralConfiguration::stylesheet_key ()
/*
The stylesheet that is opened.
*/
{
  return "stylesheet";
}


ustring GeneralConfiguration::stylesheet ()
{
  return string_get (stylesheet_key (), "");
}


void GeneralConfiguration::stylesheet_set (ustring value)
{
  string_set (stylesheet_key (), value);
}


ustring GeneralConfiguration::references_file_key ()
/*
The filename to save referencese to.
*/
{
  return "ref_file";
}


ustring GeneralConfiguration::references_file ()
{
  return string_get (references_file_key (), "");
}


void GeneralConfiguration::references_file_set (ustring value)
{
  string_set (references_file_key (), value);
}


ustring GeneralConfiguration::export_to_bibleworks_filename_key ()
/*
When exporting a project to Bibleworks, save to this file.
*/
{
  return "exp_bw_file";
}


ustring GeneralConfiguration::export_to_bibleworks_filename ()
{
  return string_get (export_to_bibleworks_filename_key (), "");
}


void GeneralConfiguration::export_to_bibleworks_filename_set (ustring value)
{
  string_set (export_to_bibleworks_filename_key (), value);
}


ustring GeneralConfiguration::export_to_sword_module_path_key ()
/*
The path to save the module to when exporting to SWORD.
*/
{
  return "exp_sword_mod";
}


ustring GeneralConfiguration::export_to_sword_module_path ()
{
  return string_get (export_to_sword_module_path_key (), g_get_home_dir ());
}


void GeneralConfiguration::export_to_sword_module_path_set (ustring value)
{
  string_set (export_to_sword_module_path_key (), value);
}


ustring GeneralConfiguration::export_to_sword_install_path_key ()
/*
The path where to install the module when exporting to SWORD.
*/
{
  return "exp_sword_inst";
}


ustring GeneralConfiguration::export_to_sword_install_path ()
{
  return string_get (export_to_sword_install_path_key (), gw_build_filename (g_get_home_dir (), ".sword"));
}


void GeneralConfiguration::export_to_sword_install_path_set (ustring value)
{
  string_set (export_to_sword_install_path_key (), value);
}


ustring GeneralConfiguration::editor_font_name_key ()
/*
The font used in the editor.
*/
{
  return "edit_font";
}


ustring GeneralConfiguration::editor_font_name ()
{
  return string_get (editor_font_name_key (), "Sans 14");
}


void GeneralConfiguration::editor_font_name_set (ustring value)
{
  string_set (editor_font_name_key (), value);
}


ustring GeneralConfiguration::paper_format_key ()
/*
Paper format to be used when printing.
*/
{
  return "paper_format";
}


ustring GeneralConfiguration::paper_format ()
{
  return string_get (paper_format_key (), "A4");
}


void GeneralConfiguration::paper_format_set (ustring value)
{
  string_set (paper_format_key (), value);
}


ustring GeneralConfiguration::paper_width_key ()
/*
The paper width, in centimeters, to be used when printing.
*/
{
  return "paper_width";
}


double GeneralConfiguration::paper_width ()
{
  return double_get (paper_width_key (), 21.0);
}


void GeneralConfiguration::paper_width_set (double value)
{
  double_set (paper_width_key (), value);
}


ustring GeneralConfiguration::paper_height_key ()
/*
The paper height, in centimeters, to be used when printing.
*/
{
  return "paper_height";
}


double GeneralConfiguration::paper_height ()
{
  return double_get (paper_height_key (), 29.7);
}


void GeneralConfiguration::paper_height_set (double value)
{
  double_set (paper_height_key (), value);
}


ustring GeneralConfiguration::paper_left_margin_key ()
/*
The left margin, in centimeters, of the paper, when printing.
*/
{
  return "paper_l_margin";
}


double GeneralConfiguration::paper_left_margin ()
{
  return double_get (paper_left_margin_key (), 2);
}


void GeneralConfiguration::paper_left_margin_set (double value)
{
  double_set (paper_left_margin_key (), value);
}


ustring GeneralConfiguration::paper_right_margin_key ()
/*
The right margin, in centimeters, of the paper, when printing.
*/
{
  return "paper_r_margin";
}


double GeneralConfiguration::paper_right_margin ()
{
  return double_get (paper_right_margin_key (), 2);
}


void GeneralConfiguration::paper_right_margin_set (double value)
{
  double_set (paper_right_margin_key (), value);
}


ustring GeneralConfiguration::paper_top_margin_key ()
/*
The top margin, in centimeters, of the paper, when printing.
*/
{
  return "paper_t_margin";
}


double GeneralConfiguration::paper_top_margin ()
{
  return double_get (paper_top_margin_key (), 2);
}


void GeneralConfiguration::paper_top_margin_set (double value)
{
  double_set (paper_top_margin_key (), value);
}


ustring GeneralConfiguration::paper_bottom_margin_key ()
/*
The bottom margin, in centimeters, of the paper, when printing.
*/
{
  return "paper_b_margin";
}


double GeneralConfiguration::paper_bottom_margin ()
{
  return double_get (paper_bottom_margin_key (), 2);
}


void GeneralConfiguration::paper_bottom_margin_set (double value)
{
  double_set (paper_bottom_margin_key (), value);
}


ustring GeneralConfiguration::notes_print_full_page_width_key ()
/*
Whether footnotes will be printed over the full width of the page.
*/
{
  return "print_note_full_width";
}


bool GeneralConfiguration::notes_print_full_page_width ()
{
  return bool_get (notes_print_full_page_width_key (), false);
}


void GeneralConfiguration::notes_print_full_page_width_set (bool value)
{
  bool_set (notes_print_full_page_width_key (), value);
}


ustring GeneralConfiguration::java_binary_key ()
/*
Full path to the Java binary.
*/
{
  return "java_bin";
}


ustring GeneralConfiguration::java_binary ()
{
  return string_get (java_binary_key (), "");
}


void GeneralConfiguration::java_binary_set (ustring value)
{
  string_set (java_binary_key (), value);
}


ustring GeneralConfiguration::xep_home_key ()
/*
The directory where XEP has been installed.
*/
{
  return "xep_home";
}


ustring GeneralConfiguration::xep_home ()
{
  return string_get (xep_home_key (), "");
}


void GeneralConfiguration::xep_home_set (ustring value)
{
  string_set (xep_home_key (), value);
}


ustring GeneralConfiguration::printdate_key ()
/*
Whether to print the current date on each page.
*/
{
  return "print_date";
}


bool GeneralConfiguration::printdate ()
{
  return bool_get (printdate_key (), true);
}


void GeneralConfiguration::printdate_set (bool value)
{
  bool_set (printdate_key (), value);
}


ustring GeneralConfiguration::print_changes_only_key ()
/*
Whether to print changes only when comparing two projects.
*/
{
  return "print_changes_only";
}


bool GeneralConfiguration::print_changes_only ()
{
  return bool_get (print_changes_only_key (), false);
}


void GeneralConfiguration::print_changes_only_set (bool value)
{
  bool_set (print_changes_only_key (), value);
}


ustring GeneralConfiguration::project_to_compare_with_key ()
/*
Which project to compare the current project with, when comparing two projects.
*/
{
  return "project_compare";
}


ustring GeneralConfiguration::project_to_compare_with ()
{
  return string_get (project_to_compare_with_key (), "");  
}


void GeneralConfiguration::project_to_compare_with_set (ustring value)
{
  string_set (project_to_compare_with_key (), value);
}


ustring GeneralConfiguration::show_notes_selector_key ()
/*
A selector that decides which set of notes to show.
*/
{
  return "show_note_select";
}


int GeneralConfiguration::show_notes_selector ()
{
  return int_get (show_notes_selector_key (), 0);
}


void GeneralConfiguration::show_notes_selector_set (int value)
{
  int_set (show_notes_selector_key (), value);
}


ustring GeneralConfiguration::show_notes_for_current_project_only_key ()
/*
Whether to show the notes for the current project only.
*/
{
  return "show_note_curr_proj_only";
}


bool GeneralConfiguration::show_notes_for_current_project_only ()
{
  return bool_get (show_notes_for_current_project_only_key (), false);
}


void GeneralConfiguration::show_notes_for_current_project_only_set (bool value)
{
  bool_set (show_notes_for_current_project_only_key (), value);
}


ustring GeneralConfiguration::notes_show_project_key ()
/*
Whether to show the project with each note.
*/
{
  return "note_show_proj";
}


bool GeneralConfiguration::notes_show_project ()
{
  return bool_get (notes_show_project_key (), false);
}


void GeneralConfiguration::notes_show_project_set (bool value)
{
  bool_set (notes_show_project_key (), value);
}


ustring GeneralConfiguration::notes_sort_round_ref_key ()
/*
Sort the notes so that the most relevant ones are near the cursor.
*/
{
  return "sort_notes_round_ref";
}


bool GeneralConfiguration::notes_sort_round_ref ()
{
  return bool_get (notes_sort_round_ref_key (), false);
}


void GeneralConfiguration::notes_sort_round_ref_set (bool value)
{
  bool_set (notes_sort_round_ref_key (), value);
}


ustring GeneralConfiguration::show_notes_category_key ()
/*
Which category of notes to show.
*/
{
  return "show_note_cat";
}


ustring GeneralConfiguration::show_notes_category ()
{
  return string_get (show_notes_category_key (), "");
}


void GeneralConfiguration::show_notes_category_set (ustring value)
{
  string_set (show_notes_category_key (), value);
}


ustring GeneralConfiguration::show_notes_date_range_from_key ()
/*
Show notes edited from this date and on.
*/
{
  return "show_note_date_from";
}


int GeneralConfiguration::show_notes_date_range_from ()
{
  return int_get (show_notes_date_range_from_key (), date_time_julian_day_get_current());
}


void GeneralConfiguration::show_notes_date_range_from_set (int value)
{
  int_set (show_notes_date_range_from_key (), value);
}


ustring GeneralConfiguration::show_notes_date_range_to_key ()
/*
Show notes edited until this date.
*/
{
  return "show_note_date_to";
}


int GeneralConfiguration::show_notes_date_range_to ()
{
  return int_get (show_notes_date_range_to_key (), date_time_julian_day_get_current());
}


void GeneralConfiguration::show_notes_date_range_to_set (int value)
{
  int_set (show_notes_date_range_to_key (), value);
}


ustring GeneralConfiguration::bibleworks_executable_key ()
/*
The path of the BibleWorks executable.
*/
{
  return "bw_exe";
}


ustring GeneralConfiguration::bibleworks_executable ()
{
  return string_get (bibleworks_executable_key (), "bw700.exe");
}


void GeneralConfiguration::bibleworks_executable_set (ustring value)
{
  string_set (bibleworks_executable_key (), value);
}


ustring GeneralConfiguration::reference_exchange_send_to_bibleworks_key ()
/*
Whether to send references to BibleWorks.
*/
{
  return "ref_to_bw";
}


bool GeneralConfiguration::reference_exchange_send_to_bibleworks ()
{
  return bool_get (reference_exchange_send_to_bibleworks_key (), false);
}


void GeneralConfiguration::reference_exchange_send_to_bibleworks_set (bool value)
{
  bool_set (reference_exchange_send_to_bibleworks_key (), value);
}


ustring GeneralConfiguration::reference_exchange_receive_from_bibleworks_key ()
/*
Whether to receive references from BibleWorks.
*/
{
  return "ref_from_bw";
}


bool GeneralConfiguration::reference_exchange_receive_from_bibleworks ()
{
  return bool_get (reference_exchange_receive_from_bibleworks_key (), false);
}


void GeneralConfiguration::reference_exchange_receive_from_bibleworks_set (bool value)
{
  bool_set (reference_exchange_receive_from_bibleworks_key (), value);
}


ustring GeneralConfiguration::reference_exchange_send_to_bibletime_key ()
/*
Whether to send references to BibleTime.
*/
{
  return "ref_to_bt";
}


bool GeneralConfiguration::reference_exchange_send_to_bibletime ()
{
  return bool_get (reference_exchange_send_to_bibletime_key (), false);
}


void GeneralConfiguration::reference_exchange_send_to_bibletime_set (bool value)
{
  bool_set (reference_exchange_send_to_bibletime_key (), value);
}


ustring GeneralConfiguration::reference_exchange_receive_from_bibletime_key ()
/*
Whether to receive references from BibleTime.
*/
{
  return "ref_from_bt";
}


bool GeneralConfiguration::reference_exchange_receive_from_bibletime ()
{
  return bool_get (reference_exchange_receive_from_bibletime_key (), false);
}


void GeneralConfiguration::reference_exchange_receive_from_bibletime_set (bool value)
{
  bool_set (reference_exchange_receive_from_bibletime_key (), value);
}


ustring GeneralConfiguration::reference_exchange_send_to_santafefocus_key ()
/*
Whether to send references to SantaFeFocus enabled programs.
*/
{
  return "ref_to_sff";
}


bool GeneralConfiguration::reference_exchange_send_to_santafefocus ()
{
  return bool_get (reference_exchange_send_to_santafefocus_key (), true);
}


void GeneralConfiguration::reference_exchange_send_to_santafefocus_set (bool value)
{
  bool_set (reference_exchange_send_to_santafefocus_key (), value);
}


ustring GeneralConfiguration::reference_exchange_receive_from_santafefocus_key ()
/*
Whether to receive references from SantaFeFocus enabled programs.
*/
{
  return "ref_from_sff";
}


bool GeneralConfiguration::reference_exchange_receive_from_santafefocus ()
{
  return bool_get (reference_exchange_receive_from_santafefocus_key (), false);
}


void GeneralConfiguration::reference_exchange_receive_from_santafefocus_set (bool value)
{
  bool_set (reference_exchange_receive_from_santafefocus_key (), value);
}


ustring GeneralConfiguration::check_markers_compare_project_key ()
/*
When comparing USFM markers, which project to compare with.
*/
{
  return "chk_mark_comp_with";
}


ustring GeneralConfiguration::check_markers_compare_project ()
{
  return string_get (check_markers_compare_project_key (), "");
}


void GeneralConfiguration::check_markers_compare_project_set (ustring value)
{
  string_set (check_markers_compare_project_key (), value);
}


ustring GeneralConfiguration::check_markers_compare_all_markers_key ()
/*
When comparing USFM markers, whether to compare all markers.
*/
{
  return "chk_mark_comp_all";
}


bool GeneralConfiguration::check_markers_compare_all_markers ()
{
  return bool_get (check_markers_compare_all_markers_key (), true);
}


void GeneralConfiguration::check_markers_compare_all_markers_set (bool value)
{
  bool_set (check_markers_compare_all_markers_key (), value);
}


ustring GeneralConfiguration::check_markers_compare_include_only_key ()
/*
When comparing USFM markers, which markers to include only.
*/
{
  return "chk_mark_comp_only";
}


ustring GeneralConfiguration::check_markers_compare_include_only ()
{
  return string_get (check_markers_compare_include_only_key (), "");
}


void GeneralConfiguration::check_markers_compare_include_only_set (ustring value)
{
  string_set (check_markers_compare_include_only_key (), value);
}


ustring GeneralConfiguration::check_markers_compare_ignore_key ()
/*
When comparing USFM markers, which markers to ignore.
*/
{
  return "chk_mark_comp_ign";
}


ustring GeneralConfiguration::check_markers_compare_ignore ()
{
  return string_get (check_markers_compare_ignore_key (), "");
}


void GeneralConfiguration::check_markers_compare_ignore_set (ustring value)
{
  string_set (check_markers_compare_ignore_key (), value);
}


ustring GeneralConfiguration::check_markers_compare_ignore_verse_zero_key ()
/*
When comparing USFM markers, whether to ignore anything in verse zero.
*/
{
  return "chk_mark_ign_v0";
}


bool GeneralConfiguration::check_markers_compare_ignore_verse_zero ()
{
  return bool_get (check_markers_compare_ignore_verse_zero_key (), false);
}


void GeneralConfiguration::check_markers_compare_ignore_verse_zero_set (bool value)
{
  bool_set (check_markers_compare_ignore_verse_zero_key (), value);
}


ustring GeneralConfiguration::check_capitalization_punctuation_key ()
/*
When checking capitalization, this gives the punctuation to be followed by a
capitalized letter.
*/
{
  return "chk_cap_punct";
}


ustring GeneralConfiguration::check_capitalization_punctuation ()
{
  return string_get (check_capitalization_punctuation_key (), CAPITALIZATION_PUNCTUATION);
}


void GeneralConfiguration::check_capitalization_punctuation_set (ustring value)
{
  string_set (check_capitalization_punctuation_key (), value);
}


ustring GeneralConfiguration::check_capitalization_ignore_key ()
/*
When checking capitalization, this lists characters after which a lowercase 
letter is acceptable without giving an error message.
*/
{
  return "chk_cap_ign";
}


ustring GeneralConfiguration::check_capitalization_ignore ()
{
  return string_get (check_capitalization_ignore_key (), CAPITALIZATION_LOWERCASE);
}


void GeneralConfiguration::check_capitalization_ignore_set (ustring value)
{
  string_set (check_capitalization_ignore_key (), value);
}


ustring GeneralConfiguration::check_capitalization_allow_any_prefixes_key ()
/*
When checking capitalization, whether to allow any prefixes.
*/
{
  return "chk_cap_any_pref";
}


bool GeneralConfiguration::check_capitalization_allow_any_prefixes ()
{
  return bool_get (check_capitalization_allow_any_prefixes_key (), false);
}


void GeneralConfiguration::check_capitalization_allow_any_prefixes_set (bool value)
{
  bool_set (check_capitalization_allow_any_prefixes_key (), value);
}


ustring GeneralConfiguration::check_repetition_ignore_case_key ()
/*
When checking repeating words, whether to ignore the case.
*/
{
  return "chk_repeat_ign_case";
}


bool GeneralConfiguration::check_repetition_ignore_case ()
{
  return bool_get (check_repetition_ignore_case_key (), false);
}


void GeneralConfiguration::check_repetition_ignore_case_set (bool value)
{
  bool_set (check_repetition_ignore_case_key (), value);
}


ustring GeneralConfiguration::check_repetition_show_only_these_key ()
/*
When checking repeating words, whether to show only certain words.
*/
{
  return "chk_repeat_only_these";
}


bool GeneralConfiguration::check_repetition_show_only_these ()
{
  return bool_get (check_repetition_show_only_these_key (), false);
}


void GeneralConfiguration::check_repetition_show_only_these_set (bool value)
{
  bool_set (check_repetition_show_only_these_key (), value);
}


ustring GeneralConfiguration::check_repetition_ignore_these_key ()
/*
When checking repeating words, whether to ignore certain words.
*/
{
  return "chk_repeat_ign_these";
}


bool GeneralConfiguration::check_repetition_ignore_these ()
{
  return bool_get (check_repetition_ignore_these_key (), false);
}


void GeneralConfiguration::check_repetition_ignore_these_set (bool value)
{
  bool_set (check_repetition_ignore_these_key (), value);
}


ustring GeneralConfiguration::check_matching_pairs_ignore_key ()
/*
When checking matching pairs of punctuation, which punctuation to ignore.
*/
{
  return "chk_pairs_ign";
}


ustring GeneralConfiguration::check_matching_pairs_ignore ()
{
  return string_get (check_matching_pairs_ignore_key (), "");
}


void GeneralConfiguration::check_matching_pairs_ignore_set (ustring value)
{
  string_set (check_matching_pairs_ignore_key (), value);
}


ustring GeneralConfiguration::check_words_inventory_not_include_words_count_key ()
/*
When making a word inventory, words with a count of n and up are not given.
*/
{
  return "chk_wrd_inv_excl_count";
}


int GeneralConfiguration::check_words_inventory_not_include_words_count ()
{
  return int_get (check_words_inventory_not_include_words_count_key (), 0);
}


void GeneralConfiguration::check_words_inventory_not_include_words_count_set (int value)
{
  int_set (check_words_inventory_not_include_words_count_key (), value);
}


ustring GeneralConfiguration::check_words_inventory_word_forming_characters_key ()
/*
When making a word inventory, these are extra characters that form a word.
*/
{
  return "chk_wrd_extra_chars";
}


ustring GeneralConfiguration::check_words_inventory_word_forming_characters ()
{
  return string_get (check_words_inventory_word_forming_characters_key (), "");
}


void GeneralConfiguration::check_words_inventory_word_forming_characters_set (ustring value)
{
  string_set (check_words_inventory_word_forming_characters_key (), value);
}


ustring GeneralConfiguration::check_markers_spacing_include_key ()
/*
When checking spacing of USFM markers, which markers to include.
*/
{
  return "chk_mark_spac";
}


ustring GeneralConfiguration::check_markers_spacing_include ()
{
  return string_get (check_markers_spacing_include_key (), "xo");
}


void GeneralConfiguration::check_markers_spacing_include_set (ustring value)
{
  string_set (check_markers_spacing_include_key (), value);
}


ustring GeneralConfiguration::tools_area_page_number_key ()
/*
The number of the page now being displayed in the Tools Area.
*/
{
  return "tools_area_page";
}


int GeneralConfiguration::tools_area_page_number ()
{
  return int_get (tools_area_page_number_key (), 0);
}


void GeneralConfiguration::tools_area_page_number_set (int value)
{
  int_set (tools_area_page_number_key (), value);
}


ustring GeneralConfiguration::styles_category_expanded_key ()
/*
Which categories in the Styles Area are expanded.
*/
{
  return "style_cat_exp";
}


vector<bool> GeneralConfiguration::styles_category_expanded ()
{
  return vector_bool_get (styles_category_expanded_key ());
}


void GeneralConfiguration::styles_category_expanded_set (vector<bool> value)
{
  vector_bool_set (styles_category_expanded_key (), value);
}


ustring GeneralConfiguration::insert_footnote_template_key ()
/*
Current template for inserting a footnote.
*/
{
  return "ins_footnote_templ";
}


ustring GeneralConfiguration::insert_footnote_template ()
{
  return string_get (insert_footnote_template_key (), "");
}


void GeneralConfiguration::insert_footnote_template_set (ustring value)
{
  string_set (insert_footnote_template_key (), value);
}


ustring GeneralConfiguration::insert_endnote_template_key ()
/*
Current template for inserting an endnote.
*/
{
  return "ins_endnote_templ";
}


ustring GeneralConfiguration::insert_endnote_template ()
{
  return string_get (insert_endnote_template_key (), "");
}


void GeneralConfiguration::insert_endnote_template_set (ustring value)
{
  string_set (insert_endnote_template_key (), value);
}


ustring GeneralConfiguration::insert_xref_template_key ()
/*
Current template for inserting a crossreference.
*/
{
  return "ins_xref_templ";
}


ustring GeneralConfiguration::insert_xref_template ()
{
  return string_get (insert_xref_template_key (), "");
}


void GeneralConfiguration::insert_xref_template_set (ustring value)
{
  string_set (insert_xref_template_key (), value);
}


ustring GeneralConfiguration::parallel_bible_keep_verses_together_key ()
/*
When printing a Parallel Bible, whether to keep the verses together on one page.
*/
{
  return "par_bible_keep_vs";
}


bool GeneralConfiguration::parallel_bible_keep_verses_together ()
{
  return bool_get (parallel_bible_keep_verses_together_key (), true);
}


void GeneralConfiguration::parallel_bible_keep_verses_together_set (bool value)
{
  bool_set (parallel_bible_keep_verses_together_key (), value);
}


ustring GeneralConfiguration::parallel_bible_chapters_verses_key ()
/*
When printing a Parallel Bible, which chapters and verses, i.e., which portion 
to print.
*/
{
  return "par_bible_ch_vs";
}


ustring GeneralConfiguration::parallel_bible_chapters_verses ()
{
  return string_get (parallel_bible_chapters_verses_key (), CHAPTER_VERSE_SELECTION_ALL);
}


void GeneralConfiguration::parallel_bible_chapters_verses_set (ustring value)
{
  string_set (parallel_bible_chapters_verses_key (), value);
}


ustring GeneralConfiguration::parallel_bible_include_verse_zero_key ()
/*
When printing a Parallel Bible, whether to include verse 0.
*/
{
  return "par_bible_vs0";
}


bool GeneralConfiguration::parallel_bible_include_verse_zero ()
{
  return bool_get (parallel_bible_include_verse_zero_key (), false);
}


void GeneralConfiguration::parallel_bible_include_verse_zero_set (bool value)
{
  bool_set (parallel_bible_include_verse_zero_key (), value);
}


ustring GeneralConfiguration::printing_fonts_key ()
/*
When printing something, this is the list of fonts to use.
*/
{
  return "print_fonts2";
}


vector<ustring> GeneralConfiguration::printing_fonts ()
{
  return vector_string_get (printing_fonts_key ());
}


void GeneralConfiguration::printing_fonts_set (vector<ustring> value)
{
  vector_string_set (printing_fonts_key (), value);
}


ustring GeneralConfiguration::parallel_bible_projects_key ()
/*
The projects to include in a parallel Bible.
*/
{
  return "par_bible_prj";
}


vector<ustring> GeneralConfiguration::parallel_bible_projects ()
{
  return vector_string_get (parallel_bible_projects_key ());
}


void GeneralConfiguration::parallel_bible_projects_set (vector<ustring> value)
{
  vector_string_set (parallel_bible_projects_key (), value);
}


ustring GeneralConfiguration::use_outpost_key ()
/*
Whether to use the Windows Outpost.
*/
{
  return "use_outpost";
}


bool GeneralConfiguration::use_outpost ()
{
  return bool_get (use_outpost_key (), true);
}


void GeneralConfiguration::use_outpost_set (bool value)
{
  bool_set (use_outpost_key (), value);
}


ustring GeneralConfiguration::outpost_networked_key ()
/*
Whether Windows Outpost is on the network.
*/
{
  return "outpost_networked";
}


bool GeneralConfiguration::outpost_networked ()
{
  return bool_get (outpost_networked_key (), false);
}


void GeneralConfiguration::outpost_networked_set (bool value)
{
  bool_set (outpost_networked_key (), value);
}


ustring GeneralConfiguration::outpost_host_key ()
/*
The host on which the Windows Outpost runs.
*/
{
  return "outpost_host";
}


ustring GeneralConfiguration::outpost_host ()
{
  return string_get (outpost_host_key (), "");
}


void GeneralConfiguration::outpost_host_set (ustring value)
{
  string_set (outpost_host_key (), value);
}


ustring GeneralConfiguration::outpost_path_key ()
/*
The path to the Windows Outpost.
*/
{
  return "outpost_path";
}


ustring GeneralConfiguration::outpost_path ()
{
  return string_get (outpost_path_key (), gw_build_filename (g_get_home_dir (), ".wine/drive_c/Program Files/Bibledit Windows Outpost/bwoutpost.exe"));
}


void GeneralConfiguration::outpost_path_set (ustring value)
{
  string_set (outpost_path_key (), value);
}


ustring GeneralConfiguration::wine_path_key ()
/*
The path to the Wine binary.
*/
{
  return "wine_path";
}


ustring GeneralConfiguration::wine_path ()
{
  return string_get (wine_path_key (), "wine");
}


void GeneralConfiguration::wine_path_set (ustring value)
{
  string_set (wine_path_key (), value);
}


ustring GeneralConfiguration::outpost_command_key ()
/*
The path to the Wine binary.
*/
{
  return "outpost_command";
}


ustring GeneralConfiguration::outpost_command ()
{
  return string_get (outpost_command_key (), wine_path () + shell_quote_space (outpost_path ()));
}


void GeneralConfiguration::outpost_command_set (ustring value)
{
  string_set (outpost_command_key (), value);
}


ustring GeneralConfiguration::mychecks_key ()
/*
My checks.
*/
{
  return "mychecks";
}


ustring GeneralConfiguration::mychecks ()
{
  return string_get (mychecks_key (), "");
}


void GeneralConfiguration::mychecks_set (ustring value)
{
  string_set (mychecks_key (), value);
}


ustring GeneralConfiguration::tidy_translate_key ()
/*
Whether Tidy should translate book abbreviations.
*/
{
  return "tidy_translate";
}


bool GeneralConfiguration::tidy_translate ()
{
  return bool_get (tidy_translate_key (), false);
}


void GeneralConfiguration::tidy_translate_set (bool value)
{
  bool_set (tidy_translate_key (), value);
}


ustring GeneralConfiguration::tidy_books_key ()
/*
List of ids of the book abbreviations for tidying them.
*/
{
  return "tidy_books";
}


vector<int> GeneralConfiguration::tidy_books ()
{
  return vector_int_get (tidy_books_key ());
}


void GeneralConfiguration::tidy_books_set (vector<int> value)
{
  vector_int_set (tidy_books_key (), value);
}


ustring GeneralConfiguration::tidy_texts_key ()
/*
List of texts for tidying book abbreviations.
*/
{
  return "tidy_texts";
}


vector<ustring> GeneralConfiguration::tidy_texts ()
{
  return vector_string_get (tidy_texts_key ());
}


void GeneralConfiguration::tidy_texts_set (vector<ustring> value)
{
  vector_string_set (tidy_texts_key (), value);
}


ustring GeneralConfiguration::tidy_normalize_hyphens_key ()
/*
Whether Tidy should normalize hyphens.
*/
{
  return "tidy_n_hyphens";
}


bool GeneralConfiguration::tidy_normalize_hyphens ()
{
  return bool_get (tidy_normalize_hyphens_key (), false);
}


void GeneralConfiguration::tidy_normalize_hyphens_set (bool value)
{
  bool_set (tidy_normalize_hyphens_key (), value);
}


ustring GeneralConfiguration::tidy_space_between_chapter_verse_key ()
/*
Whether Tidy should remove the space beteeen chapter and verse.
*/
{
  return "tidy_space_btw_ch_vs";
}


bool GeneralConfiguration::tidy_space_between_chapter_verse ()
{
  return bool_get (tidy_space_between_chapter_verse_key (), false);
}


void GeneralConfiguration::tidy_space_between_chapter_verse_set (bool value)
{
  bool_set (tidy_space_between_chapter_verse_key (), value);
}


ustring GeneralConfiguration::tidy_space_series_verses_key ()
/*
Whether Tidy should remove the space from a series of verses.
*/
{
  return "tidy_space_verses";
}


bool GeneralConfiguration::tidy_space_series_verses ()
{
  return bool_get (tidy_space_series_verses_key (), false);
}


void GeneralConfiguration::tidy_space_series_verses_set (bool value)
{
  bool_set (tidy_space_series_verses_key (), value);
}


ustring GeneralConfiguration::tidy_full_stop_ends_text_key ()
/*
Whether Tidy should ensure that text is ended by a full stop.
*/
{
  return "tidy_end_full_stop";
}


bool GeneralConfiguration::tidy_full_stop_ends_text ()
{
  return bool_get (tidy_full_stop_ends_text_key (), false);
}


void GeneralConfiguration::tidy_full_stop_ends_text_set (bool value)
{
  bool_set (tidy_full_stop_ends_text_key (), value);
}


ustring GeneralConfiguration::tidy_ampersand_semicolon_key ()
/*
Whether Tidy replace the ampersand with a semicolon in e.g. Mat. 10.1 & 11.2.
*/
{
  return "tidy_amp_sem";
}


bool GeneralConfiguration::tidy_ampersand_semicolon ()
{
  return bool_get (tidy_ampersand_semicolon_key (), false);
}


void GeneralConfiguration::tidy_ampersand_semicolon_set (bool value)
{
  bool_set (tidy_ampersand_semicolon_key (), value);
}


ustring GeneralConfiguration::tidy_space_before_punctuation_key ()
/*
Whether Tidy should remove a space before a variety of punctuation.
*/
{
  return "tidy_rem_sp_punct";
}


bool GeneralConfiguration::tidy_space_before_punctuation ()
{
  return bool_get (tidy_space_before_punctuation_key (), false);
}


void GeneralConfiguration::tidy_space_before_punctuation_set (bool value)
{
  bool_set (tidy_space_before_punctuation_key (), value);
}
