/*  Audacious-DUMB Plug-in to enable audacious to use DUMB for IT/XM/S3M/MOD files
 *  Copyright (C) 2006-2007 Christian Birchinger
 *  Audacious port based on previous plugins for XMMS and BMP
 *  Copyright (C) 2002-2003  Ben Davis
 *  Incorporates code from the wav plug-in,
 *  Copyright (C) 1998-2000  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "audacious-dumb.h"

#include <gtk/gtk.h>
#include "interface.h"
#include "support.h"

DuhFile *duh_file = NULL;
static gboolean disable_amiga_mods = FALSE;
static const gchar duh_custom_title_format[] = "${file-name}${?title: - ${title}}";
static gint output_frequency;
static const gint output_frequencies[]={22050, 44100, 48000, 64000, 96000, 192000};
static GtkWidget *prefs;
static GMutex *dumblock;

static GMutex *control_mutex;
static GCond *control_cond;
static gboolean stop_flag = FALSE;

static const gchar *fmts[] = { "mod", "s3m", "it", "xm", "duh", NULL };

InputPlugin duh_ip = {
    .description = "DUMB Plugin " VERSION,
    .init = duh_init,
    .cleanup = duh_cleanup,
    .about = duh_about,
    .configure = duh_configure,
    .play = play_start,
    .stop = stop,
    .pause = duh_pause,
    .file_info_box = file_info_box,
    .is_our_file_from_vfs = is_our_file_from_vfs,
    .vfs_extensions = fmts,
    .mseek = mseek,
    .probe_for_tuple = duh_probe_for_tuple,
    .priority = 0 /* Would be nice to be above Modplug but minus values don't work */ 
};

InputPlugin *duh_iplist[] = { &duh_ip, NULL };

SIMPLE_INPUT_PLUGIN(duh_ip, duh_iplist)

static gboolean duh_init(void)
{
	mcs_handle_t *db;
	db = aud_cfg_db_open();

	aud_cfg_db_get_bool(db, "DUMB", "disable_amiga_mods", &disable_amiga_mods);
	if (!disable_amiga_mods) /* force db creation since defined false equals none-existant */
		aud_cfg_db_set_bool(db, "DUMB", "disable_amiga_mods", FALSE);

	aud_cfg_db_get_int(db, "DUMB", "output_frequency", &output_frequency);	
	if (output_frequency < 22050 || output_frequency > 192000) {
		output_frequency = 44100;
		aud_cfg_db_set_int(db, "DUMB", "output_frequency", output_frequency);
	}
		
	aud_cfg_db_close(db);

	/* dumb_register_stdfiles(); */
	/*  Seems only needed when opening files with dumb directly * 
	 *  and not when VFS puts them to a memory location first   */
	
	dumb_it_max_to_mix = 256;
	
	dumblock = g_mutex_new();
	control_mutex = g_mutex_new();
	control_cond = g_cond_new();
	
	return TRUE;
}

static void duh_cleanup(void)
{
	g_mutex_free(dumblock);
	g_mutex_free(control_mutex);
	g_cond_free(control_cond);  
}	

static int is_our_file_from_vfs(const gchar *filename, VFSFile *file)
{
	gchar magic[4];
	const gchar *ext;

	g_return_val_if_fail(filename != NULL, FALSE);
	g_return_val_if_fail(file != NULL, FALSE);

	vfs_fread(magic, 1, 4, file);
	if (!memcmp(magic, XM_MAGIC, 4))
                return TRUE;
	if (!memcmp(magic, IT_MAGIC, 4))
		return TRUE;

	vfs_fseek(file, 44, SEEK_SET);
	vfs_fread(magic, 1, 4, file);
	if (!memcmp(magic, S3M_MAGIC, 4))
		return TRUE;

	vfs_fseek(file, 1080, SEEK_SET);
	vfs_fread(magic, 1, 4, file);   
	if (!memcmp(magic, MOD_MAGIC_FASTTRACKER6, 4))
		return TRUE;
	if (!memcmp(magic, MOD_MAGIC_FASTTRACKER8, 4))
		return TRUE;

/*
	printf("Is %s our file?\n", filename);

	if (disable_amiga_mods) {
		printf("Amiga support is disabled\n");
	}
*/
	if (!disable_amiga_mods) {
		if (!memcmp(magic, MOD_MAGIC_PROTRACKER4, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_PROTRACKER4X, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_NOISETRACKER, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_STARTRACKER4, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_STARTRACKER8, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_STARTRACKER4X, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_STARTRACKER8X, 4))
			return TRUE;
		if (!memcmp(magic, MOD_MAGIC_FASTTRACKER4, 4))
			return TRUE;
	}

	ext = strrchr(filename, '.');
	if (ext) {
		if (!strcasecmp(ext, ".duh")) return TRUE;
		if (!strcasecmp(ext, ".it")) return TRUE;
		if (!strcasecmp(ext, ".xm")) return TRUE;
		if (!strcasecmp(ext, ".s3m")) return TRUE;
		if (!disable_amiga_mods) {
			if (!strcasecmp(ext, ".mod")) return TRUE;
		}
	}

	return FALSE;
}

static gint duh_universal_load_vfs(DUH **duh, const char *filename, VFSFile *extfd, const short testmask)
{
	DUMBFILE *filedumb;

	gint ext_index = -1;
	VFSFile *fd = NULL;
	off_t filesize, readsize;
	gchar *filemap;
	
	*duh = NULL;

	if (extfd)
		fd = extfd;
	else {
		fd = vfs_fopen(filename, "rb");
		g_return_val_if_fail(fd != NULL, -1);
	}

	filesize = vfs_fsize(fd);
	g_return_val_if_fail(filesize > 0, -1);
	
	filemap = malloc(filesize);
	g_return_val_if_fail(filemap != NULL, -1);

	readsize = vfs_fread(filemap, 1, filesize, fd);

	if (!extfd)
		vfs_fclose(fd);

	if (readsize == 0) {
		g_warning("audacious-dumb: Couldn't read from %s", filename);
		free(filemap); filemap = NULL;
		return -1;
	}

	/*
	   The close and re-opening after reach read attempt is needed.

	   Quote from the documentation:

	   WARNING: The behaviour of these functions are undefined if you pass a
	   DUMBFILE from which data have already been read; they are likely
	   not to work. This oversight will be fixed in future releases.
	*/

	int count, total_count = sizeof(uniread) / sizeof(uniread_struct_t);

	for (count = 0; count < total_count; count++) {
		if (uniread[count].testmask & testmask) {
			g_mutex_lock(dumblock);
			filedumb = dumbfile_open_memory(filemap, readsize);
			*duh = uniread[count].read(filedumb);
			dumbfile_close(filedumb);
			g_mutex_unlock(dumblock);
			if (*duh) {
				ext_index = count;
				break;
			}
		}
	}

	free(filemap); filemap = NULL;

	return ext_index;
}

static Tuple *get_tuple_info_from_duh(DUH *duh, const gchar *filename)
{
	const gchar *mod_title_raw;
	gchar *mod_title;

	g_return_val_if_fail(filename != NULL, NULL);
	g_return_val_if_fail(duh != NULL, NULL);

	Tuple *tuple = tuple_new_from_filename(filename);

	tuple_associate_string(tuple, FIELD_FORMATTER, NULL, duh_custom_title_format);
	tuple_associate_string(tuple, FIELD_QUALITY, NULL, "sequenced");
	tuple_associate_int(tuple, FIELD_LENGTH, NULL, (int)(duh_get_length(duh) * 1000LL >> 16) );
	
	mod_title_raw = duh_get_tag(duh, "TITLE");
	if (mod_title_raw) {
		/* DOS (CP850) all my XM IT and S3M use this (MODs should be 7bit ASCII) */ 
		mod_title = g_convert(mod_title_raw, -1, "UTF-8", "CP850", NULL, NULL, NULL);
		/* Raw copy but GTK2 doesn't like none-UTF8 stuff*/ 
		/* mod_title = g_strdup(mod_title_raw); */

		g_strstrip(mod_title);

		if (mod_title[0])
			tuple_associate_string(tuple, FIELD_TITLE, NULL, mod_title);

		g_free(mod_title); mod_title = NULL;
	}
	mod_title_raw = NULL;

	return tuple;
}

static Tuple *duh_probe_for_tuple(const gchar *filename, VFSFile *fd)
{
	Tuple *input = NULL;
	DUH *duh = NULL;
	int ext_index;

	g_return_val_if_fail(filename != NULL, NULL);
	g_return_val_if_fail(fd != NULL, NULL);

	if (!is_our_file_from_vfs(filename, fd))
		return NULL;
		
	vfs_rewind(fd);

	ext_index = duh_universal_load_vfs(&duh, filename, fd, UNIREAD_ALL);
	if (ext_index == -1)
		return NULL; /* Return empty Tuple if module was not loadable */
	
	input = get_tuple_info_from_duh(duh, filename);
	tuple_associate_string(input, FIELD_CODEC, NULL, uniread[ext_index].description);	
		
	return input;
}

static void install_callbacks(DUH_SIGRENDERER *sr)
{
	DUMB_IT_SIGRENDERER *itsr = duh_get_it_sigrenderer(sr);
	dumb_it_set_loop_callback(itsr, &dumb_it_callback_terminate, NULL);
	dumb_it_set_xm_speed_zero_callback(itsr, &dumb_it_callback_terminate, NULL);
}

static gboolean play_start(InputPlayback *playback, const gchar *filename,
 VFSFile *file, gint start_time, gint stop_time, gboolean pause)
{
	Tuple *tuple;
	int rate;

	const int buffer_size = 16 * 1024;
	char *buffer;
	int actual_read, render_size;
	float delta, volume;

	if (file == NULL)
		return FALSE;

	duh_file = g_malloc(sizeof (DuhFile));
	memset(duh_file, 0, sizeof (DuhFile));

	duh_universal_load_vfs(&duh_file->duh, filename, file, UNIREAD_ALL);
	if (!duh_file->duh) {
		g_warning("audacious-dumb: Unable to play %s", filename);
		g_free(duh_file);
		duh_file = NULL;
		return FALSE;
	}

	if ((duh_file->file = duh_start_sigrenderer(duh_file->duh, 0, 2, 0)))
	{
		install_callbacks(duh_file->file);
		duh_file->channels = 2;
		duh_file->samples_per_sec = output_frequency;
		duh_file->bits_per_sample = 16;

		if (playback->output->open_audio((duh_file->bits_per_sample == 16) ? FMT_S16_LE : FMT_U8, duh_file->samples_per_sec, duh_file->channels) == 0)
		{
			duh_end_sigrenderer(duh_file->file);
			unload_duh(duh_file->duh);
			g_free(duh_file);
			duh_file = NULL;
			return FALSE;
		}

		if (pause)
			playback->output->pause(TRUE);
		
		rate = duh_file->samples_per_sec * duh_file->channels * (duh_file->bits_per_sample / 8);
		tuple = get_tuple_info_from_duh(duh_file->duh, filename);
		playback->set_tuple(playback, tuple);
		playback->set_params(playback, 8 * rate, duh_file->samples_per_sec, duh_file->channels);

		g_mutex_lock(control_mutex);
		duh_file->seek_to = (start_time > 0) ? start_time : -1;
		stop_flag = FALSE;
		playback->set_pb_ready(playback);
		g_mutex_unlock(control_mutex);

		/* Values taken from the dumbout.c example */
		delta = 65536.0f / duh_file->samples_per_sec;
		volume = 1.0f;

		buffer = g_malloc(buffer_size);

		/* The docs say output buffer equals render_size * channels (2)
		 * but it seems not enough so we're keeping the old calculated value (4)
		 */
		render_size = buffer_size / ((duh_file->bits_per_sample / 8) * duh_file->channels);

		while (!stop_flag)
		{
			g_mutex_lock(control_mutex);
			if (duh_file->seek_to != -1)
			{
				duh_end_sigrenderer(duh_file->file);
				duh_file->file = duh_start_sigrenderer(duh_file->duh, 0, duh_file->channels, (unsigned long)((gint64)duh_file->seek_to << 16) / 1000L);
				install_callbacks(duh_file->file);
				playback->output->flush(duh_file->seek_to); 
				duh_file->seek_to = -1;
				g_cond_signal(control_cond);
			}

			g_mutex_unlock(control_mutex);

			if (!duh_file->eof)
			{
				actual_read = duh_render(duh_file->file, duh_file->bits_per_sample, 0, volume, delta, render_size, buffer);
				actual_read *= (duh_file->bits_per_sample / 8) * duh_file->channels;
	
				if (actual_read == 0)
					duh_file->eof = TRUE;
				else
				{
					if (!stop_flag && duh_file->seek_to == -1)
						playback->output->write_audio(buffer, actual_read);
				}
			}
			else
			{
				while (!stop_flag && playback->output->buffer_playing())
					g_usleep(20000);
			
				break;
			} 
		}

		playback->output->close_audio();

		g_mutex_lock(control_mutex);
		stop_flag = TRUE;
		duh_file->eof = TRUE;
		g_mutex_unlock(control_mutex);

		duh_end_sigrenderer(duh_file->file);
		unload_duh(duh_file->duh);
		g_free(buffer);
		g_free(duh_file);
		duh_file = NULL;

		return TRUE;
	}
	return FALSE;
}

static void stop(InputPlayback *playback)
{
	g_mutex_lock(control_mutex);

	if (!stop_flag)
	{
		stop_flag = TRUE;
		playback->output->abort_write();
		g_cond_signal(control_cond);
	}

	g_mutex_unlock (control_mutex);
}

static void duh_pause(InputPlayback *playback, gboolean pause)
{
	g_mutex_lock(control_mutex);

	if (!stop_flag)
		playback->output->pause(pause);
            
	g_mutex_unlock(control_mutex);
}

static void mseek(InputPlayback *playback, gint time)
{
	g_mutex_lock(control_mutex);

	if (!stop_flag) {
		duh_file->seek_to = time;
		duh_file->eof = FALSE;
	        g_cond_signal(control_cond);
                g_cond_wait(control_cond, control_mutex);
	}
	
	g_mutex_unlock(control_mutex);
}

static void file_info_box(const gchar *filename)
{
	GtkWidget *songinfo;
	GtkStyle *style = NULL;
	GtkTextBuffer *TextBuffer;
	PangoFontDescription *font_desc = NULL;

	DUMB_IT_SIGDATA *sd;
	DUH *duh;

	if ( duh_universal_load_vfs(&duh, filename, NULL, UNIREAD_MOD) == -1 )
		return;
	
	sd = duh_get_it_sigdata(duh);

	songinfo = create_songinfo ();
	if (!songinfo) return;

	font_desc = pango_font_description_from_string ("monospace 9");

	if (font_desc) {
		style = gtk_style_copy(gtk_widget_get_style(songinfo));
		style->font_desc = font_desc;
	}
	
	{
		int n = dumb_it_sd_get_n_instruments(sd);
		if (n) {
			GtkCList *l = GTK_CLIST(lookup_widget(songinfo, "songinfo_instruments_clist"));
			int i;
			if (style) gtk_widget_set_style(GTK_WIDGET(l), style);
			gtk_clist_column_titles_passive(l);
			gtk_clist_set_column_justification(l, 0, GTK_JUSTIFY_RIGHT);
			for (i = 0; i < n; i++) {
				char str[3];
				const gchar *text[] = {
					str,
					(gchar*) dumb_it_sd_get_instrument_name(sd, i),
					(gchar*) dumb_it_sd_get_instrument_filename(sd, i)
				};
				sprintf(str, "%d", i+1);
				gtk_clist_insert(l, i, (gchar **)text);
			}
		} else
			gtk_notebook_remove_page(GTK_NOTEBOOK(lookup_widget(songinfo, "songinfo_notebook")), 2);
	}

	{
		GtkCList *l = GTK_CLIST(lookup_widget(songinfo, "songinfo_samples_clist"));
		int i, n = dumb_it_sd_get_n_samples(sd);
		if (style) gtk_widget_set_style(GTK_WIDGET(l), style);
		gtk_clist_column_titles_passive(l);
		gtk_clist_set_column_justification(l, 0, GTK_JUSTIFY_RIGHT);
		for (i = 0; i < n; i++) {
			char str[3];
			const gchar *text[] = {
				str,
				(gchar*) dumb_it_sd_get_sample_name(sd, i),
				(gchar*) dumb_it_sd_get_sample_filename(sd, i)
			};
			sprintf(str, "%d", i+1);
			gtk_clist_insert(l, i, (gchar **)text);
		}
	}

	{
		GtkTextView *l = GTK_TEXT_VIEW(lookup_widget(songinfo, "songinfo_message_text"));
		const char *message = (char*) dumb_it_sd_get_song_message(sd);
		if (message) {
			char *m = strdup(message);
			int i;
			for (i = 0; m[i]; i++)
				if (m[i] == '\r') m[i] = '\n';

			TextBuffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (lookup_widget(songinfo, "songinfo_message_text")));
			if (style) gtk_widget_set_style(GTK_WIDGET(l), style);
			gtk_text_buffer_set_text(TextBuffer, (gchar*) m, -1);

			free(m);
		} else
			gtk_notebook_remove_page(GTK_NOTEBOOK(lookup_widget(songinfo, "songinfo_notebook")), 0);
	}

	/* I hope that's ok here. Previously there was no unload_duh at all (leak?) */
	if (duh)
		unload_duh(duh);

	gtk_widget_show (songinfo);
}

static void duh_configure(void) {
	if (prefs) return;
	prefs = create_prefs();
	if (!prefs) return;

	gint output_frequency_idx = 4; /* 44100 default/fallback */
	int i;
	gchar freqstring[6];

	for (i = 0; i < (int) (sizeof(output_frequencies) / sizeof(output_frequencies[0])); i++) {
		snprintf(freqstring, sizeof(freqstring), "%d", output_frequencies[i]);
		gtk_combo_box_append_text (GTK_COMBO_BOX(lookup_widget(prefs, "prefs_freq_combobox")), freqstring);
		if (output_frequencies[i] == output_frequency)
			output_frequency_idx = i;
	}
	
 	gtk_combo_box_set_active(GTK_COMBO_BOX(lookup_widget(prefs, "prefs_freq_combobox")), output_frequency_idx);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(prefs, "prefs_checkbox_disableamiga")), disable_amiga_mods);

	gtk_signal_connect(GTK_OBJECT(prefs), "destroy",         
			   GTK_SIGNAL_FUNC(gtk_widget_destroyed), &prefs);

	gtk_signal_connect_object(GTK_OBJECT(lookup_widget(prefs, "prefs_button_cancel")), "clicked",
				  GTK_SIGNAL_FUNC(gtk_widget_destroy), 
				  GTK_OBJECT(prefs));

	gtk_signal_connect_object(GTK_OBJECT(lookup_widget(prefs, "prefs_button_ok")), "clicked",
				  GTK_SIGNAL_FUNC(duh_configure_ok_cb), 
				  NULL);

	gtk_widget_show(prefs);
}

static void duh_configure_ok_cb(GtkButton *w, gpointer data) {
	mcs_handle_t *db;
	gint output_frequency_idx;

	db = aud_cfg_db_open();

	disable_amiga_mods = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(prefs, "prefs_checkbox_disableamiga")));
	aud_cfg_db_set_bool(db, "DUMB", "disable_amiga_mods", disable_amiga_mods);

	output_frequency_idx = gtk_combo_box_get_active(GTK_COMBO_BOX(lookup_widget(prefs, "prefs_freq_combobox")));
	output_frequency = output_frequencies[output_frequency_idx];
	aud_cfg_db_set_int(db, "DUMB", "output_frequency", output_frequency);	

	aud_cfg_db_close(db);
	gtk_widget_destroy(prefs);
}

static void duh_about(void) {
	static GtkWidget *about;
	gchar *about_text;
	
	if (about)
		return;

        about_text = g_strjoin("", _("DUMB Input Plugin "), VERSION,
				   _("\n"
				     "by Christian Birchinger <joker@netswarm.net>\n"
				     "\n"
				     "Based on the original XMMS plugin by Ben Davis\n"
				     "and the BMP port by Michael Doering\n"
				     "\n"
				     "Built with DUMB version "), DUMB_VERSION_STR, NULL);
                                                                                
	audgui_simple_message (& about, GTK_MESSAGE_INFO, "About DUMB Plugin", about_text);

	g_free(about_text);
}
