/*
 * Argus Software
 * Copyright (c) 2000-2008 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* 
 * $Id: //depot/argus/argus-3.0/clients/clients/rasplit.c#21 $
 * $DateTime: 2006/04/07 19:27:03 $
 * $Change: 812 $
 */

/*
 * rasplit.c  - split records into files using various strategies.
 *       
 * written by Carter Bullard
 * QoSient, LLC
 *
 */

#if defined(CYGWIN)
#define USE_IPV6
#endif

#ifndef _REENTRANT
#define _REENTRANT
#endif

#include <compat.h>
#include <sys/wait.h>

#include <rabins.h>
#include <rasplit.h>

struct ArgusAdjustStruct adata, *nadp = &adata;

void
ArgusClientInit (struct ArgusParserStruct *parser)
{
   struct ArgusWfileStruct *wfile = NULL;
   struct ArgusModeStruct *mode = NULL;
   char outputfile[MAXSTRLEN];
   int i = 0, ind = 0, count = 0;

   parser->RaWriteOut = 0;
   bzero(outputfile, sizeof(outputfile));

   if (!(parser->RaInitialized)) {

      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);
      (void) signal (SIGTERM, (void (*)(int)) RaParseComplete);
      (void) signal (SIGQUIT, (void (*)(int)) RaParseComplete);
      (void) signal (SIGINT,  (void (*)(int)) RaParseComplete);

      if (parser->dflag) {
         int pid;

         if (parser->Sflag)
            parser->ArgusReliableConnection++;

         ArgusLog(LOG_WARNING, "started");
         if (chdir ("/") < 0)
            ArgusLog (LOG_ERR, "Can't chdir to / %s", strerror(errno));

         if ((pid = fork ()) < 0) {
            ArgusLog (LOG_ERR, "Can't fork daemon %s", strerror(errno));
         } else {
            if (pid) {
               struct timespec ts = {0, 20000000};
               int status;
               nanosleep(&ts, NULL);   
               waitpid(pid, &status, WNOHANG);
               if (kill(pid, 0) < 0) {
                  exit (1);
               } else
                  exit (0);
            } else {
               parser->ArgusSessionId = setsid();
               fclose (stdin);
               fclose (stdout);
               fclose (stderr);
            }
         }
      }

      bzero((char *)nadp, sizeof(*nadp));

      nadp->mode      = -1;
      nadp->modify    =  0;
      nadp->slen =  2;

      if (parser->aflag)
         nadp->slen = parser->aflag;

      if (ArgusParser->ArgusWfileList && (ArgusParser->ArgusWfileList->start != NULL)) {
         count = ArgusParser->ArgusWfileList->count;
         for (i = 0; i < count; i++) {
            if ((wfile = (struct ArgusWfileStruct *) ArgusPopFrontList(parser->ArgusWfileList, ARGUS_NOLOCK)) != NULL) {
               if ((ArgusParser->exceptfile == NULL) || strcmp(wfile->filename, ArgusParser->exceptfile)) {
                  strncpy (outputfile, wfile->filename, MAXSTRLEN);
                  count++;
                  break;
               } else
                  ArgusPushBackList(ArgusParser->ArgusWfileList, (struct ArgusListRecord *) wfile, ARGUS_NOLOCK);
            }
         }
      } else {
         bzero (outputfile, MAXSTRLEN);
         *outputfile = 'x';
      }

      if ((mode = parser->ArgusModeList) != NULL) {
         while (mode) {
            for (i = 0, ind = -1; i < ARGUSSPLITMODENUM; i++) {
               if (!(strncasecmp (mode->mode, RaSplitModes[i], 3))) {
                  ind = i;
                  break;
               }
            }
            if (ind < 0)
               usage();
            
            switch (ind) {
               case ARGUSSPLITTIME:
                  nadp->mode   = ind;
                  nadp->modify =  1;
                  if ((mode = mode->nxt) != NULL) {
                     if (isdigit((int)*mode->mode)) {
                        char *ptr = NULL;
                        nadp->value = strtol(mode->mode, (char **)&ptr, 10);
                        if (ptr == mode->mode)
                           usage();
                        else {
                           switch (*ptr) {
                              case 'y': nadp->qual = ARGUSSPLITYEAR;   break;
                              case 'M': nadp->qual = ARGUSSPLITMONTH;  break;
                              case 'w': nadp->qual = ARGUSSPLITWEEK;   break;
                              case 'd': nadp->qual = ARGUSSPLITDAY;    break;
                              case 'h': nadp->qual = ARGUSSPLITHOUR;   break;
                              case 'm': nadp->qual = ARGUSSPLITMINUTE; break;
                               default: nadp->qual = ARGUSSPLITSECOND; break;
                           }
                        }
                     }
                  }
                  break;

               case ARGUSSPLITSIZE:
               case ARGUSSPLITCOUNT:
                  nadp->mode = ind;
                  nadp->count = 1;

                  if ((mode = mode->nxt) != NULL) {
                     if (isdigit((int)*mode->mode)) {
                        char *ptr = NULL;
                        nadp->value = strtol(mode->mode, (char **)&ptr, 10);
                        if (ptr == mode->mode)
                           usage();
                        else {
                           switch (*ptr) {
                              case 'B':   
                              case 'b':  nadp->value *= 1000000000; break;
                               
                              case 'M':   
                              case 'm':  nadp->value *= 1000000; break;
                               
                              case 'K':   
                              case 'k':  nadp->value *= 1000; break;
                           }
                        }
                     }
                  }
                  break;

               case ARGUSSPLITFLOW: {
                  nadp->mode = ind;
                  if ((mode = mode->nxt) != NULL) {
                     nadp->filterstr = strdup(mode->mode);

                     if (ArgusFilterCompile (&nadp->filter, nadp->filterstr, ArgusParser->Oflag) < 0)
                        ArgusLog (LOG_ERR, "flow filter parse error");

                     if (ArgusParser->bflag) {
                        nff_dump(&nadp->filter, ArgusParser->bflag);
                        exit (0);
                     }
                  }
                  break;
               }

               case ARGUSSPLITPATTERN:
                  break;

               case ARGUSSPLITNOMODIFY:
                  nadp->modify = 0;
            }

            mode = mode->nxt;
         }
      }

      if (nadp->mode < 0) {
         nadp->mode = ARGUSSPLITCOUNT;
         nadp->value = 10000;
         nadp->count = 1;
      }

      /* if content substitution, either time or any field, is used,
         size and count modes will not work properly.  If using
         the default count, set the value so that we generate only
         one filename.

         if no substitution, then we need to add "aa" suffix to the
         output file for count and size modes.
      */
 
      if ((strchr(outputfile, '%')) || (strchr(outputfile, '$'))) {
         switch (nadp->mode) {
            case ARGUSSPLITCOUNT:
               nadp->count = -1;
               break;

            case ARGUSSPLITSIZE:
            case ARGUSSPLITFLOW:
               for (i = 0; i < nadp->slen; i++) 
                  strcat(outputfile, "a");
               break;
         }

      } else {
         switch (nadp->mode) {
            case ARGUSSPLITSIZE:
            case ARGUSSPLITCOUNT:
            case ARGUSSPLITFLOW:
               for (i = 0; i < nadp->slen; i++) 
                  strcat(outputfile, "a");
               break;
         }
      }

      if (!(strchr(outputfile, '%'))) {
         switch (nadp->mode) {
            case ARGUSSPLITTIME:
            /* if strftime() labels are not in use, need to add suffix */
              if (outputfile[strlen(outputfile) - 1] != '.')
                 strcat(outputfile, ".");
              strcat(outputfile, "%Y.%m.%d.%H.%M.%S");
              break;
         }
      }

      nadp->filename = strdup(outputfile);
      setArgusWfile (parser, outputfile, NULL);

      parser->RaClientTimeout.tv_sec  = 0;
      parser->RaClientTimeout.tv_usec = 330000;
      parser->RaInitialized++;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientInit()\n");
#endif
}


void RaArgusInputComplete (struct ArgusInput *input) { return; }


void
RaParseComplete (int sig)
{
   if (sig >= 0) {
      if ((sig == SIGINT) || (sig == SIGQUIT))
         exit(0);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaParseComplete(%d)\n", sig);
#endif
}


void
ArgusClientTimeout ()
{
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientTimeout()\n");
#endif
}

void parse_arg (int argc, char**argv) {}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Rasplit Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusParser->ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -S remoteServer  [- filter-expression]\n", ArgusParser->ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusParser->ArgusProgramName);

   fprintf (stderr, "options: -b                 dump packet-matching code.\n");
   fprintf (stderr, "         -C <[host]:port>   specify remote Cisco Netflow source.\n");
#if defined (ARGUSDEBUG)
   fprintf (stderr, "         -D <level>         specify debug level\n");
#endif
   fprintf (stderr, "         -E <file>          write records that are rejected by the filter\n");
   fprintf (stderr, "                            into <file>\n");
   fprintf (stderr, "         -F <conffile>      read configuration from <conffile>.\n");
   fprintf (stderr, "         -h                 print help.\n");

   fprintf (stderr, "         -M <mode>          supported modes of operation:\n");
   fprintf (stderr, "            time n[smhdwmy]\n");
   fprintf (stderr, "           count n[kmb]\n");
   fprintf (stderr, "            size n[kmb]\n");
   fprintf (stderr, "            nomodify\n");

   fprintf (stderr, "         -r <file>          read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -S <host[:port]>   specify remote argus <host> and optional port\n");
   fprintf (stderr, "                            number.\n");
   fprintf (stderr, "         -t <timerange>     specify <timerange> for reading records.\n");
   fprintf (stderr, "                   format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                            timeSpecification: [[[yyyy/]mm/]dd.]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                                 [yyyy/]mm/dd\n");
   fprintf (stderr, "                                                 -%%d{yMdhms}\n");
   fprintf (stderr, "         -T <secs>          attach to remote server for T seconds.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth>     specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "         -w <file>          write output to <file>. '-' denotes stdout.\n");
   exit(1);
}


void
RaProcessRecord (struct ArgusParserStruct *parser, struct ArgusRecordStruct *ns)
{
   struct ArgusRecordStruct *tns = NULL;
   char *filename = NULL, *tptr = NULL, *pptr = NULL;
   char tmpbuf[MAXSTRLEN]; 
   int newfilename = 0;

   if (ns->hdr.type & ARGUS_MAR) {
   } else {
      adata.tperiod = 0.0;

      while ((tns = ArgusAlignRecord(parser, ns, &adata)) != NULL) {
         struct ArgusWfileStruct *wfile = NULL;
         int i, count = parser->ArgusWfileList->count;

         for (i = 0; i < count; i++) {
            if ((wfile = (struct ArgusWfileStruct *) ArgusFrontList(parser->ArgusWfileList)) != NULL) {
               if ((ArgusParser->exceptfile != NULL) && (strcmp(wfile->filename, ArgusParser->exceptfile) == 0)) {
                  if ((wfile = (struct ArgusWfileStruct *) ArgusPopFrontList(parser->ArgusWfileList, ARGUS_NOLOCK)) != NULL)
                     ArgusPushBackList(ArgusParser->ArgusWfileList, (struct ArgusListRecord *) wfile, ARGUS_NOLOCK);
                  wfile = NULL;
               } else
                  break;
            }
         }


         if ((wfile = (struct ArgusWfileStruct *)ArgusFrontList(parser->ArgusWfileList)) == NULL)
            ArgusLog (LOG_ERR, "RaProcessRecord: no output file specified");

         switch (nadp->mode) {
            case ARGUSSPLITTIME: {
               time_t fileSecs;
               struct tm tmval;

               if ((tns->canon.time.src.start.tv_sec == 0) && (tns->canon.time.dst.start.tv_sec == 0))
                  ArgusLog (LOG_ERR, "RaProcessRecord: stime is zero");

               fileSecs = mktime(&nadp->RaStartTmStruct);

               if (strftime(tmpbuf, MAXSTRLEN, nadp->filename, localtime_r(&fileSecs, &tmval)) <= 0)
                  ArgusLog (LOG_ERR, "RaSendArgusRecord () ArgusCalloc %s\n", strerror(errno));

               RaProcessSplitOptions(parser, tmpbuf, MAXSTRLEN, ns);
                 
               if (strcmp(wfile->filename, tmpbuf)) {
                  filename = tmpbuf;
                  newfilename++;
               }
               break;
            }

            case ARGUSSPLITCOUNT:
               sprintf (tmpbuf, "%s", nadp->filename);
               RaProcessSplitOptions(parser, tmpbuf, MAXSTRLEN, ns);
                 
               if (strcmp(wfile->filename, tmpbuf)) {
                  filename = tmpbuf;
                  newfilename++;
               }

               if ((nadp->value > 1) && (!(nadp->count % nadp->value)))
                  newfilename++;

               if (nadp->count > 0)
                  nadp->count++;
               break;

            case ARGUSSPLITSIZE:
               sprintf (tmpbuf, "%s", nadp->filename);
               RaProcessSplitOptions(parser, tmpbuf, MAXSTRLEN, ns);
                 
               if (strcmp(wfile->filename, tmpbuf)) {
                  filename = tmpbuf;
                  newfilename++;
               }

               if ((nadp->value > 0) && (stat (wfile->filename, &wfile->statbuf) == 0))
                  if ((wfile->statbuf.st_size + (ns->hdr.len * 4)) > nadp->value)
                     newfilename++;
               break;

            case ARGUSSPLITFLOW: {
               struct nff_insn *fcode = nadp->filter.bf_insns;
               int retn = 0;

               sprintf (tmpbuf, "%s", nadp->filename);
               RaProcessSplitOptions(parser, tmpbuf, MAXSTRLEN, ns);
                 
               if (strcmp(wfile->filename, tmpbuf)) {
                  filename = tmpbuf;
                  newfilename++;
               }

               if ((retn = ArgusFilterRecord (fcode, ns)) != 0)
                  newfilename++;
               break;
            }
         }

         if (newfilename) {
            if (wfile->filename != NULL) {
               free(wfile->filename);
               wfile->filename = NULL;
            } 
 
            if (filename == NULL)
               if ((filename = RaSplitFilename(nadp)) == NULL)
                  ArgusLog(LOG_ERR, "RaProcessRecord filename beyond space");

            wfile->filename = strdup(filename);

            if (wfile->fd != NULL) {
               fclose (wfile->fd);
               wfile->fd = NULL;
            }

            /* got new filename, need to check the
               path to be sure that all the directories exist */

            strncpy (tmpbuf, wfile->filename, MAXSTRLEN);
            if ((tptr = strrchr(tmpbuf, (int) '/')) != NULL) {   /* if there is a path */
               *tptr = '\0';
               pptr = tptr;

               while ((pptr != NULL) && ((stat(tmpbuf, &wfile->statbuf)) < 0)) {
                  switch (errno) {
                     case ENOENT:
                        if ((pptr = strrchr(tmpbuf, (int) '/')) != NULL) {
                           if (pptr != tmpbuf) {
                              *pptr = '\0';
                           } else {
                              pptr = NULL;
                           }
                        }
                        break;

                     default:
                        ArgusLog (LOG_ERR, "stat: %s %s\n", tmpbuf, strerror(errno));
                  }
               }

               while (&tmpbuf[strlen(tmpbuf)] <= tptr) {
                  if ((mkdir(tmpbuf, 0777)) < 0) {
                     if (errno != EEXIST)
                        ArgusLog (LOG_ERR, "mkdir: %s %s\n", tmpbuf, strerror(errno));
                  }
                  tmpbuf[strlen(tmpbuf)] = '/';
               }
               *tptr = '/';
            }
         }

         RaSendArgusRecord (tns);
         ArgusDeleteRecordStruct (parser, tns);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessRecord (0x%x) done\n", ns); 
#endif
}


int
RaSendArgusRecord(struct ArgusRecordStruct *argus)
{
   int retn = 1;
 
   if (ArgusParser->ArgusWfileList != NULL) {
      struct ArgusWfileStruct *wfile = NULL;
      struct ArgusListObjectStruct *lobj = NULL;
      int i, count = ArgusParser->ArgusWfileList->count;

      if ((lobj = ArgusParser->ArgusWfileList->start) != NULL) {
         for (i = 0; i < count; i++) {
            if ((wfile = (struct ArgusWfileStruct *) lobj) != NULL) {
               int pass = 1;
               if (wfile->filterstr) {
                  struct nff_insn *wfcode = wfile->filter.bf_insns;
                  pass = ArgusFilterRecord (wfcode, argus);
               }

               if (pass != 0) {
                  if ((ArgusParser->exceptfile == NULL) || strcmp(wfile->filename, ArgusParser->exceptfile)) {
                     struct ArgusRecord *argusrec = NULL;
                     char buf[2048];
                     if ((argusrec = ArgusGenerateRecord (argus, 0L, buf)) != NULL) {
#ifdef _LITTLE_ENDIAN
                        ArgusHtoN(argusrec);
#endif
                        ArgusWriteNewLogfile (ArgusParser, argus->input, wfile, argusrec);
                     }
                  }
               }
            }
            lobj = lobj->nxt;
         }
      }
   }

   return (retn);
}


void ArgusWindowClose(void);

void ArgusWindowClose(void) { 
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusWindowClose () returning\n"); 
#endif
}


char *
RaSplitFilename (struct ArgusAdjustStruct *nadp)
{
   char *retn = NULL, tmpbuf[MAXSTRLEN];
   char *filename = nadp->filename;
   int len, i = 1;

   if (filename != NULL) {
      len = strlen(filename);

      for (i = 0; i < nadp->slen; i++) {
         if (filename[len - (i + 1)] == 'z') {
            filename[len - (i + 1)] = 'a';
         } else {
            filename[len - (i + 1)]++;
            break;
         }
      }

      if (filename[len - nadp->slen] == 'z') {
         snprintf(tmpbuf, MAXSTRLEN, "%sa", filename);

         if (nadp->filename)
            free(nadp->filename);

         nadp->filename = strdup(tmpbuf);
      }

      retn = nadp->filename;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaSplitFilename (0x%x) returning %s\n", nadp, retn); 
#endif

   return (retn);
}

int
RaProcessSplitOptions(struct ArgusParserStruct *parser, char *str, int len, struct ArgusRecordStruct *ns)
{
   char resultbuf[MAXSTRLEN], tmpbuf[MAXSTRLEN];
   char *ptr = NULL, *cptr = NULL, *tptr = str;
   int retn = 0, i, x, slen = 0;

   bzero(resultbuf, MAXSTRLEN);
   bzero(tmpbuf, MAXSTRLEN);

   while ((ptr = strchr (tptr, '$')) != NULL) {
      *ptr++ = '\0';
      slen = strlen(resultbuf);
      snprintf (&resultbuf[slen], MAXSTRLEN - slen, tptr);

      for (i = 0, x = 0; x < MAX_PRINT_ALG_TYPES; x++) {
         if (!strncmp (RaPrintAlgorithmTable[x].field, ptr, strlen(RaPrintAlgorithmTable[x].field))) {
            RaPrintAlgorithmTable[x].print(parser, tmpbuf, ns, RaPrintAlgorithmTable[x].length);

            while (isspace((int)tmpbuf[strlen(tmpbuf) - 1]))
               tmpbuf[strlen(tmpbuf) - 1] = '\0';

            while (isspace((int)tmpbuf[i])) i++;
            slen = strlen(resultbuf);
            snprintf (&resultbuf[slen], MAXSTRLEN - slen, "%s", &tmpbuf[i]);

            ptr += strlen(RaPrintAlgorithmTable[x].field);
            cptr = &resultbuf[strlen(resultbuf)];

            while (*ptr && (*ptr != '$')) {
               *cptr++ = *ptr++;
            }
            *cptr = '\0';
            break;
         }
      }

      tptr = ptr;
      retn++;
   }

   if (retn) {
      int len = strlen(resultbuf);
      bcopy (resultbuf, str, strlen(resultbuf));
      str[len] = '\0';
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaProcessSplitOptions(%s, %d, 0x%x): returns %d", str, len, ns, retn);
#endif

   return (retn);
}
