/*
 * Argus Software
 * Copyright (c) 2000-2008 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * rabins - time based bin processor. 
 *    this routine will take in an argus stream and align it to
 *    to a time array, and hold it for a hold period, and then
 *    output the bin countents as an argus stream.
 *
 *    this is the basis for all stream block processors.
 *    used by ragraph() to structure the data into graphing
 *    regions.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */

/* 
 * $Id: //depot/argus/argus-3.0/clients/clients/rabins.c#41 $
 * $DateTime: 2006/03/31 13:25:33 $
 * $Change: 793 $
 */

#if defined(CYGWIN)
#define USE_IPV6
#endif

#include <unistd.h>
#include <stdlib.h>

#include <math.h>

#include <compat.h>

#include <argus_util.h>
#include <argus_client.h>
#include <argus_main.h>
#include <argus_filter.h>

#include <rabins.h>
#include <rasplit.h>
#include <argus_sort.h>
#include <argus_cluster.h>

#include <signal.h>
#include <ctype.h>

int RaRealTime = 0;
float RaUpdateRate = 1.0;

struct timeval ArgusLastRealTime = {0, 0};
struct timeval ArgusLastTime     = {0, 0};
struct timeval ArgusThisTime     = {0, 0};


struct timeval dLastTime = {0, 0};
struct timeval dRealTime = {0, 0};
struct timeval dThisTime = {0, 0};
struct timeval dTime     = {0, 0};

long long thisUsec = 0;

struct RaBinProcessStruct *RaBinProcess = NULL;

int ArgusRmonMode = 0;



void
ArgusClientInit (struct ArgusParserStruct *parser)
{
   struct ArgusAdjustStruct *nadp;
   struct ArgusModeStruct *mode = NULL;
   char outputfile[MAXSTRLEN];
   int i = 0, ind = 0, size = 1;

   parser->RaWriteOut = 0;
   *outputfile = '\0';

   if (!(parser->RaInitialized)) {
      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);
      (void) signal (SIGTERM, (void (*)(int)) RaParseComplete);
      (void) signal (SIGQUIT, (void (*)(int)) RaParseComplete);
      (void) signal (SIGINT,  (void (*)(int)) RaParseComplete);

      if ((RaBinProcess = (struct RaBinProcessStruct *)ArgusCalloc(1, sizeof(*RaBinProcess))) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusCalloc error %s", strerror(errno));

      nadp = &RaBinProcess->nadp;
      RaBinProcess->size = size;

      bzero((char *)nadp, sizeof(*nadp));

      nadp->mode      = -1;
      nadp->modify    =  1;
      nadp->slen =  2;

      if (parser->aflag)
         nadp->slen = parser->aflag;

      if (parser->vflag)
         ArgusReverseSortDir++;

      parser->RaCumulativeMerge = 1;

      if ((ArgusSorter = ArgusNewSorter()) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusNewSorter error %s", strerror(errno));
 
      if ((mode = parser->ArgusModeList) != NULL) {
         while (mode) {
            if (isdigit((int) *mode->mode)) {
               ind = 0;
            } else {
               if (!(strncasecmp (mode->mode, "rtime", 5)) ||
                  (!(strncasecmp (mode->mode, "realtime", 8)))) {
                  char *ptr = NULL;
                  RaRealTime++;
                  if ((ptr = strchr(mode->mode, ':')) != NULL) {
                     double value = 0.0;
                     char *endptr = NULL;
                     ptr++;
                     value = strtod(ptr, &endptr);
                     if (ptr != endptr) {
                        RaUpdateRate = value;
                     }
                  }
               } else
               if (!(strncasecmp (mode->mode, "rmon", 4)))
                  parser->RaMonMode++;
               else
               if (!(strncasecmp (mode->mode, "norep", 5)))
                  parser->RaAgMode++;
               else {
                  for (i = 0, ind = -1; i < ARGUSSPLITMODENUM; i++) {
                     if (!(strncasecmp (mode->mode, RaSplitModes[i], 3))) {
                        ind = i;
                        switch (ind) {
                           case ARGUSSPLITTIME:
                           case ARGUSSPLITSIZE:
                           case ARGUSSPLITCOUNT:
                              if ((mode = mode->nxt) == NULL)
                                 usage();
                              break;
                        }
                     }
                  }
               }
            }

            if (ind < 0)
               usage();

            switch (ind) {
               case ARGUSSPLITTIME:
                  nadp->mode = ind;
                  if (isdigit((int)*mode->mode)) {
                     char *ptr = NULL;
                     nadp->value = strtol(mode->mode, (char **)&ptr, 10);
                     if (ptr == mode->mode)
                        usage();
                     else {
                        switch (*ptr) {
                           case 'y':
                              nadp->qual = ARGUSSPLITYEAR;  
                              size = nadp->value * 31556926;
                              break;
                           case 'M':
                              nadp->qual = ARGUSSPLITMONTH; 
                              size = nadp->value * 2629744;
                              break;
                           case 'w':
                              nadp->qual = ARGUSSPLITWEEK;  
                              size = nadp->value * 604800;
                              break;
                           case 'd':
                              nadp->qual = ARGUSSPLITDAY;   
                              size = nadp->value * 86400;
                              break;
                           case 'h':
                              nadp->qual = ARGUSSPLITHOUR;  
                              size = nadp->value * 3600;
                              break;
                           case 'm':
                              nadp->qual = ARGUSSPLITMINUTE;
                              size = nadp->value * 60;
                              break;
                            default:
                              nadp->qual = ARGUSSPLITSECOND;
                              size = nadp->value;
                              break;
                        }
                     }
                  }
                  ArgusSorter->ArgusSortAlgorithms[0] = ArgusSortStartTime;
                  break;

               case ARGUSSPLITSIZE:
               case ARGUSSPLITCOUNT:
                  nadp->mode = ind;
                  nadp->count = 1;

                  if (isdigit((int)*mode->mode)) {
                     char *ptr = NULL;
                     nadp->value = strtol(mode->mode, (char **)&ptr, 10);
                     if (ptr == mode->mode)
                        usage();
                     else {
                        switch (*ptr) {
                           case 'B':   
                           case 'b':  nadp->value *= 1000000000; break;
                            
                           case 'M':   
                           case 'm':  nadp->value *= 1000000; break;
                            
                           case 'K':   
                           case 'k':  nadp->value *= 1000; break;
                        }
                     }
                  }
                  ArgusSorter->ArgusSortAlgorithms[0] = NULL;
                  break;

               case ARGUSSPLITNOMODIFY:
                  nadp->modify = 0;
                  break;

               case ARGUSSPLITSOFT:
                  nadp->soft++;
                  break;

               case ARGUSSPLITZERO:
                  nadp->zero++;
                  break;
            }

            mode = mode->nxt;
         }
      }

      RaBinProcess->size  = size;
      if (!(nadp->value))
         nadp->value = 1;

      if (nadp->mode < 0) {
         nadp->value = 1;
         nadp->count = 1;
      }

      /* if content substitution, either time or any field, is used,
         size and count modes will not work properly.  If using
         the default count, set the value so that we generate only
         one filename.

         if no substitution, then we need to add "aa" suffix to the
         output file for count and size modes.
      */

      if (parser->ArgusWfileList != NULL) {
         struct ArgusWfileStruct *wfile = NULL;

         if ((wfile = (struct ArgusWfileStruct *)ArgusPopFrontList(parser->ArgusWfileList, ARGUS_NOLOCK)) != NULL) {
            if (strcmp(wfile->filename, "-")) {
               strncpy (outputfile, wfile->filename, MAXSTRLEN);
               if ((strchr(outputfile, '%')) || (strchr(outputfile, '$'))) {
                  switch (nadp->mode) {
                     case ARGUSSPLITCOUNT:
                        nadp->count = -1;
                        break;

                     case ARGUSSPLITSIZE:
                        for (i = 0; i < nadp->slen; i++) 
                           strcat(outputfile, "a");
                        break;
                  }

               } else {
                  switch (nadp->mode) {
                     case ARGUSSPLITSIZE:
                     case ARGUSSPLITCOUNT:
                        for (i = 0; i < nadp->slen; i++) 
                           strcat(outputfile, "a");
                        break;
                  }
               }

               if (!(strchr(outputfile, '%'))) {
                  switch (nadp->mode) {
                     case ARGUSSPLITTIME:
                       break;
                  }
               }

               nadp->filename = strdup(outputfile);
               setArgusWfile (parser, outputfile, NULL);
            } else
               setArgusWfile (parser, "-", NULL);
         }
      }

      parser->RaClientTimeout.tv_sec  = 0;
      parser->RaClientTimeout.tv_usec = 274895;
      parser->RaInitialized++;

      if (ArgusParser->Bflag > 0) {
         RaBinProcess->rtime.tv_sec = ArgusParser->ArgusRealTime.tv_sec;
      }

      if (ArgusParser->startime_t && ArgusParser->lasttime_t) {
         nadp->count = ((ArgusParser->lasttime_t - ArgusParser->startime_t)/size) + 1;
      } else {
         nadp->count = ((size > parser->Bflag) ? size : parser->Bflag)/nadp->value;
         nadp->count += 10000;
      }

      if (parser->Gflag) {
         parser->uflag++;
         parser->RaFieldDelimiter = ',';
      }

      for (i = 0; parser->RaPrintAlgorithmList[i] != NULL; i++) {
         if (parser->RaPrintAlgorithmList[i]->print == ArgusPrintProto) {
            parser->RaPrintMode |= RA_PRINTPROTO;
            break;
         }
         if (parser->RaPrintAlgorithmList[i]->print == ArgusPrintSrcPort) {
            break;
         }
         if (parser->RaPrintAlgorithmList[i]->print == ArgusPrintDstPort) {
            break;
         }
         if (parser->RaPrintAlgorithmList[i]->print == ArgusPrintSourceID) {
            parser->RaPrintMode |= RA_PRINTSRCID;
            break;
         }
      }

      if ((parser->ArgusAggregator = ArgusNewAggregator(parser, NULL)) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusNewAggregator error");
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientInit()\n");
#endif
}


void RaArgusInputComplete (struct ArgusInput *input) { return; }

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   int startsecs = 0, endsecs = 0, i;
   
   if (sig >= 0)
      if ((sig == SIGINT) || (sig == SIGQUIT))
         exit(0);

   if (!(RaParseCompleting++)) {
      if (RaBinProcess != NULL) {
         struct RaBinStruct *bin = NULL;
         struct ArgusRecordStruct *ns = NULL;

         if ((ArgusParser->ArgusWfileList == NULL) && (ArgusParser->Gflag)) {
            if (ArgusParser->startime_t && ArgusParser->lasttime_t) {
               startsecs = ArgusParser->startime_t;
               endsecs   = ArgusParser->lasttime_t;
            } else {
               for (i = 0; i < RaBinProcess->len; i++) {
                  if ((bin = RaBinProcess->array[i]) != NULL) {
                     if (startsecs == 0)
                        startsecs = bin->stime.tv_sec;
                     endsecs = bin->stime.tv_sec;
                  }
               }
            }

            if (ArgusParser->Hstr != NULL) {

            } else {
               char stimebuf[128], dtimebuf[128], etimebuf[128];

               RaBinProcess->endpt.tv_sec   = endsecs;

               if (!(ArgusParser->tflag)) {
                  RaBinProcess->startpt.tv_sec = startsecs;
                  RaBinProcess->scalesecs      = (endsecs - startsecs) + RaBinProcess->size;
               } else {
                  RaBinProcess->startpt.tv_sec = ArgusParser->startime_t;
                  RaBinProcess->scalesecs      = ArgusParser->lasttime_t - ArgusParser->startime_t;
               }

               if ((ArgusParser->RaEndTime.tv_sec >  RaBinProcess->endpt.tv_sec) ||
                  ((ArgusParser->RaEndTime.tv_sec == RaBinProcess->endpt.tv_sec) &&
                   (ArgusParser->RaEndTime.tv_usec > RaBinProcess->endpt.tv_usec)))
                  ArgusParser->RaEndTime = RaBinProcess->endpt;

               ArgusPrintTime(ArgusParser, stimebuf, &RaBinProcess->startpt);
               ArgusPrintTime(ArgusParser, dtimebuf, &RaBinProcess->endpt);
               ArgusPrintTime(ArgusParser, etimebuf, &ArgusParser->RaEndTime);

               stimebuf[strlen(stimebuf) - 1] = '\0';
               dtimebuf[strlen(dtimebuf) - 1] = '\0';
               etimebuf[strlen(etimebuf) - 1] = '\0';

               printf ("StartTime=%s\n", stimebuf);
               printf ("StopTime=%s\n",  dtimebuf);
               printf ("LastTime=%s\n",  etimebuf);
               printf ("Seconds=%d\n", RaBinProcess->scalesecs);
               printf ("BinSize=%1.*f\n", ArgusParser->pflag, RaBinProcess->size * 1.0);
               printf ("Bins=%d\n", (RaBinProcess->scalesecs + (RaBinProcess->size - 1))/RaBinProcess->size);
            }
         }

         for (i = 0; i < RaBinProcess->len; i++) {
            if ((bin = RaBinProcess->array[i]) != NULL) {
               ArgusSortQueue(ArgusSorter, bin->queue);

               while ((ns = (struct ArgusRecordStruct *) ArgusPopQueue(bin->queue, ARGUS_NOLOCK)) != NULL) {
                  RaSendArgusRecord (ns);
                  ArgusDeleteRecordStruct(ArgusParser, ns);
               }

            } else {
               if (RaBinProcess->nadp.zero && ((i >= RaBinProcess->index) && (((i - RaBinProcess->index) * RaBinProcess->size) < RaBinProcess->scalesecs))) {
                  ns = ArgusGenerateRecordStruct(NULL, NULL, NULL);
                  ((struct ArgusTimeObject *)ns->dsrs[ARGUS_TIME_INDEX])->src.start.tv_sec  = RaBinProcess->start + (RaBinProcess->size * (i - RaBinProcess->index));
                  ((struct ArgusTimeObject *)ns->dsrs[ARGUS_TIME_INDEX])->src.start.tv_usec = 0;
                  ((struct ArgusTimeObject *)ns->dsrs[ARGUS_TIME_INDEX])->src.end.tv_sec    = RaBinProcess->start + (RaBinProcess->size * ((i + 1) - RaBinProcess->index));
                  ((struct ArgusTimeObject *)ns->dsrs[ARGUS_TIME_INDEX])->src.end.tv_usec   = 0;

                  RaSendArgusRecord (ns);
               }
            }
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaParseComplete(%d)\n", sig);
#endif
}


void
ArgusClientTimeout ()
{
   struct RaBinProcessStruct *rbps = RaBinProcess;
   struct ArgusRecordStruct *ns = NULL;
   struct RaBinStruct *bin = NULL;
   int i, count;

   if (RaRealTime) {  /* establish value for time comparison */
      gettimeofday(&ArgusParser->ArgusRealTime, 0);
      ArgusAdjustGlobalTime(ArgusParser, &ArgusParser->ArgusRealTime);

      if (ArgusLastTime.tv_sec != 0) {
         if (ArgusLastRealTime.tv_sec > 0) {
            dRealTime = *RaDiffTime(&ArgusParser->ArgusRealTime, &ArgusLastRealTime);
            thisUsec = ((dRealTime.tv_sec * 1000000) + dRealTime.tv_usec) * RaUpdateRate;
            dRealTime.tv_sec  = thisUsec / 1000000;
            dRealTime.tv_usec = thisUsec % 1000000;


            ArgusLastTime.tv_sec  += dRealTime.tv_sec;
            ArgusLastTime.tv_usec += dRealTime.tv_usec;

            if (ArgusLastTime.tv_usec > 1000000) {
               ArgusLastTime.tv_sec++;
               ArgusLastTime.tv_usec -= 1000000;
            }
         }

         ArgusLastRealTime = ArgusParser->ArgusRealTime;
      }
   }


   if ((ArgusParser->Bflag > 0) && rbps->rtime.tv_sec) {
      if ((RaDiffTime(&ArgusParser->ArgusRealTime, &rbps->rtime))->tv_sec >= ArgusParser->Bflag) {

         count = (rbps->endpt.tv_sec - rbps->startpt.tv_sec)/rbps->size;

         if (rbps->array != NULL) {
            if ((bin = rbps->array[rbps->index]) != NULL) {
               ArgusSortQueue(ArgusSorter, bin->queue);
               RaDeleteBin(ArgusParser, bin);

            } else {
               if (RaBinProcess->nadp.zero && ((i >= RaBinProcess->index) && (((i - RaBinProcess->index) * RaBinProcess->size) < RaBinProcess->scalesecs))) {
                  ns = ArgusGenerateRecordStruct(NULL, NULL, NULL);
                  ((struct ArgusTimeStruct *)ns->dsrs[ARGUS_TIME_INDEX])->start.tv_sec  = RaBinProcess->start + (RaBinProcess->size * (i - RaBinProcess->index));
                  ((struct ArgusTimeStruct *)ns->dsrs[ARGUS_TIME_INDEX])->start.tv_usec = 0;
                  ((struct ArgusTimeStruct *)ns->dsrs[ARGUS_TIME_INDEX])->end.tv_sec    = RaBinProcess->start + (RaBinProcess->size * ((i + 1) - RaBinProcess->index));
                  ((struct ArgusTimeStruct *)ns->dsrs[ARGUS_TIME_INDEX])->end.tv_usec   = 0;
                  RaSendArgusRecord (ns);
                  ArgusDeleteRecordStruct(ArgusParser, ns);
#ifdef ARGUSDEBUG
                  ArgusDebug (2, "ArgusClientTimeout() RaBinProcess: creating zero record\n");
#endif
               }
            }

            for (i = 0; i < count; i++)
               rbps->array[rbps->index + i] = rbps->array[rbps->index + (i + 1)];
    
            rbps->start += rbps->size;
            rbps->end   += rbps->size;
            rbps->array[rbps->index + count] = NULL;
            rbps->startpt.tv_sec  += rbps->size;

            if ((ArgusParser->ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusParser->ArgusWfileList)))) {
               struct ArgusWfileStruct *wfile = NULL;
               struct ArgusListObjectStruct *lobj = NULL;
               int i, count = ArgusParser->ArgusWfileList->count;

               if ((lobj = ArgusParser->ArgusWfileList->start) != NULL) {
                  for (i = 0; i < count; i++) {
                     if ((wfile = (struct ArgusWfileStruct *) lobj) != NULL) {
                        if (wfile->fd != NULL)
                           fflush(wfile->fd);
                     }
                     lobj = lobj->nxt;
                  }
               }
            }
         }

#ifdef ARGUSDEBUG
         ArgusDebug (2, "ArgusClientTimeout() RaBinProcess: Bflag %d rtime %d start %d end %d size %d arraylen %d count %d index %d\n",
            ArgusParser->Bflag, RaBinProcess->rtime.tv_sec, RaBinProcess->startpt.tv_sec, RaBinProcess->endpt.tv_sec,
            RaBinProcess->size, RaBinProcess->arraylen, RaBinProcess->count, RaBinProcess->index);
#endif
         rbps->rtime.tv_sec += rbps->size;
      }
   }

   if (rbps->rtime.tv_sec == 0) {
      rbps->rtime.tv_sec = ArgusParser->ArgusRealTime.tv_sec;
   }
}

void parse_arg (int argc, char**argv) {}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Rabins Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusParser->ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -S remoteServer  [- filter-expression]\n", ArgusParser->ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusParser->ArgusProgramName);

   fprintf (stderr, "options: -b                 dump packet-matching code.\n");
   fprintf (stderr, "         -C <[host]:<port>  specify remote Cisco Netflow source.\n");
#if defined (ARGUSDEBUG)
   fprintf (stderr, "         -D <level>         specify debug level\n");
#endif
   fprintf (stderr, "         -E <file>          write records that are rejected by the filter\n");
   fprintf (stderr, "                            into <file>\n");
   fprintf (stderr, "         -F <conffile>      read configuration from <conffile>.\n");
   fprintf (stderr, "         -h                 print help.\n");

   fprintf (stderr, "         -M <mode>          supported modes of operation:\n");
   fprintf (stderr, "            time n[smhdwmy] split output into time series bins\n");
   fprintf (stderr, "            nomodify        don't modify input records\n");
   fprintf (stderr, "            zero            generate zero records for gaps\n");

   fprintf (stderr, "         -m <mode>          supported aggregation objects:\n");
   fprintf (stderr, "           none             no flow key\n");
   fprintf (stderr, "           saddr            include the source address\n");
   fprintf (stderr, "           daddr            include the destination address\n");
   fprintf (stderr, "           proto            include the destination proto\n");
   fprintf (stderr, "           sport            include the source port\n");
   fprintf (stderr, "           dport            include the destination port\n");
   fprintf (stderr, "           srcid            include the source identifier\n");

   fprintf (stderr, "         -r <file>          read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -S <host[:port]>   specify remote argus <host> and optional port\n");
   fprintf (stderr, "                            number.\n");
   fprintf (stderr, "         -t <timerange>     specify <timerange> for reading records.\n");
   fprintf (stderr, "                   format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                            timeSpecification: [[[yyyy/]mm/]dd.]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                                 [yyyy/]mm/dd\n");
   fprintf (stderr, "                                                 -%%d{yMdhms}\n");
   fprintf (stderr, "         -T <secs>          attach to remote server for T seconds.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth>     specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "         -w <file>          write output to <file>. '-' denotes stdout.\n");
   exit(1);
}

void RaProcessThisRecord (struct ArgusParserStruct *, struct ArgusRecordStruct *);

void
RaProcessRecord (struct ArgusParserStruct *parser, struct ArgusRecordStruct *ns)
{
   if (ns->hdr.type & ARGUS_MAR) {
   } else {

      if (ns->dsrs[ARGUS_FLOW_INDEX] == NULL)
         return;

      ArgusThisTime.tv_sec  = ns->canon.time.src.start.tv_sec;
      ArgusThisTime.tv_usec = ns->canon.time.src.start.tv_usec;

      if (RaRealTime) {
         if (ArgusLastTime.tv_sec == 0)
            ArgusLastTime = ArgusThisTime;

         if (!((ArgusLastTime.tv_sec  > ArgusThisTime.tv_sec) ||
            ((ArgusLastTime.tv_sec == ArgusThisTime.tv_sec) &&
             (ArgusLastTime.tv_usec > ArgusThisTime.tv_usec)))) {

            while ((ArgusThisTime.tv_sec  > ArgusLastTime.tv_sec) ||
                  ((ArgusThisTime.tv_sec == ArgusLastTime.tv_sec) &&
                   (ArgusThisTime.tv_usec > ArgusLastTime.tv_usec))) {
               struct timespec ts = {0, 0};
               int thisRate;

               dThisTime = *RaDiffTime(&ArgusThisTime, &ArgusLastTime);
               thisRate = ((dThisTime.tv_sec * 1000000) + dThisTime.tv_usec)/RaUpdateRate;
               thisRate = (thisRate > 100000) ? 100000 : thisRate;

               ts.tv_nsec =  thisRate * 1000;
               nanosleep (&ts, NULL);

               ArgusClientTimeout ();

               gettimeofday(&parser->ArgusRealTime, 0);

               if (ArgusLastRealTime.tv_sec > 0) {
                  dRealTime = *RaDiffTime(&parser->ArgusRealTime, &ArgusLastRealTime);
                  thisUsec = ((dRealTime.tv_sec * 1000000) + dRealTime.tv_usec) * RaUpdateRate;
                  dRealTime.tv_sec  = thisUsec / 1000000;
                  dRealTime.tv_usec = thisUsec % 1000000;

                  ArgusLastTime.tv_sec  += dRealTime.tv_sec;
                  ArgusLastTime.tv_usec += dRealTime.tv_usec;
                  if (ArgusLastTime.tv_usec > 1000000) {
                     ArgusLastTime.tv_sec++;
                     ArgusLastTime.tv_usec -= 1000000;
                  }
               }
               ArgusLastRealTime = parser->ArgusRealTime;
            }
         }
      } else
         ArgusLastTime = parser->ArgusRealTime;

      if (parser->RaMonMode) {
         struct ArgusRecordStruct *tns = ArgusCopyRecordStruct(ns);

         ns->canon.flow.hdr.subtype &= ~ARGUS_REVERSE;
         ns->canon.flow.hdr.argus_dsrvl8.qual &= ~ARGUS_DIRECTION;
 
         RaProcessThisRecord(parser, ns);

         ArgusReverseRecord(tns);

         tns->canon.flow.hdr.subtype &= ~ARGUS_REVERSE;
         tns->canon.flow.hdr.argus_dsrvl8.qual &= ~ARGUS_DIRECTION;

         RaProcessThisRecord(parser, tns);
         ArgusDeleteRecordStruct(parser, tns);

      } else {
         struct ArgusAggregatorStruct *agg = parser->ArgusAggregator;

         if (agg && agg->ArgusMatrixMode) {
            struct ArgusFlow *flow = (struct ArgusFlow *) &ns->canon.flow;

            switch (flow->hdr.subtype & 0x3F) {
               case ARGUS_FLOW_LAYER_3_MATRIX:
               case ARGUS_FLOW_CLASSIC5TUPLE: {
                  switch (flow->hdr.argus_dsrvl8.qual & 0x1F) {
                     case ARGUS_TYPE_IPV4: {
                        if (flow->ip_flow.ip_src > flow->ip_flow.ip_dst)
                           ArgusReverseRecord(ns);
                     }
                     break;

                     case ARGUS_TYPE_IPV6: {
                        int i;
                        for (i = 0; i < 4; i++) {
                           if (flow->ipv6_flow.ip_src[i] < flow->ipv6_flow.ip_dst[i])
                              break;

                           if (flow->ipv6_flow.ip_src[i] > flow->ipv6_flow.ip_dst[i]) {
                              ArgusReverseRecord(ns);
                              break;
                           }
                        }
                     }
                     break;
                  }
                  break;
               }

               default:
                  break;
            }
         }
         
         RaProcessThisRecord(parser, ns);
      }
   }
}

void
RaProcessThisRecord (struct ArgusParserStruct *parser, struct ArgusRecordStruct *ns)
{
   extern struct RaBinProcessStruct *RaBinProcess;
   struct ArgusRecordStruct *tns = NULL;
   int retn, offset = ArgusParser->Bflag, tstrat;
   extern int ArgusTimeRangeStrategy;

   if (ns->hdr.type & ARGUS_MAR) {
   } else {
      struct ArgusAggregatorStruct *agg = parser->ArgusAggregator;
      struct nff_insn *fcode = agg->filter.bf_insns;

      if (ArgusParser->tflag) {
         tstrat = ArgusTimeRangeStrategy;
         ArgusTimeRangeStrategy = 1;
      }

      if ((retn = ArgusFilterRecord (fcode, ns)) != 0) {
         if ((agg->rap = RaFlowModelOverRides(agg, ns)) == NULL)
            agg->rap = agg->drap;

         ArgusGenerateNewFlow(agg, ns);

         RaBinProcess->nadp.tperiod = 0.0;

         while ((tns = ArgusAlignRecord(parser, ns, &RaBinProcess->nadp)) != NULL) {
            if ((retn = ArgusCheckTime (parser, tns)) != 0) {
               if ((tns->canon.metric.src.pkts + tns->canon.metric.dst.pkts) > 0) {
                  if ((retn = ArgusInsertRecord(parser, RaBinProcess, tns, offset) < 0)) {
                     ArgusDeleteRecordStruct(parser, tns);
                  }
               } else 
                  ArgusDeleteRecordStruct(parser, tns);
            } else
               ArgusDeleteRecordStruct(parser, tns);
         }
      }

      if (ArgusParser->tflag)
         ArgusTimeRangeStrategy = tstrat;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessThisRecord (0x%x) done\n", ns); 
#endif
}


int
RaSendArgusRecord(struct ArgusRecordStruct *argus)
{
   int retn = 1;
   char buf[0x10000];

   if (!(retn = ArgusCheckTime (ArgusParser, argus)))
      return (retn);

   if (RaBinProcess->nadp.soft) {
      struct ArgusAdjustStruct *nadp = &RaBinProcess->nadp;
      unsigned int dsecs;

      dsecs = ((argus->canon.time.src.start.tv_sec - nadp->startSecs) / nadp->size) * nadp->size;

      argus->canon.time.src.start.tv_sec = nadp->startSecs + dsecs;
      argus->canon.time.src.end.tv_sec = argus->canon.time.src.start.tv_sec + nadp->size;

      argus->canon.time.src.start.tv_usec = 0;
      argus->canon.time.src.end.tv_usec = 0;
   }
 
   if ((ArgusParser->ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusParser->ArgusWfileList)))) {
      struct ArgusWfileStruct *wfile = NULL;
      struct ArgusListObjectStruct *lobj = NULL;
      int i, count = ArgusParser->ArgusWfileList->count;

      if ((lobj = ArgusParser->ArgusWfileList->start) != NULL) {
         for (i = 0; i < count; i++) {
            if ((wfile = (struct ArgusWfileStruct *) lobj) != NULL) {
               int retn = 1;
               if (wfile->filterstr) {
                  struct nff_insn *wfcode = wfile->filter.bf_insns;
                  retn = ArgusFilterRecord (wfcode, argus);
               }

               if (retn != 0) {
                  if ((ArgusParser->exceptfile == NULL) || strcmp(wfile->filename, ArgusParser->exceptfile)) {
                     struct ArgusRecord *argusrec = NULL;
                     char buf[2048];

                     if ((argusrec = ArgusGenerateRecord (argus, 0L, buf)) != NULL) {
#ifdef _LITTLE_ENDIAN
                        ArgusHtoN(argusrec);
#endif
                        ArgusWriteNewLogfile (ArgusParser, argus->input, wfile, argusrec);
                     }
                  }
               }
            }
            lobj = lobj->nxt;
         }
      }

   } else {
      if (!ArgusParser->qflag) {
         if (ArgusParser->Lflag) {
            if (ArgusParser->RaLabel == NULL)
               ArgusParser->RaLabel = ArgusGenerateLabel(ArgusParser, argus);

            if (!(ArgusParser->RaLabelCounter++ % ArgusParser->Lflag)) {
               if (ArgusParser->Gflag) {
                  printf ("Columns=%s\n", ArgusParser->RaLabel);
               } else
                  printf ("%s", ArgusParser->RaLabel);
            }

            if (ArgusParser->Lflag < 0)
               ArgusParser->Lflag = 0;

            if (ArgusParser->Gflag) {
               switch (ArgusParser->RaPrintMode) {
                  case RA_PRINTSRCID:
                     printf ("Probes=\n");
                     break;

                  case RA_PRINTPROTO: {
                     printf ("Protos=\n");
                     break;
                  }

                  default: {
                     printf ("Objects=\n");
                     break;
                  }
               }
            }
            printf ("\n");
         }

         *(int *)&buf = 0;
         ArgusPrintRecord(ArgusParser, buf, argus, MAXSTRLEN);
         fprintf (stdout, "%s\n", buf);
         fflush(stdout);
      }
   }

   return (retn);
}


void ArgusWindowClose(void);

void ArgusWindowClose(void) { 
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusWindowClose () returning\n"); 
#endif
}


char *
RaSplitFilename (struct ArgusAdjustStruct *nadp)
{
   char *retn = NULL, tmpbuf[MAXSTRLEN];
   char *filename = nadp->filename;
   int len, i = 1, carry = 0;

   if (filename != NULL) {
      len = strlen(filename);

      for (i = 0; i < nadp->slen; i++)
         if (filename[len - (i + 1)] == 'z')
            carry++;

      if ((carry == (nadp->slen - 1)) && (filename[len - nadp->slen] == 'y')) {
         strncpy(tmpbuf, filename, MAXSTRLEN);
         tmpbuf[strlen(tmpbuf) - nadp->slen] = 'z';
         for (i = 0; i < nadp->slen; i++)
            strcat(tmpbuf, "a");
         nadp->slen++;

      } else {
         for (i = 0, carry = 0; i < nadp->slen; i++) {
            if (filename[len - (i + 1)] == 'z') {
               filename[len - (i + 1)] = 'a';
            } else {
               filename[len - (i + 1)]++;
               break;
            }
         }
         strncpy (tmpbuf, filename, MAXSTRLEN);
      }

      if (nadp->filename)
         free(nadp->filename);

      nadp->filename = strdup(tmpbuf);
      retn = nadp->filename;
   }


#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaSplitFilename (0x%x) returning %s\n", nadp, retn); 
#endif

   return (retn);
}

int
RaProcessSplitOptions(struct ArgusParserStruct *parser, char *str, int len, struct ArgusRecordStruct *ns)
{
   char resultbuf[MAXSTRLEN], tmpbuf[MAXSTRLEN];
   char *ptr = NULL, *tptr = str;
   int retn = 0, i, x, slen = 0;

   bzero (resultbuf, len);

   while ((ptr = strchr (tptr, '$')) != NULL) {
      *ptr++ = '\0';
      slen = strlen(resultbuf);
      snprintf (&resultbuf[slen], MAXSTRLEN - slen, tptr);

      for (i = 0, x = 0; x < MAX_PRINT_ALG_TYPES; x++) {
         if (!strncmp (RaPrintAlgorithmTable[x].field, ptr, strlen(RaPrintAlgorithmTable[x].field))) {
            bzero (tmpbuf, MAXSTRLEN);
            RaPrintAlgorithmTable[x].print(parser, tmpbuf, ns, RaPrintAlgorithmTable[x].length);

            while (isspace((int)tmpbuf[strlen(tmpbuf) - 1]))
               tmpbuf[strlen(tmpbuf) - 1] = '\0';

            while (isspace((int)tmpbuf[i])) i++;
            slen = strlen(resultbuf);
            snprintf (&resultbuf[slen], MAXSTRLEN - slen, "%s", &tmpbuf[i]);

            ptr += strlen(RaPrintAlgorithmTable[x].field);
            while (*ptr && (*ptr != '$'))
               bcopy (ptr++, &resultbuf[strlen(resultbuf)], 1);
            break;
         }
      }

      tptr = ptr;
      retn++;
   }

   if (retn) {
      bzero (str, len);
      bcopy (resultbuf, str, strlen(resultbuf));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaProcessSplitOptions(%s, %d, 0x%x): returns %d", str, len, ns, retn);
#endif

   return (retn);
}

