#!/usr/bin/perl

use warnings;
use strict;

use File::Spec;
use URI;
use lib '/usr/share/apt-cacher';
require('apt-cacher-lib.pl');

our $cfg = read_config('/etc/apt-cacher/apt-cacher.conf');

unless ($cfg->{distinct_namespaces}) {
    print "$0: Multiple namespace support is not enabled. See distinct_namespaces in man apt-cacher(1)\n";
    exit;
}

setup_ownership();

chdir $cfg->{cache_dir} || die "Unable to chdir to $cfg->{cache_dir}: $!";

my %namespaces;

foreach (glob('packages/*')) {
    my $filename = (File::Spec->splitpath($_))[2];
    next unless is_package_file($filename);
    # print "Considering $_\n";
    if (!-f "headers/$filename") {
	warn "Headers for $filename missing. Skipping";
	next;
    }
    if (my $url = get_original_url($filename)) {
	if (my $namespace = get_namespace([$url->path_segments])) {
	    print "Import $filename to $namespace\n";
	    foreach (qw(packages headers)) {
		unless (-d (my $dir = "$_/$namespace")) {
		    mkdir $dir || die "Unable to create directory $dir: $!";
		}
		link("$_/$filename", "$_/$namespace/$filename") || die "Failed to copy $_/$filename: $!";
	    }
	    $namespaces{$namespace} = 1 unless $namespaces{$namespace};
	}
	else {
	    print "No namespace for $filename. Leaving alone.\n";
	}
    }
    else {
	warn "Could not recover original URL for $filename";
    }
}

print "\n$0: Done!\n";
if (%namespaces) {
    print "Package files have been imported from $cfg->{cache_dir} into the namespaces ",
      namespaces_string(keys %namespaces),
".\nThe original files are still present, but will be removed next time
apt-cacher-cleanup.pl is run. You can do this manually, or wait for the cron
job.\n\n";
}

# Returns a string of the used namspaces, correctly formatted with ,/and separators
sub namespaces_string {
    my @namespaces = @_;

    return (@namespaces == 0) ? ''
      : (@namespaces == 1) ? $namespaces[0]
	: (@namespaces == 2) ? join(" and ", @namespaces)
	  : join(', ', @namespaces[0 .. ($#namespaces-1)]) . " and $namespaces[-1]";
}
