
#define LOCAL_DEBUG
#include "debug.h"

#include "config.h"
#include "acfg.h"
#include "dlcon.h"

#include "tcpconnect.h"
#include "fileitem.h"
#include <errno.h>
#include "dljob.h"
#include "sockio.h"

#define MAX_RETRIES 3

using namespace MYSTD;



dlcon::dlcon(bool bManualExecution, string *xff) :
m_bStopWhenIdle(bManualExecution)
{
	LOGSTART("dlcon::dlcon");
	m_wakepipe[0]=m_wakepipe[1]=-1;
	if (0==pipe(m_wakepipe))
	{
		set_nb(m_wakepipe[0]);
		set_nb(m_wakepipe[1]);
	}
	if(xff)
		m_sXForwardedFor=*xff;
}

void dlcon::EnqJob(tDlJob *todo)
{
	setLockGuard;
	
	m_qToReceive.push_back(todo);
			
	if (m_wakepipe[1]>=0)
		POKE(m_wakepipe[1]);
}

void dlcon::AddJob(tFileItemPtr m_pItem, 
		acfg::tRepoData *pBackends, const MYSTD::string & sPatSuffix)
{
	EnqJob(new tDlJob(this, m_pItem, pBackends, sPatSuffix));
}

void dlcon::AddJob(tFileItemPtr m_pItem, tHttpUrl hi)
{
	EnqJob(new tDlJob(this, m_pItem, hi.sHost, hi.sPath));
}

void dlcon::SignalStop()
{
	setLockGuard;
	
	// stop all activity as soon as possible
	m_bStopWhenIdle=true;

	/* forget all but the first, those has to be fetched by the own downloader. If first
	 * item is being downloaded it will bail out RSN because of getting into NOMOREUSERS state. */
	while(m_qToReceive.size()>1)
	{
		delete m_qToReceive.back();
		m_qToReceive.pop_back();
	}
	POKE(m_wakepipe[1]);
}

dlcon::~dlcon()
{
	LOGSTART("dlcon::~dlcon, Destroying dlcon");
	checkforceclose(m_wakepipe[0]);
	checkforceclose(m_wakepipe[1]);
}

void dlcon::WorkLoop()
{
	LOGSTART("dlcon::WorkLoop");
    string sErrorMsg;
    tSS sendBuf;
    
    // tmp stuff
    int nTolErrorCount(0);
	  
	if (!m_InBuf.init(acfg::dlbufsize))
	{
		aclog::err("500 Out of memory");
		return;
	}

	if(m_wakepipe[0]<0 || m_wakepipe[1]<0)
	{
		aclog::err("Error creating pipe file descriptors");
		return;
	}

	fd_set rfds, wfds;
	struct timeval tv;
	       
	bool bStopAddingRequests=false;

	// keep related state vars together
	struct
	{
		tTcpHandlePtr ptr;
		int state; // negative = not connected, otherwise count of requests in the pipeline
		int fd;
		inline void reset()
		{
			ptr.reset();
			state=fd=-1;
		}
		inline void recycle()
		{
			tcpconnect::RecycleIdleConnection(ptr);
			state = fd = -1;
		}
	} con;
	con.reset();

	next_dlcon_cycle:
	for(;;)
	{
		int nMaxFd=m_wakepipe[0];
		FD_ZERO(&rfds);
		FD_ZERO(&wfds);
        FD_SET(m_wakepipe[0], &rfds);
        tDlJob *pCurJob=NULL;

		{
        	// needs to protect against concurent queue modification
			setLockGuard;
			
			dbgline;
			if(m_qToReceive.empty())
			{
				dbgline;
				// if we wait then it might time out, but others may make use of it ATM
				if(0 == con.state)
					con.recycle();

				// received a signal to stop the work as soon as the process is completed?
				if(m_bStopWhenIdle)
					return;
				
				dbgline;

				// and just wait for parent notification there
				goto ready_for_select;
			}

			pCurJob = m_qToReceive.front();

			if(pCurJob->bSuggestReconnect)
			{
				pCurJob->bSuggestReconnect=false;
				ldbg("found host change flag, prepare reconnection");
				if(0 == con.state)
					con.recycle();
				else
					con.reset();
				nTolErrorCount=0; // new host, new luck...
			}

			/* fresh state and/or going for reconnection. Pick up a working hostname and
		    * connect to it.
		    * */

			if(con.state<0)
			{
				dbgline;
				// set this right now... no matter how the rest looks, current
				// requests will be resend now anyways
				//
				m_InBuf.clear();
				sendBuf.clear();
				bStopAddingRequests=false;
				// just be sure about that
				con.reset();

				// bring job queue into a fresh state
				for(dljIter it=m_qToReceive.begin(); it!=m_qToReceive.end(); )
				{
					tDlJob *p=*it;
					p->bSuggestReconnect=p->bRequestEnqueued=false;
					dbgline;
					if(p->HasBrokenStorage())
					{
						if(p == pCurJob)
							pCurJob=NULL;
						delete p;
						m_qToReceive.erase(it++);
						continue;
					}
					++it;
				}
				if(!pCurJob) // lost front job, check the state
					continue;

				if(!pCurJob->FindConfig()) // bad, no usable host in the first item...
				{
					pCurJob->UnregDownloader(sErrorMsg.empty()
							? "503 No usable download host found"
									: sErrorMsg);
					delete pCurJob;
					m_qToReceive.pop_front();
					continue;
				}
				
				// can do internal work without locking queue (against modifications)
				__lockguard.unLock();

				const string sTargetHost = pCurJob->GetPeerName();
				ldbg("new target host: "<<sTargetHost);


				acfg::tRepoData::IHookHandler *pobs = pCurJob->GetConnStateObserver();
				if(pobs)
					pobs->JobConnect();

				con.ptr = tcpconnect::CreateConnected(sTargetHost, sErrorMsg);

				ldbg("Connection state: "<<(con.ptr?"ok":"failed"));

				if(con.ptr)
				{
	                dbgline;
					con.ptr->SetStateObserver(pobs);
					con.state = 0;
					con.fd = con.ptr->GetFD();
				}
				else
					pCurJob->BlacklistBackend();

                // queue modification possible again
				__lockguard.reLock();
				dbgline;
				
				// recheck the queue, drop those without any possible download strategy
				for(dljIter it=m_qToReceive.begin(); it!=m_qToReceive.end(); )
				{
					tDlJob *p=*it;
					if(!p->FindConfig()) // oh no, no more potential mirrors. Must tell the user.
					{
						p->UnregDownloader(sErrorMsg);
						if(pCurJob == p)
							pCurJob = NULL;
						delete p;
						m_qToReceive.erase(it++);
						continue;
					}
					++it;
				}
				// ooops, lost current job?
				if(!pCurJob)
				{
					// can shift the pointer to the next if the connection is usable?
					if(con.ptr && !m_qToReceive.empty()
							&& m_qToReceive.front()->GetPeerName() == con.ptr->GetHostname())
					{
						pCurJob=m_qToReceive.front();
					}
					else // this was a fresh connection, reuse it
						con.recycle();
					continue;
				}
				
				if(!con.ptr) // failed above? will recheck the connection state
					continue;

				for(dljIter it=m_qToReceive.begin(); it!=m_qToReceive.end(); it++)
				{
					dbgline;
					tDlJob *p=*it;
					if(p->AppendRequest(sTargetHost, sendBuf, con.ptr->GetProxyData()))
						con.state++;
					else
					{
						// refused? -> needs to reconnect later
						p->bSuggestReconnect=true;
						bStopAddingRequests=true;  // MUST set this
						dbgline;
						break;
					}
				}

			} // (re)connect done

			dbgline;
			// connected!
          
		} // end of locked section	
		
		ready_for_select:

		if(con.fd>=0)
		{
			FD_SET(con.fd, &rfds);
			nMaxFd=std::max(con.fd, nMaxFd);
			
			if(sendBuf.size())
			{
				ldbg("Needs to send " << sendBuf.size() << " bytes")
				FD_SET(con.fd, &wfds);
				nMaxFd=std::max(con.fd, nMaxFd);
			}
		}

		ldbg("select dlcon");
		tv.tv_sec = acfg::nettimeout;
		tv.tv_usec = 0;
		
		int r=select(nMaxFd+1, &rfds, &wfds, NULL, &tv);
		
		if (r<0)
		{
			if (EINTR == errno)
				continue;

			aclog::errnoFmter fer("FAILURE: select, ");
			LOG(fer);
		    sErrorMsg=string("500 Internal malfunction, ")+fer;
		    goto drop_and_restart_stream;
		}
		else if(r==0) // must be a timeout
		{
			if(con.state>0)
			{
				sErrorMsg=aclog::errnoFmter("500 Connection abort, ");
				aclog::err("Warning, disconnected during package download");
			}
			goto drop_and_restart_stream;
		}

		if (FD_ISSET(m_wakepipe[0], &rfds))
		{
			dbgline;
			for(int tmp; read(m_wakepipe[0], &tmp, 1) > 0; ) ;
			
			setLockGuard;
			// got new stuff? and needs to prepare requests? and queue is not empty?
			// walk around and send requests for them, if possible
			if(	!bStopAddingRequests && con.ptr && !m_qToReceive.empty())
			{
				ldbg("Preparing requests for new stuff...");
				
				// walks backward from the end to find a good position where previous
				// request-sending was interrupted instead of just starting from the beginning.
				
				dljIter it=m_qToReceive.end();
				for(it--; it!=m_qToReceive.begin(); it--)
					if( (*it)->bRequestEnqueued )
						break;
				
				for(; it!=m_qToReceive.end(); it++)
				{
					tDlJob *pjn=*it;
					// one position to far in backwalk
					if(pjn->bRequestEnqueued)
						continue;
					
					// found the first unseen, do stuff
					
					pjn->bRequestEnqueued=true;
					if(!pjn->FindConfig() || !pjn->AppendRequest(con.ptr->GetHostname(),
							sendBuf, con.ptr->GetProxyData()))
					{
						// bad stuff, to be reported later
						bStopAddingRequests=true;
						pjn->bSuggestReconnect=true;
						break;
					}
					else
						con.state++;
				}
			}
		}

		if (con.fd>=0 && sendBuf.size() && FD_ISSET(con.fd, &wfds))
		{
			FD_CLR(con.fd, &wfds);

			ldbg("Sending data...\n" << sendBuf);
			int s=::send(con.fd, sendBuf.data(), sendBuf.length(), MSG_NOSIGNAL);
			ldbg("Sent " << s << " bytes from " << sendBuf.length());
			if (s<0)
			{
				if(errno!=EAGAIN && errno!=EINTR)
                {
                    sErrorMsg="502 Peer communication error";
                    goto drop_and_restart_stream;
                }
                // else retry later...
			}
            else 
                sendBuf.drop(s);
		}
	    
		if (con.fd>=0 && FD_ISSET(con.fd, &rfds))
		{
			if(!pCurJob) // huh, sends something without request? Maybe disconnect hang (zero read)?
			{
				ldbg("BUG: not expected data");
				// just disconnect, and maybe reconnect
				con.reset();
				continue;
			}
			int r = m_InBuf.sysread(con.fd);
			if (r <= 0)
			{
				// pickup the error code for later and kill current connection ASAP
				sErrorMsg=aclog::errnoFmter("502 ");
				con.reset();
				
				if(pCurJob)
				{
					/*
					 * disconnected? Maybe because of end of Keep-Alive sequence, or
					 * having hit the timeout at the moment of resuming... 
					 * Trying to work around (n retries) if possible
					 */
					
					if(++nTolErrorCount < MAX_RETRIES)
					{
						
						if( ! pCurJob->HasStarted())
						{
							ldbg("MAX_RETRIES not reached, just reconnect");
							con.reset();
							continue;
						}
					}
					else
					{   // too many retries for this host :-(
						pCurJob->BlacklistBackend();
						nTolErrorCount=0;
					}
				}
				
                goto drop_and_restart_stream;
			}
        }

		while( ! m_InBuf.empty() && pCurJob)
		{
			ldbg("Processing job for " << pCurJob->RemoteUri() );
			tDlJob::tDlResult res=pCurJob->ProcessIncomming(m_InBuf, sErrorMsg);
			ldbg("... incomming data processing result: " << res);
			switch(res)
			{
				case(tDlJob::R_MOREINPUT):
				case(tDlJob::R_NOOP):
					goto next_dlcon_cycle; // will get more in the next loop
				case(tDlJob::R_SKIPITEM): // will restart the stream
				// item was ours but download aborted... might do more fine recovery in the future
				case(tDlJob::R_ERROR_LOCAL):
				case(tDlJob::R_ERROR_REMOTE):
				case(tDlJob::R_NEXTSTATE): // this one should never appear, but who knows...
					goto drop_and_restart_stream;
									
				case(tDlJob::R_DONE):
				{
					nTolErrorCount=0; // well done and can sleep now, should be resumed carefully later
					con.state--;
					
					setLockGuard;
					delete pCurJob;
					m_qToReceive.pop_front();
					pCurJob = m_qToReceive.empty() ? NULL : m_qToReceive.front();
					ldbg("Remaining dlitems: " << m_qToReceive.size());
					continue;
				}
				break;
			}
        }
		
		continue;
		
		drop_and_restart_stream:

		ldbg("Resetting pCurJob, resetting connection");
		con.reset();
		if(pCurJob)
		{
			pCurJob->UnregDownloader(sErrorMsg);
			setLockGuard;
			delete pCurJob;
			m_qToReceive.pop_front();
			pCurJob=NULL;
			continue;
		}
	}
}
