package org.inria.bmajwatcher.client.ui;

import java.util.List;
import java.util.Map;

import org.inria.bmajwatcher.client.i18n.BmajWatcherConstants;
import org.inria.bmajwatcher.client.services.BankDetailServiceAsync;
import org.inria.bmajwatcher.client.services.BankStatServiceAsync;
import org.inria.bmajwatcher.client.ui.styles.Resources;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.DecoratedTabPanel;
import com.google.gwt.user.client.ui.DialogBox;
import com.google.gwt.user.client.ui.DisclosurePanel;
import com.google.gwt.user.client.ui.Grid;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.Panel;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.visualization.client.DataTable;
import com.google.gwt.visualization.client.AbstractDataTable.ColumnType;
import com.google.gwt.visualization.client.visualizations.OrgChart;
import com.smartgwt.client.data.Record;
import com.smartgwt.client.util.SC;
import com.smartgwt.client.widgets.tree.TreeGrid;
import com.smartgwt.client.widgets.tree.TreeGridField;

/**
 * Dialog that displays charts and detailed information about a bank.
 * 
 * @author rsabas
 *
 */
public class BankDetailDialog extends DialogBox implements ClickHandler {
	
	private VerticalPanel graphPanel; 
	private LineChartGenerator cGen;
	
	private boolean graphDrawn = false;
	private boolean detailDrawn = false;
	
	private Anchor sizeAnchor = new Anchor(BmajWatcherConstants.INSTANCE.size());
	private Anchor bandwidthAnchor = new Anchor(BmajWatcherConstants.INSTANCE.bandwidth());
	private Anchor fileAnchor = new Anchor(BmajWatcherConstants.INSTANCE.downloadedFiles());
	private Anchor sessionAnchor = new Anchor(BmajWatcherConstants.INSTANCE.sessionDuration());
	private Anchor globalTimeAnchor = new Anchor(BmajWatcherConstants.INSTANCE.globalTime());
	
	private TreeGrid tree = null;
	

	public BankDetailDialog(final Record bank, final boolean adminMode) {
		super(false, false);
		
		final String bankName = bank.getAttribute(BankGridDataSource.NAME);
		
		final VerticalPanel overviewPanel = new VerticalPanel();
		
		BankDetailServiceAsync.INSTANCE.getDetailedBankInfo(bankName, new AsyncCallback<Map<String,String>>() {
			
			@Override
			public void onFailure(Throwable caught) {
				SC.warn("Bank detail loading failure");
			}

			@Override
			public void onSuccess(final Map<String, String> info) {
				
				
				BankDetailServiceAsync.INSTANCE.getFields(adminMode, new AsyncCallback<Map<String,List<String>>>() {
					
					@Override
					public void onFailure(Throwable caught) {
						SC.warn("Release fields loading failure");
					}
					
					@Override
					public void onSuccess(Map<String, List<String>> result) {
						for (String gridName : result.keySet()) {
							List<String> attributes = result.get(gridName);
							Grid grid = new Grid(attributes.size(), 2);
							grid.setStyleName(Resources.INSTANCE.mainCss().detailGrid());
							for (int i = 0; i < attributes.size(); i++) {
								String[] split = attributes.get(i).split("_:_");
								grid.setText(i, 0, split[1]);
								
								if (split[1].equals("Last log") && info.get(split[0]) != null
										&& !info.get(split[0]).trim().isEmpty()) {
									String url = info.get(split[0]);
									// Get the 3 last /
									if (url != null) {
										url = url.substring(0, url.lastIndexOf('/'));
										url = url.substring(0, url.lastIndexOf('/'));
										int index = url.lastIndexOf('/') + 1;
										
										String relativePath = info.get(split[0]).substring(index);
										
										
										grid.setHTML(i, 1, "<a href=\"logs/" + relativePath + "\">" + info.get(split[0]) + "</a>");
									}
								} else if (split[1].equals("Url")) {
									grid.setHTML(i, 1, "<a href=\"" + info.get(split[0]) + "\">" + info.get(split[0]) + "</a>");
								} else if (split[1].equals("Production directories")) {
									grid.setHTML(i, 1, info.get(split[0]));
								} else {
									if (adminMode && (split[1].equalsIgnoreCase("Postprocesses") || split[1].equalsIgnoreCase("Preprocesses"))) {
										grid.setWidget(i, 1, getProcessTree(info.get(split[0])));
									}
									else
										grid.setText(i, 1, info.get(split[0]));
								}
								
								grid.getCellFormatter().setStyleName(i, 0, Resources.INSTANCE.mainCss().cellLabel());
								grid.getCellFormatter().setStyleName(i, 1, Resources.INSTANCE.mainCss().cellValue());
							}
							
							DisclosurePanel closeablePanel = new DisclosurePanel(gridName);
							closeablePanel.setOpen(true);
							closeablePanel.setContent(grid);
							if (!gridName.equals("Post-processes") && !gridName.equals("Pre-processes"))
								overviewPanel.insert(closeablePanel, 0);
							else
								overviewPanel.add(closeablePanel);
						}
					}
					
				});
			}
		});

		HorizontalPanel mainGraphPanel = new HorizontalPanel();
		VerticalPanel graphMenu = new VerticalPanel();
		VerticalPanel topMenu = new VerticalPanel();
		graphMenu.add(topMenu);
		graphMenu.setStylePrimaryName(Resources.INSTANCE.mainCss().graphMenu());
		mainGraphPanel.add(graphMenu);
		graphPanel = new VerticalPanel();
		graphPanel.add(new Label("Loading..."));
		mainGraphPanel.add(graphPanel);
		
		final HorizontalPanel detailPanel = new HorizontalPanel();
		
		/*
		 * TabPanel
		 */
		DecoratedTabPanel tabPanel = new DecoratedTabPanel();
		tabPanel.add(overviewPanel, BmajWatcherConstants.INSTANCE.bankOverview());
		if (adminMode)
			tabPanel.add(detailPanel, BmajWatcherConstants.INSTANCE.bankSessions());
		tabPanel.add(mainGraphPanel, BmajWatcherConstants.INSTANCE.bankStats());
		tabPanel.selectTab(0);
		
		tabPanel.addSelectionHandler(new SelectionHandler<Integer>() {
			
			@Override
			public void onSelection(SelectionEvent<Integer> event) {
				if (event.getSelectedItem() == (2 - (adminMode ? 0 : 1)) 
						&& !graphDrawn) {
					BankStatServiceAsync.INSTANCE.getBankStats(bankName, new AsyncCallback<List<Map<String,String>>>() {
						
						@Override
						public void onSuccess(List<Map<String, String>> result) {
							graphPanel.clear();
							cGen = new LineChartGenerator(result);
							graphPanel.add(cGen.getSizeEvolution());
							graphDrawn = true;
						}
						
						@Override
						public void onFailure(Throwable caught) {
							SC.warn("Failure");
						}
					});
				} else if (adminMode && event.getSelectedItem() == 1 && !detailDrawn) {
//					Peut etre que c'est pas reinit ?? plantage ?
					getBankDetail(detailPanel, bankName);
					detailDrawn = true;
				}
				
			}
		});
		
		sizeAnchor.addClickHandler(this);
		bandwidthAnchor.addClickHandler(this);
		fileAnchor.addClickHandler(this);
		sessionAnchor.addClickHandler(this);
		globalTimeAnchor.addClickHandler(this);
		topMenu.setHorizontalAlignment(HasHorizontalAlignment.ALIGN_RIGHT);
		topMenu.add(sizeAnchor);
		topMenu.add(bandwidthAnchor);
		topMenu.add(fileAnchor);
		topMenu.add(sessionAnchor);
		topMenu.add(globalTimeAnchor);
		
		Button close = new Button("Close");
		close.addClickHandler(new com.google.gwt.event.dom.client.ClickHandler() {
			
			@Override
			public void onClick(com.google.gwt.event.dom.client.ClickEvent event) {
				if (tree != null)
					tree.hide();
				hide();
			}
		});
		
	  	setHTML("<b>" + bankName + "</b>");
		addStyleName(Resources.INSTANCE.mainCss().bankDetail());
		VerticalPanel vp = new VerticalPanel();
		vp.add(tabPanel);
		vp.add(close);
		vp.setCellHorizontalAlignment(close, HasHorizontalAlignment.ALIGN_RIGHT);
		add(vp);
	}
	
	
	private void getBankDetail(Panel pnl, String bankName) {
		
		tree = new TreeGrid();
		tree.setLoadDataOnDemand(true);
		tree.setWidth(600);
		tree.setHeight(400);
		tree.setAnimateFolders(false);
		tree.setDataSource(BankTreeDataSource.getInstance(bankName));
		tree.setAutoFetchData(true);
		TreeGridField field = new TreeGridField("value");
		tree.setFields(field);
		
		pnl.add(tree);
		tree.redraw();
	}
	
	private OrgChart getProcessTree(String processes) {
		
		if (processes != null && !processes.trim().isEmpty()) {
			String[] split = processes.split(";");
			int currentIndex = 0;
			DataTable currentData = DataTable.create();
			currentData.addColumn(ColumnType.STRING, "Process");
			currentData.addColumn(ColumnType.STRING, "Parent");
			currentData.addColumn(ColumnType.STRING, "Tooltip");
			String parentBlock = "";
			String parentMeta = "";
			
			currentData.addRow();
			currentData.setValue(currentIndex++, 0, "Root");
			
			for (String elt : split) {
				String[] pair = elt.split("=");
				if (pair[0].equals("BLOCK")) {
					currentData.addRow();
					if (pair.length == 2) {
						currentData.setValue(currentIndex++, 0, pair[1]);
						parentBlock = pair[1];
					} else {
						currentData.setValue(currentIndex++, 0, "BLOCK");
						parentBlock = "BLOCK";
					}
					currentData.setValue(currentIndex - 1, 1, "Root");
					
				} else if (pair[0].equals("META")) {
					currentData.addRow();
					currentData.setValue(currentIndex++, 0, pair[1]); // position 0 is element name
					currentData.setValue(currentIndex - 1, 1, parentBlock); // position 1 is element parent name
					parentMeta = pair[1];
				} else { // PS
					currentData.addRow();
					String keyName = pair[1].split("_:_")[0];
					String exe = pair[1].split("_:_")[1];
					currentData.setValue(currentIndex++, 0, "<span style=\"color:#A44B98\">" + keyName + "</span>");
					currentData.setValue(currentIndex - 1, 1, parentMeta);
					currentData.setValue(currentIndex - 1, 2, exe); // tooltip
				}
			}
			
			OrgChart.Options opts = OrgChart.Options.create();
			opts.setSize(OrgChart.Size.MEDIUM);
			opts.setAllowHtml(true);
				
			return new OrgChart(currentData, opts);
		}
		
		return null;
	}
	
	/*
	 * 
	 * EVENT HANDLING WHEN ANCHOR IS CLICKED
	 * 
	 */
	@Override
	public void onClick(ClickEvent event) {
		graphPanel.clear();

		if (event.getSource() == sizeAnchor) {
			graphPanel.add(cGen.getSizeEvolution());
		} else if (event.getSource() == fileAnchor) {
			graphPanel.add(cGen.getFileDlCount());
		} else if (event.getSource() == globalTimeAnchor) {
			graphPanel.add(cGen.getGlobalTimeEvolution());
		} else if (event.getSource() == sessionAnchor) {
			graphPanel.add(cGen.getSessionDuration());
		} else if (event.getSource() == bandwidthAnchor) {
			graphPanel.add(cGen.getBandwidthEvolution());
		}
	}
}
