// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/arc/input_method_manager/input_connection_impl.h"

#include "base/strings/utf_string_conversions.h"
#include "base/test/scoped_task_environment.h"
#include "chrome/browser/chromeos/arc/input_method_manager/test_input_method_manager_bridge.h"
#include "chrome/browser/ui/ash/keyboard/chrome_keyboard_controller_client_test_helper.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "ui/base/ime/chromeos/mock_input_method_manager.h"
#include "ui/base/ime/dummy_text_input_client.h"
#include "ui/base/ime/ime_bridge.h"
#include "ui/base/ime/mock_ime_input_context_handler.h"
#include "ui/base/ime/mock_input_method.h"

namespace arc {

namespace {

class DummyInputMethodEngineObserver
    : public input_method::InputMethodEngineBase::Observer {
 public:
  DummyInputMethodEngineObserver() = default;
  ~DummyInputMethodEngineObserver() override = default;

  void OnActivate(const std::string& engine_id) override {}
  void OnFocus(
      const ui::IMEEngineHandlerInterface::InputContext& context) override {}
  void OnBlur(int context_id) override {}
  void OnKeyEvent(
      const std::string& engine_id,
      const input_method::InputMethodEngineBase::KeyboardEvent& event,
      ui::IMEEngineHandlerInterface::KeyEventDoneCallback key_data) override {}
  void OnReset(const std::string& engine_id) override {}
  void OnDeactivated(const std::string& engine_id) override {}
  void OnCompositionBoundsChanged(
      const std::vector<gfx::Rect>& bounds) override {}
  bool IsInterestedInKeyEvent() const override { return true; }
  void OnSurroundingTextChanged(const std::string& engine_id,
                                const std::string& text,
                                int cursor_pos,
                                int anchor_pos,
                                int offset_pos) override {}
  void OnInputContextUpdate(
      const ui::IMEEngineHandlerInterface::InputContext& context) override {}
  void OnCandidateClicked(
      const std::string& component_id,
      int candidate_id,
      input_method::InputMethodEngineBase::MouseButtonEvent button) override {}
  void OnMenuItemActivated(const std::string& component_id,
                           const std::string& menu_id) override {}
  void OnScreenProjectionChanged(bool is_projected) override {}

 private:
  DISALLOW_COPY_AND_ASSIGN(DummyInputMethodEngineObserver);
};

class TestInputMethodManager
    : public chromeos::input_method::MockInputMethodManager {
 public:
  TestInputMethodManager()
      : state_(base::MakeRefCounted<
               chromeos::input_method::MockInputMethodManager::State>()) {}
  ~TestInputMethodManager() override = default;

  scoped_refptr<InputMethodManager::State> GetActiveIMEState() override {
    return state_;
  }

 private:
  scoped_refptr<State> state_;

  DISALLOW_COPY_AND_ASSIGN(TestInputMethodManager);
};

class TestIMEInputContextHandler : public ui::MockIMEInputContextHandler {
 public:
  explicit TestIMEInputContextHandler(ui::InputMethod* input_method)
      : input_method_(input_method) {}
  ~TestIMEInputContextHandler() override = default;

  ui::InputMethod* GetInputMethod() override { return input_method_; }

  void SendKeyEvent(ui::KeyEvent* event) override {
    ui::MockIMEInputContextHandler::SendKeyEvent(event);
    ++send_key_event_call_count_;
  }

  void Reset() {
    ui::MockIMEInputContextHandler::Reset();
    send_key_event_call_count_ = 0;
  }

  int send_key_event_call_count() const { return send_key_event_call_count_; }

 private:
  ui::InputMethod* const input_method_;

  int send_key_event_call_count_ = 0;

  DISALLOW_COPY_AND_ASSIGN(TestIMEInputContextHandler);
};

class InputConnectionImplTest : public testing::Test {
 public:
  InputConnectionImplTest() = default;
  ~InputConnectionImplTest() override = default;

  std::unique_ptr<InputConnectionImpl> createNewConnection(int context_id) {
    return std::make_unique<InputConnectionImpl>(engine_.get(), bridge_.get(),
                                                 context_id);
  }

  chromeos::InputMethodEngine* engine() { return engine_.get(); }

  TestIMEInputContextHandler* context_handler() { return &context_handler_; }

  ui::DummyTextInputClient* client() { return &text_input_client_; }

  ui::IMEEngineHandlerInterface::InputContext context() {
    return ui::IMEEngineHandlerInterface::InputContext{
        1,
        ui::TEXT_INPUT_TYPE_TEXT,
        ui::TEXT_INPUT_MODE_DEFAULT,
        0 /* flags */,
        ui::TextInputClient::FOCUS_REASON_MOUSE,
        true /* should_do_learning */};
  }

  void SetUp() override {
    ui::IMEBridge::Initialize();
    chromeos::input_method::InputMethodManager::Initialize(
        new TestInputMethodManager);
    bridge_ = std::make_unique<TestInputMethodManagerBridge>();
    engine_ = std::make_unique<chromeos::InputMethodEngine>();
    engine_->Initialize(std::make_unique<DummyInputMethodEngineObserver>(),
                        "test_extension_id", nullptr);
    chrome_keyboard_controller_client_test_helper_ =
        ChromeKeyboardControllerClientTestHelper::InitializeWithFake();

    // Enable InputMethodEngine.
    ui::IMEBridge::Get()->SetInputContextHandler(&context_handler_);
    input_method_.SetFocusedTextInputClient(&text_input_client_);
    engine()->Enable("test_component_id");
  }

  void TearDown() override {
    chrome_keyboard_controller_client_test_helper_.reset();
    ui::IMEBridge::Get()->SetInputContextHandler(nullptr);
    engine_.reset();
    bridge_.reset();
    chromeos::input_method::InputMethodManager::Shutdown();
    ui::IMEBridge::Shutdown();
  }

 private:
  base::test::ScopedTaskEnvironment scoped_task_environment_;
  std::unique_ptr<TestInputMethodManagerBridge> bridge_;
  std::unique_ptr<chromeos::InputMethodEngine> engine_;
  ui::DummyTextInputClient text_input_client_;
  ui::MockInputMethod input_method_{nullptr};
  TestIMEInputContextHandler context_handler_{&input_method_};
  std::unique_ptr<ChromeKeyboardControllerClientTestHelper>
      chrome_keyboard_controller_client_test_helper_;

  DISALLOW_COPY_AND_ASSIGN(InputConnectionImplTest);
};

}  // anonymous namespace

TEST_F(InputConnectionImplTest, CommitText) {
  auto connection = createNewConnection(1);
  engine()->FocusIn(context());

  context_handler()->Reset();
  connection->CommitText(base::ASCIIToUTF16("text"), 1);
  EXPECT_EQ(1, context_handler()->commit_text_call_count());
  EXPECT_EQ("text", context_handler()->last_commit_text());

  // Calling Commit() with '\n' invokes SendKeyEvent.
  context_handler()->Reset();
  connection->CommitText(base::ASCIIToUTF16("\n"), 1);
  EXPECT_EQ(0, context_handler()->commit_text_call_count());
  EXPECT_EQ(2, context_handler()->send_key_event_call_count());
  EXPECT_EQ(ui::VKEY_RETURN,
            context_handler()->last_sent_key_event().key_code());
  EXPECT_EQ(ui::ET_KEY_RELEASED,
            context_handler()->last_sent_key_event().type());

  engine()->FocusOut();
}

TEST_F(InputConnectionImplTest, DeleteSurroundingText) {
  auto connection = createNewConnection(1);
  engine()->FocusIn(context());

  context_handler()->Reset();
  connection->DeleteSurroundingText(1, 1);
  EXPECT_EQ(1, context_handler()->delete_surrounding_text_call_count());

  engine()->FocusOut();
}

TEST_F(InputConnectionImplTest, FinishComposingText) {
  auto connection = createNewConnection(1);
  engine()->FocusIn(context());

  // If there is no composing text, FinishComposingText() does nothing.
  context_handler()->Reset();
  connection->FinishComposingText();
  EXPECT_EQ(0, context_handler()->commit_text_call_count());

  // If there is composing text, FinishComposingText() calls CommitText() with
  // the text.
  context_handler()->Reset();
  connection->SetComposingText(base::ASCIIToUTF16("composing"), 0,
                               base::nullopt);
  EXPECT_EQ(0, context_handler()->commit_text_call_count());
  connection->FinishComposingText();
  EXPECT_EQ(1, context_handler()->commit_text_call_count());
  EXPECT_EQ("composing", context_handler()->last_commit_text());

  engine()->FocusOut();
}

TEST_F(InputConnectionImplTest, SetComposingText) {
  const base::string16 text = base::ASCIIToUTF16("text");
  auto connection = createNewConnection(1);
  engine()->FocusIn(context());

  context_handler()->Reset();
  connection->SetComposingText(text, 0, base::nullopt);
  EXPECT_EQ(1, context_handler()->update_preedit_text_call_count());
  EXPECT_EQ(
      text,
      context_handler()->last_update_composition_arg().composition_text.text);
  EXPECT_EQ(3u, context_handler()
                    ->last_update_composition_arg()
                    .composition_text.selection.start());
  // Commitiing the composing text calls ClearComposition() and CommitText().
  connection->CommitText(text, 0);
  EXPECT_EQ(2, context_handler()->update_preedit_text_call_count());
  EXPECT_EQ(
      base::ASCIIToUTF16(""),
      context_handler()->last_update_composition_arg().composition_text.text);
  EXPECT_EQ(1, context_handler()->commit_text_call_count());

  // CommitText should clear the composing text.
  connection->FinishComposingText();
  // commit_text_call_count() doesn't change.
  EXPECT_EQ(1, context_handler()->commit_text_call_count());

  // Selection range
  context_handler()->Reset();
  connection->SetComposingText(text, 0, base::make_optional<gfx::Range>(1, 3));
  EXPECT_EQ(1u, context_handler()
                    ->last_update_composition_arg()
                    .composition_text.selection.start());
  EXPECT_EQ(3u, context_handler()
                    ->last_update_composition_arg()
                    .composition_text.selection.end());

  engine()->FocusOut();
}

TEST_F(InputConnectionImplTest, SetSelection) {
  auto connection = createNewConnection(1);
  engine()->FocusIn(context());
  ASSERT_TRUE(client()->selection_history().empty());

  context_handler()->Reset();
  connection->SetSelection(gfx::Range(2, 4));
  EXPECT_FALSE(client()->selection_history().empty());
  EXPECT_EQ(2u, client()->selection_history().back().start());
  EXPECT_EQ(4u, client()->selection_history().back().end());

  engine()->FocusOut();
}

}  // namespace arc
