import { IIterator, IterableOrArrayLike } from '@lumino/algorithm';
import { IDisposable } from '@lumino/disposable';
import { ISignal } from '@lumino/signaling';
/**
 * A list which can be observed for changes.
 */
export interface IObservableList<T> extends IDisposable {
    /**
     * A signal emitted when the list has changed.
     */
    readonly changed: ISignal<this, IObservableList.IChangedArgs<T>>;
    /**
     * The type of this object.
     */
    readonly type: 'List';
    /**
     * The length of the list.
     *
     * #### Notes
     * This is a read-only property.
     */
    length: number;
    /**
     * Create an iterator over the values in the list.
     *
     * @returns A new iterator starting at the front of the list.
     *
     * #### Complexity
     * Constant.
     *
     * #### Iterator Validity
     * No changes.
     */
    iter(): IIterator<T>;
    /**
     * Remove all values from the list.
     *
     * #### Complexity
     * Linear.
     *
     * #### Iterator Validity
     * All current iterators are invalidated.
     */
    clear(): void;
    /**
     * Get the value at the specified index.
     *
     * @param index - The positive integer index of interest.
     *
     * @returns The value at the specified index.
     *
     * #### Undefined Behavior
     * An `index` which is non-integral or out of range.
     */
    get(index: number): T;
    /**
     * Insert a value into the list at a specific index.
     *
     * @param index - The index at which to insert the value.
     *
     * @param value - The value to set at the specified index.
     *
     * #### Complexity
     * Linear.
     *
     * #### Iterator Validity
     * No changes.
     *
     * #### Notes
     * The `index` will be clamped to the bounds of the list.
     *
     * #### Undefined Behavior
     * An `index` which is non-integral.
     */
    insert(index: number, value: T): void;
    /**
     * Insert a set of items into the list at the specified index.
     *
     * @param index - The index at which to insert the values.
     *
     * @param values - The values to insert at the specified index.
     *
     * #### Complexity.
     * Linear.
     *
     * #### Iterator Validity
     * No changes.
     *
     * #### Notes
     * The `index` will be clamped to the bounds of the list.
     *
     * #### Undefined Behavior.
     * An `index` which is non-integral.
     */
    insertAll(index: number, values: IterableOrArrayLike<T>): void;
    /**
     * Move a value from one index to another.
     *
     * @parm fromIndex - The index of the element to move.
     *
     * @param toIndex - The index to move the element to.
     *
     * #### Complexity
     * Constant.
     *
     * #### Iterator Validity
     * Iterators pointing at the lesser of the `fromIndex` and the `toIndex`
     * and beyond are invalidated.
     *
     * #### Undefined Behavior
     * A `fromIndex` or a `toIndex` which is non-integral.
     */
    move(fromIndex: number, toIndex: number): void;
    /**
     * Add a value to the back of the list.
     *
     * @param value - The value to add to the back of the list.
     *
     * @returns The new length of the list.
     *
     * #### Complexity
     * Constant.
     *
     * #### Iterator Validity
     * No changes.
     */
    push(value: T): number;
    /**
     * Push a set of values to the back of the list.
     *
     * @param values - An iterable or array-like set of values to add.
     *
     * @returns The new length of the list.
     *
     * #### Complexity
     * Linear.
     *
     * #### Iterator Validity
     * No changes.
     */
    pushAll(values: IterableOrArrayLike<T>): number;
    /**
     * Remove and return the value at a specific index.
     *
     * @param index - The index of the value of interest.
     *
     * @returns The value at the specified index, or `undefined` if the
     *   index is out of range.
     *
     * #### Complexity
     * Constant.
     *
     * #### Iterator Validity
     * Iterators pointing at the removed value and beyond are invalidated.
     *
     * #### Undefined Behavior
     * An `index` which is non-integral.
     */
    remove(index: number): T | undefined;
    /**
     * Remove a range of items from the list.
     *
     * @param startIndex - The start index of the range to remove (inclusive).
     *
     * @param endIndex - The end index of the range to remove (exclusive).
     *
     * @returns The new length of the list.
     *
     * #### Complexity
     * Linear.
     *
     * #### Iterator Validity
     * Iterators pointing to the first removed value and beyond are invalid.
     *
     * #### Undefined Behavior
     * A `startIndex` or `endIndex` which is non-integral.
     */
    removeRange(startIndex: number, endIndex: number): number;
    /**
     * Remove the first occurrence of a value from the list.
     *
     * @param value - The value of interest.
     *
     * @returns The index of the removed value, or `-1` if the value
     *   is not contained in the list.
     *
     * #### Complexity
     * Linear.
     *
     * #### Iterator Validity
     * Iterators pointing at the removed value and beyond are invalidated.
     */
    removeValue(value: T): number;
    /**
     * Set the value at the specified index.
     *
     * @param index - The positive integer index of interest.
     *
     * @param value - The value to set at the specified index.
     *
     * #### Complexity
     * Constant.
     *
     * #### Iterator Validity
     * No changes.
     *
     * #### Undefined Behavior
     * An `index` which is non-integral or out of range.
     */
    set(index: number, value: T): void;
}
/**
 * The namespace for IObservableList related interfaces.
 */
export declare namespace IObservableList {
    /**
     * The change types which occur on an observable list.
     */
    type ChangeType = 
    /**
     * Item(s) were added to the list.
     */
    'add'
    /**
     * An item was moved within the list.
     */
     | 'move'
    /**
     * Item(s) were removed from the list.
     */
     | 'remove'
    /**
     * An item was set in the list.
     */
     | 'set';
    /**
     * The changed args object which is emitted by an observable list.
     */
    interface IChangedArgs<T> {
        /**
         * The type of change undergone by the vector.
         */
        type: ChangeType;
        /**
         * The new index associated with the change.
         */
        newIndex: number;
        /**
         * The new values associated with the change.
         *
         * #### Notes
         * The values will be contiguous starting at the `newIndex`.
         */
        newValues: T[];
        /**
         * The old index associated with the change.
         */
        oldIndex: number;
        /**
         * The old values associated with the change.
         *
         * #### Notes
         * The values will be contiguous starting at the `oldIndex`.
         */
        oldValues: T[];
    }
}
/**
 * A concrete implementation of [[IObservableList]].
 */
export declare class ObservableList<T> implements IObservableList<T> {
    /**
     * Construct a new observable map.
     */
    constructor(options?: ObservableList.IOptions<T>);
    /**
     * The type of this object.
     */
    get type(): 'List';
    /**
     * A signal emitted when the list has changed.
     */
    get changed(): ISignal<this, IObservableList.IChangedArgs<T>>;
    /**
     * The length of the list.
     */
    get length(): number;
    /**
     * Test whether the list has been disposed.
     */
    get isDisposed(): boolean;
    /**
     * Dispose of the resources held by the list.
     */
    dispose(): void;
    /**
     * Create an iterator over the values in the list.
     *
     * @returns A new iterator starting at the front of the list.
     *
     * #### Complexity
     * Constant.
     *
     * #### Iterator Validity
     * No changes.
     */
    iter(): IIterator<T>;
    /**
     * Get the value at the specified index.
     *
     * @param index - The positive integer index of interest.
     *
     * @returns The value at the specified index.
     *
     * #### Undefined Behavior
     * An `index` which is non-integral or out of range.
     */
    get(index: number): T;
    /**
     * Set the value at the specified index.
     *
     * @param index - The positive integer index of interest.
     *
     * @param value - The value to set at the specified index.
     *
     * #### Complexity
     * Constant.
     *
     * #### Iterator Validity
     * No changes.
     *
     * #### Undefined Behavior
     * An `index` which is non-integral or out of range.
     */
    set(index: number, value: T): void;
    /**
     * Add a value to the end of the list.
     *
     * @param value - The value to add to the end of the list.
     *
     * @returns The new length of the list.
     *
     * #### Complexity
     * Constant.
     *
     * #### Notes
     * By convention, the oldIndex is set to -1 to indicate
     * an push operation.
     *
     * #### Iterator Validity
     * No changes.
     */
    push(value: T): number;
    /**
     * Insert a value into the list at a specific index.
     *
     * @param index - The index at which to insert the value.
     *
     * @param value - The value to set at the specified index.
     *
     * #### Complexity
     * Linear.
     *
     * #### Iterator Validity
     * No changes.
     *
     * #### Notes
     * The `index` will be clamped to the bounds of the list.
     *
     * By convention, the oldIndex is set to -2 to indicate
     * an insert operation.
     *
     * The value -2 as oldIndex can be used to distinguish from the push
     * method which will use a value -1.
     *
     * #### Undefined Behavior
     * An `index` which is non-integral.
     */
    insert(index: number, value: T): void;
    /**
     * Remove the first occurrence of a value from the list.
     *
     * @param value - The value of interest.
     *
     * @returns The index of the removed value, or `-1` if the value
     *   is not contained in the list.
     *
     * #### Complexity
     * Linear.
     *
     * #### Iterator Validity
     * Iterators pointing at the removed value and beyond are invalidated.
     */
    removeValue(value: T): number;
    /**
     * Remove and return the value at a specific index.
     *
     * @param index - The index of the value of interest.
     *
     * @returns The value at the specified index, or `undefined` if the
     *   index is out of range.
     *
     * #### Complexity
     * Constant.
     *
     * #### Iterator Validity
     * Iterators pointing at the removed value and beyond are invalidated.
     *
     * #### Undefined Behavior
     * An `index` which is non-integral.
     */
    remove(index: number): T | undefined;
    /**
     * Remove all values from the list.
     *
     * #### Complexity
     * Linear.
     *
     * #### Iterator Validity
     * All current iterators are invalidated.
     */
    clear(): void;
    /**
     * Move a value from one index to another.
     *
     * @parm fromIndex - The index of the element to move.
     *
     * @param toIndex - The index to move the element to.
     *
     * #### Complexity
     * Constant.
     *
     * #### Iterator Validity
     * Iterators pointing at the lesser of the `fromIndex` and the `toIndex`
     * and beyond are invalidated.
     *
     * #### Undefined Behavior
     * A `fromIndex` or a `toIndex` which is non-integral.
     */
    move(fromIndex: number, toIndex: number): void;
    /**
     * Push a set of values to the back of the list.
     *
     * @param values - An iterable or array-like set of values to add.
     *
     * @returns The new length of the list.
     *
     * #### Complexity
     * Linear.
     *
     * #### Notes
     * By convention, the oldIndex is set to -1 to indicate
     * an push operation.
     *
     * #### Iterator Validity
     * No changes.
     */
    pushAll(values: IterableOrArrayLike<T>): number;
    /**
     * Insert a set of items into the list at the specified index.
     *
     * @param index - The index at which to insert the values.
     *
     * @param values - The values to insert at the specified index.
     *
     * #### Complexity.
     * Linear.
     *
     * #### Iterator Validity
     * No changes.
     *
     * #### Notes
     * The `index` will be clamped to the bounds of the list.
     * By convention, the oldIndex is set to -2 to indicate
     * an insert operation.
     *
     * #### Undefined Behavior.
     * An `index` which is non-integral.
     */
    insertAll(index: number, values: IterableOrArrayLike<T>): void;
    /**
     * Remove a range of items from the list.
     *
     * @param startIndex - The start index of the range to remove (inclusive).
     *
     * @param endIndex - The end index of the range to remove (exclusive).
     *
     * @returns The new length of the list.
     *
     * #### Complexity
     * Linear.
     *
     * #### Iterator Validity
     * Iterators pointing to the first removed value and beyond are invalid.
     *
     * #### Undefined Behavior
     * A `startIndex` or `endIndex` which is non-integral.
     */
    removeRange(startIndex: number, endIndex: number): number;
    private _array;
    private _isDisposed;
    private _itemCmp;
    private _changed;
}
/**
 * The namespace for `ObservableList` class statics.
 */
export declare namespace ObservableList {
    /**
     * The options used to initialize an observable map.
     */
    interface IOptions<T> {
        /**
         * An optional initial set of values.
         */
        values?: IterableOrArrayLike<T>;
        /**
         * The item comparison function for change detection on `set`.
         *
         * If not given, strict `===` equality will be used.
         */
        itemCmp?: (first: T, second: T) => boolean;
    }
}
