/*
 *   meiomacs.h -- MEIO Manager device definitions
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
//-------------------------------------------------------------------------
//
// define the basic structure for a RESOURCE and its pointer
//
//-------------------------------------------------------------------------
struct resource;   
struct meio_connection;
typedef struct resource FAR *PRESOURCE;      
typedef PRESOURCE       FAR *PRSET;          
typedef struct meio_connection  FAR *PMEIO_CONNECTION; 

#define PRSNULL     (PRESOURCE)0

/*-------------------------------------------------------------------------
 *  define the resource owner list node
 *
 *------------------------------------------------------------------------*/
struct resource_owner
{
   PMEIO_CONNECTION       owner;
   struct resource_owner FAR *next;
};
typedef struct resource_owner FAR *RESOURCE_LIST;



/*
 * define the type for a Resource function
 */
#define RF_ARGS                              \
    PMEIO_CONNECTION      pConnection,  /* owning connection              */ \
    PRESOURCE             lpResource,   /* pointer to resource            */ \
    USHORT                usMessage,    /* message to pass to resource    */ \
    ULONG                 lParam1,      /* define attribute to update.    */ \
    ULONG                 lParam2       /* value of attribute.            */

#define RF_ARGLIST                           \
                    pConnection,  /* owning connection              */ \
                    lpResource,   /* pointer to resource            */ \
                    usMessage,    /* message to pass to resource    */ \
                    lParam1,      /* define attribute to update.    */ \
                    lParam2       /* value of attribute.            */

#define RF_NoPARMS                        \
                    pConnection,  /* owning connection              */ \
                    lpResource,   /* pointer to resource            */ \
                    usMessage     /* message to pass to resource    */

#define VOID_ARGS                             \
              (void)pConnection;  /* owning connection              */ \
              (void)lpResource;   /* pointer to resource            */ \
              (void)usMessage;    /* message to pass to resource    */ \
              (void)lParam1;      /* define attribute to update.    */ \
              (void)lParam2       /* value of attribute.            */

#define RF_NEWARGS(prs)                      \
                    pConnection,  /* owning connection              */ \
                    prs,          /* pointer to resource            */ \
                    usMessage,    /* message to pass to resource    */ \
                    lParam1,      /* define attribute to update.    */ \
                    lParam2       /* value of attribute.            */

/*
 * define pointer to a resource's methods
 */
typedef ULONG (FAR *PRESMETH)(RF_ARGS);

/*  OK, now we can define resource structure */
typedef struct resource
{
  USHORT                usHandle;
  USHORT                usType;
  USHORT                usState;         /* ownership state of the resource */
  USHORT                usOwnerCount;    /* number of owners */
  RESOURCE_LIST         lpOwnerList;     /* list of owners */
  PRESMETH              lpMethods;
} RESOURCE;


/*------------------------------------------------------------------------*/
/*   define user macros                                                   */
/*------------------------------------------------------------------------*/
/* calling syntax is...

ResourceDef(Rname,type) = { Rparms(handle,type,methods) , data };

*/

#define ResourceDef(name,type)                                 \
type##_RESOURCE FAR name

#define ResourceRef(name,type)                                 \
extern type##_RESOURCE FAR name


#define Rparms(handle,type,methods)                            \
                  {                                            \
                    handle                  /* usHandle    */  \
                   ,type                    /* usType      */  \
                   ,MEIO_RS_UNOWNED         /* usState     */  \
                   ,0                       /* usOwnerCount*/  \
                   ,(RESOURCE_LIST)0        /* lpOwnerList */  \
                   ,methods                 /* lpMethods   */  \
                  }           /* reset is initialized after call */


/* calling sequence is...

ResourceSet(RSname) resourcelist;

*/

#define ResourceSet(name)                                                     \
extern PVOID rs##name[];             /* create ref  */                \
ResourceDef(r##name,RESOURCESET)={Rparms(h##name,RESOURCESET,name),(PRSET)rs##name}; \
PVOID rs##name[] =                   /* create set  */

#define ResourceSRef(name)          \
ResourceRef(r##name,RESOURCESET)          /* create reference for object */

/*------------------------------------------------------------------------*/
/*   define resource types                                                */
/*------------------------------------------------------------------------*/
#define ANALOG_OWNER_PORT   0x1000
#define ANALOG_USER_PORT    0x2000
#define DIGITAL_OWNER_PORT  0x3000
#define DIGITAL_USER_PORT   0x4000
#define ANALOG_SWITCH       0x5000
#define DIGITAL_SWITCH      0x6000
#define ANALOG_INTERNAL     0x7000
#define RESOURCESET         0x8000

#if defined(_USE_NEGATIVE_DEFINITION)
#define ARESOURCESET        -28672  /* 0x8000 | ANALOG_OWNER_PORT  for dumb compilers */
#define DRESOURCESET        -20480  /* 0x8000 | DIGITAL_OWNER_PORT for dumb compilers */
#define BIOSTASK            -16384  /* 0xC000                      for dumb compilers */
#else
#define ARESOURCESET        0x8000 | ANALOG_OWNER_PORT   /* -28672 for dumb compilers */
#define DRESOURCESET        0x8000 | DIGITAL_OWNER_PORT  /* -20480 for dumb compilers */
#define BIOSTASK            0xC000                       /* -16384 for dumb compilers */
#endif

#define TYPEMASK            0xF000 // mask to isolate resource type from handle

/*------------------------------------------------------------------------*/
/*   define resource attributes                                           */
/*------------------------------------------------------------------------*/
typedef struct
{
    short gain;             // for current gain setting (signed value)
   USHORT control;          // for AGC, Mixing, etc
}  ANALOG_DATA;
#define ANALOG0   {0,0}

/*-------------------------------------------------------------------------*/
/* FEATURES BYTE BIT MAP                                                   */
/*-------------------------------------------------------------------------*/
#define F_AGC_AVAILABLE      0x01  /* Features AGC                         */
#define CW_ENABLE_AGC        0x02  /* AGC Enabled state                    */
#define F_MIX_AVAILABLE      0x10  /* Features Mixing Available            */
#define F_MIX_PERMANENT      0x20  /* Features Mixing Permanatly Enable    */
#define CW_ENABLE_MIXING     0x40  /* Mixing Enabled                       */
                                   /*--------------------------------------*/


typedef struct { RESOURCE Resource; PRSET lpResourceSet;} RESOURCESET_RESOURCE;

typedef struct { RESOURCE Resource; ANALOG_DATA data;} ANALOG_OWNER_PORT_RESOURCE;
typedef struct { RESOURCE Resource; ANALOG_DATA data;} ANALOG_USER_PORT_RESOURCE;
typedef struct { RESOURCE Resource; ANALOG_DATA data;} ANALOG_INTERNAL_RESOURCE;

typedef struct { RESOURCE Resource; USHORT data;} ANALOG_SWITCH_RESOURCE;

typedef struct { RESOURCE Resource; USHORT data;} DIGITAL_OWNER_PORT_RESOURCE;
typedef struct { RESOURCE Resource; USHORT data;} DIGITAL_USER_PORT_RESOURCE;
typedef struct { RESOURCE Resource; USHORT data;} DIGITAL_SWITCH_RESOURCE;
#define DIGITAL0   0

typedef struct { RESOURCE Resource;
                 struct
                 {
                    HMTASK hmTask;       // task handle for all operations
                    ULONG  ITCB_Addr;    // address of BIOS's ITCB
                    USHORT State;        // loaded or unloaded
                 } data;
               } BIOSTASK_RESOURCE;
#define BIOSTASK0  {0,0,0}


typedef RESOURCESET_RESOURCE FAR *PRS;
#define PRS_Entry(name)   (&r##name)
#define PRS_NULL          ((PRS)0)
#define GETRP(obj)        (&((obj)->Resource))

/*------------------------------------------------------------------------*/
/*   define default function prototypes                                   */
/*------------------------------------------------------------------------*/
ULONG     FAR AnalogOwnerDefault(RF_ARGS);
ULONG     FAR AnalogUserDefault(RF_ARGS);
ULONG     FAR AnalogSwitchDefault(RF_ARGS);
ULONG     FAR AnalogInternalDefault(RF_ARGS);
ULONG     FAR DigitalOwnerDefault(RF_ARGS);
ULONG     FAR DigitalUserDefault(RF_ARGS);
ULONG     FAR DigitalSwitchDefault(RF_ARGS);
ULONG     FAR generic_cf(RF_ARGS);
ULONG     FAR generic_scf(RF_ARGS);
ULONG     FAR generic_rf(RF_ARGS);
ULONG     FAR RS_iterate(RF_ARGS);
ULONG     FAR RS_SendMsg(RF_ARGS);


ULONG     FAR meSearchResource(RF_ARGS, PULONG pPlaceholder);
PRESOURCE FAR meFindResource(HMEIO hMeio, USHORT Port);
ULONG     FAR meFindResourceSet(HMEIO hMeio,         // MEIO handle (has DSP info)
                              USHORT OwnerPort,    // owner and user handles
                              USHORT UserPort,
                              PRS FAR *pprs);      // returned resourceset pointer

ULONG     FAR ProcessUpdate(RF_ARGS);
ULONG     FAR ProcessQuery(RF_ARGS);
ULONG     FAR ProcessSharing(RF_ARGS);
ULONG     FAR ProcessMixing(RF_ARGS);
ULONG     FAR QuerySharing(RF_ARGS);


/*------------------------------------------------------------------------*/
/*   define useful constants for bit parameters                           */
/*------------------------------------------------------------------------*/
#define B0        0x000001L
#define B1        0x000002L
#define B2        0x000004L
#define B3        0x000008L
#define B4        0x000010L
#define B5        0x000020L
#define B6        0x000040L
#define B7        0x000080L
#define B8        0x000100L
#define B9        0x000200L
#define B10       0x000400L
#define B11       0x000800L
#define B12       0x001000L
#define B13       0x002000L
#define B14       0x004000L
#define B15       0x008000L


/*------------------------------------------------------------------
 *
 * define RESOURCE messages
 *
 *------------------------------------------------------------------*/
enum {
      mRESETmsg
     ,mREFRESHmsg
     ,mQUERYmsg
     ,mTESTUPDATEmsg
     ,mUPDATEmsg
     ,mTESTmsg
     ,mRESERVEmsg
     ,mRELEASEmsg
     ,mMATCHmsg
     };

/* various Resource shareability states */
enum {
       MEIO_RS_UNOWNED
      ,MEIO_RS_EXCLUSIVE
      ,MEIO_RS_SHAREABLE
      ,MEIO_RS_SHARED
     };


/*
 * these variables MUST be defined in ADAPTER SPECIFIC DATA
 */
extern PRS    AllResources[];             /* list of all Resource Sets */
extern PVOID  AllHandles[];               /* list of all primary Resources */
extern USHORT roOwnersSize;               /* number of simultaneously "owned" */
                                          /* primary resources */
extern struct resource_owner FAR roOwners[];  /* static array for resource owner list */


/* =======================================================================
 *
 * define general purpose bounds macro as gain limiter
 *
 * =======================================================================
 *
 *  calling syntax is...
 *    value  = bound(lowerlimit,  value, upperlimit);
 */
#define bound(ll,v,ul)    ((ll<v)?((v<ul)? v : ul) : ll)
#define absval( v )       ( v<0 ? -v : v )


/*   define common tests */
#define CHECK_UPDATE_OCGAIN (usMessage==mUPDATEmsg &&                 \
       ( lParam1==MEIO_OWNER_GAIN || lParam1 == MEIO_CONNECTION_GAIN))

#define CHECK_QUERY_OCGAIN  (  usMessage==mQUERYmsg &&                \
       ( lParam1==MEIO_OWNER_GAIN || lParam1 == MEIO_CONNECTION_GAIN))

#define CHECK_UPDATE_UCGAIN (usMessage==mUPDATEmsg &&                 \
       ( lParam1==MEIO_USER_GAIN || lParam1 == MEIO_CONNECTION_GAIN))

#define CHECK_QUERY_UCGAIN  (  usMessage==mQUERYmsg &&                \
       ( lParam1==MEIO_USER_GAIN || lParam1 == MEIO_CONNECTION_GAIN))

#define CHECK_UPDATE_CGAIN (usMessage==mUPDATEmsg &&                 \
       ( lParam1 == MEIO_CONNECTION_GAIN))

#define CHECK_QUERY_CGAIN  (  usMessage==mQUERYmsg &&                \
       ( lParam1 == MEIO_CONNECTION_GAIN))

#define CHECK_UPDATE_UGAIN (usMessage==mUPDATEmsg &&                 \
       ( lParam1==MEIO_USER_GAIN))

#define CHECK_QUERY_UGAIN  (  usMessage==mQUERYmsg &&                \
       ( lParam1==MEIO_USER_GAIN))

#define CHECK_UPDATE_OGAIN (usMessage==mUPDATEmsg &&                 \
       ( lParam1==MEIO_OWNER_GAIN))

#define CHECK_QUERY_OGAIN  (  usMessage==mQUERYmsg &&                \
       ( lParam1==MEIO_OWNER_GAIN))

#define CHECK_UPDATE_CONNECTION (usMessage==mUPDATEmsg &&                 \
       ( lParam1==MEIO_CONNECTION_STATE))

#define CHECK_QUERY_CONNECTION  (  usMessage==mQUERYmsg &&                \
       ( lParam1==MEIO_CONNECTION_STATE))
