/*
 *   dspcalle.c -- Front-ends to Mwave device driver calls
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include "dspmgr.h"

#include "dspmgri.h"                   // Equates
#include "mwavepub.h"                  // DD external definitions.
#include "dspbios.h"                   // Function prototypes
#include <linux/major.h>
#include <sys/ioctl.h>

#define  DSP_DATA_SIZE   0x10000	   // Size of data memory
#define  DSP_DATA_WORD_SIZE   2

#define DSP_INST_SIZE    0x8000 	   // Size of instuction memory

extern HANDLE hDrv;
extern HANDLE g_hIPCMutex;                 // Igor - 20/09/95

/******************************************************************************/
/*                                                                            */
/*  DspBIOS_RegisterIPCEvent                                                  */
/*                                                                            */
/*  Purpose:    Register an IPC event with the Mwave kernel driver.           */
/*                                                                            */
/*  Format:     x( hDriver, usIPCNum, dwDrvIPCHandle )                        */
/*                                                                            */
/*              hDriver:        Handle of kernel driver                       */
/*              usIPCNum:       IPC table index                               */
/*              dwDrvIPCHandle: Event handle suitable for kernel driver       */
/*                                                                            */
/*  Return:     RC                                                            */
/*                                                                            */
/*  Remarks:    This functions registers the IPC event handle with the Mwave  */
/*              kernel driver. If successful, it returns DSP_NOERROR,         */
/*              otherwise it returns DSP_INTERNAL_ERROR.                      */
/*                                                                            */
/******************************************************************************/
RC PASCAL DspBIOS_RegisterIPCEvent( HANDLE hDriver, USHORT usIPCNum)
{
    DWORD       ddrc;

    MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_RegisterIPCEvent entry hDriver %x usIPCNum %x \n",
	   hDriver,usIPCNum);

    ddrc = ioctl(hDriver,
		 IOCTL_MW_REGISTER_IPC,
		 usIPCNum);

    return (ddrc != 0) ? DSP_INTERNAL_ERROR : DSP_NOERROR;
}


/******************************************************************************/
/*                                                                            */
/*  DspBIOS_GetIPCEvent                                                       */
/*                                                                            */
/*  Purpose:    suspend thread in device driver waiting for IPC event         */
/*                                                                            */
/*  Format:     x( hDriver, usIPCNum, dwDrvIPCHandle )                        */
/*                                                                            */
/*              hDriver:        Handle of kernel driver                       */
/*              usIPCNum:       IPC table index                               */
/*              dwDrvIPCHandle: Event handle suitable for kernel driver       */
/*                                                                            */
/*  Return:     RC                                                            */
/*                                                                            */
/*  Remarks:    This functions registers the IPC event handle with the Mwave  */
/*              kernel driver. If successful, it returns DSP_NOERROR,         */
/*              otherwise it returns DSP_INTERNAL_ERROR.                      */
/*                                                                            */
/******************************************************************************/
RC PASCAL DspBIOS_GetIPCEvent( HANDLE hDriver, USHORT usIPCNum)
{
    DWORD       ddrc;

    MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_GetIPCEvent entry hDriver %x usIPCNum %x \n",
	   hDriver,usIPCNum);

    ddrc = ioctl(hDriver,
		 IOCTL_MW_GET_IPC,
		 usIPCNum);

    /* @PORT @TBD. A flag should be checked here to see if this call was cancelled due to an
       IOCTL_MW_UNREGISTER_IPC */

    return (ddrc != 0) ? DSP_INTERNAL_ERROR : DSP_NOERROR;
}



/******************************************************************************/
/*                                                                            */
/*  DspBIOS_UnregisterIPCEvent                                                */
/*                                                                            */
/*  Purpose:    Unregisters an IPC event with the Mwave kernel driver.        */
/*                                                                            */
/*  Format:     x( hDriver, usIPCNum, dwDrvIPCHandle )                        */
/*                                                                            */
/*              hDriver:        Handle of kernel driver                       */
/*              usIPCNum:       IPC table index                               */
/*              dwDrvIPCHandle: Event handle suitable for kernel driver       */
/*                                                                            */
/*  Return:     RC                                                            */
/*                                                                            */
/*  Remarks:    This functions unregisters the IPC event handle with the      */
/*              Mwave kernel driver. If successful, it returns DSP_NOERROR,   */
/*              otherwise it returns DSP_INTERNAL_ERROR.                      */
/*                                                                            */
/******************************************************************************/
RC PASCAL DspBIOS_UnregisterIPCEvent( HANDLE hDriver, USHORT usIPCNum)
{
    DWORD       ddrc;

    MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_UnregisterIPCEvent entry hDriver %x usIPCNum %x \n",
	   hDriver,usIPCNum);

    /* @PORT @TBD. A flag should be set here if a Get IPC is pending. That way when the suspended thread
       returns it will be able to determine that the cause was due to an unregister */

    ddrc = ioctl(hDriver,
		 IOCTL_MW_UNREGISTER_IPC,
		 usIPCNum);

    return (ddrc != 0) ? DSP_INTERNAL_ERROR : DSP_NOERROR;
}



/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: DspBIOS_D_Read                                          */
/*                                                                          */
/* FUNCTION: This function reads DSP Data memory.                           */
/*                                                                          */
/* INPUT:                                                                   */
/*       PRDSP prdsp                                                        */
/*       ULONG ulDspAddress                                                 */
/*       ULONG ulSize                                                       */
/*       PVOID pBuf                                                         */
/*                                                                          */
/* OUTPUT: error code                                                       */
/*                                                                          */
/* SIDE EFFECTS: None                                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/
RC PASCAL DspBIOS_D_Read(PRDSP prdsp,ULONG ulDspAddress,ULONG ulSize,PVOID pBuf)
{
  DWORD 	   ddrc=OK;
  MW_READWRITE   *pData=0;
  
  MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_D_Read entry prdsp %x ulDspAddress %lx ulSize %lx\n",
	 (int)prdsp,ulDspAddress,ulSize);
  
  /* check for dumb errors like zero read size                             */
  if (ulSize == 0)
    return (DSP_READ_FAULT);
  
  // Assume High two bytes are Application junk...
  ulDspAddress &= 0x0000FFFF;
  
  if (ulDspAddress >= DSP_DATA_SIZE)
    return DSP_INV_ADDRESS;
  
  if (ulDspAddress + DSP_DATA_WORD_SIZE*ulSize > DSP_DATA_SIZE)
    return DSP_INV_XFER_SIZE;
  
  pData=malloc(sizeof(MW_READWRITE));
  if (pData) {
    pData->usDspAddress=(USHORT)ulDspAddress;
    pData->ulDataLength=ulSize; /* number of words to read in */
    pData->pBuf=pBuf;
    
    ddrc = ioctl(hDrv,
		 IOCTL_MW_READ_DATA,
		 pData);
    
    if (ddrc) {
      MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_D_Read error rc %lx from ioctl\n",ddrc);
      ddrc = DSP_INTERNAL_ERROR;
    } else {
      /* memcpy(pBuf,pData->Data,pData->ulDataLength*sizeof(WORD)); */
    }
    free(pData);
  } else {
    ddrc=DSP_INTERNAL_ERROR;
  }
    
  MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_D_Read exit ddrc %lx\n",ddrc);

  return ddrc;
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: DspBIOS_D_Write                                         */
/*                                                                          */
/* FUNCTION: This function writes DSP Data memory.                          */
/*                                                                          */
/* INPUT:                                                                   */
/*       PRDSP prdsp                                                        */
/*       ULONG ulDspAddress                                                 */
/*       ULONG ulSize                                                       */
/*       PVOID pBuf                                                         */
/*                                                                          */
/* OUTPUT: error code                                                       */
/*                                                                          */
/* SIDE EFFECTS: None                                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC PASCAL DspBIOS_D_Write(PRDSP prdsp,ULONG ulDspAddress,ULONG ulSize,PVOID pBuf)

{
  DWORD 	   ddrc=OK;
  MW_READWRITE   *pData=0;
  
  MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_D_Write entry prdsp %x ulDspAddress %lx ulSize %lx\n",
	 (int)prdsp,ulDspAddress,ulSize);
  
  if (ulSize == 0)
    return (DSP_READ_FAULT);
  
  ulDspAddress &= 0x0000FFFF;
  
  if (ulDspAddress >= DSP_DATA_SIZE)
    return DSP_INV_ADDRESS;
  
  if (ulDspAddress + DSP_DATA_WORD_SIZE*ulSize > DSP_DATA_SIZE)
    return DSP_INV_XFER_SIZE;
  
  pData = malloc(sizeof(MW_READWRITE));
  
  if (pData) {
    pData->usDspAddress=(USHORT)ulDspAddress;
    pData->ulDataLength=ulSize; /* number of words to write */
    pData->pBuf=pBuf;
    /* memcpy(pData->Data,pBuf,ulSize*sizeof(WORD)); */
    
    ddrc = ioctl(hDrv,
		 IOCTL_MW_WRITE_DATA,
		 pData);
    if (ddrc) {
      MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_D_Write error rc %lx from ioctl\n",ddrc);
      ddrc = DSP_INTERNAL_ERROR;
    }
    
    free(pData);
    
  } else {
    ddrc=DSP_INTERNAL_ERROR;
  }
  
  return ddrc;
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: DspBIOS_I_Read                                          */
/*                                                                          */
/* FUNCTION: This function reads DSP Insrtruction memory.                   */
/*                                                                          */
/* INPUT:                                                                   */
/*       PRDSP prdsp                                                        */
/*       ULONG ulDspAddress                                                 */
/*       ULONG ulSize                                                       */
/*       PVOID pBuf                                                         */
/*                                                                          */
/* OUTPUT: error code                                                       */
/*                                                                          */
/* SIDE EFFECTS: None                                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC PASCAL DspBIOS_I_Read(PRDSP prdsp,ULONG ulDspAddress,ULONG ulSize,PVOID pBuf)
{
  DWORD 	   ddrc=OK;
  MW_READWRITE    *pData=0;

  MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_I_Read entry prdsp %x ulDspAddress %lx ulSize %lx\n",
	 (int)prdsp,ulDspAddress,ulSize);

  if (ulSize == 0)
    return (DSP_READ_FAULT);

  ulDspAddress &= 0x0000FFFF;
  
  if (ulDspAddress >= DSP_INST_SIZE)
    return DSP_INV_ADDRESS;

  pData=malloc(sizeof(MW_READWRITE));
  if (pData) {
    pData->usDspAddress=(USHORT)ulDspAddress;
    pData->ulDataLength=ulSize; /* number of words to read */
    pData->pBuf=pBuf;
    
   ddrc=ioctl(hDrv,	             
	      IOCTL_MW_READ_INST,    
	      pData);

   if (ddrc) {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_I_Read error rc %lx from ioctl\n",ddrc);
     ddrc = DSP_INTERNAL_ERROR;
    } else {
      /* memcpy(pBuf,pData->Data,pData->ulDataLength*sizeof(WORD)); */
    }
    free(pData);
  } else {
    ddrc=DSP_INTERNAL_ERROR;
  }
   
  return ddrc;
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: DspBIOS_I_Write                                         */
/*                                                                          */
/* FUNCTION: This function writes DSP instruction memory.                   */
/*                                                                          */
/* INPUT:                                                                   */
/*       PRDSP prdsp                                                        */
/*       ULONG ulDspAddress                                                 */
/*       ULONG ulSize                                                       */
/*       PVOID pBuf                                                         */
/*                                                                          */
/* OUTPUT: error code                                                       */
/*                                                                          */
/* SIDE EFFECTS: None                                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC PASCAL DspBIOS_I_Write(PRDSP prdsp,ULONG ulDspAddress,ULONG ulSize,PVOID pBuf)

{
  DWORD 	   ddrc;
  MW_READWRITE   *pData=0;
  
  MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_I_Write entry prdsp %x ulDspAddress %lx ulSize %lx\n",
	 (int)prdsp,ulDspAddress,ulSize);
  
  if (ulSize == 0)
    return (DSP_READ_FAULT);
  
  ulDspAddress &= 0x0000FFFF;
  
  if (ulDspAddress >= DSP_INST_SIZE)
    return DSP_INV_ADDRESS;
  
  pData = malloc(sizeof(MW_READWRITE));
  
  if (pData) {
    pData->usDspAddress=(USHORT)ulDspAddress;
    pData->ulDataLength=ulSize; /* number of words to write */
    pData->pBuf=pBuf;
    /* memcpy(pData->Data,pBuf,ulSize*sizeof(WORD)); */
    
    ddrc = ioctl(hDrv,
		 IOCTL_MW_WRITE_INST,
		 pData);
    if (ddrc) {
      MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_I_Write error rc %lx from ioctl\n",ddrc);
      ddrc = DSP_INTERNAL_ERROR;
    }
    
    free(pData);
    
  } else {
    ddrc=DSP_INTERNAL_ERROR;
  }
  MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_I_Write exit ddrc %lx\n",ddrc);

  return ddrc;
}


RC DspBIOS_Abilities(PRDSP prdsp,PUSHORT pusSize,PVOID InfoBuf)
{

#ifndef NODSP
   MW_ABILITIES	        mwAbilities;
   DWORD 	        ddrc;
   RDSP_ABILITIES_INFO *mwDspAbilities = (RDSP_ABILITIES_INFO *)InfoBuf;
   int			i;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_Abilities entry prdsp %x\n",(int)prdsp);

   if ( *pusSize < sizeof(RDSP_ABILITIES_INFO) ) {
     *pusSize = sizeof(RDSP_ABILITIES_INFO);
     MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_ABilities error DSP_INS_BUFFER\n");
     return DSP_INS_BUFFER;
   }
   
   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_Abilities calling IOCTL\n");
   ddrc = ioctl(hDrv,
		IOCTL_MW_DSP_ABILITIES,
		&mwAbilities);

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_Abilites return from ioctl ddrc %lx\n",ddrc);
   
   if (ddrc)
     return DSP_INTERNAL_ERROR;

   mwDspAbilities->DSPABILITIES_usResv1            = 0;
   mwDspAbilities->DSPABILITIES_usResv2            = 0;
   mwDspAbilities->DSPABILITIES_usResv3            = 0;
   mwDspAbilities->DSPABILITIES_ulResv4            = 0L;
   mwDspAbilities->DSPABILITIES_ulIPS              = mwAbilities.instr_per_sec;
   mwDspAbilities->DSPABILITIES_ulDataStore        = mwAbilities.data_size;
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_Abilities saving mwAbilities.inst_size %lx\n",mwAbilities.inst_size);
   mwDspAbilities->DSPABILITIES_ulInstStore        = mwAbilities.inst_size;
   mwDspAbilities->DSPABILITIES_ulSys_Bus_Capacity = mwAbilities.bus_dma_bw;
   mwDspAbilities->DSPABILITIES_usSlot_Number      = 0;
   mwDspAbilities->DSPABILITIES_usAdapter_ID       = 0;
   mwDspAbilities->DSPABILITIES_usCOM              = 1; /* COM1 */
   mwDspAbilities->DSPABILITIES_usResv5            = mwAbilities.component_count;
   
   for (i=0; i<7; i++)
     mwDspAbilities->DSPABILITIES_ulResv6[i] = mwAbilities.component_list[i];
   
   // should be fixed in the driver XXX
   mwDspAbilities->DSPABILITIES_ulResv6[7] = 0;
   
   //memcpy(InfoBuf, &mwDspAbilities, sizeof(RDSP_ABILITIES_INFO));
   
   return DSP_NOERROR;
   
#else
  return (DSP_NOERROR);
#endif
}



/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: DspBIOS_Reset                                           */
/*                                                                          */
/* FUNCTION: This function Resets the DSP.                                  */
/*                                                                          */
/* INPUT:                                                                   */
/*       PRDSP prdsp                                                        */
/*                                                                          */
/* OUTPUT: error code                                                       */
/*                                                                          */
/* SIDE EFFECTS: None                                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */

/************************** END OF SPECIFICATIONS ***************************/

RC PASCAL DspBIOS_Reset(PRDSP prdsp)
{
  DWORD 	        ddrc;
  
  MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_Reset entry prdsp %x\n",(int)prdsp);

  ddrc=ioctl(hDrv,
	     IOCTL_MW_RESET,   
	     0);

  MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_Reset exit ddrc %lx\n",ddrc);
  if (!ddrc)
    return DSP_NOERROR;
  else
    return DSP_INTERNAL_ERROR;

}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: DspBIOS_Run                                             */
/*                                                                          */
/* FUNCTION: This function "unhalts" the DSP.                               */
/*                                                                          */
/* INPUT:                                                                   */
/*       PRDSP prdsp                                                        */
/*                                                                          */
/* OUTPUT: error code                                                       */
/*                                                                          */
/* SIDE EFFECTS: None                                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC PASCAL DspBIOS_Run(PRDSP prdsp)
{
   DWORD 	        ddrc;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspcalle::DspBIOS_Run entry prdsp %x\n",(int)prdsp);

   ddrc = ioctl(hDrv,
                IOCTL_MW_RUN, 
		0);

   if (!ddrc)
     return DSP_NOERROR;
   else
     return DSP_INTERNAL_ERROR;

}


