/*
 * distance.c --- distance package for Natch
 *
 * Copyright (c) 1997-2002, 2004 by Pascal Wassong All Rights Reserved.
 *
 * Time-stamp: <2004-07-01 13:07:43 pascal>
 *
 * This file is part of Natch.
 *
 * Natch is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Natch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include	"common.h"

#include	"distance.h"

#include	"pcpj.h"
#include	"twin_squares.h"

#include	<string.h>	/* For memset & memcpy	*/
#include	<assert.h>

/*--------------------------------------------------------------------------*/

#define	Nombre_Maximum_Tableaux_Dans_Pile	34
static	distance_t*	Pile_Tableaux[Nombre_Maximum_Tableaux_Dans_Pile];
static	unsigned int	Nombre_Elements_Pile_Tableaux = 0;

/* Not static because used in the MACRO */
distance_t*		Tableau_Distance_Actuel;
static	distance_t*	Tableau_Distance_Pion_Actuel;
static	distance_t*	Tableau_Distance_Precedent;

/*--------------------------------------------------------------------------*/

typedef		void (*create_piece_table_for_square_t)(
    distance_t*	table,
    square_t	square );

static void create_king_tables( distance_t* boards );
static void create_king_table_for_square(
    distance_t* table, square_t square );

static void create_queen_tables( distance_t* boards );
static void create_queen_table_for_square(
    distance_t* table, square_t square );

static void create_rook_tables( distance_t* boards );
static void create_rook_table_for_square(
    distance_t* table, square_t square );

static void create_bishop_tables( distance_t* boards );
static void create_bishop_table_for_square(
    distance_t* table, square_t square );

static void create_knight_tables( distance_t* boards );
static void create_knight_table_for_square(
    distance_t* table, square_t square );

/* #define DEBUG_PRINT_TABLE */
#ifdef DEBUG_PRINT_TABLE
static void print_table( distance_t* distance_table );
void print_current_distance_table();
#endif

/*--------------------------------------------------------------------------*/
/*MACRO distance_t
distance(piece_type_t piece, colour_t camp, square_t caseDep, square_t caseArr)
{
    return (Tableau_Distance_Actuel[type_piece * SIZE_PIECE +
    (camp == BLANC ? 0 : SIZE_PIECE / 2) +
    ((row(caseDep) >> 1) +
    column(caseDep)) * SIZE_ECHIQUIER +
    caseArr]);
} */

distance_t
distancePion(colour_t camp, square_t caseDep, square_t caseArr)
{
    return (Tableau_Distance_Pion_Actuel[(camp == BLANC ? 0 : SIZE_PIECE / 2) +
					((row(caseDep) >> 1) +
					 column(caseDep)) * SIZE_ECHIQUIER +
					caseArr]);
}

/*--------------------------------------------------------------------------*/

static void
create_king_tables( distance_t* boards )
{
    square_t	square ;

    memset( boards, INFINI, SIZE_PIECE );

    for ( square = a1 ; square <= h8 ; square++ )
    {
	if ( ! CasesInterdites[ square ] )
	{
	    distance_t*	table = boards +
		( (row( square ) >> 1) + column( square ) ) * SIZE_ECHIQUIER ;
	    create_king_table_for_square( table, square );
	}
    }
    memcpy( boards + SIZE_PIECE / 2, boards, SIZE_PIECE / 2 );

}

static void
create_king_table_for_square( distance_t* table, square_t square )
{
    square_t	todo[ 64 ], twin_square, current_square, new_square ;
    distance_t	current_distance ;
    int		max, current ;

    todo[ 0 ] 	    = square ;
    current 	    = 0	;
    max 	    = 1	;
    table[ square ] = 0	;

    while ( current < max )
    {
	current_square   = todo[ current++ ];
	current_distance = table[ current_square ] + 1 ;
	twin_square = find_twin_square( current_square, TRUE, ROI, BLANC );

	new_square = current_square - 1 ;
	if ((!CasesInterdites[ new_square ]) && table[ new_square ] == INFINI &&
	    new_square != twin_square)
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square + 1 ;
	if ((!CasesInterdites[ new_square ]) && table[ new_square ] == INFINI &&
	    new_square != twin_square)
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square - 16 ;
	if ((!CasesInterdites[ new_square ]) && table[ new_square ] == INFINI &&
	    new_square != twin_square)
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square + 16 ;
	if ((!CasesInterdites[ new_square ]) && table[ new_square ] == INFINI &&
	    new_square != twin_square)
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}

	twin_square = find_twin_square( current_square, FALSE, ROI, BLANC );
	new_square = current_square - 17 ;
	if ((!CasesInterdites[ new_square ]) && table[ new_square ] == INFINI &&
	    new_square != twin_square)
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square - 15 ;
	if ((!CasesInterdites[ new_square ]) && table[ new_square ] == INFINI &&
	    new_square != twin_square)
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square + 17 ;
	if ((!CasesInterdites[ new_square ]) && table[ new_square ] == INFINI &&
	    new_square != twin_square)
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square + 15 ;
	if ((!CasesInterdites[ new_square ]) && table[ new_square ] == INFINI &&
	    new_square != twin_square)
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
    }
}

/*--------------------------------------------------------------------------*/

static void
create_pawn_table_for_one_colour(distance_t* boards, colour_t colour)
{
    int			i;
    square_t		origin_square;
    distance_t*		board;
    square_delta_t	delta_Y;
    row_t		origin_row;
    row_t		promotion_row;
    square_t		todo[64];
    int			max, current;
    distance_t		current_distance;
    square_t		square, new_square;

    if (colour == BLANC)
    {
	delta_Y       = 0x10;
	origin_row    = 0x10;
	promotion_row = 0x70;
    }
    else
    {
	delta_Y       = 0xF0;
	origin_row    = 0x60;
	promotion_row = 0x00;
    }

    /* It costs nothing for a pawn to stay on a promotion square */
    for (i=0; i<8; i++)
    {
	board = boards + ((promotion_row >> 1) + i) * SIZE_ECHIQUIER;
	board[promotion_row + i] = 0;
    }

    for (origin_square=a2; origin_square<=h7; origin_square++)
    {
	board = boards + ( (row(origin_square) >> 1) + column(origin_square) )
	    * SIZE_ECHIQUIER;
	board[origin_square] = 0;

	todo[0] = origin_square;
	current = 0;
	max = 1;

	while (current < max)
	{
	    square = todo[current++];
	    current_distance = board[square] + 1;

	    new_square = square + delta_Y;
	    if ((! CasesInterdites[new_square]) && board[new_square] == INFINI)
	    {
		board[new_square] = current_distance;
		todo[max++] = new_square;

		if (row(square) == origin_row)
		{
		    new_square += delta_Y;
		    if ( ( !CasesInterdites[ new_square ] )
			 && board[ new_square ] == INFINI )
		    {
			board[new_square] = current_distance;
			todo[max++] = new_square;
		    }
		}
	    }
	    new_square = square + delta_Y + 1;
	    if ((! CasesInterdites[new_square]) && board[new_square] == INFINI)
	    {
		board[new_square] = current_distance;
		todo[max++] = new_square;
	    }
	    new_square = square + delta_Y - 1;
	    if ((! CasesInterdites[new_square]) && board[new_square] == INFINI)
	    {
		board[new_square] = current_distance;
		todo[max++] = new_square;
	    }
	}
    }
}

static void
distCreeTabPion(distance_t* pawn_distance_table)
{
    memset(pawn_distance_table, INFINI, SIZE_PIECE);
    create_pawn_table_for_one_colour(pawn_distance_table, BLANC);
    create_pawn_table_for_one_colour(pawn_distance_table + SIZE_PIECE/2, NOIR);
}

/*--------------------------------------------------------------------------*/

static void
create_queen_tables( distance_t* boards )
{
    square_t 	square ;

    memset( boards, INFINI, SIZE_PIECE );

    for ( square = a1 ; square <= h8 ; square++ )
    {
	if ( ! CasesInterdites[ square ] )
	{
	    distance_t*	table = boards +
		( (row( square ) >> 1) + column( square ) ) * SIZE_ECHIQUIER ;
	    create_queen_table_for_square( table, square );
	}
    }
    memcpy( boards + SIZE_PIECE / 2, boards, SIZE_PIECE / 2 );

}

static void
create_queen_table_for_square( distance_t* table, square_t square )
{
    distance_t	current_distance ;
    square_t 	current_square, new_square ;
    square_t	current_twin ;
    square_t	todo[ 64 ];
    int		max, current ;
    square_t	twin_square, twin_square2 ;

    todo[ 0 ] 	    = square ;
    current 	    = 0	;
    max 	    = 1	;
    table[ square ] = 0	;

    while ( current < max )
    {
	current_square 	 = todo[ current++ ];
	current_distance = table[ current_square ] + 1 ;
	current_twin = find_twin_square( current_square, TRUE, DAME, BLANC );

	new_square = current_square + 1 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance;
		todo[ max++ ] 	    = new_square;
	    }
	    twin_square2 = find_twin_square( new_square, TRUE, DAME, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square++ ;
	}
	new_square = current_square - 1 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, TRUE, DAME, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square-- ;
	}
	new_square = current_square + 16 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, TRUE, DAME, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square += 16 ;
	}
	new_square = current_square - 16 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance;
		todo[ max++ ] = new_square;
	    }
	    twin_square2 = find_twin_square( new_square, TRUE, DAME, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square -= 16;
	}
	new_square = current_square + 15;
	current_twin = find_twin_square( current_square, FALSE, DAME, BLANC );
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, FALSE, DAME, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square += 15 ;
	}
	new_square = current_square - 15 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, FALSE, DAME, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square -= 15 ;
	}
	new_square = current_square + 17 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, FALSE, DAME, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square += 17 ;
	}
	new_square = current_square - 17 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, FALSE, DAME, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square -= 17 ;
	}
    }
}

/*--------------------------------------------------------------------------*/

static void
create_rook_tables( distance_t* boards )
{
    square_t 	square ;

    memset( boards, INFINI, SIZE_PIECE );

    for ( square = a1 ; square <= h8 ; square++ )
    {
	if ( ! CasesInterdites[ square ] )
	{
	    distance_t*	table = boards +
		( (row( square ) >> 1) + column( square ) ) * SIZE_ECHIQUIER ;
	    create_rook_table_for_square( table, square );
	}
    }
    memcpy( boards + SIZE_PIECE / 2, boards, SIZE_PIECE / 2 );

}

static void
create_rook_table_for_square( distance_t* table, square_t square )
{
    distance_t	current_distance;
    square_t 	current_square, new_square ;
    square_t	current_twin ;
    square_t	todo[ 64 ];
    int		max, current ;
    square_t	twin_square, twin_square2 ;

    todo[ 0 ] 	    = square ;
    current 	    = 0	;
    max 	    = 1	;
    table[ square ] = 0	;

    while ( current < max )
    {
	current_square = todo[ current++ ];
	current_distance = table[ current_square ] + 1 ;
	current_twin = find_twin_square( current_square, TRUE, TOUR, BLANC );

	new_square  = current_square + 1 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, TRUE, TOUR, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square++;
	}
	new_square = current_square - 1 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, TRUE, TOUR, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square--;
	}
	new_square = current_square + 16 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance;
		todo[ max++ ] 	    = new_square;
	    }
	    twin_square2 = find_twin_square( new_square, TRUE, TOUR, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square += 16 ;
	}
	new_square = current_square - 16 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, TRUE, TOUR, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square -= 16 ;
	}
    }
}

/*--------------------------------------------------------------------------*/

static void
create_bishop_tables( distance_t* boards )
{
    square_t 	square ;

    memset( boards, INFINI, SIZE_PIECE );

    for ( square = a1 ; square <= h8 ; square++ )
    {
	if ( ! CasesInterdites[ square ] )
	{
	    distance_t*	table = boards +
		( (row( square ) >> 1) + column( square ) ) * SIZE_ECHIQUIER ;
	    create_bishop_table_for_square( table, square );
	}
    }
    memcpy( boards + SIZE_PIECE / 2, boards, SIZE_PIECE / 2 );

}

static void
create_bishop_table_for_square( distance_t* table, square_t square )
{
    distance_t	current_distance;
    square_t 	current_square, new_square;
    square_t	current_twin ;
    square_t	todo[ 64 ];
    int		max, current;
    square_t	twin_square, twin_square2;

    todo[ 0 ] 	    = square ;
    current 	    = 0	;
    max 	    = 1	;
    table[ square ] = 0	;

    while ( current < max )
    {
	current_square   = todo[ current++ ];
	current_distance = table[ current_square ] + 1 ;
	current_twin = find_twin_square( current_square, FALSE, FOU, BLANC );

	new_square = current_square + 15 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, FALSE, FOU, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square += 15 ;
	}
	new_square = current_square - 15 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, FALSE, FOU, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square -= 15 ;
	}
	new_square = current_square + 17 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, FALSE, FOU, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square += 17 ;
	}
	new_square = current_square - 17 ;
	twin_square = current_twin ;
	while ( !CasesInterdites[ new_square ]
		&& new_square != twin_square
		&& new_square != current_twin )
	{
	    if ( table[ new_square ] == INFINI )
	    {
		table[ new_square ] = current_distance ;
		todo[ max++ ] 	    = new_square ;
	    }
	    twin_square2 = find_twin_square( new_square, FALSE, FOU, BLANC );
	    if ( twin_square2 != No_Twin_Square )
	    {
		twin_square = twin_square2 ;
	    }
	    new_square -= 17 ;
	}
    }
}

/*--------------------------------------------------------------------------*/

static void
create_knight_tables( distance_t* boards )
{
    square_t 	square ;

    memset( boards, INFINI, SIZE_PIECE );

    for ( square = a1 ; square <= h8 ; square++ )
    {
	if ( ! CasesInterdites[ square ] )
	{
	    distance_t*	table = boards +
		( (row( square ) >> 1) + column( square ) ) * SIZE_ECHIQUIER ;
	    create_knight_table_for_square( table, square );
	}
    }
    memcpy( boards + SIZE_PIECE / 2, boards, SIZE_PIECE / 2 );

}

static void
create_knight_table_for_square( distance_t* table, square_t square )
{
    distance_t	current_distance ;
    square_t 	current_square, new_square ;
    square_t	todo[ 64 ];
    int		max, current ;

    todo[ 0 ] 	    = square ;
    current 	    = 0	;
    max 	    = 1	;
    table[ square ] = 0	;

    while ( current < max )
    {
	current_square = todo[ current++ ];
	current_distance = table[ current_square ] + 1 ;
	new_square = current_square - 33 ;
	if ( ! CasesInterdites[ new_square ] && table[ new_square ] == INFINI )
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square - 31 ;
	if ( ! CasesInterdites[ new_square ] && table[ new_square ] == INFINI )
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square - 18 ;
	if ( ! CasesInterdites[ new_square ] && table[ new_square ] == INFINI )
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square - 14 ;
	if ( ! CasesInterdites[ new_square ] && table[ new_square ] == INFINI )
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square + 33 ;
	if ( ! CasesInterdites[ new_square ] && table[ new_square ] == INFINI )
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square + 31 ;
	if ( ! CasesInterdites[ new_square ] && table[ new_square ] == INFINI )
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square + 18 ;
	if ( ! CasesInterdites[ new_square ] && table[ new_square ] == INFINI )
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
	new_square = current_square + 14 ;
	if ( ! CasesInterdites[ new_square ] && table[ new_square ] == INFINI )
	{
	    table[ new_square ] = current_distance ;
	    todo[ max++ ] 	= new_square ;
	}
    }
}

/*--------------------------------------------------------------------------*/

void
distCreeTabPionRoiCavalier(distance_t* distance_table)
{
    distCreeTabPion(&distance_table[PION * SIZE_PIECE]);

    /* We skip the kings : they don't collide with themselves, neither do they
     * with knights.
     */
    create_twin_squares( &ListeSures[ 2 ], NbSures - 2 );

    create_king_tables  ( &distance_table[ ROI      * SIZE_PIECE ] );
    create_knight_tables( &distance_table[ CAVALIER * SIZE_PIECE ] );
}

void
distCreeTabPiece(distance_t* distance_table)
{
    create_twin_squares( ListeSures, NbSures + NbRestantes );

    create_queen_tables ( &distance_table[ DAME * SIZE_PIECE ] );
    create_rook_tables  ( &distance_table[ TOUR * SIZE_PIECE ] );
    create_bishop_tables( &distance_table[ FOU  * SIZE_PIECE ] );
}

void
distance_create_table( distance_t* distance_table )
{
    create_twin_squares( ListeSures, NbSures + NbRestantes );

    distCreeTabPion     ( &distance_table[ PION     * SIZE_PIECE ]);
    create_king_tables  ( &distance_table[ ROI      * SIZE_PIECE ]);
    create_knight_tables( &distance_table[ CAVALIER * SIZE_PIECE ]);
    create_queen_tables ( &distance_table[ DAME     * SIZE_PIECE ]);
    create_rook_tables  ( &distance_table[ TOUR     * SIZE_PIECE ]);
    create_bishop_tables( &distance_table[ FOU      * SIZE_PIECE ]);
}

static void
create_table_for_square(
    distance_t*				table,
    square_t*				squares,
    unsigned int			nb_squares,
    create_piece_table_for_square_t	function )
{
    unsigned int	i ;

    memset( table, INFINI, SIZE_PIECE );

    for ( i = 0 ; i < nb_squares ; i++ )
    {
	square_t	square = squares[ i ];
	distance_t*	one_table  = table +
	    ( (row( square ) >> 1) + column( square ) ) * SIZE_ECHIQUIER ;
	(*function)( one_table, square );
    }

    memcpy( table + SIZE_PIECE / 2, table, SIZE_PIECE / 2 );

}

void
distance_create_table_for_squares(
    distance_t*		distance_table,
    square_t*		king_squares,
    unsigned int	nb_king_squares,
    square_t*		pawn_squares,
    unsigned int	nb_pawn_squares,
    square_t*		queen_squares,
    unsigned int	nb_queen_squares,
    square_t*		rook_squares,
    unsigned int	nb_rook_squares,
    square_t*		bishop_squares,
    unsigned int	nb_bishop_squares,
    square_t*		knight_squares,
    unsigned int	nb_knight_squares )
{
    create_table_for_square(
	&distance_table[ ROI * SIZE_PIECE ],
	king_squares,
	nb_king_squares,
	create_king_table_for_square );

    distCreeTabPion( &distance_table[ PION * SIZE_PIECE ] );

    create_table_for_square(
	&distance_table[ DAME * SIZE_PIECE ],
	queen_squares,
	nb_queen_squares,
	create_queen_table_for_square );
    create_table_for_square(
	&distance_table[ TOUR * SIZE_PIECE ],
	rook_squares,
	nb_rook_squares,
	create_rook_table_for_square );
    create_table_for_square(
	&distance_table[ FOU * SIZE_PIECE ],
	bishop_squares,
	nb_bishop_squares,
	create_bishop_table_for_square );
    create_table_for_square(
	&distance_table[ CAVALIER * SIZE_PIECE ],
	knight_squares,
	nb_knight_squares,
	create_knight_table_for_square );
}

/*--------------------------------------------------------------------------*/

#ifdef DEBUG_PRINT_TABLE
static void
print_table( distance_t* distance_table )
{
    distance_t*		ptr ;
    int			i, j ;
    square_t		caseDep    = a1;
    piece_type_t	piece_type = TOUR;
    colour_t		camp 	   = BLANC;
    writeAssociation( "Printing table :" );
    ptr = &distance_table[ piece_type * SIZE_PIECE +
			 ( camp == BLANC ? 0 : SIZE_PIECE / 2 ) +
			 ( ( row( caseDep ) >> 1 ) +
			   column( caseDep ) ) * SIZE_ECHIQUIER ];
    fprintf( MainFD, "CasesInterdites : \n" );
    for( i = 7; i >= 0; i-- )
    {
	for( j = 0; j < 8; j++ )
	{
	    fprintf( MainFD, "%3.3d ", CasesInterdites[ i * 16 + j ]);
	}
	fprintf( MainFD, "\n" );
    }
    fprintf( MainFD, "distance : \n" );
    for( i = 7; i >= 0; i-- )
    {
	for( j = 0; j < 8; j++ )
	{
	    fprintf( MainFD, "%3.3d ", ptr[ i * 16 + j ]);
	}
	fprintf( MainFD, "\n" );
    }
}

void
print_current_distance_table()
{
    print_table( Tableau_Distance_Actuel );
}
#endif /* DEBUG_PRINT_TABLE */

/*--------------------------------------------------------------------------*/

void
distance_push( distance_t* distance_table )
{
    assert( Nombre_Elements_Pile_Tableaux < Nombre_Maximum_Tableaux_Dans_Pile );

    if (Nombre_Elements_Pile_Tableaux == 0)
    {
	Tableau_Distance_Precedent = distance_table;
    }
    else
    {
	Tableau_Distance_Precedent =
	    Pile_Tableaux[Nombre_Elements_Pile_Tableaux - 1];
    }

    Pile_Tableaux[Nombre_Elements_Pile_Tableaux] = distance_table;
    Tableau_Distance_Actuel = distance_table;
    Tableau_Distance_Pion_Actuel = &(distance_table[PION * SIZE_PIECE]);
    Nombre_Elements_Pile_Tableaux++;

}

void
distance_pop()
{
    assert( Nombre_Elements_Pile_Tableaux > 0 );

    Nombre_Elements_Pile_Tableaux--;
    Tableau_Distance_Actuel = Pile_Tableaux[Nombre_Elements_Pile_Tableaux - 1];
    Tableau_Distance_Pion_Actuel =
	&(Tableau_Distance_Actuel[ PION * SIZE_PIECE ]);
    if (Nombre_Elements_Pile_Tableaux > 1)
    {
	Tableau_Distance_Precedent =
	    Pile_Tableaux[Nombre_Elements_Pile_Tableaux - 2];
    }
}

distance_t
distance_delta(
    piece_type_t	type_piece,
    colour_t		camp,
    square_t		case_depart,
    square_t		case_arrivee )
{
    return (distance(type_piece, camp, case_depart, case_arrivee) -
	    Tableau_Distance_Precedent[type_piece * SIZE_PIECE +
				      (camp == BLANC ? 0 : SIZE_PIECE / 2) +
				      ((row(case_depart) >> 1) +
				       column(case_depart)) * SIZE_ECHIQUIER +
				      case_arrivee] );
}
