/* Copyright (c) 2006, Google Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/* minidump_exception_win32.h: Definitions of exception codes for
 * Win32 platform
 *
 * (This is C99 source, please don't corrupt it with C++.)
 *
 * Author: Mark Mentovai
 * Split into its own file: Neal Sidhwaney */


#ifndef GOOGLE_BREAKPAD_COMMON_MINIDUMP_EXCEPTION_WIN32_H__
#define GOOGLE_BREAKPAD_COMMON_MINIDUMP_EXCEPTION_WIN32_H__

#include <stddef.h>

#include "google_breakpad/common/breakpad_types.h"


/* For (MDException).exception_code.  These values come from WinBase.h */
typedef enum {
  MD_EXCEPTION_CODE_WIN_GUARD_PAGE_VIOLATION     = 0x80000001,
      /* EXCEPTION_GUARD_PAGE */
  MD_EXCEPTION_CODE_WIN_DATATYPE_MISALIGNMENT    = 0x80000002,
      /* EXCEPTION_DATATYPE_MISALIGNMENT */
  MD_EXCEPTION_CODE_WIN_BREAKPOINT               = 0x80000003,
      /* EXCEPTION_BREAKPOINT */
  MD_EXCEPTION_CODE_WIN_SINGLE_STEP              = 0x80000004,
      /* EXCEPTION_SINGLE_STEP */
  MD_EXCEPTION_CODE_WIN_ACCESS_VIOLATION         = 0xc0000005,
      /* EXCEPTION_ACCESS_VIOLATION */
  MD_EXCEPTION_CODE_WIN_IN_PAGE_ERROR            = 0xc0000006,
      /* EXCEPTION_IN_PAGE_ERROR */
  MD_EXCEPTION_CODE_WIN_INVALID_HANDLE           = 0xc0000008,
      /* EXCEPTION_INVALID_HANDLE */
  MD_EXCEPTION_CODE_WIN_ILLEGAL_INSTRUCTION      = 0xc000001d,
      /* EXCEPTION_ILLEGAL_INSTRUCTION */
  MD_EXCEPTION_CODE_WIN_NONCONTINUABLE_EXCEPTION = 0xc0000025,
      /* EXCEPTION_NONCONTINUABLE_EXCEPTION */
  MD_EXCEPTION_CODE_WIN_INVALID_DISPOSITION      = 0xc0000026,
      /* EXCEPTION_INVALID_DISPOSITION */
  MD_EXCEPTION_CODE_WIN_ARRAY_BOUNDS_EXCEEDED    = 0xc000008c,
      /* EXCEPTION_BOUNDS_EXCEEDED */
  MD_EXCEPTION_CODE_WIN_FLOAT_DENORMAL_OPERAND   = 0xc000008d,
      /* EXCEPTION_FLT_DENORMAL_OPERAND */
  MD_EXCEPTION_CODE_WIN_FLOAT_DIVIDE_BY_ZERO     = 0xc000008e,
      /* EXCEPTION_FLT_DIVIDE_BY_ZERO */
  MD_EXCEPTION_CODE_WIN_FLOAT_INEXACT_RESULT     = 0xc000008f,
      /* EXCEPTION_FLT_INEXACT_RESULT */
  MD_EXCEPTION_CODE_WIN_FLOAT_INVALID_OPERATION  = 0xc0000090,
      /* EXCEPTION_FLT_INVALID_OPERATION */
  MD_EXCEPTION_CODE_WIN_FLOAT_OVERFLOW           = 0xc0000091,
      /* EXCEPTION_FLT_OVERFLOW */
  MD_EXCEPTION_CODE_WIN_FLOAT_STACK_CHECK        = 0xc0000092,
      /* EXCEPTION_FLT_STACK_CHECK */
  MD_EXCEPTION_CODE_WIN_FLOAT_UNDERFLOW          = 0xc0000093,
      /* EXCEPTION_FLT_UNDERFLOW */
  MD_EXCEPTION_CODE_WIN_INTEGER_DIVIDE_BY_ZERO   = 0xc0000094,
      /* EXCEPTION_INT_DIVIDE_BY_ZERO */
  MD_EXCEPTION_CODE_WIN_INTEGER_OVERFLOW         = 0xc0000095,
      /* EXCEPTION_INT_OVERFLOW */
  MD_EXCEPTION_CODE_WIN_PRIVILEGED_INSTRUCTION   = 0xc0000096,
      /* EXCEPTION_PRIV_INSTRUCTION */
  MD_EXCEPTION_CODE_WIN_STACK_OVERFLOW           = 0xc00000fd,
      /* EXCEPTION_STACK_OVERFLOW */
  MD_EXCEPTION_CODE_WIN_BAD_FUNCTION_TABLE       = 0xc00000ff,
      /* EXCEPTION_BAD_FUNCTION_TABLE */
  MD_EXCEPTION_CODE_WIN_POSSIBLE_DEADLOCK        = 0xc0000194,
      /* EXCEPTION_POSSIBLE_DEADLOCK */
  MD_EXCEPTION_OUT_OF_MEMORY                     = 0xe0000008,
      /* Exception thrown by Chromium allocators to indicate OOM.
	     See base/process/memory.h in Chromium for rationale. */
  MD_EXCEPTION_CODE_WIN_UNHANDLED_CPP_EXCEPTION  = 0xe06d7363,
      /* Per http://support.microsoft.com/kb/185294,
         generated by Visual C++ compiler */
  MD_EXCEPTION_CODE_WIN_SIMULATED                = 0x0517a7ed
      /* Fake exception code used by Crashpad's
         CrashpadClient::DumpWithoutCrash. */
} MDExceptionCodeWin;


/* For (MDException).exception_information[2], when (MDException).exception_code
 * is MD_EXCEPTION_CODE_WIN_IN_PAGE_ERROR. This describes the underlying reason
 * for the error. These values come from ntstatus.h.
 *
 * The content of this enum was created from ntstatus.h in the 10 SDK
 * (version 10.0.19041.0) with
 *
 * egrep '#define [A-Z_0-9]+\s+\(\(NTSTATUS\)0x[048C][0-9A-F]+L\)' ntstatus.h
 * | tr -d '\r'
 * | sed -r 's@#define ([A-Z_0-9]+)\s+\(\(NTSTATUS\)(0x[048C][0-9A-F]+)L\).*@\2 \1@'
 * | sort
 * | sed -r 's@(0x[048C][0-9A-F]+) ([A-Z_0-9]+)@  MD_NTSTATUS_WIN_\2 = \1,@'
 *
 * With easy copy to clipboard with
 * | xclip -selection c  # on linux
 * | clip  # on windows
 * | pbcopy  # on mac */
typedef enum {
  MD_NTSTATUS_WIN_STATUS_SUCCESS = 0x00000000,
  MD_NTSTATUS_WIN_STATUS_WAIT_0 = 0x00000000,
  MD_NTSTATUS_WIN_STATUS_WAIT_1 = 0x00000001,
  MD_NTSTATUS_WIN_STATUS_WAIT_2 = 0x00000002,
  MD_NTSTATUS_WIN_STATUS_WAIT_3 = 0x00000003,
  MD_NTSTATUS_WIN_STATUS_WAIT_63 = 0x0000003F,
  MD_NTSTATUS_WIN_STATUS_ABANDONED = 0x00000080,
  MD_NTSTATUS_WIN_STATUS_ABANDONED_WAIT_0 = 0x00000080,
  MD_NTSTATUS_WIN_STATUS_ABANDONED_WAIT_63 = 0x000000BF,
  MD_NTSTATUS_WIN_STATUS_USER_APC = 0x000000C0,
  MD_NTSTATUS_WIN_STATUS_ALREADY_COMPLETE = 0x000000FF,
  MD_NTSTATUS_WIN_STATUS_KERNEL_APC = 0x00000100,
  MD_NTSTATUS_WIN_STATUS_ALERTED = 0x00000101,
  MD_NTSTATUS_WIN_STATUS_TIMEOUT = 0x00000102,
  MD_NTSTATUS_WIN_STATUS_PENDING = 0x00000103,
  MD_NTSTATUS_WIN_STATUS_REPARSE = 0x00000104,
  MD_NTSTATUS_WIN_STATUS_MORE_ENTRIES = 0x00000105,
  MD_NTSTATUS_WIN_STATUS_NOT_ALL_ASSIGNED = 0x00000106,
  MD_NTSTATUS_WIN_STATUS_SOME_NOT_MAPPED = 0x00000107,
  MD_NTSTATUS_WIN_STATUS_OPLOCK_BREAK_IN_PROGRESS = 0x00000108,
  MD_NTSTATUS_WIN_STATUS_VOLUME_MOUNTED = 0x00000109,
  MD_NTSTATUS_WIN_STATUS_RXACT_COMMITTED = 0x0000010A,
  MD_NTSTATUS_WIN_STATUS_NOTIFY_CLEANUP = 0x0000010B,
  MD_NTSTATUS_WIN_STATUS_NOTIFY_ENUM_DIR = 0x0000010C,
  MD_NTSTATUS_WIN_STATUS_NO_QUOTAS_FOR_ACCOUNT = 0x0000010D,
  MD_NTSTATUS_WIN_STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED = 0x0000010E,
  MD_NTSTATUS_WIN_STATUS_PAGE_FAULT_TRANSITION = 0x00000110,
  MD_NTSTATUS_WIN_STATUS_PAGE_FAULT_DEMAND_ZERO = 0x00000111,
  MD_NTSTATUS_WIN_STATUS_PAGE_FAULT_COPY_ON_WRITE = 0x00000112,
  MD_NTSTATUS_WIN_STATUS_PAGE_FAULT_GUARD_PAGE = 0x00000113,
  MD_NTSTATUS_WIN_STATUS_PAGE_FAULT_PAGING_FILE = 0x00000114,
  MD_NTSTATUS_WIN_STATUS_CACHE_PAGE_LOCKED = 0x00000115,
  MD_NTSTATUS_WIN_STATUS_CRASH_DUMP = 0x00000116,
  MD_NTSTATUS_WIN_STATUS_BUFFER_ALL_ZEROS = 0x00000117,
  MD_NTSTATUS_WIN_STATUS_REPARSE_OBJECT = 0x00000118,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_REQUIREMENTS_CHANGED = 0x00000119,
  MD_NTSTATUS_WIN_STATUS_TRANSLATION_COMPLETE = 0x00000120,
  MD_NTSTATUS_WIN_STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY = 0x00000121,
  MD_NTSTATUS_WIN_STATUS_NOTHING_TO_TERMINATE = 0x00000122,
  MD_NTSTATUS_WIN_STATUS_PROCESS_NOT_IN_JOB = 0x00000123,
  MD_NTSTATUS_WIN_STATUS_PROCESS_IN_JOB = 0x00000124,
  MD_NTSTATUS_WIN_STATUS_VOLSNAP_HIBERNATE_READY = 0x00000125,
  MD_NTSTATUS_WIN_STATUS_FSFILTER_OP_COMPLETED_SUCCESSFULLY = 0x00000126,
  MD_NTSTATUS_WIN_STATUS_INTERRUPT_VECTOR_ALREADY_CONNECTED = 0x00000127,
  MD_NTSTATUS_WIN_STATUS_INTERRUPT_STILL_CONNECTED = 0x00000128,
  MD_NTSTATUS_WIN_STATUS_PROCESS_CLONED = 0x00000129,
  MD_NTSTATUS_WIN_STATUS_FILE_LOCKED_WITH_ONLY_READERS = 0x0000012A,
  MD_NTSTATUS_WIN_STATUS_FILE_LOCKED_WITH_WRITERS = 0x0000012B,
  MD_NTSTATUS_WIN_STATUS_VALID_IMAGE_HASH = 0x0000012C,
  MD_NTSTATUS_WIN_STATUS_VALID_CATALOG_HASH = 0x0000012D,
  MD_NTSTATUS_WIN_STATUS_VALID_STRONG_CODE_HASH = 0x0000012E,
  MD_NTSTATUS_WIN_STATUS_GHOSTED = 0x0000012F,
  MD_NTSTATUS_WIN_STATUS_DATA_OVERWRITTEN = 0x00000130,
  MD_NTSTATUS_WIN_STATUS_RESOURCEMANAGER_READ_ONLY = 0x00000202,
  MD_NTSTATUS_WIN_STATUS_RING_PREVIOUSLY_EMPTY = 0x00000210,
  MD_NTSTATUS_WIN_STATUS_RING_PREVIOUSLY_FULL = 0x00000211,
  MD_NTSTATUS_WIN_STATUS_RING_PREVIOUSLY_ABOVE_QUOTA = 0x00000212,
  MD_NTSTATUS_WIN_STATUS_RING_NEWLY_EMPTY = 0x00000213,
  MD_NTSTATUS_WIN_STATUS_RING_SIGNAL_OPPOSITE_ENDPOINT = 0x00000214,
  MD_NTSTATUS_WIN_STATUS_OPLOCK_SWITCHED_TO_NEW_HANDLE = 0x00000215,
  MD_NTSTATUS_WIN_STATUS_OPLOCK_HANDLE_CLOSED = 0x00000216,
  MD_NTSTATUS_WIN_STATUS_WAIT_FOR_OPLOCK = 0x00000367,
  MD_NTSTATUS_WIN_STATUS_REPARSE_GLOBAL = 0x00000368,
  MD_NTSTATUS_WIN_DBG_EXCEPTION_HANDLED = 0x00010001,
  MD_NTSTATUS_WIN_DBG_CONTINUE = 0x00010002,
  MD_NTSTATUS_WIN_STATUS_FLT_IO_COMPLETE = 0x001C0001,
  MD_NTSTATUS_WIN_STATUS_RTPM_CONTEXT_CONTINUE = 0x00293000,
  MD_NTSTATUS_WIN_STATUS_RTPM_CONTEXT_COMPLETE = 0x00293001,
  MD_NTSTATUS_WIN_STATUS_HV_PENDING_PAGE_REQUESTS = 0x00350059,
  MD_NTSTATUS_WIN_STATUS_SPACES_REPAIRED = 0x00E70000,
  MD_NTSTATUS_WIN_STATUS_SPACES_PAUSE = 0x00E70001,
  MD_NTSTATUS_WIN_STATUS_SPACES_COMPLETE = 0x00E70002,
  MD_NTSTATUS_WIN_STATUS_SPACES_REDIRECT = 0x00E70003,
  MD_NTSTATUS_WIN_STATUS_OBJECT_NAME_EXISTS = 0x40000000,
  MD_NTSTATUS_WIN_STATUS_THREAD_WAS_SUSPENDED = 0x40000001,
  MD_NTSTATUS_WIN_STATUS_WORKING_SET_LIMIT_RANGE = 0x40000002,
  MD_NTSTATUS_WIN_STATUS_IMAGE_NOT_AT_BASE = 0x40000003,
  MD_NTSTATUS_WIN_STATUS_RXACT_STATE_CREATED = 0x40000004,
  MD_NTSTATUS_WIN_STATUS_SEGMENT_NOTIFICATION = 0x40000005,
  MD_NTSTATUS_WIN_STATUS_LOCAL_USER_SESSION_KEY = 0x40000006,
  MD_NTSTATUS_WIN_STATUS_BAD_CURRENT_DIRECTORY = 0x40000007,
  MD_NTSTATUS_WIN_STATUS_SERIAL_MORE_WRITES = 0x40000008,
  MD_NTSTATUS_WIN_STATUS_REGISTRY_RECOVERED = 0x40000009,
  MD_NTSTATUS_WIN_STATUS_FT_READ_RECOVERY_FROM_BACKUP = 0x4000000A,
  MD_NTSTATUS_WIN_STATUS_FT_WRITE_RECOVERY = 0x4000000B,
  MD_NTSTATUS_WIN_STATUS_SERIAL_COUNTER_TIMEOUT = 0x4000000C,
  MD_NTSTATUS_WIN_STATUS_NULL_LM_PASSWORD = 0x4000000D,
  MD_NTSTATUS_WIN_STATUS_IMAGE_MACHINE_TYPE_MISMATCH = 0x4000000E,
  MD_NTSTATUS_WIN_STATUS_RECEIVE_PARTIAL = 0x4000000F,
  MD_NTSTATUS_WIN_STATUS_RECEIVE_EXPEDITED = 0x40000010,
  MD_NTSTATUS_WIN_STATUS_RECEIVE_PARTIAL_EXPEDITED = 0x40000011,
  MD_NTSTATUS_WIN_STATUS_EVENT_DONE = 0x40000012,
  MD_NTSTATUS_WIN_STATUS_EVENT_PENDING = 0x40000013,
  MD_NTSTATUS_WIN_STATUS_CHECKING_FILE_SYSTEM = 0x40000014,
  MD_NTSTATUS_WIN_STATUS_FATAL_APP_EXIT = 0x40000015,
  MD_NTSTATUS_WIN_STATUS_PREDEFINED_HANDLE = 0x40000016,
  MD_NTSTATUS_WIN_STATUS_WAS_UNLOCKED = 0x40000017,
  MD_NTSTATUS_WIN_STATUS_SERVICE_NOTIFICATION = 0x40000018,
  MD_NTSTATUS_WIN_STATUS_WAS_LOCKED = 0x40000019,
  MD_NTSTATUS_WIN_STATUS_LOG_HARD_ERROR = 0x4000001A,
  MD_NTSTATUS_WIN_STATUS_ALREADY_WIN32 = 0x4000001B,
  MD_NTSTATUS_WIN_STATUS_WX86_UNSIMULATE = 0x4000001C,
  MD_NTSTATUS_WIN_STATUS_WX86_CONTINUE = 0x4000001D,
  MD_NTSTATUS_WIN_STATUS_WX86_SINGLE_STEP = 0x4000001E,
  MD_NTSTATUS_WIN_STATUS_WX86_BREAKPOINT = 0x4000001F,
  MD_NTSTATUS_WIN_STATUS_WX86_EXCEPTION_CONTINUE = 0x40000020,
  MD_NTSTATUS_WIN_STATUS_WX86_EXCEPTION_LASTCHANCE = 0x40000021,
  MD_NTSTATUS_WIN_STATUS_WX86_EXCEPTION_CHAIN = 0x40000022,
  MD_NTSTATUS_WIN_STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE = 0x40000023,
  MD_NTSTATUS_WIN_STATUS_NO_YIELD_PERFORMED = 0x40000024,
  MD_NTSTATUS_WIN_STATUS_TIMER_RESUME_IGNORED = 0x40000025,
  MD_NTSTATUS_WIN_STATUS_ARBITRATION_UNHANDLED = 0x40000026,
  MD_NTSTATUS_WIN_STATUS_CARDBUS_NOT_SUPPORTED = 0x40000027,
  MD_NTSTATUS_WIN_STATUS_WX86_CREATEWX86TIB = 0x40000028,
  MD_NTSTATUS_WIN_STATUS_MP_PROCESSOR_MISMATCH = 0x40000029,
  MD_NTSTATUS_WIN_STATUS_HIBERNATED = 0x4000002A,
  MD_NTSTATUS_WIN_STATUS_RESUME_HIBERNATION = 0x4000002B,
  MD_NTSTATUS_WIN_STATUS_FIRMWARE_UPDATED = 0x4000002C,
  MD_NTSTATUS_WIN_STATUS_DRIVERS_LEAKING_LOCKED_PAGES = 0x4000002D,
  MD_NTSTATUS_WIN_STATUS_MESSAGE_RETRIEVED = 0x4000002E,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_POWERSTATE_TRANSITION = 0x4000002F,
  MD_NTSTATUS_WIN_STATUS_ALPC_CHECK_COMPLETION_LIST = 0x40000030,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_POWERSTATE_COMPLEX_TRANSITION = 0x40000031,
  MD_NTSTATUS_WIN_STATUS_ACCESS_AUDIT_BY_POLICY = 0x40000032,
  MD_NTSTATUS_WIN_STATUS_ABANDON_HIBERFILE = 0x40000033,
  MD_NTSTATUS_WIN_STATUS_BIZRULES_NOT_ENABLED = 0x40000034,
  MD_NTSTATUS_WIN_STATUS_FT_READ_FROM_COPY = 0x40000035,
  MD_NTSTATUS_WIN_STATUS_IMAGE_AT_DIFFERENT_BASE = 0x40000036,
  MD_NTSTATUS_WIN_STATUS_PATCH_DEFERRED = 0x40000037,
  MD_NTSTATUS_WIN_STATUS_WAKE_SYSTEM = 0x40000294,
  MD_NTSTATUS_WIN_STATUS_DS_SHUTTING_DOWN = 0x40000370,
  MD_NTSTATUS_WIN_STATUS_DISK_REPAIR_REDIRECTED = 0x40000807,
  MD_NTSTATUS_WIN_STATUS_SERVICES_FAILED_AUTOSTART = 0x4000A144,
  MD_NTSTATUS_WIN_DBG_REPLY_LATER = 0x40010001,
  MD_NTSTATUS_WIN_DBG_UNABLE_TO_PROVIDE_HANDLE = 0x40010002,
  MD_NTSTATUS_WIN_DBG_TERMINATE_THREAD = 0x40010003,
  MD_NTSTATUS_WIN_DBG_TERMINATE_PROCESS = 0x40010004,
  MD_NTSTATUS_WIN_DBG_CONTROL_C = 0x40010005,
  MD_NTSTATUS_WIN_DBG_PRINTEXCEPTION_C = 0x40010006,
  MD_NTSTATUS_WIN_DBG_RIPEXCEPTION = 0x40010007,
  MD_NTSTATUS_WIN_DBG_CONTROL_BREAK = 0x40010008,
  MD_NTSTATUS_WIN_DBG_COMMAND_EXCEPTION = 0x40010009,
  MD_NTSTATUS_WIN_DBG_PRINTEXCEPTION_WIDE_C = 0x4001000A,
  MD_NTSTATUS_WIN_RPC_NT_UUID_LOCAL_ONLY = 0x40020056,
  MD_NTSTATUS_WIN_RPC_NT_SEND_INCOMPLETE = 0x400200AF,
  MD_NTSTATUS_WIN_STATUS_CTX_CDM_CONNECT = 0x400A0004,
  MD_NTSTATUS_WIN_STATUS_CTX_CDM_DISCONNECT = 0x400A0005,
  MD_NTSTATUS_WIN_STATUS_SXS_RELEASE_ACTIVATION_CONTEXT = 0x4015000D,
  MD_NTSTATUS_WIN_STATUS_HEURISTIC_DAMAGE_POSSIBLE = 0x40190001,
  MD_NTSTATUS_WIN_STATUS_RECOVERY_NOT_NEEDED = 0x40190034,
  MD_NTSTATUS_WIN_STATUS_RM_ALREADY_STARTED = 0x40190035,
  MD_NTSTATUS_WIN_STATUS_LOG_NO_RESTART = 0x401A000C,
  MD_NTSTATUS_WIN_STATUS_VIDEO_DRIVER_DEBUG_REPORT_REQUEST = 0x401B00EC,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PARTIAL_DATA_POPULATED = 0x401E000A,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_SKIP_ALLOCATION_PREPARATION = 0x401E0201,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MODE_NOT_PINNED = 0x401E0307,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_PREFERRED_MODE = 0x401E031E,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DATASET_IS_EMPTY = 0x401E034B,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_MORE_ELEMENTS_IN_DATASET = 0x401E034C,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_PINNED = 0x401E0351,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_UNKNOWN_CHILD_STATUS = 0x401E042F,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_LEADLINK_START_DEFERRED = 0x401E0437,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_POLLING_TOO_FREQUENTLY = 0x401E0439,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_START_DEFERRED = 0x401E043A,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DEPENDABLE_CHILD_STATUS = 0x401E043C,
  MD_NTSTATUS_WIN_STATUS_NDIS_INDICATION_REQUIRED = 0x40230001,
  MD_NTSTATUS_WIN_STATUS_PCP_UNSUPPORTED_PSS_SALT = 0x40292023,
  MD_NTSTATUS_WIN_STATUS_GUARD_PAGE_VIOLATION = 0x80000001,
  MD_NTSTATUS_WIN_STATUS_DATATYPE_MISALIGNMENT = 0x80000002,
  MD_NTSTATUS_WIN_STATUS_BREAKPOINT = 0x80000003,
  MD_NTSTATUS_WIN_STATUS_SINGLE_STEP = 0x80000004,
  MD_NTSTATUS_WIN_STATUS_BUFFER_OVERFLOW = 0x80000005,
  MD_NTSTATUS_WIN_STATUS_NO_MORE_FILES = 0x80000006,
  MD_NTSTATUS_WIN_STATUS_WAKE_SYSTEM_DEBUGGER = 0x80000007,
  MD_NTSTATUS_WIN_STATUS_HANDLES_CLOSED = 0x8000000A,
  MD_NTSTATUS_WIN_STATUS_NO_INHERITANCE = 0x8000000B,
  MD_NTSTATUS_WIN_STATUS_GUID_SUBSTITUTION_MADE = 0x8000000C,
  MD_NTSTATUS_WIN_STATUS_PARTIAL_COPY = 0x8000000D,
  MD_NTSTATUS_WIN_STATUS_DEVICE_PAPER_EMPTY = 0x8000000E,
  MD_NTSTATUS_WIN_STATUS_DEVICE_POWERED_OFF = 0x8000000F,
  MD_NTSTATUS_WIN_STATUS_DEVICE_OFF_LINE = 0x80000010,
  MD_NTSTATUS_WIN_STATUS_DEVICE_BUSY = 0x80000011,
  MD_NTSTATUS_WIN_STATUS_NO_MORE_EAS = 0x80000012,
  MD_NTSTATUS_WIN_STATUS_INVALID_EA_NAME = 0x80000013,
  MD_NTSTATUS_WIN_STATUS_EA_LIST_INCONSISTENT = 0x80000014,
  MD_NTSTATUS_WIN_STATUS_INVALID_EA_FLAG = 0x80000015,
  MD_NTSTATUS_WIN_STATUS_VERIFY_REQUIRED = 0x80000016,
  MD_NTSTATUS_WIN_STATUS_EXTRANEOUS_INFORMATION = 0x80000017,
  MD_NTSTATUS_WIN_STATUS_RXACT_COMMIT_NECESSARY = 0x80000018,
  MD_NTSTATUS_WIN_STATUS_NO_MORE_ENTRIES = 0x8000001A,
  MD_NTSTATUS_WIN_STATUS_FILEMARK_DETECTED = 0x8000001B,
  MD_NTSTATUS_WIN_STATUS_MEDIA_CHANGED = 0x8000001C,
  MD_NTSTATUS_WIN_STATUS_BUS_RESET = 0x8000001D,
  MD_NTSTATUS_WIN_STATUS_END_OF_MEDIA = 0x8000001E,
  MD_NTSTATUS_WIN_STATUS_BEGINNING_OF_MEDIA = 0x8000001F,
  MD_NTSTATUS_WIN_STATUS_MEDIA_CHECK = 0x80000020,
  MD_NTSTATUS_WIN_STATUS_SETMARK_DETECTED = 0x80000021,
  MD_NTSTATUS_WIN_STATUS_NO_DATA_DETECTED = 0x80000022,
  MD_NTSTATUS_WIN_STATUS_REDIRECTOR_HAS_OPEN_HANDLES = 0x80000023,
  MD_NTSTATUS_WIN_STATUS_SERVER_HAS_OPEN_HANDLES = 0x80000024,
  MD_NTSTATUS_WIN_STATUS_ALREADY_DISCONNECTED = 0x80000025,
  MD_NTSTATUS_WIN_STATUS_LONGJUMP = 0x80000026,
  MD_NTSTATUS_WIN_STATUS_CLEANER_CARTRIDGE_INSTALLED = 0x80000027,
  MD_NTSTATUS_WIN_STATUS_PLUGPLAY_QUERY_VETOED = 0x80000028,
  MD_NTSTATUS_WIN_STATUS_UNWIND_CONSOLIDATE = 0x80000029,
  MD_NTSTATUS_WIN_STATUS_REGISTRY_HIVE_RECOVERED = 0x8000002A,
  MD_NTSTATUS_WIN_STATUS_DLL_MIGHT_BE_INSECURE = 0x8000002B,
  MD_NTSTATUS_WIN_STATUS_DLL_MIGHT_BE_INCOMPATIBLE = 0x8000002C,
  MD_NTSTATUS_WIN_STATUS_STOPPED_ON_SYMLINK = 0x8000002D,
  MD_NTSTATUS_WIN_STATUS_CANNOT_GRANT_REQUESTED_OPLOCK = 0x8000002E,
  MD_NTSTATUS_WIN_STATUS_NO_ACE_CONDITION = 0x8000002F,
  MD_NTSTATUS_WIN_STATUS_DEVICE_SUPPORT_IN_PROGRESS = 0x80000030,
  MD_NTSTATUS_WIN_STATUS_DEVICE_POWER_CYCLE_REQUIRED = 0x80000031,
  MD_NTSTATUS_WIN_STATUS_NO_WORK_DONE = 0x80000032,
  MD_NTSTATUS_WIN_STATUS_RETURN_ADDRESS_HIJACK_ATTEMPT = 0x80000033,
  MD_NTSTATUS_WIN_STATUS_DEVICE_REQUIRES_CLEANING = 0x80000288,
  MD_NTSTATUS_WIN_STATUS_DEVICE_DOOR_OPEN = 0x80000289,
  MD_NTSTATUS_WIN_STATUS_DATA_LOST_REPAIR = 0x80000803,
  MD_NTSTATUS_WIN_STATUS_GPIO_INTERRUPT_ALREADY_UNMASKED = 0x8000A127,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROPERTY_BLOB_CHECKSUM_MISMATCH = 0x8000CF00,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROPERTY_BLOB_TOO_LARGE = 0x8000CF04,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_TOO_MANY_PROPERTY_BLOBS = 0x8000CF05,
  MD_NTSTATUS_WIN_DBG_EXCEPTION_NOT_HANDLED = 0x80010001,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_ALREADY_UP = 0x80130001,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_ALREADY_DOWN = 0x80130002,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETWORK_ALREADY_ONLINE = 0x80130003,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETWORK_ALREADY_OFFLINE = 0x80130004,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_ALREADY_MEMBER = 0x80130005,
  MD_NTSTATUS_WIN_STATUS_COULD_NOT_RESIZE_LOG = 0x80190009,
  MD_NTSTATUS_WIN_STATUS_NO_TXF_METADATA = 0x80190029,
  MD_NTSTATUS_WIN_STATUS_CANT_RECOVER_WITH_HANDLE_OPEN = 0x80190031,
  MD_NTSTATUS_WIN_STATUS_TXF_METADATA_ALREADY_PRESENT = 0x80190041,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_SCOPE_CALLBACKS_NOT_SET = 0x80190042,
  MD_NTSTATUS_WIN_STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD_RECOVERED = 0x801B00EB,
  MD_NTSTATUS_WIN_STATUS_FLT_BUFFER_TOO_SMALL = 0x801C0001,
  MD_NTSTATUS_WIN_STATUS_FVE_PARTIAL_METADATA = 0x80210001,
  MD_NTSTATUS_WIN_STATUS_FVE_TRANSIENT_STATE = 0x80210002,
  MD_NTSTATUS_WIN_STATUS_VID_REMOTE_NODE_PARENT_GPA_PAGES_USED = 0x80370001,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_INCOMPLETE_REGENERATION = 0x80380001,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_INCOMPLETE_DISK_MIGRATION = 0x80380002,
  MD_NTSTATUS_WIN_STATUS_BCD_NOT_ALL_ENTRIES_IMPORTED = 0x80390001,
  MD_NTSTATUS_WIN_STATUS_BCD_NOT_ALL_ENTRIES_SYNCHRONIZED = 0x80390003,
  MD_NTSTATUS_WIN_STATUS_QUERY_STORAGE_ERROR = 0x803A0001,
  MD_NTSTATUS_WIN_STATUS_GDI_HANDLE_LEAK = 0x803F0001,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_NOT_ENABLED = 0x80430006,
  MD_NTSTATUS_WIN_STATUS_UNSUCCESSFUL = 0xC0000001,
  MD_NTSTATUS_WIN_STATUS_NOT_IMPLEMENTED = 0xC0000002,
  MD_NTSTATUS_WIN_STATUS_INVALID_INFO_CLASS = 0xC0000003,
  MD_NTSTATUS_WIN_STATUS_INFO_LENGTH_MISMATCH = 0xC0000004,
  MD_NTSTATUS_WIN_STATUS_ACCESS_VIOLATION = 0xC0000005,
  MD_NTSTATUS_WIN_STATUS_IN_PAGE_ERROR = 0xC0000006,
  MD_NTSTATUS_WIN_STATUS_PAGEFILE_QUOTA = 0xC0000007,
  MD_NTSTATUS_WIN_STATUS_INVALID_HANDLE = 0xC0000008,
  MD_NTSTATUS_WIN_STATUS_BAD_INITIAL_STACK = 0xC0000009,
  MD_NTSTATUS_WIN_STATUS_BAD_INITIAL_PC = 0xC000000A,
  MD_NTSTATUS_WIN_STATUS_INVALID_CID = 0xC000000B,
  MD_NTSTATUS_WIN_STATUS_TIMER_NOT_CANCELED = 0xC000000C,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER = 0xC000000D,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_DEVICE = 0xC000000E,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_FILE = 0xC000000F,
  MD_NTSTATUS_WIN_STATUS_INVALID_DEVICE_REQUEST = 0xC0000010,
  MD_NTSTATUS_WIN_STATUS_END_OF_FILE = 0xC0000011,
  MD_NTSTATUS_WIN_STATUS_WRONG_VOLUME = 0xC0000012,
  MD_NTSTATUS_WIN_STATUS_NO_MEDIA_IN_DEVICE = 0xC0000013,
  MD_NTSTATUS_WIN_STATUS_UNRECOGNIZED_MEDIA = 0xC0000014,
  MD_NTSTATUS_WIN_STATUS_NONEXISTENT_SECTOR = 0xC0000015,
  MD_NTSTATUS_WIN_STATUS_MORE_PROCESSING_REQUIRED = 0xC0000016,
  MD_NTSTATUS_WIN_STATUS_NO_MEMORY = 0xC0000017,
  MD_NTSTATUS_WIN_STATUS_CONFLICTING_ADDRESSES = 0xC0000018,
  MD_NTSTATUS_WIN_STATUS_NOT_MAPPED_VIEW = 0xC0000019,
  MD_NTSTATUS_WIN_STATUS_UNABLE_TO_FREE_VM = 0xC000001A,
  MD_NTSTATUS_WIN_STATUS_UNABLE_TO_DELETE_SECTION = 0xC000001B,
  MD_NTSTATUS_WIN_STATUS_INVALID_SYSTEM_SERVICE = 0xC000001C,
  MD_NTSTATUS_WIN_STATUS_ILLEGAL_INSTRUCTION = 0xC000001D,
  MD_NTSTATUS_WIN_STATUS_INVALID_LOCK_SEQUENCE = 0xC000001E,
  MD_NTSTATUS_WIN_STATUS_INVALID_VIEW_SIZE = 0xC000001F,
  MD_NTSTATUS_WIN_STATUS_INVALID_FILE_FOR_SECTION = 0xC0000020,
  MD_NTSTATUS_WIN_STATUS_ALREADY_COMMITTED = 0xC0000021,
  MD_NTSTATUS_WIN_STATUS_ACCESS_DENIED = 0xC0000022,
  MD_NTSTATUS_WIN_STATUS_BUFFER_TOO_SMALL = 0xC0000023,
  MD_NTSTATUS_WIN_STATUS_OBJECT_TYPE_MISMATCH = 0xC0000024,
  MD_NTSTATUS_WIN_STATUS_NONCONTINUABLE_EXCEPTION = 0xC0000025,
  MD_NTSTATUS_WIN_STATUS_INVALID_DISPOSITION = 0xC0000026,
  MD_NTSTATUS_WIN_STATUS_UNWIND = 0xC0000027,
  MD_NTSTATUS_WIN_STATUS_BAD_STACK = 0xC0000028,
  MD_NTSTATUS_WIN_STATUS_INVALID_UNWIND_TARGET = 0xC0000029,
  MD_NTSTATUS_WIN_STATUS_NOT_LOCKED = 0xC000002A,
  MD_NTSTATUS_WIN_STATUS_PARITY_ERROR = 0xC000002B,
  MD_NTSTATUS_WIN_STATUS_UNABLE_TO_DECOMMIT_VM = 0xC000002C,
  MD_NTSTATUS_WIN_STATUS_NOT_COMMITTED = 0xC000002D,
  MD_NTSTATUS_WIN_STATUS_INVALID_PORT_ATTRIBUTES = 0xC000002E,
  MD_NTSTATUS_WIN_STATUS_PORT_MESSAGE_TOO_LONG = 0xC000002F,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_MIX = 0xC0000030,
  MD_NTSTATUS_WIN_STATUS_INVALID_QUOTA_LOWER = 0xC0000031,
  MD_NTSTATUS_WIN_STATUS_DISK_CORRUPT_ERROR = 0xC0000032,
  MD_NTSTATUS_WIN_STATUS_OBJECT_NAME_INVALID = 0xC0000033,
  MD_NTSTATUS_WIN_STATUS_OBJECT_NAME_NOT_FOUND = 0xC0000034,
  MD_NTSTATUS_WIN_STATUS_OBJECT_NAME_COLLISION = 0xC0000035,
  MD_NTSTATUS_WIN_STATUS_PORT_DO_NOT_DISTURB = 0xC0000036,
  MD_NTSTATUS_WIN_STATUS_PORT_DISCONNECTED = 0xC0000037,
  MD_NTSTATUS_WIN_STATUS_DEVICE_ALREADY_ATTACHED = 0xC0000038,
  MD_NTSTATUS_WIN_STATUS_OBJECT_PATH_INVALID = 0xC0000039,
  MD_NTSTATUS_WIN_STATUS_OBJECT_PATH_NOT_FOUND = 0xC000003A,
  MD_NTSTATUS_WIN_STATUS_OBJECT_PATH_SYNTAX_BAD = 0xC000003B,
  MD_NTSTATUS_WIN_STATUS_DATA_OVERRUN = 0xC000003C,
  MD_NTSTATUS_WIN_STATUS_DATA_LATE_ERROR = 0xC000003D,
  MD_NTSTATUS_WIN_STATUS_DATA_ERROR = 0xC000003E,
  MD_NTSTATUS_WIN_STATUS_CRC_ERROR = 0xC000003F,
  MD_NTSTATUS_WIN_STATUS_SECTION_TOO_BIG = 0xC0000040,
  MD_NTSTATUS_WIN_STATUS_PORT_CONNECTION_REFUSED = 0xC0000041,
  MD_NTSTATUS_WIN_STATUS_INVALID_PORT_HANDLE = 0xC0000042,
  MD_NTSTATUS_WIN_STATUS_SHARING_VIOLATION = 0xC0000043,
  MD_NTSTATUS_WIN_STATUS_QUOTA_EXCEEDED = 0xC0000044,
  MD_NTSTATUS_WIN_STATUS_INVALID_PAGE_PROTECTION = 0xC0000045,
  MD_NTSTATUS_WIN_STATUS_MUTANT_NOT_OWNED = 0xC0000046,
  MD_NTSTATUS_WIN_STATUS_SEMAPHORE_LIMIT_EXCEEDED = 0xC0000047,
  MD_NTSTATUS_WIN_STATUS_PORT_ALREADY_SET = 0xC0000048,
  MD_NTSTATUS_WIN_STATUS_SECTION_NOT_IMAGE = 0xC0000049,
  MD_NTSTATUS_WIN_STATUS_SUSPEND_COUNT_EXCEEDED = 0xC000004A,
  MD_NTSTATUS_WIN_STATUS_THREAD_IS_TERMINATING = 0xC000004B,
  MD_NTSTATUS_WIN_STATUS_BAD_WORKING_SET_LIMIT = 0xC000004C,
  MD_NTSTATUS_WIN_STATUS_INCOMPATIBLE_FILE_MAP = 0xC000004D,
  MD_NTSTATUS_WIN_STATUS_SECTION_PROTECTION = 0xC000004E,
  MD_NTSTATUS_WIN_STATUS_EAS_NOT_SUPPORTED = 0xC000004F,
  MD_NTSTATUS_WIN_STATUS_EA_TOO_LARGE = 0xC0000050,
  MD_NTSTATUS_WIN_STATUS_NONEXISTENT_EA_ENTRY = 0xC0000051,
  MD_NTSTATUS_WIN_STATUS_NO_EAS_ON_FILE = 0xC0000052,
  MD_NTSTATUS_WIN_STATUS_EA_CORRUPT_ERROR = 0xC0000053,
  MD_NTSTATUS_WIN_STATUS_FILE_LOCK_CONFLICT = 0xC0000054,
  MD_NTSTATUS_WIN_STATUS_LOCK_NOT_GRANTED = 0xC0000055,
  MD_NTSTATUS_WIN_STATUS_DELETE_PENDING = 0xC0000056,
  MD_NTSTATUS_WIN_STATUS_CTL_FILE_NOT_SUPPORTED = 0xC0000057,
  MD_NTSTATUS_WIN_STATUS_UNKNOWN_REVISION = 0xC0000058,
  MD_NTSTATUS_WIN_STATUS_REVISION_MISMATCH = 0xC0000059,
  MD_NTSTATUS_WIN_STATUS_INVALID_OWNER = 0xC000005A,
  MD_NTSTATUS_WIN_STATUS_INVALID_PRIMARY_GROUP = 0xC000005B,
  MD_NTSTATUS_WIN_STATUS_NO_IMPERSONATION_TOKEN = 0xC000005C,
  MD_NTSTATUS_WIN_STATUS_CANT_DISABLE_MANDATORY = 0xC000005D,
  MD_NTSTATUS_WIN_STATUS_NO_LOGON_SERVERS = 0xC000005E,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_LOGON_SESSION = 0xC000005F,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_PRIVILEGE = 0xC0000060,
  MD_NTSTATUS_WIN_STATUS_PRIVILEGE_NOT_HELD = 0xC0000061,
  MD_NTSTATUS_WIN_STATUS_INVALID_ACCOUNT_NAME = 0xC0000062,
  MD_NTSTATUS_WIN_STATUS_USER_EXISTS = 0xC0000063,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_USER = 0xC0000064,
  MD_NTSTATUS_WIN_STATUS_GROUP_EXISTS = 0xC0000065,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_GROUP = 0xC0000066,
  MD_NTSTATUS_WIN_STATUS_MEMBER_IN_GROUP = 0xC0000067,
  MD_NTSTATUS_WIN_STATUS_MEMBER_NOT_IN_GROUP = 0xC0000068,
  MD_NTSTATUS_WIN_STATUS_LAST_ADMIN = 0xC0000069,
  MD_NTSTATUS_WIN_STATUS_WRONG_PASSWORD = 0xC000006A,
  MD_NTSTATUS_WIN_STATUS_ILL_FORMED_PASSWORD = 0xC000006B,
  MD_NTSTATUS_WIN_STATUS_PASSWORD_RESTRICTION = 0xC000006C,
  MD_NTSTATUS_WIN_STATUS_LOGON_FAILURE = 0xC000006D,
  MD_NTSTATUS_WIN_STATUS_ACCOUNT_RESTRICTION = 0xC000006E,
  MD_NTSTATUS_WIN_STATUS_INVALID_LOGON_HOURS = 0xC000006F,
  MD_NTSTATUS_WIN_STATUS_INVALID_WORKSTATION = 0xC0000070,
  MD_NTSTATUS_WIN_STATUS_PASSWORD_EXPIRED = 0xC0000071,
  MD_NTSTATUS_WIN_STATUS_ACCOUNT_DISABLED = 0xC0000072,
  MD_NTSTATUS_WIN_STATUS_NONE_MAPPED = 0xC0000073,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_LUIDS_REQUESTED = 0xC0000074,
  MD_NTSTATUS_WIN_STATUS_LUIDS_EXHAUSTED = 0xC0000075,
  MD_NTSTATUS_WIN_STATUS_INVALID_SUB_AUTHORITY = 0xC0000076,
  MD_NTSTATUS_WIN_STATUS_INVALID_ACL = 0xC0000077,
  MD_NTSTATUS_WIN_STATUS_INVALID_SID = 0xC0000078,
  MD_NTSTATUS_WIN_STATUS_INVALID_SECURITY_DESCR = 0xC0000079,
  MD_NTSTATUS_WIN_STATUS_PROCEDURE_NOT_FOUND = 0xC000007A,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_FORMAT = 0xC000007B,
  MD_NTSTATUS_WIN_STATUS_NO_TOKEN = 0xC000007C,
  MD_NTSTATUS_WIN_STATUS_BAD_INHERITANCE_ACL = 0xC000007D,
  MD_NTSTATUS_WIN_STATUS_RANGE_NOT_LOCKED = 0xC000007E,
  MD_NTSTATUS_WIN_STATUS_DISK_FULL = 0xC000007F,
  MD_NTSTATUS_WIN_STATUS_SERVER_DISABLED = 0xC0000080,
  MD_NTSTATUS_WIN_STATUS_SERVER_NOT_DISABLED = 0xC0000081,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_GUIDS_REQUESTED = 0xC0000082,
  MD_NTSTATUS_WIN_STATUS_GUIDS_EXHAUSTED = 0xC0000083,
  MD_NTSTATUS_WIN_STATUS_INVALID_ID_AUTHORITY = 0xC0000084,
  MD_NTSTATUS_WIN_STATUS_AGENTS_EXHAUSTED = 0xC0000085,
  MD_NTSTATUS_WIN_STATUS_INVALID_VOLUME_LABEL = 0xC0000086,
  MD_NTSTATUS_WIN_STATUS_SECTION_NOT_EXTENDED = 0xC0000087,
  MD_NTSTATUS_WIN_STATUS_NOT_MAPPED_DATA = 0xC0000088,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_DATA_NOT_FOUND = 0xC0000089,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_TYPE_NOT_FOUND = 0xC000008A,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_NAME_NOT_FOUND = 0xC000008B,
  MD_NTSTATUS_WIN_STATUS_ARRAY_BOUNDS_EXCEEDED = 0xC000008C,
  MD_NTSTATUS_WIN_STATUS_FLOAT_DENORMAL_OPERAND = 0xC000008D,
  MD_NTSTATUS_WIN_STATUS_FLOAT_DIVIDE_BY_ZERO = 0xC000008E,
  MD_NTSTATUS_WIN_STATUS_FLOAT_INEXACT_RESULT = 0xC000008F,
  MD_NTSTATUS_WIN_STATUS_FLOAT_INVALID_OPERATION = 0xC0000090,
  MD_NTSTATUS_WIN_STATUS_FLOAT_OVERFLOW = 0xC0000091,
  MD_NTSTATUS_WIN_STATUS_FLOAT_STACK_CHECK = 0xC0000092,
  MD_NTSTATUS_WIN_STATUS_FLOAT_UNDERFLOW = 0xC0000093,
  MD_NTSTATUS_WIN_STATUS_INTEGER_DIVIDE_BY_ZERO = 0xC0000094,
  MD_NTSTATUS_WIN_STATUS_INTEGER_OVERFLOW = 0xC0000095,
  MD_NTSTATUS_WIN_STATUS_PRIVILEGED_INSTRUCTION = 0xC0000096,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_PAGING_FILES = 0xC0000097,
  MD_NTSTATUS_WIN_STATUS_FILE_INVALID = 0xC0000098,
  MD_NTSTATUS_WIN_STATUS_ALLOTTED_SPACE_EXCEEDED = 0xC0000099,
  MD_NTSTATUS_WIN_STATUS_INSUFFICIENT_RESOURCES = 0xC000009A,
  MD_NTSTATUS_WIN_STATUS_DFS_EXIT_PATH_FOUND = 0xC000009B,
  MD_NTSTATUS_WIN_STATUS_DEVICE_DATA_ERROR = 0xC000009C,
  MD_NTSTATUS_WIN_STATUS_DEVICE_NOT_CONNECTED = 0xC000009D,
  MD_NTSTATUS_WIN_STATUS_DEVICE_POWER_FAILURE = 0xC000009E,
  MD_NTSTATUS_WIN_STATUS_FREE_VM_NOT_AT_BASE = 0xC000009F,
  MD_NTSTATUS_WIN_STATUS_MEMORY_NOT_ALLOCATED = 0xC00000A0,
  MD_NTSTATUS_WIN_STATUS_WORKING_SET_QUOTA = 0xC00000A1,
  MD_NTSTATUS_WIN_STATUS_MEDIA_WRITE_PROTECTED = 0xC00000A2,
  MD_NTSTATUS_WIN_STATUS_DEVICE_NOT_READY = 0xC00000A3,
  MD_NTSTATUS_WIN_STATUS_INVALID_GROUP_ATTRIBUTES = 0xC00000A4,
  MD_NTSTATUS_WIN_STATUS_BAD_IMPERSONATION_LEVEL = 0xC00000A5,
  MD_NTSTATUS_WIN_STATUS_CANT_OPEN_ANONYMOUS = 0xC00000A6,
  MD_NTSTATUS_WIN_STATUS_BAD_VALIDATION_CLASS = 0xC00000A7,
  MD_NTSTATUS_WIN_STATUS_BAD_TOKEN_TYPE = 0xC00000A8,
  MD_NTSTATUS_WIN_STATUS_BAD_MASTER_BOOT_RECORD = 0xC00000A9,
  MD_NTSTATUS_WIN_STATUS_INSTRUCTION_MISALIGNMENT = 0xC00000AA,
  MD_NTSTATUS_WIN_STATUS_INSTANCE_NOT_AVAILABLE = 0xC00000AB,
  MD_NTSTATUS_WIN_STATUS_PIPE_NOT_AVAILABLE = 0xC00000AC,
  MD_NTSTATUS_WIN_STATUS_INVALID_PIPE_STATE = 0xC00000AD,
  MD_NTSTATUS_WIN_STATUS_PIPE_BUSY = 0xC00000AE,
  MD_NTSTATUS_WIN_STATUS_ILLEGAL_FUNCTION = 0xC00000AF,
  MD_NTSTATUS_WIN_STATUS_PIPE_DISCONNECTED = 0xC00000B0,
  MD_NTSTATUS_WIN_STATUS_PIPE_CLOSING = 0xC00000B1,
  MD_NTSTATUS_WIN_STATUS_PIPE_CONNECTED = 0xC00000B2,
  MD_NTSTATUS_WIN_STATUS_PIPE_LISTENING = 0xC00000B3,
  MD_NTSTATUS_WIN_STATUS_INVALID_READ_MODE = 0xC00000B4,
  MD_NTSTATUS_WIN_STATUS_IO_TIMEOUT = 0xC00000B5,
  MD_NTSTATUS_WIN_STATUS_FILE_FORCED_CLOSED = 0xC00000B6,
  MD_NTSTATUS_WIN_STATUS_PROFILING_NOT_STARTED = 0xC00000B7,
  MD_NTSTATUS_WIN_STATUS_PROFILING_NOT_STOPPED = 0xC00000B8,
  MD_NTSTATUS_WIN_STATUS_COULD_NOT_INTERPRET = 0xC00000B9,
  MD_NTSTATUS_WIN_STATUS_FILE_IS_A_DIRECTORY = 0xC00000BA,
  MD_NTSTATUS_WIN_STATUS_NOT_SUPPORTED = 0xC00000BB,
  MD_NTSTATUS_WIN_STATUS_REMOTE_NOT_LISTENING = 0xC00000BC,
  MD_NTSTATUS_WIN_STATUS_DUPLICATE_NAME = 0xC00000BD,
  MD_NTSTATUS_WIN_STATUS_BAD_NETWORK_PATH = 0xC00000BE,
  MD_NTSTATUS_WIN_STATUS_NETWORK_BUSY = 0xC00000BF,
  MD_NTSTATUS_WIN_STATUS_DEVICE_DOES_NOT_EXIST = 0xC00000C0,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_COMMANDS = 0xC00000C1,
  MD_NTSTATUS_WIN_STATUS_ADAPTER_HARDWARE_ERROR = 0xC00000C2,
  MD_NTSTATUS_WIN_STATUS_INVALID_NETWORK_RESPONSE = 0xC00000C3,
  MD_NTSTATUS_WIN_STATUS_UNEXPECTED_NETWORK_ERROR = 0xC00000C4,
  MD_NTSTATUS_WIN_STATUS_BAD_REMOTE_ADAPTER = 0xC00000C5,
  MD_NTSTATUS_WIN_STATUS_PRINT_QUEUE_FULL = 0xC00000C6,
  MD_NTSTATUS_WIN_STATUS_NO_SPOOL_SPACE = 0xC00000C7,
  MD_NTSTATUS_WIN_STATUS_PRINT_CANCELLED = 0xC00000C8,
  MD_NTSTATUS_WIN_STATUS_NETWORK_NAME_DELETED = 0xC00000C9,
  MD_NTSTATUS_WIN_STATUS_NETWORK_ACCESS_DENIED = 0xC00000CA,
  MD_NTSTATUS_WIN_STATUS_BAD_DEVICE_TYPE = 0xC00000CB,
  MD_NTSTATUS_WIN_STATUS_BAD_NETWORK_NAME = 0xC00000CC,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_NAMES = 0xC00000CD,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_SESSIONS = 0xC00000CE,
  MD_NTSTATUS_WIN_STATUS_SHARING_PAUSED = 0xC00000CF,
  MD_NTSTATUS_WIN_STATUS_REQUEST_NOT_ACCEPTED = 0xC00000D0,
  MD_NTSTATUS_WIN_STATUS_REDIRECTOR_PAUSED = 0xC00000D1,
  MD_NTSTATUS_WIN_STATUS_NET_WRITE_FAULT = 0xC00000D2,
  MD_NTSTATUS_WIN_STATUS_PROFILING_AT_LIMIT = 0xC00000D3,
  MD_NTSTATUS_WIN_STATUS_NOT_SAME_DEVICE = 0xC00000D4,
  MD_NTSTATUS_WIN_STATUS_FILE_RENAMED = 0xC00000D5,
  MD_NTSTATUS_WIN_STATUS_VIRTUAL_CIRCUIT_CLOSED = 0xC00000D6,
  MD_NTSTATUS_WIN_STATUS_NO_SECURITY_ON_OBJECT = 0xC00000D7,
  MD_NTSTATUS_WIN_STATUS_CANT_WAIT = 0xC00000D8,
  MD_NTSTATUS_WIN_STATUS_PIPE_EMPTY = 0xC00000D9,
  MD_NTSTATUS_WIN_STATUS_CANT_ACCESS_DOMAIN_INFO = 0xC00000DA,
  MD_NTSTATUS_WIN_STATUS_CANT_TERMINATE_SELF = 0xC00000DB,
  MD_NTSTATUS_WIN_STATUS_INVALID_SERVER_STATE = 0xC00000DC,
  MD_NTSTATUS_WIN_STATUS_INVALID_DOMAIN_STATE = 0xC00000DD,
  MD_NTSTATUS_WIN_STATUS_INVALID_DOMAIN_ROLE = 0xC00000DE,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_DOMAIN = 0xC00000DF,
  MD_NTSTATUS_WIN_STATUS_DOMAIN_EXISTS = 0xC00000E0,
  MD_NTSTATUS_WIN_STATUS_DOMAIN_LIMIT_EXCEEDED = 0xC00000E1,
  MD_NTSTATUS_WIN_STATUS_OPLOCK_NOT_GRANTED = 0xC00000E2,
  MD_NTSTATUS_WIN_STATUS_INVALID_OPLOCK_PROTOCOL = 0xC00000E3,
  MD_NTSTATUS_WIN_STATUS_INTERNAL_DB_CORRUPTION = 0xC00000E4,
  MD_NTSTATUS_WIN_STATUS_INTERNAL_ERROR = 0xC00000E5,
  MD_NTSTATUS_WIN_STATUS_GENERIC_NOT_MAPPED = 0xC00000E6,
  MD_NTSTATUS_WIN_STATUS_BAD_DESCRIPTOR_FORMAT = 0xC00000E7,
  MD_NTSTATUS_WIN_STATUS_INVALID_USER_BUFFER = 0xC00000E8,
  MD_NTSTATUS_WIN_STATUS_UNEXPECTED_IO_ERROR = 0xC00000E9,
  MD_NTSTATUS_WIN_STATUS_UNEXPECTED_MM_CREATE_ERR = 0xC00000EA,
  MD_NTSTATUS_WIN_STATUS_UNEXPECTED_MM_MAP_ERROR = 0xC00000EB,
  MD_NTSTATUS_WIN_STATUS_UNEXPECTED_MM_EXTEND_ERR = 0xC00000EC,
  MD_NTSTATUS_WIN_STATUS_NOT_LOGON_PROCESS = 0xC00000ED,
  MD_NTSTATUS_WIN_STATUS_LOGON_SESSION_EXISTS = 0xC00000EE,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_1 = 0xC00000EF,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_2 = 0xC00000F0,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_3 = 0xC00000F1,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_4 = 0xC00000F2,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_5 = 0xC00000F3,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_6 = 0xC00000F4,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_7 = 0xC00000F5,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_8 = 0xC00000F6,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_9 = 0xC00000F7,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_10 = 0xC00000F8,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_11 = 0xC00000F9,
  MD_NTSTATUS_WIN_STATUS_INVALID_PARAMETER_12 = 0xC00000FA,
  MD_NTSTATUS_WIN_STATUS_REDIRECTOR_NOT_STARTED = 0xC00000FB,
  MD_NTSTATUS_WIN_STATUS_REDIRECTOR_STARTED = 0xC00000FC,
  MD_NTSTATUS_WIN_STATUS_STACK_OVERFLOW = 0xC00000FD,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_PACKAGE = 0xC00000FE,
  MD_NTSTATUS_WIN_STATUS_BAD_FUNCTION_TABLE = 0xC00000FF,
  MD_NTSTATUS_WIN_STATUS_VARIABLE_NOT_FOUND = 0xC0000100,
  MD_NTSTATUS_WIN_STATUS_DIRECTORY_NOT_EMPTY = 0xC0000101,
  MD_NTSTATUS_WIN_STATUS_FILE_CORRUPT_ERROR = 0xC0000102,
  MD_NTSTATUS_WIN_STATUS_NOT_A_DIRECTORY = 0xC0000103,
  MD_NTSTATUS_WIN_STATUS_BAD_LOGON_SESSION_STATE = 0xC0000104,
  MD_NTSTATUS_WIN_STATUS_LOGON_SESSION_COLLISION = 0xC0000105,
  MD_NTSTATUS_WIN_STATUS_NAME_TOO_LONG = 0xC0000106,
  MD_NTSTATUS_WIN_STATUS_FILES_OPEN = 0xC0000107,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_IN_USE = 0xC0000108,
  MD_NTSTATUS_WIN_STATUS_MESSAGE_NOT_FOUND = 0xC0000109,
  MD_NTSTATUS_WIN_STATUS_PROCESS_IS_TERMINATING = 0xC000010A,
  MD_NTSTATUS_WIN_STATUS_INVALID_LOGON_TYPE = 0xC000010B,
  MD_NTSTATUS_WIN_STATUS_NO_GUID_TRANSLATION = 0xC000010C,
  MD_NTSTATUS_WIN_STATUS_CANNOT_IMPERSONATE = 0xC000010D,
  MD_NTSTATUS_WIN_STATUS_IMAGE_ALREADY_LOADED = 0xC000010E,
  MD_NTSTATUS_WIN_STATUS_ABIOS_NOT_PRESENT = 0xC000010F,
  MD_NTSTATUS_WIN_STATUS_ABIOS_LID_NOT_EXIST = 0xC0000110,
  MD_NTSTATUS_WIN_STATUS_ABIOS_LID_ALREADY_OWNED = 0xC0000111,
  MD_NTSTATUS_WIN_STATUS_ABIOS_NOT_LID_OWNER = 0xC0000112,
  MD_NTSTATUS_WIN_STATUS_ABIOS_INVALID_COMMAND = 0xC0000113,
  MD_NTSTATUS_WIN_STATUS_ABIOS_INVALID_LID = 0xC0000114,
  MD_NTSTATUS_WIN_STATUS_ABIOS_SELECTOR_NOT_AVAILABLE = 0xC0000115,
  MD_NTSTATUS_WIN_STATUS_ABIOS_INVALID_SELECTOR = 0xC0000116,
  MD_NTSTATUS_WIN_STATUS_NO_LDT = 0xC0000117,
  MD_NTSTATUS_WIN_STATUS_INVALID_LDT_SIZE = 0xC0000118,
  MD_NTSTATUS_WIN_STATUS_INVALID_LDT_OFFSET = 0xC0000119,
  MD_NTSTATUS_WIN_STATUS_INVALID_LDT_DESCRIPTOR = 0xC000011A,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_NE_FORMAT = 0xC000011B,
  MD_NTSTATUS_WIN_STATUS_RXACT_INVALID_STATE = 0xC000011C,
  MD_NTSTATUS_WIN_STATUS_RXACT_COMMIT_FAILURE = 0xC000011D,
  MD_NTSTATUS_WIN_STATUS_MAPPED_FILE_SIZE_ZERO = 0xC000011E,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_OPENED_FILES = 0xC000011F,
  MD_NTSTATUS_WIN_STATUS_CANCELLED = 0xC0000120,
  MD_NTSTATUS_WIN_STATUS_CANNOT_DELETE = 0xC0000121,
  MD_NTSTATUS_WIN_STATUS_INVALID_COMPUTER_NAME = 0xC0000122,
  MD_NTSTATUS_WIN_STATUS_FILE_DELETED = 0xC0000123,
  MD_NTSTATUS_WIN_STATUS_SPECIAL_ACCOUNT = 0xC0000124,
  MD_NTSTATUS_WIN_STATUS_SPECIAL_GROUP = 0xC0000125,
  MD_NTSTATUS_WIN_STATUS_SPECIAL_USER = 0xC0000126,
  MD_NTSTATUS_WIN_STATUS_MEMBERS_PRIMARY_GROUP = 0xC0000127,
  MD_NTSTATUS_WIN_STATUS_FILE_CLOSED = 0xC0000128,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_THREADS = 0xC0000129,
  MD_NTSTATUS_WIN_STATUS_THREAD_NOT_IN_PROCESS = 0xC000012A,
  MD_NTSTATUS_WIN_STATUS_TOKEN_ALREADY_IN_USE = 0xC000012B,
  MD_NTSTATUS_WIN_STATUS_PAGEFILE_QUOTA_EXCEEDED = 0xC000012C,
  MD_NTSTATUS_WIN_STATUS_COMMITMENT_LIMIT = 0xC000012D,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_LE_FORMAT = 0xC000012E,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_NOT_MZ = 0xC000012F,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_PROTECT = 0xC0000130,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_WIN_16 = 0xC0000131,
  MD_NTSTATUS_WIN_STATUS_LOGON_SERVER_CONFLICT = 0xC0000132,
  MD_NTSTATUS_WIN_STATUS_TIME_DIFFERENCE_AT_DC = 0xC0000133,
  MD_NTSTATUS_WIN_STATUS_SYNCHRONIZATION_REQUIRED = 0xC0000134,
  MD_NTSTATUS_WIN_STATUS_DLL_NOT_FOUND = 0xC0000135,
  MD_NTSTATUS_WIN_STATUS_OPEN_FAILED = 0xC0000136,
  MD_NTSTATUS_WIN_STATUS_IO_PRIVILEGE_FAILED = 0xC0000137,
  MD_NTSTATUS_WIN_STATUS_ORDINAL_NOT_FOUND = 0xC0000138,
  MD_NTSTATUS_WIN_STATUS_ENTRYPOINT_NOT_FOUND = 0xC0000139,
  MD_NTSTATUS_WIN_STATUS_CONTROL_C_EXIT = 0xC000013A,
  MD_NTSTATUS_WIN_STATUS_LOCAL_DISCONNECT = 0xC000013B,
  MD_NTSTATUS_WIN_STATUS_REMOTE_DISCONNECT = 0xC000013C,
  MD_NTSTATUS_WIN_STATUS_REMOTE_RESOURCES = 0xC000013D,
  MD_NTSTATUS_WIN_STATUS_LINK_FAILED = 0xC000013E,
  MD_NTSTATUS_WIN_STATUS_LINK_TIMEOUT = 0xC000013F,
  MD_NTSTATUS_WIN_STATUS_INVALID_CONNECTION = 0xC0000140,
  MD_NTSTATUS_WIN_STATUS_INVALID_ADDRESS = 0xC0000141,
  MD_NTSTATUS_WIN_STATUS_DLL_INIT_FAILED = 0xC0000142,
  MD_NTSTATUS_WIN_STATUS_MISSING_SYSTEMFILE = 0xC0000143,
  MD_NTSTATUS_WIN_STATUS_UNHANDLED_EXCEPTION = 0xC0000144,
  MD_NTSTATUS_WIN_STATUS_APP_INIT_FAILURE = 0xC0000145,
  MD_NTSTATUS_WIN_STATUS_PAGEFILE_CREATE_FAILED = 0xC0000146,
  MD_NTSTATUS_WIN_STATUS_NO_PAGEFILE = 0xC0000147,
  MD_NTSTATUS_WIN_STATUS_INVALID_LEVEL = 0xC0000148,
  MD_NTSTATUS_WIN_STATUS_WRONG_PASSWORD_CORE = 0xC0000149,
  MD_NTSTATUS_WIN_STATUS_ILLEGAL_FLOAT_CONTEXT = 0xC000014A,
  MD_NTSTATUS_WIN_STATUS_PIPE_BROKEN = 0xC000014B,
  MD_NTSTATUS_WIN_STATUS_REGISTRY_CORRUPT = 0xC000014C,
  MD_NTSTATUS_WIN_STATUS_REGISTRY_IO_FAILED = 0xC000014D,
  MD_NTSTATUS_WIN_STATUS_NO_EVENT_PAIR = 0xC000014E,
  MD_NTSTATUS_WIN_STATUS_UNRECOGNIZED_VOLUME = 0xC000014F,
  MD_NTSTATUS_WIN_STATUS_SERIAL_NO_DEVICE_INITED = 0xC0000150,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_ALIAS = 0xC0000151,
  MD_NTSTATUS_WIN_STATUS_MEMBER_NOT_IN_ALIAS = 0xC0000152,
  MD_NTSTATUS_WIN_STATUS_MEMBER_IN_ALIAS = 0xC0000153,
  MD_NTSTATUS_WIN_STATUS_ALIAS_EXISTS = 0xC0000154,
  MD_NTSTATUS_WIN_STATUS_LOGON_NOT_GRANTED = 0xC0000155,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_SECRETS = 0xC0000156,
  MD_NTSTATUS_WIN_STATUS_SECRET_TOO_LONG = 0xC0000157,
  MD_NTSTATUS_WIN_STATUS_INTERNAL_DB_ERROR = 0xC0000158,
  MD_NTSTATUS_WIN_STATUS_FULLSCREEN_MODE = 0xC0000159,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_CONTEXT_IDS = 0xC000015A,
  MD_NTSTATUS_WIN_STATUS_LOGON_TYPE_NOT_GRANTED = 0xC000015B,
  MD_NTSTATUS_WIN_STATUS_NOT_REGISTRY_FILE = 0xC000015C,
  MD_NTSTATUS_WIN_STATUS_NT_CROSS_ENCRYPTION_REQUIRED = 0xC000015D,
  MD_NTSTATUS_WIN_STATUS_DOMAIN_CTRLR_CONFIG_ERROR = 0xC000015E,
  MD_NTSTATUS_WIN_STATUS_FT_MISSING_MEMBER = 0xC000015F,
  MD_NTSTATUS_WIN_STATUS_ILL_FORMED_SERVICE_ENTRY = 0xC0000160,
  MD_NTSTATUS_WIN_STATUS_ILLEGAL_CHARACTER = 0xC0000161,
  MD_NTSTATUS_WIN_STATUS_UNMAPPABLE_CHARACTER = 0xC0000162,
  MD_NTSTATUS_WIN_STATUS_UNDEFINED_CHARACTER = 0xC0000163,
  MD_NTSTATUS_WIN_STATUS_FLOPPY_VOLUME = 0xC0000164,
  MD_NTSTATUS_WIN_STATUS_FLOPPY_ID_MARK_NOT_FOUND = 0xC0000165,
  MD_NTSTATUS_WIN_STATUS_FLOPPY_WRONG_CYLINDER = 0xC0000166,
  MD_NTSTATUS_WIN_STATUS_FLOPPY_UNKNOWN_ERROR = 0xC0000167,
  MD_NTSTATUS_WIN_STATUS_FLOPPY_BAD_REGISTERS = 0xC0000168,
  MD_NTSTATUS_WIN_STATUS_DISK_RECALIBRATE_FAILED = 0xC0000169,
  MD_NTSTATUS_WIN_STATUS_DISK_OPERATION_FAILED = 0xC000016A,
  MD_NTSTATUS_WIN_STATUS_DISK_RESET_FAILED = 0xC000016B,
  MD_NTSTATUS_WIN_STATUS_SHARED_IRQ_BUSY = 0xC000016C,
  MD_NTSTATUS_WIN_STATUS_FT_ORPHANING = 0xC000016D,
  MD_NTSTATUS_WIN_STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT = 0xC000016E,
  MD_NTSTATUS_WIN_STATUS_PARTITION_FAILURE = 0xC0000172,
  MD_NTSTATUS_WIN_STATUS_INVALID_BLOCK_LENGTH = 0xC0000173,
  MD_NTSTATUS_WIN_STATUS_DEVICE_NOT_PARTITIONED = 0xC0000174,
  MD_NTSTATUS_WIN_STATUS_UNABLE_TO_LOCK_MEDIA = 0xC0000175,
  MD_NTSTATUS_WIN_STATUS_UNABLE_TO_UNLOAD_MEDIA = 0xC0000176,
  MD_NTSTATUS_WIN_STATUS_EOM_OVERFLOW = 0xC0000177,
  MD_NTSTATUS_WIN_STATUS_NO_MEDIA = 0xC0000178,
  MD_NTSTATUS_WIN_STATUS_NO_SUCH_MEMBER = 0xC000017A,
  MD_NTSTATUS_WIN_STATUS_INVALID_MEMBER = 0xC000017B,
  MD_NTSTATUS_WIN_STATUS_KEY_DELETED = 0xC000017C,
  MD_NTSTATUS_WIN_STATUS_NO_LOG_SPACE = 0xC000017D,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_SIDS = 0xC000017E,
  MD_NTSTATUS_WIN_STATUS_LM_CROSS_ENCRYPTION_REQUIRED = 0xC000017F,
  MD_NTSTATUS_WIN_STATUS_KEY_HAS_CHILDREN = 0xC0000180,
  MD_NTSTATUS_WIN_STATUS_CHILD_MUST_BE_VOLATILE = 0xC0000181,
  MD_NTSTATUS_WIN_STATUS_DEVICE_CONFIGURATION_ERROR = 0xC0000182,
  MD_NTSTATUS_WIN_STATUS_DRIVER_INTERNAL_ERROR = 0xC0000183,
  MD_NTSTATUS_WIN_STATUS_INVALID_DEVICE_STATE = 0xC0000184,
  MD_NTSTATUS_WIN_STATUS_IO_DEVICE_ERROR = 0xC0000185,
  MD_NTSTATUS_WIN_STATUS_DEVICE_PROTOCOL_ERROR = 0xC0000186,
  MD_NTSTATUS_WIN_STATUS_BACKUP_CONTROLLER = 0xC0000187,
  MD_NTSTATUS_WIN_STATUS_LOG_FILE_FULL = 0xC0000188,
  MD_NTSTATUS_WIN_STATUS_TOO_LATE = 0xC0000189,
  MD_NTSTATUS_WIN_STATUS_NO_TRUST_LSA_SECRET = 0xC000018A,
  MD_NTSTATUS_WIN_STATUS_NO_TRUST_SAM_ACCOUNT = 0xC000018B,
  MD_NTSTATUS_WIN_STATUS_TRUSTED_DOMAIN_FAILURE = 0xC000018C,
  MD_NTSTATUS_WIN_STATUS_TRUSTED_RELATIONSHIP_FAILURE = 0xC000018D,
  MD_NTSTATUS_WIN_STATUS_EVENTLOG_FILE_CORRUPT = 0xC000018E,
  MD_NTSTATUS_WIN_STATUS_EVENTLOG_CANT_START = 0xC000018F,
  MD_NTSTATUS_WIN_STATUS_TRUST_FAILURE = 0xC0000190,
  MD_NTSTATUS_WIN_STATUS_MUTANT_LIMIT_EXCEEDED = 0xC0000191,
  MD_NTSTATUS_WIN_STATUS_NETLOGON_NOT_STARTED = 0xC0000192,
  MD_NTSTATUS_WIN_STATUS_ACCOUNT_EXPIRED = 0xC0000193,
  MD_NTSTATUS_WIN_STATUS_POSSIBLE_DEADLOCK = 0xC0000194,
  MD_NTSTATUS_WIN_STATUS_NETWORK_CREDENTIAL_CONFLICT = 0xC0000195,
  MD_NTSTATUS_WIN_STATUS_REMOTE_SESSION_LIMIT = 0xC0000196,
  MD_NTSTATUS_WIN_STATUS_EVENTLOG_FILE_CHANGED = 0xC0000197,
  MD_NTSTATUS_WIN_STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT = 0xC0000198,
  MD_NTSTATUS_WIN_STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT = 0xC0000199,
  MD_NTSTATUS_WIN_STATUS_NOLOGON_SERVER_TRUST_ACCOUNT = 0xC000019A,
  MD_NTSTATUS_WIN_STATUS_DOMAIN_TRUST_INCONSISTENT = 0xC000019B,
  MD_NTSTATUS_WIN_STATUS_FS_DRIVER_REQUIRED = 0xC000019C,
  MD_NTSTATUS_WIN_STATUS_IMAGE_ALREADY_LOADED_AS_DLL = 0xC000019D,
  MD_NTSTATUS_WIN_STATUS_INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING = 0xC000019E,
  MD_NTSTATUS_WIN_STATUS_SHORT_NAMES_NOT_ENABLED_ON_VOLUME = 0xC000019F,
  MD_NTSTATUS_WIN_STATUS_SECURITY_STREAM_IS_INCONSISTENT = 0xC00001A0,
  MD_NTSTATUS_WIN_STATUS_INVALID_LOCK_RANGE = 0xC00001A1,
  MD_NTSTATUS_WIN_STATUS_INVALID_ACE_CONDITION = 0xC00001A2,
  MD_NTSTATUS_WIN_STATUS_IMAGE_SUBSYSTEM_NOT_PRESENT = 0xC00001A3,
  MD_NTSTATUS_WIN_STATUS_NOTIFICATION_GUID_ALREADY_DEFINED = 0xC00001A4,
  MD_NTSTATUS_WIN_STATUS_INVALID_EXCEPTION_HANDLER = 0xC00001A5,
  MD_NTSTATUS_WIN_STATUS_DUPLICATE_PRIVILEGES = 0xC00001A6,
  MD_NTSTATUS_WIN_STATUS_NOT_ALLOWED_ON_SYSTEM_FILE = 0xC00001A7,
  MD_NTSTATUS_WIN_STATUS_REPAIR_NEEDED = 0xC00001A8,
  MD_NTSTATUS_WIN_STATUS_QUOTA_NOT_ENABLED = 0xC00001A9,
  MD_NTSTATUS_WIN_STATUS_NO_APPLICATION_PACKAGE = 0xC00001AA,
  MD_NTSTATUS_WIN_STATUS_FILE_METADATA_OPTIMIZATION_IN_PROGRESS = 0xC00001AB,
  MD_NTSTATUS_WIN_STATUS_NOT_SAME_OBJECT = 0xC00001AC,
  MD_NTSTATUS_WIN_STATUS_FATAL_MEMORY_EXHAUSTION = 0xC00001AD,
  MD_NTSTATUS_WIN_STATUS_ERROR_PROCESS_NOT_IN_JOB = 0xC00001AE,
  MD_NTSTATUS_WIN_STATUS_CPU_SET_INVALID = 0xC00001AF,
  MD_NTSTATUS_WIN_STATUS_IO_DEVICE_INVALID_DATA = 0xC00001B0,
  MD_NTSTATUS_WIN_STATUS_IO_UNALIGNED_WRITE = 0xC00001B1,
  MD_NTSTATUS_WIN_STATUS_CONTROL_STACK_VIOLATION = 0xC00001B2,
  MD_NTSTATUS_WIN_STATUS_NETWORK_OPEN_RESTRICTION = 0xC0000201,
  MD_NTSTATUS_WIN_STATUS_NO_USER_SESSION_KEY = 0xC0000202,
  MD_NTSTATUS_WIN_STATUS_USER_SESSION_DELETED = 0xC0000203,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_LANG_NOT_FOUND = 0xC0000204,
  MD_NTSTATUS_WIN_STATUS_INSUFF_SERVER_RESOURCES = 0xC0000205,
  MD_NTSTATUS_WIN_STATUS_INVALID_BUFFER_SIZE = 0xC0000206,
  MD_NTSTATUS_WIN_STATUS_INVALID_ADDRESS_COMPONENT = 0xC0000207,
  MD_NTSTATUS_WIN_STATUS_INVALID_ADDRESS_WILDCARD = 0xC0000208,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_ADDRESSES = 0xC0000209,
  MD_NTSTATUS_WIN_STATUS_ADDRESS_ALREADY_EXISTS = 0xC000020A,
  MD_NTSTATUS_WIN_STATUS_ADDRESS_CLOSED = 0xC000020B,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_DISCONNECTED = 0xC000020C,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_RESET = 0xC000020D,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_NODES = 0xC000020E,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_ABORTED = 0xC000020F,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_TIMED_OUT = 0xC0000210,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NO_RELEASE = 0xC0000211,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NO_MATCH = 0xC0000212,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_RESPONDED = 0xC0000213,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_INVALID_ID = 0xC0000214,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_INVALID_TYPE = 0xC0000215,
  MD_NTSTATUS_WIN_STATUS_NOT_SERVER_SESSION = 0xC0000216,
  MD_NTSTATUS_WIN_STATUS_NOT_CLIENT_SESSION = 0xC0000217,
  MD_NTSTATUS_WIN_STATUS_CANNOT_LOAD_REGISTRY_FILE = 0xC0000218,
  MD_NTSTATUS_WIN_STATUS_DEBUG_ATTACH_FAILED = 0xC0000219,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_PROCESS_TERMINATED = 0xC000021A,
  MD_NTSTATUS_WIN_STATUS_DATA_NOT_ACCEPTED = 0xC000021B,
  MD_NTSTATUS_WIN_STATUS_NO_BROWSER_SERVERS_FOUND = 0xC000021C,
  MD_NTSTATUS_WIN_STATUS_VDM_HARD_ERROR = 0xC000021D,
  MD_NTSTATUS_WIN_STATUS_DRIVER_CANCEL_TIMEOUT = 0xC000021E,
  MD_NTSTATUS_WIN_STATUS_REPLY_MESSAGE_MISMATCH = 0xC000021F,
  MD_NTSTATUS_WIN_STATUS_MAPPED_ALIGNMENT = 0xC0000220,
  MD_NTSTATUS_WIN_STATUS_IMAGE_CHECKSUM_MISMATCH = 0xC0000221,
  MD_NTSTATUS_WIN_STATUS_LOST_WRITEBEHIND_DATA = 0xC0000222,
  MD_NTSTATUS_WIN_STATUS_CLIENT_SERVER_PARAMETERS_INVALID = 0xC0000223,
  MD_NTSTATUS_WIN_STATUS_PASSWORD_MUST_CHANGE = 0xC0000224,
  MD_NTSTATUS_WIN_STATUS_NOT_FOUND = 0xC0000225,
  MD_NTSTATUS_WIN_STATUS_NOT_TINY_STREAM = 0xC0000226,
  MD_NTSTATUS_WIN_STATUS_RECOVERY_FAILURE = 0xC0000227,
  MD_NTSTATUS_WIN_STATUS_STACK_OVERFLOW_READ = 0xC0000228,
  MD_NTSTATUS_WIN_STATUS_FAIL_CHECK = 0xC0000229,
  MD_NTSTATUS_WIN_STATUS_DUPLICATE_OBJECTID = 0xC000022A,
  MD_NTSTATUS_WIN_STATUS_OBJECTID_EXISTS = 0xC000022B,
  MD_NTSTATUS_WIN_STATUS_CONVERT_TO_LARGE = 0xC000022C,
  MD_NTSTATUS_WIN_STATUS_RETRY = 0xC000022D,
  MD_NTSTATUS_WIN_STATUS_FOUND_OUT_OF_SCOPE = 0xC000022E,
  MD_NTSTATUS_WIN_STATUS_ALLOCATE_BUCKET = 0xC000022F,
  MD_NTSTATUS_WIN_STATUS_PROPSET_NOT_FOUND = 0xC0000230,
  MD_NTSTATUS_WIN_STATUS_MARSHALL_OVERFLOW = 0xC0000231,
  MD_NTSTATUS_WIN_STATUS_INVALID_VARIANT = 0xC0000232,
  MD_NTSTATUS_WIN_STATUS_DOMAIN_CONTROLLER_NOT_FOUND = 0xC0000233,
  MD_NTSTATUS_WIN_STATUS_ACCOUNT_LOCKED_OUT = 0xC0000234,
  MD_NTSTATUS_WIN_STATUS_HANDLE_NOT_CLOSABLE = 0xC0000235,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_REFUSED = 0xC0000236,
  MD_NTSTATUS_WIN_STATUS_GRACEFUL_DISCONNECT = 0xC0000237,
  MD_NTSTATUS_WIN_STATUS_ADDRESS_ALREADY_ASSOCIATED = 0xC0000238,
  MD_NTSTATUS_WIN_STATUS_ADDRESS_NOT_ASSOCIATED = 0xC0000239,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_INVALID = 0xC000023A,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_ACTIVE = 0xC000023B,
  MD_NTSTATUS_WIN_STATUS_NETWORK_UNREACHABLE = 0xC000023C,
  MD_NTSTATUS_WIN_STATUS_HOST_UNREACHABLE = 0xC000023D,
  MD_NTSTATUS_WIN_STATUS_PROTOCOL_UNREACHABLE = 0xC000023E,
  MD_NTSTATUS_WIN_STATUS_PORT_UNREACHABLE = 0xC000023F,
  MD_NTSTATUS_WIN_STATUS_REQUEST_ABORTED = 0xC0000240,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_ABORTED = 0xC0000241,
  MD_NTSTATUS_WIN_STATUS_BAD_COMPRESSION_BUFFER = 0xC0000242,
  MD_NTSTATUS_WIN_STATUS_USER_MAPPED_FILE = 0xC0000243,
  MD_NTSTATUS_WIN_STATUS_AUDIT_FAILED = 0xC0000244,
  MD_NTSTATUS_WIN_STATUS_TIMER_RESOLUTION_NOT_SET = 0xC0000245,
  MD_NTSTATUS_WIN_STATUS_CONNECTION_COUNT_LIMIT = 0xC0000246,
  MD_NTSTATUS_WIN_STATUS_LOGIN_TIME_RESTRICTION = 0xC0000247,
  MD_NTSTATUS_WIN_STATUS_LOGIN_WKSTA_RESTRICTION = 0xC0000248,
  MD_NTSTATUS_WIN_STATUS_IMAGE_MP_UP_MISMATCH = 0xC0000249,
  MD_NTSTATUS_WIN_STATUS_INSUFFICIENT_LOGON_INFO = 0xC0000250,
  MD_NTSTATUS_WIN_STATUS_BAD_DLL_ENTRYPOINT = 0xC0000251,
  MD_NTSTATUS_WIN_STATUS_BAD_SERVICE_ENTRYPOINT = 0xC0000252,
  MD_NTSTATUS_WIN_STATUS_LPC_REPLY_LOST = 0xC0000253,
  MD_NTSTATUS_WIN_STATUS_IP_ADDRESS_CONFLICT1 = 0xC0000254,
  MD_NTSTATUS_WIN_STATUS_IP_ADDRESS_CONFLICT2 = 0xC0000255,
  MD_NTSTATUS_WIN_STATUS_REGISTRY_QUOTA_LIMIT = 0xC0000256,
  MD_NTSTATUS_WIN_STATUS_PATH_NOT_COVERED = 0xC0000257,
  MD_NTSTATUS_WIN_STATUS_NO_CALLBACK_ACTIVE = 0xC0000258,
  MD_NTSTATUS_WIN_STATUS_LICENSE_QUOTA_EXCEEDED = 0xC0000259,
  MD_NTSTATUS_WIN_STATUS_PWD_TOO_SHORT = 0xC000025A,
  MD_NTSTATUS_WIN_STATUS_PWD_TOO_RECENT = 0xC000025B,
  MD_NTSTATUS_WIN_STATUS_PWD_HISTORY_CONFLICT = 0xC000025C,
  MD_NTSTATUS_WIN_STATUS_PLUGPLAY_NO_DEVICE = 0xC000025E,
  MD_NTSTATUS_WIN_STATUS_UNSUPPORTED_COMPRESSION = 0xC000025F,
  MD_NTSTATUS_WIN_STATUS_INVALID_HW_PROFILE = 0xC0000260,
  MD_NTSTATUS_WIN_STATUS_INVALID_PLUGPLAY_DEVICE_PATH = 0xC0000261,
  MD_NTSTATUS_WIN_STATUS_DRIVER_ORDINAL_NOT_FOUND = 0xC0000262,
  MD_NTSTATUS_WIN_STATUS_DRIVER_ENTRYPOINT_NOT_FOUND = 0xC0000263,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_NOT_OWNED = 0xC0000264,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_LINKS = 0xC0000265,
  MD_NTSTATUS_WIN_STATUS_QUOTA_LIST_INCONSISTENT = 0xC0000266,
  MD_NTSTATUS_WIN_STATUS_FILE_IS_OFFLINE = 0xC0000267,
  MD_NTSTATUS_WIN_STATUS_EVALUATION_EXPIRATION = 0xC0000268,
  MD_NTSTATUS_WIN_STATUS_ILLEGAL_DLL_RELOCATION = 0xC0000269,
  MD_NTSTATUS_WIN_STATUS_LICENSE_VIOLATION = 0xC000026A,
  MD_NTSTATUS_WIN_STATUS_DLL_INIT_FAILED_LOGOFF = 0xC000026B,
  MD_NTSTATUS_WIN_STATUS_DRIVER_UNABLE_TO_LOAD = 0xC000026C,
  MD_NTSTATUS_WIN_STATUS_DFS_UNAVAILABLE = 0xC000026D,
  MD_NTSTATUS_WIN_STATUS_VOLUME_DISMOUNTED = 0xC000026E,
  MD_NTSTATUS_WIN_STATUS_WX86_INTERNAL_ERROR = 0xC000026F,
  MD_NTSTATUS_WIN_STATUS_WX86_FLOAT_STACK_CHECK = 0xC0000270,
  MD_NTSTATUS_WIN_STATUS_VALIDATE_CONTINUE = 0xC0000271,
  MD_NTSTATUS_WIN_STATUS_NO_MATCH = 0xC0000272,
  MD_NTSTATUS_WIN_STATUS_NO_MORE_MATCHES = 0xC0000273,
  MD_NTSTATUS_WIN_STATUS_NOT_A_REPARSE_POINT = 0xC0000275,
  MD_NTSTATUS_WIN_STATUS_IO_REPARSE_TAG_INVALID = 0xC0000276,
  MD_NTSTATUS_WIN_STATUS_IO_REPARSE_TAG_MISMATCH = 0xC0000277,
  MD_NTSTATUS_WIN_STATUS_IO_REPARSE_DATA_INVALID = 0xC0000278,
  MD_NTSTATUS_WIN_STATUS_IO_REPARSE_TAG_NOT_HANDLED = 0xC0000279,
  MD_NTSTATUS_WIN_STATUS_PWD_TOO_LONG = 0xC000027A,
  MD_NTSTATUS_WIN_STATUS_STOWED_EXCEPTION = 0xC000027B,
  MD_NTSTATUS_WIN_STATUS_CONTEXT_STOWED_EXCEPTION = 0xC000027C,
  MD_NTSTATUS_WIN_STATUS_REPARSE_POINT_NOT_RESOLVED = 0xC0000280,
  MD_NTSTATUS_WIN_STATUS_DIRECTORY_IS_A_REPARSE_POINT = 0xC0000281,
  MD_NTSTATUS_WIN_STATUS_RANGE_LIST_CONFLICT = 0xC0000282,
  MD_NTSTATUS_WIN_STATUS_SOURCE_ELEMENT_EMPTY = 0xC0000283,
  MD_NTSTATUS_WIN_STATUS_DESTINATION_ELEMENT_FULL = 0xC0000284,
  MD_NTSTATUS_WIN_STATUS_ILLEGAL_ELEMENT_ADDRESS = 0xC0000285,
  MD_NTSTATUS_WIN_STATUS_MAGAZINE_NOT_PRESENT = 0xC0000286,
  MD_NTSTATUS_WIN_STATUS_REINITIALIZATION_NEEDED = 0xC0000287,
  MD_NTSTATUS_WIN_STATUS_ENCRYPTION_FAILED = 0xC000028A,
  MD_NTSTATUS_WIN_STATUS_DECRYPTION_FAILED = 0xC000028B,
  MD_NTSTATUS_WIN_STATUS_RANGE_NOT_FOUND = 0xC000028C,
  MD_NTSTATUS_WIN_STATUS_NO_RECOVERY_POLICY = 0xC000028D,
  MD_NTSTATUS_WIN_STATUS_NO_EFS = 0xC000028E,
  MD_NTSTATUS_WIN_STATUS_WRONG_EFS = 0xC000028F,
  MD_NTSTATUS_WIN_STATUS_NO_USER_KEYS = 0xC0000290,
  MD_NTSTATUS_WIN_STATUS_FILE_NOT_ENCRYPTED = 0xC0000291,
  MD_NTSTATUS_WIN_STATUS_NOT_EXPORT_FORMAT = 0xC0000292,
  MD_NTSTATUS_WIN_STATUS_FILE_ENCRYPTED = 0xC0000293,
  MD_NTSTATUS_WIN_STATUS_WMI_GUID_NOT_FOUND = 0xC0000295,
  MD_NTSTATUS_WIN_STATUS_WMI_INSTANCE_NOT_FOUND = 0xC0000296,
  MD_NTSTATUS_WIN_STATUS_WMI_ITEMID_NOT_FOUND = 0xC0000297,
  MD_NTSTATUS_WIN_STATUS_WMI_TRY_AGAIN = 0xC0000298,
  MD_NTSTATUS_WIN_STATUS_SHARED_POLICY = 0xC0000299,
  MD_NTSTATUS_WIN_STATUS_POLICY_OBJECT_NOT_FOUND = 0xC000029A,
  MD_NTSTATUS_WIN_STATUS_POLICY_ONLY_IN_DS = 0xC000029B,
  MD_NTSTATUS_WIN_STATUS_VOLUME_NOT_UPGRADED = 0xC000029C,
  MD_NTSTATUS_WIN_STATUS_REMOTE_STORAGE_NOT_ACTIVE = 0xC000029D,
  MD_NTSTATUS_WIN_STATUS_REMOTE_STORAGE_MEDIA_ERROR = 0xC000029E,
  MD_NTSTATUS_WIN_STATUS_NO_TRACKING_SERVICE = 0xC000029F,
  MD_NTSTATUS_WIN_STATUS_SERVER_SID_MISMATCH = 0xC00002A0,
  MD_NTSTATUS_WIN_STATUS_DS_NO_ATTRIBUTE_OR_VALUE = 0xC00002A1,
  MD_NTSTATUS_WIN_STATUS_DS_INVALID_ATTRIBUTE_SYNTAX = 0xC00002A2,
  MD_NTSTATUS_WIN_STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED = 0xC00002A3,
  MD_NTSTATUS_WIN_STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS = 0xC00002A4,
  MD_NTSTATUS_WIN_STATUS_DS_BUSY = 0xC00002A5,
  MD_NTSTATUS_WIN_STATUS_DS_UNAVAILABLE = 0xC00002A6,
  MD_NTSTATUS_WIN_STATUS_DS_NO_RIDS_ALLOCATED = 0xC00002A7,
  MD_NTSTATUS_WIN_STATUS_DS_NO_MORE_RIDS = 0xC00002A8,
  MD_NTSTATUS_WIN_STATUS_DS_INCORRECT_ROLE_OWNER = 0xC00002A9,
  MD_NTSTATUS_WIN_STATUS_DS_RIDMGR_INIT_ERROR = 0xC00002AA,
  MD_NTSTATUS_WIN_STATUS_DS_OBJ_CLASS_VIOLATION = 0xC00002AB,
  MD_NTSTATUS_WIN_STATUS_DS_CANT_ON_NON_LEAF = 0xC00002AC,
  MD_NTSTATUS_WIN_STATUS_DS_CANT_ON_RDN = 0xC00002AD,
  MD_NTSTATUS_WIN_STATUS_DS_CANT_MOD_OBJ_CLASS = 0xC00002AE,
  MD_NTSTATUS_WIN_STATUS_DS_CROSS_DOM_MOVE_FAILED = 0xC00002AF,
  MD_NTSTATUS_WIN_STATUS_DS_GC_NOT_AVAILABLE = 0xC00002B0,
  MD_NTSTATUS_WIN_STATUS_DIRECTORY_SERVICE_REQUIRED = 0xC00002B1,
  MD_NTSTATUS_WIN_STATUS_REPARSE_ATTRIBUTE_CONFLICT = 0xC00002B2,
  MD_NTSTATUS_WIN_STATUS_CANT_ENABLE_DENY_ONLY = 0xC00002B3,
  MD_NTSTATUS_WIN_STATUS_FLOAT_MULTIPLE_FAULTS = 0xC00002B4,
  MD_NTSTATUS_WIN_STATUS_FLOAT_MULTIPLE_TRAPS = 0xC00002B5,
  MD_NTSTATUS_WIN_STATUS_DEVICE_REMOVED = 0xC00002B6,
  MD_NTSTATUS_WIN_STATUS_JOURNAL_DELETE_IN_PROGRESS = 0xC00002B7,
  MD_NTSTATUS_WIN_STATUS_JOURNAL_NOT_ACTIVE = 0xC00002B8,
  MD_NTSTATUS_WIN_STATUS_NOINTERFACE = 0xC00002B9,
  MD_NTSTATUS_WIN_STATUS_DS_RIDMGR_DISABLED = 0xC00002BA,
  MD_NTSTATUS_WIN_STATUS_DS_ADMIN_LIMIT_EXCEEDED = 0xC00002C1,
  MD_NTSTATUS_WIN_STATUS_DRIVER_FAILED_SLEEP = 0xC00002C2,
  MD_NTSTATUS_WIN_STATUS_MUTUAL_AUTHENTICATION_FAILED = 0xC00002C3,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_SYSTEM_FILE = 0xC00002C4,
  MD_NTSTATUS_WIN_STATUS_DATATYPE_MISALIGNMENT_ERROR = 0xC00002C5,
  MD_NTSTATUS_WIN_STATUS_WMI_READ_ONLY = 0xC00002C6,
  MD_NTSTATUS_WIN_STATUS_WMI_SET_FAILURE = 0xC00002C7,
  MD_NTSTATUS_WIN_STATUS_COMMITMENT_MINIMUM = 0xC00002C8,
  MD_NTSTATUS_WIN_STATUS_REG_NAT_CONSUMPTION = 0xC00002C9,
  MD_NTSTATUS_WIN_STATUS_TRANSPORT_FULL = 0xC00002CA,
  MD_NTSTATUS_WIN_STATUS_DS_SAM_INIT_FAILURE = 0xC00002CB,
  MD_NTSTATUS_WIN_STATUS_ONLY_IF_CONNECTED = 0xC00002CC,
  MD_NTSTATUS_WIN_STATUS_DS_SENSITIVE_GROUP_VIOLATION = 0xC00002CD,
  MD_NTSTATUS_WIN_STATUS_PNP_RESTART_ENUMERATION = 0xC00002CE,
  MD_NTSTATUS_WIN_STATUS_JOURNAL_ENTRY_DELETED = 0xC00002CF,
  MD_NTSTATUS_WIN_STATUS_DS_CANT_MOD_PRIMARYGROUPID = 0xC00002D0,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_IMAGE_BAD_SIGNATURE = 0xC00002D1,
  MD_NTSTATUS_WIN_STATUS_PNP_REBOOT_REQUIRED = 0xC00002D2,
  MD_NTSTATUS_WIN_STATUS_POWER_STATE_INVALID = 0xC00002D3,
  MD_NTSTATUS_WIN_STATUS_DS_INVALID_GROUP_TYPE = 0xC00002D4,
  MD_NTSTATUS_WIN_STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN = 0xC00002D5,
  MD_NTSTATUS_WIN_STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN = 0xC00002D6,
  MD_NTSTATUS_WIN_STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER = 0xC00002D7,
  MD_NTSTATUS_WIN_STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER = 0xC00002D8,
  MD_NTSTATUS_WIN_STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER = 0xC00002D9,
  MD_NTSTATUS_WIN_STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER = 0xC00002DA,
  MD_NTSTATUS_WIN_STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER = 0xC00002DB,
  MD_NTSTATUS_WIN_STATUS_DS_HAVE_PRIMARY_MEMBERS = 0xC00002DC,
  MD_NTSTATUS_WIN_STATUS_WMI_NOT_SUPPORTED = 0xC00002DD,
  MD_NTSTATUS_WIN_STATUS_INSUFFICIENT_POWER = 0xC00002DE,
  MD_NTSTATUS_WIN_STATUS_SAM_NEED_BOOTKEY_PASSWORD = 0xC00002DF,
  MD_NTSTATUS_WIN_STATUS_SAM_NEED_BOOTKEY_FLOPPY = 0xC00002E0,
  MD_NTSTATUS_WIN_STATUS_DS_CANT_START = 0xC00002E1,
  MD_NTSTATUS_WIN_STATUS_DS_INIT_FAILURE = 0xC00002E2,
  MD_NTSTATUS_WIN_STATUS_SAM_INIT_FAILURE = 0xC00002E3,
  MD_NTSTATUS_WIN_STATUS_DS_GC_REQUIRED = 0xC00002E4,
  MD_NTSTATUS_WIN_STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY = 0xC00002E5,
  MD_NTSTATUS_WIN_STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS = 0xC00002E6,
  MD_NTSTATUS_WIN_STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED = 0xC00002E7,
  MD_NTSTATUS_WIN_STATUS_MULTIPLE_FAULT_VIOLATION = 0xC00002E8,
  MD_NTSTATUS_WIN_STATUS_CURRENT_DOMAIN_NOT_ALLOWED = 0xC00002E9,
  MD_NTSTATUS_WIN_STATUS_CANNOT_MAKE = 0xC00002EA,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_SHUTDOWN = 0xC00002EB,
  MD_NTSTATUS_WIN_STATUS_DS_INIT_FAILURE_CONSOLE = 0xC00002EC,
  MD_NTSTATUS_WIN_STATUS_DS_SAM_INIT_FAILURE_CONSOLE = 0xC00002ED,
  MD_NTSTATUS_WIN_STATUS_UNFINISHED_CONTEXT_DELETED = 0xC00002EE,
  MD_NTSTATUS_WIN_STATUS_NO_TGT_REPLY = 0xC00002EF,
  MD_NTSTATUS_WIN_STATUS_OBJECTID_NOT_FOUND = 0xC00002F0,
  MD_NTSTATUS_WIN_STATUS_NO_IP_ADDRESSES = 0xC00002F1,
  MD_NTSTATUS_WIN_STATUS_WRONG_CREDENTIAL_HANDLE = 0xC00002F2,
  MD_NTSTATUS_WIN_STATUS_CRYPTO_SYSTEM_INVALID = 0xC00002F3,
  MD_NTSTATUS_WIN_STATUS_MAX_REFERRALS_EXCEEDED = 0xC00002F4,
  MD_NTSTATUS_WIN_STATUS_MUST_BE_KDC = 0xC00002F5,
  MD_NTSTATUS_WIN_STATUS_STRONG_CRYPTO_NOT_SUPPORTED = 0xC00002F6,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_PRINCIPALS = 0xC00002F7,
  MD_NTSTATUS_WIN_STATUS_NO_PA_DATA = 0xC00002F8,
  MD_NTSTATUS_WIN_STATUS_PKINIT_NAME_MISMATCH = 0xC00002F9,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_LOGON_REQUIRED = 0xC00002FA,
  MD_NTSTATUS_WIN_STATUS_KDC_INVALID_REQUEST = 0xC00002FB,
  MD_NTSTATUS_WIN_STATUS_KDC_UNABLE_TO_REFER = 0xC00002FC,
  MD_NTSTATUS_WIN_STATUS_KDC_UNKNOWN_ETYPE = 0xC00002FD,
  MD_NTSTATUS_WIN_STATUS_SHUTDOWN_IN_PROGRESS = 0xC00002FE,
  MD_NTSTATUS_WIN_STATUS_SERVER_SHUTDOWN_IN_PROGRESS = 0xC00002FF,
  MD_NTSTATUS_WIN_STATUS_NOT_SUPPORTED_ON_SBS = 0xC0000300,
  MD_NTSTATUS_WIN_STATUS_WMI_GUID_DISCONNECTED = 0xC0000301,
  MD_NTSTATUS_WIN_STATUS_WMI_ALREADY_DISABLED = 0xC0000302,
  MD_NTSTATUS_WIN_STATUS_WMI_ALREADY_ENABLED = 0xC0000303,
  MD_NTSTATUS_WIN_STATUS_MFT_TOO_FRAGMENTED = 0xC0000304,
  MD_NTSTATUS_WIN_STATUS_COPY_PROTECTION_FAILURE = 0xC0000305,
  MD_NTSTATUS_WIN_STATUS_CSS_AUTHENTICATION_FAILURE = 0xC0000306,
  MD_NTSTATUS_WIN_STATUS_CSS_KEY_NOT_PRESENT = 0xC0000307,
  MD_NTSTATUS_WIN_STATUS_CSS_KEY_NOT_ESTABLISHED = 0xC0000308,
  MD_NTSTATUS_WIN_STATUS_CSS_SCRAMBLED_SECTOR = 0xC0000309,
  MD_NTSTATUS_WIN_STATUS_CSS_REGION_MISMATCH = 0xC000030A,
  MD_NTSTATUS_WIN_STATUS_CSS_RESETS_EXHAUSTED = 0xC000030B,
  MD_NTSTATUS_WIN_STATUS_PASSWORD_CHANGE_REQUIRED = 0xC000030C,
  MD_NTSTATUS_WIN_STATUS_LOST_MODE_LOGON_RESTRICTION = 0xC000030D,
  MD_NTSTATUS_WIN_STATUS_PKINIT_FAILURE = 0xC0000320,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_SUBSYSTEM_FAILURE = 0xC0000321,
  MD_NTSTATUS_WIN_STATUS_NO_KERB_KEY = 0xC0000322,
  MD_NTSTATUS_WIN_STATUS_HOST_DOWN = 0xC0000350,
  MD_NTSTATUS_WIN_STATUS_UNSUPPORTED_PREAUTH = 0xC0000351,
  MD_NTSTATUS_WIN_STATUS_EFS_ALG_BLOB_TOO_BIG = 0xC0000352,
  MD_NTSTATUS_WIN_STATUS_PORT_NOT_SET = 0xC0000353,
  MD_NTSTATUS_WIN_STATUS_DEBUGGER_INACTIVE = 0xC0000354,
  MD_NTSTATUS_WIN_STATUS_DS_VERSION_CHECK_FAILURE = 0xC0000355,
  MD_NTSTATUS_WIN_STATUS_AUDITING_DISABLED = 0xC0000356,
  MD_NTSTATUS_WIN_STATUS_PRENT4_MACHINE_ACCOUNT = 0xC0000357,
  MD_NTSTATUS_WIN_STATUS_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER = 0xC0000358,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_WIN_32 = 0xC0000359,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_WIN_64 = 0xC000035A,
  MD_NTSTATUS_WIN_STATUS_BAD_BINDINGS = 0xC000035B,
  MD_NTSTATUS_WIN_STATUS_NETWORK_SESSION_EXPIRED = 0xC000035C,
  MD_NTSTATUS_WIN_STATUS_APPHELP_BLOCK = 0xC000035D,
  MD_NTSTATUS_WIN_STATUS_ALL_SIDS_FILTERED = 0xC000035E,
  MD_NTSTATUS_WIN_STATUS_NOT_SAFE_MODE_DRIVER = 0xC000035F,
  MD_NTSTATUS_WIN_STATUS_ACCESS_DISABLED_BY_POLICY_DEFAULT = 0xC0000361,
  MD_NTSTATUS_WIN_STATUS_ACCESS_DISABLED_BY_POLICY_PATH = 0xC0000362,
  MD_NTSTATUS_WIN_STATUS_ACCESS_DISABLED_BY_POLICY_PUBLISHER = 0xC0000363,
  MD_NTSTATUS_WIN_STATUS_ACCESS_DISABLED_BY_POLICY_OTHER = 0xC0000364,
  MD_NTSTATUS_WIN_STATUS_FAILED_DRIVER_ENTRY = 0xC0000365,
  MD_NTSTATUS_WIN_STATUS_DEVICE_ENUMERATION_ERROR = 0xC0000366,
  MD_NTSTATUS_WIN_STATUS_MOUNT_POINT_NOT_RESOLVED = 0xC0000368,
  MD_NTSTATUS_WIN_STATUS_INVALID_DEVICE_OBJECT_PARAMETER = 0xC0000369,
  MD_NTSTATUS_WIN_STATUS_MCA_OCCURED = 0xC000036A,
  MD_NTSTATUS_WIN_STATUS_DRIVER_BLOCKED_CRITICAL = 0xC000036B,
  MD_NTSTATUS_WIN_STATUS_DRIVER_BLOCKED = 0xC000036C,
  MD_NTSTATUS_WIN_STATUS_DRIVER_DATABASE_ERROR = 0xC000036D,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_HIVE_TOO_LARGE = 0xC000036E,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMPORT_OF_NON_DLL = 0xC000036F,
  MD_NTSTATUS_WIN_STATUS_NO_SECRETS = 0xC0000371,
  MD_NTSTATUS_WIN_STATUS_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY = 0xC0000372,
  MD_NTSTATUS_WIN_STATUS_FAILED_STACK_SWITCH = 0xC0000373,
  MD_NTSTATUS_WIN_STATUS_HEAP_CORRUPTION = 0xC0000374,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_WRONG_PIN = 0xC0000380,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_CARD_BLOCKED = 0xC0000381,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_CARD_NOT_AUTHENTICATED = 0xC0000382,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_NO_CARD = 0xC0000383,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_NO_KEY_CONTAINER = 0xC0000384,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_NO_CERTIFICATE = 0xC0000385,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_NO_KEYSET = 0xC0000386,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_IO_ERROR = 0xC0000387,
  MD_NTSTATUS_WIN_STATUS_DOWNGRADE_DETECTED = 0xC0000388,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_CERT_REVOKED = 0xC0000389,
  MD_NTSTATUS_WIN_STATUS_ISSUING_CA_UNTRUSTED = 0xC000038A,
  MD_NTSTATUS_WIN_STATUS_REVOCATION_OFFLINE_C = 0xC000038B,
  MD_NTSTATUS_WIN_STATUS_PKINIT_CLIENT_FAILURE = 0xC000038C,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_CERT_EXPIRED = 0xC000038D,
  MD_NTSTATUS_WIN_STATUS_DRIVER_FAILED_PRIOR_UNLOAD = 0xC000038E,
  MD_NTSTATUS_WIN_STATUS_SMARTCARD_SILENT_CONTEXT = 0xC000038F,
  MD_NTSTATUS_WIN_STATUS_PER_USER_TRUST_QUOTA_EXCEEDED = 0xC0000401,
  MD_NTSTATUS_WIN_STATUS_ALL_USER_TRUST_QUOTA_EXCEEDED = 0xC0000402,
  MD_NTSTATUS_WIN_STATUS_USER_DELETE_TRUST_QUOTA_EXCEEDED = 0xC0000403,
  MD_NTSTATUS_WIN_STATUS_DS_NAME_NOT_UNIQUE = 0xC0000404,
  MD_NTSTATUS_WIN_STATUS_DS_DUPLICATE_ID_FOUND = 0xC0000405,
  MD_NTSTATUS_WIN_STATUS_DS_GROUP_CONVERSION_ERROR = 0xC0000406,
  MD_NTSTATUS_WIN_STATUS_VOLSNAP_PREPARE_HIBERNATE = 0xC0000407,
  MD_NTSTATUS_WIN_STATUS_USER2USER_REQUIRED = 0xC0000408,
  MD_NTSTATUS_WIN_STATUS_STACK_BUFFER_OVERRUN = 0xC0000409,
  MD_NTSTATUS_WIN_STATUS_NO_S4U_PROT_SUPPORT = 0xC000040A,
  MD_NTSTATUS_WIN_STATUS_CROSSREALM_DELEGATION_FAILURE = 0xC000040B,
  MD_NTSTATUS_WIN_STATUS_REVOCATION_OFFLINE_KDC = 0xC000040C,
  MD_NTSTATUS_WIN_STATUS_ISSUING_CA_UNTRUSTED_KDC = 0xC000040D,
  MD_NTSTATUS_WIN_STATUS_KDC_CERT_EXPIRED = 0xC000040E,
  MD_NTSTATUS_WIN_STATUS_KDC_CERT_REVOKED = 0xC000040F,
  MD_NTSTATUS_WIN_STATUS_PARAMETER_QUOTA_EXCEEDED = 0xC0000410,
  MD_NTSTATUS_WIN_STATUS_HIBERNATION_FAILURE = 0xC0000411,
  MD_NTSTATUS_WIN_STATUS_DELAY_LOAD_FAILED = 0xC0000412,
  MD_NTSTATUS_WIN_STATUS_AUTHENTICATION_FIREWALL_FAILED = 0xC0000413,
  MD_NTSTATUS_WIN_STATUS_VDM_DISALLOWED = 0xC0000414,
  MD_NTSTATUS_WIN_STATUS_HUNG_DISPLAY_DRIVER_THREAD = 0xC0000415,
  MD_NTSTATUS_WIN_STATUS_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE = 0xC0000416,
  MD_NTSTATUS_WIN_STATUS_INVALID_CRUNTIME_PARAMETER = 0xC0000417,
  MD_NTSTATUS_WIN_STATUS_NTLM_BLOCKED = 0xC0000418,
  MD_NTSTATUS_WIN_STATUS_DS_SRC_SID_EXISTS_IN_FOREST = 0xC0000419,
  MD_NTSTATUS_WIN_STATUS_DS_DOMAIN_NAME_EXISTS_IN_FOREST = 0xC000041A,
  MD_NTSTATUS_WIN_STATUS_DS_FLAT_NAME_EXISTS_IN_FOREST = 0xC000041B,
  MD_NTSTATUS_WIN_STATUS_INVALID_USER_PRINCIPAL_NAME = 0xC000041C,
  MD_NTSTATUS_WIN_STATUS_FATAL_USER_CALLBACK_EXCEPTION = 0xC000041D,
  MD_NTSTATUS_WIN_STATUS_ASSERTION_FAILURE = 0xC0000420,
  MD_NTSTATUS_WIN_STATUS_VERIFIER_STOP = 0xC0000421,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_POP_STACK = 0xC0000423,
  MD_NTSTATUS_WIN_STATUS_INCOMPATIBLE_DRIVER_BLOCKED = 0xC0000424,
  MD_NTSTATUS_WIN_STATUS_HIVE_UNLOADED = 0xC0000425,
  MD_NTSTATUS_WIN_STATUS_COMPRESSION_DISABLED = 0xC0000426,
  MD_NTSTATUS_WIN_STATUS_FILE_SYSTEM_LIMITATION = 0xC0000427,
  MD_NTSTATUS_WIN_STATUS_INVALID_IMAGE_HASH = 0xC0000428,
  MD_NTSTATUS_WIN_STATUS_NOT_CAPABLE = 0xC0000429,
  MD_NTSTATUS_WIN_STATUS_REQUEST_OUT_OF_SEQUENCE = 0xC000042A,
  MD_NTSTATUS_WIN_STATUS_IMPLEMENTATION_LIMIT = 0xC000042B,
  MD_NTSTATUS_WIN_STATUS_ELEVATION_REQUIRED = 0xC000042C,
  MD_NTSTATUS_WIN_STATUS_NO_SECURITY_CONTEXT = 0xC000042D,
  MD_NTSTATUS_WIN_STATUS_PKU2U_CERT_FAILURE = 0xC000042F,
  MD_NTSTATUS_WIN_STATUS_BEYOND_VDL = 0xC0000432,
  MD_NTSTATUS_WIN_STATUS_ENCOUNTERED_WRITE_IN_PROGRESS = 0xC0000433,
  MD_NTSTATUS_WIN_STATUS_PTE_CHANGED = 0xC0000434,
  MD_NTSTATUS_WIN_STATUS_PURGE_FAILED = 0xC0000435,
  MD_NTSTATUS_WIN_STATUS_CRED_REQUIRES_CONFIRMATION = 0xC0000440,
  MD_NTSTATUS_WIN_STATUS_CS_ENCRYPTION_INVALID_SERVER_RESPONSE = 0xC0000441,
  MD_NTSTATUS_WIN_STATUS_CS_ENCRYPTION_UNSUPPORTED_SERVER = 0xC0000442,
  MD_NTSTATUS_WIN_STATUS_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE = 0xC0000443,
  MD_NTSTATUS_WIN_STATUS_CS_ENCRYPTION_NEW_ENCRYPTED_FILE = 0xC0000444,
  MD_NTSTATUS_WIN_STATUS_CS_ENCRYPTION_FILE_NOT_CSE = 0xC0000445,
  MD_NTSTATUS_WIN_STATUS_INVALID_LABEL = 0xC0000446,
  MD_NTSTATUS_WIN_STATUS_DRIVER_PROCESS_TERMINATED = 0xC0000450,
  MD_NTSTATUS_WIN_STATUS_AMBIGUOUS_SYSTEM_DEVICE = 0xC0000451,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_DEVICE_NOT_FOUND = 0xC0000452,
  MD_NTSTATUS_WIN_STATUS_RESTART_BOOT_APPLICATION = 0xC0000453,
  MD_NTSTATUS_WIN_STATUS_INSUFFICIENT_NVRAM_RESOURCES = 0xC0000454,
  MD_NTSTATUS_WIN_STATUS_INVALID_SESSION = 0xC0000455,
  MD_NTSTATUS_WIN_STATUS_THREAD_ALREADY_IN_SESSION = 0xC0000456,
  MD_NTSTATUS_WIN_STATUS_THREAD_NOT_IN_SESSION = 0xC0000457,
  MD_NTSTATUS_WIN_STATUS_INVALID_WEIGHT = 0xC0000458,
  MD_NTSTATUS_WIN_STATUS_REQUEST_PAUSED = 0xC0000459,
  MD_NTSTATUS_WIN_STATUS_NO_RANGES_PROCESSED = 0xC0000460,
  MD_NTSTATUS_WIN_STATUS_DISK_RESOURCES_EXHAUSTED = 0xC0000461,
  MD_NTSTATUS_WIN_STATUS_NEEDS_REMEDIATION = 0xC0000462,
  MD_NTSTATUS_WIN_STATUS_DEVICE_FEATURE_NOT_SUPPORTED = 0xC0000463,
  MD_NTSTATUS_WIN_STATUS_DEVICE_UNREACHABLE = 0xC0000464,
  MD_NTSTATUS_WIN_STATUS_INVALID_TOKEN = 0xC0000465,
  MD_NTSTATUS_WIN_STATUS_SERVER_UNAVAILABLE = 0xC0000466,
  MD_NTSTATUS_WIN_STATUS_FILE_NOT_AVAILABLE = 0xC0000467,
  MD_NTSTATUS_WIN_STATUS_DEVICE_INSUFFICIENT_RESOURCES = 0xC0000468,
  MD_NTSTATUS_WIN_STATUS_PACKAGE_UPDATING = 0xC0000469,
  MD_NTSTATUS_WIN_STATUS_NOT_READ_FROM_COPY = 0xC000046A,
  MD_NTSTATUS_WIN_STATUS_FT_WRITE_FAILURE = 0xC000046B,
  MD_NTSTATUS_WIN_STATUS_FT_DI_SCAN_REQUIRED = 0xC000046C,
  MD_NTSTATUS_WIN_STATUS_OBJECT_NOT_EXTERNALLY_BACKED = 0xC000046D,
  MD_NTSTATUS_WIN_STATUS_EXTERNAL_BACKING_PROVIDER_UNKNOWN = 0xC000046E,
  MD_NTSTATUS_WIN_STATUS_COMPRESSION_NOT_BENEFICIAL = 0xC000046F,
  MD_NTSTATUS_WIN_STATUS_DATA_CHECKSUM_ERROR = 0xC0000470,
  MD_NTSTATUS_WIN_STATUS_INTERMIXED_KERNEL_EA_OPERATION = 0xC0000471,
  MD_NTSTATUS_WIN_STATUS_TRIM_READ_ZERO_NOT_SUPPORTED = 0xC0000472,
  MD_NTSTATUS_WIN_STATUS_TOO_MANY_SEGMENT_DESCRIPTORS = 0xC0000473,
  MD_NTSTATUS_WIN_STATUS_INVALID_OFFSET_ALIGNMENT = 0xC0000474,
  MD_NTSTATUS_WIN_STATUS_INVALID_FIELD_IN_PARAMETER_LIST = 0xC0000475,
  MD_NTSTATUS_WIN_STATUS_OPERATION_IN_PROGRESS = 0xC0000476,
  MD_NTSTATUS_WIN_STATUS_INVALID_INITIATOR_TARGET_PATH = 0xC0000477,
  MD_NTSTATUS_WIN_STATUS_SCRUB_DATA_DISABLED = 0xC0000478,
  MD_NTSTATUS_WIN_STATUS_NOT_REDUNDANT_STORAGE = 0xC0000479,
  MD_NTSTATUS_WIN_STATUS_RESIDENT_FILE_NOT_SUPPORTED = 0xC000047A,
  MD_NTSTATUS_WIN_STATUS_COMPRESSED_FILE_NOT_SUPPORTED = 0xC000047B,
  MD_NTSTATUS_WIN_STATUS_DIRECTORY_NOT_SUPPORTED = 0xC000047C,
  MD_NTSTATUS_WIN_STATUS_IO_OPERATION_TIMEOUT = 0xC000047D,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_NEEDS_REMEDIATION = 0xC000047E,
  MD_NTSTATUS_WIN_STATUS_APPX_INTEGRITY_FAILURE_CLR_NGEN = 0xC000047F,
  MD_NTSTATUS_WIN_STATUS_SHARE_UNAVAILABLE = 0xC0000480,
  MD_NTSTATUS_WIN_STATUS_APISET_NOT_HOSTED = 0xC0000481,
  MD_NTSTATUS_WIN_STATUS_APISET_NOT_PRESENT = 0xC0000482,
  MD_NTSTATUS_WIN_STATUS_DEVICE_HARDWARE_ERROR = 0xC0000483,
  MD_NTSTATUS_WIN_STATUS_FIRMWARE_SLOT_INVALID = 0xC0000484,
  MD_NTSTATUS_WIN_STATUS_FIRMWARE_IMAGE_INVALID = 0xC0000485,
  MD_NTSTATUS_WIN_STATUS_STORAGE_TOPOLOGY_ID_MISMATCH = 0xC0000486,
  MD_NTSTATUS_WIN_STATUS_WIM_NOT_BOOTABLE = 0xC0000487,
  MD_NTSTATUS_WIN_STATUS_BLOCKED_BY_PARENTAL_CONTROLS = 0xC0000488,
  MD_NTSTATUS_WIN_STATUS_NEEDS_REGISTRATION = 0xC0000489,
  MD_NTSTATUS_WIN_STATUS_QUOTA_ACTIVITY = 0xC000048A,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_INVOKE_INLINE = 0xC000048B,
  MD_NTSTATUS_WIN_STATUS_BLOCK_TOO_MANY_REFERENCES = 0xC000048C,
  MD_NTSTATUS_WIN_STATUS_MARKED_TO_DISALLOW_WRITES = 0xC000048D,
  MD_NTSTATUS_WIN_STATUS_NETWORK_ACCESS_DENIED_EDP = 0xC000048E,
  MD_NTSTATUS_WIN_STATUS_ENCLAVE_FAILURE = 0xC000048F,
  MD_NTSTATUS_WIN_STATUS_PNP_NO_COMPAT_DRIVERS = 0xC0000490,
  MD_NTSTATUS_WIN_STATUS_PNP_DRIVER_PACKAGE_NOT_FOUND = 0xC0000491,
  MD_NTSTATUS_WIN_STATUS_PNP_DRIVER_CONFIGURATION_NOT_FOUND = 0xC0000492,
  MD_NTSTATUS_WIN_STATUS_PNP_DRIVER_CONFIGURATION_INCOMPLETE = 0xC0000493,
  MD_NTSTATUS_WIN_STATUS_PNP_FUNCTION_DRIVER_REQUIRED = 0xC0000494,
  MD_NTSTATUS_WIN_STATUS_PNP_DEVICE_CONFIGURATION_PENDING = 0xC0000495,
  MD_NTSTATUS_WIN_STATUS_DEVICE_HINT_NAME_BUFFER_TOO_SMALL = 0xC0000496,
  MD_NTSTATUS_WIN_STATUS_PACKAGE_NOT_AVAILABLE = 0xC0000497,
  MD_NTSTATUS_WIN_STATUS_DEVICE_IN_MAINTENANCE = 0xC0000499,
  MD_NTSTATUS_WIN_STATUS_NOT_SUPPORTED_ON_DAX = 0xC000049A,
  MD_NTSTATUS_WIN_STATUS_FREE_SPACE_TOO_FRAGMENTED = 0xC000049B,
  MD_NTSTATUS_WIN_STATUS_DAX_MAPPING_EXISTS = 0xC000049C,
  MD_NTSTATUS_WIN_STATUS_CHILD_PROCESS_BLOCKED = 0xC000049D,
  MD_NTSTATUS_WIN_STATUS_STORAGE_LOST_DATA_PERSISTENCE = 0xC000049E,
  MD_NTSTATUS_WIN_STATUS_VRF_CFG_AND_IO_ENABLED = 0xC000049F,
  MD_NTSTATUS_WIN_STATUS_PARTITION_TERMINATING = 0xC00004A0,
  MD_NTSTATUS_WIN_STATUS_EXTERNAL_SYSKEY_NOT_SUPPORTED = 0xC00004A1,
  MD_NTSTATUS_WIN_STATUS_ENCLAVE_VIOLATION = 0xC00004A2,
  MD_NTSTATUS_WIN_STATUS_FILE_PROTECTED_UNDER_DPL = 0xC00004A3,
  MD_NTSTATUS_WIN_STATUS_VOLUME_NOT_CLUSTER_ALIGNED = 0xC00004A4,
  MD_NTSTATUS_WIN_STATUS_NO_PHYSICALLY_ALIGNED_FREE_SPACE_FOUND = 0xC00004A5,
  MD_NTSTATUS_WIN_STATUS_APPX_FILE_NOT_ENCRYPTED = 0xC00004A6,
  MD_NTSTATUS_WIN_STATUS_RWRAW_ENCRYPTED_FILE_NOT_ENCRYPTED = 0xC00004A7,
  MD_NTSTATUS_WIN_STATUS_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILEOFFSET = 0xC00004A8,
  MD_NTSTATUS_WIN_STATUS_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILERANGE = 0xC00004A9,
  MD_NTSTATUS_WIN_STATUS_RWRAW_ENCRYPTED_INVALID_EDATAINFO_PARAMETER = 0xC00004AA,
  MD_NTSTATUS_WIN_STATUS_FT_READ_FAILURE = 0xC00004AB,
  MD_NTSTATUS_WIN_STATUS_PATCH_CONFLICT = 0xC00004AC,
  MD_NTSTATUS_WIN_STATUS_STORAGE_RESERVE_ID_INVALID = 0xC00004AD,
  MD_NTSTATUS_WIN_STATUS_STORAGE_RESERVE_DOES_NOT_EXIST = 0xC00004AE,
  MD_NTSTATUS_WIN_STATUS_STORAGE_RESERVE_ALREADY_EXISTS = 0xC00004AF,
  MD_NTSTATUS_WIN_STATUS_STORAGE_RESERVE_NOT_EMPTY = 0xC00004B0,
  MD_NTSTATUS_WIN_STATUS_NOT_A_DAX_VOLUME = 0xC00004B1,
  MD_NTSTATUS_WIN_STATUS_NOT_DAX_MAPPABLE = 0xC00004B2,
  MD_NTSTATUS_WIN_STATUS_CASE_DIFFERING_NAMES_IN_DIR = 0xC00004B3,
  MD_NTSTATUS_WIN_STATUS_FILE_NOT_SUPPORTED = 0xC00004B4,
  MD_NTSTATUS_WIN_STATUS_NOT_SUPPORTED_WITH_BTT = 0xC00004B5,
  MD_NTSTATUS_WIN_STATUS_ENCRYPTION_DISABLED = 0xC00004B6,
  MD_NTSTATUS_WIN_STATUS_ENCRYPTING_METADATA_DISALLOWED = 0xC00004B7,
  MD_NTSTATUS_WIN_STATUS_CANT_CLEAR_ENCRYPTION_FLAG = 0xC00004B8,
  MD_NTSTATUS_WIN_STATUS_UNSATISFIED_DEPENDENCIES = 0xC00004B9,
  MD_NTSTATUS_WIN_STATUS_CASE_SENSITIVE_PATH = 0xC00004BA,
  MD_NTSTATUS_WIN_STATUS_HAS_SYSTEM_CRITICAL_FILES = 0xC00004BD,
  MD_NTSTATUS_WIN_STATUS_INVALID_TASK_NAME = 0xC0000500,
  MD_NTSTATUS_WIN_STATUS_INVALID_TASK_INDEX = 0xC0000501,
  MD_NTSTATUS_WIN_STATUS_THREAD_ALREADY_IN_TASK = 0xC0000502,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_BYPASS = 0xC0000503,
  MD_NTSTATUS_WIN_STATUS_UNDEFINED_SCOPE = 0xC0000504,
  MD_NTSTATUS_WIN_STATUS_INVALID_CAP = 0xC0000505,
  MD_NTSTATUS_WIN_STATUS_NOT_GUI_PROCESS = 0xC0000506,
  MD_NTSTATUS_WIN_STATUS_DEVICE_HUNG = 0xC0000507,
  MD_NTSTATUS_WIN_STATUS_CONTAINER_ASSIGNED = 0xC0000508,
  MD_NTSTATUS_WIN_STATUS_JOB_NO_CONTAINER = 0xC0000509,
  MD_NTSTATUS_WIN_STATUS_DEVICE_UNRESPONSIVE = 0xC000050A,
  MD_NTSTATUS_WIN_STATUS_REPARSE_POINT_ENCOUNTERED = 0xC000050B,
  MD_NTSTATUS_WIN_STATUS_ATTRIBUTE_NOT_PRESENT = 0xC000050C,
  MD_NTSTATUS_WIN_STATUS_NOT_A_TIERED_VOLUME = 0xC000050D,
  MD_NTSTATUS_WIN_STATUS_ALREADY_HAS_STREAM_ID = 0xC000050E,
  MD_NTSTATUS_WIN_STATUS_JOB_NOT_EMPTY = 0xC000050F,
  MD_NTSTATUS_WIN_STATUS_ALREADY_INITIALIZED = 0xC0000510,
  MD_NTSTATUS_WIN_STATUS_ENCLAVE_NOT_TERMINATED = 0xC0000511,
  MD_NTSTATUS_WIN_STATUS_ENCLAVE_IS_TERMINATING = 0xC0000512,
  MD_NTSTATUS_WIN_STATUS_SMB1_NOT_AVAILABLE = 0xC0000513,
  MD_NTSTATUS_WIN_STATUS_SMR_GARBAGE_COLLECTION_REQUIRED = 0xC0000514,
  MD_NTSTATUS_WIN_STATUS_INTERRUPTED = 0xC0000515,
  MD_NTSTATUS_WIN_STATUS_THREAD_NOT_RUNNING = 0xC0000516,
  MD_NTSTATUS_WIN_STATUS_FAIL_FAST_EXCEPTION = 0xC0000602,
  MD_NTSTATUS_WIN_STATUS_IMAGE_CERT_REVOKED = 0xC0000603,
  MD_NTSTATUS_WIN_STATUS_DYNAMIC_CODE_BLOCKED = 0xC0000604,
  MD_NTSTATUS_WIN_STATUS_IMAGE_CERT_EXPIRED = 0xC0000605,
  MD_NTSTATUS_WIN_STATUS_STRICT_CFG_VIOLATION = 0xC0000606,
  MD_NTSTATUS_WIN_STATUS_SET_CONTEXT_DENIED = 0xC000060A,
  MD_NTSTATUS_WIN_STATUS_CROSS_PARTITION_VIOLATION = 0xC000060B,
  MD_NTSTATUS_WIN_STATUS_PORT_CLOSED = 0xC0000700,
  MD_NTSTATUS_WIN_STATUS_MESSAGE_LOST = 0xC0000701,
  MD_NTSTATUS_WIN_STATUS_INVALID_MESSAGE = 0xC0000702,
  MD_NTSTATUS_WIN_STATUS_REQUEST_CANCELED = 0xC0000703,
  MD_NTSTATUS_WIN_STATUS_RECURSIVE_DISPATCH = 0xC0000704,
  MD_NTSTATUS_WIN_STATUS_LPC_RECEIVE_BUFFER_EXPECTED = 0xC0000705,
  MD_NTSTATUS_WIN_STATUS_LPC_INVALID_CONNECTION_USAGE = 0xC0000706,
  MD_NTSTATUS_WIN_STATUS_LPC_REQUESTS_NOT_ALLOWED = 0xC0000707,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_IN_USE = 0xC0000708,
  MD_NTSTATUS_WIN_STATUS_HARDWARE_MEMORY_ERROR = 0xC0000709,
  MD_NTSTATUS_WIN_STATUS_THREADPOOL_HANDLE_EXCEPTION = 0xC000070A,
  MD_NTSTATUS_WIN_STATUS_THREADPOOL_SET_EVENT_ON_COMPLETION_FAILED = 0xC000070B,
  MD_NTSTATUS_WIN_STATUS_THREADPOOL_RELEASE_SEMAPHORE_ON_COMPLETION_FAILED = 0xC000070C,
  MD_NTSTATUS_WIN_STATUS_THREADPOOL_RELEASE_MUTEX_ON_COMPLETION_FAILED = 0xC000070D,
  MD_NTSTATUS_WIN_STATUS_THREADPOOL_FREE_LIBRARY_ON_COMPLETION_FAILED = 0xC000070E,
  MD_NTSTATUS_WIN_STATUS_THREADPOOL_RELEASED_DURING_OPERATION = 0xC000070F,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_WHILE_IMPERSONATING = 0xC0000710,
  MD_NTSTATUS_WIN_STATUS_APC_RETURNED_WHILE_IMPERSONATING = 0xC0000711,
  MD_NTSTATUS_WIN_STATUS_PROCESS_IS_PROTECTED = 0xC0000712,
  MD_NTSTATUS_WIN_STATUS_MCA_EXCEPTION = 0xC0000713,
  MD_NTSTATUS_WIN_STATUS_CERTIFICATE_MAPPING_NOT_UNIQUE = 0xC0000714,
  MD_NTSTATUS_WIN_STATUS_SYMLINK_CLASS_DISABLED = 0xC0000715,
  MD_NTSTATUS_WIN_STATUS_INVALID_IDN_NORMALIZATION = 0xC0000716,
  MD_NTSTATUS_WIN_STATUS_NO_UNICODE_TRANSLATION = 0xC0000717,
  MD_NTSTATUS_WIN_STATUS_ALREADY_REGISTERED = 0xC0000718,
  MD_NTSTATUS_WIN_STATUS_CONTEXT_MISMATCH = 0xC0000719,
  MD_NTSTATUS_WIN_STATUS_PORT_ALREADY_HAS_COMPLETION_LIST = 0xC000071A,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_THREAD_PRIORITY = 0xC000071B,
  MD_NTSTATUS_WIN_STATUS_INVALID_THREAD = 0xC000071C,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_TRANSACTION = 0xC000071D,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_LDR_LOCK = 0xC000071E,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_LANG = 0xC000071F,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_PRI_BACK = 0xC0000720,
  MD_NTSTATUS_WIN_STATUS_CALLBACK_RETURNED_THREAD_AFFINITY = 0xC0000721,
  MD_NTSTATUS_WIN_STATUS_LPC_HANDLE_COUNT_EXCEEDED = 0xC0000722,
  MD_NTSTATUS_WIN_STATUS_EXECUTABLE_MEMORY_WRITE = 0xC0000723,
  MD_NTSTATUS_WIN_STATUS_KERNEL_EXECUTABLE_MEMORY_WRITE = 0xC0000724,
  MD_NTSTATUS_WIN_STATUS_ATTACHED_EXECUTABLE_MEMORY_WRITE = 0xC0000725,
  MD_NTSTATUS_WIN_STATUS_TRIGGERED_EXECUTABLE_MEMORY_WRITE = 0xC0000726,
  MD_NTSTATUS_WIN_STATUS_DISK_REPAIR_DISABLED = 0xC0000800,
  MD_NTSTATUS_WIN_STATUS_DS_DOMAIN_RENAME_IN_PROGRESS = 0xC0000801,
  MD_NTSTATUS_WIN_STATUS_DISK_QUOTA_EXCEEDED = 0xC0000802,
  MD_NTSTATUS_WIN_STATUS_CONTENT_BLOCKED = 0xC0000804,
  MD_NTSTATUS_WIN_STATUS_BAD_CLUSTERS = 0xC0000805,
  MD_NTSTATUS_WIN_STATUS_VOLUME_DIRTY = 0xC0000806,
  MD_NTSTATUS_WIN_STATUS_DISK_REPAIR_UNSUCCESSFUL = 0xC0000808,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_LOG_OVERFULL = 0xC0000809,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_LOG_CORRUPTED = 0xC000080A,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_LOG_UNAVAILABLE = 0xC000080B,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_LOG_DELETED_FULL = 0xC000080C,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_LOG_CLEARED = 0xC000080D,
  MD_NTSTATUS_WIN_STATUS_ORPHAN_NAME_EXHAUSTED = 0xC000080E,
  MD_NTSTATUS_WIN_STATUS_PROACTIVE_SCAN_IN_PROGRESS = 0xC000080F,
  MD_NTSTATUS_WIN_STATUS_ENCRYPTED_IO_NOT_POSSIBLE = 0xC0000810,
  MD_NTSTATUS_WIN_STATUS_CORRUPT_LOG_UPLEVEL_RECORDS = 0xC0000811,
  MD_NTSTATUS_WIN_STATUS_FILE_CHECKED_OUT = 0xC0000901,
  MD_NTSTATUS_WIN_STATUS_CHECKOUT_REQUIRED = 0xC0000902,
  MD_NTSTATUS_WIN_STATUS_BAD_FILE_TYPE = 0xC0000903,
  MD_NTSTATUS_WIN_STATUS_FILE_TOO_LARGE = 0xC0000904,
  MD_NTSTATUS_WIN_STATUS_FORMS_AUTH_REQUIRED = 0xC0000905,
  MD_NTSTATUS_WIN_STATUS_VIRUS_INFECTED = 0xC0000906,
  MD_NTSTATUS_WIN_STATUS_VIRUS_DELETED = 0xC0000907,
  MD_NTSTATUS_WIN_STATUS_BAD_MCFG_TABLE = 0xC0000908,
  MD_NTSTATUS_WIN_STATUS_CANNOT_BREAK_OPLOCK = 0xC0000909,
  MD_NTSTATUS_WIN_STATUS_BAD_KEY = 0xC000090A,
  MD_NTSTATUS_WIN_STATUS_BAD_DATA = 0xC000090B,
  MD_NTSTATUS_WIN_STATUS_NO_KEY = 0xC000090C,
  MD_NTSTATUS_WIN_STATUS_FILE_HANDLE_REVOKED = 0xC0000910,
  MD_NTSTATUS_WIN_STATUS_WOW_ASSERTION = 0xC0009898,
  MD_NTSTATUS_WIN_STATUS_INVALID_SIGNATURE = 0xC000A000,
  MD_NTSTATUS_WIN_STATUS_HMAC_NOT_SUPPORTED = 0xC000A001,
  MD_NTSTATUS_WIN_STATUS_AUTH_TAG_MISMATCH = 0xC000A002,
  MD_NTSTATUS_WIN_STATUS_INVALID_STATE_TRANSITION = 0xC000A003,
  MD_NTSTATUS_WIN_STATUS_INVALID_KERNEL_INFO_VERSION = 0xC000A004,
  MD_NTSTATUS_WIN_STATUS_INVALID_PEP_INFO_VERSION = 0xC000A005,
  MD_NTSTATUS_WIN_STATUS_HANDLE_REVOKED = 0xC000A006,
  MD_NTSTATUS_WIN_STATUS_EOF_ON_GHOSTED_RANGE = 0xC000A007,
  MD_NTSTATUS_WIN_STATUS_CC_NEEDS_CALLBACK_SECTION_DRAIN = 0xC000A008,
  MD_NTSTATUS_WIN_STATUS_IPSEC_QUEUE_OVERFLOW = 0xC000A010,
  MD_NTSTATUS_WIN_STATUS_ND_QUEUE_OVERFLOW = 0xC000A011,
  MD_NTSTATUS_WIN_STATUS_HOPLIMIT_EXCEEDED = 0xC000A012,
  MD_NTSTATUS_WIN_STATUS_PROTOCOL_NOT_SUPPORTED = 0xC000A013,
  MD_NTSTATUS_WIN_STATUS_FASTPATH_REJECTED = 0xC000A014,
  MD_NTSTATUS_WIN_STATUS_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED = 0xC000A080,
  MD_NTSTATUS_WIN_STATUS_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR = 0xC000A081,
  MD_NTSTATUS_WIN_STATUS_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR = 0xC000A082,
  MD_NTSTATUS_WIN_STATUS_XML_PARSE_ERROR = 0xC000A083,
  MD_NTSTATUS_WIN_STATUS_XMLDSIG_ERROR = 0xC000A084,
  MD_NTSTATUS_WIN_STATUS_WRONG_COMPARTMENT = 0xC000A085,
  MD_NTSTATUS_WIN_STATUS_AUTHIP_FAILURE = 0xC000A086,
  MD_NTSTATUS_WIN_STATUS_DS_OID_MAPPED_GROUP_CANT_HAVE_MEMBERS = 0xC000A087,
  MD_NTSTATUS_WIN_STATUS_DS_OID_NOT_FOUND = 0xC000A088,
  MD_NTSTATUS_WIN_STATUS_INCORRECT_ACCOUNT_TYPE = 0xC000A089,
  MD_NTSTATUS_WIN_STATUS_HASH_NOT_SUPPORTED = 0xC000A100,
  MD_NTSTATUS_WIN_STATUS_HASH_NOT_PRESENT = 0xC000A101,
  MD_NTSTATUS_WIN_STATUS_SECONDARY_IC_PROVIDER_NOT_REGISTERED = 0xC000A121,
  MD_NTSTATUS_WIN_STATUS_GPIO_CLIENT_INFORMATION_INVALID = 0xC000A122,
  MD_NTSTATUS_WIN_STATUS_GPIO_VERSION_NOT_SUPPORTED = 0xC000A123,
  MD_NTSTATUS_WIN_STATUS_GPIO_INVALID_REGISTRATION_PACKET = 0xC000A124,
  MD_NTSTATUS_WIN_STATUS_GPIO_OPERATION_DENIED = 0xC000A125,
  MD_NTSTATUS_WIN_STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE = 0xC000A126,
  MD_NTSTATUS_WIN_STATUS_CANNOT_SWITCH_RUNLEVEL = 0xC000A141,
  MD_NTSTATUS_WIN_STATUS_INVALID_RUNLEVEL_SETTING = 0xC000A142,
  MD_NTSTATUS_WIN_STATUS_RUNLEVEL_SWITCH_TIMEOUT = 0xC000A143,
  MD_NTSTATUS_WIN_STATUS_RUNLEVEL_SWITCH_AGENT_TIMEOUT = 0xC000A145,
  MD_NTSTATUS_WIN_STATUS_RUNLEVEL_SWITCH_IN_PROGRESS = 0xC000A146,
  MD_NTSTATUS_WIN_STATUS_NOT_APPCONTAINER = 0xC000A200,
  MD_NTSTATUS_WIN_STATUS_NOT_SUPPORTED_IN_APPCONTAINER = 0xC000A201,
  MD_NTSTATUS_WIN_STATUS_INVALID_PACKAGE_SID_LENGTH = 0xC000A202,
  MD_NTSTATUS_WIN_STATUS_LPAC_ACCESS_DENIED = 0xC000A203,
  MD_NTSTATUS_WIN_STATUS_ADMINLESS_ACCESS_DENIED = 0xC000A204,
  MD_NTSTATUS_WIN_STATUS_APP_DATA_NOT_FOUND = 0xC000A281,
  MD_NTSTATUS_WIN_STATUS_APP_DATA_EXPIRED = 0xC000A282,
  MD_NTSTATUS_WIN_STATUS_APP_DATA_CORRUPT = 0xC000A283,
  MD_NTSTATUS_WIN_STATUS_APP_DATA_LIMIT_EXCEEDED = 0xC000A284,
  MD_NTSTATUS_WIN_STATUS_APP_DATA_REBOOT_REQUIRED = 0xC000A285,
  MD_NTSTATUS_WIN_STATUS_OFFLOAD_READ_FLT_NOT_SUPPORTED = 0xC000A2A1,
  MD_NTSTATUS_WIN_STATUS_OFFLOAD_WRITE_FLT_NOT_SUPPORTED = 0xC000A2A2,
  MD_NTSTATUS_WIN_STATUS_OFFLOAD_READ_FILE_NOT_SUPPORTED = 0xC000A2A3,
  MD_NTSTATUS_WIN_STATUS_OFFLOAD_WRITE_FILE_NOT_SUPPORTED = 0xC000A2A4,
  MD_NTSTATUS_WIN_STATUS_WOF_WIM_HEADER_CORRUPT = 0xC000A2A5,
  MD_NTSTATUS_WIN_STATUS_WOF_WIM_RESOURCE_TABLE_CORRUPT = 0xC000A2A6,
  MD_NTSTATUS_WIN_STATUS_WOF_FILE_RESOURCE_TABLE_CORRUPT = 0xC000A2A7,
  MD_NTSTATUS_WIN_STATUS_CIMFS_IMAGE_CORRUPT = 0xC000C001,
  MD_NTSTATUS_WIN_STATUS_FILE_SYSTEM_VIRTUALIZATION_UNAVAILABLE = 0xC000CE01,
  MD_NTSTATUS_WIN_STATUS_FILE_SYSTEM_VIRTUALIZATION_METADATA_CORRUPT = 0xC000CE02,
  MD_NTSTATUS_WIN_STATUS_FILE_SYSTEM_VIRTUALIZATION_BUSY = 0xC000CE03,
  MD_NTSTATUS_WIN_STATUS_FILE_SYSTEM_VIRTUALIZATION_PROVIDER_UNKNOWN = 0xC000CE04,
  MD_NTSTATUS_WIN_STATUS_FILE_SYSTEM_VIRTUALIZATION_INVALID_OPERATION = 0xC000CE05,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_SYNC_ROOT_METADATA_CORRUPT = 0xC000CF00,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROVIDER_NOT_RUNNING = 0xC000CF01,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_METADATA_CORRUPT = 0xC000CF02,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_METADATA_TOO_LARGE = 0xC000CF03,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROPERTY_VERSION_NOT_SUPPORTED = 0xC000CF06,
  MD_NTSTATUS_WIN_STATUS_NOT_A_CLOUD_FILE = 0xC000CF07,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_NOT_IN_SYNC = 0xC000CF08,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_ALREADY_CONNECTED = 0xC000CF09,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_NOT_SUPPORTED = 0xC000CF0A,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_INVALID_REQUEST = 0xC000CF0B,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_READ_ONLY_VOLUME = 0xC000CF0C,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_CONNECTED_PROVIDER_ONLY = 0xC000CF0D,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_VALIDATION_FAILED = 0xC000CF0E,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_AUTHENTICATION_FAILED = 0xC000CF0F,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_INSUFFICIENT_RESOURCES = 0xC000CF10,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_NETWORK_UNAVAILABLE = 0xC000CF11,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_UNSUCCESSFUL = 0xC000CF12,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_NOT_UNDER_SYNC_ROOT = 0xC000CF13,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_IN_USE = 0xC000CF14,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PINNED = 0xC000CF15,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_REQUEST_ABORTED = 0xC000CF16,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROPERTY_CORRUPT = 0xC000CF17,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_ACCESS_DENIED = 0xC000CF18,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_INCOMPATIBLE_HARDLINKS = 0xC000CF19,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROPERTY_LOCK_CONFLICT = 0xC000CF1A,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_REQUEST_CANCELED = 0xC000CF1B,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_PROVIDER_TERMINATED = 0xC000CF1D,
  MD_NTSTATUS_WIN_STATUS_NOT_A_CLOUD_SYNC_ROOT = 0xC000CF1E,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_REQUEST_TIMEOUT = 0xC000CF1F,
  MD_NTSTATUS_WIN_STATUS_CLOUD_FILE_DEHYDRATION_DISALLOWED = 0xC000CF20,
  MD_NTSTATUS_WIN_STATUS_FILE_SNAP_IN_PROGRESS = 0xC000F500,
  MD_NTSTATUS_WIN_STATUS_FILE_SNAP_USER_SECTION_NOT_SUPPORTED = 0xC000F501,
  MD_NTSTATUS_WIN_STATUS_FILE_SNAP_MODIFY_NOT_SUPPORTED = 0xC000F502,
  MD_NTSTATUS_WIN_STATUS_FILE_SNAP_IO_NOT_COORDINATED = 0xC000F503,
  MD_NTSTATUS_WIN_STATUS_FILE_SNAP_UNEXPECTED_ERROR = 0xC000F504,
  MD_NTSTATUS_WIN_STATUS_FILE_SNAP_INVALID_PARAMETER = 0xC000F505,
  MD_NTSTATUS_WIN_DBG_NO_STATE_CHANGE = 0xC0010001,
  MD_NTSTATUS_WIN_DBG_APP_NOT_IDLE = 0xC0010002,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_STRING_BINDING = 0xC0020001,
  MD_NTSTATUS_WIN_RPC_NT_WRONG_KIND_OF_BINDING = 0xC0020002,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_BINDING = 0xC0020003,
  MD_NTSTATUS_WIN_RPC_NT_PROTSEQ_NOT_SUPPORTED = 0xC0020004,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_RPC_PROTSEQ = 0xC0020005,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_STRING_UUID = 0xC0020006,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_ENDPOINT_FORMAT = 0xC0020007,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_NET_ADDR = 0xC0020008,
  MD_NTSTATUS_WIN_RPC_NT_NO_ENDPOINT_FOUND = 0xC0020009,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_TIMEOUT = 0xC002000A,
  MD_NTSTATUS_WIN_RPC_NT_OBJECT_NOT_FOUND = 0xC002000B,
  MD_NTSTATUS_WIN_RPC_NT_ALREADY_REGISTERED = 0xC002000C,
  MD_NTSTATUS_WIN_RPC_NT_TYPE_ALREADY_REGISTERED = 0xC002000D,
  MD_NTSTATUS_WIN_RPC_NT_ALREADY_LISTENING = 0xC002000E,
  MD_NTSTATUS_WIN_RPC_NT_NO_PROTSEQS_REGISTERED = 0xC002000F,
  MD_NTSTATUS_WIN_RPC_NT_NOT_LISTENING = 0xC0020010,
  MD_NTSTATUS_WIN_RPC_NT_UNKNOWN_MGR_TYPE = 0xC0020011,
  MD_NTSTATUS_WIN_RPC_NT_UNKNOWN_IF = 0xC0020012,
  MD_NTSTATUS_WIN_RPC_NT_NO_BINDINGS = 0xC0020013,
  MD_NTSTATUS_WIN_RPC_NT_NO_PROTSEQS = 0xC0020014,
  MD_NTSTATUS_WIN_RPC_NT_CANT_CREATE_ENDPOINT = 0xC0020015,
  MD_NTSTATUS_WIN_RPC_NT_OUT_OF_RESOURCES = 0xC0020016,
  MD_NTSTATUS_WIN_RPC_NT_SERVER_UNAVAILABLE = 0xC0020017,
  MD_NTSTATUS_WIN_RPC_NT_SERVER_TOO_BUSY = 0xC0020018,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_NETWORK_OPTIONS = 0xC0020019,
  MD_NTSTATUS_WIN_RPC_NT_NO_CALL_ACTIVE = 0xC002001A,
  MD_NTSTATUS_WIN_RPC_NT_CALL_FAILED = 0xC002001B,
  MD_NTSTATUS_WIN_RPC_NT_CALL_FAILED_DNE = 0xC002001C,
  MD_NTSTATUS_WIN_RPC_NT_PROTOCOL_ERROR = 0xC002001D,
  MD_NTSTATUS_WIN_RPC_NT_UNSUPPORTED_TRANS_SYN = 0xC002001F,
  MD_NTSTATUS_WIN_RPC_NT_UNSUPPORTED_TYPE = 0xC0020021,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_TAG = 0xC0020022,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_BOUND = 0xC0020023,
  MD_NTSTATUS_WIN_RPC_NT_NO_ENTRY_NAME = 0xC0020024,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_NAME_SYNTAX = 0xC0020025,
  MD_NTSTATUS_WIN_RPC_NT_UNSUPPORTED_NAME_SYNTAX = 0xC0020026,
  MD_NTSTATUS_WIN_RPC_NT_UUID_NO_ADDRESS = 0xC0020028,
  MD_NTSTATUS_WIN_RPC_NT_DUPLICATE_ENDPOINT = 0xC0020029,
  MD_NTSTATUS_WIN_RPC_NT_UNKNOWN_AUTHN_TYPE = 0xC002002A,
  MD_NTSTATUS_WIN_RPC_NT_MAX_CALLS_TOO_SMALL = 0xC002002B,
  MD_NTSTATUS_WIN_RPC_NT_STRING_TOO_LONG = 0xC002002C,
  MD_NTSTATUS_WIN_RPC_NT_PROTSEQ_NOT_FOUND = 0xC002002D,
  MD_NTSTATUS_WIN_RPC_NT_PROCNUM_OUT_OF_RANGE = 0xC002002E,
  MD_NTSTATUS_WIN_RPC_NT_BINDING_HAS_NO_AUTH = 0xC002002F,
  MD_NTSTATUS_WIN_RPC_NT_UNKNOWN_AUTHN_SERVICE = 0xC0020030,
  MD_NTSTATUS_WIN_RPC_NT_UNKNOWN_AUTHN_LEVEL = 0xC0020031,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_AUTH_IDENTITY = 0xC0020032,
  MD_NTSTATUS_WIN_RPC_NT_UNKNOWN_AUTHZ_SERVICE = 0xC0020033,
  MD_NTSTATUS_WIN_EPT_NT_INVALID_ENTRY = 0xC0020034,
  MD_NTSTATUS_WIN_EPT_NT_CANT_PERFORM_OP = 0xC0020035,
  MD_NTSTATUS_WIN_EPT_NT_NOT_REGISTERED = 0xC0020036,
  MD_NTSTATUS_WIN_RPC_NT_NOTHING_TO_EXPORT = 0xC0020037,
  MD_NTSTATUS_WIN_RPC_NT_INCOMPLETE_NAME = 0xC0020038,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_VERS_OPTION = 0xC0020039,
  MD_NTSTATUS_WIN_RPC_NT_NO_MORE_MEMBERS = 0xC002003A,
  MD_NTSTATUS_WIN_RPC_NT_NOT_ALL_OBJS_UNEXPORTED = 0xC002003B,
  MD_NTSTATUS_WIN_RPC_NT_INTERFACE_NOT_FOUND = 0xC002003C,
  MD_NTSTATUS_WIN_RPC_NT_ENTRY_ALREADY_EXISTS = 0xC002003D,
  MD_NTSTATUS_WIN_RPC_NT_ENTRY_NOT_FOUND = 0xC002003E,
  MD_NTSTATUS_WIN_RPC_NT_NAME_SERVICE_UNAVAILABLE = 0xC002003F,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_NAF_ID = 0xC0020040,
  MD_NTSTATUS_WIN_RPC_NT_CANNOT_SUPPORT = 0xC0020041,
  MD_NTSTATUS_WIN_RPC_NT_NO_CONTEXT_AVAILABLE = 0xC0020042,
  MD_NTSTATUS_WIN_RPC_NT_INTERNAL_ERROR = 0xC0020043,
  MD_NTSTATUS_WIN_RPC_NT_ZERO_DIVIDE = 0xC0020044,
  MD_NTSTATUS_WIN_RPC_NT_ADDRESS_ERROR = 0xC0020045,
  MD_NTSTATUS_WIN_RPC_NT_FP_DIV_ZERO = 0xC0020046,
  MD_NTSTATUS_WIN_RPC_NT_FP_UNDERFLOW = 0xC0020047,
  MD_NTSTATUS_WIN_RPC_NT_FP_OVERFLOW = 0xC0020048,
  MD_NTSTATUS_WIN_RPC_NT_CALL_IN_PROGRESS = 0xC0020049,
  MD_NTSTATUS_WIN_RPC_NT_NO_MORE_BINDINGS = 0xC002004A,
  MD_NTSTATUS_WIN_RPC_NT_GROUP_MEMBER_NOT_FOUND = 0xC002004B,
  MD_NTSTATUS_WIN_EPT_NT_CANT_CREATE = 0xC002004C,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_OBJECT = 0xC002004D,
  MD_NTSTATUS_WIN_RPC_NT_NO_INTERFACES = 0xC002004F,
  MD_NTSTATUS_WIN_RPC_NT_CALL_CANCELLED = 0xC0020050,
  MD_NTSTATUS_WIN_RPC_NT_BINDING_INCOMPLETE = 0xC0020051,
  MD_NTSTATUS_WIN_RPC_NT_COMM_FAILURE = 0xC0020052,
  MD_NTSTATUS_WIN_RPC_NT_UNSUPPORTED_AUTHN_LEVEL = 0xC0020053,
  MD_NTSTATUS_WIN_RPC_NT_NO_PRINC_NAME = 0xC0020054,
  MD_NTSTATUS_WIN_RPC_NT_NOT_RPC_ERROR = 0xC0020055,
  MD_NTSTATUS_WIN_RPC_NT_SEC_PKG_ERROR = 0xC0020057,
  MD_NTSTATUS_WIN_RPC_NT_NOT_CANCELLED = 0xC0020058,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_ASYNC_HANDLE = 0xC0020062,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_ASYNC_CALL = 0xC0020063,
  MD_NTSTATUS_WIN_RPC_NT_PROXY_ACCESS_DENIED = 0xC0020064,
  MD_NTSTATUS_WIN_RPC_NT_COOKIE_AUTH_FAILED = 0xC0020065,
  MD_NTSTATUS_WIN_RPC_NT_NO_MORE_ENTRIES = 0xC0030001,
  MD_NTSTATUS_WIN_RPC_NT_SS_CHAR_TRANS_OPEN_FAIL = 0xC0030002,
  MD_NTSTATUS_WIN_RPC_NT_SS_CHAR_TRANS_SHORT_FILE = 0xC0030003,
  MD_NTSTATUS_WIN_RPC_NT_SS_IN_NULL_CONTEXT = 0xC0030004,
  MD_NTSTATUS_WIN_RPC_NT_SS_CONTEXT_MISMATCH = 0xC0030005,
  MD_NTSTATUS_WIN_RPC_NT_SS_CONTEXT_DAMAGED = 0xC0030006,
  MD_NTSTATUS_WIN_RPC_NT_SS_HANDLES_MISMATCH = 0xC0030007,
  MD_NTSTATUS_WIN_RPC_NT_SS_CANNOT_GET_CALL_HANDLE = 0xC0030008,
  MD_NTSTATUS_WIN_RPC_NT_NULL_REF_POINTER = 0xC0030009,
  MD_NTSTATUS_WIN_RPC_NT_ENUM_VALUE_OUT_OF_RANGE = 0xC003000A,
  MD_NTSTATUS_WIN_RPC_NT_BYTE_COUNT_TOO_SMALL = 0xC003000B,
  MD_NTSTATUS_WIN_RPC_NT_BAD_STUB_DATA = 0xC003000C,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_ES_ACTION = 0xC0030059,
  MD_NTSTATUS_WIN_RPC_NT_WRONG_ES_VERSION = 0xC003005A,
  MD_NTSTATUS_WIN_RPC_NT_WRONG_STUB_VERSION = 0xC003005B,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_PIPE_OBJECT = 0xC003005C,
  MD_NTSTATUS_WIN_RPC_NT_INVALID_PIPE_OPERATION = 0xC003005D,
  MD_NTSTATUS_WIN_RPC_NT_WRONG_PIPE_VERSION = 0xC003005E,
  MD_NTSTATUS_WIN_RPC_NT_PIPE_CLOSED = 0xC003005F,
  MD_NTSTATUS_WIN_RPC_NT_PIPE_DISCIPLINE_ERROR = 0xC0030060,
  MD_NTSTATUS_WIN_RPC_NT_PIPE_EMPTY = 0xC0030061,
  MD_NTSTATUS_WIN_STATUS_PNP_BAD_MPS_TABLE = 0xC0040035,
  MD_NTSTATUS_WIN_STATUS_PNP_TRANSLATION_FAILED = 0xC0040036,
  MD_NTSTATUS_WIN_STATUS_PNP_IRQ_TRANSLATION_FAILED = 0xC0040037,
  MD_NTSTATUS_WIN_STATUS_PNP_INVALID_ID = 0xC0040038,
  MD_NTSTATUS_WIN_STATUS_IO_REISSUE_AS_CACHED = 0xC0040039,
  MD_NTSTATUS_WIN_STATUS_CTX_WINSTATION_NAME_INVALID = 0xC00A0001,
  MD_NTSTATUS_WIN_STATUS_CTX_INVALID_PD = 0xC00A0002,
  MD_NTSTATUS_WIN_STATUS_CTX_PD_NOT_FOUND = 0xC00A0003,
  MD_NTSTATUS_WIN_STATUS_CTX_CLOSE_PENDING = 0xC00A0006,
  MD_NTSTATUS_WIN_STATUS_CTX_NO_OUTBUF = 0xC00A0007,
  MD_NTSTATUS_WIN_STATUS_CTX_MODEM_INF_NOT_FOUND = 0xC00A0008,
  MD_NTSTATUS_WIN_STATUS_CTX_INVALID_MODEMNAME = 0xC00A0009,
  MD_NTSTATUS_WIN_STATUS_CTX_RESPONSE_ERROR = 0xC00A000A,
  MD_NTSTATUS_WIN_STATUS_CTX_MODEM_RESPONSE_TIMEOUT = 0xC00A000B,
  MD_NTSTATUS_WIN_STATUS_CTX_MODEM_RESPONSE_NO_CARRIER = 0xC00A000C,
  MD_NTSTATUS_WIN_STATUS_CTX_MODEM_RESPONSE_NO_DIALTONE = 0xC00A000D,
  MD_NTSTATUS_WIN_STATUS_CTX_MODEM_RESPONSE_BUSY = 0xC00A000E,
  MD_NTSTATUS_WIN_STATUS_CTX_MODEM_RESPONSE_VOICE = 0xC00A000F,
  MD_NTSTATUS_WIN_STATUS_CTX_TD_ERROR = 0xC00A0010,
  MD_NTSTATUS_WIN_STATUS_CTX_LICENSE_CLIENT_INVALID = 0xC00A0012,
  MD_NTSTATUS_WIN_STATUS_CTX_LICENSE_NOT_AVAILABLE = 0xC00A0013,
  MD_NTSTATUS_WIN_STATUS_CTX_LICENSE_EXPIRED = 0xC00A0014,
  MD_NTSTATUS_WIN_STATUS_CTX_WINSTATION_NOT_FOUND = 0xC00A0015,
  MD_NTSTATUS_WIN_STATUS_CTX_WINSTATION_NAME_COLLISION = 0xC00A0016,
  MD_NTSTATUS_WIN_STATUS_CTX_WINSTATION_BUSY = 0xC00A0017,
  MD_NTSTATUS_WIN_STATUS_CTX_BAD_VIDEO_MODE = 0xC00A0018,
  MD_NTSTATUS_WIN_STATUS_CTX_GRAPHICS_INVALID = 0xC00A0022,
  MD_NTSTATUS_WIN_STATUS_CTX_NOT_CONSOLE = 0xC00A0024,
  MD_NTSTATUS_WIN_STATUS_CTX_CLIENT_QUERY_TIMEOUT = 0xC00A0026,
  MD_NTSTATUS_WIN_STATUS_CTX_CONSOLE_DISCONNECT = 0xC00A0027,
  MD_NTSTATUS_WIN_STATUS_CTX_CONSOLE_CONNECT = 0xC00A0028,
  MD_NTSTATUS_WIN_STATUS_CTX_SHADOW_DENIED = 0xC00A002A,
  MD_NTSTATUS_WIN_STATUS_CTX_WINSTATION_ACCESS_DENIED = 0xC00A002B,
  MD_NTSTATUS_WIN_STATUS_CTX_INVALID_WD = 0xC00A002E,
  MD_NTSTATUS_WIN_STATUS_CTX_WD_NOT_FOUND = 0xC00A002F,
  MD_NTSTATUS_WIN_STATUS_CTX_SHADOW_INVALID = 0xC00A0030,
  MD_NTSTATUS_WIN_STATUS_CTX_SHADOW_DISABLED = 0xC00A0031,
  MD_NTSTATUS_WIN_STATUS_RDP_PROTOCOL_ERROR = 0xC00A0032,
  MD_NTSTATUS_WIN_STATUS_CTX_CLIENT_LICENSE_NOT_SET = 0xC00A0033,
  MD_NTSTATUS_WIN_STATUS_CTX_CLIENT_LICENSE_IN_USE = 0xC00A0034,
  MD_NTSTATUS_WIN_STATUS_CTX_SHADOW_ENDED_BY_MODE_CHANGE = 0xC00A0035,
  MD_NTSTATUS_WIN_STATUS_CTX_SHADOW_NOT_RUNNING = 0xC00A0036,
  MD_NTSTATUS_WIN_STATUS_CTX_LOGON_DISABLED = 0xC00A0037,
  MD_NTSTATUS_WIN_STATUS_CTX_SECURITY_LAYER_ERROR = 0xC00A0038,
  MD_NTSTATUS_WIN_STATUS_TS_INCOMPATIBLE_SESSIONS = 0xC00A0039,
  MD_NTSTATUS_WIN_STATUS_TS_VIDEO_SUBSYSTEM_ERROR = 0xC00A003A,
  MD_NTSTATUS_WIN_STATUS_MUI_FILE_NOT_FOUND = 0xC00B0001,
  MD_NTSTATUS_WIN_STATUS_MUI_INVALID_FILE = 0xC00B0002,
  MD_NTSTATUS_WIN_STATUS_MUI_INVALID_RC_CONFIG = 0xC00B0003,
  MD_NTSTATUS_WIN_STATUS_MUI_INVALID_LOCALE_NAME = 0xC00B0004,
  MD_NTSTATUS_WIN_STATUS_MUI_INVALID_ULTIMATEFALLBACK_NAME = 0xC00B0005,
  MD_NTSTATUS_WIN_STATUS_MUI_FILE_NOT_LOADED = 0xC00B0006,
  MD_NTSTATUS_WIN_STATUS_RESOURCE_ENUM_USER_STOP = 0xC00B0007,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_INVALID_NODE = 0xC0130001,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_EXISTS = 0xC0130002,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_JOIN_IN_PROGRESS = 0xC0130003,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_NOT_FOUND = 0xC0130004,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_LOCAL_NODE_NOT_FOUND = 0xC0130005,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETWORK_EXISTS = 0xC0130006,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETWORK_NOT_FOUND = 0xC0130007,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETINTERFACE_EXISTS = 0xC0130008,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETINTERFACE_NOT_FOUND = 0xC0130009,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_INVALID_REQUEST = 0xC013000A,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_INVALID_NETWORK_PROVIDER = 0xC013000B,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_DOWN = 0xC013000C,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_UNREACHABLE = 0xC013000D,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_NOT_MEMBER = 0xC013000E,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_JOIN_NOT_IN_PROGRESS = 0xC013000F,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_INVALID_NETWORK = 0xC0130010,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NO_NET_ADAPTERS = 0xC0130011,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_UP = 0xC0130012,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_PAUSED = 0xC0130013,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NODE_NOT_PAUSED = 0xC0130014,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NO_SECURITY_CONTEXT = 0xC0130015,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NETWORK_NOT_INTERNAL = 0xC0130016,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_POISONED = 0xC0130017,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_NON_CSV_PATH = 0xC0130018,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_VOLUME_NOT_LOCAL = 0xC0130019,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_READ_OPLOCK_BREAK_IN_PROGRESS = 0xC0130020,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_AUTO_PAUSE_ERROR = 0xC0130021,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_REDIRECTED = 0xC0130022,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_NOT_REDIRECTED = 0xC0130023,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_VOLUME_DRAINING = 0xC0130024,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_SNAPSHOT_CREATION_IN_PROGRESS = 0xC0130025,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_VOLUME_DRAINING_SUCCEEDED_DOWNLEVEL = 0xC0130026,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_NO_SNAPSHOTS = 0xC0130027,
  MD_NTSTATUS_WIN_STATUS_CSV_IO_PAUSE_TIMEOUT = 0xC0130028,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_INVALID_HANDLE = 0xC0130029,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CSV_SUPPORTED_ONLY_ON_COORDINATOR = 0xC0130030,
  MD_NTSTATUS_WIN_STATUS_CLUSTER_CAM_TICKET_REPLAY_DETECTED = 0xC0130031,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_OPCODE = 0xC0140001,
  MD_NTSTATUS_WIN_STATUS_ACPI_STACK_OVERFLOW = 0xC0140002,
  MD_NTSTATUS_WIN_STATUS_ACPI_ASSERT_FAILED = 0xC0140003,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_INDEX = 0xC0140004,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_ARGUMENT = 0xC0140005,
  MD_NTSTATUS_WIN_STATUS_ACPI_FATAL = 0xC0140006,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_SUPERNAME = 0xC0140007,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_ARGTYPE = 0xC0140008,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_OBJTYPE = 0xC0140009,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_TARGETTYPE = 0xC014000A,
  MD_NTSTATUS_WIN_STATUS_ACPI_INCORRECT_ARGUMENT_COUNT = 0xC014000B,
  MD_NTSTATUS_WIN_STATUS_ACPI_ADDRESS_NOT_MAPPED = 0xC014000C,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_EVENTTYPE = 0xC014000D,
  MD_NTSTATUS_WIN_STATUS_ACPI_HANDLER_COLLISION = 0xC014000E,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_DATA = 0xC014000F,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_REGION = 0xC0140010,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_ACCESS_SIZE = 0xC0140011,
  MD_NTSTATUS_WIN_STATUS_ACPI_ACQUIRE_GLOBAL_LOCK = 0xC0140012,
  MD_NTSTATUS_WIN_STATUS_ACPI_ALREADY_INITIALIZED = 0xC0140013,
  MD_NTSTATUS_WIN_STATUS_ACPI_NOT_INITIALIZED = 0xC0140014,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_MUTEX_LEVEL = 0xC0140015,
  MD_NTSTATUS_WIN_STATUS_ACPI_MUTEX_NOT_OWNED = 0xC0140016,
  MD_NTSTATUS_WIN_STATUS_ACPI_MUTEX_NOT_OWNER = 0xC0140017,
  MD_NTSTATUS_WIN_STATUS_ACPI_RS_ACCESS = 0xC0140018,
  MD_NTSTATUS_WIN_STATUS_ACPI_INVALID_TABLE = 0xC0140019,
  MD_NTSTATUS_WIN_STATUS_ACPI_REG_HANDLER_FAILED = 0xC0140020,
  MD_NTSTATUS_WIN_STATUS_ACPI_POWER_REQUEST_FAILED = 0xC0140021,
  MD_NTSTATUS_WIN_STATUS_SXS_SECTION_NOT_FOUND = 0xC0150001,
  MD_NTSTATUS_WIN_STATUS_SXS_CANT_GEN_ACTCTX = 0xC0150002,
  MD_NTSTATUS_WIN_STATUS_SXS_INVALID_ACTCTXDATA_FORMAT = 0xC0150003,
  MD_NTSTATUS_WIN_STATUS_SXS_ASSEMBLY_NOT_FOUND = 0xC0150004,
  MD_NTSTATUS_WIN_STATUS_SXS_MANIFEST_FORMAT_ERROR = 0xC0150005,
  MD_NTSTATUS_WIN_STATUS_SXS_MANIFEST_PARSE_ERROR = 0xC0150006,
  MD_NTSTATUS_WIN_STATUS_SXS_ACTIVATION_CONTEXT_DISABLED = 0xC0150007,
  MD_NTSTATUS_WIN_STATUS_SXS_KEY_NOT_FOUND = 0xC0150008,
  MD_NTSTATUS_WIN_STATUS_SXS_VERSION_CONFLICT = 0xC0150009,
  MD_NTSTATUS_WIN_STATUS_SXS_WRONG_SECTION_TYPE = 0xC015000A,
  MD_NTSTATUS_WIN_STATUS_SXS_THREAD_QUERIES_DISABLED = 0xC015000B,
  MD_NTSTATUS_WIN_STATUS_SXS_ASSEMBLY_MISSING = 0xC015000C,
  MD_NTSTATUS_WIN_STATUS_SXS_PROCESS_DEFAULT_ALREADY_SET = 0xC015000E,
  MD_NTSTATUS_WIN_STATUS_SXS_EARLY_DEACTIVATION = 0xC015000F,
  MD_NTSTATUS_WIN_STATUS_SXS_INVALID_DEACTIVATION = 0xC0150010,
  MD_NTSTATUS_WIN_STATUS_SXS_MULTIPLE_DEACTIVATION = 0xC0150011,
  MD_NTSTATUS_WIN_STATUS_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY = 0xC0150012,
  MD_NTSTATUS_WIN_STATUS_SXS_PROCESS_TERMINATION_REQUESTED = 0xC0150013,
  MD_NTSTATUS_WIN_STATUS_SXS_CORRUPT_ACTIVATION_STACK = 0xC0150014,
  MD_NTSTATUS_WIN_STATUS_SXS_CORRUPTION = 0xC0150015,
  MD_NTSTATUS_WIN_STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE = 0xC0150016,
  MD_NTSTATUS_WIN_STATUS_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME = 0xC0150017,
  MD_NTSTATUS_WIN_STATUS_SXS_IDENTITY_DUPLICATE_ATTRIBUTE = 0xC0150018,
  MD_NTSTATUS_WIN_STATUS_SXS_IDENTITY_PARSE_ERROR = 0xC0150019,
  MD_NTSTATUS_WIN_STATUS_SXS_COMPONENT_STORE_CORRUPT = 0xC015001A,
  MD_NTSTATUS_WIN_STATUS_SXS_FILE_HASH_MISMATCH = 0xC015001B,
  MD_NTSTATUS_WIN_STATUS_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT = 0xC015001C,
  MD_NTSTATUS_WIN_STATUS_SXS_IDENTITIES_DIFFERENT = 0xC015001D,
  MD_NTSTATUS_WIN_STATUS_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT = 0xC015001E,
  MD_NTSTATUS_WIN_STATUS_SXS_FILE_NOT_PART_OF_ASSEMBLY = 0xC015001F,
  MD_NTSTATUS_WIN_STATUS_ADVANCED_INSTALLER_FAILED = 0xC0150020,
  MD_NTSTATUS_WIN_STATUS_XML_ENCODING_MISMATCH = 0xC0150021,
  MD_NTSTATUS_WIN_STATUS_SXS_MANIFEST_TOO_BIG = 0xC0150022,
  MD_NTSTATUS_WIN_STATUS_SXS_SETTING_NOT_REGISTERED = 0xC0150023,
  MD_NTSTATUS_WIN_STATUS_SXS_TRANSACTION_CLOSURE_INCOMPLETE = 0xC0150024,
  MD_NTSTATUS_WIN_STATUS_SMI_PRIMITIVE_INSTALLER_FAILED = 0xC0150025,
  MD_NTSTATUS_WIN_STATUS_GENERIC_COMMAND_FAILED = 0xC0150026,
  MD_NTSTATUS_WIN_STATUS_SXS_FILE_HASH_MISSING = 0xC0150027,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONAL_CONFLICT = 0xC0190001,
  MD_NTSTATUS_WIN_STATUS_INVALID_TRANSACTION = 0xC0190002,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NOT_ACTIVE = 0xC0190003,
  MD_NTSTATUS_WIN_STATUS_TM_INITIALIZATION_FAILED = 0xC0190004,
  MD_NTSTATUS_WIN_STATUS_RM_NOT_ACTIVE = 0xC0190005,
  MD_NTSTATUS_WIN_STATUS_RM_METADATA_CORRUPT = 0xC0190006,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NOT_JOINED = 0xC0190007,
  MD_NTSTATUS_WIN_STATUS_DIRECTORY_NOT_RM = 0xC0190008,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONS_UNSUPPORTED_REMOTE = 0xC019000A,
  MD_NTSTATUS_WIN_STATUS_LOG_RESIZE_INVALID_SIZE = 0xC019000B,
  MD_NTSTATUS_WIN_STATUS_REMOTE_FILE_VERSION_MISMATCH = 0xC019000C,
  MD_NTSTATUS_WIN_STATUS_CRM_PROTOCOL_ALREADY_EXISTS = 0xC019000F,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_PROPAGATION_FAILED = 0xC0190010,
  MD_NTSTATUS_WIN_STATUS_CRM_PROTOCOL_NOT_FOUND = 0xC0190011,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_SUPERIOR_EXISTS = 0xC0190012,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_REQUEST_NOT_VALID = 0xC0190013,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NOT_REQUESTED = 0xC0190014,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_ALREADY_ABORTED = 0xC0190015,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_ALREADY_COMMITTED = 0xC0190016,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_INVALID_MARSHALL_BUFFER = 0xC0190017,
  MD_NTSTATUS_WIN_STATUS_CURRENT_TRANSACTION_NOT_VALID = 0xC0190018,
  MD_NTSTATUS_WIN_STATUS_LOG_GROWTH_FAILED = 0xC0190019,
  MD_NTSTATUS_WIN_STATUS_OBJECT_NO_LONGER_EXISTS = 0xC0190021,
  MD_NTSTATUS_WIN_STATUS_STREAM_MINIVERSION_NOT_FOUND = 0xC0190022,
  MD_NTSTATUS_WIN_STATUS_STREAM_MINIVERSION_NOT_VALID = 0xC0190023,
  MD_NTSTATUS_WIN_STATUS_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION = 0xC0190024,
  MD_NTSTATUS_WIN_STATUS_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT = 0xC0190025,
  MD_NTSTATUS_WIN_STATUS_CANT_CREATE_MORE_STREAM_MINIVERSIONS = 0xC0190026,
  MD_NTSTATUS_WIN_STATUS_HANDLE_NO_LONGER_VALID = 0xC0190028,
  MD_NTSTATUS_WIN_STATUS_LOG_CORRUPTION_DETECTED = 0xC0190030,
  MD_NTSTATUS_WIN_STATUS_RM_DISCONNECTED = 0xC0190032,
  MD_NTSTATUS_WIN_STATUS_ENLISTMENT_NOT_SUPERIOR = 0xC0190033,
  MD_NTSTATUS_WIN_STATUS_FILE_IDENTITY_NOT_PERSISTENT = 0xC0190036,
  MD_NTSTATUS_WIN_STATUS_CANT_BREAK_TRANSACTIONAL_DEPENDENCY = 0xC0190037,
  MD_NTSTATUS_WIN_STATUS_CANT_CROSS_RM_BOUNDARY = 0xC0190038,
  MD_NTSTATUS_WIN_STATUS_TXF_DIR_NOT_EMPTY = 0xC0190039,
  MD_NTSTATUS_WIN_STATUS_INDOUBT_TRANSACTIONS_EXIST = 0xC019003A,
  MD_NTSTATUS_WIN_STATUS_TM_VOLATILE = 0xC019003B,
  MD_NTSTATUS_WIN_STATUS_ROLLBACK_TIMER_EXPIRED = 0xC019003C,
  MD_NTSTATUS_WIN_STATUS_TXF_ATTRIBUTE_CORRUPT = 0xC019003D,
  MD_NTSTATUS_WIN_STATUS_EFS_NOT_ALLOWED_IN_TRANSACTION = 0xC019003E,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONAL_OPEN_NOT_ALLOWED = 0xC019003F,
  MD_NTSTATUS_WIN_STATUS_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE = 0xC0190040,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_REQUIRED_PROMOTION = 0xC0190043,
  MD_NTSTATUS_WIN_STATUS_CANNOT_EXECUTE_FILE_IN_TRANSACTION = 0xC0190044,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONS_NOT_FROZEN = 0xC0190045,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_FREEZE_IN_PROGRESS = 0xC0190046,
  MD_NTSTATUS_WIN_STATUS_NOT_SNAPSHOT_VOLUME = 0xC0190047,
  MD_NTSTATUS_WIN_STATUS_NO_SAVEPOINT_WITH_OPEN_FILES = 0xC0190048,
  MD_NTSTATUS_WIN_STATUS_SPARSE_NOT_ALLOWED_IN_TRANSACTION = 0xC0190049,
  MD_NTSTATUS_WIN_STATUS_TM_IDENTITY_MISMATCH = 0xC019004A,
  MD_NTSTATUS_WIN_STATUS_FLOATED_SECTION = 0xC019004B,
  MD_NTSTATUS_WIN_STATUS_CANNOT_ACCEPT_TRANSACTED_WORK = 0xC019004C,
  MD_NTSTATUS_WIN_STATUS_CANNOT_ABORT_TRANSACTIONS = 0xC019004D,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NOT_FOUND = 0xC019004E,
  MD_NTSTATUS_WIN_STATUS_RESOURCEMANAGER_NOT_FOUND = 0xC019004F,
  MD_NTSTATUS_WIN_STATUS_ENLISTMENT_NOT_FOUND = 0xC0190050,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONMANAGER_NOT_FOUND = 0xC0190051,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONMANAGER_NOT_ONLINE = 0xC0190052,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION = 0xC0190053,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NOT_ROOT = 0xC0190054,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_OBJECT_EXPIRED = 0xC0190055,
  MD_NTSTATUS_WIN_STATUS_COMPRESSION_NOT_ALLOWED_IN_TRANSACTION = 0xC0190056,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_RESPONSE_NOT_ENLISTED = 0xC0190057,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_RECORD_TOO_LONG = 0xC0190058,
  MD_NTSTATUS_WIN_STATUS_NO_LINK_TRACKING_IN_TRANSACTION = 0xC0190059,
  MD_NTSTATUS_WIN_STATUS_OPERATION_NOT_SUPPORTED_IN_TRANSACTION = 0xC019005A,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_INTEGRITY_VIOLATED = 0xC019005B,
  MD_NTSTATUS_WIN_STATUS_TRANSACTIONMANAGER_IDENTITY_MISMATCH = 0xC019005C,
  MD_NTSTATUS_WIN_STATUS_RM_CANNOT_BE_FROZEN_FOR_SNAPSHOT = 0xC019005D,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_MUST_WRITETHROUGH = 0xC019005E,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NO_SUPERIOR = 0xC019005F,
  MD_NTSTATUS_WIN_STATUS_EXPIRED_HANDLE = 0xC0190060,
  MD_NTSTATUS_WIN_STATUS_TRANSACTION_NOT_ENLISTED = 0xC0190061,
  MD_NTSTATUS_WIN_STATUS_LOG_SECTOR_INVALID = 0xC01A0001,
  MD_NTSTATUS_WIN_STATUS_LOG_SECTOR_PARITY_INVALID = 0xC01A0002,
  MD_NTSTATUS_WIN_STATUS_LOG_SECTOR_REMAPPED = 0xC01A0003,
  MD_NTSTATUS_WIN_STATUS_LOG_BLOCK_INCOMPLETE = 0xC01A0004,
  MD_NTSTATUS_WIN_STATUS_LOG_INVALID_RANGE = 0xC01A0005,
  MD_NTSTATUS_WIN_STATUS_LOG_BLOCKS_EXHAUSTED = 0xC01A0006,
  MD_NTSTATUS_WIN_STATUS_LOG_READ_CONTEXT_INVALID = 0xC01A0007,
  MD_NTSTATUS_WIN_STATUS_LOG_RESTART_INVALID = 0xC01A0008,
  MD_NTSTATUS_WIN_STATUS_LOG_BLOCK_VERSION = 0xC01A0009,
  MD_NTSTATUS_WIN_STATUS_LOG_BLOCK_INVALID = 0xC01A000A,
  MD_NTSTATUS_WIN_STATUS_LOG_READ_MODE_INVALID = 0xC01A000B,
  MD_NTSTATUS_WIN_STATUS_LOG_METADATA_CORRUPT = 0xC01A000D,
  MD_NTSTATUS_WIN_STATUS_LOG_METADATA_INVALID = 0xC01A000E,
  MD_NTSTATUS_WIN_STATUS_LOG_METADATA_INCONSISTENT = 0xC01A000F,
  MD_NTSTATUS_WIN_STATUS_LOG_RESERVATION_INVALID = 0xC01A0010,
  MD_NTSTATUS_WIN_STATUS_LOG_CANT_DELETE = 0xC01A0011,
  MD_NTSTATUS_WIN_STATUS_LOG_CONTAINER_LIMIT_EXCEEDED = 0xC01A0012,
  MD_NTSTATUS_WIN_STATUS_LOG_START_OF_LOG = 0xC01A0013,
  MD_NTSTATUS_WIN_STATUS_LOG_POLICY_ALREADY_INSTALLED = 0xC01A0014,
  MD_NTSTATUS_WIN_STATUS_LOG_POLICY_NOT_INSTALLED = 0xC01A0015,
  MD_NTSTATUS_WIN_STATUS_LOG_POLICY_INVALID = 0xC01A0016,
  MD_NTSTATUS_WIN_STATUS_LOG_POLICY_CONFLICT = 0xC01A0017,
  MD_NTSTATUS_WIN_STATUS_LOG_PINNED_ARCHIVE_TAIL = 0xC01A0018,
  MD_NTSTATUS_WIN_STATUS_LOG_RECORD_NONEXISTENT = 0xC01A0019,
  MD_NTSTATUS_WIN_STATUS_LOG_RECORDS_RESERVED_INVALID = 0xC01A001A,
  MD_NTSTATUS_WIN_STATUS_LOG_SPACE_RESERVED_INVALID = 0xC01A001B,
  MD_NTSTATUS_WIN_STATUS_LOG_TAIL_INVALID = 0xC01A001C,
  MD_NTSTATUS_WIN_STATUS_LOG_FULL = 0xC01A001D,
  MD_NTSTATUS_WIN_STATUS_LOG_MULTIPLEXED = 0xC01A001E,
  MD_NTSTATUS_WIN_STATUS_LOG_DEDICATED = 0xC01A001F,
  MD_NTSTATUS_WIN_STATUS_LOG_ARCHIVE_NOT_IN_PROGRESS = 0xC01A0020,
  MD_NTSTATUS_WIN_STATUS_LOG_ARCHIVE_IN_PROGRESS = 0xC01A0021,
  MD_NTSTATUS_WIN_STATUS_LOG_EPHEMERAL = 0xC01A0022,
  MD_NTSTATUS_WIN_STATUS_LOG_NOT_ENOUGH_CONTAINERS = 0xC01A0023,
  MD_NTSTATUS_WIN_STATUS_LOG_CLIENT_ALREADY_REGISTERED = 0xC01A0024,
  MD_NTSTATUS_WIN_STATUS_LOG_CLIENT_NOT_REGISTERED = 0xC01A0025,
  MD_NTSTATUS_WIN_STATUS_LOG_FULL_HANDLER_IN_PROGRESS = 0xC01A0026,
  MD_NTSTATUS_WIN_STATUS_LOG_CONTAINER_READ_FAILED = 0xC01A0027,
  MD_NTSTATUS_WIN_STATUS_LOG_CONTAINER_WRITE_FAILED = 0xC01A0028,
  MD_NTSTATUS_WIN_STATUS_LOG_CONTAINER_OPEN_FAILED = 0xC01A0029,
  MD_NTSTATUS_WIN_STATUS_LOG_CONTAINER_STATE_INVALID = 0xC01A002A,
  MD_NTSTATUS_WIN_STATUS_LOG_STATE_INVALID = 0xC01A002B,
  MD_NTSTATUS_WIN_STATUS_LOG_PINNED = 0xC01A002C,
  MD_NTSTATUS_WIN_STATUS_LOG_METADATA_FLUSH_FAILED = 0xC01A002D,
  MD_NTSTATUS_WIN_STATUS_LOG_INCONSISTENT_SECURITY = 0xC01A002E,
  MD_NTSTATUS_WIN_STATUS_LOG_APPENDED_FLUSH_FAILED = 0xC01A002F,
  MD_NTSTATUS_WIN_STATUS_LOG_PINNED_RESERVATION = 0xC01A0030,
  MD_NTSTATUS_WIN_STATUS_VIDEO_HUNG_DISPLAY_DRIVER_THREAD = 0xC01B00EA,
  MD_NTSTATUS_WIN_STATUS_FLT_NO_HANDLER_DEFINED = 0xC01C0001,
  MD_NTSTATUS_WIN_STATUS_FLT_CONTEXT_ALREADY_DEFINED = 0xC01C0002,
  MD_NTSTATUS_WIN_STATUS_FLT_INVALID_ASYNCHRONOUS_REQUEST = 0xC01C0003,
  MD_NTSTATUS_WIN_STATUS_FLT_DISALLOW_FAST_IO = 0xC01C0004,
  MD_NTSTATUS_WIN_STATUS_FLT_INVALID_NAME_REQUEST = 0xC01C0005,
  MD_NTSTATUS_WIN_STATUS_FLT_NOT_SAFE_TO_POST_OPERATION = 0xC01C0006,
  MD_NTSTATUS_WIN_STATUS_FLT_NOT_INITIALIZED = 0xC01C0007,
  MD_NTSTATUS_WIN_STATUS_FLT_FILTER_NOT_READY = 0xC01C0008,
  MD_NTSTATUS_WIN_STATUS_FLT_POST_OPERATION_CLEANUP = 0xC01C0009,
  MD_NTSTATUS_WIN_STATUS_FLT_INTERNAL_ERROR = 0xC01C000A,
  MD_NTSTATUS_WIN_STATUS_FLT_DELETING_OBJECT = 0xC01C000B,
  MD_NTSTATUS_WIN_STATUS_FLT_MUST_BE_NONPAGED_POOL = 0xC01C000C,
  MD_NTSTATUS_WIN_STATUS_FLT_DUPLICATE_ENTRY = 0xC01C000D,
  MD_NTSTATUS_WIN_STATUS_FLT_CBDQ_DISABLED = 0xC01C000E,
  MD_NTSTATUS_WIN_STATUS_FLT_DO_NOT_ATTACH = 0xC01C000F,
  MD_NTSTATUS_WIN_STATUS_FLT_DO_NOT_DETACH = 0xC01C0010,
  MD_NTSTATUS_WIN_STATUS_FLT_INSTANCE_ALTITUDE_COLLISION = 0xC01C0011,
  MD_NTSTATUS_WIN_STATUS_FLT_INSTANCE_NAME_COLLISION = 0xC01C0012,
  MD_NTSTATUS_WIN_STATUS_FLT_FILTER_NOT_FOUND = 0xC01C0013,
  MD_NTSTATUS_WIN_STATUS_FLT_VOLUME_NOT_FOUND = 0xC01C0014,
  MD_NTSTATUS_WIN_STATUS_FLT_INSTANCE_NOT_FOUND = 0xC01C0015,
  MD_NTSTATUS_WIN_STATUS_FLT_CONTEXT_ALLOCATION_NOT_FOUND = 0xC01C0016,
  MD_NTSTATUS_WIN_STATUS_FLT_INVALID_CONTEXT_REGISTRATION = 0xC01C0017,
  MD_NTSTATUS_WIN_STATUS_FLT_NAME_CACHE_MISS = 0xC01C0018,
  MD_NTSTATUS_WIN_STATUS_FLT_NO_DEVICE_OBJECT = 0xC01C0019,
  MD_NTSTATUS_WIN_STATUS_FLT_VOLUME_ALREADY_MOUNTED = 0xC01C001A,
  MD_NTSTATUS_WIN_STATUS_FLT_ALREADY_ENLISTED = 0xC01C001B,
  MD_NTSTATUS_WIN_STATUS_FLT_CONTEXT_ALREADY_LINKED = 0xC01C001C,
  MD_NTSTATUS_WIN_STATUS_FLT_NO_WAITER_FOR_REPLY = 0xC01C0020,
  MD_NTSTATUS_WIN_STATUS_FLT_REGISTRATION_BUSY = 0xC01C0023,
  MD_NTSTATUS_WIN_STATUS_MONITOR_NO_DESCRIPTOR = 0xC01D0001,
  MD_NTSTATUS_WIN_STATUS_MONITOR_UNKNOWN_DESCRIPTOR_FORMAT = 0xC01D0002,
  MD_NTSTATUS_WIN_STATUS_MONITOR_INVALID_DESCRIPTOR_CHECKSUM = 0xC01D0003,
  MD_NTSTATUS_WIN_STATUS_MONITOR_INVALID_STANDARD_TIMING_BLOCK = 0xC01D0004,
  MD_NTSTATUS_WIN_STATUS_MONITOR_WMI_DATABLOCK_REGISTRATION_FAILED = 0xC01D0005,
  MD_NTSTATUS_WIN_STATUS_MONITOR_INVALID_SERIAL_NUMBER_MONDSC_BLOCK = 0xC01D0006,
  MD_NTSTATUS_WIN_STATUS_MONITOR_INVALID_USER_FRIENDLY_MONDSC_BLOCK = 0xC01D0007,
  MD_NTSTATUS_WIN_STATUS_MONITOR_NO_MORE_DESCRIPTOR_DATA = 0xC01D0008,
  MD_NTSTATUS_WIN_STATUS_MONITOR_INVALID_DETAILED_TIMING_BLOCK = 0xC01D0009,
  MD_NTSTATUS_WIN_STATUS_MONITOR_INVALID_MANUFACTURE_DATE = 0xC01D000A,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NOT_EXCLUSIVE_MODE_OWNER = 0xC01E0000,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INSUFFICIENT_DMA_BUFFER = 0xC01E0001,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_DISPLAY_ADAPTER = 0xC01E0002,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ADAPTER_WAS_RESET = 0xC01E0003,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_DRIVER_MODEL = 0xC01E0004,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_MODE_CHANGED = 0xC01E0005,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_OCCLUDED = 0xC01E0006,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_DENIED = 0xC01E0007,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CANNOTCOLORCONVERT = 0xC01E0008,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DRIVER_MISMATCH = 0xC01E0009,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_REDIRECTION_DISABLED = 0xC01E000B,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_UNOCCLUDED = 0xC01E000C,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_WINDOWDC_NOT_AVAILABLE = 0xC01E000D,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_WINDOWLESS_PRESENT_DISABLED = 0xC01E000E,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_INVALID_WINDOW = 0xC01E000F,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PRESENT_BUFFER_NOT_BOUND = 0xC01E0010,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_VAIL_STATE_CHANGED = 0xC01E0011,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INDIRECT_DISPLAY_ABANDON_SWAPCHAIN = 0xC01E0012,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INDIRECT_DISPLAY_DEVICE_STOPPED = 0xC01E0013,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_VIDEO_MEMORY = 0xC01E0100,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CANT_LOCK_MEMORY = 0xC01E0101,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ALLOCATION_BUSY = 0xC01E0102,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TOO_MANY_REFERENCES = 0xC01E0103,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TRY_AGAIN_LATER = 0xC01E0104,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TRY_AGAIN_NOW = 0xC01E0105,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ALLOCATION_INVALID = 0xC01E0106,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNAVAILABLE = 0xC01E0107,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_UNSWIZZLING_APERTURE_UNSUPPORTED = 0xC01E0108,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CANT_EVICT_PINNED_ALLOCATION = 0xC01E0109,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_ALLOCATION_USAGE = 0xC01E0110,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CANT_RENDER_LOCKED_ALLOCATION = 0xC01E0111,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ALLOCATION_CLOSED = 0xC01E0112,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_ALLOCATION_INSTANCE = 0xC01E0113,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_ALLOCATION_HANDLE = 0xC01E0114,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_WRONG_ALLOCATION_DEVICE = 0xC01E0115,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ALLOCATION_CONTENT_LOST = 0xC01E0116,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_GPU_EXCEPTION_ON_DEVICE = 0xC01E0200,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY = 0xC01E0300,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_VIDPN_TOPOLOGY_NOT_SUPPORTED = 0xC01E0301,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_VIDPN_TOPOLOGY_CURRENTLY_NOT_SUPPORTED = 0xC01E0302,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN = 0xC01E0303,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE = 0xC01E0304,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET = 0xC01E0305,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_VIDPN_MODALITY_NOT_SUPPORTED = 0xC01E0306,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN_SOURCEMODESET = 0xC01E0308,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN_TARGETMODESET = 0xC01E0309,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_FREQUENCY = 0xC01E030A,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_ACTIVE_REGION = 0xC01E030B,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_TOTAL_REGION = 0xC01E030C,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE_MODE = 0xC01E0310,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET_MODE = 0xC01E0311,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PINNED_MODE_MUST_REMAIN_IN_SET = 0xC01E0312,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PATH_ALREADY_IN_TOPOLOGY = 0xC01E0313,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MODE_ALREADY_IN_MODESET = 0xC01E0314,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDEOPRESENTSOURCESET = 0xC01E0315,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDEOPRESENTTARGETSET = 0xC01E0316,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_SOURCE_ALREADY_IN_SET = 0xC01E0317,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TARGET_ALREADY_IN_SET = 0xC01E0318,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN_PRESENT_PATH = 0xC01E0319,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_RECOMMENDED_VIDPN_TOPOLOGY = 0xC01E031A,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGESET = 0xC01E031B,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE = 0xC01E031C,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_FREQUENCYRANGE_NOT_IN_SET = 0xC01E031D,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_FREQUENCYRANGE_ALREADY_IN_SET = 0xC01E031F,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_STALE_MODESET = 0xC01E0320,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITOR_SOURCEMODESET = 0xC01E0321,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITOR_SOURCE_MODE = 0xC01E0322,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_RECOMMENDED_FUNCTIONAL_VIDPN = 0xC01E0323,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MODE_ID_MUST_BE_UNIQUE = 0xC01E0324,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_EMPTY_ADAPTER_MONITOR_MODE_SUPPORT_INTERSECTION = 0xC01E0325,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_VIDEO_PRESENT_TARGETS_LESS_THAN_SOURCES = 0xC01E0326,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PATH_NOT_IN_TOPOLOGY = 0xC01E0327,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_SOURCE = 0xC01E0328,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_TARGET = 0xC01E0329,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITORDESCRIPTORSET = 0xC01E032A,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITORDESCRIPTOR = 0xC01E032B,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MONITORDESCRIPTOR_NOT_IN_SET = 0xC01E032C,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MONITORDESCRIPTOR_ALREADY_IN_SET = 0xC01E032D,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MONITORDESCRIPTOR_ID_MUST_BE_UNIQUE = 0xC01E032E,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN_TARGET_SUBSET_TYPE = 0xC01E032F,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_RESOURCES_NOT_RELATED = 0xC01E0330,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_SOURCE_ID_MUST_BE_UNIQUE = 0xC01E0331,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TARGET_ID_MUST_BE_UNIQUE = 0xC01E0332,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_AVAILABLE_VIDPN_TARGET = 0xC01E0333,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MONITOR_COULD_NOT_BE_ASSOCIATED_WITH_ADAPTER = 0xC01E0334,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_VIDPNMGR = 0xC01E0335,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_ACTIVE_VIDPN = 0xC01E0336,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_STALE_VIDPN_TOPOLOGY = 0xC01E0337,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MONITOR_NOT_CONNECTED = 0xC01E0338,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_SOURCE_NOT_IN_TOPOLOGY = 0xC01E0339,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PRIMARYSURFACE_SIZE = 0xC01E033A,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VISIBLEREGION_SIZE = 0xC01E033B,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_STRIDE = 0xC01E033C,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PIXELFORMAT = 0xC01E033D,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_COLORBASIS = 0xC01E033E,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PIXELVALUEACCESSMODE = 0xC01E033F,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TARGET_NOT_IN_TOPOLOGY = 0xC01E0340,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_DISPLAY_MODE_MANAGEMENT_SUPPORT = 0xC01E0341,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_VIDPN_SOURCE_IN_USE = 0xC01E0342,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CANT_ACCESS_ACTIVE_VIDPN = 0xC01E0343,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PATH_IMPORTANCE_ORDINAL = 0xC01E0344,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PATH_CONTENT_GEOMETRY_TRANSFORMATION = 0xC01E0345,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_SUPPORTED = 0xC01E0346,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_GAMMA_RAMP = 0xC01E0347,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_GAMMA_RAMP_NOT_SUPPORTED = 0xC01E0348,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MULTISAMPLING_NOT_SUPPORTED = 0xC01E0349,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MODE_NOT_IN_MODESET = 0xC01E034A,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_VIDPN_TOPOLOGY_RECOMMENDATION_REASON = 0xC01E034D,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PATH_CONTENT_TYPE = 0xC01E034E,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_COPYPROTECTION_TYPE = 0xC01E034F,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_UNASSIGNED_MODESET_ALREADY_EXISTS = 0xC01E0350,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_SCANLINE_ORDERING = 0xC01E0352,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_TOPOLOGY_CHANGES_NOT_ALLOWED = 0xC01E0353,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_AVAILABLE_IMPORTANCE_ORDINALS = 0xC01E0354,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INCOMPATIBLE_PRIVATE_FORMAT = 0xC01E0355,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MODE_PRUNING_ALGORITHM = 0xC01E0356,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITOR_CAPABILITY_ORIGIN = 0xC01E0357,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE_CONSTRAINT = 0xC01E0358,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MAX_NUM_PATHS_REACHED = 0xC01E0359,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CANCEL_VIDPN_TOPOLOGY_AUGMENTATION = 0xC01E035A,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_CLIENT_TYPE = 0xC01E035B,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CLIENTVIDPN_NOT_SET = 0xC01E035C,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_SPECIFIED_CHILD_ALREADY_CONNECTED = 0xC01E0400,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CHILD_DESCRIPTOR_NOT_SUPPORTED = 0xC01E0401,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NOT_A_LINKED_ADAPTER = 0xC01E0430,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_LEADLINK_NOT_ENUMERATED = 0xC01E0431,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CHAINLINKS_NOT_ENUMERATED = 0xC01E0432,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ADAPTER_CHAIN_NOT_READY = 0xC01E0433,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CHAINLINKS_NOT_STARTED = 0xC01E0434,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_CHAINLINKS_NOT_POWERED_ON = 0xC01E0435,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INCONSISTENT_DEVICE_LINK_STATE = 0xC01E0436,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NOT_POST_DEVICE_DRIVER = 0xC01E0438,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ADAPTER_ACCESS_NOT_EXCLUDED = 0xC01E043B,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_NOT_SUPPORTED = 0xC01E0500,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_COPP_NOT_SUPPORTED = 0xC01E0501,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_UAB_NOT_SUPPORTED = 0xC01E0502,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_INVALID_ENCRYPTED_PARAMETERS = 0xC01E0503,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_NO_PROTECTED_OUTPUTS_EXIST = 0xC01E0505,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_INTERNAL_ERROR = 0xC01E050B,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_INVALID_HANDLE = 0xC01E050C,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PVP_INVALID_CERTIFICATE_LENGTH = 0xC01E050E,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_SPANNING_MODE_ENABLED = 0xC01E050F,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_THEATER_MODE_ENABLED = 0xC01E0510,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PVP_HFS_FAILED = 0xC01E0511,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_INVALID_SRM = 0xC01E0512,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_HDCP = 0xC01E0513,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_ACP = 0xC01E0514,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_CGMSA = 0xC01E0515,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_HDCP_SRM_NEVER_SET = 0xC01E0516,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_RESOLUTION_TOO_HIGH = 0xC01E0517,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_ALL_HDCP_HARDWARE_ALREADY_IN_USE = 0xC01E0518,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_NO_LONGER_EXISTS = 0xC01E051A,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_COPP_SEMANTICS = 0xC01E051C,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_INVALID_INFORMATION_REQUEST = 0xC01E051D,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_DRIVER_INTERNAL_ERROR = 0xC01E051E,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_PROTECTED_OUTPUT_DOES_NOT_HAVE_OPM_SEMANTICS = 0xC01E051F,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_SIGNALING_NOT_SUPPORTED = 0xC01E0520,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_OPM_INVALID_CONFIGURATION_REQUEST = 0xC01E0521,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_I2C_NOT_SUPPORTED = 0xC01E0580,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_I2C_DEVICE_DOES_NOT_EXIST = 0xC01E0581,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_I2C_ERROR_TRANSMITTING_DATA = 0xC01E0582,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_I2C_ERROR_RECEIVING_DATA = 0xC01E0583,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_VCP_NOT_SUPPORTED = 0xC01E0584,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_INVALID_DATA = 0xC01E0585,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_MONITOR_RETURNED_INVALID_TIMING_STATUS_BYTE = 0xC01E0586,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_INVALID_CAPABILITIES_STRING = 0xC01E0587,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MCA_INTERNAL_ERROR = 0xC01E0588,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_COMMAND = 0xC01E0589,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_LENGTH = 0xC01E058A,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DDCCI_INVALID_MESSAGE_CHECKSUM = 0xC01E058B,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_PHYSICAL_MONITOR_HANDLE = 0xC01E058C,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MONITOR_NO_LONGER_EXISTS = 0xC01E058D,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_ONLY_CONSOLE_SESSION_SUPPORTED = 0xC01E05E0,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_DISPLAY_DEVICE_CORRESPONDS_TO_NAME = 0xC01E05E1,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_DISPLAY_DEVICE_NOT_ATTACHED_TO_DESKTOP = 0xC01E05E2,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_MIRRORING_DEVICES_NOT_SUPPORTED = 0xC01E05E3,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INVALID_POINTER = 0xC01E05E4,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_NO_MONITORS_CORRESPOND_TO_DISPLAY_DEVICE = 0xC01E05E5,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_PARAMETER_ARRAY_TOO_SMALL = 0xC01E05E6,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_INTERNAL_ERROR = 0xC01E05E7,
  MD_NTSTATUS_WIN_STATUS_GRAPHICS_SESSION_TYPE_CHANGE_IN_PROGRESS = 0xC01E05E8,
  MD_NTSTATUS_WIN_STATUS_FVE_LOCKED_VOLUME = 0xC0210000,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_ENCRYPTED = 0xC0210001,
  MD_NTSTATUS_WIN_STATUS_FVE_BAD_INFORMATION = 0xC0210002,
  MD_NTSTATUS_WIN_STATUS_FVE_TOO_SMALL = 0xC0210003,
  MD_NTSTATUS_WIN_STATUS_FVE_FAILED_WRONG_FS = 0xC0210004,
  MD_NTSTATUS_WIN_STATUS_FVE_BAD_PARTITION_SIZE = 0xC0210005,
  MD_NTSTATUS_WIN_STATUS_FVE_FS_NOT_EXTENDED = 0xC0210006,
  MD_NTSTATUS_WIN_STATUS_FVE_FS_MOUNTED = 0xC0210007,
  MD_NTSTATUS_WIN_STATUS_FVE_NO_LICENSE = 0xC0210008,
  MD_NTSTATUS_WIN_STATUS_FVE_ACTION_NOT_ALLOWED = 0xC0210009,
  MD_NTSTATUS_WIN_STATUS_FVE_BAD_DATA = 0xC021000A,
  MD_NTSTATUS_WIN_STATUS_FVE_VOLUME_NOT_BOUND = 0xC021000B,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_DATA_VOLUME = 0xC021000C,
  MD_NTSTATUS_WIN_STATUS_FVE_CONV_READ_ERROR = 0xC021000D,
  MD_NTSTATUS_WIN_STATUS_FVE_CONV_WRITE_ERROR = 0xC021000E,
  MD_NTSTATUS_WIN_STATUS_FVE_OVERLAPPED_UPDATE = 0xC021000F,
  MD_NTSTATUS_WIN_STATUS_FVE_FAILED_SECTOR_SIZE = 0xC0210010,
  MD_NTSTATUS_WIN_STATUS_FVE_FAILED_AUTHENTICATION = 0xC0210011,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_OS_VOLUME = 0xC0210012,
  MD_NTSTATUS_WIN_STATUS_FVE_KEYFILE_NOT_FOUND = 0xC0210013,
  MD_NTSTATUS_WIN_STATUS_FVE_KEYFILE_INVALID = 0xC0210014,
  MD_NTSTATUS_WIN_STATUS_FVE_KEYFILE_NO_VMK = 0xC0210015,
  MD_NTSTATUS_WIN_STATUS_FVE_TPM_DISABLED = 0xC0210016,
  MD_NTSTATUS_WIN_STATUS_FVE_TPM_SRK_AUTH_NOT_ZERO = 0xC0210017,
  MD_NTSTATUS_WIN_STATUS_FVE_TPM_INVALID_PCR = 0xC0210018,
  MD_NTSTATUS_WIN_STATUS_FVE_TPM_NO_VMK = 0xC0210019,
  MD_NTSTATUS_WIN_STATUS_FVE_PIN_INVALID = 0xC021001A,
  MD_NTSTATUS_WIN_STATUS_FVE_AUTH_INVALID_APPLICATION = 0xC021001B,
  MD_NTSTATUS_WIN_STATUS_FVE_AUTH_INVALID_CONFIG = 0xC021001C,
  MD_NTSTATUS_WIN_STATUS_FVE_DEBUGGER_ENABLED = 0xC021001D,
  MD_NTSTATUS_WIN_STATUS_FVE_DRY_RUN_FAILED = 0xC021001E,
  MD_NTSTATUS_WIN_STATUS_FVE_BAD_METADATA_POINTER = 0xC021001F,
  MD_NTSTATUS_WIN_STATUS_FVE_OLD_METADATA_COPY = 0xC0210020,
  MD_NTSTATUS_WIN_STATUS_FVE_REBOOT_REQUIRED = 0xC0210021,
  MD_NTSTATUS_WIN_STATUS_FVE_RAW_ACCESS = 0xC0210022,
  MD_NTSTATUS_WIN_STATUS_FVE_RAW_BLOCKED = 0xC0210023,
  MD_NTSTATUS_WIN_STATUS_FVE_NO_AUTOUNLOCK_MASTER_KEY = 0xC0210024,
  MD_NTSTATUS_WIN_STATUS_FVE_MOR_FAILED = 0xC0210025,
  MD_NTSTATUS_WIN_STATUS_FVE_NO_FEATURE_LICENSE = 0xC0210026,
  MD_NTSTATUS_WIN_STATUS_FVE_POLICY_USER_DISABLE_RDV_NOT_ALLOWED = 0xC0210027,
  MD_NTSTATUS_WIN_STATUS_FVE_CONV_RECOVERY_FAILED = 0xC0210028,
  MD_NTSTATUS_WIN_STATUS_FVE_VIRTUALIZED_SPACE_TOO_BIG = 0xC0210029,
  MD_NTSTATUS_WIN_STATUS_FVE_INVALID_DATUM_TYPE = 0xC021002A,
  MD_NTSTATUS_WIN_STATUS_FVE_VOLUME_TOO_SMALL = 0xC0210030,
  MD_NTSTATUS_WIN_STATUS_FVE_ENH_PIN_INVALID = 0xC0210031,
  MD_NTSTATUS_WIN_STATUS_FVE_FULL_ENCRYPTION_NOT_ALLOWED_ON_TP_STORAGE = 0xC0210032,
  MD_NTSTATUS_WIN_STATUS_FVE_WIPE_NOT_ALLOWED_ON_TP_STORAGE = 0xC0210033,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_ALLOWED_ON_CSV_STACK = 0xC0210034,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_ALLOWED_ON_CLUSTER = 0xC0210035,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_ALLOWED_TO_UPGRADE_WHILE_CONVERTING = 0xC0210036,
  MD_NTSTATUS_WIN_STATUS_FVE_WIPE_CANCEL_NOT_APPLICABLE = 0xC0210037,
  MD_NTSTATUS_WIN_STATUS_FVE_EDRIVE_DRY_RUN_FAILED = 0xC0210038,
  MD_NTSTATUS_WIN_STATUS_FVE_SECUREBOOT_DISABLED = 0xC0210039,
  MD_NTSTATUS_WIN_STATUS_FVE_SECUREBOOT_CONFIG_CHANGE = 0xC021003A,
  MD_NTSTATUS_WIN_STATUS_FVE_DEVICE_LOCKEDOUT = 0xC021003B,
  MD_NTSTATUS_WIN_STATUS_FVE_VOLUME_EXTEND_PREVENTS_EOW_DECRYPT = 0xC021003C,
  MD_NTSTATUS_WIN_STATUS_FVE_NOT_DE_VOLUME = 0xC021003D,
  MD_NTSTATUS_WIN_STATUS_FVE_PROTECTION_DISABLED = 0xC021003E,
  MD_NTSTATUS_WIN_STATUS_FVE_PROTECTION_CANNOT_BE_DISABLED = 0xC021003F,
  MD_NTSTATUS_WIN_STATUS_FVE_OSV_KSR_NOT_ALLOWED = 0xC0210040,
  MD_NTSTATUS_WIN_STATUS_FWP_CALLOUT_NOT_FOUND = 0xC0220001,
  MD_NTSTATUS_WIN_STATUS_FWP_CONDITION_NOT_FOUND = 0xC0220002,
  MD_NTSTATUS_WIN_STATUS_FWP_FILTER_NOT_FOUND = 0xC0220003,
  MD_NTSTATUS_WIN_STATUS_FWP_LAYER_NOT_FOUND = 0xC0220004,
  MD_NTSTATUS_WIN_STATUS_FWP_PROVIDER_NOT_FOUND = 0xC0220005,
  MD_NTSTATUS_WIN_STATUS_FWP_PROVIDER_CONTEXT_NOT_FOUND = 0xC0220006,
  MD_NTSTATUS_WIN_STATUS_FWP_SUBLAYER_NOT_FOUND = 0xC0220007,
  MD_NTSTATUS_WIN_STATUS_FWP_NOT_FOUND = 0xC0220008,
  MD_NTSTATUS_WIN_STATUS_FWP_ALREADY_EXISTS = 0xC0220009,
  MD_NTSTATUS_WIN_STATUS_FWP_IN_USE = 0xC022000A,
  MD_NTSTATUS_WIN_STATUS_FWP_DYNAMIC_SESSION_IN_PROGRESS = 0xC022000B,
  MD_NTSTATUS_WIN_STATUS_FWP_WRONG_SESSION = 0xC022000C,
  MD_NTSTATUS_WIN_STATUS_FWP_NO_TXN_IN_PROGRESS = 0xC022000D,
  MD_NTSTATUS_WIN_STATUS_FWP_TXN_IN_PROGRESS = 0xC022000E,
  MD_NTSTATUS_WIN_STATUS_FWP_TXN_ABORTED = 0xC022000F,
  MD_NTSTATUS_WIN_STATUS_FWP_SESSION_ABORTED = 0xC0220010,
  MD_NTSTATUS_WIN_STATUS_FWP_INCOMPATIBLE_TXN = 0xC0220011,
  MD_NTSTATUS_WIN_STATUS_FWP_TIMEOUT = 0xC0220012,
  MD_NTSTATUS_WIN_STATUS_FWP_NET_EVENTS_DISABLED = 0xC0220013,
  MD_NTSTATUS_WIN_STATUS_FWP_INCOMPATIBLE_LAYER = 0xC0220014,
  MD_NTSTATUS_WIN_STATUS_FWP_KM_CLIENTS_ONLY = 0xC0220015,
  MD_NTSTATUS_WIN_STATUS_FWP_LIFETIME_MISMATCH = 0xC0220016,
  MD_NTSTATUS_WIN_STATUS_FWP_BUILTIN_OBJECT = 0xC0220017,
  MD_NTSTATUS_WIN_STATUS_FWP_TOO_MANY_CALLOUTS = 0xC0220018,
  MD_NTSTATUS_WIN_STATUS_FWP_NOTIFICATION_DROPPED = 0xC0220019,
  MD_NTSTATUS_WIN_STATUS_FWP_TRAFFIC_MISMATCH = 0xC022001A,
  MD_NTSTATUS_WIN_STATUS_FWP_INCOMPATIBLE_SA_STATE = 0xC022001B,
  MD_NTSTATUS_WIN_STATUS_FWP_NULL_POINTER = 0xC022001C,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_ENUMERATOR = 0xC022001D,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_FLAGS = 0xC022001E,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_NET_MASK = 0xC022001F,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_RANGE = 0xC0220020,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_INTERVAL = 0xC0220021,
  MD_NTSTATUS_WIN_STATUS_FWP_ZERO_LENGTH_ARRAY = 0xC0220022,
  MD_NTSTATUS_WIN_STATUS_FWP_NULL_DISPLAY_NAME = 0xC0220023,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_ACTION_TYPE = 0xC0220024,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_WEIGHT = 0xC0220025,
  MD_NTSTATUS_WIN_STATUS_FWP_MATCH_TYPE_MISMATCH = 0xC0220026,
  MD_NTSTATUS_WIN_STATUS_FWP_TYPE_MISMATCH = 0xC0220027,
  MD_NTSTATUS_WIN_STATUS_FWP_OUT_OF_BOUNDS = 0xC0220028,
  MD_NTSTATUS_WIN_STATUS_FWP_RESERVED = 0xC0220029,
  MD_NTSTATUS_WIN_STATUS_FWP_DUPLICATE_CONDITION = 0xC022002A,
  MD_NTSTATUS_WIN_STATUS_FWP_DUPLICATE_KEYMOD = 0xC022002B,
  MD_NTSTATUS_WIN_STATUS_FWP_ACTION_INCOMPATIBLE_WITH_LAYER = 0xC022002C,
  MD_NTSTATUS_WIN_STATUS_FWP_ACTION_INCOMPATIBLE_WITH_SUBLAYER = 0xC022002D,
  MD_NTSTATUS_WIN_STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_LAYER = 0xC022002E,
  MD_NTSTATUS_WIN_STATUS_FWP_CONTEXT_INCOMPATIBLE_WITH_CALLOUT = 0xC022002F,
  MD_NTSTATUS_WIN_STATUS_FWP_INCOMPATIBLE_AUTH_METHOD = 0xC0220030,
  MD_NTSTATUS_WIN_STATUS_FWP_INCOMPATIBLE_DH_GROUP = 0xC0220031,
  MD_NTSTATUS_WIN_STATUS_FWP_EM_NOT_SUPPORTED = 0xC0220032,
  MD_NTSTATUS_WIN_STATUS_FWP_NEVER_MATCH = 0xC0220033,
  MD_NTSTATUS_WIN_STATUS_FWP_PROVIDER_CONTEXT_MISMATCH = 0xC0220034,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_PARAMETER = 0xC0220035,
  MD_NTSTATUS_WIN_STATUS_FWP_TOO_MANY_SUBLAYERS = 0xC0220036,
  MD_NTSTATUS_WIN_STATUS_FWP_CALLOUT_NOTIFICATION_FAILED = 0xC0220037,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_AUTH_TRANSFORM = 0xC0220038,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_CIPHER_TRANSFORM = 0xC0220039,
  MD_NTSTATUS_WIN_STATUS_FWP_INCOMPATIBLE_CIPHER_TRANSFORM = 0xC022003A,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_TRANSFORM_COMBINATION = 0xC022003B,
  MD_NTSTATUS_WIN_STATUS_FWP_DUPLICATE_AUTH_METHOD = 0xC022003C,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_TUNNEL_ENDPOINT = 0xC022003D,
  MD_NTSTATUS_WIN_STATUS_FWP_L2_DRIVER_NOT_READY = 0xC022003E,
  MD_NTSTATUS_WIN_STATUS_FWP_KEY_DICTATOR_ALREADY_REGISTERED = 0xC022003F,
  MD_NTSTATUS_WIN_STATUS_FWP_KEY_DICTATION_INVALID_KEYING_MATERIAL = 0xC0220040,
  MD_NTSTATUS_WIN_STATUS_FWP_CONNECTIONS_DISABLED = 0xC0220041,
  MD_NTSTATUS_WIN_STATUS_FWP_INVALID_DNS_NAME = 0xC0220042,
  MD_NTSTATUS_WIN_STATUS_FWP_STILL_ON = 0xC0220043,
  MD_NTSTATUS_WIN_STATUS_FWP_IKEEXT_NOT_RUNNING = 0xC0220044,
  MD_NTSTATUS_WIN_STATUS_FWP_TCPIP_NOT_READY = 0xC0220100,
  MD_NTSTATUS_WIN_STATUS_FWP_INJECT_HANDLE_CLOSING = 0xC0220101,
  MD_NTSTATUS_WIN_STATUS_FWP_INJECT_HANDLE_STALE = 0xC0220102,
  MD_NTSTATUS_WIN_STATUS_FWP_CANNOT_PEND = 0xC0220103,
  MD_NTSTATUS_WIN_STATUS_FWP_DROP_NOICMP = 0xC0220104,
  MD_NTSTATUS_WIN_STATUS_NDIS_CLOSING = 0xC0230002,
  MD_NTSTATUS_WIN_STATUS_NDIS_BAD_VERSION = 0xC0230004,
  MD_NTSTATUS_WIN_STATUS_NDIS_BAD_CHARACTERISTICS = 0xC0230005,
  MD_NTSTATUS_WIN_STATUS_NDIS_ADAPTER_NOT_FOUND = 0xC0230006,
  MD_NTSTATUS_WIN_STATUS_NDIS_OPEN_FAILED = 0xC0230007,
  MD_NTSTATUS_WIN_STATUS_NDIS_DEVICE_FAILED = 0xC0230008,
  MD_NTSTATUS_WIN_STATUS_NDIS_MULTICAST_FULL = 0xC0230009,
  MD_NTSTATUS_WIN_STATUS_NDIS_MULTICAST_EXISTS = 0xC023000A,
  MD_NTSTATUS_WIN_STATUS_NDIS_MULTICAST_NOT_FOUND = 0xC023000B,
  MD_NTSTATUS_WIN_STATUS_NDIS_REQUEST_ABORTED = 0xC023000C,
  MD_NTSTATUS_WIN_STATUS_NDIS_RESET_IN_PROGRESS = 0xC023000D,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_PACKET = 0xC023000F,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_DEVICE_REQUEST = 0xC0230010,
  MD_NTSTATUS_WIN_STATUS_NDIS_ADAPTER_NOT_READY = 0xC0230011,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_LENGTH = 0xC0230014,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_DATA = 0xC0230015,
  MD_NTSTATUS_WIN_STATUS_NDIS_BUFFER_TOO_SHORT = 0xC0230016,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_OID = 0xC0230017,
  MD_NTSTATUS_WIN_STATUS_NDIS_ADAPTER_REMOVED = 0xC0230018,
  MD_NTSTATUS_WIN_STATUS_NDIS_UNSUPPORTED_MEDIA = 0xC0230019,
  MD_NTSTATUS_WIN_STATUS_NDIS_GROUP_ADDRESS_IN_USE = 0xC023001A,
  MD_NTSTATUS_WIN_STATUS_NDIS_FILE_NOT_FOUND = 0xC023001B,
  MD_NTSTATUS_WIN_STATUS_NDIS_ERROR_READING_FILE = 0xC023001C,
  MD_NTSTATUS_WIN_STATUS_NDIS_ALREADY_MAPPED = 0xC023001D,
  MD_NTSTATUS_WIN_STATUS_NDIS_RESOURCE_CONFLICT = 0xC023001E,
  MD_NTSTATUS_WIN_STATUS_NDIS_MEDIA_DISCONNECTED = 0xC023001F,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_ADDRESS = 0xC0230022,
  MD_NTSTATUS_WIN_STATUS_NDIS_PAUSED = 0xC023002A,
  MD_NTSTATUS_WIN_STATUS_NDIS_INTERFACE_NOT_FOUND = 0xC023002B,
  MD_NTSTATUS_WIN_STATUS_NDIS_UNSUPPORTED_REVISION = 0xC023002C,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_PORT = 0xC023002D,
  MD_NTSTATUS_WIN_STATUS_NDIS_INVALID_PORT_STATE = 0xC023002E,
  MD_NTSTATUS_WIN_STATUS_NDIS_LOW_POWER_STATE = 0xC023002F,
  MD_NTSTATUS_WIN_STATUS_NDIS_REINIT_REQUIRED = 0xC0230030,
  MD_NTSTATUS_WIN_STATUS_NDIS_NO_QUEUES = 0xC0230031,
  MD_NTSTATUS_WIN_STATUS_NDIS_NOT_SUPPORTED = 0xC02300BB,
  MD_NTSTATUS_WIN_STATUS_NDIS_OFFLOAD_POLICY = 0xC023100F,
  MD_NTSTATUS_WIN_STATUS_NDIS_OFFLOAD_CONNECTION_REJECTED = 0xC0231012,
  MD_NTSTATUS_WIN_STATUS_NDIS_OFFLOAD_PATH_REJECTED = 0xC0231013,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_AUTO_CONFIG_ENABLED = 0xC0232000,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_MEDIA_IN_USE = 0xC0232001,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_POWER_STATE_INVALID = 0xC0232002,
  MD_NTSTATUS_WIN_STATUS_NDIS_PM_WOL_PATTERN_LIST_FULL = 0xC0232003,
  MD_NTSTATUS_WIN_STATUS_NDIS_PM_PROTOCOL_OFFLOAD_LIST_FULL = 0xC0232004,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_AP_CHANNEL_CURRENTLY_NOT_AVAILABLE = 0xC0232005,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_AP_BAND_CURRENTLY_NOT_AVAILABLE = 0xC0232006,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_AP_CHANNEL_NOT_ALLOWED = 0xC0232007,
  MD_NTSTATUS_WIN_STATUS_NDIS_DOT11_AP_BAND_NOT_ALLOWED = 0xC0232008,
  MD_NTSTATUS_WIN_STATUS_QUIC_HANDSHAKE_FAILURE = 0xC0240000,
  MD_NTSTATUS_WIN_STATUS_QUIC_VER_NEG_FAILURE = 0xC0240001,
  MD_NTSTATUS_WIN_STATUS_TPM_ERROR_MASK = 0xC0290000,
  MD_NTSTATUS_WIN_STATUS_TPM_AUTHFAIL = 0xC0290001,
  MD_NTSTATUS_WIN_STATUS_TPM_BADINDEX = 0xC0290002,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_PARAMETER = 0xC0290003,
  MD_NTSTATUS_WIN_STATUS_TPM_AUDITFAILURE = 0xC0290004,
  MD_NTSTATUS_WIN_STATUS_TPM_CLEAR_DISABLED = 0xC0290005,
  MD_NTSTATUS_WIN_STATUS_TPM_DEACTIVATED = 0xC0290006,
  MD_NTSTATUS_WIN_STATUS_TPM_DISABLED = 0xC0290007,
  MD_NTSTATUS_WIN_STATUS_TPM_DISABLED_CMD = 0xC0290008,
  MD_NTSTATUS_WIN_STATUS_TPM_FAIL = 0xC0290009,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_ORDINAL = 0xC029000A,
  MD_NTSTATUS_WIN_STATUS_TPM_INSTALL_DISABLED = 0xC029000B,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_KEYHANDLE = 0xC029000C,
  MD_NTSTATUS_WIN_STATUS_TPM_KEYNOTFOUND = 0xC029000D,
  MD_NTSTATUS_WIN_STATUS_TPM_INAPPROPRIATE_ENC = 0xC029000E,
  MD_NTSTATUS_WIN_STATUS_TPM_MIGRATEFAIL = 0xC029000F,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_PCR_INFO = 0xC0290010,
  MD_NTSTATUS_WIN_STATUS_TPM_NOSPACE = 0xC0290011,
  MD_NTSTATUS_WIN_STATUS_TPM_NOSRK = 0xC0290012,
  MD_NTSTATUS_WIN_STATUS_TPM_NOTSEALED_BLOB = 0xC0290013,
  MD_NTSTATUS_WIN_STATUS_TPM_OWNER_SET = 0xC0290014,
  MD_NTSTATUS_WIN_STATUS_TPM_RESOURCES = 0xC0290015,
  MD_NTSTATUS_WIN_STATUS_TPM_SHORTRANDOM = 0xC0290016,
  MD_NTSTATUS_WIN_STATUS_TPM_SIZE = 0xC0290017,
  MD_NTSTATUS_WIN_STATUS_TPM_WRONGPCRVAL = 0xC0290018,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_PARAM_SIZE = 0xC0290019,
  MD_NTSTATUS_WIN_STATUS_TPM_SHA_THREAD = 0xC029001A,
  MD_NTSTATUS_WIN_STATUS_TPM_SHA_ERROR = 0xC029001B,
  MD_NTSTATUS_WIN_STATUS_TPM_FAILEDSELFTEST = 0xC029001C,
  MD_NTSTATUS_WIN_STATUS_TPM_AUTH2FAIL = 0xC029001D,
  MD_NTSTATUS_WIN_STATUS_TPM_BADTAG = 0xC029001E,
  MD_NTSTATUS_WIN_STATUS_TPM_IOERROR = 0xC029001F,
  MD_NTSTATUS_WIN_STATUS_TPM_ENCRYPT_ERROR = 0xC0290020,
  MD_NTSTATUS_WIN_STATUS_TPM_DECRYPT_ERROR = 0xC0290021,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_AUTHHANDLE = 0xC0290022,
  MD_NTSTATUS_WIN_STATUS_TPM_NO_ENDORSEMENT = 0xC0290023,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_KEYUSAGE = 0xC0290024,
  MD_NTSTATUS_WIN_STATUS_TPM_WRONG_ENTITYTYPE = 0xC0290025,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_POSTINIT = 0xC0290026,
  MD_NTSTATUS_WIN_STATUS_TPM_INAPPROPRIATE_SIG = 0xC0290027,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_KEY_PROPERTY = 0xC0290028,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_MIGRATION = 0xC0290029,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_SCHEME = 0xC029002A,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_DATASIZE = 0xC029002B,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_MODE = 0xC029002C,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_PRESENCE = 0xC029002D,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_VERSION = 0xC029002E,
  MD_NTSTATUS_WIN_STATUS_TPM_NO_WRAP_TRANSPORT = 0xC029002F,
  MD_NTSTATUS_WIN_STATUS_TPM_AUDITFAIL_UNSUCCESSFUL = 0xC0290030,
  MD_NTSTATUS_WIN_STATUS_TPM_AUDITFAIL_SUCCESSFUL = 0xC0290031,
  MD_NTSTATUS_WIN_STATUS_TPM_NOTRESETABLE = 0xC0290032,
  MD_NTSTATUS_WIN_STATUS_TPM_NOTLOCAL = 0xC0290033,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_TYPE = 0xC0290034,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_RESOURCE = 0xC0290035,
  MD_NTSTATUS_WIN_STATUS_TPM_NOTFIPS = 0xC0290036,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_FAMILY = 0xC0290037,
  MD_NTSTATUS_WIN_STATUS_TPM_NO_NV_PERMISSION = 0xC0290038,
  MD_NTSTATUS_WIN_STATUS_TPM_REQUIRES_SIGN = 0xC0290039,
  MD_NTSTATUS_WIN_STATUS_TPM_KEY_NOTSUPPORTED = 0xC029003A,
  MD_NTSTATUS_WIN_STATUS_TPM_AUTH_CONFLICT = 0xC029003B,
  MD_NTSTATUS_WIN_STATUS_TPM_AREA_LOCKED = 0xC029003C,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_LOCALITY = 0xC029003D,
  MD_NTSTATUS_WIN_STATUS_TPM_READ_ONLY = 0xC029003E,
  MD_NTSTATUS_WIN_STATUS_TPM_PER_NOWRITE = 0xC029003F,
  MD_NTSTATUS_WIN_STATUS_TPM_FAMILYCOUNT = 0xC0290040,
  MD_NTSTATUS_WIN_STATUS_TPM_WRITE_LOCKED = 0xC0290041,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_ATTRIBUTES = 0xC0290042,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_STRUCTURE = 0xC0290043,
  MD_NTSTATUS_WIN_STATUS_TPM_KEY_OWNER_CONTROL = 0xC0290044,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_COUNTER = 0xC0290045,
  MD_NTSTATUS_WIN_STATUS_TPM_NOT_FULLWRITE = 0xC0290046,
  MD_NTSTATUS_WIN_STATUS_TPM_CONTEXT_GAP = 0xC0290047,
  MD_NTSTATUS_WIN_STATUS_TPM_MAXNVWRITES = 0xC0290048,
  MD_NTSTATUS_WIN_STATUS_TPM_NOOPERATOR = 0xC0290049,
  MD_NTSTATUS_WIN_STATUS_TPM_RESOURCEMISSING = 0xC029004A,
  MD_NTSTATUS_WIN_STATUS_TPM_DELEGATE_LOCK = 0xC029004B,
  MD_NTSTATUS_WIN_STATUS_TPM_DELEGATE_FAMILY = 0xC029004C,
  MD_NTSTATUS_WIN_STATUS_TPM_DELEGATE_ADMIN = 0xC029004D,
  MD_NTSTATUS_WIN_STATUS_TPM_TRANSPORT_NOTEXCLUSIVE = 0xC029004E,
  MD_NTSTATUS_WIN_STATUS_TPM_OWNER_CONTROL = 0xC029004F,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_RESOURCES = 0xC0290050,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_INPUT_DATA0 = 0xC0290051,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_INPUT_DATA1 = 0xC0290052,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_ISSUER_SETTINGS = 0xC0290053,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_TPM_SETTINGS = 0xC0290054,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_STAGE = 0xC0290055,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_ISSUER_VALIDITY = 0xC0290056,
  MD_NTSTATUS_WIN_STATUS_TPM_DAA_WRONG_W = 0xC0290057,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_HANDLE = 0xC0290058,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_DELEGATE = 0xC0290059,
  MD_NTSTATUS_WIN_STATUS_TPM_BADCONTEXT = 0xC029005A,
  MD_NTSTATUS_WIN_STATUS_TPM_TOOMANYCONTEXTS = 0xC029005B,
  MD_NTSTATUS_WIN_STATUS_TPM_MA_TICKET_SIGNATURE = 0xC029005C,
  MD_NTSTATUS_WIN_STATUS_TPM_MA_DESTINATION = 0xC029005D,
  MD_NTSTATUS_WIN_STATUS_TPM_MA_SOURCE = 0xC029005E,
  MD_NTSTATUS_WIN_STATUS_TPM_MA_AUTHORITY = 0xC029005F,
  MD_NTSTATUS_WIN_STATUS_TPM_PERMANENTEK = 0xC0290061,
  MD_NTSTATUS_WIN_STATUS_TPM_BAD_SIGNATURE = 0xC0290062,
  MD_NTSTATUS_WIN_STATUS_TPM_NOCONTEXTSPACE = 0xC0290063,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_ASYMMETRIC = 0xC0290081,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_ATTRIBUTES = 0xC0290082,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_HASH = 0xC0290083,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_VALUE = 0xC0290084,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_HIERARCHY = 0xC0290085,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_KEY_SIZE = 0xC0290087,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_MGF = 0xC0290088,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_MODE = 0xC0290089,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_TYPE = 0xC029008A,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_HANDLE = 0xC029008B,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_KDF = 0xC029008C,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_RANGE = 0xC029008D,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_AUTH_FAIL = 0xC029008E,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NONCE = 0xC029008F,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_PP = 0xC0290090,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SCHEME = 0xC0290092,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SIZE = 0xC0290095,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SYMMETRIC = 0xC0290096,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_TAG = 0xC0290097,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SELECTOR = 0xC0290098,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_INSUFFICIENT = 0xC029009A,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SIGNATURE = 0xC029009B,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_KEY = 0xC029009C,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_POLICY_FAIL = 0xC029009D,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_INTEGRITY = 0xC029009F,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_TICKET = 0xC02900A0,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_RESERVED_BITS = 0xC02900A1,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_BAD_AUTH = 0xC02900A2,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_EXPIRED = 0xC02900A3,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_POLICY_CC = 0xC02900A4,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_BINDING = 0xC02900A5,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_CURVE = 0xC02900A6,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_ECC_POINT = 0xC02900A7,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_INITIALIZE = 0xC0290100,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_FAILURE = 0xC0290101,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SEQUENCE = 0xC0290103,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_PRIVATE = 0xC029010B,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_HMAC = 0xC0290119,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_DISABLED = 0xC0290120,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_EXCLUSIVE = 0xC0290121,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_ECC_CURVE = 0xC0290123,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_AUTH_TYPE = 0xC0290124,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_AUTH_MISSING = 0xC0290125,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_POLICY = 0xC0290126,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_PCR = 0xC0290127,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_PCR_CHANGED = 0xC0290128,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_UPGRADE = 0xC029012D,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_TOO_MANY_CONTEXTS = 0xC029012E,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_AUTH_UNAVAILABLE = 0xC029012F,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_REBOOT = 0xC0290130,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_UNBALANCED = 0xC0290131,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_COMMAND_SIZE = 0xC0290142,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_COMMAND_CODE = 0xC0290143,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_AUTHSIZE = 0xC0290144,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_AUTH_CONTEXT = 0xC0290145,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_RANGE = 0xC0290146,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_SIZE = 0xC0290147,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_LOCKED = 0xC0290148,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_AUTHORIZATION = 0xC0290149,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_UNINITIALIZED = 0xC029014A,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_SPACE = 0xC029014B,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NV_DEFINED = 0xC029014C,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_BAD_CONTEXT = 0xC0290150,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_CPHASH = 0xC0290151,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_PARENT = 0xC0290152,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NEEDS_TEST = 0xC0290153,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_NO_RESULT = 0xC0290154,
  MD_NTSTATUS_WIN_STATUS_TPM_20_E_SENSITIVE = 0xC0290155,
  MD_NTSTATUS_WIN_STATUS_TPM_COMMAND_BLOCKED = 0xC0290400,
  MD_NTSTATUS_WIN_STATUS_TPM_INVALID_HANDLE = 0xC0290401,
  MD_NTSTATUS_WIN_STATUS_TPM_DUPLICATE_VHANDLE = 0xC0290402,
  MD_NTSTATUS_WIN_STATUS_TPM_EMBEDDED_COMMAND_BLOCKED = 0xC0290403,
  MD_NTSTATUS_WIN_STATUS_TPM_EMBEDDED_COMMAND_UNSUPPORTED = 0xC0290404,
  MD_NTSTATUS_WIN_STATUS_TPM_RETRY = 0xC0290800,
  MD_NTSTATUS_WIN_STATUS_TPM_NEEDS_SELFTEST = 0xC0290801,
  MD_NTSTATUS_WIN_STATUS_TPM_DOING_SELFTEST = 0xC0290802,
  MD_NTSTATUS_WIN_STATUS_TPM_DEFEND_LOCK_RUNNING = 0xC0290803,
  MD_NTSTATUS_WIN_STATUS_TPM_COMMAND_CANCELED = 0xC0291001,
  MD_NTSTATUS_WIN_STATUS_TPM_TOO_MANY_CONTEXTS = 0xC0291002,
  MD_NTSTATUS_WIN_STATUS_TPM_NOT_FOUND = 0xC0291003,
  MD_NTSTATUS_WIN_STATUS_TPM_ACCESS_DENIED = 0xC0291004,
  MD_NTSTATUS_WIN_STATUS_TPM_INSUFFICIENT_BUFFER = 0xC0291005,
  MD_NTSTATUS_WIN_STATUS_TPM_PPI_FUNCTION_UNSUPPORTED = 0xC0291006,
  MD_NTSTATUS_WIN_STATUS_PCP_ERROR_MASK = 0xC0292000,
  MD_NTSTATUS_WIN_STATUS_PCP_DEVICE_NOT_READY = 0xC0292001,
  MD_NTSTATUS_WIN_STATUS_PCP_INVALID_HANDLE = 0xC0292002,
  MD_NTSTATUS_WIN_STATUS_PCP_INVALID_PARAMETER = 0xC0292003,
  MD_NTSTATUS_WIN_STATUS_PCP_FLAG_NOT_SUPPORTED = 0xC0292004,
  MD_NTSTATUS_WIN_STATUS_PCP_NOT_SUPPORTED = 0xC0292005,
  MD_NTSTATUS_WIN_STATUS_PCP_BUFFER_TOO_SMALL = 0xC0292006,
  MD_NTSTATUS_WIN_STATUS_PCP_INTERNAL_ERROR = 0xC0292007,
  MD_NTSTATUS_WIN_STATUS_PCP_AUTHENTICATION_FAILED = 0xC0292008,
  MD_NTSTATUS_WIN_STATUS_PCP_AUTHENTICATION_IGNORED = 0xC0292009,
  MD_NTSTATUS_WIN_STATUS_PCP_POLICY_NOT_FOUND = 0xC029200A,
  MD_NTSTATUS_WIN_STATUS_PCP_PROFILE_NOT_FOUND = 0xC029200B,
  MD_NTSTATUS_WIN_STATUS_PCP_VALIDATION_FAILED = 0xC029200C,
  MD_NTSTATUS_WIN_STATUS_PCP_DEVICE_NOT_FOUND = 0xC029200D,
  MD_NTSTATUS_WIN_STATUS_PCP_WRONG_PARENT = 0xC029200E,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_NOT_LOADED = 0xC029200F,
  MD_NTSTATUS_WIN_STATUS_PCP_NO_KEY_CERTIFICATION = 0xC0292010,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_NOT_FINALIZED = 0xC0292011,
  MD_NTSTATUS_WIN_STATUS_PCP_ATTESTATION_CHALLENGE_NOT_SET = 0xC0292012,
  MD_NTSTATUS_WIN_STATUS_PCP_NOT_PCR_BOUND = 0xC0292013,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_ALREADY_FINALIZED = 0xC0292014,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_USAGE_POLICY_NOT_SUPPORTED = 0xC0292015,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_USAGE_POLICY_INVALID = 0xC0292016,
  MD_NTSTATUS_WIN_STATUS_PCP_SOFT_KEY_ERROR = 0xC0292017,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_NOT_AUTHENTICATED = 0xC0292018,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_NOT_AIK = 0xC0292019,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_NOT_SIGNING_KEY = 0xC029201A,
  MD_NTSTATUS_WIN_STATUS_PCP_LOCKED_OUT = 0xC029201B,
  MD_NTSTATUS_WIN_STATUS_PCP_CLAIM_TYPE_NOT_SUPPORTED = 0xC029201C,
  MD_NTSTATUS_WIN_STATUS_PCP_TPM_VERSION_NOT_SUPPORTED = 0xC029201D,
  MD_NTSTATUS_WIN_STATUS_PCP_BUFFER_LENGTH_MISMATCH = 0xC029201E,
  MD_NTSTATUS_WIN_STATUS_PCP_IFX_RSA_KEY_CREATION_BLOCKED = 0xC029201F,
  MD_NTSTATUS_WIN_STATUS_PCP_TICKET_MISSING = 0xC0292020,
  MD_NTSTATUS_WIN_STATUS_PCP_RAW_POLICY_NOT_SUPPORTED = 0xC0292021,
  MD_NTSTATUS_WIN_STATUS_PCP_KEY_HANDLE_INVALIDATED = 0xC0292022,
  MD_NTSTATUS_WIN_STATUS_RTPM_NO_RESULT = 0xC0293002,
  MD_NTSTATUS_WIN_STATUS_RTPM_PCR_READ_INCOMPLETE = 0xC0293003,
  MD_NTSTATUS_WIN_STATUS_RTPM_INVALID_CONTEXT = 0xC0293004,
  MD_NTSTATUS_WIN_STATUS_RTPM_UNSUPPORTED_CMD = 0xC0293005,
  MD_NTSTATUS_WIN_STATUS_TPM_ZERO_EXHAUST_ENABLED = 0xC0294000,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_HYPERCALL_CODE = 0xC0350002,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_HYPERCALL_INPUT = 0xC0350003,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_ALIGNMENT = 0xC0350004,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_PARAMETER = 0xC0350005,
  MD_NTSTATUS_WIN_STATUS_HV_ACCESS_DENIED = 0xC0350006,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_PARTITION_STATE = 0xC0350007,
  MD_NTSTATUS_WIN_STATUS_HV_OPERATION_DENIED = 0xC0350008,
  MD_NTSTATUS_WIN_STATUS_HV_UNKNOWN_PROPERTY = 0xC0350009,
  MD_NTSTATUS_WIN_STATUS_HV_PROPERTY_VALUE_OUT_OF_RANGE = 0xC035000A,
  MD_NTSTATUS_WIN_STATUS_HV_INSUFFICIENT_MEMORY = 0xC035000B,
  MD_NTSTATUS_WIN_STATUS_HV_PARTITION_TOO_DEEP = 0xC035000C,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_PARTITION_ID = 0xC035000D,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_VP_INDEX = 0xC035000E,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_PORT_ID = 0xC0350011,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_CONNECTION_ID = 0xC0350012,
  MD_NTSTATUS_WIN_STATUS_HV_INSUFFICIENT_BUFFERS = 0xC0350013,
  MD_NTSTATUS_WIN_STATUS_HV_NOT_ACKNOWLEDGED = 0xC0350014,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_VP_STATE = 0xC0350015,
  MD_NTSTATUS_WIN_STATUS_HV_ACKNOWLEDGED = 0xC0350016,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_SAVE_RESTORE_STATE = 0xC0350017,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_SYNIC_STATE = 0xC0350018,
  MD_NTSTATUS_WIN_STATUS_HV_OBJECT_IN_USE = 0xC0350019,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_PROXIMITY_DOMAIN_INFO = 0xC035001A,
  MD_NTSTATUS_WIN_STATUS_HV_NO_DATA = 0xC035001B,
  MD_NTSTATUS_WIN_STATUS_HV_INACTIVE = 0xC035001C,
  MD_NTSTATUS_WIN_STATUS_HV_NO_RESOURCES = 0xC035001D,
  MD_NTSTATUS_WIN_STATUS_HV_FEATURE_UNAVAILABLE = 0xC035001E,
  MD_NTSTATUS_WIN_STATUS_HV_INSUFFICIENT_BUFFER = 0xC0350033,
  MD_NTSTATUS_WIN_STATUS_HV_INSUFFICIENT_DEVICE_DOMAINS = 0xC0350038,
  MD_NTSTATUS_WIN_STATUS_HV_CPUID_FEATURE_VALIDATION_ERROR = 0xC035003C,
  MD_NTSTATUS_WIN_STATUS_HV_CPUID_XSAVE_FEATURE_VALIDATION_ERROR = 0xC035003D,
  MD_NTSTATUS_WIN_STATUS_HV_PROCESSOR_STARTUP_TIMEOUT = 0xC035003E,
  MD_NTSTATUS_WIN_STATUS_HV_SMX_ENABLED = 0xC035003F,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_LP_INDEX = 0xC0350041,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_REGISTER_VALUE = 0xC0350050,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_VTL_STATE = 0xC0350051,
  MD_NTSTATUS_WIN_STATUS_HV_NX_NOT_DETECTED = 0xC0350055,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_DEVICE_ID = 0xC0350057,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_DEVICE_STATE = 0xC0350058,
  MD_NTSTATUS_WIN_STATUS_HV_PAGE_REQUEST_INVALID = 0xC0350060,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_CPU_GROUP_ID = 0xC035006F,
  MD_NTSTATUS_WIN_STATUS_HV_INVALID_CPU_GROUP_STATE = 0xC0350070,
  MD_NTSTATUS_WIN_STATUS_HV_OPERATION_FAILED = 0xC0350071,
  MD_NTSTATUS_WIN_STATUS_HV_NOT_ALLOWED_WITH_NESTED_VIRT_ACTIVE = 0xC0350072,
  MD_NTSTATUS_WIN_STATUS_HV_INSUFFICIENT_ROOT_MEMORY = 0xC0350073,
  MD_NTSTATUS_WIN_STATUS_HV_EVENT_BUFFER_ALREADY_FREED = 0xC0350074,
  MD_NTSTATUS_WIN_STATUS_HV_INSUFFICIENT_CONTIGUOUS_MEMORY = 0xC0350075,
  MD_NTSTATUS_WIN_STATUS_HV_NOT_PRESENT = 0xC0351000,
  MD_NTSTATUS_WIN_STATUS_IPSEC_BAD_SPI = 0xC0360001,
  MD_NTSTATUS_WIN_STATUS_IPSEC_SA_LIFETIME_EXPIRED = 0xC0360002,
  MD_NTSTATUS_WIN_STATUS_IPSEC_WRONG_SA = 0xC0360003,
  MD_NTSTATUS_WIN_STATUS_IPSEC_REPLAY_CHECK_FAILED = 0xC0360004,
  MD_NTSTATUS_WIN_STATUS_IPSEC_INVALID_PACKET = 0xC0360005,
  MD_NTSTATUS_WIN_STATUS_IPSEC_INTEGRITY_CHECK_FAILED = 0xC0360006,
  MD_NTSTATUS_WIN_STATUS_IPSEC_CLEAR_TEXT_DROP = 0xC0360007,
  MD_NTSTATUS_WIN_STATUS_IPSEC_AUTH_FIREWALL_DROP = 0xC0360008,
  MD_NTSTATUS_WIN_STATUS_IPSEC_THROTTLE_DROP = 0xC0360009,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_BLOCK = 0xC0368000,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_RECEIVED_MULTICAST = 0xC0368001,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_INVALID_PACKET = 0xC0368002,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_STATE_LOOKUP_FAILED = 0xC0368003,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_MAX_ENTRIES = 0xC0368004,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_KEYMOD_NOT_ALLOWED = 0xC0368005,
  MD_NTSTATUS_WIN_STATUS_IPSEC_DOSP_MAX_PER_IP_RATELIMIT_QUEUES = 0xC0368006,
  MD_NTSTATUS_WIN_STATUS_VID_DUPLICATE_HANDLER = 0xC0370001,
  MD_NTSTATUS_WIN_STATUS_VID_TOO_MANY_HANDLERS = 0xC0370002,
  MD_NTSTATUS_WIN_STATUS_VID_QUEUE_FULL = 0xC0370003,
  MD_NTSTATUS_WIN_STATUS_VID_HANDLER_NOT_PRESENT = 0xC0370004,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_OBJECT_NAME = 0xC0370005,
  MD_NTSTATUS_WIN_STATUS_VID_PARTITION_NAME_TOO_LONG = 0xC0370006,
  MD_NTSTATUS_WIN_STATUS_VID_MESSAGE_QUEUE_NAME_TOO_LONG = 0xC0370007,
  MD_NTSTATUS_WIN_STATUS_VID_PARTITION_ALREADY_EXISTS = 0xC0370008,
  MD_NTSTATUS_WIN_STATUS_VID_PARTITION_DOES_NOT_EXIST = 0xC0370009,
  MD_NTSTATUS_WIN_STATUS_VID_PARTITION_NAME_NOT_FOUND = 0xC037000A,
  MD_NTSTATUS_WIN_STATUS_VID_MESSAGE_QUEUE_ALREADY_EXISTS = 0xC037000B,
  MD_NTSTATUS_WIN_STATUS_VID_EXCEEDED_MBP_ENTRY_MAP_LIMIT = 0xC037000C,
  MD_NTSTATUS_WIN_STATUS_VID_MB_STILL_REFERENCED = 0xC037000D,
  MD_NTSTATUS_WIN_STATUS_VID_CHILD_GPA_PAGE_SET_CORRUPTED = 0xC037000E,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_NUMA_SETTINGS = 0xC037000F,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_NUMA_NODE_INDEX = 0xC0370010,
  MD_NTSTATUS_WIN_STATUS_VID_NOTIFICATION_QUEUE_ALREADY_ASSOCIATED = 0xC0370011,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_MEMORY_BLOCK_HANDLE = 0xC0370012,
  MD_NTSTATUS_WIN_STATUS_VID_PAGE_RANGE_OVERFLOW = 0xC0370013,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_MESSAGE_QUEUE_HANDLE = 0xC0370014,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_GPA_RANGE_HANDLE = 0xC0370015,
  MD_NTSTATUS_WIN_STATUS_VID_NO_MEMORY_BLOCK_NOTIFICATION_QUEUE = 0xC0370016,
  MD_NTSTATUS_WIN_STATUS_VID_MEMORY_BLOCK_LOCK_COUNT_EXCEEDED = 0xC0370017,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_PPM_HANDLE = 0xC0370018,
  MD_NTSTATUS_WIN_STATUS_VID_MBPS_ARE_LOCKED = 0xC0370019,
  MD_NTSTATUS_WIN_STATUS_VID_MESSAGE_QUEUE_CLOSED = 0xC037001A,
  MD_NTSTATUS_WIN_STATUS_VID_VIRTUAL_PROCESSOR_LIMIT_EXCEEDED = 0xC037001B,
  MD_NTSTATUS_WIN_STATUS_VID_STOP_PENDING = 0xC037001C,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_PROCESSOR_STATE = 0xC037001D,
  MD_NTSTATUS_WIN_STATUS_VID_EXCEEDED_KM_CONTEXT_COUNT_LIMIT = 0xC037001E,
  MD_NTSTATUS_WIN_STATUS_VID_KM_INTERFACE_ALREADY_INITIALIZED = 0xC037001F,
  MD_NTSTATUS_WIN_STATUS_VID_MB_PROPERTY_ALREADY_SET_RESET = 0xC0370020,
  MD_NTSTATUS_WIN_STATUS_VID_MMIO_RANGE_DESTROYED = 0xC0370021,
  MD_NTSTATUS_WIN_STATUS_VID_INVALID_CHILD_GPA_PAGE_SET = 0xC0370022,
  MD_NTSTATUS_WIN_STATUS_VID_RESERVE_PAGE_SET_IS_BEING_USED = 0xC0370023,
  MD_NTSTATUS_WIN_STATUS_VID_RESERVE_PAGE_SET_TOO_SMALL = 0xC0370024,
  MD_NTSTATUS_WIN_STATUS_VID_MBP_ALREADY_LOCKED_USING_RESERVED_PAGE = 0xC0370025,
  MD_NTSTATUS_WIN_STATUS_VID_MBP_COUNT_EXCEEDED_LIMIT = 0xC0370026,
  MD_NTSTATUS_WIN_STATUS_VID_SAVED_STATE_CORRUPT = 0xC0370027,
  MD_NTSTATUS_WIN_STATUS_VID_SAVED_STATE_UNRECOGNIZED_ITEM = 0xC0370028,
  MD_NTSTATUS_WIN_STATUS_VID_SAVED_STATE_INCOMPATIBLE = 0xC0370029,
  MD_NTSTATUS_WIN_STATUS_VID_VTL_ACCESS_DENIED = 0xC037002A,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DATABASE_FULL = 0xC0380001,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_CONFIGURATION_CORRUPTED = 0xC0380002,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_CONFIGURATION_NOT_IN_SYNC = 0xC0380003,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_CONFIG_UPDATE_FAILED = 0xC0380004,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_CONTAINS_NON_SIMPLE_VOLUME = 0xC0380005,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_DUPLICATE = 0xC0380006,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_DYNAMIC = 0xC0380007,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_ID_INVALID = 0xC0380008,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_INVALID = 0xC0380009,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAST_VOTER = 0xC038000A,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAYOUT_INVALID = 0xC038000B,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAYOUT_NON_BASIC_BETWEEN_BASIC_PARTITIONS = 0xC038000C,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAYOUT_NOT_CYLINDER_ALIGNED = 0xC038000D,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAYOUT_PARTITIONS_TOO_SMALL = 0xC038000E,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAYOUT_PRIMARY_BETWEEN_LOGICAL_PARTITIONS = 0xC038000F,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_LAYOUT_TOO_MANY_PARTITIONS = 0xC0380010,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_MISSING = 0xC0380011,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_NOT_EMPTY = 0xC0380012,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_NOT_ENOUGH_SPACE = 0xC0380013,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_REVECTORING_FAILED = 0xC0380014,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_SECTOR_SIZE_INVALID = 0xC0380015,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_SET_NOT_CONTAINED = 0xC0380016,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_USED_BY_MULTIPLE_MEMBERS = 0xC0380017,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DISK_USED_BY_MULTIPLE_PLEXES = 0xC0380018,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DYNAMIC_DISK_NOT_SUPPORTED = 0xC0380019,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_EXTENT_ALREADY_USED = 0xC038001A,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_EXTENT_NOT_CONTIGUOUS = 0xC038001B,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_EXTENT_NOT_IN_PUBLIC_REGION = 0xC038001C,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_EXTENT_NOT_SECTOR_ALIGNED = 0xC038001D,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_EXTENT_OVERLAPS_EBR_PARTITION = 0xC038001E,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_EXTENT_VOLUME_LENGTHS_DO_NOT_MATCH = 0xC038001F,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_FAULT_TOLERANT_NOT_SUPPORTED = 0xC0380020,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_INTERLEAVE_LENGTH_INVALID = 0xC0380021,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MAXIMUM_REGISTERED_USERS = 0xC0380022,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MEMBER_IN_SYNC = 0xC0380023,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MEMBER_INDEX_DUPLICATE = 0xC0380024,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MEMBER_INDEX_INVALID = 0xC0380025,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MEMBER_MISSING = 0xC0380026,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MEMBER_NOT_DETACHED = 0xC0380027,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MEMBER_REGENERATING = 0xC0380028,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_ALL_DISKS_FAILED = 0xC0380029,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NO_REGISTERED_USERS = 0xC038002A,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NO_SUCH_USER = 0xC038002B,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NOTIFICATION_RESET = 0xC038002C,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NUMBER_OF_MEMBERS_INVALID = 0xC038002D,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NUMBER_OF_PLEXES_INVALID = 0xC038002E,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_DUPLICATE = 0xC038002F,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_ID_INVALID = 0xC0380030,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_INVALID = 0xC0380031,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_NAME_INVALID = 0xC0380032,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_OFFLINE = 0xC0380033,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_HAS_QUORUM = 0xC0380034,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_WITHOUT_QUORUM = 0xC0380035,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PARTITION_STYLE_INVALID = 0xC0380036,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PARTITION_UPDATE_FAILED = 0xC0380037,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_IN_SYNC = 0xC0380038,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_INDEX_DUPLICATE = 0xC0380039,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_INDEX_INVALID = 0xC038003A,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_LAST_ACTIVE = 0xC038003B,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_MISSING = 0xC038003C,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_REGENERATING = 0xC038003D,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_TYPE_INVALID = 0xC038003E,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_NOT_RAID5 = 0xC038003F,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_NOT_SIMPLE = 0xC0380040,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_STRUCTURE_SIZE_INVALID = 0xC0380041,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_TOO_MANY_NOTIFICATION_REQUESTS = 0xC0380042,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_TRANSACTION_IN_PROGRESS = 0xC0380043,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_UNEXPECTED_DISK_LAYOUT_CHANGE = 0xC0380044,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_CONTAINS_MISSING_DISK = 0xC0380045,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_ID_INVALID = 0xC0380046,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_LENGTH_INVALID = 0xC0380047,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_LENGTH_NOT_SECTOR_SIZE_MULTIPLE = 0xC0380048,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_NOT_MIRRORED = 0xC0380049,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_NOT_RETAINED = 0xC038004A,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_OFFLINE = 0xC038004B,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_RETAINED = 0xC038004C,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NUMBER_OF_EXTENTS_INVALID = 0xC038004D,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_DIFFERENT_SECTOR_SIZE = 0xC038004E,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_BAD_BOOT_DISK = 0xC038004F,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_CONFIG_OFFLINE = 0xC0380050,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_CONFIG_ONLINE = 0xC0380051,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NOT_PRIMARY_PACK = 0xC0380052,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PACK_LOG_UPDATE_FAILED = 0xC0380053,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NUMBER_OF_DISKS_IN_PLEX_INVALID = 0xC0380054,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NUMBER_OF_DISKS_IN_MEMBER_INVALID = 0xC0380055,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_VOLUME_MIRRORED = 0xC0380056,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PLEX_NOT_SIMPLE_SPANNED = 0xC0380057,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NO_VALID_LOG_COPIES = 0xC0380058,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_PRIMARY_PACK_PRESENT = 0xC0380059,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_NUMBER_OF_DISKS_INVALID = 0xC038005A,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_MIRROR_NOT_SUPPORTED = 0xC038005B,
  MD_NTSTATUS_WIN_STATUS_VOLMGR_RAID5_NOT_SUPPORTED = 0xC038005C,
  MD_NTSTATUS_WIN_STATUS_BCD_TOO_MANY_ELEMENTS = 0xC0390002,
  MD_NTSTATUS_WIN_STATUS_VHD_DRIVE_FOOTER_MISSING = 0xC03A0001,
  MD_NTSTATUS_WIN_STATUS_VHD_DRIVE_FOOTER_CHECKSUM_MISMATCH = 0xC03A0002,
  MD_NTSTATUS_WIN_STATUS_VHD_DRIVE_FOOTER_CORRUPT = 0xC03A0003,
  MD_NTSTATUS_WIN_STATUS_VHD_FORMAT_UNKNOWN = 0xC03A0004,
  MD_NTSTATUS_WIN_STATUS_VHD_FORMAT_UNSUPPORTED_VERSION = 0xC03A0005,
  MD_NTSTATUS_WIN_STATUS_VHD_SPARSE_HEADER_CHECKSUM_MISMATCH = 0xC03A0006,
  MD_NTSTATUS_WIN_STATUS_VHD_SPARSE_HEADER_UNSUPPORTED_VERSION = 0xC03A0007,
  MD_NTSTATUS_WIN_STATUS_VHD_SPARSE_HEADER_CORRUPT = 0xC03A0008,
  MD_NTSTATUS_WIN_STATUS_VHD_BLOCK_ALLOCATION_FAILURE = 0xC03A0009,
  MD_NTSTATUS_WIN_STATUS_VHD_BLOCK_ALLOCATION_TABLE_CORRUPT = 0xC03A000A,
  MD_NTSTATUS_WIN_STATUS_VHD_INVALID_BLOCK_SIZE = 0xC03A000B,
  MD_NTSTATUS_WIN_STATUS_VHD_BITMAP_MISMATCH = 0xC03A000C,
  MD_NTSTATUS_WIN_STATUS_VHD_PARENT_VHD_NOT_FOUND = 0xC03A000D,
  MD_NTSTATUS_WIN_STATUS_VHD_CHILD_PARENT_ID_MISMATCH = 0xC03A000E,
  MD_NTSTATUS_WIN_STATUS_VHD_CHILD_PARENT_TIMESTAMP_MISMATCH = 0xC03A000F,
  MD_NTSTATUS_WIN_STATUS_VHD_METADATA_READ_FAILURE = 0xC03A0010,
  MD_NTSTATUS_WIN_STATUS_VHD_METADATA_WRITE_FAILURE = 0xC03A0011,
  MD_NTSTATUS_WIN_STATUS_VHD_INVALID_SIZE = 0xC03A0012,
  MD_NTSTATUS_WIN_STATUS_VHD_INVALID_FILE_SIZE = 0xC03A0013,
  MD_NTSTATUS_WIN_STATUS_VIRTDISK_PROVIDER_NOT_FOUND = 0xC03A0014,
  MD_NTSTATUS_WIN_STATUS_VIRTDISK_NOT_VIRTUAL_DISK = 0xC03A0015,
  MD_NTSTATUS_WIN_STATUS_VHD_PARENT_VHD_ACCESS_DENIED = 0xC03A0016,
  MD_NTSTATUS_WIN_STATUS_VHD_CHILD_PARENT_SIZE_MISMATCH = 0xC03A0017,
  MD_NTSTATUS_WIN_STATUS_VHD_DIFFERENCING_CHAIN_CYCLE_DETECTED = 0xC03A0018,
  MD_NTSTATUS_WIN_STATUS_VHD_DIFFERENCING_CHAIN_ERROR_IN_PARENT = 0xC03A0019,
  MD_NTSTATUS_WIN_STATUS_VIRTUAL_DISK_LIMITATION = 0xC03A001A,
  MD_NTSTATUS_WIN_STATUS_VHD_INVALID_TYPE = 0xC03A001B,
  MD_NTSTATUS_WIN_STATUS_VHD_INVALID_STATE = 0xC03A001C,
  MD_NTSTATUS_WIN_STATUS_VIRTDISK_UNSUPPORTED_DISK_SECTOR_SIZE = 0xC03A001D,
  MD_NTSTATUS_WIN_STATUS_VIRTDISK_DISK_ALREADY_OWNED = 0xC03A001E,
  MD_NTSTATUS_WIN_STATUS_VIRTDISK_DISK_ONLINE_AND_WRITABLE = 0xC03A001F,
  MD_NTSTATUS_WIN_STATUS_CTLOG_TRACKING_NOT_INITIALIZED = 0xC03A0020,
  MD_NTSTATUS_WIN_STATUS_CTLOG_LOGFILE_SIZE_EXCEEDED_MAXSIZE = 0xC03A0021,
  MD_NTSTATUS_WIN_STATUS_CTLOG_VHD_CHANGED_OFFLINE = 0xC03A0022,
  MD_NTSTATUS_WIN_STATUS_CTLOG_INVALID_TRACKING_STATE = 0xC03A0023,
  MD_NTSTATUS_WIN_STATUS_CTLOG_INCONSISTENT_TRACKING_FILE = 0xC03A0024,
  MD_NTSTATUS_WIN_STATUS_VHD_METADATA_FULL = 0xC03A0028,
  MD_NTSTATUS_WIN_STATUS_VHD_INVALID_CHANGE_TRACKING_ID = 0xC03A0029,
  MD_NTSTATUS_WIN_STATUS_VHD_CHANGE_TRACKING_DISABLED = 0xC03A002A,
  MD_NTSTATUS_WIN_STATUS_VHD_MISSING_CHANGE_TRACKING_INFORMATION = 0xC03A0030,
  MD_NTSTATUS_WIN_STATUS_VHD_RESIZE_WOULD_TRUNCATE_DATA = 0xC03A0031,
  MD_NTSTATUS_WIN_STATUS_VHD_COULD_NOT_COMPUTE_MINIMUM_VIRTUAL_SIZE = 0xC03A0032,
  MD_NTSTATUS_WIN_STATUS_VHD_ALREADY_AT_OR_BELOW_MINIMUM_VIRTUAL_SIZE = 0xC03A0033,
  MD_NTSTATUS_WIN_STATUS_RKF_KEY_NOT_FOUND = 0xC0400001,
  MD_NTSTATUS_WIN_STATUS_RKF_DUPLICATE_KEY = 0xC0400002,
  MD_NTSTATUS_WIN_STATUS_RKF_BLOB_FULL = 0xC0400003,
  MD_NTSTATUS_WIN_STATUS_RKF_STORE_FULL = 0xC0400004,
  MD_NTSTATUS_WIN_STATUS_RKF_FILE_BLOCKED = 0xC0400005,
  MD_NTSTATUS_WIN_STATUS_RKF_ACTIVE_KEY = 0xC0400006,
  MD_NTSTATUS_WIN_STATUS_RDBSS_RESTART_OPERATION = 0xC0410001,
  MD_NTSTATUS_WIN_STATUS_RDBSS_CONTINUE_OPERATION = 0xC0410002,
  MD_NTSTATUS_WIN_STATUS_RDBSS_POST_OPERATION = 0xC0410003,
  MD_NTSTATUS_WIN_STATUS_RDBSS_RETRY_LOOKUP = 0xC0410004,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INVALID_HANDLE = 0xC0420001,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_READ_NOT_PERMITTED = 0xC0420002,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_WRITE_NOT_PERMITTED = 0xC0420003,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INVALID_PDU = 0xC0420004,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INSUFFICIENT_AUTHENTICATION = 0xC0420005,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_REQUEST_NOT_SUPPORTED = 0xC0420006,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INVALID_OFFSET = 0xC0420007,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INSUFFICIENT_AUTHORIZATION = 0xC0420008,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_PREPARE_QUEUE_FULL = 0xC0420009,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_ATTRIBUTE_NOT_FOUND = 0xC042000A,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_ATTRIBUTE_NOT_LONG = 0xC042000B,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INSUFFICIENT_ENCRYPTION_KEY_SIZE = 0xC042000C,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INVALID_ATTRIBUTE_VALUE_LENGTH = 0xC042000D,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_UNLIKELY = 0xC042000E,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INSUFFICIENT_ENCRYPTION = 0xC042000F,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_UNSUPPORTED_GROUP_TYPE = 0xC0420010,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_INSUFFICIENT_RESOURCES = 0xC0420011,
  MD_NTSTATUS_WIN_STATUS_BTH_ATT_UNKNOWN_ERROR = 0xC0421000,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_ROLLBACK_DETECTED = 0xC0430001,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_VIOLATION = 0xC0430002,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_INVALID_POLICY = 0xC0430003,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_PUBLISHER_NOT_FOUND = 0xC0430004,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_NOT_SIGNED = 0xC0430005,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_FILE_REPLACED = 0xC0430007,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_NOT_AUTHORIZED = 0xC0430008,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_UNKNOWN = 0xC0430009,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_MISSING_ANTIROLLBACKVERSION = 0xC043000A,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_PLATFORM_ID_MISMATCH = 0xC043000B,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_ROLLBACK_DETECTED = 0xC043000C,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_POLICY_UPGRADE_MISMATCH = 0xC043000D,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_REQUIRED_POLICY_FILE_MISSING = 0xC043000E,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_NOT_BASE_POLICY = 0xC043000F,
  MD_NTSTATUS_WIN_STATUS_SECUREBOOT_NOT_SUPPLEMENTAL_POLICY = 0xC0430010,
  MD_NTSTATUS_WIN_STATUS_AUDIO_ENGINE_NODE_NOT_FOUND = 0xC0440001,
  MD_NTSTATUS_WIN_STATUS_HDAUDIO_EMPTY_CONNECTION_LIST = 0xC0440002,
  MD_NTSTATUS_WIN_STATUS_HDAUDIO_CONNECTION_LIST_NOT_SUPPORTED = 0xC0440003,
  MD_NTSTATUS_WIN_STATUS_HDAUDIO_NO_LOGICAL_DEVICES_CREATED = 0xC0440004,
  MD_NTSTATUS_WIN_STATUS_HDAUDIO_NULL_LINKED_LIST_ENTRY = 0xC0440005,
  MD_NTSTATUS_WIN_STATUS_VSM_NOT_INITIALIZED = 0xC0450000,
  MD_NTSTATUS_WIN_STATUS_VSM_DMA_PROTECTION_NOT_IN_USE = 0xC0450001,
  MD_NTSTATUS_WIN_STATUS_VOLSNAP_BOOTFILE_NOT_VALID = 0xC0500003,
  MD_NTSTATUS_WIN_STATUS_VOLSNAP_ACTIVATION_TIMEOUT = 0xC0500004,
  MD_NTSTATUS_WIN_STATUS_IO_PREEMPTED = 0xC0510001,
  MD_NTSTATUS_WIN_STATUS_SVHDX_ERROR_STORED = 0xC05C0000,
  MD_NTSTATUS_WIN_STATUS_SVHDX_ERROR_NOT_AVAILABLE = 0xC05CFF00,
  MD_NTSTATUS_WIN_STATUS_SVHDX_UNIT_ATTENTION_AVAILABLE = 0xC05CFF01,
  MD_NTSTATUS_WIN_STATUS_SVHDX_UNIT_ATTENTION_CAPACITY_DATA_CHANGED = 0xC05CFF02,
  MD_NTSTATUS_WIN_STATUS_SVHDX_UNIT_ATTENTION_RESERVATIONS_PREEMPTED = 0xC05CFF03,
  MD_NTSTATUS_WIN_STATUS_SVHDX_UNIT_ATTENTION_RESERVATIONS_RELEASED = 0xC05CFF04,
  MD_NTSTATUS_WIN_STATUS_SVHDX_UNIT_ATTENTION_REGISTRATIONS_PREEMPTED = 0xC05CFF05,
  MD_NTSTATUS_WIN_STATUS_SVHDX_UNIT_ATTENTION_OPERATING_DEFINITION_CHANGED = 0xC05CFF06,
  MD_NTSTATUS_WIN_STATUS_SVHDX_RESERVATION_CONFLICT = 0xC05CFF07,
  MD_NTSTATUS_WIN_STATUS_SVHDX_WRONG_FILE_TYPE = 0xC05CFF08,
  MD_NTSTATUS_WIN_STATUS_SVHDX_VERSION_MISMATCH = 0xC05CFF09,
  MD_NTSTATUS_WIN_STATUS_VHD_SHARED = 0xC05CFF0A,
  MD_NTSTATUS_WIN_STATUS_SVHDX_NO_INITIATOR = 0xC05CFF0B,
  MD_NTSTATUS_WIN_STATUS_VHDSET_BACKING_STORAGE_NOT_FOUND = 0xC05CFF0C,
  MD_NTSTATUS_WIN_STATUS_SMB_NO_PREAUTH_INTEGRITY_HASH_OVERLAP = 0xC05D0000,
  MD_NTSTATUS_WIN_STATUS_SMB_BAD_CLUSTER_DIALECT = 0xC05D0001,
  MD_NTSTATUS_WIN_STATUS_SMB_GUEST_LOGON_BLOCKED = 0xC05D0002,
  MD_NTSTATUS_WIN_STATUS_SPACES_FAULT_DOMAIN_TYPE_INVALID = 0xC0E70001,
  MD_NTSTATUS_WIN_STATUS_SPACES_RESILIENCY_TYPE_INVALID = 0xC0E70003,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRIVE_SECTOR_SIZE_INVALID = 0xC0E70004,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRIVE_REDUNDANCY_INVALID = 0xC0E70006,
  MD_NTSTATUS_WIN_STATUS_SPACES_NUMBER_OF_DATA_COPIES_INVALID = 0xC0E70007,
  MD_NTSTATUS_WIN_STATUS_SPACES_INTERLEAVE_LENGTH_INVALID = 0xC0E70009,
  MD_NTSTATUS_WIN_STATUS_SPACES_NUMBER_OF_COLUMNS_INVALID = 0xC0E7000A,
  MD_NTSTATUS_WIN_STATUS_SPACES_NOT_ENOUGH_DRIVES = 0xC0E7000B,
  MD_NTSTATUS_WIN_STATUS_SPACES_EXTENDED_ERROR = 0xC0E7000C,
  MD_NTSTATUS_WIN_STATUS_SPACES_PROVISIONING_TYPE_INVALID = 0xC0E7000D,
  MD_NTSTATUS_WIN_STATUS_SPACES_ALLOCATION_SIZE_INVALID = 0xC0E7000E,
  MD_NTSTATUS_WIN_STATUS_SPACES_ENCLOSURE_AWARE_INVALID = 0xC0E7000F,
  MD_NTSTATUS_WIN_STATUS_SPACES_WRITE_CACHE_SIZE_INVALID = 0xC0E70010,
  MD_NTSTATUS_WIN_STATUS_SPACES_NUMBER_OF_GROUPS_INVALID = 0xC0E70011,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRIVE_OPERATIONAL_STATE_INVALID = 0xC0E70012,
  MD_NTSTATUS_WIN_STATUS_SPACES_UPDATE_COLUMN_STATE = 0xC0E70013,
  MD_NTSTATUS_WIN_STATUS_SPACES_MAP_REQUIRED = 0xC0E70014,
  MD_NTSTATUS_WIN_STATUS_SPACES_UNSUPPORTED_VERSION = 0xC0E70015,
  MD_NTSTATUS_WIN_STATUS_SPACES_CORRUPT_METADATA = 0xC0E70016,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRT_FULL = 0xC0E70017,
  MD_NTSTATUS_WIN_STATUS_SPACES_INCONSISTENCY = 0xC0E70018,
  MD_NTSTATUS_WIN_STATUS_SPACES_LOG_NOT_READY = 0xC0E70019,
  MD_NTSTATUS_WIN_STATUS_SPACES_NO_REDUNDANCY = 0xC0E7001A,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRIVE_NOT_READY = 0xC0E7001B,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRIVE_SPLIT = 0xC0E7001C,
  MD_NTSTATUS_WIN_STATUS_SPACES_DRIVE_LOST_DATA = 0xC0E7001D,
  MD_NTSTATUS_WIN_STATUS_SPACES_ENTRY_INCOMPLETE = 0xC0E7001E,
  MD_NTSTATUS_WIN_STATUS_SPACES_ENTRY_INVALID = 0xC0E7001F,
  MD_NTSTATUS_WIN_STATUS_SPACES_MARK_DIRTY = 0xC0E70020,
  MD_NTSTATUS_WIN_STATUS_SECCORE_INVALID_COMMAND = 0xC0E80000,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_INTEGRITY_ROLLBACK_DETECTED = 0xC0E90001,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_INTEGRITY_POLICY_VIOLATION = 0xC0E90002,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_INTEGRITY_INVALID_POLICY = 0xC0E90003,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_INTEGRITY_POLICY_NOT_SIGNED = 0xC0E90004,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_INTEGRITY_TOO_MANY_POLICIES = 0xC0E90005,
  MD_NTSTATUS_WIN_STATUS_SYSTEM_INTEGRITY_SUPPLEMENTAL_POLICY_NOT_AUTHORIZED = 0xC0E90006,
  MD_NTSTATUS_WIN_STATUS_NO_APPLICABLE_APP_LICENSES_FOUND = 0xC0EA0001,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_NOT_FOUND = 0xC0EA0002,
  MD_NTSTATUS_WIN_STATUS_CLIP_DEVICE_LICENSE_MISSING = 0xC0EA0003,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_INVALID_SIGNATURE = 0xC0EA0004,
  MD_NTSTATUS_WIN_STATUS_CLIP_KEYHOLDER_LICENSE_MISSING_OR_INVALID = 0xC0EA0005,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_EXPIRED = 0xC0EA0006,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_SIGNED_BY_UNKNOWN_SOURCE = 0xC0EA0007,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_NOT_SIGNED = 0xC0EA0008,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_HARDWARE_ID_OUT_OF_TOLERANCE = 0xC0EA0009,
  MD_NTSTATUS_WIN_STATUS_CLIP_LICENSE_DEVICE_ID_MISMATCH = 0xC0EA000A,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_NOT_AUTHORIZED = 0xC0EB0001,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_INVALID = 0xC0EB0002,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_FILE_NOT_AUTHORIZED = 0xC0EB0003,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_CATALOG_NOT_AUTHORIZED = 0xC0EB0004,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_BINARY_ID_NOT_FOUND = 0xC0EB0005,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_NOT_ACTIVE = 0xC0EB0006,
  MD_NTSTATUS_WIN_STATUS_PLATFORM_MANIFEST_NOT_SIGNED = 0xC0EB0007,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_CONDITION_NOT_SATISFIED = 0xC0EC0000,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_HANDLE_INVALIDATED = 0xC0EC0001,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_INVALID_HOST_GENERATION = 0xC0EC0002,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_UNEXPECTED_PROCESS_REGISTRATION = 0xC0EC0003,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_INVALID_HOST_STATE = 0xC0EC0004,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_NO_DONOR = 0xC0EC0005,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_HOST_ID_MISMATCH = 0xC0EC0006,
  MD_NTSTATUS_WIN_STATUS_APPEXEC_UNKNOWN_USER = 0xC0EC0007,
} MDNTStatusCodeWin;

// These constants are defined in the MSDN documentation of
// the EXCEPTION_RECORD structure.
typedef enum {
  MD_ACCESS_VIOLATION_WIN_READ  = 0,
  MD_ACCESS_VIOLATION_WIN_WRITE = 1,
  MD_ACCESS_VIOLATION_WIN_EXEC  = 8
} MDAccessViolationTypeWin;

// These constants are defined in the MSDN documentation of
// the EXCEPTION_RECORD structure.
typedef enum {
  MD_IN_PAGE_ERROR_WIN_READ  = 0,
  MD_IN_PAGE_ERROR_WIN_WRITE = 1,
  MD_IN_PAGE_ERROR_WIN_EXEC  = 8
} MDInPageErrorTypeWin;

/* For (MDException).exception_information[0], when (MDException).exception_code
 * is MD_EXCEPTION_CODE_WIN_STACK_BUFFER_OVERRUN. This describes the underlying
 * reason for the crash. These values come from winnt.h.
 *
 * The content of this enum was created from winnt.h in the 10 SDK
 * (version 10.0.19041.0) with
 *
 * egrep '#define FAST_FAIL_[A-Z_0-9]+\s+[0-9]' winnt.h
 * | tr -d '\r'
 * | sed -r 's@#define FAST_FAIL_([A-Z_0-9]+)\s+([0-9]+).*@\2 \1@'
 * | sed -r 's@([0-9]+) ([A-Z_0-9]+)@  MD_FAST_FAIL_WIN_\2 = \1,@' */
typedef enum {
  MD_FAST_FAIL_WIN_LEGACY_GS_VIOLATION = 0,
  MD_FAST_FAIL_WIN_VTGUARD_CHECK_FAILURE = 1,
  MD_FAST_FAIL_WIN_STACK_COOKIE_CHECK_FAILURE = 2,
  MD_FAST_FAIL_WIN_CORRUPT_LIST_ENTRY = 3,
  MD_FAST_FAIL_WIN_INCORRECT_STACK = 4,
  MD_FAST_FAIL_WIN_INVALID_ARG = 5,
  MD_FAST_FAIL_WIN_GS_COOKIE_INIT = 6,
  MD_FAST_FAIL_WIN_FATAL_APP_EXIT = 7,
  MD_FAST_FAIL_WIN_RANGE_CHECK_FAILURE = 8,
  MD_FAST_FAIL_WIN_UNSAFE_REGISTRY_ACCESS = 9,
  MD_FAST_FAIL_WIN_GUARD_ICALL_CHECK_FAILURE = 10,
  MD_FAST_FAIL_WIN_GUARD_WRITE_CHECK_FAILURE = 11,
  MD_FAST_FAIL_WIN_INVALID_FIBER_SWITCH = 12,
  MD_FAST_FAIL_WIN_INVALID_SET_OF_CONTEXT = 13,
  MD_FAST_FAIL_WIN_INVALID_REFERENCE_COUNT = 14,
  MD_FAST_FAIL_WIN_INVALID_JUMP_BUFFER = 18,
  MD_FAST_FAIL_WIN_MRDATA_MODIFIED = 19,
  MD_FAST_FAIL_WIN_CERTIFICATION_FAILURE = 20,
  MD_FAST_FAIL_WIN_INVALID_EXCEPTION_CHAIN = 21,
  MD_FAST_FAIL_WIN_CRYPTO_LIBRARY = 22,
  MD_FAST_FAIL_WIN_INVALID_CALL_IN_DLL_CALLOUT = 23,
  MD_FAST_FAIL_WIN_INVALID_IMAGE_BASE = 24,
  MD_FAST_FAIL_WIN_DLOAD_PROTECTION_FAILURE = 25,
  MD_FAST_FAIL_WIN_UNSAFE_EXTENSION_CALL = 26,
  MD_FAST_FAIL_WIN_DEPRECATED_SERVICE_INVOKED = 27,
  MD_FAST_FAIL_WIN_INVALID_BUFFER_ACCESS = 28,
  MD_FAST_FAIL_WIN_INVALID_BALANCED_TREE = 29,
  MD_FAST_FAIL_WIN_INVALID_NEXT_THREAD = 30,
  MD_FAST_FAIL_WIN_GUARD_ICALL_CHECK_SUPPRESSED = 31,
  MD_FAST_FAIL_WIN_APCS_DISABLED = 32,
  MD_FAST_FAIL_WIN_INVALID_IDLE_STATE = 33,
  MD_FAST_FAIL_WIN_MRDATA_PROTECTION_FAILURE = 34,
  MD_FAST_FAIL_WIN_UNEXPECTED_HEAP_EXCEPTION = 35,
  MD_FAST_FAIL_WIN_INVALID_LOCK_STATE = 36,
  MD_FAST_FAIL_WIN_GUARD_JUMPTABLE = 37,
  MD_FAST_FAIL_WIN_INVALID_LONGJUMP_TARGET = 38,
  MD_FAST_FAIL_WIN_INVALID_DISPATCH_CONTEXT = 39,
  MD_FAST_FAIL_WIN_INVALID_THREAD = 40,
  MD_FAST_FAIL_WIN_INVALID_SYSCALL_NUMBER = 41,
  MD_FAST_FAIL_WIN_INVALID_FILE_OPERATION = 42,
  MD_FAST_FAIL_WIN_LPAC_ACCESS_DENIED = 43,
  MD_FAST_FAIL_WIN_GUARD_SS_FAILURE = 44,
  MD_FAST_FAIL_WIN_LOADER_CONTINUITY_FAILURE = 45,
  MD_FAST_FAIL_WIN_GUARD_EXPORT_SUPPRESSION_FAILURE = 46,
  MD_FAST_FAIL_WIN_INVALID_CONTROL_STACK = 47,
  MD_FAST_FAIL_WIN_SET_CONTEXT_DENIED = 48,
  MD_FAST_FAIL_WIN_INVALID_IAT = 49,
  MD_FAST_FAIL_WIN_HEAP_METADATA_CORRUPTION = 50,
  MD_FAST_FAIL_WIN_PAYLOAD_RESTRICTION_VIOLATION = 51,
  MD_FAST_FAIL_WIN_LOW_LABEL_ACCESS_DENIED = 52,
  MD_FAST_FAIL_WIN_ENCLAVE_CALL_FAILURE = 53,
  MD_FAST_FAIL_WIN_UNHANDLED_LSS_EXCEPTON = 54,
  MD_FAST_FAIL_WIN_ADMINLESS_ACCESS_DENIED = 55,
  MD_FAST_FAIL_WIN_UNEXPECTED_CALL = 56,
  MD_FAST_FAIL_WIN_CONTROL_INVALID_RETURN_ADDRESS = 57,
  MD_FAST_FAIL_WIN_UNEXPECTED_HOST_BEHAVIOR = 58,
  MD_FAST_FAIL_WIN_FLAGS_CORRUPTION = 59,
  MD_FAST_FAIL_WIN_VEH_CORRUPTION = 60,
  MD_FAST_FAIL_WIN_ETW_CORRUPTION = 61,
  MD_FAST_FAIL_WIN_RIO_ABORT = 62,
  MD_FAST_FAIL_WIN_INVALID_PFN = 63,
} MDFastFailWin;

/* For various (MDException).exception_information entries. This describes the
* underlying reason for the crash. These values come from winerror.h.
 *
 * The content of this enum was created from winnt.h in the 10 SDK
 * (version 10.0.19041.0) with
 *
 * egrep -o '#define ERROR_[A-Z_0-9]+\s+[0-9]+L' winerror.h
 * | tr -d '\r'
 * | sed -r 's@#define ERROR_([A-Z_0-9]+)\s+([0-9]+)L@\2 \1@'
 * | sort -n
 * | sed -r 's@([0-9]+) ([A-Z_0-9]+)@  MD_ERROR_WIN_\2 = \1,@' */
typedef enum {
  MD_ERROR_WIN_SUCCESS = 0,
  MD_ERROR_WIN_INVALID_FUNCTION = 1,
  MD_ERROR_WIN_FILE_NOT_FOUND = 2,
  MD_ERROR_WIN_PATH_NOT_FOUND = 3,
  MD_ERROR_WIN_TOO_MANY_OPEN_FILES = 4,
  MD_ERROR_WIN_ACCESS_DENIED = 5,
  MD_ERROR_WIN_INVALID_HANDLE = 6,
  MD_ERROR_WIN_ARENA_TRASHED = 7,
  MD_ERROR_WIN_NOT_ENOUGH_MEMORY = 8,
  MD_ERROR_WIN_INVALID_BLOCK = 9,
  MD_ERROR_WIN_BAD_ENVIRONMENT = 10,
  MD_ERROR_WIN_BAD_FORMAT = 11,
  MD_ERROR_WIN_INVALID_ACCESS = 12,
  MD_ERROR_WIN_INVALID_DATA = 13,
  MD_ERROR_WIN_OUTOFMEMORY = 14,
  MD_ERROR_WIN_INVALID_DRIVE = 15,
  MD_ERROR_WIN_CURRENT_DIRECTORY = 16,
  MD_ERROR_WIN_NOT_SAME_DEVICE = 17,
  MD_ERROR_WIN_NO_MORE_FILES = 18,
  MD_ERROR_WIN_WRITE_PROTECT = 19,
  MD_ERROR_WIN_BAD_UNIT = 20,
  MD_ERROR_WIN_NOT_READY = 21,
  MD_ERROR_WIN_BAD_COMMAND = 22,
  MD_ERROR_WIN_CRC = 23,
  MD_ERROR_WIN_BAD_LENGTH = 24,
  MD_ERROR_WIN_SEEK = 25,
  MD_ERROR_WIN_NOT_DOS_DISK = 26,
  MD_ERROR_WIN_SECTOR_NOT_FOUND = 27,
  MD_ERROR_WIN_OUT_OF_PAPER = 28,
  MD_ERROR_WIN_WRITE_FAULT = 29,
  MD_ERROR_WIN_READ_FAULT = 30,
  MD_ERROR_WIN_GEN_FAILURE = 31,
  MD_ERROR_WIN_SHARING_VIOLATION = 32,
  MD_ERROR_WIN_LOCK_VIOLATION = 33,
  MD_ERROR_WIN_WRONG_DISK = 34,
  MD_ERROR_WIN_SHARING_BUFFER_EXCEEDED = 36,
  MD_ERROR_WIN_HANDLE_EOF = 38,
  MD_ERROR_WIN_HANDLE_DISK_FULL = 39,
  MD_ERROR_WIN_NOT_SUPPORTED = 50,
  MD_ERROR_WIN_REM_NOT_LIST = 51,
  MD_ERROR_WIN_DUP_NAME = 52,
  MD_ERROR_WIN_BAD_NETPATH = 53,
  MD_ERROR_WIN_NETWORK_BUSY = 54,
  MD_ERROR_WIN_DEV_NOT_EXIST = 55,
  MD_ERROR_WIN_TOO_MANY_CMDS = 56,
  MD_ERROR_WIN_ADAP_HDW_ERR = 57,
  MD_ERROR_WIN_BAD_NET_RESP = 58,
  MD_ERROR_WIN_UNEXP_NET_ERR = 59,
  MD_ERROR_WIN_BAD_REM_ADAP = 60,
  MD_ERROR_WIN_PRINTQ_FULL = 61,
  MD_ERROR_WIN_NO_SPOOL_SPACE = 62,
  MD_ERROR_WIN_PRINT_CANCELLED = 63,
  MD_ERROR_WIN_NETNAME_DELETED = 64,
  MD_ERROR_WIN_NETWORK_ACCESS_DENIED = 65,
  MD_ERROR_WIN_BAD_DEV_TYPE = 66,
  MD_ERROR_WIN_BAD_NET_NAME = 67,
  MD_ERROR_WIN_TOO_MANY_NAMES = 68,
  MD_ERROR_WIN_TOO_MANY_SESS = 69,
  MD_ERROR_WIN_SHARING_PAUSED = 70,
  MD_ERROR_WIN_REQ_NOT_ACCEP = 71,
  MD_ERROR_WIN_REDIR_PAUSED = 72,
  MD_ERROR_WIN_FILE_EXISTS = 80,
  MD_ERROR_WIN_CANNOT_MAKE = 82,
  MD_ERROR_WIN_FAIL_I24 = 83,
  MD_ERROR_WIN_OUT_OF_STRUCTURES = 84,
  MD_ERROR_WIN_ALREADY_ASSIGNED = 85,
  MD_ERROR_WIN_INVALID_PASSWORD = 86,
  MD_ERROR_WIN_INVALID_PARAMETER = 87,
  MD_ERROR_WIN_NET_WRITE_FAULT = 88,
  MD_ERROR_WIN_NO_PROC_SLOTS = 89,
  MD_ERROR_WIN_TOO_MANY_SEMAPHORES = 100,
  MD_ERROR_WIN_EXCL_SEM_ALREADY_OWNED = 101,
  MD_ERROR_WIN_SEM_IS_SET = 102,
  MD_ERROR_WIN_TOO_MANY_SEM_REQUESTS = 103,
  MD_ERROR_WIN_INVALID_AT_INTERRUPT_TIME = 104,
  MD_ERROR_WIN_SEM_OWNER_DIED = 105,
  MD_ERROR_WIN_SEM_USER_LIMIT = 106,
  MD_ERROR_WIN_DISK_CHANGE = 107,
  MD_ERROR_WIN_DRIVE_LOCKED = 108,
  MD_ERROR_WIN_BROKEN_PIPE = 109,
  MD_ERROR_WIN_OPEN_FAILED = 110,
  MD_ERROR_WIN_BUFFER_OVERFLOW = 111,
  MD_ERROR_WIN_DISK_FULL = 112,
  MD_ERROR_WIN_NO_MORE_SEARCH_HANDLES = 113,
  MD_ERROR_WIN_INVALID_TARGET_HANDLE = 114,
  MD_ERROR_WIN_INVALID_CATEGORY = 117,
  MD_ERROR_WIN_INVALID_VERIFY_SWITCH = 118,
  MD_ERROR_WIN_BAD_DRIVER_LEVEL = 119,
  MD_ERROR_WIN_CALL_NOT_IMPLEMENTED = 120,
  MD_ERROR_WIN_SEM_TIMEOUT = 121,
  MD_ERROR_WIN_INSUFFICIENT_BUFFER = 122,
  MD_ERROR_WIN_INVALID_NAME = 123,
  MD_ERROR_WIN_INVALID_LEVEL = 124,
  MD_ERROR_WIN_NO_VOLUME_LABEL = 125,
  MD_ERROR_WIN_MOD_NOT_FOUND = 126,
  MD_ERROR_WIN_PROC_NOT_FOUND = 127,
  MD_ERROR_WIN_WAIT_NO_CHILDREN = 128,
  MD_ERROR_WIN_CHILD_NOT_COMPLETE = 129,
  MD_ERROR_WIN_DIRECT_ACCESS_HANDLE = 130,
  MD_ERROR_WIN_NEGATIVE_SEEK = 131,
  MD_ERROR_WIN_SEEK_ON_DEVICE = 132,
  MD_ERROR_WIN_IS_JOIN_TARGET = 133,
  MD_ERROR_WIN_IS_JOINED = 134,
  MD_ERROR_WIN_IS_SUBSTED = 135,
  MD_ERROR_WIN_NOT_JOINED = 136,
  MD_ERROR_WIN_NOT_SUBSTED = 137,
  MD_ERROR_WIN_JOIN_TO_JOIN = 138,
  MD_ERROR_WIN_SUBST_TO_SUBST = 139,
  MD_ERROR_WIN_JOIN_TO_SUBST = 140,
  MD_ERROR_WIN_SUBST_TO_JOIN = 141,
  MD_ERROR_WIN_BUSY_DRIVE = 142,
  MD_ERROR_WIN_SAME_DRIVE = 143,
  MD_ERROR_WIN_DIR_NOT_ROOT = 144,
  MD_ERROR_WIN_DIR_NOT_EMPTY = 145,
  MD_ERROR_WIN_IS_SUBST_PATH = 146,
  MD_ERROR_WIN_IS_JOIN_PATH = 147,
  MD_ERROR_WIN_PATH_BUSY = 148,
  MD_ERROR_WIN_IS_SUBST_TARGET = 149,
  MD_ERROR_WIN_SYSTEM_TRACE = 150,
  MD_ERROR_WIN_INVALID_EVENT_COUNT = 151,
  MD_ERROR_WIN_TOO_MANY_MUXWAITERS = 152,
  MD_ERROR_WIN_INVALID_LIST_FORMAT = 153,
  MD_ERROR_WIN_LABEL_TOO_LONG = 154,
  MD_ERROR_WIN_TOO_MANY_TCBS = 155,
  MD_ERROR_WIN_SIGNAL_REFUSED = 156,
  MD_ERROR_WIN_DISCARDED = 157,
  MD_ERROR_WIN_NOT_LOCKED = 158,
  MD_ERROR_WIN_BAD_THREADID_ADDR = 159,
  MD_ERROR_WIN_BAD_ARGUMENTS = 160,
  MD_ERROR_WIN_BAD_PATHNAME = 161,
  MD_ERROR_WIN_SIGNAL_PENDING = 162,
  MD_ERROR_WIN_MAX_THRDS_REACHED = 164,
  MD_ERROR_WIN_LOCK_FAILED = 167,
  MD_ERROR_WIN_BUSY = 170,
  MD_ERROR_WIN_DEVICE_SUPPORT_IN_PROGRESS = 171,
  MD_ERROR_WIN_CANCEL_VIOLATION = 173,
  MD_ERROR_WIN_ATOMIC_LOCKS_NOT_SUPPORTED = 174,
  MD_ERROR_WIN_INVALID_SEGMENT_NUMBER = 180,
  MD_ERROR_WIN_INVALID_ORDINAL = 182,
  MD_ERROR_WIN_ALREADY_EXISTS = 183,
  MD_ERROR_WIN_INVALID_FLAG_NUMBER = 186,
  MD_ERROR_WIN_SEM_NOT_FOUND = 187,
  MD_ERROR_WIN_INVALID_STARTING_CODESEG = 188,
  MD_ERROR_WIN_INVALID_STACKSEG = 189,
  MD_ERROR_WIN_INVALID_MODULETYPE = 190,
  MD_ERROR_WIN_INVALID_EXE_SIGNATURE = 191,
  MD_ERROR_WIN_EXE_MARKED_INVALID = 192,
  MD_ERROR_WIN_BAD_EXE_FORMAT = 193,
  MD_ERROR_WIN_INVALID_MINALLOCSIZE = 195,
  MD_ERROR_WIN_DYNLINK_FROM_INVALID_RING = 196,
  MD_ERROR_WIN_IOPL_NOT_ENABLED = 197,
  MD_ERROR_WIN_INVALID_SEGDPL = 198,
  MD_ERROR_WIN_RING2SEG_MUST_BE_MOVABLE = 200,
  MD_ERROR_WIN_RELOC_CHAIN_XEEDS_SEGLIM = 201,
  MD_ERROR_WIN_INFLOOP_IN_RELOC_CHAIN = 202,
  MD_ERROR_WIN_ENVVAR_NOT_FOUND = 203,
  MD_ERROR_WIN_NO_SIGNAL_SENT = 205,
  MD_ERROR_WIN_FILENAME_EXCED_RANGE = 206,
  MD_ERROR_WIN_RING2_STACK_IN_USE = 207,
  MD_ERROR_WIN_META_EXPANSION_TOO_LONG = 208,
  MD_ERROR_WIN_INVALID_SIGNAL_NUMBER = 209,
  MD_ERROR_WIN_THREAD_1_INACTIVE = 210,
  MD_ERROR_WIN_LOCKED = 212,
  MD_ERROR_WIN_TOO_MANY_MODULES = 214,
  MD_ERROR_WIN_NESTING_NOT_ALLOWED = 215,
  MD_ERROR_WIN_EXE_MACHINE_TYPE_MISMATCH = 216,
  MD_ERROR_WIN_EXE_CANNOT_MODIFY_SIGNED_BINARY = 217,
  MD_ERROR_WIN_EXE_CANNOT_MODIFY_STRONG_SIGNED_BINARY = 218,
  MD_ERROR_WIN_FILE_CHECKED_OUT = 220,
  MD_ERROR_WIN_CHECKOUT_REQUIRED = 221,
  MD_ERROR_WIN_BAD_FILE_TYPE = 222,
  MD_ERROR_WIN_FILE_TOO_LARGE = 223,
  MD_ERROR_WIN_FORMS_AUTH_REQUIRED = 224,
  MD_ERROR_WIN_VIRUS_INFECTED = 225,
  MD_ERROR_WIN_VIRUS_DELETED = 226,
  MD_ERROR_WIN_PIPE_LOCAL = 229,
  MD_ERROR_WIN_BAD_PIPE = 230,
  MD_ERROR_WIN_PIPE_BUSY = 231,
  MD_ERROR_WIN_NO_DATA = 232,
  MD_ERROR_WIN_PIPE_NOT_CONNECTED = 233,
  MD_ERROR_WIN_MORE_DATA = 234,
  MD_ERROR_WIN_NO_WORK_DONE = 235,
  MD_ERROR_WIN_VC_DISCONNECTED = 240,
  MD_ERROR_WIN_INVALID_EA_NAME = 254,
  MD_ERROR_WIN_EA_LIST_INCONSISTENT = 255,
  MD_ERROR_WIN_NO_MORE_ITEMS = 259,
  MD_ERROR_WIN_CANNOT_COPY = 266,
  MD_ERROR_WIN_DIRECTORY = 267,
  MD_ERROR_WIN_EAS_DIDNT_FIT = 275,
  MD_ERROR_WIN_EA_FILE_CORRUPT = 276,
  MD_ERROR_WIN_EA_TABLE_FULL = 277,
  MD_ERROR_WIN_INVALID_EA_HANDLE = 278,
  MD_ERROR_WIN_EAS_NOT_SUPPORTED = 282,
  MD_ERROR_WIN_NOT_OWNER = 288,
  MD_ERROR_WIN_TOO_MANY_POSTS = 298,
  MD_ERROR_WIN_PARTIAL_COPY = 299,
  MD_ERROR_WIN_OPLOCK_NOT_GRANTED = 300,
  MD_ERROR_WIN_INVALID_OPLOCK_PROTOCOL = 301,
  MD_ERROR_WIN_DISK_TOO_FRAGMENTED = 302,
  MD_ERROR_WIN_DELETE_PENDING = 303,
  MD_ERROR_WIN_INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING = 304,
  MD_ERROR_WIN_SHORT_NAMES_NOT_ENABLED_ON_VOLUME = 305,
  MD_ERROR_WIN_SECURITY_STREAM_IS_INCONSISTENT = 306,
  MD_ERROR_WIN_INVALID_LOCK_RANGE = 307,
  MD_ERROR_WIN_IMAGE_SUBSYSTEM_NOT_PRESENT = 308,
  MD_ERROR_WIN_NOTIFICATION_GUID_ALREADY_DEFINED = 309,
  MD_ERROR_WIN_INVALID_EXCEPTION_HANDLER = 310,
  MD_ERROR_WIN_DUPLICATE_PRIVILEGES = 311,
  MD_ERROR_WIN_NO_RANGES_PROCESSED = 312,
  MD_ERROR_WIN_NOT_ALLOWED_ON_SYSTEM_FILE = 313,
  MD_ERROR_WIN_DISK_RESOURCES_EXHAUSTED = 314,
  MD_ERROR_WIN_INVALID_TOKEN = 315,
  MD_ERROR_WIN_DEVICE_FEATURE_NOT_SUPPORTED = 316,
  MD_ERROR_WIN_MR_MID_NOT_FOUND = 317,
  MD_ERROR_WIN_SCOPE_NOT_FOUND = 318,
  MD_ERROR_WIN_UNDEFINED_SCOPE = 319,
  MD_ERROR_WIN_INVALID_CAP = 320,
  MD_ERROR_WIN_DEVICE_UNREACHABLE = 321,
  MD_ERROR_WIN_DEVICE_NO_RESOURCES = 322,
  MD_ERROR_WIN_DATA_CHECKSUM_ERROR = 323,
  MD_ERROR_WIN_INTERMIXED_KERNEL_EA_OPERATION = 324,
  MD_ERROR_WIN_FILE_LEVEL_TRIM_NOT_SUPPORTED = 326,
  MD_ERROR_WIN_OFFSET_ALIGNMENT_VIOLATION = 327,
  MD_ERROR_WIN_INVALID_FIELD_IN_PARAMETER_LIST = 328,
  MD_ERROR_WIN_OPERATION_IN_PROGRESS = 329,
  MD_ERROR_WIN_BAD_DEVICE_PATH = 330,
  MD_ERROR_WIN_TOO_MANY_DESCRIPTORS = 331,
  MD_ERROR_WIN_SCRUB_DATA_DISABLED = 332,
  MD_ERROR_WIN_NOT_REDUNDANT_STORAGE = 333,
  MD_ERROR_WIN_RESIDENT_FILE_NOT_SUPPORTED = 334,
  MD_ERROR_WIN_COMPRESSED_FILE_NOT_SUPPORTED = 335,
  MD_ERROR_WIN_DIRECTORY_NOT_SUPPORTED = 336,
  MD_ERROR_WIN_NOT_READ_FROM_COPY = 337,
  MD_ERROR_WIN_FT_WRITE_FAILURE = 338,
  MD_ERROR_WIN_FT_DI_SCAN_REQUIRED = 339,
  MD_ERROR_WIN_INVALID_KERNEL_INFO_VERSION = 340,
  MD_ERROR_WIN_INVALID_PEP_INFO_VERSION = 341,
  MD_ERROR_WIN_OBJECT_NOT_EXTERNALLY_BACKED = 342,
  MD_ERROR_WIN_EXTERNAL_BACKING_PROVIDER_UNKNOWN = 343,
  MD_ERROR_WIN_COMPRESSION_NOT_BENEFICIAL = 344,
  MD_ERROR_WIN_STORAGE_TOPOLOGY_ID_MISMATCH = 345,
  MD_ERROR_WIN_BLOCKED_BY_PARENTAL_CONTROLS = 346,
  MD_ERROR_WIN_BLOCK_TOO_MANY_REFERENCES = 347,
  MD_ERROR_WIN_MARKED_TO_DISALLOW_WRITES = 348,
  MD_ERROR_WIN_ENCLAVE_FAILURE = 349,
  MD_ERROR_WIN_FAIL_NOACTION_REBOOT = 350,
  MD_ERROR_WIN_FAIL_SHUTDOWN = 351,
  MD_ERROR_WIN_FAIL_RESTART = 352,
  MD_ERROR_WIN_MAX_SESSIONS_REACHED = 353,
  MD_ERROR_WIN_NETWORK_ACCESS_DENIED_EDP = 354,
  MD_ERROR_WIN_DEVICE_HINT_NAME_BUFFER_TOO_SMALL = 355,
  MD_ERROR_WIN_EDP_POLICY_DENIES_OPERATION = 356,
  MD_ERROR_WIN_EDP_DPL_POLICY_CANT_BE_SATISFIED = 357,
  MD_ERROR_WIN_CLOUD_FILE_SYNC_ROOT_METADATA_CORRUPT = 358,
  MD_ERROR_WIN_DEVICE_IN_MAINTENANCE = 359,
  MD_ERROR_WIN_NOT_SUPPORTED_ON_DAX = 360,
  MD_ERROR_WIN_DAX_MAPPING_EXISTS = 361,
  MD_ERROR_WIN_CLOUD_FILE_PROVIDER_NOT_RUNNING = 362,
  MD_ERROR_WIN_CLOUD_FILE_METADATA_CORRUPT = 363,
  MD_ERROR_WIN_CLOUD_FILE_METADATA_TOO_LARGE = 364,
  MD_ERROR_WIN_CLOUD_FILE_PROPERTY_BLOB_TOO_LARGE = 365,
  MD_ERROR_WIN_CLOUD_FILE_PROPERTY_BLOB_CHECKSUM_MISMATCH = 366,
  MD_ERROR_WIN_CHILD_PROCESS_BLOCKED = 367,
  MD_ERROR_WIN_STORAGE_LOST_DATA_PERSISTENCE = 368,
  MD_ERROR_WIN_FILE_SYSTEM_VIRTUALIZATION_UNAVAILABLE = 369,
  MD_ERROR_WIN_FILE_SYSTEM_VIRTUALIZATION_METADATA_CORRUPT = 370,
  MD_ERROR_WIN_FILE_SYSTEM_VIRTUALIZATION_BUSY = 371,
  MD_ERROR_WIN_FILE_SYSTEM_VIRTUALIZATION_PROVIDER_UNKNOWN = 372,
  MD_ERROR_WIN_GDI_HANDLE_LEAK = 373,
  MD_ERROR_WIN_CLOUD_FILE_TOO_MANY_PROPERTY_BLOBS = 374,
  MD_ERROR_WIN_CLOUD_FILE_PROPERTY_VERSION_NOT_SUPPORTED = 375,
  MD_ERROR_WIN_NOT_A_CLOUD_FILE = 376,
  MD_ERROR_WIN_CLOUD_FILE_NOT_IN_SYNC = 377,
  MD_ERROR_WIN_CLOUD_FILE_ALREADY_CONNECTED = 378,
  MD_ERROR_WIN_CLOUD_FILE_NOT_SUPPORTED = 379,
  MD_ERROR_WIN_CLOUD_FILE_INVALID_REQUEST = 380,
  MD_ERROR_WIN_CLOUD_FILE_READ_ONLY_VOLUME = 381,
  MD_ERROR_WIN_CLOUD_FILE_CONNECTED_PROVIDER_ONLY = 382,
  MD_ERROR_WIN_CLOUD_FILE_VALIDATION_FAILED = 383,
  MD_ERROR_WIN_SMB1_NOT_AVAILABLE = 384,
  MD_ERROR_WIN_FILE_SYSTEM_VIRTUALIZATION_INVALID_OPERATION = 385,
  MD_ERROR_WIN_CLOUD_FILE_AUTHENTICATION_FAILED = 386,
  MD_ERROR_WIN_CLOUD_FILE_INSUFFICIENT_RESOURCES = 387,
  MD_ERROR_WIN_CLOUD_FILE_NETWORK_UNAVAILABLE = 388,
  MD_ERROR_WIN_CLOUD_FILE_UNSUCCESSFUL = 389,
  MD_ERROR_WIN_CLOUD_FILE_NOT_UNDER_SYNC_ROOT = 390,
  MD_ERROR_WIN_CLOUD_FILE_IN_USE = 391,
  MD_ERROR_WIN_CLOUD_FILE_PINNED = 392,
  MD_ERROR_WIN_CLOUD_FILE_REQUEST_ABORTED = 393,
  MD_ERROR_WIN_CLOUD_FILE_PROPERTY_CORRUPT = 394,
  MD_ERROR_WIN_CLOUD_FILE_ACCESS_DENIED = 395,
  MD_ERROR_WIN_CLOUD_FILE_INCOMPATIBLE_HARDLINKS = 396,
  MD_ERROR_WIN_CLOUD_FILE_PROPERTY_LOCK_CONFLICT = 397,
  MD_ERROR_WIN_CLOUD_FILE_REQUEST_CANCELED = 398,
  MD_ERROR_WIN_EXTERNAL_SYSKEY_NOT_SUPPORTED = 399,
  MD_ERROR_WIN_THREAD_MODE_ALREADY_BACKGROUND = 400,
  MD_ERROR_WIN_THREAD_MODE_NOT_BACKGROUND = 401,
  MD_ERROR_WIN_PROCESS_MODE_ALREADY_BACKGROUND = 402,
  MD_ERROR_WIN_PROCESS_MODE_NOT_BACKGROUND = 403,
  MD_ERROR_WIN_CLOUD_FILE_PROVIDER_TERMINATED = 404,
  MD_ERROR_WIN_NOT_A_CLOUD_SYNC_ROOT = 405,
  MD_ERROR_WIN_FILE_PROTECTED_UNDER_DPL = 406,
  MD_ERROR_WIN_VOLUME_NOT_CLUSTER_ALIGNED = 407,
  MD_ERROR_WIN_NO_PHYSICALLY_ALIGNED_FREE_SPACE_FOUND = 408,
  MD_ERROR_WIN_APPX_FILE_NOT_ENCRYPTED = 409,
  MD_ERROR_WIN_RWRAW_ENCRYPTED_FILE_NOT_ENCRYPTED = 410,
  MD_ERROR_WIN_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILEOFFSET = 411,
  MD_ERROR_WIN_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILERANGE = 412,
  MD_ERROR_WIN_RWRAW_ENCRYPTED_INVALID_EDATAINFO_PARAMETER = 413,
  MD_ERROR_WIN_LINUX_SUBSYSTEM_NOT_PRESENT = 414,
  MD_ERROR_WIN_FT_READ_FAILURE = 415,
  MD_ERROR_WIN_STORAGE_RESERVE_ID_INVALID = 416,
  MD_ERROR_WIN_STORAGE_RESERVE_DOES_NOT_EXIST = 417,
  MD_ERROR_WIN_STORAGE_RESERVE_ALREADY_EXISTS = 418,
  MD_ERROR_WIN_STORAGE_RESERVE_NOT_EMPTY = 419,
  MD_ERROR_WIN_NOT_A_DAX_VOLUME = 420,
  MD_ERROR_WIN_NOT_DAX_MAPPABLE = 421,
  MD_ERROR_WIN_TIME_SENSITIVE_THREAD = 422,
  MD_ERROR_WIN_DPL_NOT_SUPPORTED_FOR_USER = 423,
  MD_ERROR_WIN_CASE_DIFFERING_NAMES_IN_DIR = 424,
  MD_ERROR_WIN_FILE_NOT_SUPPORTED = 425,
  MD_ERROR_WIN_CLOUD_FILE_REQUEST_TIMEOUT = 426,
  MD_ERROR_WIN_NO_TASK_QUEUE = 427,
  MD_ERROR_WIN_SRC_SRV_DLL_LOAD_FAILED = 428,
  MD_ERROR_WIN_NOT_SUPPORTED_WITH_BTT = 429,
  MD_ERROR_WIN_ENCRYPTION_DISABLED = 430,
  MD_ERROR_WIN_ENCRYPTING_METADATA_DISALLOWED = 431,
  MD_ERROR_WIN_CANT_CLEAR_ENCRYPTION_FLAG = 432,
  MD_ERROR_WIN_NO_SUCH_DEVICE = 433,
  MD_ERROR_WIN_CLOUD_FILE_DEHYDRATION_DISALLOWED = 434,
  MD_ERROR_WIN_FILE_SNAP_IN_PROGRESS = 435,
  MD_ERROR_WIN_FILE_SNAP_USER_SECTION_NOT_SUPPORTED = 436,
  MD_ERROR_WIN_FILE_SNAP_MODIFY_NOT_SUPPORTED = 437,
  MD_ERROR_WIN_FILE_SNAP_IO_NOT_COORDINATED = 438,
  MD_ERROR_WIN_FILE_SNAP_UNEXPECTED_ERROR = 439,
  MD_ERROR_WIN_FILE_SNAP_INVALID_PARAMETER = 440,
  MD_ERROR_WIN_UNSATISFIED_DEPENDENCIES = 441,
  MD_ERROR_WIN_CASE_SENSITIVE_PATH = 442,
  MD_ERROR_WIN_UNEXPECTED_NTCACHEMANAGER_ERROR = 443,
  MD_ERROR_WIN_LINUX_SUBSYSTEM_UPDATE_REQUIRED = 444,
  MD_ERROR_WIN_DLP_POLICY_WARNS_AGAINST_OPERATION = 445,
  MD_ERROR_WIN_DLP_POLICY_DENIES_OPERATION = 446,
  MD_ERROR_WIN_DLP_POLICY_SILENTLY_FAIL = 449,
  MD_ERROR_WIN_CAPAUTHZ_NOT_DEVUNLOCKED = 450,
  MD_ERROR_WIN_CAPAUTHZ_CHANGE_TYPE = 451,
  MD_ERROR_WIN_CAPAUTHZ_NOT_PROVISIONED = 452,
  MD_ERROR_WIN_CAPAUTHZ_NOT_AUTHORIZED = 453,
  MD_ERROR_WIN_CAPAUTHZ_NO_POLICY = 454,
  MD_ERROR_WIN_CAPAUTHZ_DB_CORRUPTED = 455,
  MD_ERROR_WIN_CAPAUTHZ_SCCD_INVALID_CATALOG = 456,
  MD_ERROR_WIN_CAPAUTHZ_SCCD_NO_AUTH_ENTITY = 457,
  MD_ERROR_WIN_CAPAUTHZ_SCCD_PARSE_ERROR = 458,
  MD_ERROR_WIN_CAPAUTHZ_SCCD_DEV_MODE_REQUIRED = 459,
  MD_ERROR_WIN_CAPAUTHZ_SCCD_NO_CAPABILITY_MATCH = 460,
  MD_ERROR_WIN_CIMFS_IMAGE_CORRUPT = 470,
  MD_ERROR_WIN_PNP_QUERY_REMOVE_DEVICE_TIMEOUT = 480,
  MD_ERROR_WIN_PNP_QUERY_REMOVE_RELATED_DEVICE_TIMEOUT = 481,
  MD_ERROR_WIN_PNP_QUERY_REMOVE_UNRELATED_DEVICE_TIMEOUT = 482,
  MD_ERROR_WIN_DEVICE_HARDWARE_ERROR = 483,
  MD_ERROR_WIN_INVALID_ADDRESS = 487,
  MD_ERROR_WIN_HAS_SYSTEM_CRITICAL_FILES = 488,
  MD_ERROR_WIN_USER_PROFILE_LOAD = 500,
  MD_ERROR_WIN_ARITHMETIC_OVERFLOW = 534,
  MD_ERROR_WIN_PIPE_CONNECTED = 535,
  MD_ERROR_WIN_PIPE_LISTENING = 536,
  MD_ERROR_WIN_VERIFIER_STOP = 537,
  MD_ERROR_WIN_ABIOS_ERROR = 538,
  MD_ERROR_WIN_WX86_WARNING = 539,
  MD_ERROR_WIN_WX86_ERROR = 540,
  MD_ERROR_WIN_TIMER_NOT_CANCELED = 541,
  MD_ERROR_WIN_UNWIND = 542,
  MD_ERROR_WIN_BAD_STACK = 543,
  MD_ERROR_WIN_INVALID_UNWIND_TARGET = 544,
  MD_ERROR_WIN_INVALID_PORT_ATTRIBUTES = 545,
  MD_ERROR_WIN_PORT_MESSAGE_TOO_LONG = 546,
  MD_ERROR_WIN_INVALID_QUOTA_LOWER = 547,
  MD_ERROR_WIN_DEVICE_ALREADY_ATTACHED = 548,
  MD_ERROR_WIN_INSTRUCTION_MISALIGNMENT = 549,
  MD_ERROR_WIN_PROFILING_NOT_STARTED = 550,
  MD_ERROR_WIN_PROFILING_NOT_STOPPED = 551,
  MD_ERROR_WIN_COULD_NOT_INTERPRET = 552,
  MD_ERROR_WIN_PROFILING_AT_LIMIT = 553,
  MD_ERROR_WIN_CANT_WAIT = 554,
  MD_ERROR_WIN_CANT_TERMINATE_SELF = 555,
  MD_ERROR_WIN_UNEXPECTED_MM_CREATE_ERR = 556,
  MD_ERROR_WIN_UNEXPECTED_MM_MAP_ERROR = 557,
  MD_ERROR_WIN_UNEXPECTED_MM_EXTEND_ERR = 558,
  MD_ERROR_WIN_BAD_FUNCTION_TABLE = 559,
  MD_ERROR_WIN_NO_GUID_TRANSLATION = 560,
  MD_ERROR_WIN_INVALID_LDT_SIZE = 561,
  MD_ERROR_WIN_INVALID_LDT_OFFSET = 563,
  MD_ERROR_WIN_INVALID_LDT_DESCRIPTOR = 564,
  MD_ERROR_WIN_TOO_MANY_THREADS = 565,
  MD_ERROR_WIN_THREAD_NOT_IN_PROCESS = 566,
  MD_ERROR_WIN_PAGEFILE_QUOTA_EXCEEDED = 567,
  MD_ERROR_WIN_LOGON_SERVER_CONFLICT = 568,
  MD_ERROR_WIN_SYNCHRONIZATION_REQUIRED = 569,
  MD_ERROR_WIN_NET_OPEN_FAILED = 570,
  MD_ERROR_WIN_IO_PRIVILEGE_FAILED = 571,
  MD_ERROR_WIN_CONTROL_C_EXIT = 572,
  MD_ERROR_WIN_MISSING_SYSTEMFILE = 573,
  MD_ERROR_WIN_UNHANDLED_EXCEPTION = 574,
  MD_ERROR_WIN_APP_INIT_FAILURE = 575,
  MD_ERROR_WIN_PAGEFILE_CREATE_FAILED = 576,
  MD_ERROR_WIN_INVALID_IMAGE_HASH = 577,
  MD_ERROR_WIN_NO_PAGEFILE = 578,
  MD_ERROR_WIN_ILLEGAL_FLOAT_CONTEXT = 579,
  MD_ERROR_WIN_NO_EVENT_PAIR = 580,
  MD_ERROR_WIN_DOMAIN_CTRLR_CONFIG_ERROR = 581,
  MD_ERROR_WIN_ILLEGAL_CHARACTER = 582,
  MD_ERROR_WIN_UNDEFINED_CHARACTER = 583,
  MD_ERROR_WIN_FLOPPY_VOLUME = 584,
  MD_ERROR_WIN_BIOS_FAILED_TO_CONNECT_INTERRUPT = 585,
  MD_ERROR_WIN_BACKUP_CONTROLLER = 586,
  MD_ERROR_WIN_MUTANT_LIMIT_EXCEEDED = 587,
  MD_ERROR_WIN_FS_DRIVER_REQUIRED = 588,
  MD_ERROR_WIN_CANNOT_LOAD_REGISTRY_FILE = 589,
  MD_ERROR_WIN_DEBUG_ATTACH_FAILED = 590,
  MD_ERROR_WIN_SYSTEM_PROCESS_TERMINATED = 591,
  MD_ERROR_WIN_DATA_NOT_ACCEPTED = 592,
  MD_ERROR_WIN_VDM_HARD_ERROR = 593,
  MD_ERROR_WIN_DRIVER_CANCEL_TIMEOUT = 594,
  MD_ERROR_WIN_REPLY_MESSAGE_MISMATCH = 595,
  MD_ERROR_WIN_LOST_WRITEBEHIND_DATA = 596,
  MD_ERROR_WIN_CLIENT_SERVER_PARAMETERS_INVALID = 597,
  MD_ERROR_WIN_NOT_TINY_STREAM = 598,
  MD_ERROR_WIN_STACK_OVERFLOW_READ = 599,
  MD_ERROR_WIN_CONVERT_TO_LARGE = 600,
  MD_ERROR_WIN_FOUND_OUT_OF_SCOPE = 601,
  MD_ERROR_WIN_ALLOCATE_BUCKET = 602,
  MD_ERROR_WIN_MARSHALL_OVERFLOW = 603,
  MD_ERROR_WIN_INVALID_VARIANT = 604,
  MD_ERROR_WIN_BAD_COMPRESSION_BUFFER = 605,
  MD_ERROR_WIN_AUDIT_FAILED = 606,
  MD_ERROR_WIN_TIMER_RESOLUTION_NOT_SET = 607,
  MD_ERROR_WIN_INSUFFICIENT_LOGON_INFO = 608,
  MD_ERROR_WIN_BAD_DLL_ENTRYPOINT = 609,
  MD_ERROR_WIN_BAD_SERVICE_ENTRYPOINT = 610,
  MD_ERROR_WIN_IP_ADDRESS_CONFLICT1 = 611,
  MD_ERROR_WIN_IP_ADDRESS_CONFLICT2 = 612,
  MD_ERROR_WIN_REGISTRY_QUOTA_LIMIT = 613,
  MD_ERROR_WIN_NO_CALLBACK_ACTIVE = 614,
  MD_ERROR_WIN_PWD_TOO_SHORT = 615,
  MD_ERROR_WIN_PWD_TOO_RECENT = 616,
  MD_ERROR_WIN_PWD_HISTORY_CONFLICT = 617,
  MD_ERROR_WIN_UNSUPPORTED_COMPRESSION = 618,
  MD_ERROR_WIN_INVALID_HW_PROFILE = 619,
  MD_ERROR_WIN_INVALID_PLUGPLAY_DEVICE_PATH = 620,
  MD_ERROR_WIN_QUOTA_LIST_INCONSISTENT = 621,
  MD_ERROR_WIN_EVALUATION_EXPIRATION = 622,
  MD_ERROR_WIN_ILLEGAL_DLL_RELOCATION = 623,
  MD_ERROR_WIN_DLL_INIT_FAILED_LOGOFF = 624,
  MD_ERROR_WIN_VALIDATE_CONTINUE = 625,
  MD_ERROR_WIN_NO_MORE_MATCHES = 626,
  MD_ERROR_WIN_RANGE_LIST_CONFLICT = 627,
  MD_ERROR_WIN_SERVER_SID_MISMATCH = 628,
  MD_ERROR_WIN_CANT_ENABLE_DENY_ONLY = 629,
  MD_ERROR_WIN_FLOAT_MULTIPLE_FAULTS = 630,
  MD_ERROR_WIN_FLOAT_MULTIPLE_TRAPS = 631,
  MD_ERROR_WIN_NOINTERFACE = 632,
  MD_ERROR_WIN_DRIVER_FAILED_SLEEP = 633,
  MD_ERROR_WIN_CORRUPT_SYSTEM_FILE = 634,
  MD_ERROR_WIN_COMMITMENT_MINIMUM = 635,
  MD_ERROR_WIN_PNP_RESTART_ENUMERATION = 636,
  MD_ERROR_WIN_SYSTEM_IMAGE_BAD_SIGNATURE = 637,
  MD_ERROR_WIN_PNP_REBOOT_REQUIRED = 638,
  MD_ERROR_WIN_INSUFFICIENT_POWER = 639,
  MD_ERROR_WIN_MULTIPLE_FAULT_VIOLATION = 640,
  MD_ERROR_WIN_SYSTEM_SHUTDOWN = 641,
  MD_ERROR_WIN_PORT_NOT_SET = 642,
  MD_ERROR_WIN_DS_VERSION_CHECK_FAILURE = 643,
  MD_ERROR_WIN_RANGE_NOT_FOUND = 644,
  MD_ERROR_WIN_NOT_SAFE_MODE_DRIVER = 646,
  MD_ERROR_WIN_FAILED_DRIVER_ENTRY = 647,
  MD_ERROR_WIN_DEVICE_ENUMERATION_ERROR = 648,
  MD_ERROR_WIN_MOUNT_POINT_NOT_RESOLVED = 649,
  MD_ERROR_WIN_INVALID_DEVICE_OBJECT_PARAMETER = 650,
  MD_ERROR_WIN_MCA_OCCURED = 651,
  MD_ERROR_WIN_DRIVER_DATABASE_ERROR = 652,
  MD_ERROR_WIN_SYSTEM_HIVE_TOO_LARGE = 653,
  MD_ERROR_WIN_DRIVER_FAILED_PRIOR_UNLOAD = 654,
  MD_ERROR_WIN_VOLSNAP_PREPARE_HIBERNATE = 655,
  MD_ERROR_WIN_HIBERNATION_FAILURE = 656,
  MD_ERROR_WIN_PWD_TOO_LONG = 657,
  MD_ERROR_WIN_FILE_SYSTEM_LIMITATION = 665,
  MD_ERROR_WIN_ASSERTION_FAILURE = 668,
  MD_ERROR_WIN_ACPI_ERROR = 669,
  MD_ERROR_WIN_WOW_ASSERTION = 670,
  MD_ERROR_WIN_PNP_BAD_MPS_TABLE = 671,
  MD_ERROR_WIN_PNP_TRANSLATION_FAILED = 672,
  MD_ERROR_WIN_PNP_IRQ_TRANSLATION_FAILED = 673,
  MD_ERROR_WIN_PNP_INVALID_ID = 674,
  MD_ERROR_WIN_WAKE_SYSTEM_DEBUGGER = 675,
  MD_ERROR_WIN_HANDLES_CLOSED = 676,
  MD_ERROR_WIN_EXTRANEOUS_INFORMATION = 677,
  MD_ERROR_WIN_RXACT_COMMIT_NECESSARY = 678,
  MD_ERROR_WIN_MEDIA_CHECK = 679,
  MD_ERROR_WIN_GUID_SUBSTITUTION_MADE = 680,
  MD_ERROR_WIN_STOPPED_ON_SYMLINK = 681,
  MD_ERROR_WIN_LONGJUMP = 682,
  MD_ERROR_WIN_PLUGPLAY_QUERY_VETOED = 683,
  MD_ERROR_WIN_UNWIND_CONSOLIDATE = 684,
  MD_ERROR_WIN_REGISTRY_HIVE_RECOVERED = 685,
  MD_ERROR_WIN_DLL_MIGHT_BE_INSECURE = 686,
  MD_ERROR_WIN_DLL_MIGHT_BE_INCOMPATIBLE = 687,
  MD_ERROR_WIN_DBG_EXCEPTION_NOT_HANDLED = 688,
  MD_ERROR_WIN_DBG_REPLY_LATER = 689,
  MD_ERROR_WIN_DBG_UNABLE_TO_PROVIDE_HANDLE = 690,
  MD_ERROR_WIN_DBG_TERMINATE_THREAD = 691,
  MD_ERROR_WIN_DBG_TERMINATE_PROCESS = 692,
  MD_ERROR_WIN_DBG_CONTROL_C = 693,
  MD_ERROR_WIN_DBG_PRINTEXCEPTION_C = 694,
  MD_ERROR_WIN_DBG_RIPEXCEPTION = 695,
  MD_ERROR_WIN_DBG_CONTROL_BREAK = 696,
  MD_ERROR_WIN_DBG_COMMAND_EXCEPTION = 697,
  MD_ERROR_WIN_OBJECT_NAME_EXISTS = 698,
  MD_ERROR_WIN_THREAD_WAS_SUSPENDED = 699,
  MD_ERROR_WIN_IMAGE_NOT_AT_BASE = 700,
  MD_ERROR_WIN_RXACT_STATE_CREATED = 701,
  MD_ERROR_WIN_SEGMENT_NOTIFICATION = 702,
  MD_ERROR_WIN_BAD_CURRENT_DIRECTORY = 703,
  MD_ERROR_WIN_FT_READ_RECOVERY_FROM_BACKUP = 704,
  MD_ERROR_WIN_FT_WRITE_RECOVERY = 705,
  MD_ERROR_WIN_IMAGE_MACHINE_TYPE_MISMATCH = 706,
  MD_ERROR_WIN_RECEIVE_PARTIAL = 707,
  MD_ERROR_WIN_RECEIVE_EXPEDITED = 708,
  MD_ERROR_WIN_RECEIVE_PARTIAL_EXPEDITED = 709,
  MD_ERROR_WIN_EVENT_DONE = 710,
  MD_ERROR_WIN_EVENT_PENDING = 711,
  MD_ERROR_WIN_CHECKING_FILE_SYSTEM = 712,
  MD_ERROR_WIN_FATAL_APP_EXIT = 713,
  MD_ERROR_WIN_PREDEFINED_HANDLE = 714,
  MD_ERROR_WIN_WAS_UNLOCKED = 715,
  MD_ERROR_WIN_SERVICE_NOTIFICATION = 716,
  MD_ERROR_WIN_WAS_LOCKED = 717,
  MD_ERROR_WIN_LOG_HARD_ERROR = 718,
  MD_ERROR_WIN_ALREADY_WIN32 = 719,
  MD_ERROR_WIN_IMAGE_MACHINE_TYPE_MISMATCH_EXE = 720,
  MD_ERROR_WIN_NO_YIELD_PERFORMED = 721,
  MD_ERROR_WIN_TIMER_RESUME_IGNORED = 722,
  MD_ERROR_WIN_ARBITRATION_UNHANDLED = 723,
  MD_ERROR_WIN_CARDBUS_NOT_SUPPORTED = 724,
  MD_ERROR_WIN_MP_PROCESSOR_MISMATCH = 725,
  MD_ERROR_WIN_HIBERNATED = 726,
  MD_ERROR_WIN_RESUME_HIBERNATION = 727,
  MD_ERROR_WIN_FIRMWARE_UPDATED = 728,
  MD_ERROR_WIN_DRIVERS_LEAKING_LOCKED_PAGES = 729,
  MD_ERROR_WIN_WAKE_SYSTEM = 730,
  MD_ERROR_WIN_WAIT_1 = 731,
  MD_ERROR_WIN_WAIT_2 = 732,
  MD_ERROR_WIN_WAIT_3 = 733,
  MD_ERROR_WIN_WAIT_63 = 734,
  MD_ERROR_WIN_ABANDONED_WAIT_0 = 735,
  MD_ERROR_WIN_ABANDONED_WAIT_63 = 736,
  MD_ERROR_WIN_USER_APC = 737,
  MD_ERROR_WIN_KERNEL_APC = 738,
  MD_ERROR_WIN_ALERTED = 739,
  MD_ERROR_WIN_ELEVATION_REQUIRED = 740,
  MD_ERROR_WIN_REPARSE = 741,
  MD_ERROR_WIN_OPLOCK_BREAK_IN_PROGRESS = 742,
  MD_ERROR_WIN_VOLUME_MOUNTED = 743,
  MD_ERROR_WIN_RXACT_COMMITTED = 744,
  MD_ERROR_WIN_NOTIFY_CLEANUP = 745,
  MD_ERROR_WIN_PRIMARY_TRANSPORT_CONNECT_FAILED = 746,
  MD_ERROR_WIN_PAGE_FAULT_TRANSITION = 747,
  MD_ERROR_WIN_PAGE_FAULT_DEMAND_ZERO = 748,
  MD_ERROR_WIN_PAGE_FAULT_COPY_ON_WRITE = 749,
  MD_ERROR_WIN_PAGE_FAULT_GUARD_PAGE = 750,
  MD_ERROR_WIN_PAGE_FAULT_PAGING_FILE = 751,
  MD_ERROR_WIN_CACHE_PAGE_LOCKED = 752,
  MD_ERROR_WIN_CRASH_DUMP = 753,
  MD_ERROR_WIN_BUFFER_ALL_ZEROS = 754,
  MD_ERROR_WIN_REPARSE_OBJECT = 755,
  MD_ERROR_WIN_RESOURCE_REQUIREMENTS_CHANGED = 756,
  MD_ERROR_WIN_TRANSLATION_COMPLETE = 757,
  MD_ERROR_WIN_NOTHING_TO_TERMINATE = 758,
  MD_ERROR_WIN_PROCESS_NOT_IN_JOB = 759,
  MD_ERROR_WIN_PROCESS_IN_JOB = 760,
  MD_ERROR_WIN_VOLSNAP_HIBERNATE_READY = 761,
  MD_ERROR_WIN_FSFILTER_OP_COMPLETED_SUCCESSFULLY = 762,
  MD_ERROR_WIN_INTERRUPT_VECTOR_ALREADY_CONNECTED = 763,
  MD_ERROR_WIN_INTERRUPT_STILL_CONNECTED = 764,
  MD_ERROR_WIN_WAIT_FOR_OPLOCK = 765,
  MD_ERROR_WIN_DBG_EXCEPTION_HANDLED = 766,
  MD_ERROR_WIN_DBG_CONTINUE = 767,
  MD_ERROR_WIN_CALLBACK_POP_STACK = 768,
  MD_ERROR_WIN_COMPRESSION_DISABLED = 769,
  MD_ERROR_WIN_CANTFETCHBACKWARDS = 770,
  MD_ERROR_WIN_CANTSCROLLBACKWARDS = 771,
  MD_ERROR_WIN_ROWSNOTRELEASED = 772,
  MD_ERROR_WIN_BAD_ACCESSOR_FLAGS = 773,
  MD_ERROR_WIN_ERRORS_ENCOUNTERED = 774,
  MD_ERROR_WIN_NOT_CAPABLE = 775,
  MD_ERROR_WIN_REQUEST_OUT_OF_SEQUENCE = 776,
  MD_ERROR_WIN_VERSION_PARSE_ERROR = 777,
  MD_ERROR_WIN_BADSTARTPOSITION = 778,
  MD_ERROR_WIN_MEMORY_HARDWARE = 779,
  MD_ERROR_WIN_DISK_REPAIR_DISABLED = 780,
  MD_ERROR_WIN_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE = 781,
  MD_ERROR_WIN_SYSTEM_POWERSTATE_TRANSITION = 782,
  MD_ERROR_WIN_SYSTEM_POWERSTATE_COMPLEX_TRANSITION = 783,
  MD_ERROR_WIN_MCA_EXCEPTION = 784,
  MD_ERROR_WIN_ACCESS_AUDIT_BY_POLICY = 785,
  MD_ERROR_WIN_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY = 786,
  MD_ERROR_WIN_ABANDON_HIBERFILE = 787,
  MD_ERROR_WIN_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED = 788,
  MD_ERROR_WIN_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR = 789,
  MD_ERROR_WIN_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR = 790,
  MD_ERROR_WIN_BAD_MCFG_TABLE = 791,
  MD_ERROR_WIN_DISK_REPAIR_REDIRECTED = 792,
  MD_ERROR_WIN_DISK_REPAIR_UNSUCCESSFUL = 793,
  MD_ERROR_WIN_CORRUPT_LOG_OVERFULL = 794,
  MD_ERROR_WIN_CORRUPT_LOG_CORRUPTED = 795,
  MD_ERROR_WIN_CORRUPT_LOG_UNAVAILABLE = 796,
  MD_ERROR_WIN_CORRUPT_LOG_DELETED_FULL = 797,
  MD_ERROR_WIN_CORRUPT_LOG_CLEARED = 798,
  MD_ERROR_WIN_ORPHAN_NAME_EXHAUSTED = 799,
  MD_ERROR_WIN_OPLOCK_SWITCHED_TO_NEW_HANDLE = 800,
  MD_ERROR_WIN_CANNOT_GRANT_REQUESTED_OPLOCK = 801,
  MD_ERROR_WIN_CANNOT_BREAK_OPLOCK = 802,
  MD_ERROR_WIN_OPLOCK_HANDLE_CLOSED = 803,
  MD_ERROR_WIN_NO_ACE_CONDITION = 804,
  MD_ERROR_WIN_INVALID_ACE_CONDITION = 805,
  MD_ERROR_WIN_FILE_HANDLE_REVOKED = 806,
  MD_ERROR_WIN_IMAGE_AT_DIFFERENT_BASE = 807,
  MD_ERROR_WIN_ENCRYPTED_IO_NOT_POSSIBLE = 808,
  MD_ERROR_WIN_FILE_METADATA_OPTIMIZATION_IN_PROGRESS = 809,
  MD_ERROR_WIN_QUOTA_ACTIVITY = 810,
  MD_ERROR_WIN_HANDLE_REVOKED = 811,
  MD_ERROR_WIN_CALLBACK_INVOKE_INLINE = 812,
  MD_ERROR_WIN_CPU_SET_INVALID = 813,
  MD_ERROR_WIN_ENCLAVE_NOT_TERMINATED = 814,
  MD_ERROR_WIN_ENCLAVE_VIOLATION = 815,
  MD_ERROR_WIN_EA_ACCESS_DENIED = 994,
  MD_ERROR_WIN_OPERATION_ABORTED = 995,
  MD_ERROR_WIN_IO_INCOMPLETE = 996,
  MD_ERROR_WIN_IO_PENDING = 997,
  MD_ERROR_WIN_NOACCESS = 998,
  MD_ERROR_WIN_SWAPERROR = 999,
  MD_ERROR_WIN_STACK_OVERFLOW = 1001,
  MD_ERROR_WIN_INVALID_MESSAGE = 1002,
  MD_ERROR_WIN_CAN_NOT_COMPLETE = 1003,
  MD_ERROR_WIN_INVALID_FLAGS = 1004,
  MD_ERROR_WIN_UNRECOGNIZED_VOLUME = 1005,
  MD_ERROR_WIN_FILE_INVALID = 1006,
  MD_ERROR_WIN_FULLSCREEN_MODE = 1007,
  MD_ERROR_WIN_NO_TOKEN = 1008,
  MD_ERROR_WIN_BADDB = 1009,
  MD_ERROR_WIN_BADKEY = 1010,
  MD_ERROR_WIN_CANTOPEN = 1011,
  MD_ERROR_WIN_CANTREAD = 1012,
  MD_ERROR_WIN_CANTWRITE = 1013,
  MD_ERROR_WIN_REGISTRY_RECOVERED = 1014,
  MD_ERROR_WIN_REGISTRY_CORRUPT = 1015,
  MD_ERROR_WIN_REGISTRY_IO_FAILED = 1016,
  MD_ERROR_WIN_NOT_REGISTRY_FILE = 1017,
  MD_ERROR_WIN_KEY_DELETED = 1018,
  MD_ERROR_WIN_NO_LOG_SPACE = 1019,
  MD_ERROR_WIN_KEY_HAS_CHILDREN = 1020,
  MD_ERROR_WIN_CHILD_MUST_BE_VOLATILE = 1021,
  MD_ERROR_WIN_NOTIFY_ENUM_DIR = 1022,
  MD_ERROR_WIN_DEPENDENT_SERVICES_RUNNING = 1051,
  MD_ERROR_WIN_INVALID_SERVICE_CONTROL = 1052,
  MD_ERROR_WIN_SERVICE_REQUEST_TIMEOUT = 1053,
  MD_ERROR_WIN_SERVICE_NO_THREAD = 1054,
  MD_ERROR_WIN_SERVICE_DATABASE_LOCKED = 1055,
  MD_ERROR_WIN_SERVICE_ALREADY_RUNNING = 1056,
  MD_ERROR_WIN_INVALID_SERVICE_ACCOUNT = 1057,
  MD_ERROR_WIN_SERVICE_DISABLED = 1058,
  MD_ERROR_WIN_CIRCULAR_DEPENDENCY = 1059,
  MD_ERROR_WIN_SERVICE_DOES_NOT_EXIST = 1060,
  MD_ERROR_WIN_SERVICE_CANNOT_ACCEPT_CTRL = 1061,
  MD_ERROR_WIN_SERVICE_NOT_ACTIVE = 1062,
  MD_ERROR_WIN_FAILED_SERVICE_CONTROLLER_CONNECT = 1063,
  MD_ERROR_WIN_EXCEPTION_IN_SERVICE = 1064,
  MD_ERROR_WIN_DATABASE_DOES_NOT_EXIST = 1065,
  MD_ERROR_WIN_SERVICE_SPECIFIC_ERROR = 1066,
  MD_ERROR_WIN_PROCESS_ABORTED = 1067,
  MD_ERROR_WIN_SERVICE_DEPENDENCY_FAIL = 1068,
  MD_ERROR_WIN_SERVICE_LOGON_FAILED = 1069,
  MD_ERROR_WIN_SERVICE_START_HANG = 1070,
  MD_ERROR_WIN_INVALID_SERVICE_LOCK = 1071,
  MD_ERROR_WIN_SERVICE_MARKED_FOR_DELETE = 1072,
  MD_ERROR_WIN_SERVICE_EXISTS = 1073,
  MD_ERROR_WIN_ALREADY_RUNNING_LKG = 1074,
  MD_ERROR_WIN_SERVICE_DEPENDENCY_DELETED = 1075,
  MD_ERROR_WIN_BOOT_ALREADY_ACCEPTED = 1076,
  MD_ERROR_WIN_SERVICE_NEVER_STARTED = 1077,
  MD_ERROR_WIN_DUPLICATE_SERVICE_NAME = 1078,
  MD_ERROR_WIN_DIFFERENT_SERVICE_ACCOUNT = 1079,
  MD_ERROR_WIN_CANNOT_DETECT_DRIVER_FAILURE = 1080,
  MD_ERROR_WIN_CANNOT_DETECT_PROCESS_ABORT = 1081,
  MD_ERROR_WIN_NO_RECOVERY_PROGRAM = 1082,
  MD_ERROR_WIN_SERVICE_NOT_IN_EXE = 1083,
  MD_ERROR_WIN_NOT_SAFEBOOT_SERVICE = 1084,
  MD_ERROR_WIN_END_OF_MEDIA = 1100,
  MD_ERROR_WIN_FILEMARK_DETECTED = 1101,
  MD_ERROR_WIN_BEGINNING_OF_MEDIA = 1102,
  MD_ERROR_WIN_SETMARK_DETECTED = 1103,
  MD_ERROR_WIN_NO_DATA_DETECTED = 1104,
  MD_ERROR_WIN_PARTITION_FAILURE = 1105,
  MD_ERROR_WIN_INVALID_BLOCK_LENGTH = 1106,
  MD_ERROR_WIN_DEVICE_NOT_PARTITIONED = 1107,
  MD_ERROR_WIN_UNABLE_TO_LOCK_MEDIA = 1108,
  MD_ERROR_WIN_UNABLE_TO_UNLOAD_MEDIA = 1109,
  MD_ERROR_WIN_MEDIA_CHANGED = 1110,
  MD_ERROR_WIN_BUS_RESET = 1111,
  MD_ERROR_WIN_NO_MEDIA_IN_DRIVE = 1112,
  MD_ERROR_WIN_NO_UNICODE_TRANSLATION = 1113,
  MD_ERROR_WIN_DLL_INIT_FAILED = 1114,
  MD_ERROR_WIN_SHUTDOWN_IN_PROGRESS = 1115,
  MD_ERROR_WIN_NO_SHUTDOWN_IN_PROGRESS = 1116,
  MD_ERROR_WIN_IO_DEVICE = 1117,
  MD_ERROR_WIN_SERIAL_NO_DEVICE = 1118,
  MD_ERROR_WIN_IRQ_BUSY = 1119,
  MD_ERROR_WIN_MORE_WRITES = 1120,
  MD_ERROR_WIN_COUNTER_TIMEOUT = 1121,
  MD_ERROR_WIN_FLOPPY_ID_MARK_NOT_FOUND = 1122,
  MD_ERROR_WIN_FLOPPY_WRONG_CYLINDER = 1123,
  MD_ERROR_WIN_FLOPPY_UNKNOWN_ERROR = 1124,
  MD_ERROR_WIN_FLOPPY_BAD_REGISTERS = 1125,
  MD_ERROR_WIN_DISK_RECALIBRATE_FAILED = 1126,
  MD_ERROR_WIN_DISK_OPERATION_FAILED = 1127,
  MD_ERROR_WIN_DISK_RESET_FAILED = 1128,
  MD_ERROR_WIN_EOM_OVERFLOW = 1129,
  MD_ERROR_WIN_NOT_ENOUGH_SERVER_MEMORY = 1130,
  MD_ERROR_WIN_POSSIBLE_DEADLOCK = 1131,
  MD_ERROR_WIN_MAPPED_ALIGNMENT = 1132,
  MD_ERROR_WIN_SET_POWER_STATE_VETOED = 1140,
  MD_ERROR_WIN_SET_POWER_STATE_FAILED = 1141,
  MD_ERROR_WIN_TOO_MANY_LINKS = 1142,
  MD_ERROR_WIN_OLD_WIN_VERSION = 1150,
  MD_ERROR_WIN_APP_WRONG_OS = 1151,
  MD_ERROR_WIN_SINGLE_INSTANCE_APP = 1152,
  MD_ERROR_WIN_RMODE_APP = 1153,
  MD_ERROR_WIN_INVALID_DLL = 1154,
  MD_ERROR_WIN_NO_ASSOCIATION = 1155,
  MD_ERROR_WIN_DDE_FAIL = 1156,
  MD_ERROR_WIN_DLL_NOT_FOUND = 1157,
  MD_ERROR_WIN_NO_MORE_USER_HANDLES = 1158,
  MD_ERROR_WIN_MESSAGE_SYNC_ONLY = 1159,
  MD_ERROR_WIN_SOURCE_ELEMENT_EMPTY = 1160,
  MD_ERROR_WIN_DESTINATION_ELEMENT_FULL = 1161,
  MD_ERROR_WIN_ILLEGAL_ELEMENT_ADDRESS = 1162,
  MD_ERROR_WIN_MAGAZINE_NOT_PRESENT = 1163,
  MD_ERROR_WIN_DEVICE_REINITIALIZATION_NEEDED = 1164,
  MD_ERROR_WIN_DEVICE_REQUIRES_CLEANING = 1165,
  MD_ERROR_WIN_DEVICE_DOOR_OPEN = 1166,
  MD_ERROR_WIN_DEVICE_NOT_CONNECTED = 1167,
  MD_ERROR_WIN_NOT_FOUND = 1168,
  MD_ERROR_WIN_NO_MATCH = 1169,
  MD_ERROR_WIN_SET_NOT_FOUND = 1170,
  MD_ERROR_WIN_POINT_NOT_FOUND = 1171,
  MD_ERROR_WIN_NO_TRACKING_SERVICE = 1172,
  MD_ERROR_WIN_NO_VOLUME_ID = 1173,
  MD_ERROR_WIN_UNABLE_TO_REMOVE_REPLACED = 1175,
  MD_ERROR_WIN_UNABLE_TO_MOVE_REPLACEMENT = 1176,
  MD_ERROR_WIN_UNABLE_TO_MOVE_REPLACEMENT_2 = 1177,
  MD_ERROR_WIN_JOURNAL_DELETE_IN_PROGRESS = 1178,
  MD_ERROR_WIN_JOURNAL_NOT_ACTIVE = 1179,
  MD_ERROR_WIN_POTENTIAL_FILE_FOUND = 1180,
  MD_ERROR_WIN_JOURNAL_ENTRY_DELETED = 1181,
  MD_ERROR_WIN_VRF_CFG_AND_IO_ENABLED = 1183,
  MD_ERROR_WIN_PARTITION_TERMINATING = 1184,
  MD_ERROR_WIN_SHUTDOWN_IS_SCHEDULED = 1190,
  MD_ERROR_WIN_SHUTDOWN_USERS_LOGGED_ON = 1191,
  MD_ERROR_WIN_BAD_DEVICE = 1200,
  MD_ERROR_WIN_CONNECTION_UNAVAIL = 1201,
  MD_ERROR_WIN_DEVICE_ALREADY_REMEMBERED = 1202,
  MD_ERROR_WIN_NO_NET_OR_BAD_PATH = 1203,
  MD_ERROR_WIN_BAD_PROVIDER = 1204,
  MD_ERROR_WIN_CANNOT_OPEN_PROFILE = 1205,
  MD_ERROR_WIN_BAD_PROFILE = 1206,
  MD_ERROR_WIN_NOT_CONTAINER = 1207,
  MD_ERROR_WIN_EXTENDED_ERROR = 1208,
  MD_ERROR_WIN_INVALID_GROUPNAME = 1209,
  MD_ERROR_WIN_INVALID_COMPUTERNAME = 1210,
  MD_ERROR_WIN_INVALID_EVENTNAME = 1211,
  MD_ERROR_WIN_INVALID_DOMAINNAME = 1212,
  MD_ERROR_WIN_INVALID_SERVICENAME = 1213,
  MD_ERROR_WIN_INVALID_NETNAME = 1214,
  MD_ERROR_WIN_INVALID_SHARENAME = 1215,
  MD_ERROR_WIN_INVALID_PASSWORDNAME = 1216,
  MD_ERROR_WIN_INVALID_MESSAGENAME = 1217,
  MD_ERROR_WIN_INVALID_MESSAGEDEST = 1218,
  MD_ERROR_WIN_SESSION_CREDENTIAL_CONFLICT = 1219,
  MD_ERROR_WIN_REMOTE_SESSION_LIMIT_EXCEEDED = 1220,
  MD_ERROR_WIN_DUP_DOMAINNAME = 1221,
  MD_ERROR_WIN_NO_NETWORK = 1222,
  MD_ERROR_WIN_CANCELLED = 1223,
  MD_ERROR_WIN_USER_MAPPED_FILE = 1224,
  MD_ERROR_WIN_CONNECTION_REFUSED = 1225,
  MD_ERROR_WIN_GRACEFUL_DISCONNECT = 1226,
  MD_ERROR_WIN_ADDRESS_ALREADY_ASSOCIATED = 1227,
  MD_ERROR_WIN_ADDRESS_NOT_ASSOCIATED = 1228,
  MD_ERROR_WIN_CONNECTION_INVALID = 1229,
  MD_ERROR_WIN_CONNECTION_ACTIVE = 1230,
  MD_ERROR_WIN_NETWORK_UNREACHABLE = 1231,
  MD_ERROR_WIN_HOST_UNREACHABLE = 1232,
  MD_ERROR_WIN_PROTOCOL_UNREACHABLE = 1233,
  MD_ERROR_WIN_PORT_UNREACHABLE = 1234,
  MD_ERROR_WIN_REQUEST_ABORTED = 1235,
  MD_ERROR_WIN_CONNECTION_ABORTED = 1236,
  MD_ERROR_WIN_RETRY = 1237,
  MD_ERROR_WIN_CONNECTION_COUNT_LIMIT = 1238,
  MD_ERROR_WIN_LOGIN_TIME_RESTRICTION = 1239,
  MD_ERROR_WIN_LOGIN_WKSTA_RESTRICTION = 1240,
  MD_ERROR_WIN_INCORRECT_ADDRESS = 1241,
  MD_ERROR_WIN_ALREADY_REGISTERED = 1242,
  MD_ERROR_WIN_SERVICE_NOT_FOUND = 1243,
  MD_ERROR_WIN_NOT_AUTHENTICATED = 1244,
  MD_ERROR_WIN_NOT_LOGGED_ON = 1245,
  MD_ERROR_WIN_CONTINUE = 1246,
  MD_ERROR_WIN_ALREADY_INITIALIZED = 1247,
  MD_ERROR_WIN_NO_MORE_DEVICES = 1248,
  MD_ERROR_WIN_NO_SUCH_SITE = 1249,
  MD_ERROR_WIN_DOMAIN_CONTROLLER_EXISTS = 1250,
  MD_ERROR_WIN_ONLY_IF_CONNECTED = 1251,
  MD_ERROR_WIN_OVERRIDE_NOCHANGES = 1252,
  MD_ERROR_WIN_BAD_USER_PROFILE = 1253,
  MD_ERROR_WIN_NOT_SUPPORTED_ON_SBS = 1254,
  MD_ERROR_WIN_SERVER_SHUTDOWN_IN_PROGRESS = 1255,
  MD_ERROR_WIN_HOST_DOWN = 1256,
  MD_ERROR_WIN_NON_ACCOUNT_SID = 1257,
  MD_ERROR_WIN_NON_DOMAIN_SID = 1258,
  MD_ERROR_WIN_APPHELP_BLOCK = 1259,
  MD_ERROR_WIN_ACCESS_DISABLED_BY_POLICY = 1260,
  MD_ERROR_WIN_REG_NAT_CONSUMPTION = 1261,
  MD_ERROR_WIN_CSCSHARE_OFFLINE = 1262,
  MD_ERROR_WIN_PKINIT_FAILURE = 1263,
  MD_ERROR_WIN_SMARTCARD_SUBSYSTEM_FAILURE = 1264,
  MD_ERROR_WIN_DOWNGRADE_DETECTED = 1265,
  MD_ERROR_WIN_MACHINE_LOCKED = 1271,
  MD_ERROR_WIN_SMB_GUEST_LOGON_BLOCKED = 1272,
  MD_ERROR_WIN_CALLBACK_SUPPLIED_INVALID_DATA = 1273,
  MD_ERROR_WIN_SYNC_FOREGROUND_REFRESH_REQUIRED = 1274,
  MD_ERROR_WIN_DRIVER_BLOCKED = 1275,
  MD_ERROR_WIN_INVALID_IMPORT_OF_NON_DLL = 1276,
  MD_ERROR_WIN_ACCESS_DISABLED_WEBBLADE = 1277,
  MD_ERROR_WIN_ACCESS_DISABLED_WEBBLADE_TAMPER = 1278,
  MD_ERROR_WIN_RECOVERY_FAILURE = 1279,
  MD_ERROR_WIN_ALREADY_FIBER = 1280,
  MD_ERROR_WIN_ALREADY_THREAD = 1281,
  MD_ERROR_WIN_STACK_BUFFER_OVERRUN = 1282,
  MD_ERROR_WIN_PARAMETER_QUOTA_EXCEEDED = 1283,
  MD_ERROR_WIN_DEBUGGER_INACTIVE = 1284,
  MD_ERROR_WIN_DELAY_LOAD_FAILED = 1285,
  MD_ERROR_WIN_VDM_DISALLOWED = 1286,
  MD_ERROR_WIN_UNIDENTIFIED_ERROR = 1287,
  MD_ERROR_WIN_INVALID_CRUNTIME_PARAMETER = 1288,
  MD_ERROR_WIN_BEYOND_VDL = 1289,
  MD_ERROR_WIN_INCOMPATIBLE_SERVICE_SID_TYPE = 1290,
  MD_ERROR_WIN_DRIVER_PROCESS_TERMINATED = 1291,
  MD_ERROR_WIN_IMPLEMENTATION_LIMIT = 1292,
  MD_ERROR_WIN_PROCESS_IS_PROTECTED = 1293,
  MD_ERROR_WIN_SERVICE_NOTIFY_CLIENT_LAGGING = 1294,
  MD_ERROR_WIN_DISK_QUOTA_EXCEEDED = 1295,
  MD_ERROR_WIN_CONTENT_BLOCKED = 1296,
  MD_ERROR_WIN_INCOMPATIBLE_SERVICE_PRIVILEGE = 1297,
  MD_ERROR_WIN_APP_HANG = 1298,
  MD_ERROR_WIN_INVALID_LABEL = 1299,
  MD_ERROR_WIN_NOT_ALL_ASSIGNED = 1300,
  MD_ERROR_WIN_SOME_NOT_MAPPED = 1301,
  MD_ERROR_WIN_NO_QUOTAS_FOR_ACCOUNT = 1302,
  MD_ERROR_WIN_LOCAL_USER_SESSION_KEY = 1303,
  MD_ERROR_WIN_NULL_LM_PASSWORD = 1304,
  MD_ERROR_WIN_UNKNOWN_REVISION = 1305,
  MD_ERROR_WIN_REVISION_MISMATCH = 1306,
  MD_ERROR_WIN_INVALID_OWNER = 1307,
  MD_ERROR_WIN_INVALID_PRIMARY_GROUP = 1308,
  MD_ERROR_WIN_NO_IMPERSONATION_TOKEN = 1309,
  MD_ERROR_WIN_CANT_DISABLE_MANDATORY = 1310,
  MD_ERROR_WIN_NO_LOGON_SERVERS = 1311,
  MD_ERROR_WIN_NO_SUCH_LOGON_SESSION = 1312,
  MD_ERROR_WIN_NO_SUCH_PRIVILEGE = 1313,
  MD_ERROR_WIN_PRIVILEGE_NOT_HELD = 1314,
  MD_ERROR_WIN_INVALID_ACCOUNT_NAME = 1315,
  MD_ERROR_WIN_USER_EXISTS = 1316,
  MD_ERROR_WIN_NO_SUCH_USER = 1317,
  MD_ERROR_WIN_GROUP_EXISTS = 1318,
  MD_ERROR_WIN_NO_SUCH_GROUP = 1319,
  MD_ERROR_WIN_MEMBER_IN_GROUP = 1320,
  MD_ERROR_WIN_MEMBER_NOT_IN_GROUP = 1321,
  MD_ERROR_WIN_LAST_ADMIN = 1322,
  MD_ERROR_WIN_WRONG_PASSWORD = 1323,
  MD_ERROR_WIN_ILL_FORMED_PASSWORD = 1324,
  MD_ERROR_WIN_PASSWORD_RESTRICTION = 1325,
  MD_ERROR_WIN_LOGON_FAILURE = 1326,
  MD_ERROR_WIN_ACCOUNT_RESTRICTION = 1327,
  MD_ERROR_WIN_INVALID_LOGON_HOURS = 1328,
  MD_ERROR_WIN_INVALID_WORKSTATION = 1329,
  MD_ERROR_WIN_PASSWORD_EXPIRED = 1330,
  MD_ERROR_WIN_ACCOUNT_DISABLED = 1331,
  MD_ERROR_WIN_NONE_MAPPED = 1332,
  MD_ERROR_WIN_TOO_MANY_LUIDS_REQUESTED = 1333,
  MD_ERROR_WIN_LUIDS_EXHAUSTED = 1334,
  MD_ERROR_WIN_INVALID_SUB_AUTHORITY = 1335,
  MD_ERROR_WIN_INVALID_ACL = 1336,
  MD_ERROR_WIN_INVALID_SID = 1337,
  MD_ERROR_WIN_INVALID_SECURITY_DESCR = 1338,
  MD_ERROR_WIN_BAD_INHERITANCE_ACL = 1340,
  MD_ERROR_WIN_SERVER_DISABLED = 1341,
  MD_ERROR_WIN_SERVER_NOT_DISABLED = 1342,
  MD_ERROR_WIN_INVALID_ID_AUTHORITY = 1343,
  MD_ERROR_WIN_ALLOTTED_SPACE_EXCEEDED = 1344,
  MD_ERROR_WIN_INVALID_GROUP_ATTRIBUTES = 1345,
  MD_ERROR_WIN_BAD_IMPERSONATION_LEVEL = 1346,
  MD_ERROR_WIN_CANT_OPEN_ANONYMOUS = 1347,
  MD_ERROR_WIN_BAD_VALIDATION_CLASS = 1348,
  MD_ERROR_WIN_BAD_TOKEN_TYPE = 1349,
  MD_ERROR_WIN_NO_SECURITY_ON_OBJECT = 1350,
  MD_ERROR_WIN_CANT_ACCESS_DOMAIN_INFO = 1351,
  MD_ERROR_WIN_INVALID_SERVER_STATE = 1352,
  MD_ERROR_WIN_INVALID_DOMAIN_STATE = 1353,
  MD_ERROR_WIN_INVALID_DOMAIN_ROLE = 1354,
  MD_ERROR_WIN_NO_SUCH_DOMAIN = 1355,
  MD_ERROR_WIN_DOMAIN_EXISTS = 1356,
  MD_ERROR_WIN_DOMAIN_LIMIT_EXCEEDED = 1357,
  MD_ERROR_WIN_INTERNAL_DB_CORRUPTION = 1358,
  MD_ERROR_WIN_INTERNAL_ERROR = 1359,
  MD_ERROR_WIN_GENERIC_NOT_MAPPED = 1360,
  MD_ERROR_WIN_BAD_DESCRIPTOR_FORMAT = 1361,
  MD_ERROR_WIN_NOT_LOGON_PROCESS = 1362,
  MD_ERROR_WIN_LOGON_SESSION_EXISTS = 1363,
  MD_ERROR_WIN_NO_SUCH_PACKAGE = 1364,
  MD_ERROR_WIN_BAD_LOGON_SESSION_STATE = 1365,
  MD_ERROR_WIN_LOGON_SESSION_COLLISION = 1366,
  MD_ERROR_WIN_INVALID_LOGON_TYPE = 1367,
  MD_ERROR_WIN_CANNOT_IMPERSONATE = 1368,
  MD_ERROR_WIN_RXACT_INVALID_STATE = 1369,
  MD_ERROR_WIN_RXACT_COMMIT_FAILURE = 1370,
  MD_ERROR_WIN_SPECIAL_ACCOUNT = 1371,
  MD_ERROR_WIN_SPECIAL_GROUP = 1372,
  MD_ERROR_WIN_SPECIAL_USER = 1373,
  MD_ERROR_WIN_MEMBERS_PRIMARY_GROUP = 1374,
  MD_ERROR_WIN_TOKEN_ALREADY_IN_USE = 1375,
  MD_ERROR_WIN_NO_SUCH_ALIAS = 1376,
  MD_ERROR_WIN_MEMBER_NOT_IN_ALIAS = 1377,
  MD_ERROR_WIN_MEMBER_IN_ALIAS = 1378,
  MD_ERROR_WIN_ALIAS_EXISTS = 1379,
  MD_ERROR_WIN_LOGON_NOT_GRANTED = 1380,
  MD_ERROR_WIN_TOO_MANY_SECRETS = 1381,
  MD_ERROR_WIN_SECRET_TOO_LONG = 1382,
  MD_ERROR_WIN_INTERNAL_DB_ERROR = 1383,
  MD_ERROR_WIN_TOO_MANY_CONTEXT_IDS = 1384,
  MD_ERROR_WIN_LOGON_TYPE_NOT_GRANTED = 1385,
  MD_ERROR_WIN_NT_CROSS_ENCRYPTION_REQUIRED = 1386,
  MD_ERROR_WIN_NO_SUCH_MEMBER = 1387,
  MD_ERROR_WIN_INVALID_MEMBER = 1388,
  MD_ERROR_WIN_TOO_MANY_SIDS = 1389,
  MD_ERROR_WIN_LM_CROSS_ENCRYPTION_REQUIRED = 1390,
  MD_ERROR_WIN_NO_INHERITANCE = 1391,
  MD_ERROR_WIN_FILE_CORRUPT = 1392,
  MD_ERROR_WIN_DISK_CORRUPT = 1393,
  MD_ERROR_WIN_NO_USER_SESSION_KEY = 1394,
  MD_ERROR_WIN_LICENSE_QUOTA_EXCEEDED = 1395,
  MD_ERROR_WIN_WRONG_TARGET_NAME = 1396,
  MD_ERROR_WIN_MUTUAL_AUTH_FAILED = 1397,
  MD_ERROR_WIN_TIME_SKEW = 1398,
  MD_ERROR_WIN_CURRENT_DOMAIN_NOT_ALLOWED = 1399,
  MD_ERROR_WIN_INVALID_WINDOW_HANDLE = 1400,
  MD_ERROR_WIN_INVALID_MENU_HANDLE = 1401,
  MD_ERROR_WIN_INVALID_CURSOR_HANDLE = 1402,
  MD_ERROR_WIN_INVALID_ACCEL_HANDLE = 1403,
  MD_ERROR_WIN_INVALID_HOOK_HANDLE = 1404,
  MD_ERROR_WIN_INVALID_DWP_HANDLE = 1405,
  MD_ERROR_WIN_TLW_WITH_WSCHILD = 1406,
  MD_ERROR_WIN_CANNOT_FIND_WND_CLASS = 1407,
  MD_ERROR_WIN_WINDOW_OF_OTHER_THREAD = 1408,
  MD_ERROR_WIN_HOTKEY_ALREADY_REGISTERED = 1409,
  MD_ERROR_WIN_CLASS_ALREADY_EXISTS = 1410,
  MD_ERROR_WIN_CLASS_DOES_NOT_EXIST = 1411,
  MD_ERROR_WIN_CLASS_HAS_WINDOWS = 1412,
  MD_ERROR_WIN_INVALID_INDEX = 1413,
  MD_ERROR_WIN_INVALID_ICON_HANDLE = 1414,
  MD_ERROR_WIN_PRIVATE_DIALOG_INDEX = 1415,
  MD_ERROR_WIN_LISTBOX_ID_NOT_FOUND = 1416,
  MD_ERROR_WIN_NO_WILDCARD_CHARACTERS = 1417,
  MD_ERROR_WIN_CLIPBOARD_NOT_OPEN = 1418,
  MD_ERROR_WIN_HOTKEY_NOT_REGISTERED = 1419,
  MD_ERROR_WIN_WINDOW_NOT_DIALOG = 1420,
  MD_ERROR_WIN_CONTROL_ID_NOT_FOUND = 1421,
  MD_ERROR_WIN_INVALID_COMBOBOX_MESSAGE = 1422,
  MD_ERROR_WIN_WINDOW_NOT_COMBOBOX = 1423,
  MD_ERROR_WIN_INVALID_EDIT_HEIGHT = 1424,
  MD_ERROR_WIN_DC_NOT_FOUND = 1425,
  MD_ERROR_WIN_INVALID_HOOK_FILTER = 1426,
  MD_ERROR_WIN_INVALID_FILTER_PROC = 1427,
  MD_ERROR_WIN_HOOK_NEEDS_HMOD = 1428,
  MD_ERROR_WIN_GLOBAL_ONLY_HOOK = 1429,
  MD_ERROR_WIN_JOURNAL_HOOK_SET = 1430,
  MD_ERROR_WIN_HOOK_NOT_INSTALLED = 1431,
  MD_ERROR_WIN_INVALID_LB_MESSAGE = 1432,
  MD_ERROR_WIN_SETCOUNT_ON_BAD_LB = 1433,
  MD_ERROR_WIN_LB_WITHOUT_TABSTOPS = 1434,
  MD_ERROR_WIN_DESTROY_OBJECT_OF_OTHER_THREAD = 1435,
  MD_ERROR_WIN_CHILD_WINDOW_MENU = 1436,
  MD_ERROR_WIN_NO_SYSTEM_MENU = 1437,
  MD_ERROR_WIN_INVALID_MSGBOX_STYLE = 1438,
  MD_ERROR_WIN_INVALID_SPI_VALUE = 1439,
  MD_ERROR_WIN_SCREEN_ALREADY_LOCKED = 1440,
  MD_ERROR_WIN_HWNDS_HAVE_DIFF_PARENT = 1441,
  MD_ERROR_WIN_NOT_CHILD_WINDOW = 1442,
  MD_ERROR_WIN_INVALID_GW_COMMAND = 1443,
  MD_ERROR_WIN_INVALID_THREAD_ID = 1444,
  MD_ERROR_WIN_NON_MDICHILD_WINDOW = 1445,
  MD_ERROR_WIN_POPUP_ALREADY_ACTIVE = 1446,
  MD_ERROR_WIN_NO_SCROLLBARS = 1447,
  MD_ERROR_WIN_INVALID_SCROLLBAR_RANGE = 1448,
  MD_ERROR_WIN_INVALID_SHOWWIN_COMMAND = 1449,
  MD_ERROR_WIN_NO_SYSTEM_RESOURCES = 1450,
  MD_ERROR_WIN_NONPAGED_SYSTEM_RESOURCES = 1451,
  MD_ERROR_WIN_PAGED_SYSTEM_RESOURCES = 1452,
  MD_ERROR_WIN_WORKING_SET_QUOTA = 1453,
  MD_ERROR_WIN_PAGEFILE_QUOTA = 1454,
  MD_ERROR_WIN_COMMITMENT_LIMIT = 1455,
  MD_ERROR_WIN_MENU_ITEM_NOT_FOUND = 1456,
  MD_ERROR_WIN_INVALID_KEYBOARD_HANDLE = 1457,
  MD_ERROR_WIN_HOOK_TYPE_NOT_ALLOWED = 1458,
  MD_ERROR_WIN_REQUIRES_INTERACTIVE_WINDOWSTATION = 1459,
  MD_ERROR_WIN_TIMEOUT = 1460,
  MD_ERROR_WIN_INVALID_MONITOR_HANDLE = 1461,
  MD_ERROR_WIN_INCORRECT_SIZE = 1462,
  MD_ERROR_WIN_SYMLINK_CLASS_DISABLED = 1463,
  MD_ERROR_WIN_SYMLINK_NOT_SUPPORTED = 1464,
  MD_ERROR_WIN_XML_PARSE_ERROR = 1465,
  MD_ERROR_WIN_XMLDSIG_ERROR = 1466,
  MD_ERROR_WIN_RESTART_APPLICATION = 1467,
  MD_ERROR_WIN_WRONG_COMPARTMENT = 1468,
  MD_ERROR_WIN_AUTHIP_FAILURE = 1469,
  MD_ERROR_WIN_NO_NVRAM_RESOURCES = 1470,
  MD_ERROR_WIN_NOT_GUI_PROCESS = 1471,
  MD_ERROR_WIN_EVENTLOG_FILE_CORRUPT = 1500,
  MD_ERROR_WIN_EVENTLOG_CANT_START = 1501,
  MD_ERROR_WIN_LOG_FILE_FULL = 1502,
  MD_ERROR_WIN_EVENTLOG_FILE_CHANGED = 1503,
  MD_ERROR_WIN_CONTAINER_ASSIGNED = 1504,
  MD_ERROR_WIN_JOB_NO_CONTAINER = 1505,
  MD_ERROR_WIN_INVALID_TASK_NAME = 1550,
  MD_ERROR_WIN_INVALID_TASK_INDEX = 1551,
  MD_ERROR_WIN_THREAD_ALREADY_IN_TASK = 1552,
  MD_ERROR_WIN_INSTALL_SERVICE_FAILURE = 1601,
  MD_ERROR_WIN_INSTALL_USEREXIT = 1602,
  MD_ERROR_WIN_INSTALL_FAILURE = 1603,
  MD_ERROR_WIN_INSTALL_SUSPEND = 1604,
  MD_ERROR_WIN_UNKNOWN_PRODUCT = 1605,
  MD_ERROR_WIN_UNKNOWN_FEATURE = 1606,
  MD_ERROR_WIN_UNKNOWN_COMPONENT = 1607,
  MD_ERROR_WIN_UNKNOWN_PROPERTY = 1608,
  MD_ERROR_WIN_INVALID_HANDLE_STATE = 1609,
  MD_ERROR_WIN_BAD_CONFIGURATION = 1610,
  MD_ERROR_WIN_INDEX_ABSENT = 1611,
  MD_ERROR_WIN_INSTALL_SOURCE_ABSENT = 1612,
  MD_ERROR_WIN_INSTALL_PACKAGE_VERSION = 1613,
  MD_ERROR_WIN_PRODUCT_UNINSTALLED = 1614,
  MD_ERROR_WIN_BAD_QUERY_SYNTAX = 1615,
  MD_ERROR_WIN_INVALID_FIELD = 1616,
  MD_ERROR_WIN_DEVICE_REMOVED = 1617,
  MD_ERROR_WIN_INSTALL_ALREADY_RUNNING = 1618,
  MD_ERROR_WIN_INSTALL_PACKAGE_OPEN_FAILED = 1619,
  MD_ERROR_WIN_INSTALL_PACKAGE_INVALID = 1620,
  MD_ERROR_WIN_INSTALL_UI_FAILURE = 1621,
  MD_ERROR_WIN_INSTALL_LOG_FAILURE = 1622,
  MD_ERROR_WIN_INSTALL_LANGUAGE_UNSUPPORTED = 1623,
  MD_ERROR_WIN_INSTALL_TRANSFORM_FAILURE = 1624,
  MD_ERROR_WIN_INSTALL_PACKAGE_REJECTED = 1625,
  MD_ERROR_WIN_FUNCTION_NOT_CALLED = 1626,
  MD_ERROR_WIN_FUNCTION_FAILED = 1627,
  MD_ERROR_WIN_INVALID_TABLE = 1628,
  MD_ERROR_WIN_DATATYPE_MISMATCH = 1629,
  MD_ERROR_WIN_UNSUPPORTED_TYPE = 1630,
  MD_ERROR_WIN_CREATE_FAILED = 1631,
  MD_ERROR_WIN_INSTALL_TEMP_UNWRITABLE = 1632,
  MD_ERROR_WIN_INSTALL_PLATFORM_UNSUPPORTED = 1633,
  MD_ERROR_WIN_INSTALL_NOTUSED = 1634,
  MD_ERROR_WIN_PATCH_PACKAGE_OPEN_FAILED = 1635,
  MD_ERROR_WIN_PATCH_PACKAGE_INVALID = 1636,
  MD_ERROR_WIN_PATCH_PACKAGE_UNSUPPORTED = 1637,
  MD_ERROR_WIN_PRODUCT_VERSION = 1638,
  MD_ERROR_WIN_INVALID_COMMAND_LINE = 1639,
  MD_ERROR_WIN_INSTALL_REMOTE_DISALLOWED = 1640,
  MD_ERROR_WIN_SUCCESS_REBOOT_INITIATED = 1641,
  MD_ERROR_WIN_PATCH_TARGET_NOT_FOUND = 1642,
  MD_ERROR_WIN_PATCH_PACKAGE_REJECTED = 1643,
  MD_ERROR_WIN_INSTALL_TRANSFORM_REJECTED = 1644,
  MD_ERROR_WIN_INSTALL_REMOTE_PROHIBITED = 1645,
  MD_ERROR_WIN_PATCH_REMOVAL_UNSUPPORTED = 1646,
  MD_ERROR_WIN_UNKNOWN_PATCH = 1647,
  MD_ERROR_WIN_PATCH_NO_SEQUENCE = 1648,
  MD_ERROR_WIN_PATCH_REMOVAL_DISALLOWED = 1649,
  MD_ERROR_WIN_INVALID_PATCH_XML = 1650,
  MD_ERROR_WIN_PATCH_MANAGED_ADVERTISED_PRODUCT = 1651,
  MD_ERROR_WIN_INSTALL_SERVICE_SAFEBOOT = 1652,
  MD_ERROR_WIN_FAIL_FAST_EXCEPTION = 1653,
  MD_ERROR_WIN_INSTALL_REJECTED = 1654,
  MD_ERROR_WIN_DYNAMIC_CODE_BLOCKED = 1655,
  MD_ERROR_WIN_NOT_SAME_OBJECT = 1656,
  MD_ERROR_WIN_STRICT_CFG_VIOLATION = 1657,
  MD_ERROR_WIN_SET_CONTEXT_DENIED = 1660,
  MD_ERROR_WIN_CROSS_PARTITION_VIOLATION = 1661,
  MD_ERROR_WIN_RETURN_ADDRESS_HIJACK_ATTEMPT = 1662,
  MD_ERROR_WIN_INVALID_USER_BUFFER = 1784,
  MD_ERROR_WIN_UNRECOGNIZED_MEDIA = 1785,
  MD_ERROR_WIN_NO_TRUST_LSA_SECRET = 1786,
  MD_ERROR_WIN_NO_TRUST_SAM_ACCOUNT = 1787,
  MD_ERROR_WIN_TRUSTED_DOMAIN_FAILURE = 1788,
  MD_ERROR_WIN_TRUSTED_RELATIONSHIP_FAILURE = 1789,
  MD_ERROR_WIN_TRUST_FAILURE = 1790,
  MD_ERROR_WIN_NETLOGON_NOT_STARTED = 1792,
  MD_ERROR_WIN_ACCOUNT_EXPIRED = 1793,
  MD_ERROR_WIN_REDIRECTOR_HAS_OPEN_HANDLES = 1794,
  MD_ERROR_WIN_PRINTER_DRIVER_ALREADY_INSTALLED = 1795,
  MD_ERROR_WIN_UNKNOWN_PORT = 1796,
  MD_ERROR_WIN_UNKNOWN_PRINTER_DRIVER = 1797,
  MD_ERROR_WIN_UNKNOWN_PRINTPROCESSOR = 1798,
  MD_ERROR_WIN_INVALID_SEPARATOR_FILE = 1799,
  MD_ERROR_WIN_INVALID_PRIORITY = 1800,
  MD_ERROR_WIN_INVALID_PRINTER_NAME = 1801,
  MD_ERROR_WIN_PRINTER_ALREADY_EXISTS = 1802,
  MD_ERROR_WIN_INVALID_PRINTER_COMMAND = 1803,
  MD_ERROR_WIN_INVALID_DATATYPE = 1804,
  MD_ERROR_WIN_INVALID_ENVIRONMENT = 1805,
  MD_ERROR_WIN_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT = 1807,
  MD_ERROR_WIN_NOLOGON_WORKSTATION_TRUST_ACCOUNT = 1808,
  MD_ERROR_WIN_NOLOGON_SERVER_TRUST_ACCOUNT = 1809,
  MD_ERROR_WIN_DOMAIN_TRUST_INCONSISTENT = 1810,
  MD_ERROR_WIN_SERVER_HAS_OPEN_HANDLES = 1811,
  MD_ERROR_WIN_RESOURCE_DATA_NOT_FOUND = 1812,
  MD_ERROR_WIN_RESOURCE_TYPE_NOT_FOUND = 1813,
  MD_ERROR_WIN_RESOURCE_NAME_NOT_FOUND = 1814,
  MD_ERROR_WIN_RESOURCE_LANG_NOT_FOUND = 1815,
  MD_ERROR_WIN_NOT_ENOUGH_QUOTA = 1816,
  MD_ERROR_WIN_INVALID_TIME = 1901,
  MD_ERROR_WIN_INVALID_FORM_NAME = 1902,
  MD_ERROR_WIN_INVALID_FORM_SIZE = 1903,
  MD_ERROR_WIN_ALREADY_WAITING = 1904,
  MD_ERROR_WIN_PRINTER_DELETED = 1905,
  MD_ERROR_WIN_INVALID_PRINTER_STATE = 1906,
  MD_ERROR_WIN_PASSWORD_MUST_CHANGE = 1907,
  MD_ERROR_WIN_DOMAIN_CONTROLLER_NOT_FOUND = 1908,
  MD_ERROR_WIN_ACCOUNT_LOCKED_OUT = 1909,
  MD_ERROR_WIN_NO_SITENAME = 1919,
  MD_ERROR_WIN_CANT_ACCESS_FILE = 1920,
  MD_ERROR_WIN_CANT_RESOLVE_FILENAME = 1921,
  MD_ERROR_WIN_KM_DRIVER_BLOCKED = 1930,
  MD_ERROR_WIN_CONTEXT_EXPIRED = 1931,
  MD_ERROR_WIN_PER_USER_TRUST_QUOTA_EXCEEDED = 1932,
  MD_ERROR_WIN_ALL_USER_TRUST_QUOTA_EXCEEDED = 1933,
  MD_ERROR_WIN_USER_DELETE_TRUST_QUOTA_EXCEEDED = 1934,
  MD_ERROR_WIN_AUTHENTICATION_FIREWALL_FAILED = 1935,
  MD_ERROR_WIN_REMOTE_PRINT_CONNECTIONS_BLOCKED = 1936,
  MD_ERROR_WIN_NTLM_BLOCKED = 1937,
  MD_ERROR_WIN_PASSWORD_CHANGE_REQUIRED = 1938,
  MD_ERROR_WIN_LOST_MODE_LOGON_RESTRICTION = 1939,
  MD_ERROR_WIN_INVALID_PIXEL_FORMAT = 2000,
  MD_ERROR_WIN_BAD_DRIVER = 2001,
  MD_ERROR_WIN_INVALID_WINDOW_STYLE = 2002,
  MD_ERROR_WIN_METAFILE_NOT_SUPPORTED = 2003,
  MD_ERROR_WIN_TRANSFORM_NOT_SUPPORTED = 2004,
  MD_ERROR_WIN_CLIPPING_NOT_SUPPORTED = 2005,
  MD_ERROR_WIN_INVALID_CMM = 2010,
  MD_ERROR_WIN_INVALID_PROFILE = 2011,
  MD_ERROR_WIN_TAG_NOT_FOUND = 2012,
  MD_ERROR_WIN_TAG_NOT_PRESENT = 2013,
  MD_ERROR_WIN_DUPLICATE_TAG = 2014,
  MD_ERROR_WIN_PROFILE_NOT_ASSOCIATED_WITH_DEVICE = 2015,
  MD_ERROR_WIN_PROFILE_NOT_FOUND = 2016,
  MD_ERROR_WIN_INVALID_COLORSPACE = 2017,
  MD_ERROR_WIN_ICM_NOT_ENABLED = 2018,
  MD_ERROR_WIN_DELETING_ICM_XFORM = 2019,
  MD_ERROR_WIN_INVALID_TRANSFORM = 2020,
  MD_ERROR_WIN_COLORSPACE_MISMATCH = 2021,
  MD_ERROR_WIN_INVALID_COLORINDEX = 2022,
  MD_ERROR_WIN_PROFILE_DOES_NOT_MATCH_DEVICE = 2023,
  MD_ERROR_WIN_CONNECTED_OTHER_PASSWORD = 2108,
  MD_ERROR_WIN_CONNECTED_OTHER_PASSWORD_DEFAULT = 2109,
  MD_ERROR_WIN_BAD_USERNAME = 2202,
  MD_ERROR_WIN_NOT_CONNECTED = 2250,
  MD_ERROR_WIN_OPEN_FILES = 2401,
  MD_ERROR_WIN_ACTIVE_CONNECTIONS = 2402,
  MD_ERROR_WIN_DEVICE_IN_USE = 2404,
  MD_ERROR_WIN_UNKNOWN_PRINT_MONITOR = 3000,
  MD_ERROR_WIN_PRINTER_DRIVER_IN_USE = 3001,
  MD_ERROR_WIN_SPOOL_FILE_NOT_FOUND = 3002,
  MD_ERROR_WIN_SPL_NO_STARTDOC = 3003,
  MD_ERROR_WIN_SPL_NO_ADDJOB = 3004,
  MD_ERROR_WIN_PRINT_PROCESSOR_ALREADY_INSTALLED = 3005,
  MD_ERROR_WIN_PRINT_MONITOR_ALREADY_INSTALLED = 3006,
  MD_ERROR_WIN_INVALID_PRINT_MONITOR = 3007,
  MD_ERROR_WIN_PRINT_MONITOR_IN_USE = 3008,
  MD_ERROR_WIN_PRINTER_HAS_JOBS_QUEUED = 3009,
  MD_ERROR_WIN_SUCCESS_REBOOT_REQUIRED = 3010,
  MD_ERROR_WIN_SUCCESS_RESTART_REQUIRED = 3011,
  MD_ERROR_WIN_PRINTER_NOT_FOUND = 3012,
  MD_ERROR_WIN_PRINTER_DRIVER_WARNED = 3013,
  MD_ERROR_WIN_PRINTER_DRIVER_BLOCKED = 3014,
  MD_ERROR_WIN_PRINTER_DRIVER_PACKAGE_IN_USE = 3015,
  MD_ERROR_WIN_CORE_DRIVER_PACKAGE_NOT_FOUND = 3016,
  MD_ERROR_WIN_FAIL_REBOOT_REQUIRED = 3017,
  MD_ERROR_WIN_FAIL_REBOOT_INITIATED = 3018,
  MD_ERROR_WIN_PRINTER_DRIVER_DOWNLOAD_NEEDED = 3019,
  MD_ERROR_WIN_PRINT_JOB_RESTART_REQUIRED = 3020,
  MD_ERROR_WIN_INVALID_PRINTER_DRIVER_MANIFEST = 3021,
  MD_ERROR_WIN_PRINTER_NOT_SHAREABLE = 3022,
  MD_ERROR_WIN_REQUEST_PAUSED = 3050,
  MD_ERROR_WIN_APPEXEC_CONDITION_NOT_SATISFIED = 3060,
  MD_ERROR_WIN_APPEXEC_HANDLE_INVALIDATED = 3061,
  MD_ERROR_WIN_APPEXEC_INVALID_HOST_GENERATION = 3062,
  MD_ERROR_WIN_APPEXEC_UNEXPECTED_PROCESS_REGISTRATION = 3063,
  MD_ERROR_WIN_APPEXEC_INVALID_HOST_STATE = 3064,
  MD_ERROR_WIN_APPEXEC_NO_DONOR = 3065,
  MD_ERROR_WIN_APPEXEC_HOST_ID_MISMATCH = 3066,
  MD_ERROR_WIN_APPEXEC_UNKNOWN_USER = 3067,
  MD_ERROR_WIN_IO_REISSUE_AS_CACHED = 3950,
  MD_ERROR_WIN_WINS_INTERNAL = 4000,
  MD_ERROR_WIN_CAN_NOT_DEL_LOCAL_WINS = 4001,
  MD_ERROR_WIN_STATIC_INIT = 4002,
  MD_ERROR_WIN_INC_BACKUP = 4003,
  MD_ERROR_WIN_FULL_BACKUP = 4004,
  MD_ERROR_WIN_REC_NON_EXISTENT = 4005,
  MD_ERROR_WIN_RPL_NOT_ALLOWED = 4006,
  MD_ERROR_WIN_DHCP_ADDRESS_CONFLICT = 4100,
  MD_ERROR_WIN_WMI_GUID_NOT_FOUND = 4200,
  MD_ERROR_WIN_WMI_INSTANCE_NOT_FOUND = 4201,
  MD_ERROR_WIN_WMI_ITEMID_NOT_FOUND = 4202,
  MD_ERROR_WIN_WMI_TRY_AGAIN = 4203,
  MD_ERROR_WIN_WMI_DP_NOT_FOUND = 4204,
  MD_ERROR_WIN_WMI_UNRESOLVED_INSTANCE_REF = 4205,
  MD_ERROR_WIN_WMI_ALREADY_ENABLED = 4206,
  MD_ERROR_WIN_WMI_GUID_DISCONNECTED = 4207,
  MD_ERROR_WIN_WMI_SERVER_UNAVAILABLE = 4208,
  MD_ERROR_WIN_WMI_DP_FAILED = 4209,
  MD_ERROR_WIN_WMI_INVALID_MOF = 4210,
  MD_ERROR_WIN_WMI_INVALID_REGINFO = 4211,
  MD_ERROR_WIN_WMI_ALREADY_DISABLED = 4212,
  MD_ERROR_WIN_WMI_READ_ONLY = 4213,
  MD_ERROR_WIN_WMI_SET_FAILURE = 4214,
  MD_ERROR_WIN_NOT_APPCONTAINER = 4250,
  MD_ERROR_WIN_APPCONTAINER_REQUIRED = 4251,
  MD_ERROR_WIN_NOT_SUPPORTED_IN_APPCONTAINER = 4252,
  MD_ERROR_WIN_INVALID_PACKAGE_SID_LENGTH = 4253,
  MD_ERROR_WIN_INVALID_MEDIA = 4300,
  MD_ERROR_WIN_INVALID_LIBRARY = 4301,
  MD_ERROR_WIN_INVALID_MEDIA_POOL = 4302,
  MD_ERROR_WIN_DRIVE_MEDIA_MISMATCH = 4303,
  MD_ERROR_WIN_MEDIA_OFFLINE = 4304,
  MD_ERROR_WIN_LIBRARY_OFFLINE = 4305,
  MD_ERROR_WIN_EMPTY = 4306,
  MD_ERROR_WIN_NOT_EMPTY = 4307,
  MD_ERROR_WIN_MEDIA_UNAVAILABLE = 4308,
  MD_ERROR_WIN_RESOURCE_DISABLED = 4309,
  MD_ERROR_WIN_INVALID_CLEANER = 4310,
  MD_ERROR_WIN_UNABLE_TO_CLEAN = 4311,
  MD_ERROR_WIN_OBJECT_NOT_FOUND = 4312,
  MD_ERROR_WIN_DATABASE_FAILURE = 4313,
  MD_ERROR_WIN_DATABASE_FULL = 4314,
  MD_ERROR_WIN_MEDIA_INCOMPATIBLE = 4315,
  MD_ERROR_WIN_RESOURCE_NOT_PRESENT = 4316,
  MD_ERROR_WIN_INVALID_OPERATION = 4317,
  MD_ERROR_WIN_MEDIA_NOT_AVAILABLE = 4318,
  MD_ERROR_WIN_DEVICE_NOT_AVAILABLE = 4319,
  MD_ERROR_WIN_REQUEST_REFUSED = 4320,
  MD_ERROR_WIN_INVALID_DRIVE_OBJECT = 4321,
  MD_ERROR_WIN_LIBRARY_FULL = 4322,
  MD_ERROR_WIN_MEDIUM_NOT_ACCESSIBLE = 4323,
  MD_ERROR_WIN_UNABLE_TO_LOAD_MEDIUM = 4324,
  MD_ERROR_WIN_UNABLE_TO_INVENTORY_DRIVE = 4325,
  MD_ERROR_WIN_UNABLE_TO_INVENTORY_SLOT = 4326,
  MD_ERROR_WIN_UNABLE_TO_INVENTORY_TRANSPORT = 4327,
  MD_ERROR_WIN_TRANSPORT_FULL = 4328,
  MD_ERROR_WIN_CONTROLLING_IEPORT = 4329,
  MD_ERROR_WIN_UNABLE_TO_EJECT_MOUNTED_MEDIA = 4330,
  MD_ERROR_WIN_CLEANER_SLOT_SET = 4331,
  MD_ERROR_WIN_CLEANER_SLOT_NOT_SET = 4332,
  MD_ERROR_WIN_CLEANER_CARTRIDGE_SPENT = 4333,
  MD_ERROR_WIN_UNEXPECTED_OMID = 4334,
  MD_ERROR_WIN_CANT_DELETE_LAST_ITEM = 4335,
  MD_ERROR_WIN_MESSAGE_EXCEEDS_MAX_SIZE = 4336,
  MD_ERROR_WIN_VOLUME_CONTAINS_SYS_FILES = 4337,
  MD_ERROR_WIN_INDIGENOUS_TYPE = 4338,
  MD_ERROR_WIN_NO_SUPPORTING_DRIVES = 4339,
  MD_ERROR_WIN_CLEANER_CARTRIDGE_INSTALLED = 4340,
  MD_ERROR_WIN_IEPORT_FULL = 4341,
  MD_ERROR_WIN_FILE_OFFLINE = 4350,
  MD_ERROR_WIN_REMOTE_STORAGE_NOT_ACTIVE = 4351,
  MD_ERROR_WIN_REMOTE_STORAGE_MEDIA_ERROR = 4352,
  MD_ERROR_WIN_NOT_A_REPARSE_POINT = 4390,
  MD_ERROR_WIN_REPARSE_ATTRIBUTE_CONFLICT = 4391,
  MD_ERROR_WIN_INVALID_REPARSE_DATA = 4392,
  MD_ERROR_WIN_REPARSE_TAG_INVALID = 4393,
  MD_ERROR_WIN_REPARSE_TAG_MISMATCH = 4394,
  MD_ERROR_WIN_REPARSE_POINT_ENCOUNTERED = 4395,
  MD_ERROR_WIN_APP_DATA_NOT_FOUND = 4400,
  MD_ERROR_WIN_APP_DATA_EXPIRED = 4401,
  MD_ERROR_WIN_APP_DATA_CORRUPT = 4402,
  MD_ERROR_WIN_APP_DATA_LIMIT_EXCEEDED = 4403,
  MD_ERROR_WIN_APP_DATA_REBOOT_REQUIRED = 4404,
  MD_ERROR_WIN_SECUREBOOT_ROLLBACK_DETECTED = 4420,
  MD_ERROR_WIN_SECUREBOOT_POLICY_VIOLATION = 4421,
  MD_ERROR_WIN_SECUREBOOT_INVALID_POLICY = 4422,
  MD_ERROR_WIN_SECUREBOOT_POLICY_PUBLISHER_NOT_FOUND = 4423,
  MD_ERROR_WIN_SECUREBOOT_POLICY_NOT_SIGNED = 4424,
  MD_ERROR_WIN_SECUREBOOT_NOT_ENABLED = 4425,
  MD_ERROR_WIN_SECUREBOOT_FILE_REPLACED = 4426,
  MD_ERROR_WIN_SECUREBOOT_POLICY_NOT_AUTHORIZED = 4427,
  MD_ERROR_WIN_SECUREBOOT_POLICY_UNKNOWN = 4428,
  MD_ERROR_WIN_SECUREBOOT_POLICY_MISSING_ANTIROLLBACKVERSION = 4429,
  MD_ERROR_WIN_SECUREBOOT_PLATFORM_ID_MISMATCH = 4430,
  MD_ERROR_WIN_SECUREBOOT_POLICY_ROLLBACK_DETECTED = 4431,
  MD_ERROR_WIN_SECUREBOOT_POLICY_UPGRADE_MISMATCH = 4432,
  MD_ERROR_WIN_SECUREBOOT_REQUIRED_POLICY_FILE_MISSING = 4433,
  MD_ERROR_WIN_SECUREBOOT_NOT_BASE_POLICY = 4434,
  MD_ERROR_WIN_SECUREBOOT_NOT_SUPPLEMENTAL_POLICY = 4435,
  MD_ERROR_WIN_OFFLOAD_READ_FLT_NOT_SUPPORTED = 4440,
  MD_ERROR_WIN_OFFLOAD_WRITE_FLT_NOT_SUPPORTED = 4441,
  MD_ERROR_WIN_OFFLOAD_READ_FILE_NOT_SUPPORTED = 4442,
  MD_ERROR_WIN_OFFLOAD_WRITE_FILE_NOT_SUPPORTED = 4443,
  MD_ERROR_WIN_ALREADY_HAS_STREAM_ID = 4444,
  MD_ERROR_WIN_SMR_GARBAGE_COLLECTION_REQUIRED = 4445,
  MD_ERROR_WIN_WOF_WIM_HEADER_CORRUPT = 4446,
  MD_ERROR_WIN_WOF_WIM_RESOURCE_TABLE_CORRUPT = 4447,
  MD_ERROR_WIN_WOF_FILE_RESOURCE_TABLE_CORRUPT = 4448,
  MD_ERROR_WIN_VOLUME_NOT_SIS_ENABLED = 4500,
  MD_ERROR_WIN_SYSTEM_INTEGRITY_ROLLBACK_DETECTED = 4550,
  MD_ERROR_WIN_SYSTEM_INTEGRITY_POLICY_VIOLATION = 4551,
  MD_ERROR_WIN_SYSTEM_INTEGRITY_INVALID_POLICY = 4552,
  MD_ERROR_WIN_SYSTEM_INTEGRITY_POLICY_NOT_SIGNED = 4553,
  MD_ERROR_WIN_SYSTEM_INTEGRITY_TOO_MANY_POLICIES = 4554,
  MD_ERROR_WIN_SYSTEM_INTEGRITY_SUPPLEMENTAL_POLICY_NOT_AUTHORIZED = 4555,
  MD_ERROR_WIN_VSM_NOT_INITIALIZED = 4560,
  MD_ERROR_WIN_VSM_DMA_PROTECTION_NOT_IN_USE = 4561,
  MD_ERROR_WIN_PLATFORM_MANIFEST_NOT_AUTHORIZED = 4570,
  MD_ERROR_WIN_PLATFORM_MANIFEST_INVALID = 4571,
  MD_ERROR_WIN_PLATFORM_MANIFEST_FILE_NOT_AUTHORIZED = 4572,
  MD_ERROR_WIN_PLATFORM_MANIFEST_CATALOG_NOT_AUTHORIZED = 4573,
  MD_ERROR_WIN_PLATFORM_MANIFEST_BINARY_ID_NOT_FOUND = 4574,
  MD_ERROR_WIN_PLATFORM_MANIFEST_NOT_ACTIVE = 4575,
  MD_ERROR_WIN_PLATFORM_MANIFEST_NOT_SIGNED = 4576,
  MD_ERROR_WIN_DEPENDENT_RESOURCE_EXISTS = 5001,
  MD_ERROR_WIN_DEPENDENCY_NOT_FOUND = 5002,
  MD_ERROR_WIN_DEPENDENCY_ALREADY_EXISTS = 5003,
  MD_ERROR_WIN_RESOURCE_NOT_ONLINE = 5004,
  MD_ERROR_WIN_HOST_NODE_NOT_AVAILABLE = 5005,
  MD_ERROR_WIN_RESOURCE_NOT_AVAILABLE = 5006,
  MD_ERROR_WIN_RESOURCE_NOT_FOUND = 5007,
  MD_ERROR_WIN_SHUTDOWN_CLUSTER = 5008,
  MD_ERROR_WIN_CANT_EVICT_ACTIVE_NODE = 5009,
  MD_ERROR_WIN_OBJECT_ALREADY_EXISTS = 5010,
  MD_ERROR_WIN_OBJECT_IN_LIST = 5011,
  MD_ERROR_WIN_GROUP_NOT_AVAILABLE = 5012,
  MD_ERROR_WIN_GROUP_NOT_FOUND = 5013,
  MD_ERROR_WIN_GROUP_NOT_ONLINE = 5014,
  MD_ERROR_WIN_HOST_NODE_NOT_RESOURCE_OWNER = 5015,
  MD_ERROR_WIN_HOST_NODE_NOT_GROUP_OWNER = 5016,
  MD_ERROR_WIN_RESMON_CREATE_FAILED = 5017,
  MD_ERROR_WIN_RESMON_ONLINE_FAILED = 5018,
  MD_ERROR_WIN_RESOURCE_ONLINE = 5019,
  MD_ERROR_WIN_QUORUM_RESOURCE = 5020,
  MD_ERROR_WIN_NOT_QUORUM_CAPABLE = 5021,
  MD_ERROR_WIN_CLUSTER_SHUTTING_DOWN = 5022,
  MD_ERROR_WIN_INVALID_STATE = 5023,
  MD_ERROR_WIN_RESOURCE_PROPERTIES_STORED = 5024,
  MD_ERROR_WIN_NOT_QUORUM_CLASS = 5025,
  MD_ERROR_WIN_CORE_RESOURCE = 5026,
  MD_ERROR_WIN_QUORUM_RESOURCE_ONLINE_FAILED = 5027,
  MD_ERROR_WIN_QUORUMLOG_OPEN_FAILED = 5028,
  MD_ERROR_WIN_CLUSTERLOG_CORRUPT = 5029,
  MD_ERROR_WIN_CLUSTERLOG_RECORD_EXCEEDS_MAXSIZE = 5030,
  MD_ERROR_WIN_CLUSTERLOG_EXCEEDS_MAXSIZE = 5031,
  MD_ERROR_WIN_CLUSTERLOG_CHKPOINT_NOT_FOUND = 5032,
  MD_ERROR_WIN_CLUSTERLOG_NOT_ENOUGH_SPACE = 5033,
  MD_ERROR_WIN_QUORUM_OWNER_ALIVE = 5034,
  MD_ERROR_WIN_NETWORK_NOT_AVAILABLE = 5035,
  MD_ERROR_WIN_NODE_NOT_AVAILABLE = 5036,
  MD_ERROR_WIN_ALL_NODES_NOT_AVAILABLE = 5037,
  MD_ERROR_WIN_RESOURCE_FAILED = 5038,
  MD_ERROR_WIN_CLUSTER_INVALID_NODE = 5039,
  MD_ERROR_WIN_CLUSTER_NODE_EXISTS = 5040,
  MD_ERROR_WIN_CLUSTER_JOIN_IN_PROGRESS = 5041,
  MD_ERROR_WIN_CLUSTER_NODE_NOT_FOUND = 5042,
  MD_ERROR_WIN_CLUSTER_LOCAL_NODE_NOT_FOUND = 5043,
  MD_ERROR_WIN_CLUSTER_NETWORK_EXISTS = 5044,
  MD_ERROR_WIN_CLUSTER_NETWORK_NOT_FOUND = 5045,
  MD_ERROR_WIN_CLUSTER_NETINTERFACE_EXISTS = 5046,
  MD_ERROR_WIN_CLUSTER_NETINTERFACE_NOT_FOUND = 5047,
  MD_ERROR_WIN_CLUSTER_INVALID_REQUEST = 5048,
  MD_ERROR_WIN_CLUSTER_INVALID_NETWORK_PROVIDER = 5049,
  MD_ERROR_WIN_CLUSTER_NODE_DOWN = 5050,
  MD_ERROR_WIN_CLUSTER_NODE_UNREACHABLE = 5051,
  MD_ERROR_WIN_CLUSTER_NODE_NOT_MEMBER = 5052,
  MD_ERROR_WIN_CLUSTER_JOIN_NOT_IN_PROGRESS = 5053,
  MD_ERROR_WIN_CLUSTER_INVALID_NETWORK = 5054,
  MD_ERROR_WIN_CLUSTER_NODE_UP = 5056,
  MD_ERROR_WIN_CLUSTER_IPADDR_IN_USE = 5057,
  MD_ERROR_WIN_CLUSTER_NODE_NOT_PAUSED = 5058,
  MD_ERROR_WIN_CLUSTER_NO_SECURITY_CONTEXT = 5059,
  MD_ERROR_WIN_CLUSTER_NETWORK_NOT_INTERNAL = 5060,
  MD_ERROR_WIN_CLUSTER_NODE_ALREADY_UP = 5061,
  MD_ERROR_WIN_CLUSTER_NODE_ALREADY_DOWN = 5062,
  MD_ERROR_WIN_CLUSTER_NETWORK_ALREADY_ONLINE = 5063,
  MD_ERROR_WIN_CLUSTER_NETWORK_ALREADY_OFFLINE = 5064,
  MD_ERROR_WIN_CLUSTER_NODE_ALREADY_MEMBER = 5065,
  MD_ERROR_WIN_CLUSTER_LAST_INTERNAL_NETWORK = 5066,
  MD_ERROR_WIN_CLUSTER_NETWORK_HAS_DEPENDENTS = 5067,
  MD_ERROR_WIN_INVALID_OPERATION_ON_QUORUM = 5068,
  MD_ERROR_WIN_DEPENDENCY_NOT_ALLOWED = 5069,
  MD_ERROR_WIN_CLUSTER_NODE_PAUSED = 5070,
  MD_ERROR_WIN_NODE_CANT_HOST_RESOURCE = 5071,
  MD_ERROR_WIN_CLUSTER_NODE_NOT_READY = 5072,
  MD_ERROR_WIN_CLUSTER_NODE_SHUTTING_DOWN = 5073,
  MD_ERROR_WIN_CLUSTER_JOIN_ABORTED = 5074,
  MD_ERROR_WIN_CLUSTER_INCOMPATIBLE_VERSIONS = 5075,
  MD_ERROR_WIN_CLUSTER_MAXNUM_OF_RESOURCES_EXCEEDED = 5076,
  MD_ERROR_WIN_CLUSTER_SYSTEM_CONFIG_CHANGED = 5077,
  MD_ERROR_WIN_CLUSTER_RESOURCE_TYPE_NOT_FOUND = 5078,
  MD_ERROR_WIN_CLUSTER_RESTYPE_NOT_SUPPORTED = 5079,
  MD_ERROR_WIN_CLUSTER_RESNAME_NOT_FOUND = 5080,
  MD_ERROR_WIN_CLUSTER_NO_RPC_PACKAGES_REGISTERED = 5081,
  MD_ERROR_WIN_CLUSTER_OWNER_NOT_IN_PREFLIST = 5082,
  MD_ERROR_WIN_CLUSTER_DATABASE_SEQMISMATCH = 5083,
  MD_ERROR_WIN_RESMON_INVALID_STATE = 5084,
  MD_ERROR_WIN_CLUSTER_GUM_NOT_LOCKER = 5085,
  MD_ERROR_WIN_QUORUM_DISK_NOT_FOUND = 5086,
  MD_ERROR_WIN_DATABASE_BACKUP_CORRUPT = 5087,
  MD_ERROR_WIN_CLUSTER_NODE_ALREADY_HAS_DFS_ROOT = 5088,
  MD_ERROR_WIN_RESOURCE_PROPERTY_UNCHANGEABLE = 5089,
  MD_ERROR_WIN_NO_ADMIN_ACCESS_POINT = 5090,
  MD_ERROR_WIN_CLUSTER_MEMBERSHIP_INVALID_STATE = 5890,
  MD_ERROR_WIN_CLUSTER_QUORUMLOG_NOT_FOUND = 5891,
  MD_ERROR_WIN_CLUSTER_MEMBERSHIP_HALT = 5892,
  MD_ERROR_WIN_CLUSTER_INSTANCE_ID_MISMATCH = 5893,
  MD_ERROR_WIN_CLUSTER_NETWORK_NOT_FOUND_FOR_IP = 5894,
  MD_ERROR_WIN_CLUSTER_PROPERTY_DATA_TYPE_MISMATCH = 5895,
  MD_ERROR_WIN_CLUSTER_EVICT_WITHOUT_CLEANUP = 5896,
  MD_ERROR_WIN_CLUSTER_PARAMETER_MISMATCH = 5897,
  MD_ERROR_WIN_NODE_CANNOT_BE_CLUSTERED = 5898,
  MD_ERROR_WIN_CLUSTER_WRONG_OS_VERSION = 5899,
  MD_ERROR_WIN_CLUSTER_CANT_CREATE_DUP_CLUSTER_NAME = 5900,
  MD_ERROR_WIN_CLUSCFG_ALREADY_COMMITTED = 5901,
  MD_ERROR_WIN_CLUSCFG_ROLLBACK_FAILED = 5902,
  MD_ERROR_WIN_CLUSCFG_SYSTEM_DISK_DRIVE_LETTER_CONFLICT = 5903,
  MD_ERROR_WIN_CLUSTER_OLD_VERSION = 5904,
  MD_ERROR_WIN_CLUSTER_MISMATCHED_COMPUTER_ACCT_NAME = 5905,
  MD_ERROR_WIN_CLUSTER_NO_NET_ADAPTERS = 5906,
  MD_ERROR_WIN_CLUSTER_POISONED = 5907,
  MD_ERROR_WIN_CLUSTER_GROUP_MOVING = 5908,
  MD_ERROR_WIN_CLUSTER_RESOURCE_TYPE_BUSY = 5909,
  MD_ERROR_WIN_RESOURCE_CALL_TIMED_OUT = 5910,
  MD_ERROR_WIN_INVALID_CLUSTER_IPV6_ADDRESS = 5911,
  MD_ERROR_WIN_CLUSTER_INTERNAL_INVALID_FUNCTION = 5912,
  MD_ERROR_WIN_CLUSTER_PARAMETER_OUT_OF_BOUNDS = 5913,
  MD_ERROR_WIN_CLUSTER_PARTIAL_SEND = 5914,
  MD_ERROR_WIN_CLUSTER_REGISTRY_INVALID_FUNCTION = 5915,
  MD_ERROR_WIN_CLUSTER_INVALID_STRING_TERMINATION = 5916,
  MD_ERROR_WIN_CLUSTER_INVALID_STRING_FORMAT = 5917,
  MD_ERROR_WIN_CLUSTER_DATABASE_TRANSACTION_IN_PROGRESS = 5918,
  MD_ERROR_WIN_CLUSTER_DATABASE_TRANSACTION_NOT_IN_PROGRESS = 5919,
  MD_ERROR_WIN_CLUSTER_NULL_DATA = 5920,
  MD_ERROR_WIN_CLUSTER_PARTIAL_READ = 5921,
  MD_ERROR_WIN_CLUSTER_PARTIAL_WRITE = 5922,
  MD_ERROR_WIN_CLUSTER_CANT_DESERIALIZE_DATA = 5923,
  MD_ERROR_WIN_DEPENDENT_RESOURCE_PROPERTY_CONFLICT = 5924,
  MD_ERROR_WIN_CLUSTER_NO_QUORUM = 5925,
  MD_ERROR_WIN_CLUSTER_INVALID_IPV6_NETWORK = 5926,
  MD_ERROR_WIN_CLUSTER_INVALID_IPV6_TUNNEL_NETWORK = 5927,
  MD_ERROR_WIN_QUORUM_NOT_ALLOWED_IN_THIS_GROUP = 5928,
  MD_ERROR_WIN_DEPENDENCY_TREE_TOO_COMPLEX = 5929,
  MD_ERROR_WIN_EXCEPTION_IN_RESOURCE_CALL = 5930,
  MD_ERROR_WIN_CLUSTER_RHS_FAILED_INITIALIZATION = 5931,
  MD_ERROR_WIN_CLUSTER_NOT_INSTALLED = 5932,
  MD_ERROR_WIN_CLUSTER_RESOURCES_MUST_BE_ONLINE_ON_THE_SAME_NODE = 5933,
  MD_ERROR_WIN_CLUSTER_MAX_NODES_IN_CLUSTER = 5934,
  MD_ERROR_WIN_CLUSTER_TOO_MANY_NODES = 5935,
  MD_ERROR_WIN_CLUSTER_OBJECT_ALREADY_USED = 5936,
  MD_ERROR_WIN_NONCORE_GROUPS_FOUND = 5937,
  MD_ERROR_WIN_FILE_SHARE_RESOURCE_CONFLICT = 5938,
  MD_ERROR_WIN_CLUSTER_EVICT_INVALID_REQUEST = 5939,
  MD_ERROR_WIN_CLUSTER_SINGLETON_RESOURCE = 5940,
  MD_ERROR_WIN_CLUSTER_GROUP_SINGLETON_RESOURCE = 5941,
  MD_ERROR_WIN_CLUSTER_RESOURCE_PROVIDER_FAILED = 5942,
  MD_ERROR_WIN_CLUSTER_RESOURCE_CONFIGURATION_ERROR = 5943,
  MD_ERROR_WIN_CLUSTER_GROUP_BUSY = 5944,
  MD_ERROR_WIN_CLUSTER_NOT_SHARED_VOLUME = 5945,
  MD_ERROR_WIN_CLUSTER_INVALID_SECURITY_DESCRIPTOR = 5946,
  MD_ERROR_WIN_CLUSTER_SHARED_VOLUMES_IN_USE = 5947,
  MD_ERROR_WIN_CLUSTER_USE_SHARED_VOLUMES_API = 5948,
  MD_ERROR_WIN_CLUSTER_BACKUP_IN_PROGRESS = 5949,
  MD_ERROR_WIN_NON_CSV_PATH = 5950,
  MD_ERROR_WIN_CSV_VOLUME_NOT_LOCAL = 5951,
  MD_ERROR_WIN_CLUSTER_WATCHDOG_TERMINATING = 5952,
  MD_ERROR_WIN_CLUSTER_RESOURCE_VETOED_MOVE_INCOMPATIBLE_NODES = 5953,
  MD_ERROR_WIN_CLUSTER_INVALID_NODE_WEIGHT = 5954,
  MD_ERROR_WIN_CLUSTER_RESOURCE_VETOED_CALL = 5955,
  MD_ERROR_WIN_RESMON_SYSTEM_RESOURCES_LACKING = 5956,
  MD_ERROR_WIN_CLUSTER_RESOURCE_VETOED_MOVE_NOT_ENOUGH_RESOURCES_ON_DESTINATION = 5957,
  MD_ERROR_WIN_CLUSTER_RESOURCE_VETOED_MOVE_NOT_ENOUGH_RESOURCES_ON_SOURCE = 5958,
  MD_ERROR_WIN_CLUSTER_GROUP_QUEUED = 5959,
  MD_ERROR_WIN_CLUSTER_RESOURCE_LOCKED_STATUS = 5960,
  MD_ERROR_WIN_CLUSTER_SHARED_VOLUME_FAILOVER_NOT_ALLOWED = 5961,
  MD_ERROR_WIN_CLUSTER_NODE_DRAIN_IN_PROGRESS = 5962,
  MD_ERROR_WIN_CLUSTER_DISK_NOT_CONNECTED = 5963,
  MD_ERROR_WIN_DISK_NOT_CSV_CAPABLE = 5964,
  MD_ERROR_WIN_RESOURCE_NOT_IN_AVAILABLE_STORAGE = 5965,
  MD_ERROR_WIN_CLUSTER_SHARED_VOLUME_REDIRECTED = 5966,
  MD_ERROR_WIN_CLUSTER_SHARED_VOLUME_NOT_REDIRECTED = 5967,
  MD_ERROR_WIN_CLUSTER_CANNOT_RETURN_PROPERTIES = 5968,
  MD_ERROR_WIN_CLUSTER_RESOURCE_CONTAINS_UNSUPPORTED_DIFF_AREA_FOR_SHARED_VOLUMES = 5969,
  MD_ERROR_WIN_CLUSTER_RESOURCE_IS_IN_MAINTENANCE_MODE = 5970,
  MD_ERROR_WIN_CLUSTER_AFFINITY_CONFLICT = 5971,
  MD_ERROR_WIN_CLUSTER_RESOURCE_IS_REPLICA_VIRTUAL_MACHINE = 5972,
  MD_ERROR_WIN_CLUSTER_UPGRADE_INCOMPATIBLE_VERSIONS = 5973,
  MD_ERROR_WIN_CLUSTER_UPGRADE_FIX_QUORUM_NOT_SUPPORTED = 5974,
  MD_ERROR_WIN_CLUSTER_UPGRADE_RESTART_REQUIRED = 5975,
  MD_ERROR_WIN_CLUSTER_UPGRADE_IN_PROGRESS = 5976,
  MD_ERROR_WIN_CLUSTER_UPGRADE_INCOMPLETE = 5977,
  MD_ERROR_WIN_CLUSTER_NODE_IN_GRACE_PERIOD = 5978,
  MD_ERROR_WIN_CLUSTER_CSV_IO_PAUSE_TIMEOUT = 5979,
  MD_ERROR_WIN_NODE_NOT_ACTIVE_CLUSTER_MEMBER = 5980,
  MD_ERROR_WIN_CLUSTER_RESOURCE_NOT_MONITORED = 5981,
  MD_ERROR_WIN_CLUSTER_RESOURCE_DOES_NOT_SUPPORT_UNMONITORED = 5982,
  MD_ERROR_WIN_CLUSTER_RESOURCE_IS_REPLICATED = 5983,
  MD_ERROR_WIN_CLUSTER_NODE_ISOLATED = 5984,
  MD_ERROR_WIN_CLUSTER_NODE_QUARANTINED = 5985,
  MD_ERROR_WIN_CLUSTER_DATABASE_UPDATE_CONDITION_FAILED = 5986,
  MD_ERROR_WIN_CLUSTER_SPACE_DEGRADED = 5987,
  MD_ERROR_WIN_CLUSTER_TOKEN_DELEGATION_NOT_SUPPORTED = 5988,
  MD_ERROR_WIN_CLUSTER_CSV_INVALID_HANDLE = 5989,
  MD_ERROR_WIN_CLUSTER_CSV_SUPPORTED_ONLY_ON_COORDINATOR = 5990,
  MD_ERROR_WIN_GROUPSET_NOT_AVAILABLE = 5991,
  MD_ERROR_WIN_GROUPSET_NOT_FOUND = 5992,
  MD_ERROR_WIN_GROUPSET_CANT_PROVIDE = 5993,
  MD_ERROR_WIN_CLUSTER_FAULT_DOMAIN_PARENT_NOT_FOUND = 5994,
  MD_ERROR_WIN_CLUSTER_FAULT_DOMAIN_INVALID_HIERARCHY = 5995,
  MD_ERROR_WIN_CLUSTER_FAULT_DOMAIN_FAILED_S2D_VALIDATION = 5996,
  MD_ERROR_WIN_CLUSTER_FAULT_DOMAIN_S2D_CONNECTIVITY_LOSS = 5997,
  MD_ERROR_WIN_CLUSTER_INVALID_INFRASTRUCTURE_FILESERVER_NAME = 5998,
  MD_ERROR_WIN_CLUSTERSET_MANAGEMENT_CLUSTER_UNREACHABLE = 5999,
  MD_ERROR_WIN_ENCRYPTION_FAILED = 6000,
  MD_ERROR_WIN_DECRYPTION_FAILED = 6001,
  MD_ERROR_WIN_FILE_ENCRYPTED = 6002,
  MD_ERROR_WIN_NO_RECOVERY_POLICY = 6003,
  MD_ERROR_WIN_NO_EFS = 6004,
  MD_ERROR_WIN_WRONG_EFS = 6005,
  MD_ERROR_WIN_NO_USER_KEYS = 6006,
  MD_ERROR_WIN_FILE_NOT_ENCRYPTED = 6007,
  MD_ERROR_WIN_NOT_EXPORT_FORMAT = 6008,
  MD_ERROR_WIN_FILE_READ_ONLY = 6009,
  MD_ERROR_WIN_DIR_EFS_DISALLOWED = 6010,
  MD_ERROR_WIN_EFS_SERVER_NOT_TRUSTED = 6011,
  MD_ERROR_WIN_BAD_RECOVERY_POLICY = 6012,
  MD_ERROR_WIN_EFS_ALG_BLOB_TOO_BIG = 6013,
  MD_ERROR_WIN_VOLUME_NOT_SUPPORT_EFS = 6014,
  MD_ERROR_WIN_EFS_DISABLED = 6015,
  MD_ERROR_WIN_EFS_VERSION_NOT_SUPPORT = 6016,
  MD_ERROR_WIN_CS_ENCRYPTION_INVALID_SERVER_RESPONSE = 6017,
  MD_ERROR_WIN_CS_ENCRYPTION_UNSUPPORTED_SERVER = 6018,
  MD_ERROR_WIN_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE = 6019,
  MD_ERROR_WIN_CS_ENCRYPTION_NEW_ENCRYPTED_FILE = 6020,
  MD_ERROR_WIN_CS_ENCRYPTION_FILE_NOT_CSE = 6021,
  MD_ERROR_WIN_ENCRYPTION_POLICY_DENIES_OPERATION = 6022,
  MD_ERROR_WIN_WIP_ENCRYPTION_FAILED = 6023,
  MD_ERROR_WIN_NO_BROWSER_SERVERS_FOUND = 6118,
  MD_ERROR_WIN_CLUSTER_OBJECT_IS_CLUSTER_SET_VM = 6250,
  MD_ERROR_WIN_LOG_SECTOR_INVALID = 6600,
  MD_ERROR_WIN_LOG_SECTOR_PARITY_INVALID = 6601,
  MD_ERROR_WIN_LOG_SECTOR_REMAPPED = 6602,
  MD_ERROR_WIN_LOG_BLOCK_INCOMPLETE = 6603,
  MD_ERROR_WIN_LOG_INVALID_RANGE = 6604,
  MD_ERROR_WIN_LOG_BLOCKS_EXHAUSTED = 6605,
  MD_ERROR_WIN_LOG_READ_CONTEXT_INVALID = 6606,
  MD_ERROR_WIN_LOG_RESTART_INVALID = 6607,
  MD_ERROR_WIN_LOG_BLOCK_VERSION = 6608,
  MD_ERROR_WIN_LOG_BLOCK_INVALID = 6609,
  MD_ERROR_WIN_LOG_READ_MODE_INVALID = 6610,
  MD_ERROR_WIN_LOG_NO_RESTART = 6611,
  MD_ERROR_WIN_LOG_METADATA_CORRUPT = 6612,
  MD_ERROR_WIN_LOG_METADATA_INVALID = 6613,
  MD_ERROR_WIN_LOG_METADATA_INCONSISTENT = 6614,
  MD_ERROR_WIN_LOG_RESERVATION_INVALID = 6615,
  MD_ERROR_WIN_LOG_CANT_DELETE = 6616,
  MD_ERROR_WIN_LOG_CONTAINER_LIMIT_EXCEEDED = 6617,
  MD_ERROR_WIN_LOG_START_OF_LOG = 6618,
  MD_ERROR_WIN_LOG_POLICY_ALREADY_INSTALLED = 6619,
  MD_ERROR_WIN_LOG_POLICY_NOT_INSTALLED = 6620,
  MD_ERROR_WIN_LOG_POLICY_INVALID = 6621,
  MD_ERROR_WIN_LOG_POLICY_CONFLICT = 6622,
  MD_ERROR_WIN_LOG_PINNED_ARCHIVE_TAIL = 6623,
  MD_ERROR_WIN_LOG_RECORD_NONEXISTENT = 6624,
  MD_ERROR_WIN_LOG_RECORDS_RESERVED_INVALID = 6625,
  MD_ERROR_WIN_LOG_SPACE_RESERVED_INVALID = 6626,
  MD_ERROR_WIN_LOG_TAIL_INVALID = 6627,
  MD_ERROR_WIN_LOG_FULL = 6628,
  MD_ERROR_WIN_COULD_NOT_RESIZE_LOG = 6629,
  MD_ERROR_WIN_LOG_MULTIPLEXED = 6630,
  MD_ERROR_WIN_LOG_DEDICATED = 6631,
  MD_ERROR_WIN_LOG_ARCHIVE_NOT_IN_PROGRESS = 6632,
  MD_ERROR_WIN_LOG_ARCHIVE_IN_PROGRESS = 6633,
  MD_ERROR_WIN_LOG_EPHEMERAL = 6634,
  MD_ERROR_WIN_LOG_NOT_ENOUGH_CONTAINERS = 6635,
  MD_ERROR_WIN_LOG_CLIENT_ALREADY_REGISTERED = 6636,
  MD_ERROR_WIN_LOG_CLIENT_NOT_REGISTERED = 6637,
  MD_ERROR_WIN_LOG_FULL_HANDLER_IN_PROGRESS = 6638,
  MD_ERROR_WIN_LOG_CONTAINER_READ_FAILED = 6639,
  MD_ERROR_WIN_LOG_CONTAINER_WRITE_FAILED = 6640,
  MD_ERROR_WIN_LOG_CONTAINER_OPEN_FAILED = 6641,
  MD_ERROR_WIN_LOG_CONTAINER_STATE_INVALID = 6642,
  MD_ERROR_WIN_LOG_STATE_INVALID = 6643,
  MD_ERROR_WIN_LOG_PINNED = 6644,
  MD_ERROR_WIN_LOG_METADATA_FLUSH_FAILED = 6645,
  MD_ERROR_WIN_LOG_INCONSISTENT_SECURITY = 6646,
  MD_ERROR_WIN_LOG_APPENDED_FLUSH_FAILED = 6647,
  MD_ERROR_WIN_LOG_PINNED_RESERVATION = 6648,
  MD_ERROR_WIN_INVALID_TRANSACTION = 6700,
  MD_ERROR_WIN_TRANSACTION_NOT_ACTIVE = 6701,
  MD_ERROR_WIN_TRANSACTION_REQUEST_NOT_VALID = 6702,
  MD_ERROR_WIN_TRANSACTION_NOT_REQUESTED = 6703,
  MD_ERROR_WIN_TRANSACTION_ALREADY_ABORTED = 6704,
  MD_ERROR_WIN_TRANSACTION_ALREADY_COMMITTED = 6705,
  MD_ERROR_WIN_TM_INITIALIZATION_FAILED = 6706,
  MD_ERROR_WIN_RESOURCEMANAGER_READ_ONLY = 6707,
  MD_ERROR_WIN_TRANSACTION_NOT_JOINED = 6708,
  MD_ERROR_WIN_TRANSACTION_SUPERIOR_EXISTS = 6709,
  MD_ERROR_WIN_CRM_PROTOCOL_ALREADY_EXISTS = 6710,
  MD_ERROR_WIN_TRANSACTION_PROPAGATION_FAILED = 6711,
  MD_ERROR_WIN_CRM_PROTOCOL_NOT_FOUND = 6712,
  MD_ERROR_WIN_TRANSACTION_INVALID_MARSHALL_BUFFER = 6713,
  MD_ERROR_WIN_CURRENT_TRANSACTION_NOT_VALID = 6714,
  MD_ERROR_WIN_TRANSACTION_NOT_FOUND = 6715,
  MD_ERROR_WIN_RESOURCEMANAGER_NOT_FOUND = 6716,
  MD_ERROR_WIN_ENLISTMENT_NOT_FOUND = 6717,
  MD_ERROR_WIN_TRANSACTIONMANAGER_NOT_FOUND = 6718,
  MD_ERROR_WIN_TRANSACTIONMANAGER_NOT_ONLINE = 6719,
  MD_ERROR_WIN_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION = 6720,
  MD_ERROR_WIN_TRANSACTION_NOT_ROOT = 6721,
  MD_ERROR_WIN_TRANSACTION_OBJECT_EXPIRED = 6722,
  MD_ERROR_WIN_TRANSACTION_RESPONSE_NOT_ENLISTED = 6723,
  MD_ERROR_WIN_TRANSACTION_RECORD_TOO_LONG = 6724,
  MD_ERROR_WIN_IMPLICIT_TRANSACTION_NOT_SUPPORTED = 6725,
  MD_ERROR_WIN_TRANSACTION_INTEGRITY_VIOLATED = 6726,
  MD_ERROR_WIN_TRANSACTIONMANAGER_IDENTITY_MISMATCH = 6727,
  MD_ERROR_WIN_RM_CANNOT_BE_FROZEN_FOR_SNAPSHOT = 6728,
  MD_ERROR_WIN_TRANSACTION_MUST_WRITETHROUGH = 6729,
  MD_ERROR_WIN_TRANSACTION_NO_SUPERIOR = 6730,
  MD_ERROR_WIN_HEURISTIC_DAMAGE_POSSIBLE = 6731,
  MD_ERROR_WIN_TRANSACTIONAL_CONFLICT = 6800,
  MD_ERROR_WIN_RM_NOT_ACTIVE = 6801,
  MD_ERROR_WIN_RM_METADATA_CORRUPT = 6802,
  MD_ERROR_WIN_DIRECTORY_NOT_RM = 6803,
  MD_ERROR_WIN_TRANSACTIONS_UNSUPPORTED_REMOTE = 6805,
  MD_ERROR_WIN_LOG_RESIZE_INVALID_SIZE = 6806,
  MD_ERROR_WIN_OBJECT_NO_LONGER_EXISTS = 6807,
  MD_ERROR_WIN_STREAM_MINIVERSION_NOT_FOUND = 6808,
  MD_ERROR_WIN_STREAM_MINIVERSION_NOT_VALID = 6809,
  MD_ERROR_WIN_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION = 6810,
  MD_ERROR_WIN_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT = 6811,
  MD_ERROR_WIN_CANT_CREATE_MORE_STREAM_MINIVERSIONS = 6812,
  MD_ERROR_WIN_REMOTE_FILE_VERSION_MISMATCH = 6814,
  MD_ERROR_WIN_HANDLE_NO_LONGER_VALID = 6815,
  MD_ERROR_WIN_NO_TXF_METADATA = 6816,
  MD_ERROR_WIN_LOG_CORRUPTION_DETECTED = 6817,
  MD_ERROR_WIN_CANT_RECOVER_WITH_HANDLE_OPEN = 6818,
  MD_ERROR_WIN_RM_DISCONNECTED = 6819,
  MD_ERROR_WIN_ENLISTMENT_NOT_SUPERIOR = 6820,
  MD_ERROR_WIN_RECOVERY_NOT_NEEDED = 6821,
  MD_ERROR_WIN_RM_ALREADY_STARTED = 6822,
  MD_ERROR_WIN_FILE_IDENTITY_NOT_PERSISTENT = 6823,
  MD_ERROR_WIN_CANT_BREAK_TRANSACTIONAL_DEPENDENCY = 6824,
  MD_ERROR_WIN_CANT_CROSS_RM_BOUNDARY = 6825,
  MD_ERROR_WIN_TXF_DIR_NOT_EMPTY = 6826,
  MD_ERROR_WIN_INDOUBT_TRANSACTIONS_EXIST = 6827,
  MD_ERROR_WIN_TM_VOLATILE = 6828,
  MD_ERROR_WIN_ROLLBACK_TIMER_EXPIRED = 6829,
  MD_ERROR_WIN_TXF_ATTRIBUTE_CORRUPT = 6830,
  MD_ERROR_WIN_EFS_NOT_ALLOWED_IN_TRANSACTION = 6831,
  MD_ERROR_WIN_TRANSACTIONAL_OPEN_NOT_ALLOWED = 6832,
  MD_ERROR_WIN_LOG_GROWTH_FAILED = 6833,
  MD_ERROR_WIN_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE = 6834,
  MD_ERROR_WIN_TXF_METADATA_ALREADY_PRESENT = 6835,
  MD_ERROR_WIN_TRANSACTION_SCOPE_CALLBACKS_NOT_SET = 6836,
  MD_ERROR_WIN_TRANSACTION_REQUIRED_PROMOTION = 6837,
  MD_ERROR_WIN_CANNOT_EXECUTE_FILE_IN_TRANSACTION = 6838,
  MD_ERROR_WIN_TRANSACTIONS_NOT_FROZEN = 6839,
  MD_ERROR_WIN_TRANSACTION_FREEZE_IN_PROGRESS = 6840,
  MD_ERROR_WIN_NOT_SNAPSHOT_VOLUME = 6841,
  MD_ERROR_WIN_NO_SAVEPOINT_WITH_OPEN_FILES = 6842,
  MD_ERROR_WIN_DATA_LOST_REPAIR = 6843,
  MD_ERROR_WIN_SPARSE_NOT_ALLOWED_IN_TRANSACTION = 6844,
  MD_ERROR_WIN_TM_IDENTITY_MISMATCH = 6845,
  MD_ERROR_WIN_FLOATED_SECTION = 6846,
  MD_ERROR_WIN_CANNOT_ACCEPT_TRANSACTED_WORK = 6847,
  MD_ERROR_WIN_CANNOT_ABORT_TRANSACTIONS = 6848,
  MD_ERROR_WIN_BAD_CLUSTERS = 6849,
  MD_ERROR_WIN_COMPRESSION_NOT_ALLOWED_IN_TRANSACTION = 6850,
  MD_ERROR_WIN_VOLUME_DIRTY = 6851,
  MD_ERROR_WIN_NO_LINK_TRACKING_IN_TRANSACTION = 6852,
  MD_ERROR_WIN_OPERATION_NOT_SUPPORTED_IN_TRANSACTION = 6853,
  MD_ERROR_WIN_EXPIRED_HANDLE = 6854,
  MD_ERROR_WIN_TRANSACTION_NOT_ENLISTED = 6855,
  MD_ERROR_WIN_CTX_WINSTATION_NAME_INVALID = 7001,
  MD_ERROR_WIN_CTX_INVALID_PD = 7002,
  MD_ERROR_WIN_CTX_PD_NOT_FOUND = 7003,
  MD_ERROR_WIN_CTX_WD_NOT_FOUND = 7004,
  MD_ERROR_WIN_CTX_CANNOT_MAKE_EVENTLOG_ENTRY = 7005,
  MD_ERROR_WIN_CTX_SERVICE_NAME_COLLISION = 7006,
  MD_ERROR_WIN_CTX_CLOSE_PENDING = 7007,
  MD_ERROR_WIN_CTX_NO_OUTBUF = 7008,
  MD_ERROR_WIN_CTX_MODEM_INF_NOT_FOUND = 7009,
  MD_ERROR_WIN_CTX_INVALID_MODEMNAME = 7010,
  MD_ERROR_WIN_CTX_MODEM_RESPONSE_ERROR = 7011,
  MD_ERROR_WIN_CTX_MODEM_RESPONSE_TIMEOUT = 7012,
  MD_ERROR_WIN_CTX_MODEM_RESPONSE_NO_CARRIER = 7013,
  MD_ERROR_WIN_CTX_MODEM_RESPONSE_NO_DIALTONE = 7014,
  MD_ERROR_WIN_CTX_MODEM_RESPONSE_BUSY = 7015,
  MD_ERROR_WIN_CTX_MODEM_RESPONSE_VOICE = 7016,
  MD_ERROR_WIN_CTX_TD_ERROR = 7017,
  MD_ERROR_WIN_CTX_WINSTATION_NOT_FOUND = 7022,
  MD_ERROR_WIN_CTX_WINSTATION_ALREADY_EXISTS = 7023,
  MD_ERROR_WIN_CTX_WINSTATION_BUSY = 7024,
  MD_ERROR_WIN_CTX_BAD_VIDEO_MODE = 7025,
  MD_ERROR_WIN_CTX_GRAPHICS_INVALID = 7035,
  MD_ERROR_WIN_CTX_LOGON_DISABLED = 7037,
  MD_ERROR_WIN_CTX_NOT_CONSOLE = 7038,
  MD_ERROR_WIN_CTX_CLIENT_QUERY_TIMEOUT = 7040,
  MD_ERROR_WIN_CTX_CONSOLE_DISCONNECT = 7041,
  MD_ERROR_WIN_CTX_CONSOLE_CONNECT = 7042,
  MD_ERROR_WIN_CTX_SHADOW_DENIED = 7044,
  MD_ERROR_WIN_CTX_WINSTATION_ACCESS_DENIED = 7045,
  MD_ERROR_WIN_CTX_INVALID_WD = 7049,
  MD_ERROR_WIN_CTX_SHADOW_INVALID = 7050,
  MD_ERROR_WIN_CTX_SHADOW_DISABLED = 7051,
  MD_ERROR_WIN_CTX_CLIENT_LICENSE_IN_USE = 7052,
  MD_ERROR_WIN_CTX_CLIENT_LICENSE_NOT_SET = 7053,
  MD_ERROR_WIN_CTX_LICENSE_NOT_AVAILABLE = 7054,
  MD_ERROR_WIN_CTX_LICENSE_CLIENT_INVALID = 7055,
  MD_ERROR_WIN_CTX_LICENSE_EXPIRED = 7056,
  MD_ERROR_WIN_CTX_SHADOW_NOT_RUNNING = 7057,
  MD_ERROR_WIN_CTX_SHADOW_ENDED_BY_MODE_CHANGE = 7058,
  MD_ERROR_WIN_ACTIVATION_COUNT_EXCEEDED = 7059,
  MD_ERROR_WIN_CTX_WINSTATIONS_DISABLED = 7060,
  MD_ERROR_WIN_CTX_ENCRYPTION_LEVEL_REQUIRED = 7061,
  MD_ERROR_WIN_CTX_SESSION_IN_USE = 7062,
  MD_ERROR_WIN_CTX_NO_FORCE_LOGOFF = 7063,
  MD_ERROR_WIN_CTX_ACCOUNT_RESTRICTION = 7064,
  MD_ERROR_WIN_RDP_PROTOCOL_ERROR = 7065,
  MD_ERROR_WIN_CTX_CDM_CONNECT = 7066,
  MD_ERROR_WIN_CTX_CDM_DISCONNECT = 7067,
  MD_ERROR_WIN_CTX_SECURITY_LAYER_ERROR = 7068,
  MD_ERROR_WIN_TS_INCOMPATIBLE_SESSIONS = 7069,
  MD_ERROR_WIN_TS_VIDEO_SUBSYSTEM_ERROR = 7070,
  MD_ERROR_WIN_DS_NOT_INSTALLED = 8200,
  MD_ERROR_WIN_DS_MEMBERSHIP_EVALUATED_LOCALLY = 8201,
  MD_ERROR_WIN_DS_NO_ATTRIBUTE_OR_VALUE = 8202,
  MD_ERROR_WIN_DS_INVALID_ATTRIBUTE_SYNTAX = 8203,
  MD_ERROR_WIN_DS_ATTRIBUTE_TYPE_UNDEFINED = 8204,
  MD_ERROR_WIN_DS_ATTRIBUTE_OR_VALUE_EXISTS = 8205,
  MD_ERROR_WIN_DS_BUSY = 8206,
  MD_ERROR_WIN_DS_UNAVAILABLE = 8207,
  MD_ERROR_WIN_DS_NO_RIDS_ALLOCATED = 8208,
  MD_ERROR_WIN_DS_NO_MORE_RIDS = 8209,
  MD_ERROR_WIN_DS_INCORRECT_ROLE_OWNER = 8210,
  MD_ERROR_WIN_DS_RIDMGR_INIT_ERROR = 8211,
  MD_ERROR_WIN_DS_OBJ_CLASS_VIOLATION = 8212,
  MD_ERROR_WIN_DS_CANT_ON_NON_LEAF = 8213,
  MD_ERROR_WIN_DS_CANT_ON_RDN = 8214,
  MD_ERROR_WIN_DS_CANT_MOD_OBJ_CLASS = 8215,
  MD_ERROR_WIN_DS_CROSS_DOM_MOVE_ERROR = 8216,
  MD_ERROR_WIN_DS_GC_NOT_AVAILABLE = 8217,
  MD_ERROR_WIN_SHARED_POLICY = 8218,
  MD_ERROR_WIN_POLICY_OBJECT_NOT_FOUND = 8219,
  MD_ERROR_WIN_POLICY_ONLY_IN_DS = 8220,
  MD_ERROR_WIN_PROMOTION_ACTIVE = 8221,
  MD_ERROR_WIN_NO_PROMOTION_ACTIVE = 8222,
  MD_ERROR_WIN_DS_OPERATIONS_ERROR = 8224,
  MD_ERROR_WIN_DS_PROTOCOL_ERROR = 8225,
  MD_ERROR_WIN_DS_TIMELIMIT_EXCEEDED = 8226,
  MD_ERROR_WIN_DS_SIZELIMIT_EXCEEDED = 8227,
  MD_ERROR_WIN_DS_ADMIN_LIMIT_EXCEEDED = 8228,
  MD_ERROR_WIN_DS_COMPARE_FALSE = 8229,
  MD_ERROR_WIN_DS_COMPARE_TRUE = 8230,
  MD_ERROR_WIN_DS_AUTH_METHOD_NOT_SUPPORTED = 8231,
  MD_ERROR_WIN_DS_STRONG_AUTH_REQUIRED = 8232,
  MD_ERROR_WIN_DS_INAPPROPRIATE_AUTH = 8233,
  MD_ERROR_WIN_DS_AUTH_UNKNOWN = 8234,
  MD_ERROR_WIN_DS_REFERRAL = 8235,
  MD_ERROR_WIN_DS_UNAVAILABLE_CRIT_EXTENSION = 8236,
  MD_ERROR_WIN_DS_CONFIDENTIALITY_REQUIRED = 8237,
  MD_ERROR_WIN_DS_INAPPROPRIATE_MATCHING = 8238,
  MD_ERROR_WIN_DS_CONSTRAINT_VIOLATION = 8239,
  MD_ERROR_WIN_DS_NO_SUCH_OBJECT = 8240,
  MD_ERROR_WIN_DS_ALIAS_PROBLEM = 8241,
  MD_ERROR_WIN_DS_INVALID_DN_SYNTAX = 8242,
  MD_ERROR_WIN_DS_IS_LEAF = 8243,
  MD_ERROR_WIN_DS_ALIAS_DEREF_PROBLEM = 8244,
  MD_ERROR_WIN_DS_UNWILLING_TO_PERFORM = 8245,
  MD_ERROR_WIN_DS_LOOP_DETECT = 8246,
  MD_ERROR_WIN_DS_NAMING_VIOLATION = 8247,
  MD_ERROR_WIN_DS_OBJECT_RESULTS_TOO_LARGE = 8248,
  MD_ERROR_WIN_DS_AFFECTS_MULTIPLE_DSAS = 8249,
  MD_ERROR_WIN_DS_SERVER_DOWN = 8250,
  MD_ERROR_WIN_DS_LOCAL_ERROR = 8251,
  MD_ERROR_WIN_DS_ENCODING_ERROR = 8252,
  MD_ERROR_WIN_DS_DECODING_ERROR = 8253,
  MD_ERROR_WIN_DS_FILTER_UNKNOWN = 8254,
  MD_ERROR_WIN_DS_PARAM_ERROR = 8255,
  MD_ERROR_WIN_DS_NOT_SUPPORTED = 8256,
  MD_ERROR_WIN_DS_NO_RESULTS_RETURNED = 8257,
  MD_ERROR_WIN_DS_CONTROL_NOT_FOUND = 8258,
  MD_ERROR_WIN_DS_CLIENT_LOOP = 8259,
  MD_ERROR_WIN_DS_REFERRAL_LIMIT_EXCEEDED = 8260,
  MD_ERROR_WIN_DS_SORT_CONTROL_MISSING = 8261,
  MD_ERROR_WIN_DS_OFFSET_RANGE_ERROR = 8262,
  MD_ERROR_WIN_DS_RIDMGR_DISABLED = 8263,
  MD_ERROR_WIN_DS_ROOT_MUST_BE_NC = 8301,
  MD_ERROR_WIN_DS_ADD_REPLICA_INHIBITED = 8302,
  MD_ERROR_WIN_DS_ATT_NOT_DEF_IN_SCHEMA = 8303,
  MD_ERROR_WIN_DS_MAX_OBJ_SIZE_EXCEEDED = 8304,
  MD_ERROR_WIN_DS_OBJ_STRING_NAME_EXISTS = 8305,
  MD_ERROR_WIN_DS_NO_RDN_DEFINED_IN_SCHEMA = 8306,
  MD_ERROR_WIN_DS_RDN_DOESNT_MATCH_SCHEMA = 8307,
  MD_ERROR_WIN_DS_NO_REQUESTED_ATTS_FOUND = 8308,
  MD_ERROR_WIN_DS_USER_BUFFER_TO_SMALL = 8309,
  MD_ERROR_WIN_DS_ATT_IS_NOT_ON_OBJ = 8310,
  MD_ERROR_WIN_DS_ILLEGAL_MOD_OPERATION = 8311,
  MD_ERROR_WIN_DS_OBJ_TOO_LARGE = 8312,
  MD_ERROR_WIN_DS_BAD_INSTANCE_TYPE = 8313,
  MD_ERROR_WIN_DS_MASTERDSA_REQUIRED = 8314,
  MD_ERROR_WIN_DS_OBJECT_CLASS_REQUIRED = 8315,
  MD_ERROR_WIN_DS_MISSING_REQUIRED_ATT = 8316,
  MD_ERROR_WIN_DS_ATT_NOT_DEF_FOR_CLASS = 8317,
  MD_ERROR_WIN_DS_ATT_ALREADY_EXISTS = 8318,
  MD_ERROR_WIN_DS_CANT_ADD_ATT_VALUES = 8320,
  MD_ERROR_WIN_DS_SINGLE_VALUE_CONSTRAINT = 8321,
  MD_ERROR_WIN_DS_RANGE_CONSTRAINT = 8322,
  MD_ERROR_WIN_DS_ATT_VAL_ALREADY_EXISTS = 8323,
  MD_ERROR_WIN_DS_CANT_REM_MISSING_ATT = 8324,
  MD_ERROR_WIN_DS_CANT_REM_MISSING_ATT_VAL = 8325,
  MD_ERROR_WIN_DS_ROOT_CANT_BE_SUBREF = 8326,
  MD_ERROR_WIN_DS_NO_CHAINING = 8327,
  MD_ERROR_WIN_DS_NO_CHAINED_EVAL = 8328,
  MD_ERROR_WIN_DS_NO_PARENT_OBJECT = 8329,
  MD_ERROR_WIN_DS_PARENT_IS_AN_ALIAS = 8330,
  MD_ERROR_WIN_DS_CANT_MIX_MASTER_AND_REPS = 8331,
  MD_ERROR_WIN_DS_CHILDREN_EXIST = 8332,
  MD_ERROR_WIN_DS_OBJ_NOT_FOUND = 8333,
  MD_ERROR_WIN_DS_ALIASED_OBJ_MISSING = 8334,
  MD_ERROR_WIN_DS_BAD_NAME_SYNTAX = 8335,
  MD_ERROR_WIN_DS_ALIAS_POINTS_TO_ALIAS = 8336,
  MD_ERROR_WIN_DS_CANT_DEREF_ALIAS = 8337,
  MD_ERROR_WIN_DS_OUT_OF_SCOPE = 8338,
  MD_ERROR_WIN_DS_OBJECT_BEING_REMOVED = 8339,
  MD_ERROR_WIN_DS_CANT_DELETE_DSA_OBJ = 8340,
  MD_ERROR_WIN_DS_GENERIC_ERROR = 8341,
  MD_ERROR_WIN_DS_DSA_MUST_BE_INT_MASTER = 8342,
  MD_ERROR_WIN_DS_CLASS_NOT_DSA = 8343,
  MD_ERROR_WIN_DS_INSUFF_ACCESS_RIGHTS = 8344,
  MD_ERROR_WIN_DS_ILLEGAL_SUPERIOR = 8345,
  MD_ERROR_WIN_DS_ATTRIBUTE_OWNED_BY_SAM = 8346,
  MD_ERROR_WIN_DS_NAME_TOO_MANY_PARTS = 8347,
  MD_ERROR_WIN_DS_NAME_TOO_LONG = 8348,
  MD_ERROR_WIN_DS_NAME_VALUE_TOO_LONG = 8349,
  MD_ERROR_WIN_DS_NAME_UNPARSEABLE = 8350,
  MD_ERROR_WIN_DS_NAME_TYPE_UNKNOWN = 8351,
  MD_ERROR_WIN_DS_NOT_AN_OBJECT = 8352,
  MD_ERROR_WIN_DS_SEC_DESC_TOO_SHORT = 8353,
  MD_ERROR_WIN_DS_SEC_DESC_INVALID = 8354,
  MD_ERROR_WIN_DS_NO_DELETED_NAME = 8355,
  MD_ERROR_WIN_DS_SUBREF_MUST_HAVE_PARENT = 8356,
  MD_ERROR_WIN_DS_NCNAME_MUST_BE_NC = 8357,
  MD_ERROR_WIN_DS_CANT_ADD_SYSTEM_ONLY = 8358,
  MD_ERROR_WIN_DS_CLASS_MUST_BE_CONCRETE = 8359,
  MD_ERROR_WIN_DS_INVALID_DMD = 8360,
  MD_ERROR_WIN_DS_OBJ_GUID_EXISTS = 8361,
  MD_ERROR_WIN_DS_NOT_ON_BACKLINK = 8362,
  MD_ERROR_WIN_DS_NO_CROSSREF_FOR_NC = 8363,
  MD_ERROR_WIN_DS_SHUTTING_DOWN = 8364,
  MD_ERROR_WIN_DS_UNKNOWN_OPERATION = 8365,
  MD_ERROR_WIN_DS_INVALID_ROLE_OWNER = 8366,
  MD_ERROR_WIN_DS_COULDNT_CONTACT_FSMO = 8367,
  MD_ERROR_WIN_DS_CROSS_NC_DN_RENAME = 8368,
  MD_ERROR_WIN_DS_CANT_MOD_SYSTEM_ONLY = 8369,
  MD_ERROR_WIN_DS_REPLICATOR_ONLY = 8370,
  MD_ERROR_WIN_DS_OBJ_CLASS_NOT_DEFINED = 8371,
  MD_ERROR_WIN_DS_OBJ_CLASS_NOT_SUBCLASS = 8372,
  MD_ERROR_WIN_DS_NAME_REFERENCE_INVALID = 8373,
  MD_ERROR_WIN_DS_CROSS_REF_EXISTS = 8374,
  MD_ERROR_WIN_DS_CANT_DEL_MASTER_CROSSREF = 8375,
  MD_ERROR_WIN_DS_SUBTREE_NOTIFY_NOT_NC_HEAD = 8376,
  MD_ERROR_WIN_DS_NOTIFY_FILTER_TOO_COMPLEX = 8377,
  MD_ERROR_WIN_DS_DUP_RDN = 8378,
  MD_ERROR_WIN_DS_DUP_OID = 8379,
  MD_ERROR_WIN_DS_DUP_MAPI_ID = 8380,
  MD_ERROR_WIN_DS_DUP_SCHEMA_ID_GUID = 8381,
  MD_ERROR_WIN_DS_DUP_LDAP_DISPLAY_NAME = 8382,
  MD_ERROR_WIN_DS_SEMANTIC_ATT_TEST = 8383,
  MD_ERROR_WIN_DS_SYNTAX_MISMATCH = 8384,
  MD_ERROR_WIN_DS_EXISTS_IN_MUST_HAVE = 8385,
  MD_ERROR_WIN_DS_EXISTS_IN_MAY_HAVE = 8386,
  MD_ERROR_WIN_DS_NONEXISTENT_MAY_HAVE = 8387,
  MD_ERROR_WIN_DS_NONEXISTENT_MUST_HAVE = 8388,
  MD_ERROR_WIN_DS_AUX_CLS_TEST_FAIL = 8389,
  MD_ERROR_WIN_DS_NONEXISTENT_POSS_SUP = 8390,
  MD_ERROR_WIN_DS_SUB_CLS_TEST_FAIL = 8391,
  MD_ERROR_WIN_DS_BAD_RDN_ATT_ID_SYNTAX = 8392,
  MD_ERROR_WIN_DS_EXISTS_IN_AUX_CLS = 8393,
  MD_ERROR_WIN_DS_EXISTS_IN_SUB_CLS = 8394,
  MD_ERROR_WIN_DS_EXISTS_IN_POSS_SUP = 8395,
  MD_ERROR_WIN_DS_RECALCSCHEMA_FAILED = 8396,
  MD_ERROR_WIN_DS_TREE_DELETE_NOT_FINISHED = 8397,
  MD_ERROR_WIN_DS_CANT_DELETE = 8398,
  MD_ERROR_WIN_DS_ATT_SCHEMA_REQ_ID = 8399,
  MD_ERROR_WIN_DS_BAD_ATT_SCHEMA_SYNTAX = 8400,
  MD_ERROR_WIN_DS_CANT_CACHE_ATT = 8401,
  MD_ERROR_WIN_DS_CANT_CACHE_CLASS = 8402,
  MD_ERROR_WIN_DS_CANT_REMOVE_ATT_CACHE = 8403,
  MD_ERROR_WIN_DS_CANT_REMOVE_CLASS_CACHE = 8404,
  MD_ERROR_WIN_DS_CANT_RETRIEVE_DN = 8405,
  MD_ERROR_WIN_DS_MISSING_SUPREF = 8406,
  MD_ERROR_WIN_DS_CANT_RETRIEVE_INSTANCE = 8407,
  MD_ERROR_WIN_DS_CODE_INCONSISTENCY = 8408,
  MD_ERROR_WIN_DS_DATABASE_ERROR = 8409,
  MD_ERROR_WIN_DS_GOVERNSID_MISSING = 8410,
  MD_ERROR_WIN_DS_MISSING_EXPECTED_ATT = 8411,
  MD_ERROR_WIN_DS_NCNAME_MISSING_CR_REF = 8412,
  MD_ERROR_WIN_DS_SECURITY_CHECKING_ERROR = 8413,
  MD_ERROR_WIN_DS_SCHEMA_NOT_LOADED = 8414,
  MD_ERROR_WIN_DS_SCHEMA_ALLOC_FAILED = 8415,
  MD_ERROR_WIN_DS_ATT_SCHEMA_REQ_SYNTAX = 8416,
  MD_ERROR_WIN_DS_GCVERIFY_ERROR = 8417,
  MD_ERROR_WIN_DS_DRA_SCHEMA_MISMATCH = 8418,
  MD_ERROR_WIN_DS_CANT_FIND_DSA_OBJ = 8419,
  MD_ERROR_WIN_DS_CANT_FIND_EXPECTED_NC = 8420,
  MD_ERROR_WIN_DS_CANT_FIND_NC_IN_CACHE = 8421,
  MD_ERROR_WIN_DS_CANT_RETRIEVE_CHILD = 8422,
  MD_ERROR_WIN_DS_SECURITY_ILLEGAL_MODIFY = 8423,
  MD_ERROR_WIN_DS_CANT_REPLACE_HIDDEN_REC = 8424,
  MD_ERROR_WIN_DS_BAD_HIERARCHY_FILE = 8425,
  MD_ERROR_WIN_DS_BUILD_HIERARCHY_TABLE_FAILED = 8426,
  MD_ERROR_WIN_DS_CONFIG_PARAM_MISSING = 8427,
  MD_ERROR_WIN_DS_COUNTING_AB_INDICES_FAILED = 8428,
  MD_ERROR_WIN_DS_HIERARCHY_TABLE_MALLOC_FAILED = 8429,
  MD_ERROR_WIN_DS_INTERNAL_FAILURE = 8430,
  MD_ERROR_WIN_DS_UNKNOWN_ERROR = 8431,
  MD_ERROR_WIN_DS_ROOT_REQUIRES_CLASS_TOP = 8432,
  MD_ERROR_WIN_DS_REFUSING_FSMO_ROLES = 8433,
  MD_ERROR_WIN_DS_MISSING_FSMO_SETTINGS = 8434,
  MD_ERROR_WIN_DS_UNABLE_TO_SURRENDER_ROLES = 8435,
  MD_ERROR_WIN_DS_DRA_GENERIC = 8436,
  MD_ERROR_WIN_DS_DRA_INVALID_PARAMETER = 8437,
  MD_ERROR_WIN_DS_DRA_BUSY = 8438,
  MD_ERROR_WIN_DS_DRA_BAD_DN = 8439,
  MD_ERROR_WIN_DS_DRA_BAD_NC = 8440,
  MD_ERROR_WIN_DS_DRA_DN_EXISTS = 8441,
  MD_ERROR_WIN_DS_DRA_INTERNAL_ERROR = 8442,
  MD_ERROR_WIN_DS_DRA_INCONSISTENT_DIT = 8443,
  MD_ERROR_WIN_DS_DRA_CONNECTION_FAILED = 8444,
  MD_ERROR_WIN_DS_DRA_BAD_INSTANCE_TYPE = 8445,
  MD_ERROR_WIN_DS_DRA_OUT_OF_MEM = 8446,
  MD_ERROR_WIN_DS_DRA_MAIL_PROBLEM = 8447,
  MD_ERROR_WIN_DS_DRA_REF_ALREADY_EXISTS = 8448,
  MD_ERROR_WIN_DS_DRA_REF_NOT_FOUND = 8449,
  MD_ERROR_WIN_DS_DRA_OBJ_IS_REP_SOURCE = 8450,
  MD_ERROR_WIN_DS_DRA_DB_ERROR = 8451,
  MD_ERROR_WIN_DS_DRA_NO_REPLICA = 8452,
  MD_ERROR_WIN_DS_DRA_ACCESS_DENIED = 8453,
  MD_ERROR_WIN_DS_DRA_NOT_SUPPORTED = 8454,
  MD_ERROR_WIN_DS_DRA_RPC_CANCELLED = 8455,
  MD_ERROR_WIN_DS_DRA_SOURCE_DISABLED = 8456,
  MD_ERROR_WIN_DS_DRA_SINK_DISABLED = 8457,
  MD_ERROR_WIN_DS_DRA_NAME_COLLISION = 8458,
  MD_ERROR_WIN_DS_DRA_SOURCE_REINSTALLED = 8459,
  MD_ERROR_WIN_DS_DRA_MISSING_PARENT = 8460,
  MD_ERROR_WIN_DS_DRA_PREEMPTED = 8461,
  MD_ERROR_WIN_DS_DRA_ABANDON_SYNC = 8462,
  MD_ERROR_WIN_DS_DRA_SHUTDOWN = 8463,
  MD_ERROR_WIN_DS_DRA_INCOMPATIBLE_PARTIAL_SET = 8464,
  MD_ERROR_WIN_DS_DRA_SOURCE_IS_PARTIAL_REPLICA = 8465,
  MD_ERROR_WIN_DS_DRA_EXTN_CONNECTION_FAILED = 8466,
  MD_ERROR_WIN_DS_INSTALL_SCHEMA_MISMATCH = 8467,
  MD_ERROR_WIN_DS_DUP_LINK_ID = 8468,
  MD_ERROR_WIN_DS_NAME_ERROR_RESOLVING = 8469,
  MD_ERROR_WIN_DS_NAME_ERROR_NOT_FOUND = 8470,
  MD_ERROR_WIN_DS_NAME_ERROR_NOT_UNIQUE = 8471,
  MD_ERROR_WIN_DS_NAME_ERROR_NO_MAPPING = 8472,
  MD_ERROR_WIN_DS_NAME_ERROR_DOMAIN_ONLY = 8473,
  MD_ERROR_WIN_DS_NAME_ERROR_NO_SYNTACTICAL_MAPPING = 8474,
  MD_ERROR_WIN_DS_CONSTRUCTED_ATT_MOD = 8475,
  MD_ERROR_WIN_DS_WRONG_OM_OBJ_CLASS = 8476,
  MD_ERROR_WIN_DS_DRA_REPL_PENDING = 8477,
  MD_ERROR_WIN_DS_DS_REQUIRED = 8478,
  MD_ERROR_WIN_DS_INVALID_LDAP_DISPLAY_NAME = 8479,
  MD_ERROR_WIN_DS_NON_BASE_SEARCH = 8480,
  MD_ERROR_WIN_DS_CANT_RETRIEVE_ATTS = 8481,
  MD_ERROR_WIN_DS_BACKLINK_WITHOUT_LINK = 8482,
  MD_ERROR_WIN_DS_EPOCH_MISMATCH = 8483,
  MD_ERROR_WIN_DS_SRC_NAME_MISMATCH = 8484,
  MD_ERROR_WIN_DS_SRC_AND_DST_NC_IDENTICAL = 8485,
  MD_ERROR_WIN_DS_DST_NC_MISMATCH = 8486,
  MD_ERROR_WIN_DS_NOT_AUTHORITIVE_FOR_DST_NC = 8487,
  MD_ERROR_WIN_DS_SRC_GUID_MISMATCH = 8488,
  MD_ERROR_WIN_DS_CANT_MOVE_DELETED_OBJECT = 8489,
  MD_ERROR_WIN_DS_PDC_OPERATION_IN_PROGRESS = 8490,
  MD_ERROR_WIN_DS_CROSS_DOMAIN_CLEANUP_REQD = 8491,
  MD_ERROR_WIN_DS_ILLEGAL_XDOM_MOVE_OPERATION = 8492,
  MD_ERROR_WIN_DS_CANT_WITH_ACCT_GROUP_MEMBERSHPS = 8493,
  MD_ERROR_WIN_DS_NC_MUST_HAVE_NC_PARENT = 8494,
  MD_ERROR_WIN_DS_CR_IMPOSSIBLE_TO_VALIDATE = 8495,
  MD_ERROR_WIN_DS_DST_DOMAIN_NOT_NATIVE = 8496,
  MD_ERROR_WIN_DS_MISSING_INFRASTRUCTURE_CONTAINER = 8497,
  MD_ERROR_WIN_DS_CANT_MOVE_ACCOUNT_GROUP = 8498,
  MD_ERROR_WIN_DS_CANT_MOVE_RESOURCE_GROUP = 8499,
  MD_ERROR_WIN_DS_INVALID_SEARCH_FLAG = 8500,
  MD_ERROR_WIN_DS_NO_TREE_DELETE_ABOVE_NC = 8501,
  MD_ERROR_WIN_DS_COULDNT_LOCK_TREE_FOR_DELETE = 8502,
  MD_ERROR_WIN_DS_COULDNT_IDENTIFY_OBJECTS_FOR_TREE_DELETE = 8503,
  MD_ERROR_WIN_DS_SAM_INIT_FAILURE = 8504,
  MD_ERROR_WIN_DS_SENSITIVE_GROUP_VIOLATION = 8505,
  MD_ERROR_WIN_DS_CANT_MOD_PRIMARYGROUPID = 8506,
  MD_ERROR_WIN_DS_ILLEGAL_BASE_SCHEMA_MOD = 8507,
  MD_ERROR_WIN_DS_NONSAFE_SCHEMA_CHANGE = 8508,
  MD_ERROR_WIN_DS_SCHEMA_UPDATE_DISALLOWED = 8509,
  MD_ERROR_WIN_DS_CANT_CREATE_UNDER_SCHEMA = 8510,
  MD_ERROR_WIN_DS_INSTALL_NO_SRC_SCH_VERSION = 8511,
  MD_ERROR_WIN_DS_INSTALL_NO_SCH_VERSION_IN_INIFILE = 8512,
  MD_ERROR_WIN_DS_INVALID_GROUP_TYPE = 8513,
  MD_ERROR_WIN_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN = 8514,
  MD_ERROR_WIN_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN = 8515,
  MD_ERROR_WIN_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER = 8516,
  MD_ERROR_WIN_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER = 8517,
  MD_ERROR_WIN_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER = 8518,
  MD_ERROR_WIN_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER = 8519,
  MD_ERROR_WIN_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER = 8520,
  MD_ERROR_WIN_DS_HAVE_PRIMARY_MEMBERS = 8521,
  MD_ERROR_WIN_DS_STRING_SD_CONVERSION_FAILED = 8522,
  MD_ERROR_WIN_DS_NAMING_MASTER_GC = 8523,
  MD_ERROR_WIN_DS_DNS_LOOKUP_FAILURE = 8524,
  MD_ERROR_WIN_DS_COULDNT_UPDATE_SPNS = 8525,
  MD_ERROR_WIN_DS_CANT_RETRIEVE_SD = 8526,
  MD_ERROR_WIN_DS_KEY_NOT_UNIQUE = 8527,
  MD_ERROR_WIN_DS_WRONG_LINKED_ATT_SYNTAX = 8528,
  MD_ERROR_WIN_DS_SAM_NEED_BOOTKEY_PASSWORD = 8529,
  MD_ERROR_WIN_DS_SAM_NEED_BOOTKEY_FLOPPY = 8530,
  MD_ERROR_WIN_DS_CANT_START = 8531,
  MD_ERROR_WIN_DS_INIT_FAILURE = 8532,
  MD_ERROR_WIN_DS_NO_PKT_PRIVACY_ON_CONNECTION = 8533,
  MD_ERROR_WIN_DS_SOURCE_DOMAIN_IN_FOREST = 8534,
  MD_ERROR_WIN_DS_DESTINATION_DOMAIN_NOT_IN_FOREST = 8535,
  MD_ERROR_WIN_DS_DESTINATION_AUDITING_NOT_ENABLED = 8536,
  MD_ERROR_WIN_DS_CANT_FIND_DC_FOR_SRC_DOMAIN = 8537,
  MD_ERROR_WIN_DS_SRC_OBJ_NOT_GROUP_OR_USER = 8538,
  MD_ERROR_WIN_DS_SRC_SID_EXISTS_IN_FOREST = 8539,
  MD_ERROR_WIN_DS_SRC_AND_DST_OBJECT_CLASS_MISMATCH = 8540,
  MD_ERROR_WIN_SAM_INIT_FAILURE = 8541,
  MD_ERROR_WIN_DS_DRA_SCHEMA_INFO_SHIP = 8542,
  MD_ERROR_WIN_DS_DRA_SCHEMA_CONFLICT = 8543,
  MD_ERROR_WIN_DS_DRA_EARLIER_SCHEMA_CONFLICT = 8544,
  MD_ERROR_WIN_DS_DRA_OBJ_NC_MISMATCH = 8545,
  MD_ERROR_WIN_DS_NC_STILL_HAS_DSAS = 8546,
  MD_ERROR_WIN_DS_GC_REQUIRED = 8547,
  MD_ERROR_WIN_DS_LOCAL_MEMBER_OF_LOCAL_ONLY = 8548,
  MD_ERROR_WIN_DS_NO_FPO_IN_UNIVERSAL_GROUPS = 8549,
  MD_ERROR_WIN_DS_CANT_ADD_TO_GC = 8550,
  MD_ERROR_WIN_DS_NO_CHECKPOINT_WITH_PDC = 8551,
  MD_ERROR_WIN_DS_SOURCE_AUDITING_NOT_ENABLED = 8552,
  MD_ERROR_WIN_DS_CANT_CREATE_IN_NONDOMAIN_NC = 8553,
  MD_ERROR_WIN_DS_INVALID_NAME_FOR_SPN = 8554,
  MD_ERROR_WIN_DS_FILTER_USES_CONTRUCTED_ATTRS = 8555,
  MD_ERROR_WIN_DS_UNICODEPWD_NOT_IN_QUOTES = 8556,
  MD_ERROR_WIN_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED = 8557,
  MD_ERROR_WIN_DS_MUST_BE_RUN_ON_DST_DC = 8558,
  MD_ERROR_WIN_DS_SRC_DC_MUST_BE_SP4_OR_GREATER = 8559,
  MD_ERROR_WIN_DS_CANT_TREE_DELETE_CRITICAL_OBJ = 8560,
  MD_ERROR_WIN_DS_INIT_FAILURE_CONSOLE = 8561,
  MD_ERROR_WIN_DS_SAM_INIT_FAILURE_CONSOLE = 8562,
  MD_ERROR_WIN_DS_FOREST_VERSION_TOO_HIGH = 8563,
  MD_ERROR_WIN_DS_DOMAIN_VERSION_TOO_HIGH = 8564,
  MD_ERROR_WIN_DS_FOREST_VERSION_TOO_LOW = 8565,
  MD_ERROR_WIN_DS_DOMAIN_VERSION_TOO_LOW = 8566,
  MD_ERROR_WIN_DS_INCOMPATIBLE_VERSION = 8567,
  MD_ERROR_WIN_DS_LOW_DSA_VERSION = 8568,
  MD_ERROR_WIN_DS_NO_BEHAVIOR_VERSION_IN_MIXEDDOMAIN = 8569,
  MD_ERROR_WIN_DS_NOT_SUPPORTED_SORT_ORDER = 8570,
  MD_ERROR_WIN_DS_NAME_NOT_UNIQUE = 8571,
  MD_ERROR_WIN_DS_MACHINE_ACCOUNT_CREATED_PRENT4 = 8572,
  MD_ERROR_WIN_DS_OUT_OF_VERSION_STORE = 8573,
  MD_ERROR_WIN_DS_INCOMPATIBLE_CONTROLS_USED = 8574,
  MD_ERROR_WIN_DS_NO_REF_DOMAIN = 8575,
  MD_ERROR_WIN_DS_RESERVED_LINK_ID = 8576,
  MD_ERROR_WIN_DS_LINK_ID_NOT_AVAILABLE = 8577,
  MD_ERROR_WIN_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER = 8578,
  MD_ERROR_WIN_DS_MODIFYDN_DISALLOWED_BY_INSTANCE_TYPE = 8579,
  MD_ERROR_WIN_DS_NO_OBJECT_MOVE_IN_SCHEMA_NC = 8580,
  MD_ERROR_WIN_DS_MODIFYDN_DISALLOWED_BY_FLAG = 8581,
  MD_ERROR_WIN_DS_MODIFYDN_WRONG_GRANDPARENT = 8582,
  MD_ERROR_WIN_DS_NAME_ERROR_TRUST_REFERRAL = 8583,
  MD_ERROR_WIN_NOT_SUPPORTED_ON_STANDARD_SERVER = 8584,
  MD_ERROR_WIN_DS_CANT_ACCESS_REMOTE_PART_OF_AD = 8585,
  MD_ERROR_WIN_DS_CR_IMPOSSIBLE_TO_VALIDATE_V2 = 8586,
  MD_ERROR_WIN_DS_THREAD_LIMIT_EXCEEDED = 8587,
  MD_ERROR_WIN_DS_NOT_CLOSEST = 8588,
  MD_ERROR_WIN_DS_CANT_DERIVE_SPN_WITHOUT_SERVER_REF = 8589,
  MD_ERROR_WIN_DS_SINGLE_USER_MODE_FAILED = 8590,
  MD_ERROR_WIN_DS_NTDSCRIPT_SYNTAX_ERROR = 8591,
  MD_ERROR_WIN_DS_NTDSCRIPT_PROCESS_ERROR = 8592,
  MD_ERROR_WIN_DS_DIFFERENT_REPL_EPOCHS = 8593,
  MD_ERROR_WIN_DS_DRS_EXTENSIONS_CHANGED = 8594,
  MD_ERROR_WIN_DS_REPLICA_SET_CHANGE_NOT_ALLOWED_ON_DISABLED_CR = 8595,
  MD_ERROR_WIN_DS_NO_MSDS_INTID = 8596,
  MD_ERROR_WIN_DS_DUP_MSDS_INTID = 8597,
  MD_ERROR_WIN_DS_EXISTS_IN_RDNATTID = 8598,
  MD_ERROR_WIN_DS_AUTHORIZATION_FAILED = 8599,
  MD_ERROR_WIN_DS_INVALID_SCRIPT = 8600,
  MD_ERROR_WIN_DS_REMOTE_CROSSREF_OP_FAILED = 8601,
  MD_ERROR_WIN_DS_CROSS_REF_BUSY = 8602,
  MD_ERROR_WIN_DS_CANT_DERIVE_SPN_FOR_DELETED_DOMAIN = 8603,
  MD_ERROR_WIN_DS_CANT_DEMOTE_WITH_WRITEABLE_NC = 8604,
  MD_ERROR_WIN_DS_DUPLICATE_ID_FOUND = 8605,
  MD_ERROR_WIN_DS_INSUFFICIENT_ATTR_TO_CREATE_OBJECT = 8606,
  MD_ERROR_WIN_DS_GROUP_CONVERSION_ERROR = 8607,
  MD_ERROR_WIN_DS_CANT_MOVE_APP_BASIC_GROUP = 8608,
  MD_ERROR_WIN_DS_CANT_MOVE_APP_QUERY_GROUP = 8609,
  MD_ERROR_WIN_DS_ROLE_NOT_VERIFIED = 8610,
  MD_ERROR_WIN_DS_WKO_CONTAINER_CANNOT_BE_SPECIAL = 8611,
  MD_ERROR_WIN_DS_DOMAIN_RENAME_IN_PROGRESS = 8612,
  MD_ERROR_WIN_DS_EXISTING_AD_CHILD_NC = 8613,
  MD_ERROR_WIN_DS_REPL_LIFETIME_EXCEEDED = 8614,
  MD_ERROR_WIN_DS_DISALLOWED_IN_SYSTEM_CONTAINER = 8615,
  MD_ERROR_WIN_DS_LDAP_SEND_QUEUE_FULL = 8616,
  MD_ERROR_WIN_DS_DRA_OUT_SCHEDULE_WINDOW = 8617,
  MD_ERROR_WIN_DS_POLICY_NOT_KNOWN = 8618,
  MD_ERROR_WIN_NO_SITE_SETTINGS_OBJECT = 8619,
  MD_ERROR_WIN_NO_SECRETS = 8620,
  MD_ERROR_WIN_NO_WRITABLE_DC_FOUND = 8621,
  MD_ERROR_WIN_DS_NO_SERVER_OBJECT = 8622,
  MD_ERROR_WIN_DS_NO_NTDSA_OBJECT = 8623,
  MD_ERROR_WIN_DS_NON_ASQ_SEARCH = 8624,
  MD_ERROR_WIN_DS_AUDIT_FAILURE = 8625,
  MD_ERROR_WIN_DS_INVALID_SEARCH_FLAG_SUBTREE = 8626,
  MD_ERROR_WIN_DS_INVALID_SEARCH_FLAG_TUPLE = 8627,
  MD_ERROR_WIN_DS_HIERARCHY_TABLE_TOO_DEEP = 8628,
  MD_ERROR_WIN_DS_DRA_CORRUPT_UTD_VECTOR = 8629,
  MD_ERROR_WIN_DS_DRA_SECRETS_DENIED = 8630,
  MD_ERROR_WIN_DS_RESERVED_MAPI_ID = 8631,
  MD_ERROR_WIN_DS_MAPI_ID_NOT_AVAILABLE = 8632,
  MD_ERROR_WIN_DS_DRA_MISSING_KRBTGT_SECRET = 8633,
  MD_ERROR_WIN_DS_DOMAIN_NAME_EXISTS_IN_FOREST = 8634,
  MD_ERROR_WIN_DS_FLAT_NAME_EXISTS_IN_FOREST = 8635,
  MD_ERROR_WIN_INVALID_USER_PRINCIPAL_NAME = 8636,
  MD_ERROR_WIN_DS_OID_MAPPED_GROUP_CANT_HAVE_MEMBERS = 8637,
  MD_ERROR_WIN_DS_OID_NOT_FOUND = 8638,
  MD_ERROR_WIN_DS_DRA_RECYCLED_TARGET = 8639,
  MD_ERROR_WIN_DS_DISALLOWED_NC_REDIRECT = 8640,
  MD_ERROR_WIN_DS_HIGH_ADLDS_FFL = 8641,
  MD_ERROR_WIN_DS_HIGH_DSA_VERSION = 8642,
  MD_ERROR_WIN_DS_LOW_ADLDS_FFL = 8643,
  MD_ERROR_WIN_DOMAIN_SID_SAME_AS_LOCAL_WORKSTATION = 8644,
  MD_ERROR_WIN_DS_UNDELETE_SAM_VALIDATION_FAILED = 8645,
  MD_ERROR_WIN_INCORRECT_ACCOUNT_TYPE = 8646,
  MD_ERROR_WIN_DS_SPN_VALUE_NOT_UNIQUE_IN_FOREST = 8647,
  MD_ERROR_WIN_DS_UPN_VALUE_NOT_UNIQUE_IN_FOREST = 8648,
  MD_ERROR_WIN_DS_MISSING_FOREST_TRUST = 8649,
  MD_ERROR_WIN_DS_VALUE_KEY_NOT_UNIQUE = 8650,
  MD_ERROR_WIN_IPSEC_QM_POLICY_EXISTS = 13000,
  MD_ERROR_WIN_IPSEC_QM_POLICY_NOT_FOUND = 13001,
  MD_ERROR_WIN_IPSEC_QM_POLICY_IN_USE = 13002,
  MD_ERROR_WIN_IPSEC_MM_POLICY_EXISTS = 13003,
  MD_ERROR_WIN_IPSEC_MM_POLICY_NOT_FOUND = 13004,
  MD_ERROR_WIN_IPSEC_MM_POLICY_IN_USE = 13005,
  MD_ERROR_WIN_IPSEC_MM_FILTER_EXISTS = 13006,
  MD_ERROR_WIN_IPSEC_MM_FILTER_NOT_FOUND = 13007,
  MD_ERROR_WIN_IPSEC_TRANSPORT_FILTER_EXISTS = 13008,
  MD_ERROR_WIN_IPSEC_TRANSPORT_FILTER_NOT_FOUND = 13009,
  MD_ERROR_WIN_IPSEC_MM_AUTH_EXISTS = 13010,
  MD_ERROR_WIN_IPSEC_MM_AUTH_NOT_FOUND = 13011,
  MD_ERROR_WIN_IPSEC_MM_AUTH_IN_USE = 13012,
  MD_ERROR_WIN_IPSEC_DEFAULT_MM_POLICY_NOT_FOUND = 13013,
  MD_ERROR_WIN_IPSEC_DEFAULT_MM_AUTH_NOT_FOUND = 13014,
  MD_ERROR_WIN_IPSEC_DEFAULT_QM_POLICY_NOT_FOUND = 13015,
  MD_ERROR_WIN_IPSEC_TUNNEL_FILTER_EXISTS = 13016,
  MD_ERROR_WIN_IPSEC_TUNNEL_FILTER_NOT_FOUND = 13017,
  MD_ERROR_WIN_IPSEC_MM_FILTER_PENDING_DELETION = 13018,
  MD_ERROR_WIN_IPSEC_TRANSPORT_FILTER_PENDING_DELETION = 13019,
  MD_ERROR_WIN_IPSEC_TUNNEL_FILTER_PENDING_DELETION = 13020,
  MD_ERROR_WIN_IPSEC_MM_POLICY_PENDING_DELETION = 13021,
  MD_ERROR_WIN_IPSEC_MM_AUTH_PENDING_DELETION = 13022,
  MD_ERROR_WIN_IPSEC_QM_POLICY_PENDING_DELETION = 13023,
  MD_ERROR_WIN_IPSEC_IKE_NEG_STATUS_BEGIN = 13800,
  MD_ERROR_WIN_IPSEC_IKE_AUTH_FAIL = 13801,
  MD_ERROR_WIN_IPSEC_IKE_ATTRIB_FAIL = 13802,
  MD_ERROR_WIN_IPSEC_IKE_NEGOTIATION_PENDING = 13803,
  MD_ERROR_WIN_IPSEC_IKE_GENERAL_PROCESSING_ERROR = 13804,
  MD_ERROR_WIN_IPSEC_IKE_TIMED_OUT = 13805,
  MD_ERROR_WIN_IPSEC_IKE_NO_CERT = 13806,
  MD_ERROR_WIN_IPSEC_IKE_SA_DELETED = 13807,
  MD_ERROR_WIN_IPSEC_IKE_SA_REAPED = 13808,
  MD_ERROR_WIN_IPSEC_IKE_MM_ACQUIRE_DROP = 13809,
  MD_ERROR_WIN_IPSEC_IKE_QM_ACQUIRE_DROP = 13810,
  MD_ERROR_WIN_IPSEC_IKE_QUEUE_DROP_MM = 13811,
  MD_ERROR_WIN_IPSEC_IKE_QUEUE_DROP_NO_MM = 13812,
  MD_ERROR_WIN_IPSEC_IKE_DROP_NO_RESPONSE = 13813,
  MD_ERROR_WIN_IPSEC_IKE_MM_DELAY_DROP = 13814,
  MD_ERROR_WIN_IPSEC_IKE_QM_DELAY_DROP = 13815,
  MD_ERROR_WIN_IPSEC_IKE_ERROR = 13816,
  MD_ERROR_WIN_IPSEC_IKE_CRL_FAILED = 13817,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_KEY_USAGE = 13818,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_CERT_TYPE = 13819,
  MD_ERROR_WIN_IPSEC_IKE_NO_PRIVATE_KEY = 13820,
  MD_ERROR_WIN_IPSEC_IKE_SIMULTANEOUS_REKEY = 13821,
  MD_ERROR_WIN_IPSEC_IKE_DH_FAIL = 13822,
  MD_ERROR_WIN_IPSEC_IKE_CRITICAL_PAYLOAD_NOT_RECOGNIZED = 13823,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_HEADER = 13824,
  MD_ERROR_WIN_IPSEC_IKE_NO_POLICY = 13825,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_SIGNATURE = 13826,
  MD_ERROR_WIN_IPSEC_IKE_KERBEROS_ERROR = 13827,
  MD_ERROR_WIN_IPSEC_IKE_NO_PUBLIC_KEY = 13828,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR = 13829,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_SA = 13830,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_PROP = 13831,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_TRANS = 13832,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_KE = 13833,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_ID = 13834,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_CERT = 13835,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_CERT_REQ = 13836,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_HASH = 13837,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_SIG = 13838,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_NONCE = 13839,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_NOTIFY = 13840,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_DELETE = 13841,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_VENDOR = 13842,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_PAYLOAD = 13843,
  MD_ERROR_WIN_IPSEC_IKE_LOAD_SOFT_SA = 13844,
  MD_ERROR_WIN_IPSEC_IKE_SOFT_SA_TORN_DOWN = 13845,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_COOKIE = 13846,
  MD_ERROR_WIN_IPSEC_IKE_NO_PEER_CERT = 13847,
  MD_ERROR_WIN_IPSEC_IKE_PEER_CRL_FAILED = 13848,
  MD_ERROR_WIN_IPSEC_IKE_POLICY_CHANGE = 13849,
  MD_ERROR_WIN_IPSEC_IKE_NO_MM_POLICY = 13850,
  MD_ERROR_WIN_IPSEC_IKE_NOTCBPRIV = 13851,
  MD_ERROR_WIN_IPSEC_IKE_SECLOADFAIL = 13852,
  MD_ERROR_WIN_IPSEC_IKE_FAILSSPINIT = 13853,
  MD_ERROR_WIN_IPSEC_IKE_FAILQUERYSSP = 13854,
  MD_ERROR_WIN_IPSEC_IKE_SRVACQFAIL = 13855,
  MD_ERROR_WIN_IPSEC_IKE_SRVQUERYCRED = 13856,
  MD_ERROR_WIN_IPSEC_IKE_GETSPIFAIL = 13857,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_FILTER = 13858,
  MD_ERROR_WIN_IPSEC_IKE_OUT_OF_MEMORY = 13859,
  MD_ERROR_WIN_IPSEC_IKE_ADD_UPDATE_KEY_FAILED = 13860,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_POLICY = 13861,
  MD_ERROR_WIN_IPSEC_IKE_UNKNOWN_DOI = 13862,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_SITUATION = 13863,
  MD_ERROR_WIN_IPSEC_IKE_DH_FAILURE = 13864,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_GROUP = 13865,
  MD_ERROR_WIN_IPSEC_IKE_ENCRYPT = 13866,
  MD_ERROR_WIN_IPSEC_IKE_DECRYPT = 13867,
  MD_ERROR_WIN_IPSEC_IKE_POLICY_MATCH = 13868,
  MD_ERROR_WIN_IPSEC_IKE_UNSUPPORTED_ID = 13869,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_HASH = 13870,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_HASH_ALG = 13871,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_HASH_SIZE = 13872,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_ENCRYPT_ALG = 13873,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_AUTH_ALG = 13874,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_SIG = 13875,
  MD_ERROR_WIN_IPSEC_IKE_LOAD_FAILED = 13876,
  MD_ERROR_WIN_IPSEC_IKE_RPC_DELETE = 13877,
  MD_ERROR_WIN_IPSEC_IKE_BENIGN_REINIT = 13878,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_RESPONDER_LIFETIME_NOTIFY = 13879,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_MAJOR_VERSION = 13880,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_CERT_KEYLEN = 13881,
  MD_ERROR_WIN_IPSEC_IKE_MM_LIMIT = 13882,
  MD_ERROR_WIN_IPSEC_IKE_NEGOTIATION_DISABLED = 13883,
  MD_ERROR_WIN_IPSEC_IKE_QM_LIMIT = 13884,
  MD_ERROR_WIN_IPSEC_IKE_MM_EXPIRED = 13885,
  MD_ERROR_WIN_IPSEC_IKE_PEER_MM_ASSUMED_INVALID = 13886,
  MD_ERROR_WIN_IPSEC_IKE_CERT_CHAIN_POLICY_MISMATCH = 13887,
  MD_ERROR_WIN_IPSEC_IKE_UNEXPECTED_MESSAGE_ID = 13888,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_AUTH_PAYLOAD = 13889,
  MD_ERROR_WIN_IPSEC_IKE_DOS_COOKIE_SENT = 13890,
  MD_ERROR_WIN_IPSEC_IKE_SHUTTING_DOWN = 13891,
  MD_ERROR_WIN_IPSEC_IKE_CGA_AUTH_FAILED = 13892,
  MD_ERROR_WIN_IPSEC_IKE_PROCESS_ERR_NATOA = 13893,
  MD_ERROR_WIN_IPSEC_IKE_INVALID_MM_FOR_QM = 13894,
  MD_ERROR_WIN_IPSEC_IKE_QM_EXPIRED = 13895,
  MD_ERROR_WIN_IPSEC_IKE_TOO_MANY_FILTERS = 13896,
  MD_ERROR_WIN_IPSEC_IKE_NEG_STATUS_END = 13897,
  MD_ERROR_WIN_IPSEC_IKE_KILL_DUMMY_NAP_TUNNEL = 13898,
  MD_ERROR_WIN_IPSEC_IKE_INNER_IP_ASSIGNMENT_FAILURE = 13899,
  MD_ERROR_WIN_IPSEC_IKE_REQUIRE_CP_PAYLOAD_MISSING = 13900,
  MD_ERROR_WIN_IPSEC_KEY_MODULE_IMPERSONATION_NEGOTIATION_PENDING = 13901,
  MD_ERROR_WIN_IPSEC_IKE_COEXISTENCE_SUPPRESS = 13902,
  MD_ERROR_WIN_IPSEC_IKE_RATELIMIT_DROP = 13903,
  MD_ERROR_WIN_IPSEC_IKE_PEER_DOESNT_SUPPORT_MOBIKE = 13904,
  MD_ERROR_WIN_IPSEC_IKE_AUTHORIZATION_FAILURE = 13905,
  MD_ERROR_WIN_IPSEC_IKE_STRONG_CRED_AUTHORIZATION_FAILURE = 13906,
  MD_ERROR_WIN_IPSEC_IKE_AUTHORIZATION_FAILURE_WITH_OPTIONAL_RETRY = 13907,
  MD_ERROR_WIN_IPSEC_IKE_STRONG_CRED_AUTHORIZATION_AND_CERTMAP_FAILURE = 13908,
  MD_ERROR_WIN_IPSEC_IKE_NEG_STATUS_EXTENDED_END = 13909,
  MD_ERROR_WIN_IPSEC_BAD_SPI = 13910,
  MD_ERROR_WIN_IPSEC_SA_LIFETIME_EXPIRED = 13911,
  MD_ERROR_WIN_IPSEC_WRONG_SA = 13912,
  MD_ERROR_WIN_IPSEC_REPLAY_CHECK_FAILED = 13913,
  MD_ERROR_WIN_IPSEC_INVALID_PACKET = 13914,
  MD_ERROR_WIN_IPSEC_INTEGRITY_CHECK_FAILED = 13915,
  MD_ERROR_WIN_IPSEC_CLEAR_TEXT_DROP = 13916,
  MD_ERROR_WIN_IPSEC_AUTH_FIREWALL_DROP = 13917,
  MD_ERROR_WIN_IPSEC_THROTTLE_DROP = 13918,
  MD_ERROR_WIN_IPSEC_DOSP_BLOCK = 13925,
  MD_ERROR_WIN_IPSEC_DOSP_RECEIVED_MULTICAST = 13926,
  MD_ERROR_WIN_IPSEC_DOSP_INVALID_PACKET = 13927,
  MD_ERROR_WIN_IPSEC_DOSP_STATE_LOOKUP_FAILED = 13928,
  MD_ERROR_WIN_IPSEC_DOSP_MAX_ENTRIES = 13929,
  MD_ERROR_WIN_IPSEC_DOSP_KEYMOD_NOT_ALLOWED = 13930,
  MD_ERROR_WIN_IPSEC_DOSP_NOT_INSTALLED = 13931,
  MD_ERROR_WIN_IPSEC_DOSP_MAX_PER_IP_RATELIMIT_QUEUES = 13932,
  MD_ERROR_WIN_SXS_SECTION_NOT_FOUND = 14000,
  MD_ERROR_WIN_SXS_CANT_GEN_ACTCTX = 14001,
  MD_ERROR_WIN_SXS_INVALID_ACTCTXDATA_FORMAT = 14002,
  MD_ERROR_WIN_SXS_ASSEMBLY_NOT_FOUND = 14003,
  MD_ERROR_WIN_SXS_MANIFEST_FORMAT_ERROR = 14004,
  MD_ERROR_WIN_SXS_MANIFEST_PARSE_ERROR = 14005,
  MD_ERROR_WIN_SXS_ACTIVATION_CONTEXT_DISABLED = 14006,
  MD_ERROR_WIN_SXS_KEY_NOT_FOUND = 14007,
  MD_ERROR_WIN_SXS_VERSION_CONFLICT = 14008,
  MD_ERROR_WIN_SXS_WRONG_SECTION_TYPE = 14009,
  MD_ERROR_WIN_SXS_THREAD_QUERIES_DISABLED = 14010,
  MD_ERROR_WIN_SXS_PROCESS_DEFAULT_ALREADY_SET = 14011,
  MD_ERROR_WIN_SXS_UNKNOWN_ENCODING_GROUP = 14012,
  MD_ERROR_WIN_SXS_UNKNOWN_ENCODING = 14013,
  MD_ERROR_WIN_SXS_INVALID_XML_NAMESPACE_URI = 14014,
  MD_ERROR_WIN_SXS_ROOT_MANIFEST_DEPENDENCY_NOT_INSTALLED = 14015,
  MD_ERROR_WIN_SXS_LEAF_MANIFEST_DEPENDENCY_NOT_INSTALLED = 14016,
  MD_ERROR_WIN_SXS_INVALID_ASSEMBLY_IDENTITY_ATTRIBUTE = 14017,
  MD_ERROR_WIN_SXS_MANIFEST_MISSING_REQUIRED_DEFAULT_NAMESPACE = 14018,
  MD_ERROR_WIN_SXS_MANIFEST_INVALID_REQUIRED_DEFAULT_NAMESPACE = 14019,
  MD_ERROR_WIN_SXS_PRIVATE_MANIFEST_CROSS_PATH_WITH_REPARSE_POINT = 14020,
  MD_ERROR_WIN_SXS_DUPLICATE_DLL_NAME = 14021,
  MD_ERROR_WIN_SXS_DUPLICATE_WINDOWCLASS_NAME = 14022,
  MD_ERROR_WIN_SXS_DUPLICATE_CLSID = 14023,
  MD_ERROR_WIN_SXS_DUPLICATE_IID = 14024,
  MD_ERROR_WIN_SXS_DUPLICATE_TLBID = 14025,
  MD_ERROR_WIN_SXS_DUPLICATE_PROGID = 14026,
  MD_ERROR_WIN_SXS_DUPLICATE_ASSEMBLY_NAME = 14027,
  MD_ERROR_WIN_SXS_FILE_HASH_MISMATCH = 14028,
  MD_ERROR_WIN_SXS_POLICY_PARSE_ERROR = 14029,
  MD_ERROR_WIN_SXS_XML_E_MISSINGQUOTE = 14030,
  MD_ERROR_WIN_SXS_XML_E_COMMENTSYNTAX = 14031,
  MD_ERROR_WIN_SXS_XML_E_BADSTARTNAMECHAR = 14032,
  MD_ERROR_WIN_SXS_XML_E_BADNAMECHAR = 14033,
  MD_ERROR_WIN_SXS_XML_E_BADCHARINSTRING = 14034,
  MD_ERROR_WIN_SXS_XML_E_XMLDECLSYNTAX = 14035,
  MD_ERROR_WIN_SXS_XML_E_BADCHARDATA = 14036,
  MD_ERROR_WIN_SXS_XML_E_MISSINGWHITESPACE = 14037,
  MD_ERROR_WIN_SXS_XML_E_EXPECTINGTAGEND = 14038,
  MD_ERROR_WIN_SXS_XML_E_MISSINGSEMICOLON = 14039,
  MD_ERROR_WIN_SXS_XML_E_UNBALANCEDPAREN = 14040,
  MD_ERROR_WIN_SXS_XML_E_INTERNALERROR = 14041,
  MD_ERROR_WIN_SXS_XML_E_UNEXPECTED_WHITESPACE = 14042,
  MD_ERROR_WIN_SXS_XML_E_INCOMPLETE_ENCODING = 14043,
  MD_ERROR_WIN_SXS_XML_E_MISSING_PAREN = 14044,
  MD_ERROR_WIN_SXS_XML_E_EXPECTINGCLOSEQUOTE = 14045,
  MD_ERROR_WIN_SXS_XML_E_MULTIPLE_COLONS = 14046,
  MD_ERROR_WIN_SXS_XML_E_INVALID_DECIMAL = 14047,
  MD_ERROR_WIN_SXS_XML_E_INVALID_HEXIDECIMAL = 14048,
  MD_ERROR_WIN_SXS_XML_E_INVALID_UNICODE = 14049,
  MD_ERROR_WIN_SXS_XML_E_WHITESPACEORQUESTIONMARK = 14050,
  MD_ERROR_WIN_SXS_XML_E_UNEXPECTEDENDTAG = 14051,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDTAG = 14052,
  MD_ERROR_WIN_SXS_XML_E_DUPLICATEATTRIBUTE = 14053,
  MD_ERROR_WIN_SXS_XML_E_MULTIPLEROOTS = 14054,
  MD_ERROR_WIN_SXS_XML_E_INVALIDATROOTLEVEL = 14055,
  MD_ERROR_WIN_SXS_XML_E_BADXMLDECL = 14056,
  MD_ERROR_WIN_SXS_XML_E_MISSINGROOT = 14057,
  MD_ERROR_WIN_SXS_XML_E_UNEXPECTEDEOF = 14058,
  MD_ERROR_WIN_SXS_XML_E_BADPEREFINSUBSET = 14059,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDSTARTTAG = 14060,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDENDTAG = 14061,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDSTRING = 14062,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDCOMMENT = 14063,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDDECL = 14064,
  MD_ERROR_WIN_SXS_XML_E_UNCLOSEDCDATA = 14065,
  MD_ERROR_WIN_SXS_XML_E_RESERVEDNAMESPACE = 14066,
  MD_ERROR_WIN_SXS_XML_E_INVALIDENCODING = 14067,
  MD_ERROR_WIN_SXS_XML_E_INVALIDSWITCH = 14068,
  MD_ERROR_WIN_SXS_XML_E_BADXMLCASE = 14069,
  MD_ERROR_WIN_SXS_XML_E_INVALID_STANDALONE = 14070,
  MD_ERROR_WIN_SXS_XML_E_UNEXPECTED_STANDALONE = 14071,
  MD_ERROR_WIN_SXS_XML_E_INVALID_VERSION = 14072,
  MD_ERROR_WIN_SXS_XML_E_MISSINGEQUALS = 14073,
  MD_ERROR_WIN_SXS_PROTECTION_RECOVERY_FAILED = 14074,
  MD_ERROR_WIN_SXS_PROTECTION_PUBLIC_KEY_TOO_SHORT = 14075,
  MD_ERROR_WIN_SXS_PROTECTION_CATALOG_NOT_VALID = 14076,
  MD_ERROR_WIN_SXS_UNTRANSLATABLE_HRESULT = 14077,
  MD_ERROR_WIN_SXS_PROTECTION_CATALOG_FILE_MISSING = 14078,
  MD_ERROR_WIN_SXS_MISSING_ASSEMBLY_IDENTITY_ATTRIBUTE = 14079,
  MD_ERROR_WIN_SXS_INVALID_ASSEMBLY_IDENTITY_ATTRIBUTE_NAME = 14080,
  MD_ERROR_WIN_SXS_ASSEMBLY_MISSING = 14081,
  MD_ERROR_WIN_SXS_CORRUPT_ACTIVATION_STACK = 14082,
  MD_ERROR_WIN_SXS_CORRUPTION = 14083,
  MD_ERROR_WIN_SXS_EARLY_DEACTIVATION = 14084,
  MD_ERROR_WIN_SXS_INVALID_DEACTIVATION = 14085,
  MD_ERROR_WIN_SXS_MULTIPLE_DEACTIVATION = 14086,
  MD_ERROR_WIN_SXS_PROCESS_TERMINATION_REQUESTED = 14087,
  MD_ERROR_WIN_SXS_RELEASE_ACTIVATION_CONTEXT = 14088,
  MD_ERROR_WIN_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY = 14089,
  MD_ERROR_WIN_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE = 14090,
  MD_ERROR_WIN_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME = 14091,
  MD_ERROR_WIN_SXS_IDENTITY_DUPLICATE_ATTRIBUTE = 14092,
  MD_ERROR_WIN_SXS_IDENTITY_PARSE_ERROR = 14093,
  MD_ERROR_WIN_MALFORMED_SUBSTITUTION_STRING = 14094,
  MD_ERROR_WIN_SXS_INCORRECT_PUBLIC_KEY_TOKEN = 14095,
  MD_ERROR_WIN_UNMAPPED_SUBSTITUTION_STRING = 14096,
  MD_ERROR_WIN_SXS_ASSEMBLY_NOT_LOCKED = 14097,
  MD_ERROR_WIN_SXS_COMPONENT_STORE_CORRUPT = 14098,
  MD_ERROR_WIN_ADVANCED_INSTALLER_FAILED = 14099,
  MD_ERROR_WIN_XML_ENCODING_MISMATCH = 14100,
  MD_ERROR_WIN_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT = 14101,
  MD_ERROR_WIN_SXS_IDENTITIES_DIFFERENT = 14102,
  MD_ERROR_WIN_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT = 14103,
  MD_ERROR_WIN_SXS_FILE_NOT_PART_OF_ASSEMBLY = 14104,
  MD_ERROR_WIN_SXS_MANIFEST_TOO_BIG = 14105,
  MD_ERROR_WIN_SXS_SETTING_NOT_REGISTERED = 14106,
  MD_ERROR_WIN_SXS_TRANSACTION_CLOSURE_INCOMPLETE = 14107,
  MD_ERROR_WIN_SMI_PRIMITIVE_INSTALLER_FAILED = 14108,
  MD_ERROR_WIN_GENERIC_COMMAND_FAILED = 14109,
  MD_ERROR_WIN_SXS_FILE_HASH_MISSING = 14110,
  MD_ERROR_WIN_SXS_DUPLICATE_ACTIVATABLE_CLASS = 14111,
  MD_ERROR_WIN_EVT_INVALID_CHANNEL_PATH = 15000,
  MD_ERROR_WIN_EVT_INVALID_QUERY = 15001,
  MD_ERROR_WIN_EVT_PUBLISHER_METADATA_NOT_FOUND = 15002,
  MD_ERROR_WIN_EVT_EVENT_TEMPLATE_NOT_FOUND = 15003,
  MD_ERROR_WIN_EVT_INVALID_PUBLISHER_NAME = 15004,
  MD_ERROR_WIN_EVT_INVALID_EVENT_DATA = 15005,
  MD_ERROR_WIN_EVT_CHANNEL_NOT_FOUND = 15007,
  MD_ERROR_WIN_EVT_MALFORMED_XML_TEXT = 15008,
  MD_ERROR_WIN_EVT_SUBSCRIPTION_TO_DIRECT_CHANNEL = 15009,
  MD_ERROR_WIN_EVT_CONFIGURATION_ERROR = 15010,
  MD_ERROR_WIN_EVT_QUERY_RESULT_STALE = 15011,
  MD_ERROR_WIN_EVT_QUERY_RESULT_INVALID_POSITION = 15012,
  MD_ERROR_WIN_EVT_NON_VALIDATING_MSXML = 15013,
  MD_ERROR_WIN_EVT_FILTER_ALREADYSCOPED = 15014,
  MD_ERROR_WIN_EVT_FILTER_NOTELTSET = 15015,
  MD_ERROR_WIN_EVT_FILTER_INVARG = 15016,
  MD_ERROR_WIN_EVT_FILTER_INVTEST = 15017,
  MD_ERROR_WIN_EVT_FILTER_INVTYPE = 15018,
  MD_ERROR_WIN_EVT_FILTER_PARSEERR = 15019,
  MD_ERROR_WIN_EVT_FILTER_UNSUPPORTEDOP = 15020,
  MD_ERROR_WIN_EVT_FILTER_UNEXPECTEDTOKEN = 15021,
  MD_ERROR_WIN_EVT_INVALID_OPERATION_OVER_ENABLED_DIRECT_CHANNEL = 15022,
  MD_ERROR_WIN_EVT_INVALID_CHANNEL_PROPERTY_VALUE = 15023,
  MD_ERROR_WIN_EVT_INVALID_PUBLISHER_PROPERTY_VALUE = 15024,
  MD_ERROR_WIN_EVT_CHANNEL_CANNOT_ACTIVATE = 15025,
  MD_ERROR_WIN_EVT_FILTER_TOO_COMPLEX = 15026,
  MD_ERROR_WIN_EVT_MESSAGE_NOT_FOUND = 15027,
  MD_ERROR_WIN_EVT_MESSAGE_ID_NOT_FOUND = 15028,
  MD_ERROR_WIN_EVT_UNRESOLVED_VALUE_INSERT = 15029,
  MD_ERROR_WIN_EVT_UNRESOLVED_PARAMETER_INSERT = 15030,
  MD_ERROR_WIN_EVT_MAX_INSERTS_REACHED = 15031,
  MD_ERROR_WIN_EVT_EVENT_DEFINITION_NOT_FOUND = 15032,
  MD_ERROR_WIN_EVT_MESSAGE_LOCALE_NOT_FOUND = 15033,
  MD_ERROR_WIN_EVT_VERSION_TOO_OLD = 15034,
  MD_ERROR_WIN_EVT_VERSION_TOO_NEW = 15035,
  MD_ERROR_WIN_EVT_CANNOT_OPEN_CHANNEL_OF_QUERY = 15036,
  MD_ERROR_WIN_EVT_PUBLISHER_DISABLED = 15037,
  MD_ERROR_WIN_EVT_FILTER_OUT_OF_RANGE = 15038,
  MD_ERROR_WIN_EC_SUBSCRIPTION_CANNOT_ACTIVATE = 15080,
  MD_ERROR_WIN_EC_LOG_DISABLED = 15081,
  MD_ERROR_WIN_EC_CIRCULAR_FORWARDING = 15082,
  MD_ERROR_WIN_EC_CREDSTORE_FULL = 15083,
  MD_ERROR_WIN_EC_CRED_NOT_FOUND = 15084,
  MD_ERROR_WIN_EC_NO_ACTIVE_CHANNEL = 15085,
  MD_ERROR_WIN_MUI_FILE_NOT_FOUND = 15100,
  MD_ERROR_WIN_MUI_INVALID_FILE = 15101,
  MD_ERROR_WIN_MUI_INVALID_RC_CONFIG = 15102,
  MD_ERROR_WIN_MUI_INVALID_LOCALE_NAME = 15103,
  MD_ERROR_WIN_MUI_INVALID_ULTIMATEFALLBACK_NAME = 15104,
  MD_ERROR_WIN_MUI_FILE_NOT_LOADED = 15105,
  MD_ERROR_WIN_RESOURCE_ENUM_USER_STOP = 15106,
  MD_ERROR_WIN_MUI_INTLSETTINGS_UILANG_NOT_INSTALLED = 15107,
  MD_ERROR_WIN_MUI_INTLSETTINGS_INVALID_LOCALE_NAME = 15108,
  MD_ERROR_WIN_MRM_RUNTIME_NO_DEFAULT_OR_NEUTRAL_RESOURCE = 15110,
  MD_ERROR_WIN_MRM_INVALID_PRICONFIG = 15111,
  MD_ERROR_WIN_MRM_INVALID_FILE_TYPE = 15112,
  MD_ERROR_WIN_MRM_UNKNOWN_QUALIFIER = 15113,
  MD_ERROR_WIN_MRM_INVALID_QUALIFIER_VALUE = 15114,
  MD_ERROR_WIN_MRM_NO_CANDIDATE = 15115,
  MD_ERROR_WIN_MRM_NO_MATCH_OR_DEFAULT_CANDIDATE = 15116,
  MD_ERROR_WIN_MRM_RESOURCE_TYPE_MISMATCH = 15117,
  MD_ERROR_WIN_MRM_DUPLICATE_MAP_NAME = 15118,
  MD_ERROR_WIN_MRM_DUPLICATE_ENTRY = 15119,
  MD_ERROR_WIN_MRM_INVALID_RESOURCE_IDENTIFIER = 15120,
  MD_ERROR_WIN_MRM_FILEPATH_TOO_LONG = 15121,
  MD_ERROR_WIN_MRM_UNSUPPORTED_DIRECTORY_TYPE = 15122,
  MD_ERROR_WIN_MRM_INVALID_PRI_FILE = 15126,
  MD_ERROR_WIN_MRM_NAMED_RESOURCE_NOT_FOUND = 15127,
  MD_ERROR_WIN_MRM_MAP_NOT_FOUND = 15135,
  MD_ERROR_WIN_MRM_UNSUPPORTED_PROFILE_TYPE = 15136,
  MD_ERROR_WIN_MRM_INVALID_QUALIFIER_OPERATOR = 15137,
  MD_ERROR_WIN_MRM_INDETERMINATE_QUALIFIER_VALUE = 15138,
  MD_ERROR_WIN_MRM_AUTOMERGE_ENABLED = 15139,
  MD_ERROR_WIN_MRM_TOO_MANY_RESOURCES = 15140,
  MD_ERROR_WIN_MRM_UNSUPPORTED_FILE_TYPE_FOR_MERGE = 15141,
  MD_ERROR_WIN_MRM_UNSUPPORTED_FILE_TYPE_FOR_LOAD_UNLOAD_PRI_FILE = 15142,
  MD_ERROR_WIN_MRM_NO_CURRENT_VIEW_ON_THREAD = 15143,
  MD_ERROR_WIN_DIFFERENT_PROFILE_RESOURCE_MANAGER_EXIST = 15144,
  MD_ERROR_WIN_OPERATION_NOT_ALLOWED_FROM_SYSTEM_COMPONENT = 15145,
  MD_ERROR_WIN_MRM_DIRECT_REF_TO_NON_DEFAULT_RESOURCE = 15146,
  MD_ERROR_WIN_MRM_GENERATION_COUNT_MISMATCH = 15147,
  MD_ERROR_WIN_PRI_MERGE_VERSION_MISMATCH = 15148,
  MD_ERROR_WIN_PRI_MERGE_MISSING_SCHEMA = 15149,
  MD_ERROR_WIN_PRI_MERGE_LOAD_FILE_FAILED = 15150,
  MD_ERROR_WIN_PRI_MERGE_ADD_FILE_FAILED = 15151,
  MD_ERROR_WIN_PRI_MERGE_WRITE_FILE_FAILED = 15152,
  MD_ERROR_WIN_PRI_MERGE_MULTIPLE_PACKAGE_FAMILIES_NOT_ALLOWED = 15153,
  MD_ERROR_WIN_PRI_MERGE_MULTIPLE_MAIN_PACKAGES_NOT_ALLOWED = 15154,
  MD_ERROR_WIN_PRI_MERGE_BUNDLE_PACKAGES_NOT_ALLOWED = 15155,
  MD_ERROR_WIN_PRI_MERGE_MAIN_PACKAGE_REQUIRED = 15156,
  MD_ERROR_WIN_PRI_MERGE_RESOURCE_PACKAGE_REQUIRED = 15157,
  MD_ERROR_WIN_PRI_MERGE_INVALID_FILE_NAME = 15158,
  MD_ERROR_WIN_MRM_PACKAGE_NOT_FOUND = 15159,
  MD_ERROR_WIN_MRM_MISSING_DEFAULT_LANGUAGE = 15160,
  MD_ERROR_WIN_MCA_INVALID_CAPABILITIES_STRING = 15200,
  MD_ERROR_WIN_MCA_INVALID_VCP_VERSION = 15201,
  MD_ERROR_WIN_MCA_MONITOR_VIOLATES_MCCS_SPECIFICATION = 15202,
  MD_ERROR_WIN_MCA_MCCS_VERSION_MISMATCH = 15203,
  MD_ERROR_WIN_MCA_UNSUPPORTED_MCCS_VERSION = 15204,
  MD_ERROR_WIN_MCA_INTERNAL_ERROR = 15205,
  MD_ERROR_WIN_MCA_INVALID_TECHNOLOGY_TYPE_RETURNED = 15206,
  MD_ERROR_WIN_MCA_UNSUPPORTED_COLOR_TEMPERATURE = 15207,
  MD_ERROR_WIN_AMBIGUOUS_SYSTEM_DEVICE = 15250,
  MD_ERROR_WIN_SYSTEM_DEVICE_NOT_FOUND = 15299,
  MD_ERROR_WIN_HASH_NOT_SUPPORTED = 15300,
  MD_ERROR_WIN_HASH_NOT_PRESENT = 15301,
  MD_ERROR_WIN_SECONDARY_IC_PROVIDER_NOT_REGISTERED = 15321,
  MD_ERROR_WIN_GPIO_CLIENT_INFORMATION_INVALID = 15322,
  MD_ERROR_WIN_GPIO_VERSION_NOT_SUPPORTED = 15323,
  MD_ERROR_WIN_GPIO_INVALID_REGISTRATION_PACKET = 15324,
  MD_ERROR_WIN_GPIO_OPERATION_DENIED = 15325,
  MD_ERROR_WIN_GPIO_INCOMPATIBLE_CONNECT_MODE = 15326,
  MD_ERROR_WIN_GPIO_INTERRUPT_ALREADY_UNMASKED = 15327,
  MD_ERROR_WIN_CANNOT_SWITCH_RUNLEVEL = 15400,
  MD_ERROR_WIN_INVALID_RUNLEVEL_SETTING = 15401,
  MD_ERROR_WIN_RUNLEVEL_SWITCH_TIMEOUT = 15402,
  MD_ERROR_WIN_RUNLEVEL_SWITCH_AGENT_TIMEOUT = 15403,
  MD_ERROR_WIN_RUNLEVEL_SWITCH_IN_PROGRESS = 15404,
  MD_ERROR_WIN_SERVICES_FAILED_AUTOSTART = 15405,
  MD_ERROR_WIN_COM_TASK_STOP_PENDING = 15501,
  MD_ERROR_WIN_INSTALL_OPEN_PACKAGE_FAILED = 15600,
  MD_ERROR_WIN_INSTALL_PACKAGE_NOT_FOUND = 15601,
  MD_ERROR_WIN_INSTALL_INVALID_PACKAGE = 15602,
  MD_ERROR_WIN_INSTALL_RESOLVE_DEPENDENCY_FAILED = 15603,
  MD_ERROR_WIN_INSTALL_OUT_OF_DISK_SPACE = 15604,
  MD_ERROR_WIN_INSTALL_NETWORK_FAILURE = 15605,
  MD_ERROR_WIN_INSTALL_REGISTRATION_FAILURE = 15606,
  MD_ERROR_WIN_INSTALL_DEREGISTRATION_FAILURE = 15607,
  MD_ERROR_WIN_INSTALL_CANCEL = 15608,
  MD_ERROR_WIN_INSTALL_FAILED = 15609,
  MD_ERROR_WIN_REMOVE_FAILED = 15610,
  MD_ERROR_WIN_PACKAGE_ALREADY_EXISTS = 15611,
  MD_ERROR_WIN_NEEDS_REMEDIATION = 15612,
  MD_ERROR_WIN_INSTALL_PREREQUISITE_FAILED = 15613,
  MD_ERROR_WIN_PACKAGE_REPOSITORY_CORRUPTED = 15614,
  MD_ERROR_WIN_INSTALL_POLICY_FAILURE = 15615,
  MD_ERROR_WIN_PACKAGE_UPDATING = 15616,
  MD_ERROR_WIN_DEPLOYMENT_BLOCKED_BY_POLICY = 15617,
  MD_ERROR_WIN_PACKAGES_IN_USE = 15618,
  MD_ERROR_WIN_RECOVERY_FILE_CORRUPT = 15619,
  MD_ERROR_WIN_INVALID_STAGED_SIGNATURE = 15620,
  MD_ERROR_WIN_DELETING_EXISTING_APPLICATIONDATA_STORE_FAILED = 15621,
  MD_ERROR_WIN_INSTALL_PACKAGE_DOWNGRADE = 15622,
  MD_ERROR_WIN_SYSTEM_NEEDS_REMEDIATION = 15623,
  MD_ERROR_WIN_APPX_INTEGRITY_FAILURE_CLR_NGEN = 15624,
  MD_ERROR_WIN_RESILIENCY_FILE_CORRUPT = 15625,
  MD_ERROR_WIN_INSTALL_FIREWALL_SERVICE_NOT_RUNNING = 15626,
  MD_ERROR_WIN_PACKAGE_MOVE_FAILED = 15627,
  MD_ERROR_WIN_INSTALL_VOLUME_NOT_EMPTY = 15628,
  MD_ERROR_WIN_INSTALL_VOLUME_OFFLINE = 15629,
  MD_ERROR_WIN_INSTALL_VOLUME_CORRUPT = 15630,
  MD_ERROR_WIN_NEEDS_REGISTRATION = 15631,
  MD_ERROR_WIN_INSTALL_WRONG_PROCESSOR_ARCHITECTURE = 15632,
  MD_ERROR_WIN_DEV_SIDELOAD_LIMIT_EXCEEDED = 15633,
  MD_ERROR_WIN_INSTALL_OPTIONAL_PACKAGE_REQUIRES_MAIN_PACKAGE = 15634,
  MD_ERROR_WIN_PACKAGE_NOT_SUPPORTED_ON_FILESYSTEM = 15635,
  MD_ERROR_WIN_PACKAGE_MOVE_BLOCKED_BY_STREAMING = 15636,
  MD_ERROR_WIN_INSTALL_OPTIONAL_PACKAGE_APPLICATIONID_NOT_UNIQUE = 15637,
  MD_ERROR_WIN_PACKAGE_STAGING_ONHOLD = 15638,
  MD_ERROR_WIN_INSTALL_INVALID_RELATED_SET_UPDATE = 15639,
  MD_ERROR_WIN_INSTALL_OPTIONAL_PACKAGE_REQUIRES_MAIN_PACKAGE_FULLTRUST_CAPABILITY = 15640,
  MD_ERROR_WIN_DEPLOYMENT_BLOCKED_BY_USER_LOG_OFF = 15641,
  MD_ERROR_WIN_PROVISION_OPTIONAL_PACKAGE_REQUIRES_MAIN_PACKAGE_PROVISIONED = 15642,
  MD_ERROR_WIN_PACKAGES_REPUTATION_CHECK_FAILED = 15643,
  MD_ERROR_WIN_PACKAGES_REPUTATION_CHECK_TIMEDOUT = 15644,
  MD_ERROR_WIN_DEPLOYMENT_OPTION_NOT_SUPPORTED = 15645,
  MD_ERROR_WIN_APPINSTALLER_ACTIVATION_BLOCKED = 15646,
  MD_ERROR_WIN_REGISTRATION_FROM_REMOTE_DRIVE_NOT_SUPPORTED = 15647,
  MD_ERROR_WIN_APPX_RAW_DATA_WRITE_FAILED = 15648,
  MD_ERROR_WIN_DEPLOYMENT_BLOCKED_BY_VOLUME_POLICY_PACKAGE = 15649,
  MD_ERROR_WIN_DEPLOYMENT_BLOCKED_BY_VOLUME_POLICY_MACHINE = 15650,
  MD_ERROR_WIN_DEPLOYMENT_BLOCKED_BY_PROFILE_POLICY = 15651,
  MD_ERROR_WIN_DEPLOYMENT_FAILED_CONFLICTING_MUTABLE_PACKAGE_DIRECTORY = 15652,
  MD_ERROR_WIN_SINGLETON_RESOURCE_INSTALLED_IN_ACTIVE_USER = 15653,
  MD_ERROR_WIN_DIFFERENT_VERSION_OF_PACKAGED_SERVICE_INSTALLED = 15654,
  MD_ERROR_WIN_SERVICE_EXISTS_AS_NON_PACKAGED_SERVICE = 15655,
  MD_ERROR_WIN_PACKAGED_SERVICE_REQUIRES_ADMIN_PRIVILEGES = 15656,
  MD_ERROR_WIN_REDIRECTION_TO_DEFAULT_ACCOUNT_NOT_ALLOWED = 15657,
  MD_ERROR_WIN_PACKAGE_LACKS_CAPABILITY_TO_DEPLOY_ON_HOST = 15658,
  MD_ERROR_WIN_UNSIGNED_PACKAGE_INVALID_CONTENT = 15659,
  MD_ERROR_WIN_UNSIGNED_PACKAGE_INVALID_PUBLISHER_NAMESPACE = 15660,
  MD_ERROR_WIN_SIGNED_PACKAGE_INVALID_PUBLISHER_NAMESPACE = 15661,
  MD_ERROR_WIN_PACKAGE_EXTERNAL_LOCATION_NOT_ALLOWED = 15662,
  MD_ERROR_WIN_INSTALL_FULLTRUST_HOSTRUNTIME_REQUIRES_MAIN_PACKAGE_FULLTRUST_CAPABILITY = 15663,
  MD_ERROR_WIN_STATE_LOAD_STORE_FAILED = 15800,
  MD_ERROR_WIN_STATE_GET_VERSION_FAILED = 15801,
  MD_ERROR_WIN_STATE_SET_VERSION_FAILED = 15802,
  MD_ERROR_WIN_STATE_STRUCTURED_RESET_FAILED = 15803,
  MD_ERROR_WIN_STATE_OPEN_CONTAINER_FAILED = 15804,
  MD_ERROR_WIN_STATE_CREATE_CONTAINER_FAILED = 15805,
  MD_ERROR_WIN_STATE_DELETE_CONTAINER_FAILED = 15806,
  MD_ERROR_WIN_STATE_READ_SETTING_FAILED = 15807,
  MD_ERROR_WIN_STATE_WRITE_SETTING_FAILED = 15808,
  MD_ERROR_WIN_STATE_DELETE_SETTING_FAILED = 15809,
  MD_ERROR_WIN_STATE_QUERY_SETTING_FAILED = 15810,
  MD_ERROR_WIN_STATE_READ_COMPOSITE_SETTING_FAILED = 15811,
  MD_ERROR_WIN_STATE_WRITE_COMPOSITE_SETTING_FAILED = 15812,
  MD_ERROR_WIN_STATE_ENUMERATE_CONTAINER_FAILED = 15813,
  MD_ERROR_WIN_STATE_ENUMERATE_SETTINGS_FAILED = 15814,
  MD_ERROR_WIN_STATE_COMPOSITE_SETTING_VALUE_SIZE_LIMIT_EXCEEDED = 15815,
  MD_ERROR_WIN_STATE_SETTING_VALUE_SIZE_LIMIT_EXCEEDED = 15816,
  MD_ERROR_WIN_STATE_SETTING_NAME_SIZE_LIMIT_EXCEEDED = 15817,
  MD_ERROR_WIN_STATE_CONTAINER_NAME_SIZE_LIMIT_EXCEEDED = 15818,
  MD_ERROR_WIN_API_UNAVAILABLE = 15841,
} MDErrorWin;

#endif  /* GOOGLE_BREAKPAD_COMMON_MINIDUMP_EXCEPTION_WIN32_H__ */
