// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief gamma gamma -> pi0pi0
  class CRYSTAL_BALL_1990_I294492 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(CRYSTAL_BALL_1990_I294492);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      // histos
      const double eval = sqrtS()/GeV;
      raiseBeamErrorIf(!inRange(eval, 0.25, 1.95));

      size_t ih=0;
      for (double emax : {0.5, 0.7, 0.9, 1.1, 1.3, 1.7}) {
        string label = toString(1000*emax);
        book(_h[label], 2, 1, ++ih);
        if (_sqs.empty() && eval < emax) _sqs = label;
      }

      if (inRange(eval, 1.1, 1.5, RangeBoundary::CLOSED, RangeBoundary::CLOSED)) {
        _sqs15 = "1500"s;
      }
      book(_h["1500"], 2, 1, 7);
      _edges = _h["1500"].binning().edges<0>();

      book(_h["npipi"], 1, 1, 1);
      for (const string& en : _h["npipi"].binning().edges<0>()) {
        if (isCompatibleWithSqrtS(stod(en))) {
          _edge = en; break;
        }
      }
      _axis = YODA::Axis<double>({0., 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8});
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      Particles part = apply<FinalState>(event,"FS").particles();
      if (part.size()!=2) vetoEvent;
      for (const Particle& p : part) {
        if (p.pid()!=PID::PI0) vetoEvent;
      }
      const double cTheta = abs(part[0].mom().z()/part[0].mom().p3().mod());
      if (cTheta<=0.8)     _h["npipi"]->fill(_edge);
      if (!_sqs.empty())   _h[_sqs]->fill(disc(cTheta));
      if (!_sqs15.empty()) _h["1500"]->fill(disc(cTheta));
    }


    string disc(const double value) const {
      size_t idx = _axis.index(value);
      if (0 < idx && idx <= _edges.size())  return _edges[idx-1];
      return "OTHER"s;
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_h, crossSection()/nanobarn/sumOfWeights());
      for (auto& item : _h) {
        if (item.first == "npipi"s)  continue;
        for (auto& b : item.second->bins()) {
          const size_t idx = b.index();
          b.scaleW(1./_axis.width(idx));
        }
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,BinnedHistoPtr<string>> _h;
    YODA::Axis<double> _axis;
    vector<string> _edges;
    string _sqs = "", _sqs15 = "", _edge = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(CRYSTAL_BALL_1990_I294492);

}
