/* dirbookmarkui.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2007-2008 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "dirbookmarkui.hh"
#include <aguix/aguix.h>
#include <aguix/awindow.h>
#include <aguix/text.h>
#include <aguix/fieldlistview.h>
#include <aguix/button.h>
#include "worker_locale.h"
#include "bookmarkdbentry.hh"
#include "bookmarkdbfilter.hh"
#include "dirbookmarkaddui.hh"
#include "dirbookmarkeditui.hh"

DirBookmarkUI::DirBookmarkUI( AGUIX &aguix, BookmarkDB &data  ) : m_aguix( aguix ),
                                                                  m_data( data )
{
    m_filtered_data = std::auto_ptr<BookmarkDBFilter>( new BookmarkDBFilter( data ) );

    m_win = std::auto_ptr<AWindow>( new AWindow( &m_aguix,
                                                 0, 0,
                                                 400, 400,
                                                 0, catalog.getLocaleCom( 51 ) ) );
    m_win->create();

    AContainer *co1 = m_win->setContainer( new AContainer( m_win.get(), 1, 4 ), true );
    co1->setMaxSpace( 5 );
    
    AContainer *co1_2 = co1->add( new AContainer( m_win.get(), 2, 1 ), 0, 0 );
    co1_2->setMaxSpace( 5 );
    co1_2->add( new Text( &m_aguix, 0, 0, catalog.getLocale( 758 ), 1 ),
                0, 0, AContainer::CFIX );
    m_infixtext = static_cast<Text*>( co1_2->add( new Text( &m_aguix, 0, 0, "", 1 ),
                                                  1, 0, AContainer::CINCW ) );
    
    m_lv = static_cast<FieldListView*>( co1->add( new FieldListView( &m_aguix, 
                                                                     0, 0, 
                                                                     400, 300, 0 ),
                                                  0, 1, AContainer::CMIN ) );
    m_lv->setNrOfFields( 2 );
    m_lv->setShowHeader( true );
    m_lv->setFieldText( 0, catalog.getLocale( 809 ) );
    m_lv->setFieldText( 1, catalog.getLocale( 810 ) );
    m_lv->setHBarState( 2 );
    m_lv->setVBarState( 2 );
    m_lv->setAcceptFocus( true );
    m_lv->setDisplayFocus( true );

    AContainer *co1_4 = co1->add( new AContainer( m_win.get(), 3, 1 ), 0, 2 );
    co1_4->setMinSpace( 5 );
    co1_4->setMaxSpace( -1 );
    co1_4->setBorderWidth( 0 );

    m_addb = (Button*)co1_4->add( new Button( &m_aguix,
                                              0,
                                              0,
                                              catalog.getLocale( 759 ),
                                              1,
                                              0,
                                              0 ), 0, 0, AContainer::CFIX );

    m_editb = (Button*)co1_4->add( new Button( &m_aguix,
                                               0,
                                               0,
                                               catalog.getLocale( 805 ),
                                               1,
                                               0,
                                               0 ), 1, 0, AContainer::CFIX );

    m_delb = (Button*)co1_4->add( new Button( &m_aguix,
                                              0,
                                              0,
                                              catalog.getLocale( 760 ),
                                              1,
                                              0,
                                              0 ), 2, 0, AContainer::CFIX );

    AContainer *co1_3 = co1->add( new AContainer( m_win.get(), 2, 1 ), 0, 3 );
    co1_3->setMinSpace( 5 );
    co1_3->setMaxSpace( -1 );
    co1_3->setBorderWidth( 0 );
    m_okb = (Button*)co1_3->add( new Button( &m_aguix,
                                             0,
                                             0,
                                             catalog.getLocale( 761 ),
                                             1,
                                             0,
                                             0 ), 0, 0, AContainer::CFIX );
    m_cancelb = (Button*)co1_3->add( new Button( &m_aguix,
                                                 0,
                                                 0,
                                                 catalog.getLocale( 633 ),
                                                 1,
                                                 0,
                                                 0 ), 1, 0, AContainer::CFIX );

    m_win->contMaximize( true );
    m_win->setDoTabCycling( true );
}

DirBookmarkUI::~DirBookmarkUI()
{
}

int DirBookmarkUI::mainLoop()
{
    m_win->show();
    showData( "" );
    m_lv->takeFocus();

    std::string current_cat = "";
    AGMessage *msg;
    int endmode = 0;
    for ( ; endmode == 0; ) {
        msg = m_aguix.WaitMessage( NULL );
        if ( msg != NULL ) {
            switch ( msg->type ) {
              case AG_CLOSEWINDOW:
                  endmode = -1;
                  break;
              case AG_BUTTONCLICKED:
                  if ( msg->button.button == m_okb ) {
                      endmode = 1;
                  } else if ( msg->button.button == m_cancelb ) {
                      endmode = -1;
                  } else if ( msg->button.button == m_addb ) {
                      DirBookmarkAddUI addui( m_aguix, m_data, m_dirname, m_basename );

                      if ( addui.mainLoop() > 0 ) {
                          m_data.write();
                          showData( current_cat );
                      }
                  } else if ( msg->button.button == m_editb ) {
                      BookmarkDBEntry active_entry = getActiveEntry( current_cat );
                      if ( active_entry.getName().length() > 0 ) {
                          DirBookmarkEditUI editui( m_aguix, m_data, active_entry );
                          
                          if ( editui.mainLoop() > 0 ) {
                              m_data.write();
                              showData( current_cat );
                          }
                      }
                  } else if ( msg->button.button == m_delb ) {
                      BookmarkDBEntry active_entry = getActiveEntry( current_cat );
                      if ( active_entry.getName().length() > 0 ) {
                          std::string buttontext = catalog.getLocale( 11 );
                          buttontext += "|";
                          buttontext += catalog.getLocale( 8 );

                          int erg = m_win->request( catalog.getLocale( 123 ),
                                                    catalog.getLocale( 762 ),
                                                    buttontext.c_str() );
                          if ( erg == 0 ) {
                              m_data.delEntry( active_entry );
                              m_data.write();
                              showData( current_cat );
                          }
                      }
                  }
                  break;
              case AG_KEYPRESSED:
                  if ( msg->key.key == XK_BackSpace ) {
                      if ( KEYSTATEMASK( msg->key.keystate ) == ShiftMask ) {
                          m_filtered_data->setInfixFilter( "" );
                          showData( current_cat );
                      } else {
                          std::string cur_infix = m_filtered_data->getInfixFilter();
                          
                          if ( cur_infix.length() > 0 ) {
                              cur_infix.resize( cur_infix.length() - 1 );
                              m_filtered_data->setInfixFilter( cur_infix );
                              showData( current_cat );
                          }
                      }
                  } else if ( msg->key.key == XK_Return ) {
                      endmode = 1;
                  } else if ( msg->key.key == XK_Escape ) {
                      endmode = -1;
                  } else if ( msg->key.key == XK_Left ) {
                  } else if ( msg->key.key == XK_Right ) {
                  } else if ( msg->key.key == XK_Up ) {
                  } else if ( msg->key.key == XK_Down ) {
                  } else if ( msg->key.key == XK_Next ) {
                  } else if ( msg->key.key == XK_Prior ) {
                  } else if ( msg->key.key == XK_Home ) {
                  } else if ( msg->key.key == XK_End ) {
                  } else if ( IsModifierKey( msg->key.key ) ||
                              IsCursorKey( msg->key.key ) ||
                              IsPFKey( msg->key.key ) ||
                              IsFunctionKey( msg->key.key ) ||
                              IsMiscFunctionKey( msg->key.key ) ||
                              ( ( msg->key.key >= XK_BackSpace ) && ( msg->key.key <= XK_Escape ) ) ) {
                  } else if ( strlen( msg->key.keybuf ) > 0 ) {
                      std::string cur_infix = m_filtered_data->getInfixFilter(), old_infix;
                      old_infix = cur_infix;
                      cur_infix += msg->key.keybuf;
                      m_filtered_data->setInfixFilter( cur_infix );
                      if ( m_filtered_data->getEntries( current_cat ).empty() == true ) {
                          m_filtered_data->setInfixFilter( old_infix );
                      }
                      showData( current_cat );
                  }
                  break;
              case AG_FIELDLV_DOUBLECLICK:
                  if ( msg->fieldlv.lv == m_lv ) {
                      // double click in lv, actual element is unimportant here
                      endmode = 1;
                  }
                  break;
            }
            m_aguix.ReplyMessage( msg );
        }
    }

    m_selected_entry.reset();

    if ( endmode == 1 ) {
        int row = m_lv->getActiveRow();
        if ( m_lv->isValidRow( row ) == true ) {
            std::list<BookmarkDBEntry> e = m_filtered_data->getEntries( current_cat );
            for ( std::list<BookmarkDBEntry>::iterator it1 = e.begin();
                  it1 != e.end();
                  it1++, row-- ) {
                if ( row == 0 ) {
                    m_selected_entry = std::auto_ptr<BookmarkDBEntry>( new BookmarkDBEntry( *it1 ) );
                }
            }
        }
    }

    m_win->hide();
    
    return endmode;
}

void DirBookmarkUI::showData( const std::string &cat )
{
    m_infixtext->setText( m_filtered_data->getInfixFilter().c_str() );

    int act_row = m_lv->getActiveRow();

    m_lv->setSize( 0 );
    
    std::list<BookmarkDBEntry> e = m_filtered_data->getEntries( cat );
    for ( std::list<BookmarkDBEntry>::iterator it1 = e.begin();
          it1 != e.end();
          it1++ ) {
        int row = m_lv->addRow();
        m_lv->setText( row, 0, it1->getName() );
        m_lv->setText( row, 1, it1->getAlias() );
        m_lv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
        
        if ( row == act_row )
            m_lv->setActiveRow( row );
    }

    if ( m_lv->isValidRow( m_lv->getActiveRow() ) == false ) {
        if ( act_row < 0 ) {
            m_lv->setActiveRow( 0 );
        } else {
            m_lv->setActiveRow( e.size() - 1 );
        }
    }

    m_lv->redraw();
}

BookmarkDBEntry DirBookmarkUI::getSelectedEntry()
{
    if ( m_selected_entry.get() != NULL ) {
        return *m_selected_entry;
    }
    throw 0;
}

void DirBookmarkUI::setCurrentDirname( const std::string &dirname )
{
    m_dirname = dirname;
}

void DirBookmarkUI::setCurrentBasename( const std::string &basename )
{
    m_basename = basename;
}

BookmarkDBEntry DirBookmarkUI::getActiveEntry( const std::string &cat )
{
    int row = m_lv->getActiveRow();
    BookmarkDBEntry dbe( "", "" );
    
    if ( m_lv->isValidRow( row ) == true ) {
        std::list<BookmarkDBEntry> e = m_filtered_data->getEntries( cat );
        for ( std::list<BookmarkDBEntry>::iterator it1 = e.begin();
              it1 != e.end();
              it1++, row-- ) {
            if ( row == 0 ) {
                dbe = *it1;
            }
        }
    }
    return dbe;
}
