%{

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "common.h"
#include "parseprg.h"
#include "parseprg-internal.h"

int yyparse();
void yy_scan_string(const char*);
void yyerror(char*);
int yylex();          // for Debian slink.
extern int yylineno;

node_stmt* program; // The result of parsing

// This is set on runtime errors:
char* runerror;

// Maximal number of iterations allowed in loops in a single execution
// of a program. We set a limit to prevent infintely executing code which 
// blocks everything:
const int max_iterations=10000;

// Number of iterations in this execution
int iterations;

//-----------------------------------------------------------------------------
// The symbol table

Symtable symtabel;  

Symtable::SymEntry* Symtable::getSymbol(char* name) {
  for(int a=0; a<symc; a++)
    if(!strcmp(name,symbols[a].name) && symbols[a].valid) // Symbol found 
      return &symbols[a];
  return 0;
}

Symtable::SymEntry* Symtable::createSymbol(char* name) {
  assert(!getSymbol(name));

  // Create the symbol:    
  if(symc==maxsymbols) { yyerror("Too many variabels"); return 0; }  
  if(strlen(name)>maxsymlen) { yyerror("Variabel name too long"); return 0; }
  strcpy(symbols[symc].name,name);
  symbols[symc].valid=0;
  return &symbols[symc++];
}
  
void Symtable::setType(SymEntry* s, ParseProgram::types type) {
  s->valid=1;
  s->type=type;
  
  switch(type) {
    case ParseProgram::tINT64: s->value.int64_value=new DynVarInt64(0); break;
    case ParseProgram::tBOOL:  s->value.bool_value=new DynVarBool(0); break;
    case ParseProgram::tMAC:   { mac_addr a; a.setZero(); s->value.mac_value=new DynVarMac(a); break; }
    case ParseProgram::tIP:    { ip_addr a; a.setZero(); s->value.ip_value=new DynVarIp(a); break; }
    default: assert(0);
  }
}
	
//-----------------------------------------------------------------------------
// Statement nodes:

class JoinStmt:public node_stmt {
  node_stmt &a, &b;
  virtual void eval() { a.eval(); b.eval(); }
  public:
  JoinStmt(node_stmt& A, node_stmt& B) : a(A), b(B) {}
};

class If:public node_stmt {
  node_boolexpr &boole;
  node_stmt& stmts;  
  virtual void eval() { if(boole.eval()) stmts.eval(); }
  public:
  If(node_boolexpr& A, node_stmt& B) : boole(A), stmts(B) {}
};

class While:public node_stmt {
  node_boolexpr &boole;
  node_stmt& stmts;  
  virtual void eval() { 
    while(boole.eval()) {
      if(iterations>=max_iterations) {
        // If we have executed too many times:
	if(!runerror) runerror="Program made too many loops. If this is not an error adjust constant in parseprg.y";
        return; // The program will not exit emditiatly but now more iteratins
	        // are performed... 
      }
      iterations++;    
      stmts.eval();      
    }
  }
  public:
  While(node_boolexpr& A, node_stmt& B) : boole(A), stmts(B) {}
};

class IfElse:public node_stmt {
  node_boolexpr &boole;
  node_stmt &stmt1, &stmt2;  
  virtual void eval() { 
    if(boole.eval()) { 
      stmt1.eval(); 
    } else { 
      stmt2.eval(); 
    } 
  }
  public:
  // B and C can be 0
  IfElse(node_boolexpr& A, node_stmt& B, node_stmt& C) : boole(A), stmt1(B), stmt2(C) {}
};

//------------------------------------------------------------------------------
// Logical operations

template<class NodeT>class LogAnd:public node_boolexpr {
  NodeT &a,&b;
  virtual Bool eval() {
    return a.eval() && b.eval();
  }
  
  public:  
  LogAnd(NodeT& A, NodeT& B) : a(A), b(B) {}  
};
typedef LogAnd<node_boolexpr> LogAndBool;

template<class NodeT> class LogOr:public node_boolexpr {
  NodeT &a,&b;
  
  virtual Bool eval() {
    return a.eval() || b.eval();
  }
  
  public:
  LogOr(NodeT& A, NodeT& B) : a(A), b(B) {}  
};
typedef LogOr<node_boolexpr> LogOrBool;

template<class NodeT> class LogNot:public node_boolexpr {
  NodeT &a;
  
  virtual Bool eval() {
    return a.eval()==0;
  }
  
  public:
  LogNot(NodeT& A) : a(A) {}
};
typedef LogNot<node_boolexpr> LogNotBool;

template<class NodeT> class Less:public node_boolexpr {
  NodeT &a,&b;
  
  virtual Bool eval() {
    return a.eval()<b.eval();
  }
  
  public:
  Less(NodeT& A, NodeT& B) : a(A), b(B) {}  
};
typedef Less<node_int64expr> LessInt;
typedef Less<node_int64expr> LessInt64;

template<class NodeT> class LessEq:public node_boolexpr {
  NodeT &a,&b;
  
  virtual Bool eval() {
    return a.eval()<=b.eval();
  }
  
  public:
  LessEq(NodeT& A, NodeT& B) : a(A), b(B) {}  
};
typedef LessEq<node_int64expr> LessEqInt;
typedef LessEq<node_int64expr> LessEqInt64;

template<class NodeT, class T> class Eq:public node_boolexpr {
  NodeT &a,&b;
  
  virtual Bool eval() {
    return a.eval()==b.eval();
  }
  
  public:
  Eq(NodeT& A, NodeT& B) : a(A), b(B) {}  
};
typedef Eq<node_int64expr,int64> EqInt;
typedef Eq<node_boolexpr,Bool> EqBool;
typedef Eq<node_macexpr,mac_addr> EqMac;
typedef Eq<node_ipexpr,ip_addr> EqIp;

template<class NodeT, class T> class Neq:public node_boolexpr {
  NodeT &a,&b;
  
  virtual Bool eval() {
    return a.eval()!=b.eval();
  }
  
  public:
  Neq(NodeT& A, NodeT& B) : a(A), b(B) {}  
};
typedef Neq<node_int64expr,int64> NeqInt;
typedef Neq<node_boolexpr,Bool> NeqBool;
typedef Neq<node_macexpr,mac_addr> NeqMac;
typedef Neq<node_ipexpr,ip_addr> NeqIp;

//------------------------------------------------------------------------------
// Arithmetic operations.

template<class NodeT, class T> class Add:public NodeT {
  NodeT &a,&b;
  virtual T eval() {
    return a.eval() + b.eval();
  }
  
  public:  
  Add(NodeT& A, NodeT& B) : a(A), b(B) {}  
};
typedef Add<node_int64expr,int64> AddInt64;

template<class NodeT, class T> class Sub:public NodeT {
  NodeT &a,&b;
  virtual T eval() {
    return a.eval() - b.eval();
  }
  
  public:  
  Sub(NodeT& A, NodeT& B) : a(A), b(B) {}  
};
typedef Sub<node_int64expr,int64> SubInt64;

template<class NodeT, class T> class Mul:public NodeT {
  NodeT &a,&b;
  virtual T eval() {
    return a.eval() * b.eval();
  }
  
  public:  
  Mul(NodeT& A, NodeT& B) : a(A), b(B) {}  
};
typedef Mul<node_int64expr,int64> MulInt64;

template<class NodeT, class T> class Div:public NodeT {
  NodeT &a,&b;
  virtual T eval() {
    T d=b.eval();
    if(d==(T)0) { if(!runerror) runerror="Division by zero"; return 0; }
    return a.eval() / d;
  }
  
  public:  
  Div(NodeT& A, NodeT& B) : a(A), b(B) {}  
};
typedef Div<node_int64expr,int64> DivInt64;

template<class NodeT, class T> class Shl:public NodeT {
  NodeT &a;
  node_int64expr &b;
  virtual T eval() {
    return a.eval() << b.eval();
  }
  
  public:  
  Shl(NodeT& A, node_int64expr& B) : a(A), b(B) {}  
};
typedef Shl<node_int64expr,int64> ShlInt64;

template<class NodeT, class T> class Shr:public NodeT {
  NodeT &a;
  node_int64expr &b;
  virtual T eval() {
    return a.eval() >> b.eval();
  }
  
  public:  
  Shr(NodeT& A, node_int64expr& B) : a(A), b(B) {}  
};
typedef Shr<node_int64expr,int64> ShrInt64;

//------------------------------------------------------------------------------
// Other nodes

template<class ExprT, class VarT, class T> class VarToExpr:public ExprT {
  VarT& a;
  virtual T eval() { return *(a.eval()); }
  public:
  VarToExpr(VarT& A): a(A) {}
};
typedef VarToExpr<node_int64expr,node_int64var,int64> VarToExprInt64;
typedef VarToExpr<node_boolexpr,node_boolvar,Bool> VarToExprBool;
typedef VarToExpr<node_macexpr,node_macvar,mac_addr> VarToExprMac;
typedef VarToExpr<node_ipexpr,node_ipvar,ip_addr> VarToExprIp;

template<class NodeT> class ExprToStmt:public node_stmt {
  NodeT& a;
  virtual void eval() { a.eval(); }
  public:
  ExprToStmt(NodeT& A) : a(A) {}
};
typedef ExprToStmt<node_int64expr> ExprToStmtInt64;
typedef ExprToStmt<node_boolexpr> ExprToStmtBool;
typedef ExprToStmt<node_macexpr> ExprToStmtMac;
typedef ExprToStmt<node_ipexpr> ExprToStmtIp;

template<class ExprT, class VarT, class T> class Assign:public ExprT {
  VarT& location;
  ExprT& value;
  
  virtual T eval() { return *location.eval()=value.eval(); }
  
  public:
  Assign(VarT& A, ExprT& B) : location(A), value(B)  {}
};
typedef Assign<node_int64expr,node_int64var,int64> AssignInt64;
typedef Assign<node_boolexpr,node_boolvar,Bool> AssignBool;
typedef Assign<node_macexpr,node_macvar,mac_addr> AssignMac;
typedef Assign<node_ipexpr,node_ipvar,ip_addr> AssignIp;

//------------------------------------------------------------------------------
// Other simple converting:

class BoolToInt64:public node_int64expr {
  node_boolexpr& i;  
  
  virtual int64 eval() { return i.eval()!=0; }

  public:
  BoolToInt64(node_boolexpr& f) : i(f) {}
};

class IpMask: public node_ipexpr {
  node_ipexpr& ip;
  node_int64expr& num;

  virtual ip_addr eval() {
    unsigned int in;
    ip_addr ret;

    in = ntohl(ip.eval().ia.s_addr);
    ret.ia.s_addr = htonl(((unsigned long)(~0)) << (32-(int) num.eval()) & in);
    return ret;
  }

public:
  IpMask(node_ipexpr& cip, node_int64expr& cnum) : ip(cip), num(cnum) { }
};

class Int64ToBool:public node_boolexpr {
  node_int64expr& i;  
  
  virtual Bool eval() { return i.eval()!=(int64)0; }

  public:
  Int64ToBool(node_int64expr& f) : i(f) {}
};

class MacToId:public node_idexpr {  
  node_macexpr& m;
  
  virtual card_id eval() {
    card_id ci; 
    memset(&ci,0,sizeof(ci));
    ci.macaddr=m.eval();
    return ci;
  }
  
  public:
  MacToId(node_macexpr& mm) : m(mm) {}
};

class IpToId:public node_idexpr {  
  node_ipexpr& m;
  
  virtual card_id eval() {
    card_id ci; 
    memset(&ci,0,sizeof(ci));
    ci.ipaddr=m.eval();
    return ci;
  }
  
  public:
  IpToId(node_ipexpr& mm) : m(mm) {}
}; 

//-----------------------------------------------------------------------------
// Printing:

class PrintExprInt:public node_stmt {
  node_int64expr& a;
  
  virtual void eval() {  
    int64 r=a.eval();
    char buf[40]; snprintf(buf,40,"%u:%u",unsigned(r>>32),unsigned(r & int64(0xffffffffU)));
    ParseProgram::instance->print(buf);
  }
  
  public:
  PrintExprInt(node_int64expr& expr) : a(expr) {}
};

class PrintExprShort:public node_stmt {
  node_int64expr& a;
  
  virtual void eval() {
    char buf[40]; snprintf(buf,40,"%u",unsigned(a.eval()));
    ParseProgram::instance->print(buf);
  }
  
  public:
  PrintExprShort(node_int64expr& expr) : a(expr) {}
};

class PrintExprBool:public node_stmt {
  node_boolexpr& a;
  
  virtual void eval() {
    ParseProgram::instance->print(a.eval()!=0 ? "true" : "false");
  }
  
  public:
  PrintExprBool(node_boolexpr& expr) : a(expr) {}
};

class PrintExprMac:public node_stmt {
  node_macexpr& a;
  
  virtual void eval() {
    ParseProgram::instance->print(a.eval().tostr());
  }
  
  public:
  PrintExprMac(node_macexpr& expr) : a(expr) {}
};

class PrintExprIp:public node_stmt {
  node_ipexpr& a;
  
  virtual void eval() {
    ParseProgram::instance->print(a.eval().tostr());
  }
  
  public:
  PrintExprIp(node_ipexpr& expr) : a(expr) {}
};

class PrintString:public node_stmt {
  char* a;
  
  virtual void eval() {
    ParseProgram::instance->print(a);
  }
  
  public:
  PrintString(char* A) : a(A) {}
};

//-----------------------------------------------------------------------------
// Table manipulation:

// Number of cards
class fncTBLcardcnt:public node_int64expr {
  virtual int64 eval() {
    return ParseProgram::instance->TBLcardcnt();
  }
};

// Number of counters:
class fncTBLcntcnt:public node_int64expr {
  virtual int64 eval() {
    return ParseProgram::instance->TBLcntcnt();
  }
};

// Deleting a card:
class fncTBLidxdel:public node_stmt {
  node_int64expr& idx;
 
  virtual void eval() {
    ParseProgram::instance->TBLidxdel(idx.eval(),runerror);
  }
  
  public:
  fncTBLidxdel(node_int64expr& a) : idx(a) {}
};

// Get index for a card_id adress
class fncTBLgetidx:public node_int64expr {
  node_idexpr& m;
 
  virtual int64 eval() {
    return ParseProgram::instance->TBLgetidx(m.eval(),runerror);
  }
  
  public:
  fncTBLgetidx(node_idexpr& a) : m(a) {}
};

// Get a MAC address:
class fncTBLgetmacadr:public node_macexpr {
  node_int64expr& idx;
  
  virtual mac_addr eval() {
    return ParseProgram::instance->TBLgetmacaddr(idx.eval(),runerror);
  }
  
  public:
  fncTBLgetmacadr(node_int64expr& a) : idx(a) {}
};

// Get an IP address:
class fncTBLgetipadr:public node_ipexpr {
  node_int64expr& idx;
  
  virtual ip_addr eval() {
    return ParseProgram::instance->TBLgetipaddr(idx.eval(),runerror);
  }
  
  public:
  fncTBLgetipadr(node_int64expr& a) : idx(a) {}
};

// Returns a counter value:
class fncTBLgetcnt:public node_int64var {
  node_idexpr& m;
  node_int64expr& i;
 
  virtual int64* eval() {
    return ParseProgram::instance->TBLgetcnt(m.eval(),i.eval(),runerror);
  }
  
  public:
  fncTBLgetcnt(node_idexpr& A, node_int64expr& B) : m(A), i(B) {}
};

//-----------------------------------------------------------------------------
%}

%union {
  node_stmt*      nstmt;      
  node_int64expr* nint64expr;
  node_boolexpr*  nboolexpr;
  node_macexpr*   nmacexpr;
  node_ipexpr*    nipexpr;
  node_idexpr*    nidexpr;
  node_int64var*  nint64var;
  node_boolvar*   nboolvar;
  node_macvar*    nmacvar;
  node_ipvar*     nipvar;
  char*           nstring;
  Symtable::SymEntry* nnotypevar;
}

%type <nstmt>      stmt stmtlist program
%type <nint64expr> int64expr
%type <nint64var>  int64var
%type <nboolexpr>  boolexpr
%type <nboolvar>   boolvar
%type <nidexpr>    idexpr
%type <nipexpr>    ipexpr
%type <nipvar>     ipvar
%type <nmacexpr>   macexpr
%type <nmacvar>    macvar

%token <nint64var>  INT64VAR
%token <nboolvar>   BOOLVAR
%token <nmacvar>    MACVAR
%token <nipvar>     IPVAR
%token <nint64expr> INT64EXPR
%token <nboolexpr>  BOOLEXPR
%token <nmacexpr>   MACEXPR
%token <nipexpr>    IPEXPR
%token <nnotypevar> NOTYPEVAR
%token <nstring>    STRING

%token IF ELSE WHILE PRINT PRINTS
%token AND OR LEQ GEQ EQ NEQ LE GE SHL SHR
%token ADDEQ SUBEQ DIVEQ MULEQ SHREQ SHLEQ
%token TBLCARDCNT TBLCNTCNT TBLGETIDX TBLGETIPADR TBLGETMACADR TBLIDXDEL
%token DECLMACA DECLIPA DECLINT64 DECLBOOL
%token ERROR

%right '='
%left OR
%left AND 
%left LE GE EQ LEQ GEQ NEQ
%left SHL SHR SHLEQ SHREQ
%left '+' '-' ADDEQ SUBEQ
%left '*' '/' MULEQ DIVEQ
%left '!'

%%
// The following grammer has a shift/reduce conflict. This is from the
// dangling else ambiguty and this is correctly resolved by yacc.

program  : /* empty */                                 { program=0; }
         | stmtlist                                    { program=$1; }

stmtlist : stmtlist stmt                               { $$=new JoinStmt(*$1,*$2); }
         | stmt                                        { $$=$1; }
	
stmt     : '{' stmtlist '}'                            { $$=$2; }
         | IF '(' boolexpr ')' stmt ELSE stmt          { $$=new IfElse(*$3,*$5,*$7); }
	 | IF '(' boolexpr ')' stmt                    { $$=new If(*$3,*$5);    }
         | WHILE '(' boolexpr ')' stmt                 { $$=new While(*$3,*$5); }	
         | TBLIDXDEL '(' int64expr ')' ';'             { $$=new fncTBLidxdel(*$3); }
	 | PRINT '(' int64expr ')' ';'                 { $$=new PrintExprInt(*$3); }
	 | PRINTS '(' int64expr ')' ';'                { $$=new PrintExprShort(*$3); }
	 | PRINT '(' boolexpr ')' ';'                  { $$=new PrintExprBool(*$3); }
	 | PRINT '(' STRING ')' ';'                    { $$=new PrintString($3); }
	 | PRINT '(' macexpr ')' ';'                   { $$=new PrintExprMac(*$3); }
	 | PRINT '(' ipexpr ')' ';'                    { $$=new PrintExprIp(*$3); }
	 | int64expr ';'                               { $$=new ExprToStmtInt64(*$1); }
  	 | boolexpr ';'                                { $$=new ExprToStmtBool(*$1);  }
	 | ipexpr ';'                                  { $$=new ExprToStmtIp(*$1);    }
	 | macexpr ';'                                 { $$=new ExprToStmtMac(*$1);   }

idexpr   : macexpr                                     { char* c=ParseProgram::instance->assert_cit(idMAC); 
                                                         if(c) { yyerror(c); YYERROR; }
                                                         $$=new MacToId(*$1); }
         | ipexpr                                      { char* c=ParseProgram::instance->assert_cit(idIP); 
                                                         if(c) { yyerror(c); YYERROR; }
                                                         $$=new IpToId(*$1); }
	 

macexpr  : macvar                       { $$=new VarToExprMac(*$1);  }
       	 | macvar '=' macexpr           { $$=new AssignMac(*$1,*$3); }
         | MACEXPR                  
         | '(' macexpr ')'              { $$=$2; }
 	 | TBLGETMACADR '(' int64expr ')' { $$=new fncTBLgetmacadr(*$3);    }
	 
macvar   : MACVAR
	 | DECLMACA NOTYPEVAR     { symtabel.setType($2,ParseProgram::tMAC); $$=$2->value.mac_value; }

ipexpr   : ipvar                  { $$=new VarToExprIp(*$1);  }	
       	 | ipvar '=' ipexpr       { $$=new AssignIp(*$1,*$3); }
         | IPEXPR                 
         | '(' ipexpr ')'         { $$=$2; }
 	 | TBLGETIPADR '(' int64expr ')'  { $$=new fncTBLgetipadr(*$3);    }
         | ipexpr '/' int64expr   { $$=new IpMask(*$1, *$3); }
	
ipvar    : IPVAR
	 | DECLIPA NOTYPEVAR      { symtabel.setType($2,ParseProgram::tIP); $$=$2->value.ip_value; }

int64expr: int64var                     { $$=new VarToExprInt64(*$1);  }	
     	 | int64var '=' int64expr       { $$=new AssignInt64(*$1,*$3); }
	 | INT64EXPR	
	 | '(' int64expr ')'            { $$=$2;                     } 
	 | int64expr '+' int64expr      { $$=new AddInt64(*$1,*$3);    }
	 | int64expr '-' int64expr      { $$=new SubInt64(*$1,*$3);    }
 	 | int64expr '*' int64expr      { $$=new MulInt64(*$1,*$3);    }
 	 | int64expr '/' int64expr      { $$=new DivInt64(*$1,*$3);    }
 	 | int64expr SHL int64expr      { $$=new ShlInt64(*$1,*$3);    }
 	 | int64expr SHR int64expr      { $$=new ShrInt64(*$1,*$3);    }
	 | int64var ADDEQ int64expr     { $$=new AssignInt64(*$1,*new AddInt64(*new VarToExprInt64(*$1),*$3)); }
	 | int64var SUBEQ int64expr     { $$=new AssignInt64(*$1,*new SubInt64(*new VarToExprInt64(*$1),*$3)); }
	 | int64var DIVEQ int64expr     { $$=new AssignInt64(*$1,*new DivInt64(*new VarToExprInt64(*$1),*$3)); }
	 | int64var MULEQ int64expr     { $$=new AssignInt64(*$1,*new MulInt64(*new VarToExprInt64(*$1),*$3)); }
	 | int64var SHLEQ int64expr     { $$=new AssignInt64(*$1,*new ShlInt64(*new VarToExprInt64(*$1),*$3)); }
	 | int64var SHREQ int64expr     { $$=new AssignInt64(*$1,*new ShrInt64(*new VarToExprInt64(*$1),*$3)); }	 
	 | TBLCARDCNT '(' ')'           { $$=new fncTBLcardcnt();      }
	 | TBLCNTCNT '(' ')'            { $$=new fncTBLcntcnt();       }
	 | TBLGETIDX '(' idexpr ')'     { $$=new fncTBLgetidx(*$3);    }
	 | DECLINT64 '(' boolexpr ')'   { $$=new BoolToInt64(*$3);     }

int64var : INT64VAR	
	 | DECLINT64 NOTYPEVAR      { symtabel.setType($2,ParseProgram::tINT64); $$=$2->value.int64_value; }
         | NOTYPEVAR                { yyerror("Undeclared variabel"); YYERROR; }
	 | idexpr '[' int64expr ']' { $$=new fncTBLgetcnt(*$1,*$3); }
		 
boolexpr : boolvar                { $$=new VarToExprBool(*$1);  }	
         | boolvar '=' boolexpr   { $$=new AssignBool(*$1,*$3); }		
	 | BOOLEXPR
         | boolexpr AND boolexpr  { $$=new LogAndBool(*$1,*$3); }
	 | boolexpr OR   boolexpr { $$=new LogOrBool(*$1,*$3);  }
	 | '!' boolexpr           { $$=new LogNotBool(*$2);     }
	 | '(' boolexpr ')'       { $$=$2;                      }	  
	 | macexpr EQ macexpr     { $$=new EqMac(*$1,*$3);      }
	 | macexpr NEQ macexpr    { $$=new NeqMac(*$1,*$3);     }
	 | ipexpr EQ ipexpr       { $$=new EqIp(*$1,*$3);       }
	 | ipexpr NEQ ipexpr      { $$=new NeqIp(*$1,*$3);      }
	 | boolexpr EQ boolexpr   { $$=new EqBool(*$1,*$3);     }
	 | boolexpr NEQ boolexpr  { $$=new NeqBool(*$1,*$3);    }	 
  	 | int64expr EQ int64expr  { $$=new EqInt(*$1,*$3);     }
	 | int64expr NEQ int64expr { $$=new NeqInt(*$1,*$3);    }
         | int64expr LE   int64expr { $$=new LessInt(*$1,*$3);   }
         | int64expr LEQ  int64expr { $$=new LessEqInt(*$1,*$3); }
	 | int64expr GE   int64expr { $$=new LessInt(*$3,*$1);   }
	 | int64expr GEQ  int64expr { $$=new LessEqInt(*$3,*$1); }	
	 | DECLBOOL '(' int64expr ')' { $$=new Int64ToBool(*$3); }

boolvar  : BOOLVAR
	 | DECLBOOL NOTYPEVAR         { symtabel.setType($2,ParseProgram::tBOOL); $$=$2->value.bool_value; }
	 	 
%%

//------------------------------------------------------------------------------
// Define module interface:

#include <assert.h>

// Compiletime error handling:
int anyerror;
int linestart;
void yyerror(char *s) {
  if(!anyerror) {
    snprintf(errmsg,sizeof(errmsg),"Error in program line %i: %s",yylineno+linestart,s);
    anyerror=1;
  }
  // We ignore all but the first error. This is because this module
  // calls yyerror and afterwards make yacc give this function a standard
  // error message
}

ParseProgram* ParseProgram::instance=0;

ParseProgram::ParseProgram() {
  assert(instance==0); // Only one instance is allowed because of global
                       // variabels.
  instance=this;		      
  
  // Reset variabels:
  symtabel.clear();

  cit=idUNKNOWN;
}

ParseProgram::~ParseProgram() {
  instance=0;
}

char* ParseProgram::setProgram(int ls, char* c) {
  linestart=ls-1;
  
  // Parse program
  yy_scan_string(c);
  anyerror=0;
  if(yyparse()) {
    if(!anyerror) yyerror("Parse error"); // make a default error message
    return errmsg;
  }
  return 0;
}

char* ParseProgram::execProgram() {
  runerror=0;
  iterations=0;
  if(program) program->eval();
  if(!runerror) return 0;
  // Runerror might point to "errmsg":
  char tmp[sizeof(errmsg)];
  snprintf(tmp,sizeof(tmp),"%s",runerror);
  snprintf(errmsg,sizeof(errmsg),"Program runtime error: %s",tmp);
  return errmsg;
}

char* ParseProgram::assert_cit(card_id_type c) {
  if(cit==idUNKNOWN) cit=c;
  
  if(cit!=c) {
    return "A card lookup was used which is not supported by the addressing mode";
  }
  
  return 0;
}
