//<copyright>
//
// Copyright (c) 1995,96,97
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        wrldraw.C
//
// Purpose:     drawing routines of VRML nodes
//
// Created:     24 Apr 95   Michael Pichler
//
// Changed:     24 Feb 97   Alexander Nussbaumer (editing)
//
// Changed:     24 Feb 97   Michael Pichler
//
// $Id: wrldraw.C,v 1.35 1997/04/30 10:52:28 mpichler Exp $
//
//</file>


#if defined(PMAX) || defined (HPUX9)
enum Part { goofyPart };        // cfront confused about QvCone::Part and QvCylinder::Part
enum Binding { goofyBinding };  // cfront confused about QvMaterialBinding/QvNormalBinding::Binding
#endif


#include <vrml/QvElement.h>
#include <vrml/QvNodes.h>
#include <vrml/QvExtensions.h>
#include <vrml/QvTransform.h>
#include <vrml/QvUnknownNode.h>

#include "arrays.h"
#include "scene3d.h"
#include "camera.h"
#include "vrmlscene.h"
#include "vecutil.h"
#include "fonts.h"
#include "gecontext.h"

#include <ge3d/ge3d.h>

#include <hyperg/utils/verbose.h>

#include <math.h>
#include <iostream.h>


static QvMaterial* draw_curmtl = 0;
static int draw_hilit = 0;  // which material component to use (QvMaterial::hilit_*)
static int draw_hilanchors = 0;  // how to highlight anchors    (- " -)
static int draw_hilnonanch = 0;  // how to draw non-anchors    (- " -)
static QvNode* draw_selected = 0;  // selected node
static int texturetrfchanged = 0;  // was QvTexture2Transform applied
static int texturehandle = 0;  // 0 if texturing turned off
static int nomaterials = 0;  // ignore Material settings


static const IntArray* draw_selected_path = 0;
static IntArray current_path (8);


/***** drawVRML *****/

// wrapper around root_->draw to set state flags

void VRMLScene::drawVRML ()
{
  static int anchorshilit [Scene3D::num_colors] =
  { // mapping of Scene anchor highlighting (Scene3D::l_*) to Material highlighting
    0, QvMaterial::hilit_bright /*brightness*/, QvMaterial::hilit_none /*BBox*/,
    QvMaterial::hilit_colshade /* color */, QvMaterial::hilit_none /*edges*/
  };
  static int nonanchhilit [Scene3D::num_colors] =
  { 0, QvMaterial::hilit_dark /*brightness*/, QvMaterial::hilit_none /*BBox*/,
    QvMaterial::hilit_greyshade /* color */, QvMaterial::hilit_none /*edges*/
  };
  // BBox hightlighting anachronism; color edge hightlighting: TODO (also in ge3d)

  if (scene_->linksActive ())
  {
    draw_hilanchors = anchorshilit [scene_->linksColor ()];
    draw_hilnonanch = nonanchhilit [scene_->linksColor ()];
  }
  else
    draw_hilanchors = draw_hilnonanch = QvMaterial::hilit_none;

  nomaterials = scene_->noMaterials () || quickwire_;

  // TODO: provide default material for anchor highlighting
  draw_curmtl = 0;  // no material active
  draw_hilit = draw_hilnonanch;  // now outside of any anchor

  // cerr << "using material " << draw_hilit << " and " << draw_hilanchors << " for anchors " << endl;

  // selection box
  draw_selected = scene_->selectedNode ();
  if (draw_selected)
    draw_selected_path = scene_->selectedPath ();
  else
    draw_selected_path = 0;

  if (QvNode::curdrawmode_ == ge3d_texturing)
    ge3dLoadTextureIdentity ();
  texturetrfchanged = 0;

  ge3d_push_matrix ();  // top node need not be a separator

  root_->draw ();  // ass.: root_ != 0
  // also sets current_path at root level

  ge3d_pop_matrix ();

  if (QvNode::curdrawmode_ == ge3d_texturing)
  {
    ge3dAlphaTest (0.0);  // turn off alpha test (only used for textures)
    if (texturetrfchanged)
      ge3dLoadTextureIdentity ();
  }

} // drawVRML



/*** selection ***/


// QvNode::draw
// draws selection highlighting (bounding cube)
// should be called at end of all selectable nodes/groups

void QvNode::draw ()
{
  // care for multiple instances (check path to selected object)
  if (draw_selected_path && draw_selected == this && hasextent_ &&
    *draw_selected_path == current_path)
  { // see also GeometricObject::draw
    // might enlarge selection cube a bit or draw "in front" (see hidden-line in ge3d)
    // to be visible for cubes as well

    // store transformation matrix of selected object
    if (scene_->manipulationAllowed ())
    {
      const QvTransform* selectedtransform = scene_->data ()->selectedTransform ();

      // puts the selected transform matrix on the stack, if necessary (for groups)
      // and gets current transformation matrix
      ge3d_push_matrix ();
      if (selectedtransform)
      {
        if (vrmlscene_->selTransformInsideGroup ())
        { // group object is selected
          ge3d_get_matrix (selectedTransformationT_);
          ge3dMultMatrix ((const float (*)[4]) selectedtransform->mat_);
          ge3d_get_matrix (selectedTransformation_);
        }
        else
        { // for geometric nodes and for groups, of which the selected transform is outside
          ge3d_get_matrix (selectedTransformation_);
          ge3dMultMatrix ((const float (*)[4]) selectedtransform->invmat_);
          ge3d_get_matrix (selectedTransformationT_);
          ge3dMultMatrix ((const float (*)[4]) selectedtransform->mat_);
        }
      }
      else
      {
        ge3d_get_matrix (selectedTransformation_);
        ge3d_get_matrix (selectedTransformationT_);
      }

      // gets the right object extent for group nodes (without the first transform) 
      point3D omin, omax;
      vrmlscene_->getObjectExtent (omin, omax);

      // draw bounding box
      const short l_pat = (short) 0x00FF;  // 16 bit
      const colorRGB& col_anchoredge = scene_->col_anchoredge;

      // dashes in complementary color
      ge3d_setlinestyle (~l_pat);
      ge3d_setlinecolor (1.0, 1.0 - col_anchoredge.G, 1.0 - col_anchoredge.B);  // reddish complement
      ge3dWireCube (&omin, &omax);
      // dashes in color anchor edge
      ge3d_setlinestyle (l_pat);
      ge3dLineColor (&col_anchoredge);
      ge3dWireCube (&omin, &omax);
      ge3d_setlinestyle (-1);

      if (draw_curmtl)  // restore material
        draw_curmtl->draw ();
      // mpi-TODO: this does not change edge color of cubes!

      ge3d_pop_matrix ();
      return;
    }

    // non-editing mode
    ge3d_get_matrix (selectedTransformation_);  // for "view selection"
    // ge3d_get_matrix (selectedTransformationT_);

    // draw bounding box
    const short l_pat = (short) 0x00FF;  // 16 bit
    const colorRGB& col_anchoredge = scene_->col_anchoredge;

    // dashes in complementary color
    ge3d_setlinestyle (~l_pat);
    ge3d_setlinecolor (1.0, 1.0 - col_anchoredge.G, 1.0 - col_anchoredge.B);  // reddish complement
    ge3dWireCube (&omin_, &omax_);
    // dashes in color anchor edge
    ge3d_setlinestyle (l_pat);
    ge3dLineColor (&col_anchoredge);
    ge3dWireCube (&omin_, &omax_);
    ge3d_setlinestyle (-1);

    if (draw_curmtl)  // restore material
      draw_curmtl->draw ();
    // known bug: materials not restored at Separators
  } // draw selection

} // QvNode



/***** groups *****/


void QvGroup::draw ()
{
  int n = getNumChildren ();
  if (draw_selected_path)  // find selected node
  {
    int level = current_path.count ();  // before children
    current_path.append (0);
    for (int i = 0; i < n; i++)
    {
      current_path.set_value (level, i);
      getChild (i)->draw ();
    }
    current_path.set_count (level);
  }
  else
    for (int i = 0; i < n; i++)
      getChild (i)->draw ();

  QvNode::draw ();
} // QvGroup


void QvLOD::draw ()
{
  Camera* cam = scene_->getCamera ();
  int numchildren = getNumChildren ();

  if (!cam || !numchildren)  // nothing to draw
  { lastdrawn_ = -1;
    return;
  }

  point3D campos, wcenter;
  cam->getposition (campos);

  // transform center_ into world coordinates
  ge3dTransformMcWc (center_, &wcenter);
  // cerr << "camera pos.: " << campos << ", world center: " << wcenter;
  dec3D (wcenter, *center_);
  float distance = norm3D (wcenter);
  // cerr << ", distance: " << distance << endl;

  int numranges = range.num;
  const float* ranges = range.values;

  int i;
  for (i = 0; i < numranges; i++)
  { if (distance < *ranges++)  // use child i
      break;
  } // to use all ranges there should be numranges+1 children

  if (i >= numchildren)  // default: last child
    i = numchildren-1;

  int level = current_path.count ();
  if (draw_selected_path)  // find selected node
    current_path.append (i);

  lastdrawn_ = i;
  getChild (i)->draw ();

  if (draw_selected_path)
    current_path.set_count (level);

  QvNode::draw ();
} // QvLOD


// texture matrix must (conceptually) be pushed/popped
// on Separators like geometry transformation matrix
// however, OpenGL texture matrix stack is very narrow,
// so save/restore the transformation manually;
// also turn off texturing if turned on in this separator

#define PUSHTEXTURETRANSFORM  \
  matrix4D texture_matrix;  \
  int savetextrfchanged = texturetrfchanged;  \
  int savetexturehandle = texturehandle;  \
  if (curdrawmode_ == ge3d_texturing)  \
  { ge3dGetTextureMatrix (texture_matrix);  \
    texturetrfchanged = 0;  \
  }

#define POPTEXTURETRANSFORM  \
  if (curdrawmode_ == ge3d_texturing)  \
  { if (texturetrfchanged)  /* within this separator */  \
      ge3dLoadTextureMatrix ((const float (*)[4]) texture_matrix);  \
    texturetrfchanged = savetextrfchanged;  /* for parent separator */  \
    if (texturehandle != savetexturehandle)  \
    { ge3dDoTexturing (savetexturehandle);  \
      if (savetexturehandle)  \
        ge3dApplyTexture (savetexturehandle);  \
    }  \
    texturehandle = savetexturehandle;  \
  }


void QvSeparator::draw ()
{
  int numlights = vrmlscene_->numLights ();
  //cerr << "QvSeparator::draw: push ()" << endl;
  ge3d_push_matrix ();
  PUSHTEXTURETRANSFORM

  int n = getNumChildren ();
  if (draw_selected_path)  // find selected node
  {
    int level = current_path.count ();  // before children
    current_path.append (0);
    for (int i = 0; i < n; i++)
    {
      current_path.set_value (level, i);
      getChild (i)->draw ();
    }
    current_path.set_count (level);
  }
  else
    for (int i = 0; i < n; i++)
      getChild (i)->draw ();

  //cerr << "QvSeparator::draw: pop ()" << endl;
  POPTEXTURETRANSFORM
  ge3d_pop_matrix ();
  vrmlscene_->deactivateLights (numlights);

  QvNode::draw ();
} // QvSeparator


void QvSwitch::draw ()
{
  int which = (int) whichChild.value;

  if (which == QV_SWITCH_NONE)  // draw nothing
    return;
  if (which == QV_SWITCH_ALL)  // draw all children
    QvGroup::draw ();
  else if (which < getNumChildren ())  // draw one child
  {
    int level = current_path.count ();
    if (draw_selected_path)  // find selected node
      current_path.append (which);

    getChild (which)->draw ();
    QvNode::draw ();

    if (draw_selected_path)
      current_path.set_count (level);
  }
} // QvSwitch


void QvTransformSeparator::draw ()
{
  ge3d_push_matrix ();
  // no effect on Texture2Transform

  int n = getNumChildren ();
  if (draw_selected_path)  // find selected node
  {
    int level = current_path.count ();  // before children
    current_path.append (0);
    for (int i = 0; i < n; i++)
    {
      current_path.set_value (level, i);
      getChild (i)->draw ();
    }
    current_path.set_count (level);
  }
  else
    for (int i = 0; i < n; i++)
      getChild (i)->draw ();

  ge3d_pop_matrix ();

  QvNode::draw ();
} // QvTransformSeparator



#define DRAW(className)  \
void className::draw ()  { }



/***** coordinates *****/


// already handled in build
DRAW(QvCoordinate3)
DRAW(QvNormal)
DRAW(QvTextureCoordinate2)



/***** properties *****/


DRAW(QvFontStyle)  // already handled in build

void QvMaterial::draw ()
{
//cerr << "QvMaterial::draw" << endl;
  // material binding (front/back) is not part of VRML (applys to both sides)

  if (nomaterials)  // even no anchor highlighting?
    return;

  ge3dMaterial (mat_front_and_back, materials_ + draw_hilit);
  // sets only basecolor in wireframe/hidden line or without lighting

// const materialsGE3D* mat = materials_ + draw_hilit;
// cerr << "num base: " << mat->num_base;
// if (mat->num_base)
// cerr << ", base RGB: "<< mat->rgb_base->R << ", " << mat->rgb_base->G << ", " << mat->rgb_base->B;
// cerr << endl;
// cerr << "num ambient: " << mat->num_ambient;
// if (mat->num_ambient)
// cerr << ", ambient RGB: "<< mat->rgb_ambient->R << ", " << mat->rgb_ambient->G << ", " << mat->rgb_ambient->B;
// cerr << endl;

  draw_curmtl = this;
}

DRAW(QvMaterialBinding)
DRAW(QvNormalBinding)


void QvShapeHints::draw ()
{
  if (scene_->twosidedpolys () == Scene3D::twosided_auto)
    ge3dHint (hint_backfaceculling, backfaceculling_);
}


void QvTexture2::draw ()
{
  if (curdrawmode_ != ge3d_texturing)  // no effect unless texturing
    return;

  if (!requested_)
  {
    requested_ = 1;
    if (hgtexture_)  // Hyper-G: texture request already registered, initiate downloading
      scene_->requestTextures ();
    else if (filename.value.getLength ())
      scene_->requestTextureURL (this, filename.value.getString (), parentURL_.getString ());
  }

  if (handle_ > 0)
  { ge3dDoTexturing (1);
    if (hasalpha_ && scene_->textureTransparency ())
      ge3dAlphaTest (0.5);  // might make transparency threshold user configurable
    ge3dApplyTexture (handle_);
    texturehandle = handle_;
    ge3dTextureRepeat (reps_, rept_);
  }
  else  // turn off texturing
  { ge3dDoTexturing (0);
    texturehandle = 0;
  }
}


void QvTexture2Transform::draw ()
{
  if (curdrawmode_ == ge3d_texturing)
  {
    ge3dMultTextureMatrix ((const float (*)[4]) mat_);
    texturetrfchanged = 1;  // for restoring in Separator
  }
}



/***** lights *****/


void QvDirectionalLight::draw ()
{
  if (curdrawmode_ < ge3d_flat_shading || scene_->dolighting () == Scene3D::lighting_never)
    return;

  if (on.value)
  { int index = vrmlscene_->nextLight ();
    ge3dSetLightSource (index, &color_, &direction_, 0 /* directional */, 0);
    ge3d_switchlight (index, 1);
  }
}


void QvPointLight::draw ()
{
  if (curdrawmode_ < ge3d_flat_shading || scene_->dolighting () == Scene3D::lighting_never)
    return;

  if (on.value)
  { int index = vrmlscene_->nextLight ();
    ge3dSetLightSource (index, &color_, position_, 1 /* positional */, 0);
    ge3d_switchlight (index, 1);
  }
}


void QvSpotLight::draw ()
{
  if (curdrawmode_ < ge3d_flat_shading || scene_->dolighting () == Scene3D::lighting_never)
    return;

  if (on.value)
  { int index = vrmlscene_->nextLight ();
    ge3dSpotLight (index, &color_, position_, direction_, dropOffRate.value, cutangle_);
    ge3d_switchlight (index, 1);
  }
}



/***** cameras *****/


void QvOrthographicCamera::draw ()
{
  // TODO (?): provide mechanism for zooming with ortho cameras
//   Camera* cam = vrmlscene_->getCamera ();
//   ge3dCamera (/*cam_orthographic | */ cam_relative,
//     pos_, rotangle_*180/M_PI, rotaxis_, height_,
//     scene_->getWinAspect (),
//     0.0, cam->getyon ()
//   );

//   // orthographic transformation already set up by global Camera
//   ge3dRotate (rotaxis_, -rotangle_);
//   ge3d_translate (-pos_->x, -pos_->y, -pos_->z);

  ge3dMultMatrix ((const float (*)[4]) mat_);
}


void QvPerspectiveCamera::draw ()
{
//   Camera* cam = vrmlscene_->getCamera ();
//   // focal distance not relevant here (may be used for default overview)
//   ge3dCamera (/* cam_perspective | */ cam_relative,
//     pos_, rotangle_*180/M_PI, rotaxis_, yangle_*180/M_PI,
//     scene_->getWinAspect (),
//     cam->gethither (), cam->getyon ()
//   );

//   // perspective transformation already set up by global Camera
//   ge3dRotate (rotaxis_, -rotangle_);
//   ge3d_translate (-pos_->x, -pos_->y, -pos_->z);

  ge3dMultMatrix ((const float (*)[4]) mat_);
}


/***** transformations *****/


// might calculate overall transformations right after loading

void QvTransform::draw ()
{
  // cast: gorasche, 19950709 for WIN32
  ge3dMultMatrix ((const float (*)[4]) mat_);
}

void QvRotation::draw ()
{
  ge3dMultMatrix ((const float (*)[4]) mat_);
}

void QvMatrixTransform::draw ()
{
  ge3dMultMatrix (mat_);
}

void QvTranslation::draw ()
{
  ge3dTranslate (trans_);
}

void QvScale::draw ()
{
  ge3dScale (scale_);
}



/***** shapes *****/


void QvAsciiText::draw ()
{
  if (!fontchars_)
    return;

  int numrows = string.num;
  QvString* rowstr = string.values;
  const point3D* roworig = roworigin_;
  // const point3D* rowmax = rowmax_;  // test
  const float* rowhspc = rowhspc_;
  const unsigned char* str;  // QvSFString: non-NULL, NULL-terminated
  FontChar* fchar;

  static const float shearmatrix [4][4] =  // shear matrix for italic font
  { {    1, 0, 0, 0 },
    { 0.25, 1, 0, 0 },  // increase x depending on y, origin unchanged (tan of slant)
    {    0, 0, 1, 0 },  // slant typically 13 to 15 degrees
    {    0, 0, 0, 1 }
  };

  if (size_)
  { ge3d_push_matrix ();
    ge3d_scale (1.0, 1.0, 1.0, size_);
  }

  if (bold_)
    ge3d_setlinewidth (2);

  while (numrows--)  // for each row
  {
    ge3d_push_matrix ();

    // ge3dWireCube (roworig, rowmax);  rowmax++;  // test: show bounding box

    ge3dTranslate (roworig++);  // line origin (usually <= 0)
    if (italic_)
      ge3dMultMatrix (shearmatrix);

    for (str = (const unsigned char*) rowstr++->getString ();  *str;  str++)
    {
      fchar = fontchars_ + *str;
      if (fchar->glyph ())
        fchar->glyph ()->draw ();
      // advance to next char
      if (fchar->width ())  // space has no glyph but width
        ge3d_translate (fchar->width () + *rowhspc, 0, 0);
      else
        ge3d_translate (*rowhspc, 0, 0);
    }

    rowhspc++;
    ge3d_pop_matrix ();
  } // for each row

  if (bold_)
    ge3d_setlinewidth (1);

  if (size_)
    ge3d_pop_matrix ();

// QvNode::draw ();  // must implement selection (pick) first
}


void QvCube::draw ()
{
  ge3dCube (&omin_, &omax_);
  QvNode::draw ();
}


void QvCone::draw ()
{
  float h = height.value;
  ge3dCylinder (bottomRadius.value, 0 /* topradius */, - h / 2.0, h, parts_);
  QvNode::draw ();
}


void QvCylinder::draw ()
{
  float r = radius.value;
  float h = height.value;
  ge3dCylinder (r, r, - h / 2.0, h, parts_);
  QvNode::draw ();
}


void QvSphere::draw ()
{
  ge3dSphere (radius.value);
  QvNode::draw ();
}


void QvIndexedFaceSet::draw ()
{
// cerr << "QvIndexedFaceSet::draw (" << numvertinds_ << ", " << numnormalinds_ << ", " << numtextureinds_ << ")" << endl;
// cerr << "vertexlist: " << vertexlist_ << ", []: " << *vertexlist_ << endl;
// cerr << "vertexindices: " << vertindices_ << ", []: " << *vertindices_ << endl;

  // rpersche, 19960603: convexify
  // may write ge3d function specialized in triangles
  // (GL_TRIANGLES, no need to check for -1 in index list etc.)
  if (numconvexinds_ && curdrawmode_ > ge3d_wireframe)
    ge3dFaceSet (vertexlist_, numconvexinds_, (const int*) convexCoordIndex.values,
      materials_ + draw_hilit, (nomaterials ? (int) matb_overall : matbinding_), nummatinds_, matindices_,
      normallist_, numnormalinds_, normalindices_, (const vector3D*) convexFaceNormals_.values,
      texvertlist_, numtextureinds_, textureindices_
    );
  else
    ge3dFaceSet (
      vertexlist_, numvertinds_, vertindices_,
      materials_ + draw_hilit, (nomaterials ? (int) matb_overall : matbinding_), nummatinds_, matindices_,
      normallist_, numnormalinds_, normalindices_, facenormals_,
      texvertlist_, numtextureinds_, textureindices_
    );
  QvNode::draw ();

} // QvIndexedFaceSet


void QvIndexedLineSet::draw ()
{
//cerr << "QvIndexedLineSet::draw" << endl;

  // assert: int and long of QvMFLong are of same size - see <vrml/QvMFLong>
  // assert: arrays of point3D's are compatible to arrays of float triples

  ge3dLineSet (vertexlist_, numvertinds_, vertindices_);
  // currently materials ignored (TODO) - (normals and textures not relevant)
  QvNode::draw ();

} // QvIndexedLineSet


void QvPointSet::draw ()
{
  ge3dPointSet (points_, num_);  // TODO: materials (normals, textures not relevant)
  QvNode::draw ();

} // QvPointSet



/***** WWW *****/


void QvWWWAnchor::draw ()
{
  int oldhilit = draw_hilit;  // care for nested anchors

  if (draw_hilit != draw_hilanchors)  // activate anchor highlighting
  {
    draw_hilit = draw_hilanchors;
    //cerr << "anchor: changing draw_hilit to " << draw_hilit << ", saved " << oldhilit << "; " << draw_curmtl << endl;
    if (draw_curmtl)
      draw_curmtl->draw ();
  }

  // spec: WWWAnchor must behave like a separator (although it is not derived from it)
  int numlights = vrmlscene_->numLights ();
  ge3d_push_matrix ();
  PUSHTEXTURETRANSFORM

  // draw children of group
  int n = getNumChildren ();
  if (draw_selected_path)  // find selected node
  {
    int level = current_path.count ();  // before children
    current_path.append (0);
    for (int i = 0; i < n; i++)
    {
      current_path.set_value (level, i);
      getChild (i)->draw ();
    }
    current_path.set_count (level);
  }
  else
    for (int i = 0; i < n; i++)
      getChild (i)->draw ();

  POPTEXTURETRANSFORM
  ge3d_pop_matrix ();
  vrmlscene_->deactivateLights (numlights);

/*
  // test: draw a bounding box around anchor objects
  if (scene_->linksActive ())
  { // color change in effect for further primitives
    ge3dLineColor (&scene_->col_anchoredge);
    ge3dWireCube (&omin_, &omax_);
  }
*/

  if (draw_hilit != oldhilit)  // restore non-anchor material
  {
    //cerr << "anchor: resetting draw_hilit to " << oldhilit << "; " << draw_curmtl << endl;
    draw_hilit = oldhilit;  // draw_hilnonanch unless nested anchors
    if (draw_curmtl)
      draw_curmtl->draw ();
  } // TODO: keep track of draw_curmtl in Separators (restore; including Anchors)

  QvNode::draw ();

} // QvWWWAnchor


void QvWWWInline::draw ()
{
  // fetch first time needed
  if (state_ == s_virgin)
  {
    DEBUGNL ("QvWWWInline first encountered. Sending request");
    state_ = s_requested;
    // the request of the inline URL is VRML specific, but was put into Scene3D to allow
    // separate treatment for different viewer types (Harmony, standalone, etc.)
    scene_->requestInlineURL (this, name.value.getString (), parentURL_.getString ());
  }

  // draw bounding box until scene fetched
  if (hasextent_ && state_ != s_completed)
  {
    ge3dWireCube (&omin_, &omax_);
  }

  QvGroup::draw ();  // draw children (if already fetched)
} // QvWWWInline


/***** misc *****/


DRAW(QvInfo)

void QvUnknownNode::draw ()
{ QvGroup::draw ();
}



/***** extensions *****/


DRAW(QvLabel)


void QvLightModel::draw ()
{
  if (scene_->dolighting () == Scene3D::lighting_auto)
    ge3dHint (hint_lighting, dolighting_);
}
