/*
 * ipc.h: A plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id$
 */


#ifndef __CON_IPC_H
#define __CON_IPC_H


#include <stdio.h>
#include <sys/types.h>

#include <vdr/thread.h>




// Interface that can be implemented,
// if one can wait (signal) for the object.


class IConsWaitable {
public:
  virtual ~IConsWaitable() {}

  virtual int SignalToWaitFor() const = 0;
};




// A List of objects we can wait for.


class cConsWaitableList {
private:
  IConsWaitable** _list;
  int  _listCount;

  fd_set _wait;

public:
  cConsWaitableList();
  ~cConsWaitableList();

  void Add(IConsWaitable* pObj);
  void Remove(IConsWaitable* pObj);
  bool Wait(int timeoutMs = 0);

  bool IsSignalled(IConsWaitable* pObj) { 
    if (pObj && pObj->SignalToWaitFor() >= 0)
      return FD_ISSET(pObj->SignalToWaitFor(), &_wait);
    return false;
  }

  bool IsSignalled(int fd) {
    return FD_ISSET(fd, &_wait);
  }
};






// This is the real pipe, but that name is taken from the vdr source.
// So I call this a simple pipe.


class cConsSimplePipe {
private:
  int _pipe[ 2 ];

public:
  cConsSimplePipe();
  virtual ~cConsSimplePipe();

  bool Open();
  void Close();

  bool IsOpen() { return _pipe[0] >= 0 && _pipe[1] >= 0; }

  int getReader() const { return _pipe[0]; }
  int getWriter() const { return _pipe[1]; }
};





// Can be used to signal an other thread.
// This could also be realized by using a semaphore.
// But by using a pipe one can wait for a signal with select.


class cConsSignal
: private cConsSimplePipe,
  public IConsWaitable
{
public:
  cConsSignal();
  virtual ~cConsSignal() {}

  void Signal();
  void Reset();

  bool IsSignalled();

//IConsWaitable
  virtual int SignalToWaitFor() const { return getReader(); }
};



#endif //__CON_IPC_H

