/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#ifndef UG_PATH_H
#define UG_PATH_H

#include <glib.h>

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Windows path looks like "drive:\folder\file_pri.file_ext"
 *
 * location is path exclude file
 * location_len = path_len - file_len
 *
 * file     = file_pri     + '.' + file_ext
 * file_len = file_pri_len +  1  + file_ext_len
 *
 * file_pri_len : length of primary filename
 * file_ext     : filename extension
 * file_ext_len : length of filename extension
 *
 * Example:
 *
 * drive            = strndup (PathPart->path, PathPart->path_drive_len);
 * filename         = strndup (PathPart->file, PathPart->file_len);
 * primary_filename = strndup (PathPart->file, PathPart->file_pri_len);
 *
 * if (PathPart->file_len == 0)
 *     printf ("No filename");
 */

#define UG_PATH_PART_MEMBERS			\
	const char*		path;				\
	unsigned int	path_len;			\
	unsigned int	path_drive_len;		\
	const char*		folder;				\
	unsigned int	folder_len;			\
	const char*		file;				\
	unsigned int	file_len;			\
	unsigned int	file_pri_len;		\
	const char*		file_ext;			\
	unsigned int	file_ext_len

typedef struct UgPathPart_		UgPathPart;

struct UgPathPart_ {
	UG_PATH_PART_MEMBERS;
//	const char*		path;
//	unsigned int	path_len;
//	unsigned int	path_drive_len;
//	const char*		folder;
//	unsigned int	folder_len;
//	const char*		file;
//	unsigned int	file_len;
//	unsigned int	file_pri_len;
//	const char*		file_ext;
//	unsigned int	file_ext_len;
};

/* UgPathPart macros */
#define		UG_PATH_PART_LOCATION_LEN(spart)				( (spart)->path_len - (spart)->file_len )
#define		ug_path_part_new(path, path_len)				ug_path_part_new_full (path, path_len, G_DIR_SEPARATOR)
#define		ug_path_part(spart, path, path_len)				ug_path_part_full (spart, path, path_len, G_DIR_SEPARATOR)

UgPathPart*	ug_path_part_new_full (const char* path, int path_len, const char folder_separator);
void		ug_path_part_free (UgPathPart* spart);

/* Return TRUE if path had drive or folder, otherwise return FALSE */
/* If path_len == -1, path is null-terminated. */
gboolean	ug_path_part_full (UgPathPart* spart, const char* path, int path_len, const char folder_separator);

/* Other utilities for path
 * These functions can handle special case like "\\\\folder1\\\\folder2" or "//folder1//folder2"
 * If path_len == -1, path is null-terminated.
 *
 * Example:
 *	char* path = "/folder/element/filename";
 *  int   path_len = strlen (path)
 *
 *	unsigned int element_length, element_offset;
 *
 *  element_offset = 0;
 *	while (element_offset < path_len) {
 *		element_offset = ug_path_element (path, path_len, element_offset, &element_length);
 *		printf ("folder : %.*s\n", element_length, path + element_offset);
 *		element_offset += element_length;
 *	}
 */
unsigned int	ug_path_element_full (const char* path, int path_len, unsigned int offset, unsigned int* len, const char folder_separator);

#define		ug_path_element(path, path_len, offset, length_ptr)		ug_path_element_full (path, path_len, offset, length_ptr, G_DIR_SEPARATOR)


#ifdef __cplusplus
}
#endif

#endif  /* UG_PATH_H */

