package tijmp.ui;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryType;
import java.lang.management.MemoryUsage;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;
import tijmp.ProfilerHandler;

/** A class to show current memory usage. 
 */
public class MemoryInfoPanel extends JPanel {

    public MemoryInfoPanel (ProfilerHandler ph) {
	MemoryMXBean mbean = ph.getMemoryMXBean ();
	MemoryUsage heapUsage = mbean.getHeapMemoryUsage ();
	MemoryUsage nonHeapUsage = mbean.getNonHeapMemoryUsage ();

	List<MemInfo> ls1 = 
	    Arrays.asList (new MemInfo ("Heap", MemoryType.HEAP, heapUsage),
			   new MemInfo ("Non heap", MemoryType.NON_HEAP, 
					nonHeapUsage));
	

	TM tm1 = new TM (ls1);
	JTable t1 = new JTable (tm1);
	fixTable (t1);
	JScrollPane sp1 = new JScrollPane (t1);
	sp1.setBorder (BorderFactory.createTitledBorder ("Memory Overview"));
	
	List<MemoryPoolMXBean> ls = ph.getMemoryPoolMXBeans ();
	List<MemInfo> ls2 = new ArrayList<MemInfo> (ls.size ());
	for (MemoryPoolMXBean mpb : ls)
	    ls2.add (new MemInfo (mpb.getName (), mpb.getType (), mpb.getUsage ()));
	TM tm2 = new TM (ls2);
	JTable t2 = new JTable (tm2);
	fixTable (t2);
	t2.getRowSorter ().toggleSortOrder (TM.COL_COMMITED);
	t2.getRowSorter ().toggleSortOrder (TM.COL_COMMITED);
	JScrollPane sp2 = new JScrollPane (t2);
	sp2.setBorder (BorderFactory.createTitledBorder ("Memory Pools"));
	
	int finilizationCount = mbean.getObjectPendingFinalizationCount ();
	JLabel fc = new JLabel ("Finilization count: " + finilizationCount);
	
	GridBagLayout gb = new GridBagLayout ();
	GridBagConstraints c = new GridBagConstraints ();
	setLayout (gb);

	c.insets = new Insets (2, 2, 2, 2);
	c.gridx = 0; 
	c.gridy = 0;
	c.weightx = 1;
	c.weighty = 3;
	c.fill = GridBagConstraints.BOTH;
	add (sp1, c);
	c.gridy++;
	c.weighty = 6;
	add (sp2, c);
	c.weighty = 0;
	c.gridy++;
	add (fc, c);
    }

    private void fixTable (JTable table) {
	table.setAutoResizeMode (JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);
	table.setSelectionMode (ListSelectionModel.SINGLE_SELECTION);
	table.setAutoCreateRowSorter (true);
	TableColumn column = null;
	for (int i = 0; i < table.getModel ().getColumnCount (); i++) {
	    column = table.getColumnModel ().getColumn (i);
	    if (i == TM.COL_NAME)
		column.setPreferredWidth (200);
	    else if (i == TM.COL_TYPE)
		column.setPreferredWidth (25);
	    else
		column.setPreferredWidth (75);
	}
	int rows = table.getModel ().getRowCount () + 1; // header 
        table.setPreferredScrollableViewportSize (new Dimension (600, 15 * rows));
	table.setDefaultRenderer (Long.class, new ReadableSizeRenderer ());
    }
}

class MemInfo {
    private String name;
    private MemoryType mt;
    private MemoryUsage mu;

    public MemInfo (String name, MemoryType mt, MemoryUsage mu) {
	this.name = name;
	this.mt = mt;
	this.mu = mu;
    }
    
    public String getName () {
	return name;
    }

    public MemoryType getType () {
	return mt;
    }

    public MemoryUsage getUsage () {
	return mu;
    }
}

class TM extends AbstractTableModel {
    private List<MemInfo> ls;
    
    private static final String[] columnNames = {"Name", "Heap", "Init", 
						 "Used", "Commited", "Max"};

    public static final int COL_NAME = 0;
    public static final int COL_TYPE = COL_NAME + 1;
    public static final int COL_INIT = COL_TYPE + 1;
    public static final int COL_USED = COL_INIT + 1;
    public static final int COL_COMMITED = COL_USED + 1;
    public static final int COL_MAX = COL_COMMITED + 1;

    public TM (List<MemInfo> ls) {
	this.ls = ls;
    }

    @Override public String getColumnName (int col) {
	return columnNames[col];
    }

    public int getRowCount () { 
	return ls.size ();
    }
    
    public int getColumnCount () { 
	return columnNames.length; 
    }

    public Object getValueAt (int row, int col) {
	MemInfo mi = ls.get (row);
	MemoryUsage m = mi.getUsage ();
	switch (col) {
	case COL_NAME:
	    return mi.getName ();
	case COL_TYPE:
	    return mi.getType () == MemoryType.HEAP;
	case COL_INIT:
	    return m.getInit ();
	case COL_USED:
	    return m.getUsed ();
	case COL_COMMITED:
	    return m.getCommitted ();
	case COL_MAX:
	    return m.getMax ();
	}
	throw new IllegalStateException ("unknow column");
    }

    @Override public Class<?> getColumnClass (int col) {
	switch (col) {
	case COL_NAME:
	    return String.class;
	case COL_TYPE:
	    return Boolean.class;
	default:
	    return Long.class;
	}
    }
    
    @Override public boolean isCellEditable (int row, int col) { 
	return false; 
    }
    
    @Override public void setValueAt (Object value, int row, int col) {
	throw new IllegalStateException ("non editable table");
    }	
}
