#include <string.h>
#include "mr_pcre/mr_pcre.h"
#include "mr_curl/mr_curl.h"

using namespace std;

size_t Curl_write_data(const void *ptr, size_t size,
		size_t nmemb, Curl *curl);
size_t Curl_write_header(const void *ptr, size_t size,
		size_t nmemb, Curl *curl);

void FormParser::SetPage(const std::string &new_html)
{
	action = "";
	elems.clear();
	html = new_html; 
}

bool FormParser::ReadForm(const string &name)
{
	bool ret = false;
	PcreWrap::Array forms;
	elems.clear();

	// szukamy wlasciwego formularza w calym htmlu
	PcreWrap::pcre_match_all("<form(.*?)<\\/form>", html, "sim", forms);
	for(size_t i=0; i<forms.size(); i++) {
		// mamy jakis formularz, teraz patrzymy czy name
		// wskazuje na szukany
		
		// wybieramy sam poczatek bloku - do pierwszego znaku >
		string form_b = forms[i][0];
		string form_t = form_b.substr(0, form_b.find(">"));

		if (!PcreWrap::pcre_match("name=\"" + name + "\"", form_b, "im")) continue;

		// ok, to jest juz ten formularz - teraz wybieramy z niego
		// wszystkie dane
		action = extract("action", form_t);

		readFields(form_b);
		ret = true;
		break;
	}
	return ret;
}

// z 'ciala' formularza wyciagnie same pary nazwa->wartosc pobrane
// z jego pol typu 'input'
void FormParser::readFields(const string &form)
{
	PcreWrap::Array inputs;

	PcreWrap::pcre_match_all("<input(.*?)>", form, "im", inputs);
	for(size_t i=0; i < inputs.size(); i++) {
		string name = extract("name", inputs[i][0]);
		string value = extract("value", inputs[i][0]);
		string type = extract("type", inputs[i][0]);

		// elementy typu image dodaja dodatkowe pola ze wspolrzedna kliknienia na obrazku
		if (name != "") {
			elems[name] = value;
			if (type == "image") {
				elems[name + ".x"] = "0";
				elems[name + ".y"] = "0";
			}
		} else if (type == "image") {
			elems["x"] = "0";
			elems["y"] = "0";
		}
	}
}

// majac string w postaci ....name="xxxx".... wyciagnie same xxx
string FormParser::extract(const string &name, const string &buf)
{
	PcreWrap::Array match;

	if (PcreWrap::pcre_match_all(name+"=\"(.*?)\"", buf, "mi", match)) {
		return match[0][0];
	} 
	return "";
}

std::string &FormParser::operator[](const std::string &name)
{
	return elems[name];
}

ostream &operator<<(ostream &os, const FormParser &form)
{
	const FormParser::Array &elems = form.getElems();
	FormParser::Array::const_iterator it;
	os << "<action> -> '" << form.action << "'" << endl;
	for(it = elems.begin(); it != elems.end(); it++)
		os << it->first << " -> '" << it->second << "'" << endl;
	return os;
}

void FormParser::set(const string &name, long val)
{
	C_STREAM conv;
	conv << val;
	string str;
	conv >> str;
	elems[name] = str;
}

///////////////////////

Curl_Form::Curl_Form(const FormParser &form) : post(NULL), last(NULL)
{
	CopyFrom(form);
}

void Curl_Form::CopyFrom(const FormParser &form)
{
	Reset();
	const FormParser::Array &elems = form.getElems();
	FormParser::Array::const_iterator it;
	for(it = elems.begin(); it != elems.end(); it++) {
		AddField(it->first, it->second);
	}
}

Curl_Form &Curl_Form::operator=(const FormParser &form)
{
	CopyFrom(form);
	return *this;
}

Curl_Form::~Curl_Form()
{
	if (post) curl_formfree(post);
}

void Curl_Form::Reset()
{
	if (post) curl_formfree(post);
	post = last = NULL;
}

string Curl_Form::PostFields()
{
	struct curl_httppost *item = post;
	string ret = "", name, content;
	char *ptr;
	
	while (item != NULL) {
		ptr = curl_escape(item->name, 0);
		name = ptr;
		curl_free(ptr);
		
		ptr = curl_escape(item->contents, 0);
		content = ptr;
		curl_free(ptr);
		
		if (ret != "") ret += "&";
		ret += name + "=" + content;
		
		item = item->next;
	}
	
	return ret; 
}

// dodaje pole typu 'input' - kopiuje wartosci do pamieci biblioteki
void Curl_Form::AddField(const std::string &name, const std::string &value)
{
	curl_formadd(&post, &last,
		CURLFORM_COPYNAME, name.c_str(),
		CURLFORM_COPYCONTENTS, value.c_str(),
		CURLFORM_END);
}

void Curl_Form::AddField(const std::string &name, long value)
{
	C_STREAM conv;
	conv << value;
	std::string str;
	conv >> str;
	AddField(name, str);
}

// dodaje pole typu 'input' z binarna zwartoscia (bufor z podanym rozmiarem),
void Curl_Form::AddBuffer(const std::string &name, const void *buffer, long buffer_len)
{
	curl_formadd(&post, &last,
		CURLFORM_COPYNAME, name.c_str(),
		CURLFORM_COPYCONTENTS, buffer,
		CURLFORM_CONTENTSLENGTH, buffer_len,
		CURLFORM_END);
}

// dodaje pole uploadu pliku (pobiera plik z dysku)
void Curl_Form::AddFile(const std::string &name, const std::string &filename)
{
	curl_formadd(&post, &last,
		CURLFORM_COPYNAME, name.c_str(),
		CURLFORM_FILE, filename.c_str(),
		CURLFORM_END);
}



///////////////////////////


void Curl_Result::SetHeaders(std::istream &is)
{
	const int line_len = 8192; 
	char *c = new char[line_len+1];
	std::string line;
	
	is.getline(c, line_len);
	result = c;

	is.getline(c, line_len);
	line = c;
	while (line[0] != '\0' && line[0] != '\r' && line[0] != '\n') {
		int colon = line.find(':');
		if (colon != -1) {
			std::string name = line.substr(0, colon),
				val = line.substr(colon+2); 
			headers[name] = val;
			if (name == "Set-Cookie") {
				std::string cookie = val.substr(0, val.find(";"));
				//cookies
			}
		}
		is.getline(c, line_len);
		line = c;
	}
	delete[] c;
}

Curl_Result::Curl_Result(const Curl_Result &cr)
{
	body = cr.body;
	result = cr.result;
	headers = cr.headers;
	cookies = cr.cookies;
}

Curl_Result &Curl_Result::operator=(const Curl_Result &cr)
{
	body = cr.body;
	headers = cr.headers;
	result = cr.result;
	cookies = cr.cookies;
	return *this;
}


////////////////////////////////

Curl::Curl()
{
	curl = curl_easy_init();
	buffer = headers = NULL;
	slist = NULL;
	error_buffer = new char[CURL_ERROR_SIZE + 1];
	*error_buffer = (char)0;
}

Curl::~Curl()
{
	FreeExtraHeaders();
	curl_easy_cleanup(curl);
	delete buffer;
	delete headers;
	delete[] error_buffer;
}

CURLcode Curl::SetOpt(CURLoption option, const void *param)
{
	return error_code = curl_easy_setopt(curl, option, param);
}

CURLcode Curl::SetPost(Curl_Form *form)
{
	return error_code = curl_easy_setopt(curl, CURLOPT_HTTPPOST, form->GetPost());
}

CURLcode Curl::PerformPost(const std::string &url, const std::string &postfields, int size)
{
	this->url = url;
	this->postfields = postfields;
	if (!size) size = this->postfields.length();
	SetOpt(CURLOPT_POST, (void*)1);
	SetOpt(CURLOPT_POSTFIELDS, this->postfields.c_str());
	SetOpt(CURLOPT_POSTFIELDSIZE, (void*)size);
	SetOpt(CURLOPT_URL, this->url.c_str());
	return Perform();
}

CURLcode Curl::PerformPost(const std::string &url, Curl_Form *form)
{
	this->url = url;
	SetOpt(CURLOPT_HTTPPOST, form->GetPost());
	SetOpt(CURLOPT_URL, this->url.c_str());
	return Perform();
}

CURLcode Curl::PerformPost(const std::string &url, Curl_Form &form)
{
	this->url = url;
	SetOpt(CURLOPT_HTTPPOST, form.GetPost());
	SetOpt(CURLOPT_URL, this->url.c_str());
	return Perform();
}

CURLcode Curl::PerformGet(const std::string &url)
{
	this->url = url;
	SetOpt(CURLOPT_HTTPGET, (void*)1);
	SetOpt(CURLOPT_URL, this->url.c_str());
	return Perform();
}

CURLcode Curl::Perform()
{
	delete buffer; delete headers;
	buffer = new C_STREAM;
	headers = new C_STREAM;
	
	SetOpt(CURLOPT_WRITEFUNCTION, (void*)Curl_write_data);
	SetOpt(CURLOPT_WRITEDATA, this);

	SetOpt(CURLOPT_HEADERFUNCTION, (void*)Curl_write_header);
	SetOpt(CURLOPT_WRITEHEADER, this);
	
	*error_buffer = (char)0;
	SetOpt(CURLOPT_ERRORBUFFER, (void*)error_buffer);
	
	if (slist) SetOpt(CURLOPT_HTTPHEADER, (void*)slist);
	
	error_code = curl_easy_perform(curl);
	*buffer << '\0';
	*headers << '\0';
	return error_code;
}

std::string Curl::Body()
{
	std::string body = buffer->str();
#ifdef _mr_use_stringstream
	if (body.length() > 0) body = body.substr(0, body.length() - 1);
#else
	buffer->rdbuf()->freeze(0);
#endif
	return body;
}

std::string Curl::Headers()
{
	std::string header = headers->str();
#ifdef _mr_use_stringstream
	if (header.length() > 0) header = header.substr(0, header.length() -1);
#else
	headers->rdbuf()->freeze(0);
#endif
	return header;
}

bool Curl::AppendExtraHeader(const std::string &header)
{
	struct curl_slist *new_slist;
	new_slist = curl_slist_append(slist, header.c_str());
	if (new_slist) slist = new_slist;
	return new_slist != NULL;
}

void Curl::FreeExtraHeaders()
{
	if (slist) curl_slist_free_all(slist);
	slist = NULL;
}

Curl_Result Curl::Result()
{
	Curl_Result r;
	r.SetBody(Body());
	r.SetHeaders(HeadersStream());
	return r;
}

const char* Curl::ErrorStr()
{
	return error_buffer;
}

void Curl::Verbose(bool verbose)
{
	SetOpt(CURLOPT_VERBOSE, (void*)verbose);
}

void Curl::AutoReferer(bool referer)
{
	SetOpt(CURLOPT_AUTOREFERER, (void*)referer);
}

void Curl::FollowLocation(bool follow, long max_redirs)
{
	SetOpt(CURLOPT_FOLLOWLOCATION, (void*)follow);
	if (follow) SetOpt(CURLOPT_MAXREDIRS, (void*)max_redirs);
}

void Curl::UserAgent(const char *ua)
{
	useragent = ua;
	SetOpt(CURLOPT_USERAGENT, useragent.c_str());
}

void Curl::Proxy(const char *host, int port, const char *login, const char *pass, int auth_type)
{
	proxy_host = host;
	proxy_port = port;
	proxy_login = (login && strlen(login)) ? login : "";
	proxy_pass = (pass && strlen(pass)) ? pass : "";
	
	// usuwamy ew. http:// z poczatku proxy_host
	if (proxy_host.substr(0, 7) == "http://") proxy_host.erase(0, 7);

	SetOpt(CURLOPT_PROXY, proxy_host.c_str());
	SetOpt(CURLOPT_PROXYPORT, (const void*)proxy_port);
	if (proxy_login != "" && auth_type != 0) {
		proxy_authstr = proxy_login + ":" + proxy_pass;
		SetOpt(CURLOPT_PROXYUSERPWD, proxy_authstr.c_str());
		SetOpt(CURLOPT_PROXYAUTH, (void*)auth_type);
	}
}

void Curl::Proxy(const std::string &proxy, const char *login, const char *pass, int auth_type)
{
	int sep;
	std::string tmp_proxy = proxy;
	
	if (tmp_proxy.substr(0, 7) == "http://") tmp_proxy.erase(0, 7);
	sep = tmp_proxy.find(":");
	if (sep != -1) {
		std::string host = tmp_proxy.substr(0, sep);
		std::string port = tmp_proxy.substr(sep+1);
		Proxy(host.c_str(), atoi(port.c_str()), login, pass, auth_type);
	}
}

void Curl::SSL_ignore()
{
	SetOpt(CURLOPT_SSL_VERIFYPEER, 0);
	SetOpt(CURLOPT_SSL_VERIFYHOST, 0);
}

const char *Curl::Version()
{
	return curl_version();
}

size_t Curl_write_data(const void *ptr, size_t size,
		size_t nmemb, Curl *curl) {
	curl->buffer->write((const char*)ptr, size * nmemb);
	return curl->buffer->bad() ? 0 : size*nmemb;
}

size_t Curl_write_header(const void *ptr, size_t size,
		size_t nmemb, Curl *curl) {
	curl->headers->write((const char*)ptr, size * nmemb);
	return curl->headers->bad() ? 0 : size*nmemb;
}

