/*
 * rawmultiply.cc -- Multiplies the Raw DV input by a factor (frame duplication)
 * Copyright (C) 2002 Raphael Amaury Jacquot <sxpert@esitcom.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <iostream>
#include <string>
#include <cmath>
using std::string;
using std::cerr;
using std::cin;
using std::endl;

#include <preferences.h>
#include <Diagnostics.h>
#include <stdio.h>
#include <DVPump.h>
#include <RawDVFileInput.h>
#include <frame.h>
#include <Threader.h>
#include <DVEncoder.h>
#include <unistd.h>

class Multiplier
{
	public:
		double number_copies;
		long extent;
		double delta;

		Multiplier( double number_copies )
		{
			this->number_copies = number_copies;
			extent = 1;
			if ( number_copies )
				delta = 1 / number_copies;
			else
				throw "Count is zero - will produce no output";			
		}
		
		Multiplier( double number_copies, long extent )
		{
			this->number_copies = number_copies;
			this->extent = extent;
			if ( number_copies )
				delta = 1 / number_copies;
			else
				throw "Count is zero - will produce no output";	
		}
		
};

class RawMultiply : public Threader, public DVEncoder
{
	private:
		int obtained;
		int defined;
		Multiplier *array[ 100 ];
		double current;
		int pump_size;
		bool auto_flush;

	public:
		RawMultiply( ) : 
			obtained( 0 ),
			defined( 0 ),
			current( 0 ), 
			pump_size( 25 ), 
			auto_flush( false )
		{
		}

		virtual ~RawMultiply( )
		{
		}

		virtual string LogId( )
		{
			return "RawMultiply";
		}

		void SetNumberCopies( char* _number_copies) 
		{
			char* ptr;
			if ( strchr( _number_copies, ':' ) == NULL )
			{
				double number_copies = strtod( _number_copies, &ptr );
				if (! ( ( _number_copies[0]!='\0') && ( ptr[0] == '\0' ) ) ) 
					throw "This is not a valid number";
				array[ defined ++ ] = new Multiplier( number_copies );
			}
			else
			{
				char *ptr2;
				int extent = strtol( _number_copies, &ptr, 10 );
				double number_copies = strtod( ptr + 1, &ptr2 );
				if ( ptr2[0] != '\0' ) 
					throw "This is not a valid extent specification";
				array[ defined ++ ] = new Multiplier( number_copies, extent );
			}
		}

		void SetPumpSize( int _pump_size ) 
		{
			pump_size = _pump_size;
		}
		
		void SetAutoFlush( bool _auto_flush )
		{
			auto_flush = _auto_flush;
		}
		
	protected:

		bool OutputFrame( Frame &frame )
		{
			fflush( stdout );
			if ( GetAudioFile( ) != "" )
			{
				// Work around for design flaw in libdv
				SetPAL( frame.IsPAL( ) );
				EncodeAudio( frame );
			}
			else
			{
				EncodeMetaData( frame );
			}
			return fwrite( frame.data, frame.GetFrameSize( ), 1, stdout ) != 0;
		}

		Multiplier &GetMultiplierFor( int frame )
		{
			int total = 0;
			int i;
			
			for( i = 0; i < defined; i ++ )
			{
				if ( array[ i ]->extent != -1 )
					total += array[ i ]->extent;
				else
					break;
			}
			
			if ( i == defined || frame < total )
			{
				int offset = frame % total;
				total = 0;
				for ( i = 0; i < defined; i ++ )
				{
					if ( array[ i ]->extent != -1 )
						total += array[ i ]->extent;
					else
						break;
					if ( offset < total )
						break;
				}
			}
			
			return *array[ i ];
		}
		
		void Thread( )
		{
			bool running = true;

			RawDVFileInput input;
			input.SetPumpSize( pump_size );
			input.SetFile( stdin );
			input.SetAutoFlush( auto_flush );

			input.ThreadStart( );

			current = 0;

			if ( defined == 0 )
				array[ defined ++ ] = new Multiplier( 1 );

			if ( array[ 0 ]->number_copies < 1 )
				current = array[ 0 ]->delta;

			SetAudioLoop( false );
			
			while( running && ThreadIsRunning( ) )
			{
				if ( input.GetOutputAvailable( ) > 0 )
				{
					Frame &frame = input.GetOutputFrame( );
					Multiplier &multiplier = GetMultiplierFor( obtained ++ );
					
					if ( multiplier.number_copies <= 1 )
					{
						current += multiplier.number_copies;
						if ( current >= 1 )
						{
							running = OutputFrame( frame );
							current = current - 1;
						}
					}
					else if ( multiplier.number_copies > 1 )
					{
						double target = ceil( current );
						if ( target == current )
							target ++;
						for ( ; running && current <= target; current += multiplier.delta ) 
							running = OutputFrame( frame );
					}

					input.QueueInputFrame( );
				}
				else
				{
					running = input.ThreadIsRunning( ) && input.PumpIsNotCleared( );
				}
			}

			input.ThreadStop( );
		}
};

static void Usage( )
{
	cerr << "Usage: rawmultiply [ options ] value" << endl;
	cerr << "Where: multiplication factor (any floating point > 0 - default 1)" << endl;
	cerr << "       -A wav   - wav file to dub (default silence)" << endl;
	cerr << "       -b size  - buffer size in frames (default: 25)" << endl;
	cerr << "       -f       - fix audio samples per frame" << endl;
	cerr << "       -a       - auto flush" << endl;
	exit( 0 );
}

int main( int argc, char **argv )
{
	RawMultiply output;

	if ( isatty( fileno( stdin ) ) )
	{
		cerr << "Input must must be obtained from a pipe or a file." << endl;
		Usage( );
	}

	if ( isatty( fileno( stdout ) ) )
	{
		cerr << "Output be must redirected to a pipe or a file." << endl;
		Usage( );
	}

	Diagnostics::SetApp( "rawmultiply" );

	try
	{
		for ( int i = 1; i < argc; i ++ )
		{
			if ( !strcmp( argv[ i ], "-A" ) )
				output.SetAudioFile( string( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-f" ) )
				output.SetFixedAudioSamples( true );
			else if ( !strcmp( argv[ i ], "-a" ) )
				output.SetAutoFlush( true );
			else if ( !strcmp( argv[ i ], "-b" ) )
				output.SetPumpSize( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-v" ) )
				Diagnostics::SetLevel( atoi( argv[ ++ i ] ) );
			else if ( strcmp( argv[ i ], "--help" ) )
				output.SetNumberCopies( argv[ i ] );
			else
				Usage( );
		}
		
		output.ThreadExecute( );
	}
	catch ( string exc )
	{
		cerr << "Exception thrown: " << exc << endl;
	}

	return 0;
}
