#include <config.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <unistd.h>
#include <string.h>
#include <locale.h>
#include <signal.h>
#include "list.h"
#include "common.h"
#include "buffer.h"
#include "charset.h"
#include "auth.h"
#include "smbctx.h"
#include "smbitem.h"
#include "reconfigure.h"
#include "function.h"
#include <fuse/fuse.h>

extern	FILE	*stdlog;
char		*mountpoint = NULL;

void sig_handler(int signum){
    fprintf(stdlog != NULL ? stdlog : stderr,
	"%d->%s: SIGILL or SIGSEGV received\n",
	getpid(), __FUNCTION__);
    print_backtrace();
    if (mountpoint != NULL) fuse_unmount(mountpoint);
    exit(signum);
}

void parse_cmdline(int argc, char *argv[]){
    int		c;
    int		print_version = 0;
    int		get_help = 0;
    
    opterr = 0;
    while(1){
	static struct option	long_options[] = {
	    {"help",		no_argument, NULL, 'h'},
	    {"version",		no_argument, NULL, 'V'},
	    {NULL, 0, NULL, 0},
	};
	
	c = getopt_long(argc, argv, "-hV", long_options, NULL);
	if (c == -1) break;
	
	switch(c){
	    case 1:
		if (mountpoint == NULL) mountpoint = optarg; 
		break;
	    case 'h':
		get_help = 1;
		break;
	    case 'V':
		print_version = 1;
		break;
	    default:
		break;
	}
    }
    optarg = NULL; optopt = '?'; optind = opterr = 1;

    if (get_help || print_version){
	if (print_version)
	    printf("%s version: %s\n", PACKAGE_NAME, PACKAGE_VERSION);
	fuse_main(argc, argv, &smb_oper);
	exit(0);
    }
}

int main(int argc, char *argv[]){

    int 			fd;
    struct sigaction		siginfo;
    struct fuse_file_info	fi;

    siginfo.sa_handler = sig_handler;
    sigemptyset(&siginfo.sa_mask);
    siginfo.sa_flags = SA_RESTART;

    if (sigaction(SIGILL, &siginfo, NULL) < 0){
	fprintf(stderr, "Can't set SIGILL handler\n");
	return 1;
    }

    if (sigaction(SIGSEGV, &siginfo, NULL) < 0){
	fprintf(stderr, "Can't set SIGSEGV handler\n");
	return 1;
    }

    sigemptyset(&siginfo.sa_mask);
    sigaddset(&siginfo.sa_mask, SIGHUP);
    sigaddset(&siginfo.sa_mask, SIGALRM);
    if (pthread_sigmask(SIG_BLOCK, &siginfo.sa_mask, NULL) != 0){
	fprintf(stderr, "Can't block SIGHUP and SIGALRM signals.\n");
	return 1;
    }
    
    setlocale(LC_ALL, "");
    parse_cmdline(argc, argv);
    if (!charset_init()){
	fprintf(stderr, "Fatal error: iconv initialization failed.\n");
	exit(1);
    }

    GetUserLogin();
    GetConfigDir();
    
    if (sizeof(fi.fh) < sizeof(char*)){
	fprintf(stderr, "SMBNetFS fatal error:\n"
	    "  There is no space to store pointer in fuse_file_info struct.\n");
	return 1;
    }
    
    if ((fd = open(".", O_RDONLY)) == -1)
	fprintf(stderr, "Can't remember current directory, error : %s\n"
	    "Current directory will be changed to read config files.\n"
	    "Please use full path to avoid mounting problem next time.\n", 
	    strerror(errno));
    ReadConfig();
    if (fd != -1){
        if (fchdir(fd) == -1)
	    fprintf(stderr, 
		"Current directory was changed to read config files.\n"
		"Please use full path to avoid mounting problem next time.\n");
	close(fd);
    }
    
    AllocateBuffers(&ArrayData);
    AllocateBuffers(&ConvertBuffer);
    AllocateBuffers(&OpenFiles);
    AllocateSmbCtx();

    int ret = fuse_main(argc, argv, &smb_oper);
    if (GetFastShutdown()) return ret;
    
    DeleteOldSmbItem(time(NULL) + 10, USERPATH | AUTOPATH);
    DestroyUnusedCtx();
    DestroyBufferList(&OpenFiles);
    DestroyBufferList(&ConvertBuffer);
    DestroyBufferList(&ArrayData);
    DeleteOldSmbAuthData(time(NULL) + 10);

    return ret;
}
