//plugins.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2012
 *
 *  This file is part of roard a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "roard.h"

#define MAX_PLUGINS    8

static struct _roard_plugin {
 struct roar_dl_lhandle     * lhandle;
 struct roard_plugins_sched * sched;
 int protocols[MAX_PROTOS];
} g_plugins[MAX_PLUGINS];
static struct _roard_plugin * _pp = NULL;

static struct _roard_plugin * _find_free(void) {
 int i;

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( g_plugins[i].lhandle == NULL ) {
   memset(&(g_plugins[i]), 0, sizeof(struct _roard_plugin));
   return &(g_plugins[i]);
  }
 }

 return NULL;
}

int plugins_preinit  (void) {
 memset(g_plugins, 0, sizeof(g_plugins));

 return 0;
}

static void inline plugins_delete(struct _roard_plugin * plugin) {
 int i;

 if ( plugin->sched != NULL ) {
  if ( plugin->sched->free != NULL ) {
   roar_dl_context_restore(plugin->lhandle);
   plugin->sched->free();
   roar_dl_context_store(plugin->lhandle);
  }
 }

 roar_dl_appsched_trigger(plugin->lhandle, ROAR_DL_APPSCHED_FREE);

 for (i = 0; i < MAX_PROTOS; i++) {
  if ( plugin->protocols[i] != -1 ) {
   clients_unregister_proto(plugin->protocols[i]);
  }
 }

 roar_dl_close(plugin->lhandle);
 memset(plugin, 0, sizeof(struct _roard_plugin));
 plugin->lhandle = NULL;
}

int plugins_init  (void) {
 int i;

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( g_plugins[i].lhandle != NULL ) {
   _pp = &(g_plugins[i]);

   _pp->sched = NULL;

   if ( roar_dl_ra_init(g_plugins[i].lhandle, NULL, NULL) == -1 ) {
    ROAR_WARN("plugins_init(void): Can not RA init lib at %p: %s", g_plugins[i].lhandle, roar_error2str(roar_error));
    plugins_delete(&(g_plugins[i]));
    continue;
   }

   if ( g_plugins[i].sched != NULL ) {
    if ( g_plugins[i].sched->init != NULL ) {
     roar_dl_context_restore(g_plugins[i].lhandle);
     g_plugins[i].sched->init();
     roar_dl_context_store(g_plugins[i].lhandle);
    }
   }

   roar_dl_appsched_trigger(g_plugins[i].lhandle, ROAR_DL_APPSCHED_INIT);

   _pp = NULL;
  }
 }

 return 0;
}

int plugins_free  (void) {
 int i;

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( g_plugins[i].lhandle != NULL ) {
   plugins_delete(&(g_plugins[i]));
  }
 }

 return plugins_preinit();
}

int plugins_update   (void) {
 int ret = 0;
 int i;

 for (i = 0; i < MAX_PLUGINS; i++) {
  if ( g_plugins[i].lhandle != NULL ) {
   if ( g_plugins[i].sched != NULL ) {
    if ( g_plugins[i].sched->update != NULL ) {
     roar_dl_context_restore(g_plugins[i].lhandle);
     if ( g_plugins[i].sched->update() == -1 )
      ret = -1;
     roar_dl_context_store(g_plugins[i].lhandle);
    }
   }
   if ( roar_dl_appsched_trigger(g_plugins[i].lhandle, ROAR_DL_APPSCHED_UPDATE) == -1 )
    if ( roar_error != ROAR_ERROR_NOENT )
     ret = -1;
  }
 }

 return ret;
}

int plugins_load  (const char * filename, const char * args) {
 struct _roard_plugin * next = _find_free();
 struct roar_dl_librarypara * para;
 int i;

 if ( next == NULL )
  return -1;

 for (i = 0; i < MAX_PROTOS; i++)
  next->protocols[i] = -1;

 if ( (para = roar_dl_para_new(args, NULL, ROARD_DL_APPNAME, ROARD_DL_ABIVERSION)) == NULL ) {
  ROAR_WARN("Can not load plugin (allocate para set): %s: %s", filename, roar_error2str(roar_error));
  return -1;
 }

 next->lhandle = roar_dl_open(filename, ROAR_DL_FLAG_DEFAULTS, 0 /* we delay this until plugins_init() */, para);
 roar_dl_para_unref(para);

 if ( next->lhandle == NULL ) {
  ROAR_ERR("plugins_load(filename='%s'): can not load plugin: %s", filename, roar_dl_errstr(NULL));
  return -1;
 }

 return 0;
}

int plugins_reg_sched(struct roard_plugins_sched * sched) {
 if ( _pp == NULL )
  return -1;

 _pp->sched = sched;

 return 0;
}

int plugins_reg_proto(struct roard_proto         * proto) {
 int i;

 if ( _pp == NULL )
  return -1;

 for (i = 0; i < MAX_PROTOS; i++) {
  if ( _pp->protocols[i] == -1 ) {
   _pp->protocols[i] = proto->proto;
   break;
  }
 }

 if ( i == MAX_PROTOS ) {
  roar_err_set(ROAR_ERROR_NOMEM);
  return -1;
 }

 return clients_register_proto(proto, _pp->lhandle);
}

//ll
