#
#  Ordinal: A Library of Ordinal Models
#  Copyright (C) 1998, 1999, 2000, 2001 P.J. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public Licence as published by
#  the Free Software Foundation; either version 2 of the Licence, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public Licence for more details.
#
#  You should have received a copy of the GNU General Public Licence
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#  print.lcr(z,digits=max(3,.Options$digits-3),print.gap=2)
#  print.kalordinal(z,digits=max(3,.Options$digits-3))
#  print.moprofile(z,nind=NULL,digits=max(3,.Options$digits-3),
#                  print.gap=2)
#  print.listmatrix(x,row.names=NULL,digits=max(3,.Options$digits-3),
#                   print.gap=2)
#
#  DESCRIPTION
#
#    Printing functions for linear categorical regression models,
#  marginal ordinal time profiles, and lists of matrices.
#

print.lcr <- function(z,digits=max(3,.Options$digits-3),print.gap=2) {
  links <- c("logit","probit","loglog","cloglog","uniform","log-normal","exponential","Pareto","Cauchy","Laplace","Levy","simplex",
             "gamma","Weibull","inverse Gauss","t","chi-square","generalized logistic","generalized extreme value","Box-Cox",
             "power exponential","Burr","Hjorth","beta","stable","generalized gamma","generalized Weibull","generalized inverse Gauss",
             "F","non-central t","non-central chi-square","Tukey","non-central beta","non-central F")
  cat("\nCall:",deparse(z$call),"",sep="\n")
  if(z$individual) {
    cat("Individual data.\n")
    cat("Total number of individuals: ",z$events,"\n",sep="")
    cat("Number of observations:      ",z$n,"\n\n",sep="")
  }
  else {
    cat("Frequency table.\n")
    cat("Number of non-empty cells: ",z$n,"\n",sep="")
    cat("Total number of events:    ",z$events,"\n\n",sep="")
  }
  switch(z$distribution,
         "binary" = {
           cat("Binary distribution.\n")
           cat("Transformation: ",z$transformation,".\n",sep="")
           cat("Link:           ",links[z$link],".\n",sep="")
         },
         "multinomial" = {
           cat("Multinomial distribution.\n")
           cat("Transformation: ",z$transformation,".\n",sep="")
         },
         "simplified multinomial" = {
           cat("Simplified multinomial distribution.\n")
           cat("Transformation: ",z$transformation,".\n",sep="")
         },
         "continuation-ratio" = switch(z$direction,
           "upwards" = {
             cat("Continuation-ratio distribution (upwards).\n")
             cat("Transformation: ",z$transformation,".\n",sep="")
           },
           "downwards" = {
             cat("Continuation-ratio distribution (downwards).\n")
             cat("Transformation: ",z$transformation,".\n",sep="")
           }
           ),
         "proportional odds" = {
           cat("Proportional odds distribution.\n")
           cat("Transformation: ",z$transformation,".\n",sep="")
           cat("Link:           ",links[z$link],".\n",sep="")
         },
         "adjacent categories" = {
           cat("Adjacent categories distribution.\n")
           cat("Transformation: ",z$transformation,".\n",sep="")
         }
         )
  if(z$distribution=="multinomial")
    np <- (z$nc-1)*(z$ncv+1)
  else
    np <- z$nc-1+z$ncv
#  print.matrix(rbind(-round(z$likelihood,2),if(z$individual) as.character(z$n-np) else as.character(z$events-np),as.character(np),-round(z$likelihood,2)+np,as.character(z$iterations)),
#               rowlab=c("-Log likelihood ","Degrees of freedom ","Number of parameters ","AIC ","Iterations "),collab="",quote=F,right=T)
  print(matrix(rbind(-round(z$likelihood,2),if(z$individual) as.character(z$n-np) else as.character(z$events-np),as.character(np),-round(z$likelihood,2)+np,as.character(z$iterations)),
               dimnames=list(c("-Log likelihood ","Degrees of freedom ","Number of parameters ","AIC ","Iterations "),"")),quote=F,right=T)
  if(z$iterlim==z$iterations)
    cat("\nWarning: Number of iterations exceeded.\n")
  if(z$info!=0&&z$iterlim!=0)
    cat("\nWarning: Exact singularity in QR decomposition.\n")
  if(z$rank!=np&&z$iterlim!=0)
    cat("\nWarning: Second derivative matrix is singular.\n")
  res <- cbind(z$coefficients,z$se)
  if(z$distribution=="binary")
    rownames(res) <- c("(Intercept)",colnames(z$tvcov$tvcov))
  else
    if(z$distribution=="multinomial"&&z$ncv!=0)
      rownames(res) <- as.vector(t(cbind(matrix(paste("(Intercept",1:(z$nc-1),")",sep=""),ncol=1),
                                         matrix(rep(colnames(z$tvcov$tvcov),(z$nc-1)),nrow=(z$nc-1),byrow=T))))
    else
      if((z$distribution=="continuation-ratio"&&z$direction=="upwards")||z$distribution=="proportional odds")
        rownames(res) <- c(paste("(Intercept",0:(z$nc-2),")",sep=""),colnames(z$tvcov$tvcov))
      else
        rownames(res) <- c(paste("(Intercept",1:(z$nc-1),")",sep=""),colnames(z$tvcov$tvcov))
  colnames(res) <- c("estimate","s.e.")
  cat("\nRegression coefficients\n")
  print.default(res,digits=digits,quote=F,print.gap=print.gap)
  if(length(z$coefficients)>1&&z$iterations>0) {
    rownames(z$correlations) <- colnames(z$correlations) <- 1:length(z$coefficients)
    cat("\nCorrelation matrix\n")
    print.default(z$correlations,digits=digits,quote=F,print.gap=print.gap)
  }
  cat("\n")
}

print.kalordinal <- function(z,digits=max(3,.Options$digits-3)) {
  nc <- length(unique(z$response$y))
  if(!is.null(z$ccov))
    nccov <- dim(z$ccov$ccov)[2]
  else
    nccov <- 0
  deppar <- (z$depend=="serial"||z$depend=="Markov")
  cat("\nCall:",deparse(z$call),sep="\n")
  cat("\n")
  if(z$optimize&&z$code>2)
    cat("Warning: no convergence - error",z$code,"\n\n")
  if(length(nobs(z))!=length(z$response$y))
    cat("Number of subjects    ",length(nobs(z)),"\n")
  cat("Number of observations",length(z$response$y),"\n")
  if(!is.null(z$response$time))
    cat("Times centred at      ",mean(z$response$time),"\n")
  if(z$depend=="independence")
    cat(" with independence\n")
  else
    cat(" with",z$depend,"dependence\n")
  cat("\n-Log likelihood   ",z$maxlike,"\n")
  cat("Degrees of freedom",z$df,"\n")
  cat("AIC               ",z$aic,"\n")
  if(z$optimize)
    cat("Iterations        ",z$iterations,"\n")
  cat("\nLocation parameters\n")
  if(!is.null(attr(z$mu,"formula")))
    cat(deparse(attr(z$mu,"formula")),sep="\n")
  else
    if(!is.null(attr(z$mu,"model"))) {
      t <- deparse(attr(z$mu,"model"))
      t[1] <- sub("expression\\(","",t[1])
      t[length(t)] <- sub("\\)$","",t[length(t)])
      cat(t,sep="\n")
    }
  coef.table <- cbind(z$coef[1:z$npr],z$se[1:z$npr])
  if(nc==2)
    cname <- "(Intercept)"
  else
    cname <- paste("(Intercept",1:(nc-1),")",sep="")
  if(inherits(z$mu,"formulafn"))
    cname <- c(cname,if(is.character(attr(z$mu,"model"))) attr(z$mu,"model") else attr(z$mu,"parameters"))
  else {
    if(nccov)
      cname <- c(cname,colnames(z$ccov$ccov))
    if(z$torder) {
      cname <- c(cname,paste("t^",1:z$torder,sep=""))
      if(length(z$interaction)>0) {
        for(i in 1:nccov)
          if(z$interaction[i]>0) {
            cname <- c(cname,paste(colnames(z$ccov$ccov)[i],".t^",1:z$interaction[i],sep=""))
          }
      }
    }
    if(!is.null(z$tvcov))
      cname <- c(cname,colnames(z$tvcov$tvcov))
  }
  dimnames(coef.table) <- list(cname, c("estimate","se"))
  print.default(coef.table, digits=digits, print.gap=2)
  if(z$init||deppar) {
    cat("\nNonlinear parameters\n")
    if(z$init)
      coef <- exp(z$coef[(z$npt-deppar):z$npt])
    else
      coef <- exp(z$coef[z$npt])
    cname <- NULL
    if(z$init)
      cname <- c(cname,"initial")
    if(deppar) {
      if(z$init)
        coef[2] <- coef[2]/(1+coef[2])
      else
        coef <- coef/(1+coef)
      cname <- c(cname,"depend")
    }
    if(z$init)
      coef.table <- cbind(z$coef[(z$npt-deppar):z$npt],z$se[(z$npt-deppar):z$npt],coef)
    else
      coef.table <- cbind(z$coef[z$npt],z$se[z$npt],coef)
    dimnames(coef.table) <- list(cname, c("estimate","se","parameter"))
    print.default(coef.table, digits=digits, print.gap=2)
    if(z$depend=="frailty") {
      tmp <- trigamma(exp(-z$coef[z$npt-deppar]))
      cat("Correlation =",tmp/(tmp+trigamma(1)),"\n")
    }
  }
  if(z$optimize&&dim(z$corr)[1]>1) {
    cat("\nCorrelation matrix\n")
    print.default(z$corr,digits=digits)
  }
}

print.moprofile <- function(z,nind=NULL,digits=max(3,.Options$digits-3),print.gap=2) {
  if(!is.null(nind)&&!z$individual) {
    warning("Option `nind' is ignored as the response corresponse to a frequency table.")
    nind <- NULL
  }
  if(!is.null(nind)) {
    if(max(nind)>z$events||min(nind)<1)
      stop("No such individual.")
    ns <- length(nind)
  }
  if(z$curve.type=="probability")
    cat("\nCategory containing highest probability:\n\n")
  else
    cat("\nCumulative probabilities:\n\n")
  tim <- unique(z$response$times)
  nobs <- nobs(z)
  ii <- covind(z)
  if(is.null(nind)) {
    if(z$individual) {
      cat("Individual data.\n")
      cat("Total number of individuals: ",z$events,"\n",sep="")
      cat("Number of observations:      ",z$n,"\n",sep="")
      cat("Categories available:        ",paste(0:(z$nc-1),collapse=" "),"\n",sep="")
      if(length(tim)>1)
        cat("Number of time points:       ",length(tim),"\n",sep="")
    }
    else {
      cat("Frequency table.\n")
      cat("Number of non-empty cells: ",z$n,"\n",sep="")
      cat("Total number of events:    ",z$events,"\n",sep="")
      cat("Categories available:      ",paste(0:(z$nc-1),collapse=" "),"\n",sep="")
      if(length(tim)>1)
        cat("Number of time points:     ",length(tim),"\n",sep="")
    }
  }
  else {
    cat("Individual data.\n")
    if(ns==1)
      cat("Individual selected:    ",nind,"\n",sep="")
    else
      cat("Individuals selected:   ",paste(nind,collapse=",",sep=""),"\n",sep="")
    cat("Number of observations: ",z$n,"\n",sep="")
    cat("Categories available:   ",paste(0:(z$nc-1),collapse=" "),"\n",sep="")
    if(length(tim)>1)
      cat("Number of time points:  ",length(tim),"\n",sep="")
  }
  cat("\n")
  if(z$curve.type=="probability") {
    if(is.null(nind)) {
      if(z$individual) {
        if(length(tim)>1) {
          res <- matrix(NA,nrow=length(nobs),ncol=length(tim),dimnames=list(1:length(nobs),paste("Time ",tim,sep="")))
          for(i in tim)
            res[ii[z$response$times==i],i] <- z$pred[z$response$times==i]
          rm(i)
        }
        else
          res <- z$pred
        print.default(res,digits=digits,quote=F,print.gap=print.gap)
        cat("\n")
      }
      else {
        if(length(tim)>1) {
          res <- NULL
          for(i in tim) {
            tmp <- matrix(NA,nrow=length(nobs),ncol=2,dimnames=list(NULL,c("Level","Freq.")))
            tmp[ii[z$response$times==i],] <- cbind(z$pred[z$response$times==i],z$response$wt[z$response$times==i])
            res <- c(res,list(tmp))
          }
          rm(i,tmp)
          names(res) <- paste("Time ",tim,sep="")
          print.listmatrix(res,row.names=1:length(nobs),digits=digits,print.gap=print.gap)
          cat("\n")
        }
        else {
          res <- cbind(z$pred,z$response$wt)
          colnames(res) <- c("Level","Freq.")
          rownames(res) <- 1:length(nobs)
          print.default(res,digits=digits,quote=F,print.gap=print.gap)
          cat("\n")
        }
      }
    }
    else {
      if(length(tim)>1) {
        res <- NULL
        for(i in tim) {
          tmp <- rep(NA,length(nind))
          for(j in 1:length(nind))
            tmp[j] <- z$pred[z$response$times==i][ii[z$response$times==i]==nind[j]]
          res <- cbind(res,tmp)
          rm(tmp,j)
        }
        rm(i)
        colnames(res) <- paste("Time ",tim,sep="")
        rownames(res) <- nind
      }
      else
        res <- z$pred[ii=nind]
      print.default(res,digits=digits,quote=F,print.gap=print.gap)
      cat("\n")
    }
  }
  else {
    if(is.null(nind)) {
      if(z$individual)
        if(length(tim)>1) {
          res <- NULL
          j <- 0
          for(i in tim) {
            j <- j+1
            res <- c(res,list(cbind(matrix(z$cpred,nrow=z$n)[z$response$times==i,,drop=F],1)))
            colnames(res[[j]]) <- c("0",apply(cbind(0,1:(z$nc-1)),1,function(x) paste(x,collapse="-")))
            rownames(res[[j]]) <- 1:length(nobs)
          }
          names(res) <- paste("Time ",tim,":",sep="")
          print.listmatrix(res,row.names=1:length(nobs),digits=digits,print.gap=print.gap)
          cat("\n")
        }
        else {
          res <- cbind(matrix(z$cpred,nrow=z$n),1)
          colnames(res) <- c("0",apply(cbind(0,1:(z$nc-1)),1,function(x) paste(x,collapse="-")))
          rownames(res) <- rep(1:length(nobs),dim(res)[1]/length(nobs))
          print.default(res,digits=digits,quote=F,print.gap=print.gap)
          cat("\n")
        }
      else
        if(length(tim)>1) {
          res <- NULL
          j <- 0
          for(i in tim) {
            j <- j+1
            res <- c(res,list(cbind(matrix(z$cpred,nrow=z$n)[z$response$times==i,],1,z$response$wt[z$response$times==i])))
            colnames(res[[j]]) <- c("0",apply(cbind(0,1:(z$nc-1)),1,function(x) paste(x,collapse="-")),"Freq.")
          }
          names(res) <- paste("Time ",tim,":",sep="")
          print.listmatrix(res,row.names=1:length(nobs),digits=digits,print.gap=print.gap)
          cat("\n")
        }
        else {
          res <- cbind(matrix(z$cpred,nrow=z$n),1,z$response$wt)
          colnames(res) <- c("0",apply(cbind(0,1:(z$nc-1)),1,function(x) paste(x,collapse="-")),"Freq.")
          rownames(res) <- rep(1:length(nobs),dim(res)[1]/length(nobs))
          print.default(res,digits=digits,quote=F,print.gap=print.gap)
          cat("\n")
        }        
    }
    else {
      if(length(tim)>1) {
        res <- NULL
        j <- 0
        for(i in tim) {
          j <- j+1
          tmp <- NULL
          for(k in nind)
            tmp <- rbind(tmp,(matrix(z$cpred,nrow=z$n)[z$response$times==i,])[ii[z$response$times==i]==k,,drop=F])
          res <- c(res,list(cbind(tmp,1)))
          colnames(res[[j]]) <- c("0",apply(cbind(0,1:(z$nc-1)),1,function(x) paste(x,collapse="-")))
          rm(tmp,k)
        }
        rm(j)
        names(res) <- paste("Time ",tim,":",sep="")
        print.listmatrix(res,row.names=nind,digits=digits,print.gap=print.gap)
        cat("\n")
      }
      else {
        res <- cbind(matrix(z$cpred,nrow=z$n)[ii==nind,,drop=F],1)
        colnames(res) <- c("0",apply(cbind(0,1:(z$nc-1)),1,function(x) paste(x,collapse="-")))
        rownames(res) <- nind
        print.default(res,digits=digits,quote=F,print.gap=print.gap)
        cat("\n")
      }
    }
  }
}

print.listmatrix <- function(x,row.names=NULL,digits=max(3,.Options$digits-3),print.gap=2) {
  res <- NULL
  for(i in 1:length(x)) {
    if(is.null(colnames(x[[i]])))
      names <- rep("",ncol(x[[i]]))
    else
      names <- colnames(x[[i]])
    dimnames(x[[i]]) <- NULL
    x[[i]] <- round(x[[i]],digits)
    mat <- array("",dim=dim(x[[i]]))
    for(j in 1:ncol(x[[i]])) {
      n <- lapply(strsplit(x[[i]][,j],"\\."),nchar)
      r <- unlist(lapply(n,length))==2
      if(any(r)) {
        len.l <- len.r <- vector(mode="numeric",length=length(n))
        len.l[r] <- unlist(n[r])[(1:length(n[r]))*2-1]
        if(any(!r))
          len.l[!r] <- unlist(n[!r])
        len.r[r] <- unlist(n[r])[(1:length(n[r]))*2]
        tot <- max(nchar(names[j]),max(len.l)+1+max(len.r))
        names[j] <- paste(paste(rep(" ",tot-nchar(names[j])),collapse="",sep=""),names[j],collapse="",sep="")
        mat[,j] <- paste(substr(as.character(x[[i]][,j]),1,len.l),".",substr(as.character(x[[i]][,j]),len.l+2,len.l+1+len.r),sep="")
        for(k in (1:length(x[[i]][,j]))[nchar(mat[,j])!=tot])
          mat[k,j] <- paste(paste(rep(" ",max(len.l)-len.l[k]),collapse="",sep=""),mat[k,j],paste(rep(0,max(len.r)-len.r[k]),collapse="",sep=""),collapse="",sep="")
        rm(n,r,len.l,len.r,tot,k)
      }
      else {
        len <- unlist(n)
        tot <- max(c(nchar(names[j]),len))
        names[j] <- paste(paste(rep(" ",tot-nchar(names[j])),collapse="",sep=""),names[j],collapse="",sep="")
        mat[,j] <- as.character(x[[i]][,j])
        for(k in (1:length(x[[i]][,j]))[nchar(mat[,j])!=tot])
          mat[k,j] <- paste(paste(rep(" ",tot-len[k]),collapse="",sep=""),mat[k,j],collapse="",sep="")
        rm(n,r,len,tot,k)
      }
    }
    mat <- rbind(names,mat)
    mat <- as.matrix(apply(mat,1,function(x) paste(x,collapse=" ")))
    dimnames(mat) <- NULL
    cname <- names(x[i])
    len <- unique(nchar(mat))
    center <- len-nchar(cname)
    if(center>0)
      cname <- paste(paste(rep(" ",floor(center/2)),collapse="",sep=""),cname,paste(rep(" ",center-floor(center/2)),collapse="",sep=""))
    else
      if(center<0)
        mat <- as.matrix(paste(paste(rep(" ",floor(abs(center)/2)),collapse="",sep=""),mat,
                               paste(rep(" ",abs(center)-floor(abs(center)/2)),collapse="",sep=""),sep=""))
    if(i!=length(x))
      mat <- as.matrix(paste(mat,paste(rep(" ",3*print.gap),collapse="",sep=""),sep=""))
    colnames(mat) <- cname
    res <- cbind(res,mat)
    rm(names,mat,j,cname,len,center)
  }
  if(is.null(row.names)||(length(row.names)+1)!=dim(res)[1])
    rownames(res) <- rep("",dim(res)[1])
  else
    rownames(res) <- c("",row.names)
  print.default(res,quote=F,print.gap=print.gap)
  invisible(res)
}
