/* -*- mode: c++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/*
 Copyright (C) 2006 Mario Pucci

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it
 under the terms of the QuantLib license.  You should have received a
 copy of the license along with this program; if not, please email
 <quantlib-dev@lists.sf.net>. The license is also available online at
 <http://quantlib.org/license.shtml>.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

#include <ql/termstructures/volatility/sabrsmilesection.hpp>
#include <ql/termstructures/volatility/sabr.hpp>
#include <ql/utilities/dataformatters.hpp>

namespace QuantLib {

    SabrSmileSection::SabrSmileSection(Time timeToExpiry,
                                       Rate forward,
                                       const std::vector<Real>& sabrParams)
    : SmileSection(timeToExpiry), forward_(forward) {

        alpha_ = sabrParams[0];
        beta_ = sabrParams[1];
        nu_ = sabrParams[2];
        rho_ = sabrParams[3];

        QL_REQUIRE(forward_>0.0, "at the money forward rate must be "
                   "positive: " << io::rate(forward_) << " not allowed");
        validateSabrParameters(alpha_, beta_, nu_, rho_);
    }

    SabrSmileSection::SabrSmileSection(const Date& d,
                                       Rate forward,
                                       const std::vector<Real>& sabrParams,
                                       const DayCounter& dc)
    : SmileSection(d, dc), forward_(forward) {

        alpha_ = sabrParams[0];
        beta_ = sabrParams[1];
        nu_ = sabrParams[2];
        rho_ = sabrParams[3];

        QL_REQUIRE(forward_>0.0, "at the money forward rate must be "
                   "positive: " << io::rate(forward_) << " not allowed");
        validateSabrParameters(alpha_, beta_, nu_, rho_);
    }

     Real SabrSmileSection::varianceImpl(Rate strike) const {
        Volatility vol = unsafeSabrVolatility(strike, forward_,
            exerciseTime(), alpha_, beta_, nu_, rho_);
        return vol*vol*exerciseTime();
     }

     Real SabrSmileSection::volatilityImpl(Rate strike) const {
        return unsafeSabrVolatility(strike, forward_,
            exerciseTime(), alpha_, beta_, nu_, rho_);
     }

}
