#!/usr/bin/env python

from os                    import remove
from cogent.util.unit_test import TestCase, main
from cogent.core.info      import Info
from cogent.app.cmfinder   import CMfinder,CombMotif

__author__ = "Shandy Wikman"
__copyright__ = "Copyright 2007-2011, The Cogent Project"
__contributors__ = ["Shandy Wikman"]
__license__ = "GPL"
__version__ = "1.5.1"
__maintainer__ = "Shandy Wikman"
__email__ = "ens01svn@cs.umu.se"
__status__ = "Development"

class CmfinderTest(TestCase):
    """Tests for Cmfinder application controller"""

    def setUp(self):
        self.input = cmfinder_input
        
    def test_input_as_lines(self):
        """Test Cmfinder stdout input as lines"""

        c = CMfinder(InputHandler='_input_as_lines')
        res = c(self.input,remove_tmp=False)
        #Filename in stdout, can't compare stdout since impossible to 
        #predict filename
        self.assertEqual(res['ExitStatus'],0)
        assert res['StdOut'] is not None
        res.cleanUp()

    def test_stdout_input_as_string(self):
        """Test Cmfinder stdout input as string"""

        c = CMfinder()
        exp ='%s\n' % '\n'.join([str(i).strip('\n') for i in cmfinder_stdout])

        f = open('input.fasta','w')
        txt = '\n'.join([str(i).strip('\n') for i in self.input])
        f.write(txt)
        f.close()
        res = c('input.fasta',remove_tmp=False)
        obs = res['StdOut'].read()
        self.assertEqual(obs,exp)
        self.assertEqual(res['ExitStatus'],0)
        res.cleanUp()
        remove('input.fasta')

    def test_get_result_path(self):
        """Tests cmfinder result path"""

        c =CMfinder(InputHandler='_input_as_lines')
        res = c(self.input,remove_tmp=False)

        self.assertEqualItems(res.keys(),['StdOut','StdErr','ExitStatus',
        'cm_1','cm_2','cm_3','motif_1','motif_2','motif_3','latest',
        '_input_filename'])

        self.assertEqual(res['ExitStatus'],0)
        assert res['cm_1'] is not None
        assert res['cm_2'] is not None
        assert res['cm_3'] is not None
        assert res['motif_1'] is not None
        assert res['motif_2'] is not None
        assert res['motif_3'] is not None
        res.cleanUp()


class CombMotifTest(TestCase):
    """Tests for CombMotif application controller

    _input_as_lines function not used!!"""

    def setUp(self):
        """ """
        a = open('input.fasta.cm.h1.1','w')
        txt = '\n'.join([str(i).strip('\n') for i in cm1])
        a.write(txt)
        a.close()

        b = open('input.fasta.cm.h1.2','w')
        txt = '\n'.join([str(i).strip('\n') for i in cm2])
        b.write(txt)
        b.close()

        c = open('input.fasta.cm.h1.3','w')
        txt = '\n'.join([str(i).strip('\n') for i in cm3])
        c.write(txt)
        c.close()

        d = open('input.fasta.motif.h1.1','w')
        txt = '\n'.join([str(i).strip('\n') for i in motif1])
        d.write(txt)
        d.close()

        e = open('input.fasta.motif.h1.2','w')
        txt = '\n'.join([str(i).strip('\n') for i in motif2])
        e.write(txt)
        e.close()

        f = open('input.fasta.motif.h1.3','w')
        txt = '\n'.join([str(i).strip('\n') for i in motif3])
        f.write(txt)
        f.close()

        g = open('input.fasta','w')
        txt = '\n'.join([str(i).strip('\n') for i in cmfinder_input])
        g.write(txt)
        g.close()

    def test_combmotif(self):
        """Tests CombMotif result path & input as string"""

        cbm =CombMotif()
        res = cbm('input.fasta')

        self.assertEqualItems(res.keys(),['StdOut','StdErr','ExitStatus',
        'comb1','latest'])
        self.assertEqual(res['ExitStatus'],0)
        assert res['comb1'] is not None
        res.cleanUp()
        remove('input.fasta')
        remove('input.fasta.cm.h1.1')
        remove('input.fasta.cm.h1.2')
        remove('input.fasta.cm.h1.3')
        remove('input.fasta.motif.h1.1')
        remove('input.fasta.motif.h1.2')
        remove('input.fasta.motif.h1.3')
        remove('input.fasta.cm.h1.2.3')

cmfinder_input = ['>seq1\n', 'GGCTAGATAGCTCAGATGGTAGAGCAGAGGATTGAAGATCCTTGTGTCGTCGGTTCGATCCCGGCTCTGGCC\n', '>seq2\n', 'GGCTAGATAGCTCAGATGGTAGAGCAGAGGATTGAAGATCCTTGTGTCGTCGGTTCGATCCCGGCTCTGGCC\n', '>seq3\n', 'GGCTAGATAGCTCAGATGGTAGAGCAGAGGATTGAAGATCCTTGTGTCGTCGGTTCGATCCCGGCTCTGGCC\n', '>seq4\n', 'GGCTAGATAGCTCAGATGGTAGAGCAGAGGATTGAAGATCCTTGTGTCGTCGGTTCGATCCCGGCTCTGGCC']

cmfinder_stdout = ['Seq_0_Cand0_20_69  -0.000000\n', 'UAGAGCAGAGGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCUG\n', '((((((...((((((((...((..........))))))))))..))))))\n', 'Seq_0_Cand1_29_61  -0.000000\n', 'GGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCC\n', '((((((((...((..........))))))))))\n', 'Seq_0_Cand2_36_68  -0.000000\n', 'AGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCU\n', '(((.((....((((......))))...)).)))\n', 'Alignment saved in file input.fasta.align.h1.1\n', 'Alignment saved in file input.fasta.align.h1.2\n', 'Alignment saved in file input.fasta.align.h1.3\n']

cm1 = ['INFERNAL-1 [0.7]\n', 'NAME   (null)\n', 'STATES 154\n', 'NODES  40\n', 'W      200\n', 'el_selfsc 0.000000\n', 'NULL  -0.363 -0.170  0.415  0.000 \n', 'MODEL:\n', '\t\t\t\t[ ROOT    0 ]\n', '     S     0    -1 0     1     4  -7.615  -7.822  -0.033  -6.236                 \n', '    IL     1     1 2     1     4  -1.686  -2.369  -1.117  -4.855                  1.023 -0.442 -0.708 -0.076 \n', '    IR     2     2 3     2     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL    1 ]\n', '    ML     3     2 3     5     3  -8.950  -0.012  -7.267                         -1.824 -2.170 -3.369  1.788 \n', '     D     4     2 3     5     3  -5.620  -0.734  -1.403                         \n', '    IL     5     5 3     5     3  -1.925  -0.554  -4.164                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATR    2 ]\n', '    MR     6     5 3     8     5  -7.656  -0.026  -7.471  -7.683  -8.575         -1.639 -3.134  1.386 -2.615 \n', '     D     7     5 3     8     5  -5.352  -0.707  -2.978  -4.409  -2.404         \n', '    IR     8     8 3     8     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    3 ]\n', '    MP     9     8 3    13     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -3.884 -3.075 -5.228  4.065 -4.750 -5.113 -1.228 -4.059 -4.861 -0.314 -6.063 -1.931 -0.504 -4.796 -2.719 -3.753 \n', '    ML    10     8 3    13     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    11     8 3    13     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    12     8 3    13     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    13    13 5    13     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    14    14 6    14     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    4 ]\n', '    MP    15    14 6    19     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML    16    14 6    19     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    17    14 6    19     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    18    14 6    19     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    19    19 5    19     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    20    20 6    20     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    5 ]\n', '    MP    21    20 6    25     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -3.884 -3.075 -5.228  4.065 -4.750 -5.113 -1.228 -4.059 -4.861 -0.314 -6.063 -1.931 -0.504 -4.796 -2.719 -3.753 \n', '    ML    22    20 6    25     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    23    20 6    25     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    24    20 6    25     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    25    25 5    25     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    26    26 6    26     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    6 ]\n', '    MP    27    26 6    31     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML    28    26 6    31     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    29    26 6    31     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    30    26 6    31     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    31    31 5    31     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    32    32 6    32     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    7 ]\n', '    MP    33    32 6    37     4  -7.615  -7.822  -0.033  -6.236                 -5.208 -5.325 -4.370 -0.894 -3.401 -4.617  3.483 -4.720 -7.134 -1.459 -5.156 -3.274  0.375 -6.308 -1.634 -4.773 \n', '    ML    34    32 6    37     4  -3.758  -3.940  -0.507  -2.670                  1.023 -0.442 -0.708 -0.076 \n', '    MR    35    32 6    37     4  -4.809  -3.838  -1.706  -0.766                  1.023 -0.442 -0.708 -0.076 \n', '     D    36    32 6    37     4  -4.568  -4.250  -2.265  -0.520                 \n', '    IL    37    37 5    37     4  -1.686  -2.369  -1.117  -4.855                  1.023 -0.442 -0.708 -0.076 \n', '    IR    38    38 6    38     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL    8 ]\n', '    ML    39    38 6    41     3  -8.593  -0.013  -7.247                          2.232 -3.180 -3.632 -2.819 \n', '     D    40    38 6    41     3  -6.174  -1.687  -0.566                         \n', '    IL    41    41 3    41     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL    9 ]\n', '    ML    42    41 3    44     3  -8.593  -0.013  -7.247                         -1.639 -3.134  1.386 -2.615 \n', '     D    43    41 3    44     3  -6.174  -1.687  -0.566                         \n', '    IL    44    44 3    44     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   10 ]\n', '    ML    45    44 3    47     3  -8.950  -0.012  -7.267                          2.232 -3.180 -3.632 -2.819 \n', '     D    46    44 3    47     3  -5.620  -0.734  -1.403                         \n', '    IL    47    47 3    47     3  -1.925  -0.554  -4.164                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATR   11 ]\n', '    MR    48    47 3    50     3  -8.950  -0.012  -7.267                         -1.639 -3.134  1.386 -2.615 \n', '     D    49    47 3    50     3  -6.390  -1.568  -0.620                         \n', '    IR    50    50 3    50     3  -1.925  -0.554  -4.164                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATR   12 ]\n', '    MR    51    50 3    53     5  -7.656  -0.026  -7.471  -7.683  -8.575         -2.124  1.984 -3.766 -2.279 \n', '     D    52    50 3    53     5  -5.352  -0.707  -2.978  -4.409  -2.404         \n', '    IR    53    53 3    53     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   13 ]\n', '    MP    54    53 3    58     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML    55    53 3    58     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    56    53 3    58     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    57    53 3    58     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    58    58 5    58     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    59    59 6    59     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   14 ]\n', '    MP    60    59 6    64     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML    61    59 6    64     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    62    59 6    64     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    63    59 6    64     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    64    64 5    64     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    65    65 6    65     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   15 ]\n', '    MP    66    65 6    70     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -3.884 -3.075 -5.228  4.065 -4.750 -5.113 -1.228 -4.059 -4.861 -0.314 -6.063 -1.931 -0.504 -4.796 -2.719 -3.753 \n', '    ML    67    65 6    70     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    68    65 6    70     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    69    65 6    70     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    70    70 5    70     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    71    71 6    71     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   16 ]\n', '    MP    72    71 6    76     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -3.687 -4.001 -5.006 -0.279 -3.289 -5.690 -0.341 -4.598 -4.707 -1.022 -5.894 -2.187  4.037 -4.560 -1.931 -3.750 \n', '    ML    73    71 6    76     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    74    71 6    76     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    75    71 6    76     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    76    76 5    76     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    77    77 6    77     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   17 ]\n', '    MP    78    77 6    82     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -4.629 -4.281 -5.803 -0.660 -4.305 -6.778 -1.192 -5.008 -5.286 -1.451 -6.272 -2.306 -0.329 -5.298  3.329 -4.356 \n', '    ML    79    77 6    82     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    80    77 6    82     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    81    77 6    82     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    82    82 5    82     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    83    83 6    83     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   18 ]\n', '    MP    84    83 6    88     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML    85    83 6    88     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    86    83 6    88     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    87    83 6    88     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    88    88 5    88     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    89    89 6    89     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   19 ]\n', '    MP    90    89 6    94     4  -7.615  -7.822  -0.033  -6.236                 -3.884 -3.075 -5.228  4.065 -4.750 -5.113 -1.228 -4.059 -4.861 -0.314 -6.063 -1.931 -0.504 -4.796 -2.719 -3.753 \n', '    ML    91    89 6    94     4  -3.758  -3.940  -0.507  -2.670                  1.023 -0.442 -0.708 -0.076 \n', '    MR    92    89 6    94     4  -4.809  -3.838  -1.706  -0.766                  1.023 -0.442 -0.708 -0.076 \n', '     D    93    89 6    94     4  -4.568  -4.250  -2.265  -0.520                 \n', '    IL    94    94 5    94     4  -1.686  -2.369  -1.117  -4.855                  1.023 -0.442 -0.708 -0.076 \n', '    IR    95    95 6    95     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   20 ]\n', '    ML    96    95 6    98     3  -8.593  -0.013  -7.247                          2.232 -3.180 -3.632 -2.819 \n', '     D    97    95 6    98     3  -6.174  -1.687  -0.566                         \n', '    IL    98    98 3    98     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   21 ]\n', '    ML    99    98 3   101     3  -8.593  -0.013  -7.247                         -1.639 -3.134  1.386 -2.615 \n', '     D   100    98 3   101     3  -6.174  -1.687  -0.566                         \n', '    IL   101   101 3   101     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   22 ]\n', '    ML   102   101 3   104     3  -8.593  -0.013  -7.247                          2.232 -3.180 -3.632 -2.819 \n', '     D   103   101 3   104     3  -6.174  -1.687  -0.566                         \n', '    IL   104   104 3   104     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   23 ]\n', '    ML   105   104 3   107     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D   106   104 3   107     3  -6.174  -1.687  -0.566                         \n', '    IL   107   107 3   107     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   24 ]\n', '    ML   108   107 3   110     3  -8.593  -0.013  -7.247                         -2.124  1.984 -3.766 -2.279 \n', '     D   109   107 3   110     3  -6.174  -1.687  -0.566                         \n', '    IL   110   110 3   110     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   25 ]\n', '    ML   111   110 3   113     3  -8.593  -0.013  -7.247                         -2.124  1.984 -3.766 -2.279 \n', '     D   112   110 3   113     3  -6.174  -1.687  -0.566                         \n', '    IL   113   113 3   113     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   26 ]\n', '    ML   114   113 3   116     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D   115   113 3   116     3  -6.174  -1.687  -0.566                         \n', '    IL   116   116 3   116     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   27 ]\n', '    ML   117   116 3   119     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D   118   116 3   119     3  -6.174  -1.687  -0.566                         \n', '    IL   119   119 3   119     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   28 ]\n', '    ML   120   119 3   122     3  -8.593  -0.013  -7.247                         -1.639 -3.134  1.386 -2.615 \n', '     D   121   119 3   122     3  -6.174  -1.687  -0.566                         \n', '    IL   122   122 3   122     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   29 ]\n', '    ML   123   122 3   125     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D   124   122 3   125     3  -6.174  -1.687  -0.566                         \n', '    IL   125   125 3   125     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   30 ]\n', '    ML   126   125 3   128     3  -8.593  -0.013  -7.247                         -1.639 -3.134  1.386 -2.615 \n', '     D   127   125 3   128     3  -6.174  -1.687  -0.566                         \n', '    IL   128   128 3   128     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   31 ]\n', '    ML   129   128 3   131     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D   130   128 3   131     3  -6.174  -1.687  -0.566                         \n', '    IL   131   131 3   131     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   32 ]\n', '    ML   132   131 3   134     3  -8.593  -0.013  -7.247                         -2.124  1.984 -3.766 -2.279 \n', '     D   133   131 3   134     3  -6.174  -1.687  -0.566                         \n', '    IL   134   134 3   134     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   33 ]\n', '    ML   135   134 3   137     3  -8.593  -0.013  -7.247                         -1.639 -3.134  1.386 -2.615 \n', '     D   136   134 3   137     3  -6.174  -1.687  -0.566                         \n', '    IL   137   137 3   137     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   34 ]\n', '    ML   138   137 3   140     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D   139   137 3   140     3  -6.174  -1.687  -0.566                         \n', '    IL   140   140 3   140     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   35 ]\n', '    ML   141   140 3   143     3  -8.593  -0.013  -7.247                         -2.124  1.984 -3.766 -2.279 \n', '     D   142   140 3   143     3  -6.174  -1.687  -0.566                         \n', '    IL   143   143 3   143     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   36 ]\n', '    ML   144   143 3   146     3  -8.593  -0.013  -7.247                         -1.639 -3.134  1.386 -2.615 \n', '     D   145   143 3   146     3  -6.174  -1.687  -0.566                         \n', '    IL   146   146 3   146     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   37 ]\n', '    ML   147   146 3   149     3  -8.593  -0.013  -7.247                         -1.639 -3.134  1.386 -2.615 \n', '     D   148   146 3   149     3  -6.174  -1.687  -0.566                         \n', '    IL   149   149 3   149     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   38 ]\n', '    ML   150   149 3   152     2  -9.084  -0.003                                 -1.824 -2.170 -3.369  1.788 \n', '     D   151   149 3   152     2  -8.445  -0.004                                 \n', '    IL   152   152 3   152     2  -1.823  -0.479                                  1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ END    39 ]\n', '     E   153   152 3    -1     0                                                 \n', '//\n']

cm2 = ['INFERNAL-1 [0.7]\n', 'NAME   (null)\n', 'STATES 103\n', 'NODES  24\n', 'W      200\n', 'el_selfsc 0.000000\n', 'NULL  -0.363 -0.170  0.415  0.000 \n', 'MODEL:\n', '\t\t\t\t[ ROOT    0 ]\n', '     S     0    -1 0     1     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 \n', '    IL     1     1 2     1     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR     2     2 3     2     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    1 ]\n', '    MP     3     2 3     7     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML     4     2 3     7     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR     5     2 3     7     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D     6     2 3     7     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL     7     7 5     7     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR     8     8 6     8     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    2 ]\n', '    MP     9     8 6    13     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML    10     8 6    13     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    11     8 6    13     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    12     8 6    13     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    13    13 5    13     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    14    14 6    14     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    3 ]\n', '    MP    15    14 6    19     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -3.884 -3.075 -5.228  4.065 -4.750 -5.113 -1.228 -4.059 -4.861 -0.314 -6.063 -1.931 -0.504 -4.796 -2.719 -3.753 \n', '    ML    16    14 6    19     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    17    14 6    19     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    18    14 6    19     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    19    19 5    19     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    20    20 6    20     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    4 ]\n', '    MP    21    20 6    25     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -3.687 -4.001 -5.006 -0.279 -3.289 -5.690 -0.341 -4.598 -4.707 -1.022 -5.894 -2.187  4.037 -4.560 -1.931 -3.750 \n', '    ML    22    20 6    25     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    23    20 6    25     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    24    20 6    25     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    25    25 5    25     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    26    26 6    26     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    5 ]\n', '    MP    27    26 6    31     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -4.629 -4.281 -5.803 -0.660 -4.305 -6.778 -1.192 -5.008 -5.286 -1.451 -6.272 -2.306 -0.329 -5.298  3.329 -4.356 \n', '    ML    28    26 6    31     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    29    26 6    31     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    30    26 6    31     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    31    31 5    31     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    32    32 6    32     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    6 ]\n', '    MP    33    32 6    37     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML    34    32 6    37     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    35    32 6    37     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    36    32 6    37     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    37    37 5    37     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    38    38 6    38     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    7 ]\n', '    MP    39    38 6    43     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -3.884 -3.075 -5.228  4.065 -4.750 -5.113 -1.228 -4.059 -4.861 -0.314 -6.063 -1.931 -0.504 -4.796 -2.719 -3.753 \n', '    ML    40    38 6    43     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    41    38 6    43     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    42    38 6    43     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    43    43 5    43     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    44    44 6    44     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    8 ]\n', '    MP    45    44 6    49     4  -7.487  -8.733  -0.052  -5.186                 -3.884 -3.075 -5.228  4.065 -4.750 -5.113 -1.228 -4.059 -4.861 -0.314 -6.063 -1.931 -0.504 -4.796 -2.719 -3.753 \n', '    ML    46    44 6    49     4  -2.408  -4.532  -1.293  -1.473                  1.023 -0.442 -0.708 -0.076 \n', '    MR    47    44 6    49     4  -4.102 -12.528  -0.390  -2.485                  1.023 -0.442 -0.708 -0.076 \n', '     D    48    44 6    49     4 -12.737 -14.007  -2.036  -0.404                 \n', '    IL    49    49 5    49     4  -2.817  -4.319  -0.613  -2.698                  1.023 -0.442 -0.708 -0.076 \n', '    IR    50    50 6    50     3  -1.925  -0.554  -4.164                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATR    9 ]\n', '    MR    51    50 6    53     3  -8.950  -0.012  -7.267                         -1.639 -3.134  1.386 -2.615 \n', '     D    52    50 6    53     3  -6.390  -1.568  -0.620                         \n', '    IR    53    53 3    53     3  -1.925  -0.554  -4.164                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATR   10 ]\n', '    MR    54    53 3    56     5  -7.656  -0.026  -7.471  -7.683  -8.575         -1.639 -3.134  1.386 -2.615 \n', '     D    55    53 3    56     5  -5.352  -0.707  -2.978  -4.409  -2.404         \n', '    IR    56    56 3    56     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   11 ]\n', '    MP    57    56 3    61     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML    58    56 3    61     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    59    56 3    61     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    60    56 3    61     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    61    61 5    61     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    62    62 6    62     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   12 ]\n', '    MP    63    62 6    67     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -3.884 -3.075 -5.228  4.065 -4.750 -5.113 -1.228 -4.059 -4.861 -0.314 -6.063 -1.931 -0.504 -4.796 -2.719 -3.753 \n', '    ML    64    62 6    67     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    65    62 6    67     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    66    62 6    67     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    67    67 5    67     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    68    68 6    68     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   13 ]\n', '    MP    69    68 6    73     4  -7.615  -7.822  -0.033  -6.236                 -4.629 -4.281 -5.803 -0.660 -4.305 -6.778 -1.192 -5.008 -5.286 -1.451 -6.272 -2.306 -0.329 -5.298  3.329 -4.356 \n', '    ML    70    68 6    73     4  -3.758  -3.940  -0.507  -2.670                  1.023 -0.442 -0.708 -0.076 \n', '    MR    71    68 6    73     4  -4.809  -3.838  -1.706  -0.766                  1.023 -0.442 -0.708 -0.076 \n', '     D    72    68 6    73     4  -4.568  -4.250  -2.265  -0.520                 \n', '    IL    73    73 5    73     4  -1.686  -2.369  -1.117  -4.855                  1.023 -0.442 -0.708 -0.076 \n', '    IR    74    74 6    74     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   14 ]\n', '    ML    75    74 6    77     3  -8.593  -0.013  -7.247                         -2.124  1.984 -3.766 -2.279 \n', '     D    76    74 6    77     3  -6.174  -1.687  -0.566                         \n', '    IL    77    77 3    77     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   15 ]\n', '    ML    78    77 3    80     3  -8.593  -0.013  -7.247                         -2.124  1.984 -3.766 -2.279 \n', '     D    79    77 3    80     3  -6.174  -1.687  -0.566                         \n', '    IL    80    80 3    80     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   16 ]\n', '    ML    81    80 3    83     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D    82    80 3    83     3  -6.174  -1.687  -0.566                         \n', '    IL    83    83 3    83     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   17 ]\n', '    ML    84    83 3    86     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D    85    83 3    86     3  -6.174  -1.687  -0.566                         \n', '    IL    86    86 3    86     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   18 ]\n', '    ML    87    86 3    89     3  -8.593  -0.013  -7.247                         -1.639 -3.134  1.386 -2.615 \n', '     D    88    86 3    89     3  -6.174  -1.687  -0.566                         \n', '    IL    89    89 3    89     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   19 ]\n', '    ML    90    89 3    92     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D    91    89 3    92     3  -6.174  -1.687  -0.566                         \n', '    IL    92    92 3    92     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   20 ]\n', '    ML    93    92 3    95     3  -8.593  -0.013  -7.247                         -1.639 -3.134  1.386 -2.615 \n', '     D    94    92 3    95     3  -6.174  -1.687  -0.566                         \n', '    IL    95    95 3    95     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   21 ]\n', '    ML    96    95 3    98     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D    97    95 3    98     3  -6.174  -1.687  -0.566                         \n', '    IL    98    98 3    98     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   22 ]\n', '    ML    99    98 3   101     2  -9.084  -0.003                                 -2.124  1.984 -3.766 -2.279 \n', '     D   100    98 3   101     2  -8.445  -0.004                                 \n', '    IL   101   101 3   101     2  -1.823  -0.479                                  1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ END    23 ]\n', '     E   102   101 3    -1     0                                                 \n', '//\n']

cm3 = ['INFERNAL-1 [0.7]\n', 'NAME   (null)\n', 'STATES 103\n', 'NODES  24\n', 'W      200\n', 'el_selfsc 0.000000\n', 'NULL  -0.363 -0.170  0.415  0.000 \n', 'MODEL:\n', '\t\t\t\t[ ROOT    0 ]\n', '     S     0    -1 0     1     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 \n', '    IL     1     1 2     1     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR     2     2 3     2     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    1 ]\n', '    MP     3     2 3     7     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML     4     2 3     7     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR     5     2 3     7     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D     6     2 3     7     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL     7     7 5     7     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR     8     8 6     8     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    2 ]\n', '    MP     9     8 6    13     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML    10     8 6    13     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    11     8 6    13     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    12     8 6    13     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    13    13 5    13     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    14    14 6    14     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    3 ]\n', '    MP    15    14 6    19     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -3.884 -3.075 -5.228  4.065 -4.750 -5.113 -1.228 -4.059 -4.861 -0.314 -6.063 -1.931 -0.504 -4.796 -2.719 -3.753 \n', '    ML    16    14 6    19     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    17    14 6    19     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    18    14 6    19     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    19    19 5    19     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    20    20 6    20     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    4 ]\n', '    MP    21    20 6    25     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -3.687 -4.001 -5.006 -0.279 -3.289 -5.690 -0.341 -4.598 -4.707 -1.022 -5.894 -2.187  4.037 -4.560 -1.931 -3.750 \n', '    ML    22    20 6    25     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    23    20 6    25     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    24    20 6    25     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    25    25 5    25     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    26    26 6    26     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    5 ]\n', '    MP    27    26 6    31     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -4.629 -4.281 -5.803 -0.660 -4.305 -6.778 -1.192 -5.008 -5.286 -1.451 -6.272 -2.306 -0.329 -5.298  3.329 -4.356 \n', '    ML    28    26 6    31     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    29    26 6    31     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    30    26 6    31     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    31    31 5    31     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    32    32 6    32     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    6 ]\n', '    MP    33    32 6    37     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML    34    32 6    37     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    35    32 6    37     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    36    32 6    37     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    37    37 5    37     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    38    38 6    38     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    7 ]\n', '    MP    39    38 6    43     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -3.884 -3.075 -5.228  4.065 -4.750 -5.113 -1.228 -4.059 -4.861 -0.314 -6.063 -1.931 -0.504 -4.796 -2.719 -3.753 \n', '    ML    40    38 6    43     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    41    38 6    43     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    42    38 6    43     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    43    43 5    43     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    44    44 6    44     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP    8 ]\n', '    MP    45    44 6    49     4  -7.487  -8.733  -0.052  -5.186                 -3.884 -3.075 -5.228  4.065 -4.750 -5.113 -1.228 -4.059 -4.861 -0.314 -6.063 -1.931 -0.504 -4.796 -2.719 -3.753 \n', '    ML    46    44 6    49     4  -2.408  -4.532  -1.293  -1.473                  1.023 -0.442 -0.708 -0.076 \n', '    MR    47    44 6    49     4  -4.102 -12.528  -0.390  -2.485                  1.023 -0.442 -0.708 -0.076 \n', '     D    48    44 6    49     4 -12.737 -14.007  -2.036  -0.404                 \n', '    IL    49    49 5    49     4  -2.817  -4.319  -0.613  -2.698                  1.023 -0.442 -0.708 -0.076 \n', '    IR    50    50 6    50     3  -1.925  -0.554  -4.164                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATR    9 ]\n', '    MR    51    50 6    53     3  -8.950  -0.012  -7.267                         -1.639 -3.134  1.386 -2.615 \n', '     D    52    50 6    53     3  -6.390  -1.568  -0.620                         \n', '    IR    53    53 3    53     3  -1.925  -0.554  -4.164                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATR   10 ]\n', '    MR    54    53 3    56     5  -7.656  -0.026  -7.471  -7.683  -8.575         -1.639 -3.134  1.386 -2.615 \n', '     D    55    53 3    56     5  -5.352  -0.707  -2.978  -4.409  -2.404         \n', '    IR    56    56 3    56     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   11 ]\n', '    MP    57    56 3    61     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -6.059 -3.399 -7.491 -0.050 -6.698 -4.090 -1.804 -6.314 -5.110  3.545 -5.426 -1.969 -1.163 -4.250 -3.694 -5.400 \n', '    ML    58    56 3    61     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    59    56 3    61     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    60    56 3    61     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    61    61 5    61     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    62    62 6    62     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   12 ]\n', '    MP    63    62 6    67     6  -9.400  -9.339  -0.017  -8.115  -8.395  -8.790 -3.884 -3.075 -5.228  4.065 -4.750 -5.113 -1.228 -4.059 -4.861 -0.314 -6.063 -1.931 -0.504 -4.796 -2.719 -3.753 \n', '    ML    64    62 6    67     6  -6.250  -6.596  -1.310  -1.005  -6.446  -3.975  1.023 -0.442 -0.708 -0.076 \n', '    MR    65    62 6    67     6  -6.988  -5.717  -1.625  -5.695  -0.829  -3.908  1.023 -0.442 -0.708 -0.076 \n', '     D    66    62 6    67     6  -9.049  -7.747  -3.544  -4.226  -4.244  -0.319 \n', '    IL    67    67 5    67     6  -2.579  -2.842  -0.760  -4.497  -5.274  -4.934  1.023 -0.442 -0.708 -0.076 \n', '    IR    68    68 6    68     5  -2.408  -0.496  -5.920  -4.087  -5.193          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATP   13 ]\n', '    MP    69    68 6    73     4  -7.615  -7.822  -0.033  -6.236                 -4.629 -4.281 -5.803 -0.660 -4.305 -6.778 -1.192 -5.008 -5.286 -1.451 -6.272 -2.306 -0.329 -5.298  3.329 -4.356 \n', '    ML    70    68 6    73     4  -3.758  -3.940  -0.507  -2.670                  1.023 -0.442 -0.708 -0.076 \n', '    MR    71    68 6    73     4  -4.809  -3.838  -1.706  -0.766                  1.023 -0.442 -0.708 -0.076 \n', '     D    72    68 6    73     4  -4.568  -4.250  -2.265  -0.520                 \n', '    IL    73    73 5    73     4  -1.686  -2.369  -1.117  -4.855                  1.023 -0.442 -0.708 -0.076 \n', '    IR    74    74 6    74     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   14 ]\n', '    ML    75    74 6    77     3  -8.593  -0.013  -7.247                         -2.124  1.984 -3.766 -2.279 \n', '     D    76    74 6    77     3  -6.174  -1.687  -0.566                         \n', '    IL    77    77 3    77     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   15 ]\n', '    ML    78    77 3    80     3  -8.593  -0.013  -7.247                         -2.124  1.984 -3.766 -2.279 \n', '     D    79    77 3    80     3  -6.174  -1.687  -0.566                         \n', '    IL    80    80 3    80     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   16 ]\n', '    ML    81    80 3    83     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D    82    80 3    83     3  -6.174  -1.687  -0.566                         \n', '    IL    83    83 3    83     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   17 ]\n', '    ML    84    83 3    86     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D    85    83 3    86     3  -6.174  -1.687  -0.566                         \n', '    IL    86    86 3    86     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   18 ]\n', '    ML    87    86 3    89     3  -8.593  -0.013  -7.247                         -1.639 -3.134  1.386 -2.615 \n', '     D    88    86 3    89     3  -6.174  -1.687  -0.566                         \n', '    IL    89    89 3    89     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   19 ]\n', '    ML    90    89 3    92     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D    91    89 3    92     3  -6.174  -1.687  -0.566                         \n', '    IL    92    92 3    92     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   20 ]\n', '    ML    93    92 3    95     3  -8.593  -0.013  -7.247                         -1.639 -3.134  1.386 -2.615 \n', '     D    94    92 3    95     3  -6.174  -1.687  -0.566                         \n', '    IL    95    95 3    95     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   21 ]\n', '    ML    96    95 3    98     3  -8.593  -0.013  -7.247                         -1.824 -2.170 -3.369  1.788 \n', '     D    97    95 3    98     3  -6.174  -1.687  -0.566                         \n', '    IL    98    98 3    98     3  -1.442  -0.798  -4.142                          1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ MATL   22 ]\n', '    ML    99    98 3   101     2  -9.084  -0.003                                 -2.124  1.984 -3.766 -2.279 \n', '     D   100    98 3   101     2  -8.445  -0.004                                 \n', '    IL   101   101 3   101     2  -1.823  -0.479                                  1.023 -0.442 -0.708 -0.076 \n', '\t\t\t\t[ END    23 ]\n', '     E   102   101 3    -1     0                                                 \n', '//\n']

motif1 = ['# STOCKHOLM 1.0\n', '#=GF AU    CMfinder 0.2\n', '\n', '#=GS seq1 WT    1.00\n', '#=GS seq2 WT    1.00\n', '#=GS seq3 WT    1.00\n', '#=GS seq4 WT    1.00\n', '\n', '#=GS seq1 DE     20.. 69\t89.835533\n', '#=GS seq2 DE     20.. 69\t89.835533\n', '#=GS seq3 DE     20.. 69\t89.835533\n', '#=GS seq4 DE     20.. 69\t89.835533\n', '\n', 'seq1             UAGAGCAGAGGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCUG\n', '#=GR seq1 SS     .<<<<<...<<<<<<<...................>>>>>>>..>>>>>.\n', 'seq2             UAGAGCAGAGGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCUG\n', '#=GR seq2 SS     .<<<<<...<<<<<<<...................>>>>>>>..>>>>>.\n', 'seq3             UAGAGCAGAGGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCUG\n', '#=GR seq3 SS     .<<<<<...<<<<<<<...................>>>>>>>..>>>>>.\n', 'seq4             UAGAGCAGAGGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCUG\n', '#=GR seq4 SS     .<<<<<...<<<<<<<...................>>>>>>>..>>>>>.\n', '#=GC SS_cons     :<<<<<---<<<<<<<___________________>>>>>>>-->>>>>:\n', '#=GC RF          UAGAGCAGAGGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCUG\n', '//\n']

motif2 = ['# STOCKHOLM 1.0\n', '#=GF AU    CMfinder 0.2\n', '\n', '#=GS seq1 WT    1.00\n', '#=GS seq2 WT    1.00\n', '#=GS seq3 WT    1.00\n', '#=GS seq4 WT    1.00\n', '\n', '#=GS seq1 DE     29.. 61\t59.381535\n', '#=GS seq2 DE     29.. 61\t59.381535\n', '#=GS seq3 DE     29.. 61\t59.381535\n', '#=GS seq4 DE     29.. 61\t59.381535\n', '\n', 'seq1             GGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCC\n', '#=GR seq1 SS     <<<<<<<<<<<.........>>>..>>>>>>>>\n', 'seq2             GGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCC\n', '#=GR seq2 SS     <<<<<<<<<<<.........>>>..>>>>>>>>\n', 'seq3             GGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCC\n', '#=GR seq3 SS     <<<<<<<<<<<.........>>>..>>>>>>>>\n', 'seq4             GGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCC\n', '#=GR seq4 SS     <<<<<<<<<<<.........>>>..>>>>>>>>\n', '#=GC SS_cons     <<<<<<<<<<<_________>>>-->>>>>>>>\n', '#=GC RF          GGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCC\n', '//\n']

motif3 = ['# STOCKHOLM 1.0\n', '#=GF AU    CMfinder 0.2\n', '\n', '#=GS seq1 WT    1.00\n', '#=GS seq2 WT    1.00\n', '#=GS seq3 WT    1.00\n', '#=GS seq4 WT    1.00\n', '\n', '#=GS seq1 DE     36.. 68\t59.230255\n', '#=GS seq2 DE     36.. 68\t59.230255\n', '#=GS seq3 DE     36.. 68\t59.230255\n', '#=GS seq4 DE     36.. 68\t59.230255\n', '\n', 'seq1             AGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCU\n', '#=GR seq1 SS     .<<........>><<<<<.......>>>>>...\n', 'seq2             AGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCU\n', '#=GR seq2 SS     .<<........>><<<<<.......>>>>>...\n', 'seq3             AGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCU\n', '#=GR seq3 SS     .<<........>><<<<<.......>>>>>...\n', 'seq4             AGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCU\n', '#=GR seq4 SS     .<<........>><<<<<.......>>>>>...\n', '#=GC SS_cons     :<<________>><<<<<_______>>>>>:::\n', '#=GC RF          AGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCU\n', '//\n']


if __name__ == '__main__':
    main()
