#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009-2011  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Ported from GNUbik
# Original filename: move-queue.c
# Original copyright and license:
#/*
#  Copyright (c) 2004  Dale Mellor
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License,  or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#*/


#  OBJECT MoveQueue
#
#  DESCRIPTION
#
#    This object is designed to hold runs of moves which are applied to a
#    cube. New moves are pushed onto the back of the queue,  and old ones may be
#    popped of the front. The queue is indexed by a moving cursor (known
#    conventionally as the 'current' place) which can be moved backwards and
#    forwards. In this way,  the GNUbik application can maintain a running list
#    of moves being applied (either by the user or from scripts),  and can
#    allow the user to move backwards and forwards through the moves giving a
#    rewind/replay facility.
#
#    The queue object also holds some (redundant) state information,  which is
#    not currently used (or even accessible),  but can be used to monitor the
#    state of the queue (length,  percentage run,  etc).

import re

from .debug import *


# A structure containing information about a movement taking place.
class MoveData(object):
    def __init__(self, maxis, mslice, mdir):
        self.axis = maxis   # int 0...
        self.slice = mslice # int 0...dim-1
        self.dir = mdir     # 0 or 1.
    def copy(self):
        return MoveData(self.axis, self.slice, self.dir)
    def __str__(self):
        return 'MoveData(maxis={}, mslice={}, mdir={})'.format(self.axis, self.slice, self.dir)


class MoveQueue(object):
    class MoveQueueItem(MoveData):
        def __init__(self, data, mark_after=False, code=None, markup=None):
            if data:
                MoveData.__init__(self, data.axis, data.slice, data.dir)
            else:
                MoveData.__init__(self, 0, 0, 0)
            self.mark_after = mark_after
            self.code = code
            self.markup = markup
        def copy(self):
            return MoveQueue.MoveQueueItem(self,
                    mark_after=self.mark_after,
                    code=self.code,
                    markup=self.markup)
    
    def __init__(self):
        self.current_place = 0      # Number of steps of current from head.
        self.moves = []
        self.converter = FormatFlubrd
    def at_start(self):
        return self.current_place == 0
    def at_end(self):
        return self.current_place == self.queue_length
    @property
    def head(self):
        return self.moves and self.moves[0] or None
    @property
    def tail(self):
        return self.moves and self.moves[-1] or None
    @property
    def _prev(self):
        return self.moves[self.current_place-1]
    @property
    def _current(self):
        return self.moves[self.current_place]
    @property
    def queue_length(self):
        return len(self.moves)
    def __str__(self):
        return 'MoveQueue(len=%s)' % self.queue_length

    # Routine to push a new MoveData object onto the _back_ of the given queue. A
    # new,  locally-managed copy is made of the incumbent datum. Return 1 (one) if
    # the operation is successful; 0 (zero) indicates that there was insufficient
    # memory to grow the queue.
    def push(self, move_data, mark_after=False, code=None, markup=None):
        self.test()
        new_element = self.MoveQueueItem(move_data,
                                mark_after=mark_after, code=code, markup=markup)
        self.moves.append(new_element)
        self.test()
        return 1

    # Procedure to copy the incumbent datum to the current place in the queue,  and
    # drop all subsequent queue entries (so that the current element becomes the
    # tail). If current points past the end of the tail (including the case when
    # there are no data in the queue),  then a regular push operation is performed
    # at the tail,  and in this case zero may be returned if there is insufficient
    # memory to grow the queue. In all other cases 1 (one) will be returned.
    def push_current(self, move_data):
        # If there are no data in the queue,  then perform a standard push
        # operation. Also do this if we are at the tail.
        self.test()
        if not self.at_end():
            self.moves[self.current_place:] = []
        self.push(move_data)
        return 1

    def prev(self):
        return None if self.at_start() else self._prev
    # Routine to get the data at the 'current' position in the queue. If there are
    # no data in the queue,  None will be returned.
    def current(self):
        return None if self.at_end() else self._current

    # Routine to retard the current pointer (move it towards the _head_ of the queue).
    def retard(self):
        if not self.at_start():
            self.current_place -= 1
    
    def rewind_start(self):
        self.current_place = 0
        
    # Remove the current object and all those that come afterwards.
    def truncate(self):
        self.test()
        if self.at_start():
            self.current_place = 0
        else:
            self.retard()
            self.advance()
        self.moves[self.current_place:] = []
        self.test()
    
    def reset(self):
        self.test()
        self.current_place = 0
        self.moves[:] = []
        self.test()
        
    # Routine to advance the current position (move it towards the _tail_). If the
    # current position is already at the tail (which will also be the case if the
    # queue is empty) then no action takes place and 0 (zero) is returned to
    # indicate that no more data are available for consideration.
    def advance(self):
        self.test()
        if self.at_end():
            return False
        self.current_place += 1
        return not self.at_end()

    def mark_current(self, mark=True):
        self.test()
        if not self.at_start():
            self._prev.mark_after = mark
            if self._prev.code is not None:
                self._prev.code = self._prev.code.replace(' ','')
                #FIXME: recreate self._prev.markup
                if mark:
                    self._prev.code += ' '
        self.test()

    def test(self):
        assert self.current_place <= self.queue_length, '%s < %s'%(self.current_place, self.queue_length)
        if self.moves and self.at_end():
            assert self._prev == self.tail
    
    @debug_func
    def format(self, size):
        #TODO: arg to use explicit converter
        code = ''
        markup = ''
        pos = 0
        for i, move in enumerate(self.moves):
            if move.code is None:
                move.code, move.markup = self.converter.format(move, size)
                #debug('new code:', move.code)
            code += move.code
            markup += move.markup
            if i < self.current_place:
                pos = len(code)
        return code, pos, markup
        
    @debug_func
    def parse_iter(self, code, pos, size):
        #TODO: arg to use explicit converter
        code = code.lstrip(' ')
        queue_pos = self.current_place
        move_code = ''
        for i, c in enumerate(code):
            if move_code and self.converter.isstart(c):
                data, mark, markup_code = self.converter.parse(move_code, size)
                #FIXME: if data is None an invalid move should be pushed
                self.push(data, mark, move_code, markup_code)
                yield data, queue_pos
                if i == pos:
                    queue_pos = self.queue_length
                move_code = c
            else:
                move_code += c
            if i < pos:
                queue_pos = self.queue_length + 1
        if move_code:
            data, mark, markup_code = self.converter.parse(move_code, size)
            self.push(data, mark, move_code, markup_code)
            if len(code)-len(move_code) < pos:
                queue_pos = self.queue_length
            yield data, queue_pos
            
    def parse(self, code, pos, size):
        qpos = 0
        for res in self.parse_iter(code, pos, size):
            qpos = res[1]
        return qpos
        
        
class FormatFlubrd:
    map_face_flubrd = 'flubrd'
    re_flubrd = re.compile("(.)(\d*)([']?)([^ ]*)( *)(.*)")
    
    @classmethod
    def isstart(cls, char):
        return char in cls.map_face_flubrd
    
    @staticmethod
    def _format_markup(mface, mslice, mdir, err1, mark, err2):
        if err1:    err1 = '<span underline="error" color="red">%s</span>' % err1
        if err2:    err2 = '<span underline="error">%s</span>' % err2
        return ''.join((mface, mslice, mdir, err1, mark, err2))
    
    @staticmethod
    def intern_to_normal_move(maxis, mslice, mdir, size):
        if mslice*2 > size-1:
            return maxis+3, size-mslice-1, 1-mdir
        else:
            return maxis, mslice, mdir
    
    @classmethod
    def format(cls, move, size):
        mface, mslice, mdir = cls.intern_to_normal_move(move.axis, move.slice, move.dir, size)
        mface = cls.map_face_flubrd[mface]
        mslice = str(mslice+1) if mslice else ''
        mdir = "'" if mdir else ""
        mark = ' ' if move.mark_after else ''
        move_code = mface + mslice + mdir + mark
        markup_code = cls._format_markup(mface, mslice, mdir, '', mark, '')
        return move_code, markup_code
        
    @staticmethod
    def normal_to_intern_move(mface, mslice, mdir, size):
        if mface >= 3:
            return mface-3, size-mslice-1, 1-mdir
        else:
            return mface, mslice, mdir
    
    @classmethod
    def parse(cls, move_code, size):
        #debug('move_code: '+move_code)
        mface, mslice, mdir, err1, mark, err2 = cls.re_flubrd.match(move_code).groups()
        markup_code = cls._format_markup(mface, mslice, mdir, err1, mark, err2)
        mface = cls.map_face_flubrd.find(mface)
        mslice = int(mslice or 1) - 1
        mdir = int(bool(mdir))
        mark = bool(mark)
        if mface < 0 or mslice < 0 or mslice >= size:
            debug('Error parsing formula')
            return None, False, move_code
        # move finished, normalize it
        maxis, mslice, mdir = cls.normal_to_intern_move(mface, mslice, mdir, size)
        return MoveData(maxis, mslice, mdir), mark, markup_code

