#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009-2011  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Ported from GNUbik
# Original filename: glarea.c
# Original copyright and license:
#/*
#  GNUbik -- A 3 dimensional magic cube game.
#  Copyright (C) 2003, 2004  John Darrington
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License,  or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#*/


import OpenGL.error
from OpenGL import GLX as glx

import gobject
import gtk
from gtk import gdk
from gtk import gtkgl
from gtk import gdkgl
import gio

from .debug import *
import cube
import drwBlock
import glarea_common
import selection
from . import cursors
from . import textures
from .confstore import confstore, COLORED, IMAGED

RDRAG_COLOUR = 0
RDRAG_FILELIST = 1
# picture rate in ms. 40ms = 25 frames per sec
picture_rate = 40


class CubeArea (gobject.GObject):
    __gsignals__ = {
        'texture-changed': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, ()),
    }
    
    def __init__(self):
        gobject.GObject.__init__(self)
        
        self.widget = None
        self.last_mouse_x = -1
        self.last_mouse_y = -1
        self.button_down_cube = False
        self.button_down_background = False
        self.renderPending = False
        self.setcursorPending = False
        self.animation_in_progress = 0
        self.timer = None
        # the time for which the mouse must stay still, for anything to happen.
        self.idle_threshold = 10
        self.mouse_xy = -1, -1
        self.cursor_angle = 0.0
        # Structure to hold copy of the last selection taken or None
        self.current_selection = None
        self.enable_disable_selection_entered = 0
        self.stop_detected = False
        self.motion = False
        confstore.callbacks.append(self.on_confstore_changed)
        
    def is_item_selected(self):
        return self.current_selection is not None
        
    def init(self, widget, app):
        self.widget = widget
        
        # this is the number of frames drawn when a slice of the cube rotates 90 degrees
        self.frameQty = confstore.frameQty
        
        attribs = [ gdkgl.RGBA,
                    gdkgl.RED_SIZE,    1,
                    gdkgl.GREEN_SIZE,  1,
                    gdkgl.BLUE_SIZE,   1,
                    
                    gdkgl.DOUBLEBUFFER,
                    
                    gdkgl.DEPTH_SIZE , 1,
                    
                    gdkgl.ACCUM_RED_SIZE,  1,
                    gdkgl.ACCUM_GREEN_SIZE,  1,
                    gdkgl.ACCUM_BLUE_SIZE,  1,
                    
                    gdkgl.ATTRIB_LIST_NONE
                ]
                
        drwBlock.glutInit([])
        glconfig = gdkgl.Config(attribs)
        if not glconfig:
            raise Exception("Cannot get a suitable visual")
        gtkgl.widget_set_gl_capability(self.widget,
                                      glconfig,
                                      None,
                                      True,
                                      gdkgl.RGBA_TYPE)
                                      
        self.widget.connect("realize", self.on_realize)
        self.widget.connect("expose-event", self.on_expose_event)
        self.widget.connect("configure-event", self.on_resize)
        
        target = [ ("text/uri-list",       0, RDRAG_FILELIST),
                   ("application/x-color", 0, RDRAG_COLOUR), ]
        self.widget.drag_dest_set(gtk.DEST_DEFAULT_ALL, target, gdk.ACTION_COPY)
        self.widget.connect("drag-data-received", self.on_drag_data_received)
        
        self.widget.add_events(gdk.KEY_PRESS_MASK | gdk.BUTTON_PRESS_MASK
                               | gdk.BUTTON_RELEASE_MASK
                               # | gdk.BUTTON_MOTION_MASK
                               | gdk.ENTER_NOTIFY_MASK | gdk.LEAVE_NOTIFY_MASK
                               | gdk.VISIBILITY_NOTIFY_MASK
                               | gdk.POINTER_MOTION_MASK)
                               
        self.widget.connect("key-press-event", self.on_key_press_event)
        if DEBUG_KEYS:
            import testutils
            self.widget.connect("key-press-event", testutils.debug_keys, app)
            
        self.widget.connect("motion-notify-event", self.on_motion_notify_event)
        self.widget.connect("scroll-event", self.on_scroll_event)
        self.widget.connect("button-press-event", self.on_button_event)
        self.widget.connect("button-release-event", self.on_button_event)
        self.widget.connect("button-press-event", self.on_button_press_event, app)
        
        # initialise the selection mechanism
        self.timer = gobject.timeout_add(self.idle_threshold, self.on_timer_motion)
        # Add a handler to for all those occasions when we don't need the
        # selection mechanism going 
        self.widget.connect("enter-notify-event", self.on_enter_leave_notify_event)
        self.widget.connect("leave-notify-event", self.on_enter_leave_notify_event)
        self.widget.connect("visibility-notify-event", self.on_enter_leave_notify_event)
        self.widget.connect("unmap-event", self.on_enter_leave_notify_event)
        
    @staticmethod
    def set_the_colours():
        for i in xrange(6):
            color = gdk.Color(confstore.colors[i].color)
            facetype = confstore.colors[i].facetype
            if facetype == COLORED:
                pattern = confstore.colors[i].pattern
                texname = textures.stock_pattern[pattern].texName if pattern >= 0 else -1
            elif facetype == IMAGED:
                imagefile = confstore.colors[i].imagefile
                try:
                    pixbuf = textures.create_pixbuf_from_file(imagefile)
                    texname = textures.create_pattern_from_pixbuf(pixbuf)
                except Exception, err:
                    debug("Cannot create image from file {}: {}".format(imagefile, err))
                    texname = -1
            else:
                texname = -1
            drwBlock.face_rendering_set(i, red=color.red / 65535.0,
                                           green=color.green / 65535.0,
                                           blue=color.blue / 65535.0,
                                           facetype=facetype,
                                           texname=texname,
                                           distr=confstore.colors[i].imagemode)
            
    def re_initialize(self):
        glcontext = gtkgl.widget_get_gl_context(self.widget)
        gldrawable = gtkgl.widget_get_gl_drawable(self.widget)
        
        if not gldrawable.gl_begin(glcontext):
            debug("Cannot initialise gl drawable")
            return
            
        #TODO: Move the initialisation code somewhere else
        try:
            self.__texInit_done
        except AttributeError:
            textures.texInit()
            self.__texInit_done = True
        glarea_common.graphics_area_init(confstore.lighting)
        self.set_the_colours()
        gldrawable.gl_end()
        
    @staticmethod
    def apply_to_glmodel(model):
        assert model.dimension
        cube.set_cube_dimension(model.dimension)
        drwBlock.init_model()
        assert model.blocks
        cube.cube_set_blocks(model.blocks)
        
    def on_realize(self, unused_widget):
        self.re_initialize()
        
    def on_expose_event(self, unused_widget, unused_event):
        self.render_idle()
        
    def on_resize(self, widget, event):
        height = event.height
        width = event.width
        glcontext = gtkgl.widget_get_gl_context(widget)
        gldrawable = gtkgl.widget_get_gl_drawable(widget)
        
        #TODO ???
        #if not gdkgl.Drawable.gl_begin(gldrawable, glcontext):
        #    return False
        if not gdkgl.Drawable.make_current(gldrawable, glcontext):
            debug("Cannot set gl drawable current")
            return
            
        try:
            glx.glXWaitGL()
            glx.glXWaitX()
        except OpenGL.error.GLError:
            pass
        glarea_common.resize(width, height)
        
        return False
        
    def on_key_press_event(self, unused_widget, event):
        shifted = 0
        if event.state & gdk.SHIFT_MASK:
            shifted = 1
        return self.do_keysyms(event.keyval, shifted)
        
    def do_keysyms(self, keysym, shifted):
        if keysym == gtk.keysyms.Right:
            if shifted:
                dir_ = 0
                axis = 2
            else:
                dir_ = 1
                axis = 1
        elif keysym == gtk.keysyms.Left:
            if shifted:
                dir_ = 1
                axis = 2
            else:
                dir_ = 0
                axis = 1
        elif keysym == gtk.keysyms.Up:
            axis = 0
            dir_ = 0
        elif keysym == gtk.keysyms.Down:
            axis = 0
            dir_ = 1
        else:
            return True
            
        self.rotate_cube(axis, dir_)
        return True
        
    def update_selection(self):
        '''This func determines which block the mouse is pointing at'''
        self.current_selection = selection.pick_polygons(self.widget, confstore.lighting, *self.mouse_xy)
        if self.is_item_selected():
            cs = self.current_selection
            self.cursor_angle = cube.get_cursor_angle(cs.block, cs.face, cs.quadrant)
        self.set_cursor()

    def on_motion_notify_event(self, unused_widget, event):
        self.mouse_xy = event.x, event.y
        self.motion = True
        self.stop_detected = False
        
        if not self.button_down_background:
            return False
            
        xmotion = 0
        ymotion = 0
        if self.last_mouse_x >= 0:
            xmotion = event.x - self.last_mouse_x
        if self.last_mouse_y >= 0:
            ymotion = event.y - self.last_mouse_y
            
        self.last_mouse_x = event.x
        self.last_mouse_y = event.y
        
        if ymotion > 0: self.rotate_cube(0, 1)
        if ymotion < 0: self.rotate_cube(0, 0)
        if xmotion > 0: self.rotate_cube(1, 1)
        if xmotion < 0: self.rotate_cube(1, 0)
        return False
        
    def on_enter_leave_notify_event(self, unused_widget, event):
        ''' When the window is not mapped, kill the selection mechanism.
        It wastes processor time'''
        # This is a kludge to work around a rather horrible bug;  for some
        # reason,  some  platforms emit a EnterNotify and LeaveNotify (in
        # that order) when animations occur.  This workaround makes sure
        # that the window is not `entered twice' 
        if event.type == gdk.ENTER_NOTIFY:
            self.enable_disable_selection_entered += 1
            if self.timer is None:
                self.timer = gobject.timeout_add(self.idle_threshold, self.on_timer_motion)
        elif event.type == gdk.LEAVE_NOTIFY:
            self.update_selection()
            self.enable_disable_selection_entered -= 1
            if self.enable_disable_selection_entered <= 0:
                if self.timer is not None:
                    gobject.source_remove(self.timer)
                    self.timer = None
        return False

    def on_timer_motion(self):
        '''This callback occurs at regular intervals. The period is determined by
        idle_threshold.  It checks to see if the mouse has moved,  since the last
        call of this function.
        Post-condition:  motion is False.'''
        if self.motion == False: # if not moved since last time 
            if not self.stop_detected:
                # in here,  things happen upon the mouse stopping 
                self.stop_detected = True
                self.update_selection()
        self.motion = False
        return True

    def on_scroll_event(self, unused_widget, event):
        '''Rotate the cube about the z axis (relative to the viewer)'''
        self.rotate_cube(2, not event.direction)
        return False
        
    def on_button_event(self, unused_widget, event):
        # buttons 4 and 5 mean mouse wheel scrolling
        if event.button == 4:
            self.rotate_cube(2, 1)
        if event.button == 5:
            self.rotate_cube(2, 0)
        if event.button != 1:
            return False
            
        if event.type == gdk.BUTTON_PRESS:
            if self.is_item_selected():
                self.button_down_cube = True
            else:
                self.button_down_background = True
            # disable selection
            if self.timer is not None:
                gobject.source_remove(self.timer)
                self.timer = None
        elif event.type == gdk.BUTTON_RELEASE:
            # enable selection
            if self.timer is None:
                self.timer = gobject.timeout_add(self.idle_threshold, self.on_timer_motion)
            self.button_down_cube = False
            self.button_down_background = False
        return False
        
    def on_button_press_event(self, widget, event, app):
        if event.type == gdk.BUTTON_PRESS:
            widget.grab_focus()
            self.do_mouse_event(event.button, app)
        return True
        
    def do_mouse_event(self, button, application):
        '''handle mouse clicks'''
        # Don't let a user make a move,  whilst one is already in progress,
        # otherwise the cube falls to bits.
        if self.animation_in_progress:
            return
            
        if button == 1:
            # Make a move
            if self.is_item_selected():
                cs = self.current_selection
                application.current_movement.request_rotation(cs.axis_o, cs.slice_o, cs.dir_o)
                application.update_statusbar()
                move_data = application.current_movement.request_play()
                if move_data:
                    self.animate_rotation(application, move_data)
            self.render()
            
    def on_drag_data_received(self, unused_widget, unused_drag_context,
                                x, y, selection_data, info, unused_timestamp):
        # The special value -1 means that this callback must
        # use pick_polygons to find out which face is to be updated
        #FIXME Redefining name ... ; anderen Namen verwenden
        fs = selection.pick_polygons(self.widget, confstore.lighting, x, y)
        if not fs:  # If nothing was actually pointed to,  then exit
            return
        swatch = fs.face
        
        if info == RDRAG_COLOUR:
            #FIXME lenght and format available in PyGTK 2.14 and above.
            #if selection_data.length < 0:
            #    return
            #if (selection_data.format != 16) or (selection_data.length != 8):
            #    debug("Received invalid color data")
            #    return
            if len(selection_data.data) != 8:
                debug("Received invalid color data")
                return
                
            vals = [ord(selection_data.data[0]) | ord(selection_data.data[1]) << 8,
                    ord(selection_data.data[2]) | ord(selection_data.data[3]) << 8,
                    ord(selection_data.data[4]) | ord(selection_data.data[5]) << 8]
            colour = gdk.Color()
            colour.red = vals[0]
            colour.green = vals[1]
            colour.blue = vals[2]
            
            drwBlock.face_rendering_set(swatch,
                                            red=vals[0] / 65535.0,
                                            green=vals[1] / 65535.0,
                                            blue=vals[2] / 65535.0,
                                            facetype=COLORED)
                                            
            confstore.set_color(swatch, colour)
            self.emit('texture-changed')
            self.render()
                
        elif info == RDRAG_FILELIST:
            start = selection_data.data.split("\r\n")
            for s in start:
                if s == "":
                    continue
                    
                fileobj = gio.file_parse_name(s)
                filename = gio.file_parse_name(s).get_path()
                if not fileobj.query_exists():
                    debug('filename "%s" not found.' % filename)
                    continue
                    
                confstore.set_image_filename(swatch, filename)
                self.emit('texture-changed')
                    
                # For now,  just use the first one.
                # Later,  we'll add some method for disambiguating multiple files
                break
                
            self.render()
        # Ignore all others
        
    def set_cursor(self):
        if not self.setcursorPending:
            gobject.idle_add(self.set_cursor_idle)
            self.setcursorPending = 1
            
    def set_cursor_idle(self):
        fg = gdk.Color(65535, 65535, 65535) # White.
        bg = gdk.Color(    0,     0,     0) # Black.
        
        if self.is_item_selected() and not self.button_down_background:
            data_bits, mask_bits, height, width, hot_x, hot_y = cursors.get_cursor(self.cursor_angle)
            source = gdk.bitmap_create_from_data(None, data_bits, width, height)
            mask = gdk.bitmap_create_from_data(None, mask_bits, width, height)
            cursor = gdk.Cursor(source, mask, fg, bg, hot_x, hot_y)
        else:
            cursor = gdk.Cursor(gdk.CROSSHAIR)
        self.widget.window.set_cursor(cursor)
        
        self.setcursorPending = 0
        return False
        
    def render(self):
        if not self.renderPending:
            gobject.idle_add(self.render_idle)
            self.renderPending = 1
            
    def render_idle(self):
        # Reset the bit planes, and render the scene
        glcontext = gtkgl.widget_get_gl_context(self.widget)
        gldrawable = gtkgl.widget_get_gl_drawable(self.widget)
        
        if not gdkgl.Drawable.make_current(gldrawable, glcontext):
            debug("Cannot set gl drawable current")
            return False
            
        glarea_common.display()
        gdkgl.Drawable.swap_buffers(gldrawable)
        
        self.renderPending = 0
        return False
        
    def set_lighting(self, enable):
        glarea_common.set_lighting(enable)
        self.render()
        
    def rotate_cube(self, axis, dir_):
        '''Rotate cube about axis (screen relative)'''
        glarea_common.rotate_cube(axis, dir_)
        self.render()
        
    ### Animation
    
    def animate_rotation(self, application, move_data):
        self.animation_in_progress = 1
        drwBlock.start_animate(move_data.axis, move_data.slice, move_data.dir)
        blocks = list(application.current_movement.current_cube_state.identify_rotation_blocks(
                                move_data.axis, move_data.slice))
        assert blocks
        cube.set_animation_blocks(blocks)
        application.playbarstate.playing(application.current_movement.mark_before)
        gobject.timeout_add(picture_rate, self._on_animate, application)
        
    def _on_animate(self, application):
        # how many degrees motion per frame
        increment = 90.0 / (self.frameQty + 1)
        
        # decrement the current angle
        unfinished = drwBlock.step_animate(increment)
        
        # and redraw it
        self.render()
        
        if unfinished and not application.current_movement.abort_requested:
            # call this timeout again instead of return True; picture_rate may changed
            gobject.timeout_add(picture_rate, self._on_animate, application)
        else:
            # we have finished the animation sequence now
            drwBlock.end_animate()
            self.apply_to_glmodel(application.current_movement.current_cube_state)
            self.animation_in_progress = 0
            self.update_selection()
            application.current_movement.abort_requested = 0
            
            if not application.current_movement.stop_requested:
                application.update_statusbar()
                move_data = application.current_movement.request_play()
                if move_data:
                    # go again
                    self.animate_rotation(application, move_data)
                    return False
                    
            # no more animations
            application.current_movement.stop_requested = 0
            application.update_ui()
        return False
        
    def on_confstore_changed(self, key, *subkeys):
        def set_color(i):
            color = gdk.Color(confstore.colors[i].color)
            #confstore.colors[i].facetype = COLORED
            drwBlock.face_rendering_set(i, red=color.red / 65535.0,
                                           green=color.green / 65535.0,
                                           blue=color.blue / 65535.0,
                                           #facetype=COLORED,
                                           #texname=-1
                                       )
        def set_pattern(i):
            pattern = confstore.colors[i].pattern
            texname = textures.stock_pattern[pattern].texName if pattern >= 0 else -1
            #confstore.colors[i].facetype = COLORED
            drwBlock.face_rendering_set(i, texname=texname)
        def set_imagefile(i):
            imagefile = confstore.colors[i].imagefile
            #debug('set_imagefile',i,imagefile)
            try:
                pixbuf = textures.create_pixbuf_from_file(imagefile)
                texname = textures.create_pattern_from_pixbuf(pixbuf)
            except Exception, err:
                debug(_("Cannot create image from file {filename}: {error}").format(
                                filename=imagefile, error=err))
            else:
                #debug('image tex',texname)
                drwBlock.face_rendering_set(i, texname=texname)
        
        if key == 'frameQty':
            self.frameQty = confstore.frameQty
        elif key == 'lighting':
            self.set_lighting(confstore.lighting)
        elif key == 'dimension':
            pass
        elif key == 'colors':
            i = int(subkeys[0])
            if subkeys[1] == 'color':
                set_color(i)
            elif subkeys[1] == 'pattern':
                set_pattern(i)
            elif subkeys[1] == 'imagefile':
                set_imagefile(i)
            elif subkeys[1] == 'facetype':
                facetype = confstore.colors[i].facetype
                if facetype == COLORED:
                    set_pattern(i)
                elif facetype == IMAGED:
                    set_imagefile(i)
                drwBlock.face_rendering_set(i, facetype=facetype)
            elif subkeys[1] == 'imagemode':
                imagemode = confstore.colors[i].imagemode
                drwBlock.face_rendering_set(i, distr=imagemode)
                #set_imagefile(i)
            else:
                debug('Unknown conf value changed:', subkeys[1])
            self.render()
        else:
            debug('Unknown conf value changed:', key)
        
        
