/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file status_layer.hpp
 * \brief The status layer contains the area where are printed the score,
 *        energy, remaining lifes of the players.
 * \author Julien Jorge
 */
#ifndef __PTB_STATUS_LAYER_HPP__
#define __PTB_STATUS_LAYER_HPP__

#include "engine/layer/gui_layer.hpp"
#include "visual/animation.hpp"
#include "visual/writing.hpp"
#include "universe/derived_item_handle.hpp"
#include "ptb/item/plee/plee.hpp"

namespace ptb
{
  /**
   * \brief The status layer contains the area where are printed the score,
   *        energy, remaining lifes of the players.
   */
  class status_layer:
    public bear::engine::gui_layer
  {
  public:
    /** \brief The type of a list of scene elements retrieved from the layer. */
    typedef bear::engine::gui_layer::scene_element_list scene_element_list;

  private:
    /**
     * \brief A horizontal bar (energy).
     */
    class horizontal_bar
    {
    public:
      horizontal_bar
      ( bear::engine::level_globals& glob,
        const bear::visual::rectangle_type& level_sprite,
        const bear::visual::rectangle_type& loss_sprite,
        const std::string& icon_name, unsigned int length );

      void set_length(unsigned int length);
      unsigned int length() const;
      unsigned int width() const;
      unsigned int height() const;

      void set_level( double lev );
      void progress( bear::universe::time_type elapsed_time );
      void render
      ( scene_element_list& e, const bear::visual::position_type& pos );

    private:
      /** \brief The icon displayed on the bar. */
      bear::visual::sprite m_icon;

      /** \brief The sprite displaying the level of the bar. */
      bear::visual::sprite m_level;

      /** \brief The sprite displaying the amount of loss. */
      bear::visual::sprite m_loss;

      /** \brief Sprite displayed over the ends of the bars. */
      bear::visual::sprite m_tube_clamp;

      /** \brief Sprite displayed over the bar. */
      bear::visual::sprite m_glass_tube;

      /** \brief The current level of the bar. */
      double m_level_value;

    }; // class horizontal_bar

    /**
     * \brief A vertical bar (oxygen, ...).
     */
    class vertical_bar
    {
    public:
      vertical_bar
      ( bear::engine::level_globals& glob,
        const bear::visual::rectangle_type& level_sprite,
        const bear::visual::rectangle_type& icon_sprite, unsigned int length );

      unsigned int width() const;
      unsigned int height() const;

      void set_level( double lev );
      void progress( bear::universe::time_type elapsed_time );
      void render
      ( scene_element_list& e, const bear::visual::position_type& pos );

    private:
      /** \brief The icon displayed on the bar. */
      bear::visual::sprite m_icon;

      /** \brief The sprite displaying the level of the bar. */
      bear::visual::sprite m_level;

      /** \brief Sprite displayed over the ends of the bars. */
      bear::visual::sprite m_tube_clamp;

      /** \brief Sprite displayed over the bar. */
      bear::visual::sprite m_glass_tube;

      /** \brief The current level of the bar. */
      double m_level_value;

      /** \brief The current loss of the bar. */
      double m_loss_value;

    }; // class vertical_bar

    /**
     * \brief All the informations we want to show about a player.
     */
    class player_status
    {
    public:
      player_status
      ( bear::engine::level_globals& glob,
        const bear::visual::font& f,
        bear::universe::const_derived_item_handle_maker<plee>::handle_type& p);

      void progress( bear::engine::level_globals& glob, 
                     bear::universe::time_type elapsed_time );
    private:
      bool update_powers(unsigned int index);
       void update_tries( bear::engine::level_globals& glob,
			  bear::universe::time_type elapsed_time,
			  unsigned int index);
      void update_stone_animation( bear::engine::level_globals& glob,
				   unsigned int index);
      void progress_gauge
      ( vertical_bar& gauge, bool is_active, double value, double max_value,
        bool& new_active, bear::visual::coordinate_type& offset_x);

    public:
      /** \brief Level of energy. */
      horizontal_bar energy;

      /** \brief Level of oxygen. */
      vertical_bar oxygen;

      /** \brief The fire gauge. */
      vertical_bar fire_gauge;

      /** \brief The ice gauge. */
      vertical_bar ice_gauge;

      /** \brief The score. */
      bear::visual::writing score;

      /** \brief Number of stone. */
      bear::visual::writing stones;

      /** \brief Number of remaining tries. */
      bear::visual::writing tries;

      /** \brief Last number of remaining tries. */
      unsigned int m_last_tries;

      /** \brief The time since the last trie. */
      bear::universe::time_type m_tries_time;

      /** \brief Pointer to the player. */
      bear::universe::const_derived_item_handle_maker<plee>::handle_type
      player;

      /** \brief Indicates if the oxygen gauge is activated. */
      bool oxygen_active;

      /** \brief The offset of the oxygen gauge. */
      bear::visual::coordinate_type oxygen_offset_x;

      /** \brief Indicates if the fire gauge is activated. */
      bool fire_gauge_active;

      /** \brief The offset of the fire gauge. */
      bear::visual::coordinate_type fire_gauge_offset_x;

      /** \brief Indicates if the ice gauge is activated. */
      bool ice_gauge_active;

      /** \brief The offset of the ice gauge. */
      bear::visual::coordinate_type ice_gauge_offset_x;

      /** \brief The current animation of stone. */
      bear::visual::animation stone_animation;

    private:
      /** \brief The air power status in the last iteration. */
      bool m_air_power;

      /** \brief The fire power status in the last iteration. */
      bool m_fire_power;

      /** \brief The water power status in the last iteration. */
      bool m_water_power;

      /** \brief The length of the bars. */
      const static double s_bar_length;

    }; // class player_status

  public:
    status_layer();
    virtual ~status_layer();

    void build();
    void progress( bear::universe::time_type elapsed_time );
    void render( scene_element_list& e ) const;

  private:
    void progress_time( bear::universe::time_type elapsed_time );
    void progress_corrupting_bonus( bear::universe::time_type elapsed_time );

    void render_player1( scene_element_list& e ) const;
    void render_player2( scene_element_list& e ) const;
    void render_corrupting_bonus( scene_element_list& e ) const;
    void render_azelnut( scene_element_list& e ) const;
    void render_honeypots( scene_element_list& e ) const;
    void search_players();
    
  private:
    /** \brief Data of the first player. */
    player_status* m_data_1;

    /** \brief Data of the second player. */
    player_status* m_data_2;

    /** \brief The first player. */
    bear::universe::const_derived_item_handle_maker<plee>::handle_type
    m_first_player;

    /** \brief The second player. */
    bear::universe::const_derived_item_handle_maker<plee>::handle_type
    m_second_player;

    /** \brief The component in which we show the time. */
    bear::visual::writing m_text_time;

    /** \brief Indicate if the time must be displayed. */
    bool m_time_on;
    
    /** \brief A small Plee next to the remaining tries. */
    bear::visual::sprite m_small_plee;

    /** \brief Animation for corrupting bonus. */
    bear::visual::animation* m_corrupting_bonus;

    /** \brief The component in which we show the corrupting bonus count. */
    bear::visual::writing m_text_corrupting_bonus;

    /** \brief Animation for azelnut. */
    bear::visual::animation* m_azelnut;

    /** \brief Animation for honeypots. */
    bear::visual::sprite m_honeypot;

    /** \brief Distance between the elements of the layer. */
    const static unsigned int s_margin;

  }; // class status_layer
} // namespace ptb

#endif // __PTB_STATUS_LAYER_HPP__
