/*
    Bear Engine - Editor library

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/impl/item_instance_field_node.tpp
 * \brief Implementation of the template methods of the
 *        bf::xml::item_instance_field_node class.
 * \author Julien Jorge
 */

#include "bf/item_instance.hpp"
#include "bf/stream_conv.hpp"
#include "bf/xml/exception.hpp"
#include "bf/xml/value_to_xml.hpp"
#include "bf/wx_facilities.hpp"
#include "bf/xml/xml_to_value.hpp"
#include "bf/xml/reader_tool.hpp"

#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the value of a field of simple type.
 * \param item The item in which we save the value.
 * \param field_name The name of the field to load.
 * \param node_name The name of the xml node.
 * \param node The node to parse.
 */
template<typename Type>
void bf::xml::item_instance_field_node::load_value
( item_instance& item, const std::string& field_name,
  const std::string& node_name, const wxXmlNode* node ) const
{
  bool found = false;
  Type v;
  const wxString wx_node_name = std_to_wx_string(node_name);

  node = reader_tool::skip_comments(node);

  if ( node == NULL )
    throw xml::missing_node( node_name );

  xml::xml_to_value<Type> xml_conv;

  // we set all values found so the field has the value of the last one.
  while ( node!=NULL )
    {
      if ( node->GetName() == wx_node_name )
        {
          found = true;
          xml_conv( v, node );
        }
      else
        claw::logger << claw::log_warning << "Ignored node '"
                     << wx_to_std_string(node->GetName()) << "'" << std::endl;

      node = reader_tool::skip_comments(node->GetNext());
    }

  if ( !found )
    throw xml::missing_node( node_name );

  item.set_value( field_name, v );
} // item_instance_field_node::load_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the value of a field of type list of simple type.
 * \param item The item in which we save the value.
 * \param field_name The name of the field to load.
 * \param node_name The name of the xml node.
 * \param node The node to parse.
 */
template<typename Type>
void bf::xml::item_instance_field_node::load_value_list
( item_instance& item, const std::string& field_name,
  const std::string& node_name, const wxXmlNode* node ) const
{
  std::list<Type> v;
  const wxString wx_node_name = std_to_wx_string(node_name);

  xml::xml_to_value<Type> xml_conv;

  node = reader_tool::skip_comments(node);

  while ( node!=NULL )
    {
      if ( node->GetName() == wx_node_name )
        {
          Type tmp;
          xml_conv( tmp, node );
          v.push_back(tmp);
        }
      else
        claw::logger << claw::log_warning << "Ignored node '"
                     << wx_to_std_string(node->GetName()) << "'" << std::endl;

      node = reader_tool::skip_comments(node->GetNext());
    }

  item.set_value( field_name, v );
} // item_instance_field_node::load_value_list()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of simple type.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 * \param node_name The name of the xml node.
 */
template<typename Type>
void bf::xml::item_instance_field_node::save_value
( std::ostream& os, const std::string& field_name,
  const item_instance& item, const std::string& node_name ) const
{
  Type v;
  item.get_value( field_name, v );
  xml::value_to_xml<Type>::write( os, node_name, v );
} // item_instance_field_node::save_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the value of a field of list of simple type.
 * \param os The stream in which we save the value.
 * \param field_name The name of the field to save.
 * \param item The item in which we take the value.
 * \param node_name The name of the xml node.
 */
template<typename Type>
void bf::xml::item_instance_field_node::save_value_list
( std::ostream& os, const std::string& field_name,
  const item_instance& item, const std::string& node_name ) const
{
  std::list<Type> v;
  typename std::list<Type>::const_iterator it;

  item.get_value( field_name, v );

  for (it=v.begin(); it!=v.end(); ++it)
    xml::value_to_xml<Type>::write( os, node_name, *it );
} // item_instance_field_node::save_value_list()
