/*
    Bear Engine - Editor library

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/path_configuration.hpp
 * \brief A singleton that stores the paths used in the programs.
 * \author Julien Jorge
 */
#ifndef __BF_PATH_CONFIGURATION_HPP__
#define __BF_PATH_CONFIGURATION_HPP__

#include "bf/libeditor_export.hpp"

#include <string>
#include <list>
#include <claw/basic_singleton.hpp>

namespace bf
{
  /**
   * \brief A singleton that stores the paths used in the programs.
   * \author Julien Jorge
   */
  class BEAR_EDITOR_EXPORT path_configuration:
    public claw::concept::basic_singleton<path_configuration>
  {
  public:
    path_configuration();

    void save() const;

    std::string get_config_directory() const;

    bool get_full_path( std::string& p ) const;
    bool expand_file_name( std::string& p ) const;
    bool expand_file_name( std::string& p, std::size_t m ) const;
    bool get_relative_path( std::string& p ) const;

  private:
    void load();

    bool create_config_directory() const;
    bool create_config_file() const;

    bool find_random_file_name( std::string& name, std::size_t m ) const;
    
    void find_all_files_in_dir
    ( const std::string& dirname, const char* pattern, std::size_t offset,
      std::size_t m, std::list<std::string>& result ) const;
    bool glob_match( const char* pattern, const char* text ) const;
    bool glob_potential_match( const char* pattern, const char* text ) const;

  public:
    /** \brief Path to the directory containing the XML files for item
        classes. */
    std::list<std::string> item_class_path;

    /** \brief Path to the data directory of the game. */
    std::list<std::string> data_path;

  private:
    /** \brief The name of the configuration file. */
    static const std::string s_config_file_name;

    /** \brief The path to the directory containing the configuration file,
        relative to the user's directory. */
    static const std::string s_config_directory;

    /** \brief Character written on the left of a section name in the
        configuration file. */
    static const char s_section_left;

    /** \brief Character written on the right of a section name in the
        configuration file. */
    static const char s_section_right;

    /** \brief Character used for comments in the configuration file. */
    static const char s_comment;

    /** \brief Character used for assigning a value to a field. */
    static const char s_field_assign;

    /** \brief The field of the directory of XML files for item classes. */
    static const std::string s_items_directory_field;

    /** \brief The field of the directory of the path of datas. */
    static const std::string s_data_directory_field;

  }; // class path_configuration
} // namespace bf

#endif // __BF_PATH_CONFIGURATION_HPP__
