/* OpenVAS Client
 * Copyright (C) 1998, 1999 Renaud Deraison
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <includes.h>
#include "report_utils.h"

/*-------------------------------------------------------------------*/
/** @todo replace by g_strcmp0 */
int
safe_strcmp (char* a, char* b)
{
 if(!a && !b)
  return 0;
 if(!a)
  return 1;
 else if(!b)
  return -1;
 else return strcmp(a, b);
}



 /*-------------------------------------------------------------------*/
int arglist_length(struct arglist * arg)
{
 if(!arg)return 0;
 return arg->next ? 1 + arglist_length(arg->next):0;
}


/*
 * Auxilliary functions
 */
/**
 * @brief Recursively counts number of log messages by port.
 * 
 * @return Number of log messages or 0 if host argument is empty or NULL.
 */
int
number_of_log_messages_by_port (struct arglist * arg)
{
  if (!arg || !arg->next)
    return 0;
  else
    {
      struct arglist * logs = arg_get_value (arg->value, "LOG");
      return arglist_length(logs) + number_of_log_messages_by_port(arg->next);
    }
}

/**
 * @brief Recursively counts number of log messages by host.
 * 
 * @return Number of log messages or 0 if host argument is empty or NULL.
 */
int
number_of_log_messages_by_host (struct arglist * hosts)
{
  if(!hosts || !hosts->next)
    return 0;
  else
    return number_of_log_messages_by_port(hosts->value)
            + number_of_log_messages_by_host(hosts->next);
}

int number_of_false_positives_by_port(struct arglist * arg)
{
 if(!arg || !arg->next)return 0;
 else {
  struct arglist * notes = arg_get_value(arg->value, "FALSE");
  return arglist_length(notes) + number_of_false_positives_by_port(arg->next);
  }
}

int number_of_false_positives_by_host(struct arglist * arg)
{
 if(!arg || !arg->next)return 0;
 else return number_of_false_positives_by_port(arg->value) +
  	     number_of_false_positives_by_host(arg->next);
}

int number_of_notes_by_port(struct arglist * arg)
{
 if(!arg || !arg->next)return 0;
 else {
  struct arglist * notes = arg_get_value(arg->value, "NOTE");
  return arglist_length(notes) + number_of_notes_by_port(arg->next);
  }
}

int number_of_notes_by_host(struct arglist * arg)
{
 if(!arg || !arg->next)return 0;
 else return number_of_notes_by_port(arg->value) +
  	     number_of_notes_by_host(arg->next);
}

int number_of_warnings_by_port(struct arglist * arg)
{
 if(!arg || !arg->next)return 0;
 else {
  struct arglist * warnings = arg_get_value(arg->value, "INFO");
  return arglist_length(warnings) + number_of_warnings_by_port(arg->next);
  }
}

int number_of_warnings_by_host(struct arglist * arg)
{
 if(!arg || !arg->next)return 0;
 else return number_of_warnings_by_port(arg->value) +
  	     number_of_warnings_by_host(arg->next);
}


int number_of_holes_by_port(struct arglist * arg)
{
 if(!arg || !arg->next)return 0;
 else {
  struct arglist * holes = arg_get_value(arg->value, "REPORT");
  return arglist_length(holes) + number_of_holes_by_port(arg->next);
  }
}

int number_of_holes_by_host(struct arglist * arg)
{
 if(!arg || !arg->next)return 0;
 else return number_of_holes_by_port(arg->value) +
  	     number_of_holes_by_host(arg->next);
}

/**
 * Recursively counts up number of log messages for each host in parameter
 * arglist hosts.
 * 
 * @return Number of log messages or 0 if parameter hosts is NULL or empty.
 */
int
number_of_log_messages (struct arglist* hosts)
{
  if (!hosts || !hosts->next)
    return 0;
  else return number_of_log_messages_by_host (hosts->value)
               + number_of_log_messages (hosts->next);
}

int number_of_false_positives(struct arglist *  arg)
{
 if(!arg || !arg->next)return 0;
 else return number_of_false_positives_by_host(arg->value) +
 	     number_of_false_positives(arg->next);
}

int number_of_notes(struct arglist *  arg)
{
 if(!arg || !arg->next)return 0;
 else return number_of_notes_by_host(arg->value) +
 	     number_of_notes(arg->next);
}

int number_of_warnings(struct arglist *  arg)
{
 if(!arg || !arg->next)return 0;
 else return number_of_warnings_by_host(arg->value) +
 	     number_of_warnings(arg->next);
}

int number_of_holes(struct arglist * arg)
{
if(!arg ||!arg->next)return 0;
 else return number_of_holes_by_host(arg->value) +
 	     number_of_holes(arg->next);
}

/*
 * The most dangerous host
 */
static struct arglist*
sub_most_dangerous_host_holes (struct arglist* host, struct arglist* current,
                               int number)
{
 if(!host || !host->next)return current;
 else 
 {
  int holes = number_of_holes_by_host(host->value);
  if(holes > number) return sub_most_dangerous_host_holes(host->next, host,
  							   holes);
	else return sub_most_dangerous_host_holes(host->next, current, number);
 }
}

static struct arglist*
sub_most_dangerous_host_warnings (struct arglist *host, struct arglist *current,
                                   int number)
{
 if(!host || !host->next)return current;
 else 
 {
  int holes = number_of_warnings_by_host(host->value);
  if(holes > number) return sub_most_dangerous_host_warnings(host->next, host,
  							   holes);
	else return sub_most_dangerous_host_warnings(host->next, current, number);
 }
}


static struct arglist*
sub_most_dangerous_host_notes(struct arglist * host, struct arglist* current,
                              int number)
{
 if (!host || !host->next)
   return current;

 else
 {
  int holes = number_of_notes_by_host(host->value);
  if(holes > number) return sub_most_dangerous_host_notes(host->next, host,
  							   holes);
	else return sub_most_dangerous_host_notes(host->next, current, number);
 }
}

struct arglist *
most_dangerous_host (struct arglist * hosts)
{
 struct arglist * most = sub_most_dangerous_host_holes(hosts, NULL, 0);
 if(most)return most;
 most = sub_most_dangerous_host_warnings(hosts, NULL, 0);
 if(most)return most;
 else return sub_most_dangerous_host_notes(hosts, NULL, 0);
}
