/* $Id: prefs_scope_tree.c,v 1.17 2007-03-01 22:58:29 jan Exp $
 *
 * Copyright (C) 2004 by Greenbone Networks GmbH
 * Author(s):
 * Thomas Arendsen Hein <thomas@intevation.de>
 *
 * This program is free software under the GNU GPL (>=v2)
 * Read the file COPYING coming with the software for details.
 */

#ifdef USE_OMP
#include <assert.h>
#include <stdio.h>
#include <openvas/omp/omp.h> /* for omp_create_task_rc */
#include <openvas/omp/xml.h> /* for entity_t */
#include <openvas_server.h>
#endif

#include <openvas/base/severity_filter.h> /* for severity_filter_apply */

#include <includes.h>
#include "openvas_i18n.h"

#ifdef USE_GTK
#include <gtk/gtk.h>
#include "globals.h"
#include "openvas-client.h"
#include "context.h"
#include "preferences.h"
#include "prefs_context.h"
#include "prefs_dialog.h"
#include "prefs_dialog_auth.h"
#include "prefs_report.h"
#include "prefs_scope_tree.h"
#include "error_dlg.h"
#include "report.h"
#include "report_utils.h"
#include "backend.h"
#include "treeview_support.h"

/**
 * @file
 * Handle the scope- tree (in GUI the area to the left where Global Settings,
 * taks and scopes are organized in a tree).
 * Scopes and Global settings have an icon, showing its connection state.
 * The scope- tree also acts as a table, allowing an overview over how many
 * security notes, warnings, holes and false positives were reported in a
 * "report"- scope.
 *
 * The counted numbers for warnings, holes etc are stored in a separate .cnt
 * file to avoid recounting.
 * The cnt file is placed next to the location of the .nbe backend file (usually
 * in the contexts directory).
 */

/* temporary, should be part of the global context (from severity_filter.c) */
extern severity_filter_t * global_filter;

/**
 * @brief Indices of the Scope- treeview.
 *
 * Will be shown in the tree of scopes.
 */
enum
{
  COL_CONTEXT = 0,
  COL_NAME,  /**< Name of the task/scope/report. */
  COL_NOTE,  /**< Nr. of "security notes" reported. */
  COL_WARN,  /**< Nr. of warnings reported. */
  COL_HOLE,  /**< Nr. of holes reported. */
  COL_FALSE, /**< Nr. of false positives reported. */
  COL_LOG,   /**< Nr. of log messages received. */
  COL_EDITABLE,
  NUM_COLS
};

#ifdef USE_OMP
static int scopetree_refresh (context_type, int);

static void scopetree_delete_recurse (struct context *);

static int
ensure_context_connected (struct context *context, gboolean writable_settings)
{
  if (context->protocol == PROTOCOL_OMP)
    {
      int ret;
      gboolean previous = prefs_dialog_auth_writable_settings;

      prefs_dialog_auth_writable_settings = writable_settings;
      ret = prefs_dialog_auth_omp_connection (context);
      prefs_dialog_auth_writable_settings = previous;
      if (!ret)
        return -1;
    }
  return 0;
}
#endif /* USE_OMP */

void
scopetree_save_treerowref (struct context* context, GtkTreeModel *model,
                           GtkTreeIter iter)
{
  GtkTreePath *path;

  gtk_tree_row_reference_free(context->treerowref);
  path = gtk_tree_model_get_path(model, &iter);
  context->treerowref = gtk_tree_row_reference_new(model, path);
}


void
scope_move_menuitem_enable (struct context *context, gboolean enable)
{
  context = context_by_type(context, CONTEXT_TASK);
  if (context && context->move_menuitem && GTK_IS_WIDGET(context->move_menuitem))
      gtk_widget_set_sensitive(context->move_menuitem, enable);
}

/**
 * @brief Moves a context under a new parent.
 */
void
scopetree_move (struct context* context, struct context* new_parent)
{
  GtkWidget *scopetree = GTK_WIDGET(
      arg_get_value(arg_get_value(MainDialog, "SCOPETREE"), "TREEVIEW"));
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(scopetree));
  GtkTreeRowReference *old_treerowref;
  GtkTreePath *path;
  GtkTreeIter iter, parent;
  struct context *next;
  const char *copy_name;
  int copy_note, copy_warn, copy_hole, copy_false, copy_log;
  gboolean copy_editable;

  /* Read old context */
  old_treerowref = context->treerowref;
  context->treerowref = NULL;
  path = gtk_tree_row_reference_get_path(old_treerowref);
  gtk_tree_model_get_iter(model, &iter, path);
  gtk_tree_path_free(path);
  gtk_tree_model_get (model, &iter,
                      COL_NAME, &copy_name,
                      COL_HOLE, &copy_hole,
                      COL_WARN, &copy_warn,
                      COL_NOTE, &copy_note,
                      COL_FALSE, &copy_false,
                      COL_LOG, &copy_log,
                      COL_EDITABLE, &copy_editable,
                      -1);

  /* Create the target context */
  path = gtk_tree_row_reference_get_path(new_parent->treerowref);
  gtk_tree_model_get_iter(model, &parent, path);
  gtk_tree_path_free(path);
  gtk_tree_store_append(GTK_TREE_STORE(model), &iter, &parent);
  gtk_tree_store_set (GTK_TREE_STORE(model), &iter,
                      COL_CONTEXT, context,
                      COL_NAME, copy_name,
                      COL_HOLE, copy_hole,
                      COL_WARN, copy_warn,
                      COL_NOTE, copy_note,
                      COL_FALSE, copy_false,
                      COL_LOG, &copy_log,
                      COL_EDITABLE, copy_editable,
                      -1);
  scopetree_save_treerowref(context, model, iter);

  /* Move children */
  next = context->children;
  while(next)
  {
    scopetree_move(next, context);
    next = next->next;
  }

  /* Remove old context */
  path = gtk_tree_row_reference_get_path(old_treerowref);
  gtk_tree_model_get_iter(model, &iter, path);
  gtk_tree_path_free(path);
  gtk_tree_row_reference_free(old_treerowref);
  gtk_tree_store_remove(GTK_TREE_STORE(model), &iter);
}

void
scope_menu_moveto (GtkMenuItem *menuitem, struct context *context)
{
  GtkWidget *scopetree = GTK_WIDGET(
      arg_get_value(arg_get_value(MainDialog, "SCOPETREE"), "TREEVIEW"));
  struct context *scope = context_by_type(Context, CONTEXT_SCOPE);
  GtkTreePath *parentpath;

  prefs_context_update(scope);
  scope_move_menuitem_enable(scope, TRUE);

  if(context_move(scope, context))
    scopetree_move(scope, context);

  parentpath = gtk_tree_row_reference_get_path(context->treerowref);
  gtk_tree_view_expand_row(GTK_TREE_VIEW(scopetree), parentpath, FALSE);
  gtk_tree_path_free(parentpath);
  prefs_context_update(scope);
}

void
scopetree_move_menuitem_add (struct context *context)
{
  GtkWidget *menu = arg_get_value(MainDialog, "MOVESCOPE_SUBMENU");
  GtkWidget *menuitem;

  menuitem = gtk_menu_item_new_with_label(prefs_get_string(context, "name"));
  gtk_widget_show(menuitem);
  gtk_container_add(GTK_CONTAINER(menu), menuitem);
  g_signal_connect(G_OBJECT(menuitem), "activate",
      G_CALLBACK(scope_menu_moveto), context);

  context->move_menuitem = menuitem;
}

void
scopetree_rename (context_type type)
{
  GtkWidget *scopetree = GTK_WIDGET(
      arg_get_value(arg_get_value(MainDialog, "SCOPETREE"), "TREEVIEW"));
  GtkTreePath *path;
  GtkTreeViewColumn *column;
  struct context *context = context_by_type(Context, type);

  if(!context)
  {
    show_error(_("scopetree_rename() called with illegal type"));
    return;
  }
  prefs_context_update(context);

  column = gtk_tree_view_get_column(GTK_TREE_VIEW(scopetree), 0);
  if(!column->editable_widget) /* avoid a bug in GTK+ 2.0.2 */
  {
    gtk_tree_view_get_cursor(GTK_TREE_VIEW(scopetree), &path, NULL);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(scopetree), path, column, TRUE);
    gtk_widget_grab_focus(scopetree);
    gtk_tree_path_free(path);
  }
}

void
task_menu_rename (GtkMenuItem *menuitem, gpointer user_data)
{
  scopetree_rename(CONTEXT_TASK);
}

void
scope_menu_rename (GtkMenuItem *menuitem, gpointer user_data)
{
  scopetree_rename(CONTEXT_SCOPE);
}

void
report_menu_rename (GtkMenuItem *menuitem, gpointer user_data)
{
  scopetree_rename(CONTEXT_REPORT);
}


struct context *
scopetree_new (context_type type, const char* name, const char* filename)
{
  return scopetree_new_with_parent(NULL, type, name, filename);
}

#ifdef USE_OMP
/* The OMP option adds a function called scopetree_add_with_parent.
 * refresh_reports in ../monitor_dialog.c uses it.  scopetree_add_with_parent
 * calls the original scopetree_new_with_parent, which is renamed
 * scopetree_add_new_with_parent to cater for both a scopetree_add_with_parent
 * and a scopetree_new_with_parent. */

struct context *
scopetree_add_new_with_parent (struct context* parentscope, context_type type,
                               const char *name, const char *filename, int new);
#endif

struct context *
scopetree_new_with_parent (struct context* parentscope, context_type type,
                           const char *name, const char *filename)
#ifdef USE_OMP
{
  return scopetree_add_new_with_parent (parentscope, type, name, filename, 1);
}

struct context *
scopetree_add_with_parent (struct context* parentscope, context_type type,
                           const char *name, const char *filename)
{
  return scopetree_add_new_with_parent (parentscope, type, name, filename, 0);
}

struct context *
scopetree_add_new_with_parent (struct context* parentscope, context_type type,
                               const char *name, const char *filename, int new)
#endif /* USE_OMP */
{
  GtkWidget *scopetree = GTK_WIDGET(
      arg_get_value(arg_get_value(MainDialog, "SCOPETREE"), "TREEVIEW"));
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(scopetree));
  GtkTreePath *path, *parentpath = NULL;
  GtkTreeViewColumn *column;
  GtkTreeIter iter, parent;
  struct context *context;

  switch(type)
  {
#ifdef USE_OMP
    case CONTEXT_SERVER:
      context = Servers;
      break;
#endif
    case CONTEXT_TASK:
      context = Global;
      break;
    case CONTEXT_SCOPE:
    case CONTEXT_REPORT:
      if (parentscope != NULL)
      {
	if (parentscope->type >= type)
	{
	  show_error(_("scopetree_new_with_parent(): parent type >= child type"));
	  return NULL;
	}
	context = parentscope;
	parentpath = gtk_tree_row_reference_get_path(context->treerowref);
	gtk_tree_model_get_iter(model, &parent, parentpath);
      }
      else
      {
	gtk_tree_view_get_cursor(GTK_TREE_VIEW(scopetree), &parentpath,
	    &column);
	while(gtk_tree_path_get_depth(parentpath) >= type)
	  gtk_tree_path_up(parentpath);

	gtk_tree_model_get_iter(model, &parent, parentpath);
	gtk_tree_model_get(model, &parent, COL_CONTEXT, &context, -1);
      }
      break;
    default:
      // FIX _rename?
      show_error(_("context_rename() called with illegal type"));
      return NULL;
  }

#ifdef USE_OMP
  if (new)
    {
      if (context->protocol == PROTOCOL_OTP || type == CONTEXT_SERVER)
        context = context_new (context, name, filename);
      else if (type == CONTEXT_SCOPE)
        {
          struct context *server = context;
          struct context *scope;

          if (type == CONTEXT_REPORT) abort ();

          char* id;
          gchar *rcfile, *rc;
          GError* error;
          gsize rc_len;

          if (ensure_context_connected (server, FALSE))
            {
              show_error (_("%s: Failed to connect to manager"), __FUNCTION__);
              return NULL;
            }

          /* Create the task. */

          // FIX handle error
          // FIX create_omp_task
          if (omp_create_task_rc (&server->session,
                                  "targets = localhost\n",
                                  strlen ("targets = localhost\n"),
                                  "unnamed task",
                                  "",
                                  &id))
            {
              openvas_server_close (server->socket, server->session);
              server->socket = 0;
              show_error (_("%s: Failed to create scope on manager"),
                          __FUNCTION__);
              return NULL;
            }

          /* Create the context in the scope tree. */

          scope = context_new (server, name, filename);

          /* FIX "Context" is now "scope".  Transfer the GUI values into the file.  */
          //prefs_context_update (scope);
          //prefs_dialog_apply(Context, MainDialog);

          /* Set ID in prefs/rcfile. */

          rcfile = preferences_get_altname (scope, NULL);

          prefs_set_string (scope, "id", id);
          /* Using prefs_set_value ensures that the value always goes in the rc. */
          //prefs_set_value (scope, "targets", (void*) "localhost", ARG_STRING);
          // FIX sync tabs w prefs?
          //         stuff in prefs_context_update (scope);
          preferences_save (scope);

          /* Set rcfile on task in the manager. */

          error = NULL;
          g_file_get_contents (rcfile, &rc, &rc_len, &error);
          if (error)
            {
              free (id);
              show_error (_("%s: Failed to get scope RC file: %s"),
                          __FUNCTION__,
                          error->message);
              g_error_free (error);
              openvas_server_close (server->socket, server->session);
              server->socket = 0;
              return NULL;
            }

          if (omp_modify_task (&server->session, id, rc, NULL, NULL))
            {
              free (id);
              g_free (rc);
              show_error (_("%s: Failed to set RC on task on manager"),
                          __FUNCTION__);
              openvas_server_close (server->socket, server->session);
              server->socket = 0;
              return NULL;
            }
          g_free (rc);

          openvas_server_close (server->socket, server->session);
          server->socket = 0;

          free (id);

          context = scope;
        }
      else
        {
          show_error (_("%s: called with illegal type"), __FUNCTION__);
          return NULL;
        }
    }
  else
    context = context_add(context, name, filename);
#else
  context = context_new(context, name, filename);
#endif
  if(context)
  {
    if(type == CONTEXT_TASK
#ifdef USE_OMP
       || type == CONTEXT_SERVER
#endif
       )
    {
      gtk_tree_store_append(GTK_TREE_STORE(model), &iter, NULL);
#ifdef USE_OMP
      if (context->protocol == PROTOCOL_OTP)
#endif
      scopetree_move_menuitem_add(context);
    }
    else
      gtk_tree_store_append(GTK_TREE_STORE(model), &iter, &parent);
    gtk_tree_store_set (GTK_TREE_STORE(model), &iter,
                        COL_CONTEXT, context,
                        COL_NAME, prefs_get_string(context, "name"),
                        COL_HOLE, -1, COL_WARN, -1, COL_NOTE, -1, COL_FALSE, -1,
                        COL_LOG, -1,
                        COL_EDITABLE, TRUE, -1);
    scopetree_save_treerowref(context, model, iter);

    if(parentpath)
      gtk_tree_view_expand_row(GTK_TREE_VIEW(scopetree), parentpath, FALSE);

    path = gtk_tree_model_get_path(model, &iter);
    column = gtk_tree_view_get_column(GTK_TREE_VIEW(scopetree), 0);

    if(!column->editable_widget) /* avoid a bug in GTK+ 2.0.2 */
    {
      gtk_tree_view_set_cursor(GTK_TREE_VIEW(scopetree), path, column, !name);
      gtk_widget_grab_focus(scopetree);
    }
    gtk_tree_path_free(path);
  }
  if(parentpath)
    gtk_tree_path_free(parentpath);

  return context;
}

#ifdef USE_OMP
void
server_menu_new (GtkMenuItem *menuitem, gpointer user_data)
{
#if 0
  scopetree_new(CONTEXT_SERVER, NULL, NULL);
#else /* not 0 */
  GtkWidget *scopetree = GTK_WIDGET(
      arg_get_value(arg_get_value(MainDialog, "SCOPETREE"), "TREEVIEW"));
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(scopetree));
  GtkTreePath *path;
  GtkTreeViewColumn *column;
  GtkTreeIter iter;
  struct context *context;

  context = context_new(Servers, NULL, NULL);
  if(context)
  {
    struct context *original_context;
    gtk_tree_store_append(GTK_TREE_STORE(model), &iter, NULL);
    scopetree_move_menuitem_add(context);
    gtk_tree_store_set (GTK_TREE_STORE(model), &iter,
                        COL_CONTEXT, context,
                        COL_NAME, prefs_get_string(context, "name"),
                        COL_HOLE, -1, COL_WARN, -1, COL_NOTE, -1, COL_FALSE, -1,
                        COL_LOG, -1,
                        COL_EDITABLE, TRUE, -1);
    scopetree_save_treerowref(context, model, iter);

    path = gtk_tree_model_get_path(model, &iter);
    column = gtk_tree_view_get_column(GTK_TREE_VIEW(scopetree), 0);

    if(!column->editable_widget) /* avoid a bug in GTK+ 2.0.2 */
    {
      gtk_tree_view_set_cursor(GTK_TREE_VIEW(scopetree), path, column, 1);
      gtk_widget_grab_focus(scopetree);
    }
    gtk_tree_path_free(path);

    original_context = Context;
    prefs_context_update (context);

    if (scopetree_refresh (CONTEXT_SERVER, 1))
      {
        prefs_context_update (original_context);
        scopetree_delete_recurse (context);
        if (context->next)
          prefs_context_update(context->next);
        else if (context->parent->children == context)
          prefs_context_update (Global);
        else
          {
            struct context *sibling = context->parent->children;

            while (sibling->next != context)
              sibling = sibling->next;
            prefs_context_update(sibling);
          }
        context_delete (context);
      }
  }
#endif /* not 0 */
}
#endif /* USE_OMP */

void
task_menu_new (GtkMenuItem *menuitem, gpointer user_data)
{
  scopetree_new(CONTEXT_TASK, NULL, NULL);
}

void
scope_menu_new (GtkMenuItem *menuitem, gpointer user_data)
{
  scopetree_new(CONTEXT_SCOPE, NULL, NULL);
}

void
scopetree_context_new (GtkWidget *button, gpointer user_data)
{
  if(Context->type >= CONTEXT_TASK)
    scopetree_new(CONTEXT_SCOPE, NULL, NULL);
#ifdef USE_OMP
  else if (Context->type == CONTEXT_GLOBAL)
    scopetree_new(CONTEXT_TASK, NULL, NULL);
  else if (Context->type >= CONTEXT_SERVER)
    scopetree_new(CONTEXT_SCOPE, NULL, NULL);
#endif /* USE_OMP */
  else
    scopetree_new(CONTEXT_TASK, NULL, NULL);
}


/**
 * Delete menu widgets, tree row references and tree rows for a subtree.
 */
static void
scopetree_delete_recurse (struct context *context)
{
  struct context *child = context->children;
  GtkWidget *scopetree = GTK_WIDGET(
      arg_get_value(arg_get_value(MainDialog, "SCOPETREE"), "TREEVIEW"));
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(scopetree));
  GtkTreePath *path;
  GtkTreeIter iter;

  while(child)
  {
    scopetree_delete_recurse(child);
    child = child->next;
  }
  if(context->move_menuitem)
    gtk_widget_destroy(context->move_menuitem);
  path = gtk_tree_row_reference_get_path(context->treerowref);
  gtk_tree_model_get_iter(model, &iter, path);
  gtk_tree_path_free(path);
  gtk_tree_row_reference_free(context->treerowref);
  gtk_tree_store_remove(GTK_TREE_STORE(model), &iter);
}

static void
scopetree_delete (context_type type)
{
  GtkWindow *window = GTK_WINDOW(arg_get_value(MainDialog, "WINDOW"));
  GtkWidget *dialog;
  const char *question;
  struct context *context = context_by_type(Context, type);

  prefs_context_update(context);

  switch(type)
  {
#ifdef USE_OMP
    case CONTEXT_SERVER:
      question = _("Really delete server?");
      break;
#endif
    case CONTEXT_TASK:
      question = _("Really delete task\n `%s'\nand all scopes and reports in it?");
      break;
    case CONTEXT_SCOPE:
      question = _("Really delete scope\n `%s'\nall reports in it?");
      break;
    case CONTEXT_REPORT:
      question = _("Really delete report\n `%s'?");
      break;
    default:
      show_error(_("scopetree_delete() called with illegal type"));
      return;
  }

  dialog = gtk_message_dialog_new(window,
      GTK_DIALOG_DESTROY_WITH_PARENT,
      GTK_MESSAGE_QUESTION,
      GTK_BUTTONS_OK_CANCEL,
      question, prefs_get_string(context, "name"));

  arg_set_value(MainDialog, "CONTEXT", -1, dialog);
  switch (gtk_dialog_run(GTK_DIALOG(dialog)))
  {
    case GTK_RESPONSE_OK:
#ifdef USE_OMP
      if (context->protocol == PROTOCOL_OMP
          && type != CONTEXT_SERVER)
        {
          /* Connect first, so that if the connection fails the tree remains.
           *
           * \TODO Connecting here keeps the dialog calls out of context.c, but
           * also means the connection may time out before or during the
           * context_delete. */
          if (ensure_context_connected (context, FALSE))
            {
              arg_set_value(MainDialog, "CONTEXT", -1, window);
              gtk_widget_destroy(dialog);
              // FIX again?
              arg_set_value(MainDialog, "CONTEXT", -1, window);
              return;
            }
        }
#endif /* USE_OMP */
      scopetree_delete_recurse(context);
      if(context->next)
	prefs_context_update(context->next);
      else if(context->parent->children == context)
        {
#ifdef USE_OMP
          if (context->parent == Servers)
            prefs_context_update (Global);
          else
#endif
          prefs_context_update(context->parent);
        }
      else
      {
	struct context *sibling = context->parent->children;

	while(sibling->next != context)
	  sibling = sibling->next;
	prefs_context_update(sibling);
      }
#ifdef USE_OMP
      context_delete (context); // FIX always succeeds
      if (context->protocol == PROTOCOL_OMP
          && type != CONTEXT_SERVER)
        {
          openvas_server_close (context->socket, context->session);
          // FIX check return
          context->socket = 0;
        }
#else
      context_delete(context);
#endif
      break;
    default:
      break;
  }
  gtk_widget_destroy(dialog);
  arg_set_value(MainDialog, "CONTEXT", -1, window);
}


#ifdef USE_OMP
/**
 * @param user_data ignored (callback).
 */
void
server_menu_delete (GtkMenuItem *menuitem, gpointer user_data)
{
  scopetree_delete (CONTEXT_SERVER);
}
#endif /* USE_OMP */

/**
 * @param user_data ignored (callback).
 */
void
task_menu_delete (GtkMenuItem *menuitem, gpointer user_data)
{
  scopetree_delete(CONTEXT_TASK);
}

/**
 * @param user_data ignored (callback).
 */
void
scope_menu_delete (GtkMenuItem *menuitem, gpointer user_data)
{
  scopetree_delete(CONTEXT_SCOPE);
}

/**
 * @param user_data ignored (callback).
 */
void
report_menu_delete (GtkMenuItem *menuitem, gpointer user_data)
{
  scopetree_delete(CONTEXT_REPORT);
}

/**
 * @param user_data ignored (callback).
 */
void
scopetree_context_delete (GtkWidget *button, gpointer user_data)
{
#ifdef USE_OMP
  if(Context->type == CONTEXT_SERVER)
    scopetree_delete(CONTEXT_SERVER);
  else
#endif
  if(Context->type == CONTEXT_TASK)
    scopetree_delete(CONTEXT_TASK);
  else if(Context->type == CONTEXT_SCOPE)
    scopetree_delete(CONTEXT_SCOPE);
  else if(Context->type >= CONTEXT_REPORT)
    scopetree_delete(CONTEXT_REPORT);
  else
    show_error(_("scopetree_context_delete() called with illegal type"));
}

/**
 * @param user_data ignored (callback).
 */
static void
scope_menu_open_ok (GtkWidget *widget, gpointer user_data)
{
  GtkWidget *dialog = GTK_WIDGET(user_data);
  gchar *filename;

  filename = g_strdup(
      gtk_file_selection_get_filename(GTK_FILE_SELECTION(dialog)));
  gtk_widget_destroy(dialog);

  if(check_is_dir(filename))
    show_error(_("Please choose a target filename."));
  else if(!check_is_file(filename))
    show_error(_("File \"%s\" doesn't exist."), filename);
  else
    scopetree_new(CONTEXT_SCOPE, NULL, filename);

  g_free(filename);
}

/**
 * @param user_data ignored (callback).
 */
void
scope_menu_open (GtkMenuItem *menuitem, gpointer user_data)
{
  GtkWindow *parent = arg_get_value(MainDialog, "CONTEXT");
  GtkWidget *dialog = gtk_file_selection_new(
      _("Open configuration for new scope"));

  if(parent)
    gtk_window_set_transient_for(GTK_WINDOW(dialog), parent);
  prefs_context_update(context_by_type(Context, CONTEXT_TASK));
  g_signal_connect(GTK_FILE_SELECTION(dialog)->ok_button,
      "clicked", G_CALLBACK(scope_menu_open_ok), dialog);
  g_signal_connect_swapped(GTK_FILE_SELECTION(dialog)->cancel_button,
      "clicked", G_CALLBACK(gtk_widget_destroy), dialog);
  gtk_widget_show(dialog);
}


/**
 * @brief Callback for click on OK in the file selection dialog that appears
 * @brief after a "save as" action.
 *
 * @param  widget     (callback).
 * @param  user_data  (callback).
 */
static void
scope_menu_save_ok (GtkWidget *widget, gpointer user_data)
{
  GtkWidget *dialog = GTK_WIDGET(user_data);
  gchar *filename;

  filename = g_strdup(
      gtk_file_selection_get_filename(GTK_FILE_SELECTION(dialog)));
  gtk_widget_destroy(dialog);

  if(check_is_dir(filename))
    show_error(_("Please choose a target filename."));
  else if(check_exists(filename))
    show_error(_("File \"%s\" already exists."), filename);
  else
    preferences_save_as(Context, filename);

  g_free(filename);
}

void
scope_menu_save_as (GtkMenuItem* menuitem, gpointer user_data)
{
  GtkWindow *parent = arg_get_value(MainDialog, "CONTEXT");
  GtkWidget *dialog = gtk_file_selection_new(_("Save scope configuration"));

  if(parent)
    gtk_window_set_transient_for(GTK_WINDOW(dialog), parent);
  prefs_context_update(context_by_type(Context, CONTEXT_SCOPE));
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(dialog), "openvasrc");
  g_signal_connect(GTK_FILE_SELECTION(dialog)->ok_button,
      "clicked", G_CALLBACK(scope_menu_save_ok), dialog);
  g_signal_connect_swapped(GTK_FILE_SELECTION(dialog)->cancel_button,
      "clicked", G_CALLBACK(gtk_widget_destroy), dialog);
  gtk_widget_show(dialog);
}

/**
 * @brief Updates the current Context, saving its openvasrc.
 *
 * @param  menuitem   Menu- item (callback).
 * @param  user_data  Ignored (callback).
 */
void
scope_menu_save (GtkMenuItem* menuitem, gpointer user_data)
{
  prefs_context_update (context_by_type(Context, CONTEXT_SCOPE));
  preferences_save (Context);
}


#ifdef USE_OMP
// FIX copied from openvas-manager/src/manage.c
/**
 * @brief Return a preference from an RC.
 *
 * @param[in]  desc  The RC.
 * @param[in]  name  The name of the preference.
 *
 * @return The preference on success, else NULL.
 */
static gchar*
rc_preference (const char* desc, const char* name)
{
  char* seek;

  if (desc == NULL)
    {
#if 0
      tracef ("   desc NULL\n");
#endif
      return NULL;
    }

  while ((seek = strchr (desc, '\n')))
    {
      char* eq = seek
                 ? memchr (desc, '=', seek - desc)
                 : strchr (desc, '=');
      if (eq)
        {
#if 0
          tracef ("   1 found: %.*s\n",
                  seek ? seek - desc : strlen (seek),
                  desc);
#endif
          if (strncmp (desc, name, eq - desc - 1) == 0)
            {
              gchar* ret;
              ret = g_strndup (eq + 2,
                               seek ? seek - (eq + 2) : strlen (seek));
              return ret;
            }
        }
      else if ((seek ? seek - desc > 7 : 1)
               && strncmp (desc, "begin(", 6) == 0)
        {
          /* Read over the section. */
          desc = seek + 1;
          while ((seek = strchr (desc, '\n')))
            {
              if ((seek ? seek - desc > 5 : 1)
                  && strncmp (desc, "end(", 4) == 0)
                {
                  break;
                }
#if 0
              tracef ("   1 skip: %.*s\n",
                      seek ? seek - desc : strlen (seek),
                      desc);
#endif
              desc = seek + 1;
            }
        }
      if (seek == NULL) break;
      desc = seek + 1;
    }
  return NULL;
}

// FIX
/** @todo defined in context.c */
extern void context_collect_recurse (struct context *, const char *);

// FIX
void
scope_create_treestore_children (GtkTreeStore *treestore,
                                 struct context *context);

int
create_omp_scope (struct context *context,
                  const char* id,
                  const char* name,
                  const char* rcfile,
                  char** scope_dir_name)
{
  gchar* servers_dir_name;
  gchar* file_name;
  gboolean ok;

  servers_dir_name = g_build_filename (prefs_get_string (context, "openvas_dir"),
                                       ".Servers",
                                       NULL);
  if (check_is_dir (servers_dir_name))
    {
      char *server_dir_name = context->dir;
      gchar *pref;

      assert (context->type == CONTEXT_SERVER);

      g_free (servers_dir_name);

      if (server_dir_name == NULL)
        {
          show_error (_("%s: server dir is NULL"), __FUNCTION__);
          return -2;
        }

      *scope_dir_name = context_newpath (NULL, server_dir_name, name);
      if (*scope_dir_name == NULL)
        {
          show_error (_("%s: failed to create scope name"), __FUNCTION__);
          return -2;
        }

      // FIX free and NULL *scope_dir_name on failures from here?

      if (g_mkdir_with_parents (*scope_dir_name, 0700 /* d-w-rwxr-- */) == -1)
        {
          show_error (_("%s: g_mkdir_with_parents"), __FUNCTION__);
          return -2;
        }

      file_name = g_build_filename (*scope_dir_name, "openvasrc", NULL);

      ok = g_file_set_contents (file_name, rcfile, strlen (rcfile), NULL);
      if (ok == FALSE)
        {
          show_error (_("%s: g_file_set_contents"), __FUNCTION__);
          g_free (file_name);
          return -3;
        }

      /* Hack to ensure name in openvasrc. */

      pref = rc_preference (rcfile, "name");
      if (pref && (strcmp (pref, name) == 0))
        g_free (pref);
      else
        {
          char *pos;
          gchar *rc;
          gsize rc_len;
          GString *string;
          GError *error;

          error = NULL;
          g_file_get_contents (file_name, &rc, &rc_len, &error);
          if (error)
            {
              show_error (_("%s: g_file_get_contents"), __FUNCTION__);
              // FIX note error
              g_error_free (error);
              g_free (file_name);
              return -4;
            }

          pos = strchr (rc, '\n');
          if (pos)
            {
              *pos = '\0';
              string = g_string_new (rc);
              /* The first \n ensures that "name" begins a line. */
              g_string_append_printf (string, "\nname = %s\n", name);
              g_string_append (string, pos + 1);
            }
          else
            {
              string = g_string_new (rc);
              /* The first \n ensures that "name" begins a line. */
              g_string_append_printf (string, "\nname = %s\n", name);
            }

          g_free (rc);
          rc = g_string_free (string, FALSE);

          error = NULL;
          g_file_set_contents (file_name, (gchar*) rc, strlen (rc), &error);
          g_free (rc);
          if (error)
            {
              show_error (_("%s: g_file_get_contents"), __FUNCTION__);
              // FIX note error
              g_error_free (error);
              g_free (file_name);
              return -5;
            }
        }

      /* Hack to ensure ID in openvasrc. */

      pref = rc_preference (rcfile, "id");
      if (pref)
        g_free (pref);
      else
        {
          char *pos;
          gchar *rc;
          gsize rc_len;
          GString *string;
          GError *error;

          error = NULL;
          g_file_get_contents (file_name, &rc, &rc_len, &error);
          if (error)
            {
              show_error (_("%s: g_file_get_contents"), __FUNCTION__);
              // FIX note error
              g_error_free (error);
              g_free (file_name);
              return -4;
            }

          pos = strchr (rc, '\n');
          if (pos)
            {
              *pos = '\0';
              string = g_string_new (rc);
              /* The first \n ensures that "id" begins a line. */
              g_string_append_printf (string, "\nid = %s\n", id);
              g_string_append (string, pos + 1);
            }
          else
            {
              string = g_string_new (rc);
              /* The first \n ensures that "id" begins a line. */
              g_string_append_printf (string, "\nid = %s\n", id);
            }

          g_free (rc);
          rc = g_string_free (string, FALSE);

          error = NULL;
          g_file_set_contents (file_name, (gchar*) rc, strlen (rc), &error);
          g_free (rc);
          if (error)
            {
              show_error (_("%s: g_file_get_contents"), __FUNCTION__);
              // FIX note error
              g_error_free (error);
              g_free (file_name);
              return -5;
            }
        }

      g_free (file_name);

      {
        int pwd;

        /* Symbolic link the server NVT cache into the scope. */

        pwd = open (".", O_RDONLY);
        if (pwd == -1)
          {
            show_error (_("%s: failed to open current directory"),
                        __FUNCTION__);
            return -6;
          }

        if (chdir (*scope_dir_name))
          {
            show_error (_("%s: failed to chdir to %s"),
                        context->dir,
                        __FUNCTION__);
            close(pwd);
            return -6;
          }

        if (symlink ("../openvas_nvt_cache", "openvas_nvt_cache"))
          {
            show_error (_("%s: failed to symlink to parent NVT cache"),
                        __FUNCTION__);
            close(pwd);
            return -6;
          }

        if (fchdir (pwd))
          show_error (_("%s: failed to fchdir back to previous dir"),
                      __FUNCTION__);

        close (pwd);
        return 0;
      }
    }
  else
    g_free (servers_dir_name);

  show_error (_("%s: check_is_dir"), __FUNCTION__);
  return -1;
}

int
create_omp_report (const char *scope_dir_name, const char *id,
                   const char *report, char **report_dir_name_return)
{
  gboolean ok;
  gchar *report_dir_name, *file_name, *rcfile;
  guchar *out;
  gsize out_len = 0, rcfile_len;

  /* Create the report directory. */

  report_dir_name = context_newpath (NULL, scope_dir_name, id);
  if (report_dir_name == NULL) return -2;

  if (g_mkdir_with_parents (report_dir_name, 0700 /* d-w-rwxr-- */) == -1)
    {
      efree (&report_dir_name);
      return -2;
    }

  /* Save the report. */

  file_name = g_build_filename (report_dir_name, "report.nbe", NULL);

  out = report[0] ? g_base64_decode (report, &out_len)
                  : (guchar*) g_strdup ("");
  ok = g_file_set_contents (file_name, (gchar*) out, out_len, NULL);
  g_free (file_name);
  if (!ok) goto fail;

  /* Copy RC and cache from scope dir. */
  // FIX should get rc used for scan from manager

  file_name = g_build_filename (scope_dir_name, "openvasrc", NULL);
  ok = g_file_get_contents (file_name, &rcfile, &rcfile_len, NULL);
  g_free (file_name);
  if (!ok) goto fail;
  file_name = g_build_filename (report_dir_name, "openvasrc", NULL);
  ok = g_file_set_contents (file_name, rcfile, strlen (rcfile), NULL);
  // -- FIX mv to preferences.c
  {
    struct context temp_context;
    memset (&temp_context, '\0', sizeof (temp_context));
    /* In new RC replace name and add ID. */
    temp_context.prefs = emalloc (sizeof (struct arglist));
    if (preferences_process_filename (&temp_context, estrdup (file_name))) goto fail;
    prefs_set_value (&temp_context, "name", (void*) id, ARG_STRING);
    prefs_set_value (&temp_context, "id", (void*) id, ARG_STRING);
    preferences_save_as (&temp_context, file_name);
    efree (&temp_context.prefs);
  }
  // --
  g_free (file_name);
  if (!ok) goto fail;

  /* Turned off for now to reduce disk usage. */
#if 0
  file_name = g_build_filename (scope_dir_name, "openvas_nvt_cache", NULL);
  ok = g_file_get_contents (file_name, &rcfile, &rcfile_len, NULL);
  g_free (file_name);
  if (!ok) goto fail;
  file_name = g_build_filename (report_dir_name, "openvas_nvt_cache", NULL);
  ok = g_file_set_contents (file_name, rcfile, strlen (rcfile), NULL);
  g_free (file_name);
#endif

  if (ok)
    {
      if (report_dir_name_return)
        *report_dir_name_return = report_dir_name;
      else
        efree (&report_dir_name);
      return 0;
    }
 fail:
  efree (&report_dir_name);
  return -3;
}

static int
scopetree_refresh (context_type type, int writable_settings)
{
  struct context *context = context_by_type(Context, type);
  entity_t statuses;
  entities_t scopes;
  GtkTreeView *treeview;
  GtkTreeStore *treestore;
  const char *message;

  switch(type)
  {
    case CONTEXT_SERVER:
      break;
    default:
      show_error(_("%s must be called with a CONTEXT_SERVER"), __FUNCTION__);
      return -1;
  }

  // FIX why?
  prefs_context_update (context);

  if (ensure_context_connected (context, writable_settings))
    {
      return -1;
    }

  /* @todo Consider creating a whole new tree node for the manager and
   *       replacing the old one with it on success. */

  /* Refresh the information about the manager. */

  message = refresh_server (context);
  /* This is necessary, at least because it calls context_sync_plugin_prefs,
   * which copies plugin prefs from context->plugins and context->scanners to
   * context->plugin_prefs. */
  prefs_context_update (context);
  /* Save the RC to disk, so that new scopes can copy it. */
  context_save_recurse (context);
  if (message)
    {
      openvas_server_close (context->socket, context->session);
      context->socket = 0;
      show_error (_("%s: %s"), __FUNCTION__, message);
      return -1;
    }

  /* Get the statuses of the OMP tasks from the manager. */

  if (omp_get_status (&context->session, NULL, 1, &statuses))
    {
      openvas_server_close (context->socket, context->session);
      context->socket = 0;
      show_error (_("%s: OMP get_status failed"), __FUNCTION__);
      return -1;
    }

  /* Remove the children from the manager in the scope tree. */

  // @TODO Consider updating the existing entries.

  delete_in_client_only = 1;
  while (context->children)
    {
      scopetree_delete_recurse (context->children);
      context_delete (context->children);
    }
  delete_in_client_only = 0;

  /* Create a scope and reports on disk for each of the OMP tasks. */

  for (scopes = statuses->entities; scopes; scopes = next_entities (scopes))
    {
      entity_t scope = first_entity (scopes);
      entity_t status, task, name, rcfile;
      entities_t reports;
      char *scope_dir_name = NULL;
      guchar *rc64;
      gsize rc64_len = 0;

      if (strcmp (entity_name (scope), "task")) continue;

      /* Save the RC. */

      name = entity_child (scope, "name");
      if (name == NULL)
        {
          show_error (_("%s: OMP get_status missing name"), __FUNCTION__);
          efree (&scope_dir_name);
          free_entity (statuses);
          openvas_server_close (context->socket, context->session);
          context->socket = 0;
          return -1;
        }
      rcfile = entity_child (scope, "rcfile");
      if (rcfile == NULL)
        {
          show_error (_("%s: OMP get_status missing rcfile"), __FUNCTION__);
          efree (&scope_dir_name);
          free_entity (statuses);
          openvas_server_close (context->socket, context->session);
          context->socket = 0;
          return -1;
        }
      if (entity_attribute (scope, "id") == NULL)
        {
          show_error (_("%s: OMP get_status missing id"), __FUNCTION__);
          efree (&scope_dir_name);
          free_entity (statuses);
          openvas_server_close (context->socket, context->session);
          context->socket = 0;
          return -1;
        }
      rc64 = entity_text (rcfile)[0]
             ? g_base64_decode (entity_text (rcfile), &rc64_len)
             : (guchar*) g_strdup ("");
      if (rc64 == NULL) rc64 = (guchar*) g_strdup ("");
      if (create_omp_scope (context,
                            entity_attribute (scope, "id"),
                            entity_text (name),
                            (char*) rc64,
                            &scope_dir_name))
        {
          show_error (_("%s: Error creating scope directory"), __FUNCTION__);
          efree (&scope_dir_name);
          g_free (rc64);
          free_entity (statuses);
          openvas_server_close (context->socket, context->session);
          context->socket = 0;
          return -1;
        }
      g_free (rc64);

      /* Get the list of reports. */

      if (omp_get_status (&context->session,
                          entity_attribute (scope, "id"),
                          0,
                          &status))
        {
          show_error (_("%s: OMP get_status id failed"), __FUNCTION__);
          efree (&scope_dir_name);
          free_entity (statuses);
          openvas_server_close (context->socket, context->session);
          context->socket = 0;
          return -1;
        }

      task = entity_child (status, "task");
      if (task == NULL)
        {
          show_error (_("%s: OMP response missing task"), __FUNCTION__);
          free_entity (status);
          efree (&scope_dir_name);
          free_entity (statuses);
          openvas_server_close (context->socket, context->session);
          context->socket = 0;
          return -1;
        }

      /* Get each report. */

      for (reports = task->entities;
           reports;
           reports = next_entities (reports))
        {
          entity_t report_summary = first_entity (reports);
          const char* id;
          entity_t response, report;

          if (strcmp (entity_name (report_summary), "report"))
            continue;

          id = entity_attribute (report_summary, "id");
          if (id == NULL)
            {
              show_error (_("%s: OMP get_status missing report id"),
                          __FUNCTION__);
              continue;
            }

          /* Get and save the report. */

          if (omp_get_report (&context->session, id, "nbe", &response))
            {
              show_error (_("%s: OMP get_report failed"), __FUNCTION__);
              continue;
            }

          report = entity_child (response, "report");
          if (report == NULL)
            {
              show_error (_("%s: OMP response missing report"), __FUNCTION__);
              free_entity (response);
              continue;
            }

          // FIX OMP should have report names? (name is in rc)
          // FIX report rcfile
          if (create_omp_report (scope_dir_name, id, entity_text (report),
                                 NULL))
            {
              show_error (_("%s: Failed to create report"), __FUNCTION__);
              free_entity (response);
              continue;
            }
          // FIX set name of report in rcfile to id (currently name copied from scope)
          // may need preferences_save (context);

          free_entity (response);
        }

      free_entity (status);
      efree (&scope_dir_name);
    }
  free_entity (statuses);
  openvas_server_close (context->socket, context->session);
  context->socket = 0;

  /* Update the context from the scopes on disk. */

  // FIX flush context children
  context_collect_recurse (context, context->dir);

  /* Create the children of the manager in the scope tree. */

  treeview = GTK_TREE_VIEW (arg_get_value (arg_get_value (MainDialog,
                                                          "SCOPETREE"),
                                           "TREEVIEW"));
  treestore = GTK_TREE_STORE (gtk_tree_view_get_model (treeview));
  scope_create_treestore_children (treestore, context);

  return 0;
}

#include "openvas_plugin.h"
#include "plugin_cache.h"
// FIX copy from report.c
static void
copy_plugins (struct context *context, struct openvas_plugin *plugin)
{
  while (plugin)
  {
    context_add_plugin (context, openvas_plugin_duplicate(plugin));
    plugin = plugin->next;
  }
}

void
scopetree_refresh_reports (struct context *context)
{
  entity_t status, task;
  entities_t reports;
  char* scope_dir_name = context->dir;

  if (!check_is_dir (scope_dir_name))
    {
      show_error (_("%s: scope dir missing"), __FUNCTION__);
      return;
    }

  /* Get the list of reports. */

  if (omp_get_status (&context->session,
                      prefs_get_string (context, "id"),
                      0,
                      &status))
    {
      show_error (_("%s: OMP get_status by id failed"), __FUNCTION__);
      return;
    }

  task = entity_child (status, "task");
  if (task == NULL)
    {
      show_error (_("%s: OMP response missing task"), __FUNCTION__);
      free_entity (status);
      return;
    }

  /* Remove the children from the task in the scope tree. */

  /** @todo Consider updating the existing entries. */

  delete_in_client_only = 1;
  while (context->children)
    {
      scopetree_delete_recurse (context->children);
      context_delete (context->children);
    }
  delete_in_client_only = 0;

  /* Get each report. */

  for (reports = task->entities;
       reports;
       reports = next_entities (reports))
    {
      entity_t report_summary = first_entity (reports);
      const char* id;
      entity_t response, report;
      char* report_dir;

      if (strcmp (entity_name (report_summary), "report"))
        continue;

      id = entity_attribute (report_summary, "id");
      if (id == NULL)
        {
          show_error (_("%s: OMP get_status missing report id"),
                      __FUNCTION__);
          return;
        }

      /* Get and save the report. */

      if (omp_get_report (&context->session, id, "nbe", &response))
        {
          show_error (_("%s: OMP get_report failed"), __FUNCTION__);
          return;
        }

      report = entity_child (response, "report");
      if (report == NULL)
        {
          show_error (_("%s: OMP response missing report"), __FUNCTION__);
          free_entity (response);
          return;
        }
      // FIX what free's "report" and "report" prefs

      // FIX OMP should have report names? (name is in rc)
      // FIX report rcfile
      report_dir = NULL;
      if (create_omp_report (scope_dir_name,
                             id,
                             entity_text (report),
                             &report_dir))
        {
          show_error (_("%s: Failed to create report"), __FUNCTION__);
          free_entity (response);
          return;
        }

      if (report_dir)
        {
          struct context *report_context;
          gchar* report_rc;

          report_rc = g_build_filename (report_dir, "openvasrc", NULL);
          report_context = scopetree_add_with_parent (context, CONTEXT_REPORT,
                                                      id, report_rc);
          g_free (report_rc);

          prefs_set_string (report_context, "name", id);
          prefs_set_string (report_context, "id", id);
          preferences_save (report_context);

          if (prefs_get_int(Global, "reports_use_plugin_cache"))
          {
            int error;

            copy_plugins(report_context, context->plugins);
            copy_plugins(report_context, context->scanners);

            error = plugin_cache_write(report_context, "");
            if(error)
            {
              show_error(_("report_save() couldn't save the plugin information"));
              return;
            }

            /* by the time we get here, the report_context may already have got a
             * plugin tree because scopetree_new_with_parent may trigger GUI
             * updates, so we have to reset the tree. */
            context_reset_plugin_tree(report_context);
          }

          // FIX report.c has save,load certificate here

          prefs_context_update(report_context);
// --

          efree (&report_dir);
        }


      free_entity (response);
    }

  free_entity (status);
}

/**
 * @param user_data ignored (callback).
 */
void
server_menu_refresh (GtkMenuItem *menuitem, gpointer user_data)
{
  scopetree_refresh (CONTEXT_SERVER, 0);
}
#endif /* USE_OMP */

/**
 * @brief Set table values for statistics (e.g. nr. of warnings) in the
 * @brief treestore. Values are read from or written to a .cnt file.
 *
 * @param treestore Treestore to manipulate.
 * @param be        Backend index, will be converted (backend_convert) if file
 *                  does not exist.
 * @param fname     Path to .cnt file.
 */
void
scopetreestore_counters_update (GtkTreeStore* treestore, GtkTreeIter iter,
                                int be, char * fname)
{
  struct arglist *hosts;
  gchar* cached_counters = NULL;
  FILE * fp;
  int    highs = 0, warns = 0, notes = 0, falsepos = 0, logs = 0;

  /* Cache the results in a '.cnt' file to avoid recounting
   * each .nbe file on startup */
  if (fname != NULL)
    {
      cached_counters = g_strdup_printf ("%s.cnt", fname);
      fp = fopen (cached_counters, "r");
    }
  else
    fp = NULL;

  if (fp == NULL)
  {
     hosts = backend_convert(be);
     highs = number_of_holes(hosts);
     warns = number_of_warnings(hosts);
     notes = number_of_notes(hosts);
     falsepos = number_of_false_positives(hosts);
     logs = number_of_log_messages(hosts);
     if(hosts) arg_free_all(hosts);
     if ( fname != NULL )
       fp = fopen (cached_counters, "w");
     else
	fp = NULL;

     if ( fp != NULL )
      {
       fprintf(fp, "%d %d %d %d %d\n", highs, warns, notes, falsepos, logs);
       fclose(fp);
      }
      else if ( fname != NULL ) perror("open ");
  }
  else
  {
    fscanf(fp, "%d %d %d %d %d", &highs, &warns, &notes, &falsepos, &logs);
    fclose(fp);
  }


  /* XXX: add these to scopes and tasks? */
  gtk_tree_store_set (treestore, &iter,
                      COL_NOTE, notes,
                      COL_WARN, warns,
                      COL_HOLE, highs,
                      COL_FALSE, falsepos,
                      COL_LOG, logs,
                      -1);
  if (cached_counters) g_free (cached_counters);
}


void
scopetreeview_counters_update (struct context *context, int be, char * fname)
{
  GtkWidget *scopetree = GTK_WIDGET(
      arg_get_value(arg_get_value(MainDialog, "SCOPETREE"), "TREEVIEW"));
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(scopetree));
  GtkTreePath *path;
  GtkTreeIter iter;

  gtk_tree_view_get_cursor(GTK_TREE_VIEW(scopetree), &path, NULL);
  gtk_tree_model_get_iter(model, &iter, path);
  scopetreestore_counters_update(GTK_TREE_STORE(model), iter, be, fname);

  gtk_tree_path_free(path);
}


void
scopetreeview_connected_update (struct context *context)
{
  GtkWidget *scopetree = GTK_WIDGET(
      arg_get_value(arg_get_value(MainDialog, "SCOPETREE"), "TREEVIEW"));
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(scopetree));
  GtkTreePath *path = gtk_tree_row_reference_get_path(context->treerowref);
  GtkTreeIter iter;

  gtk_tree_model_get_iter(model, &iter, path);
  gtk_tree_model_row_changed(model, path, &iter);
  gtk_tree_path_free(path);
}


/**
 * @brief Set the pixbuf of the cell according to the state of the context or
 *        the row.
 *
 * @param data The tree-view widget.
 */
static void
connected_data_func (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
                     GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
  GtkWidget *widget = GTK_WIDGET(data);
  GdkPixbuf *icon = NULL;
  struct context *context;

  gtk_tree_model_get(model, iter, COL_CONTEXT, &context, -1);

  /* if the row is for a context that can have a connection, choose an
   * icon showing the connection status.  Otherwise icon stays NULL,
   * i.e. no icon at all.
   */
#ifdef USE_OMP
  if (context->protocol == PROTOCOL_OMP)
    {
      /* Leave it blank for now. */
    }
  else
#endif
  if (context->type == CONTEXT_SCOPE || context->type == CONTEXT_GLOBAL)
    {
      icon = gtk_widget_render_icon (widget,
             context->socket < 0 ? "openvas-disconnect" : "openvas-connect",
             GTK_ICON_SIZE_MENU, NULL);
    }
  else if (context->type == CONTEXT_REPORT
           && context->is_severity_mapped == TRUE)
    {
      icon = gtk_widget_render_icon (widget, "gtk-convert", GTK_ICON_SIZE_MENU,
                                     NULL);
    }

  g_object_set (G_OBJECT(cell), "pixbuf", icon, NULL);

  if (icon != NULL)
    g_object_unref(icon);
}

/**
 * @brief Callback for "edited" signal.
 *
 * As the only editable column is the name column, this reflects a possible
 * change in name. Rename the context.
 *
 * @brief cellrenderertext Ignored.
 * @brief path_string      Path to the edited scope in scopetree.
 * @brief new_text         New name for the selected context.
 * @brief data             Ignored.
 */
static void
on_scope_edited (GtkCellRendererText *cellrenderertext, gchar *path_string,
                 gchar* new_text, gpointer data)
{
  struct context *context;
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(data));
  GtkTreePath *path = gtk_tree_path_new_from_string(path_string);
  GtkTreeIter iter;

  if(new_text && new_text[0])
  {
    gtk_tree_model_get_iter(model, &iter, path);
    gtk_tree_store_set(GTK_TREE_STORE(model), &iter, COL_NAME, new_text, -1);
    gtk_tree_model_get(model, &iter, COL_CONTEXT, &context, -1);
#ifdef USE_OMP
    if (context->protocol == PROTOCOL_OMP
        && context->type == CONTEXT_SCOPE)
      {
        if (ensure_context_connected (context, FALSE))
          {
            show_error(_("%s: Failed to connect to manager."), __FUNCTION__);
            prefs_context_update (context);
            gtk_tree_path_free (path);
            return;
          }
        context_rename (context, new_text);
        openvas_server_close (context->socket, context->session);
        context->socket = 0;
        gtk_tree_path_free(path);
        return;
      }
#endif
    context_rename(context, new_text);
    if(context->move_menuitem)
    {
      GtkWidget *label = gtk_bin_get_child(GTK_BIN(context->move_menuitem));

      if(GTK_IS_LABEL(label))
	gtk_label_set_text(GTK_LABEL(label), prefs_get_string(context, "name"));
      else
	show_error(_("on_scope_edited(): menuitem has no label."));
    }
    prefs_context_update(context);
  }

  gtk_tree_path_free(path);
}

/**
 * @brief Implementation of a GtkTreeSelectionFunc().
 *
 * Does autoconnect and auto-expand if the Client was configured to do so.
 * Sets the filter- state (is_severity_mapped) of the selected context according
 * to existance of global severity filter and the glabal_filter_active flag.
 *
 * @return Always TRUE, because all rows are always selectable.
 */
static gboolean
scopetreeview_selection_func (GtkTreeSelection *selection, GtkTreeModel *model,
                              GtkTreePath *path, gboolean path_currently_selected,
                              gpointer userdata)
{
  GtkTreeIter iter;
  struct context *context;

  if (!path_currently_selected &&
      !gtk_tree_selection_get_selected(selection, NULL, NULL))
  {
    gtk_tree_model_get_iter(model, &iter, path);
    gtk_tree_model_get(model, &iter, COL_CONTEXT, &context, -1);
    if (prefs_get_int (Global, "tree_autoexpand"))
      gtk_tree_view_expand_row (gtk_tree_selection_get_tree_view (selection),
                                path, FALSE);
    prefs_context_update(context);
    if (prefs_get_int (Global, "nessusd_autoconnect")
        && (context->type == CONTEXT_SCOPE))
    {
      prefs_dialog_auth_connection(context);
    }
  }

  return TRUE; /* allow selection state to change */
}

/**
 * @brief Writes a number to a cell, number is fetched from tree_model.
 *
 * Acts as cell_data_func for the columns that involve counting
 * (e.g. COL_HOLES shows number of reported holes).
 *
 * @param data In practice one of COL_HOLE, COL_WARN, COL_NOTE, COL_FALSE,
 *             used as column- index in  tree_model to receive the value.
 */
static void
cell_counts (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
             GtkTreeModel *tree_model, GtkTreeIter *iter, gpointer data)
{
  gint value;
  gchar *text;

  /* Get the int value from the model. */
  gtk_tree_model_get(tree_model, iter, GPOINTER_TO_SIZE(data), &value, -1);
  /* Now we can format the value ourselves. */
  if(value < 0)
    text = g_strdup("");
  else
    text = g_strdup_printf("%d", value);
  g_object_set(cell, "text", text, NULL);
  g_free(text);
}

#if defined (USE_OMP) && !defined (DISABLE_OTP)
static void
set_cell_context_type (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell,
                       GtkTreeModel *tree_model, GtkTreeIter *iter,
                       gpointer data)
{
  struct context *context;
  gtk_tree_model_get (tree_model, iter, COL_CONTEXT, &context, -1);
  if (context < 0)
    {
      g_object_set (G_OBJECT (cell), "text", "", NULL);
      return;
    }
  switch (context->type)
    {
      case CONTEXT_GLOBAL:
        g_object_set (G_OBJECT (cell), "text", "global", NULL);
        break;
      case CONTEXT_SERVER:
        g_object_set (G_OBJECT (cell), "text", "server", NULL);
        break;
      case CONTEXT_TASK:
        g_object_set (G_OBJECT (cell), "text", "task", NULL);
        break;
      case CONTEXT_SCOPE:
#ifndef DISABLE_OTP
        g_object_set (G_OBJECT (cell), "text", "scope", NULL);
#else
        g_object_set (G_OBJECT (cell), "text", "task", NULL);
#endif
        break;
      case CONTEXT_REPORT:
        g_object_set (G_OBJECT (cell), "text", "report", NULL);
        break;
      default:
        g_object_set (G_OBJECT (cell), "text", "error", NULL);
        break;
    }
}
#endif /* USE_OMP and not DISABLE_OTP */

/**
 * @brief Behave as if the currently selected report would have been freshly
 * @brief selected, update counters, icons etc.
 *
 * Motivation for this function is to apply filters "on the fly" on the current
 * selected report.
 */
void
prefs_scope_tree_update_report ()
{
  /* If currently a Report is selected in the GUI,
    Update the scopeview, as we want the values and icon change immediately
    according to the severity_filter.
    Ideally that should work by just sending an update() signal to the
    scopetreeview. That did not work in practice, however.
    @TODO make this meachanism clean, document flow of events in real life GUI use.  */
  if (Context->type == CONTEXT_REPORT)
    {
      GtkWidget *scopetree = GTK_WIDGET (arg_get_value(arg_get_value(MainDialog,
                                                        "SCOPETREE"), "TREEVIEW"));
      GtkTreeSelection* selection = gtk_tree_view_get_selection(GTK_TREE_VIEW (scopetree));
      GtkTreeModel* model = gtk_tree_view_get_model (GTK_TREE_VIEW (scopetree));
      GtkTreeIter iter;

      // Get the selector, and the path to the selected node.
      gtk_tree_selection_get_selected (selection, NULL, &iter);
      GtkTreePath * path = gtk_tree_model_get_path (model, &iter);
      gtk_tree_selection_select_path (selection, path);
      // Manually call the selection function to update icon that indicates filtering
      scopetreeview_selection_func (selection, model, path, FALSE, NULL);

      char* fname = report_get_filename (Context);
      int be = backend_import_report (fname);
      // Manually call counter update
      scopetreestore_counters_update (GTK_TREE_STORE(model), iter, be, NULL);
      prefs_report_update (arg_get_value (MainDialog, "REPORT"), TRUE);
    }
}

/**
 * @brief Creates the columns of the scope- tree view.
 *
 * @param view The scope-tree view.
 */
void
scopetreeview_create_columns (GtkWidget* view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  column = gtk_tree_view_column_new();
  gtk_tree_view_column_set_resizable(GTK_TREE_VIEW_COLUMN(column), 1);
  gtk_tree_view_column_set_title(column, _("Name"));
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);
  renderer = gtk_cell_renderer_text_new();
  g_signal_connect(G_OBJECT(renderer), "edited",
		   G_CALLBACK(on_scope_edited), view);
  gtk_tree_view_column_pack_start(column, renderer, TRUE);
  gtk_tree_view_column_add_attribute(column, renderer, "text", COL_NAME);
  gtk_tree_view_column_add_attribute(column, renderer, "editable",
				     COL_EDITABLE);

  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_column_pack_start(column, renderer, TRUE);
  gtk_tree_view_column_set_cell_data_func(column, renderer, connected_data_func,
      view, NULL);

#if defined (USE_OMP) && !defined (DISABLE_OTP)
  column = gtk_tree_view_column_new();
  gtk_tree_view_column_set_resizable(GTK_TREE_VIEW_COLUMN(column), 1);
  gtk_tree_view_column_set_title(column, _("Type"));
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "xalign", (gfloat) 1.0, "xpad", (guint) 8, NULL);
  gtk_tree_view_column_pack_start(column, renderer, TRUE);
#if 0
  gtk_tree_view_column_add_attribute(column, renderer, "text", COL_CONTEXT);
#endif /* 0 */
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  set_cell_context_type,
					  (gpointer) COL_CONTEXT, NULL);
#endif /* USE_OMP and not DISABLE_OTP */

  column = gtk_tree_view_column_new();
  gtk_tree_view_column_set_resizable(GTK_TREE_VIEW_COLUMN(column), 1);
  gtk_tree_view_column_set_title(column, _("High"));
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "xalign", (gfloat) 1.0, "xpad", (guint) 8, NULL);
  gtk_tree_view_column_pack_start(column, renderer, TRUE);
  gtk_tree_view_column_add_attribute(column, renderer, "text", COL_HOLE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  cell_counts,
					  (gpointer) COL_HOLE, NULL);

  column = gtk_tree_view_column_new();
  gtk_tree_view_column_set_resizable(GTK_TREE_VIEW_COLUMN(column), 1);
  gtk_tree_view_column_set_title(column, _("Medium"));
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "xalign", (gfloat) 1.0, "xpad", (guint) 8, NULL);
  gtk_tree_view_column_pack_start(column, renderer, TRUE);
  gtk_tree_view_column_add_attribute(column, renderer, "text", COL_WARN);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  cell_counts,
					  (gpointer) COL_WARN, NULL);

  column = gtk_tree_view_column_new();
  gtk_tree_view_column_set_resizable(GTK_TREE_VIEW_COLUMN(column), 1);
  gtk_tree_view_column_set_title(column, _("Low"));
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "xalign", (gfloat) 1.0, "xpad", (guint) 8, NULL);
  gtk_tree_view_column_pack_start(column, renderer, TRUE);
  gtk_tree_view_column_add_attribute(column, renderer, "text", COL_NOTE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  cell_counts,
					  (gpointer) COL_NOTE, NULL);

  column = gtk_tree_view_column_new();
  gtk_tree_view_column_set_resizable(GTK_TREE_VIEW_COLUMN(column), 1);
  gtk_tree_view_column_set_title(column, _("FP"));
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), column);
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "xalign", (gfloat) 1.0, "xpad", (guint) 8, NULL);
  gtk_tree_view_column_pack_start(column, renderer, TRUE);
  gtk_tree_view_column_add_attribute(column, renderer, "text", COL_FALSE);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  cell_counts,
					  (gpointer) COL_FALSE, NULL);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN(column), 1);
  gtk_tree_view_column_set_title (column, _("Log"));
  gtk_tree_view_append_column (GTK_TREE_VIEW(view), column);
  renderer = gtk_cell_renderer_text_new ();
  g_object_set (renderer, "xalign", (gfloat) 1.0, "xpad", (guint) 8, NULL);
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "text", COL_LOG);
  gtk_tree_view_column_set_cell_data_func (column, renderer, cell_counts,
                                           (gpointer) COL_LOG, NULL);
}

void
scope_create_treestore_scopes (GtkTreeStore *treestore,
                               GtkTreeIter* task,
                               struct context *scopes)
{
  struct context *reports;
  GtkTreeIter scope, report;
    while(scopes)
    {
    gtk_tree_store_append(treestore, &scope, task);
      gtk_tree_store_set (treestore, &scope,
                          COL_CONTEXT, scopes,
                          COL_NAME, prefs_get_string(scopes, "name"),
                          COL_HOLE, -1, COL_WARN, -1, COL_NOTE, -1, COL_FALSE, -1,
                          COL_LOG, -1,
                          COL_EDITABLE, TRUE, -1);
      scopetree_save_treerowref(scopes, GTK_TREE_MODEL(treestore), scope);

      reports = scopes->children;
      while(reports)
      {
	gtk_tree_store_append(treestore, &report, &scope);
        gtk_tree_store_set (treestore, &report,
                            COL_CONTEXT, reports,
                            COL_NAME, prefs_get_string(reports, "name"),
                            COL_HOLE, -1, COL_WARN, -1, COL_NOTE, -1, COL_FALSE, -1,
                            COL_LOG, -1,
                            COL_EDITABLE, TRUE, -1);
	scopetree_save_treerowref(reports, GTK_TREE_MODEL(treestore), report);

	{
	  /* TODO: move this to general counter handling */
	  char *fname;
	  int be;

	  fname = report_get_filename(reports);
	  be = backend_import_report(fname);
	  scopetreestore_counters_update(treestore, report, be, fname);
	  efree(&fname);
	  backend_dispose(be);
	}
	reports = reports->next;
      }
      scopes = scopes->next;
    }
}

void
scope_create_treestore_children (GtkTreeStore *treestore,
                                 struct context *context)
{
#ifdef USE_OMP
  GtkTreeIter server;

  if (context->type == CONTEXT_SERVER)
    {
      GtkTreePath *path = gtk_tree_row_reference_get_path (context->treerowref);
      gtk_tree_model_get_iter (GTK_TREE_MODEL (treestore), &server, path);
      scope_create_treestore_scopes (treestore, &server, context->children);
    }
  else
#endif /* USE_OMP */
    {
      GtkTreeIter task;
      struct context *tasks = context->children;
      while(tasks)
      {
        gtk_tree_store_append(treestore, &task, NULL);
        gtk_tree_store_set(treestore, &task,
                           COL_CONTEXT, tasks,
                           COL_NAME, prefs_get_string(tasks, "name"),
                           COL_HOLE, -1, COL_WARN, -1, COL_NOTE, -1, COL_FALSE, -1,
                           COL_LOG, -1,
                           COL_EDITABLE, TRUE,
                           -1);
        scopetree_save_treerowref(tasks, GTK_TREE_MODEL(treestore), task);

#ifndef DISABLE_OTP
        scopetree_move_menuitem_add(tasks);
#endif /* undefined DISABLE_OTP */

        scope_create_treestore_scopes (treestore, &task, tasks->children);
        tasks = tasks->next;
      }
    }
}

GtkTreeStore *
scope_create_treestore (struct context *context)
{
  GtkTreeStore *treestore;
#ifndef DISABLE_OTP
  GtkTreeIter global;
#endif
  /* TODO currently unused:
   * GtkTreeIter hostgroup, host; */

  treestore = gtk_tree_store_new (NUM_COLS,
                                  G_TYPE_POINTER, G_TYPE_STRING, G_TYPE_INT,
                                  G_TYPE_INT, G_TYPE_INT, G_TYPE_INT, G_TYPE_INT,
                                  G_TYPE_BOOLEAN);
#ifndef DISABLE_OTP
  gtk_tree_store_append (treestore, &global, NULL);

  gtk_tree_store_set (treestore, &global,
                      COL_CONTEXT, context,
                      COL_NAME, prefs_get_string(context, "name"),
                      COL_HOLE, -1, COL_WARN, -1, COL_NOTE, -1, COL_FALSE, -1,
                      COL_LOG, -1,
                      COL_EDITABLE, FALSE, -1);

  scopetree_save_treerowref (context, GTK_TREE_MODEL (treestore), global);

  scope_create_treestore_children (treestore, context);
#endif /* undefined DISABLE_OTP */

#ifdef USE_OMP
  /* FIX Assume that "context" is always the Global context, just like
   *     scopetree_new_with_parent does. */
  scope_create_treestore_children (treestore, Servers);
#endif

  return treestore;
}

/**
 * Returns an arglist with following keys (type):
 *  - FRAME (GtkScrolledWindow*)
 *  - TREEVIEW (GtkTreeView*)
 */
struct arglist *
prefs_dialog_scope_tree (struct context *context)
{
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  GtkWidget *scrolledwindow;
  GtkWidget *scopetreeview;
  struct arglist *ctrls = emalloc(sizeof(struct arglist));

  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_placement(GTK_SCROLLED_WINDOW(scrolledwindow),
				    GTK_CORNER_TOP_RIGHT);
  arg_add_value(ctrls, "FRAME", ARG_PTR, -1, scrolledwindow);

  scopetreeview = gtk_tree_view_new();
  gtk_widget_show(scopetreeview);
  gtk_container_add(GTK_CONTAINER(scrolledwindow), scopetreeview);
  gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(scopetreeview), TRUE);
  arg_add_value(ctrls, "TREEVIEW", ARG_PTR, -1, scopetreeview);

  g_signal_connect(G_OBJECT(scopetreeview), "key-press-event",
                   G_CALLBACK(onKeypressed), NULL);

  scopetreeview_create_columns(scopetreeview);

  model = GTK_TREE_MODEL (scope_create_treestore (context));
  gtk_tree_view_set_model(GTK_TREE_VIEW(scopetreeview), model);
  g_object_unref(model);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(scopetreeview));
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_BROWSE);
  gtk_tree_selection_set_select_function(selection,
					 scopetreeview_selection_func,
					 NULL, NULL);

  return ctrls;
}

#endif /* USE_GTK */
