//                                               -*- C++ -*-
/**
 *  @file  StandardEvent.cxx
 *  @brief The class that implements an event based on a composite random vector whose
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: StandardEvent.cxx 1866 2011-04-11 10:32:27Z schueller $
 */
#include "StandardEvent.hxx"
#include "NumericalMathFunction.hxx"
#include "Distribution.hxx"
#include "CompositeRandomVector.hxx"
#include "UsualRandomVector.hxx"
#include "RandomVectorImplementation.hxx"
#include "Pointer.hxx"
#include "ComparisonOperatorImplementation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      typedef Base::Func::NumericalMathFunction                   NumericalMathFunction;
      typedef Distribution::InverseIsoProbabilisticTransformation InverseIsoProbabilisticTransformation;

      CLASSNAMEINIT(StandardEvent);

      /* Default constructor */
      StandardEvent::StandardEvent()
        : Event()
      {
        // Nothing to do
      }

      /* Constructor from a RandomVector */
      StandardEvent::StandardEvent(const RandomVector & antecedent,
                                   const ComparisonOperator & op,
                                   const NumericalScalar threshold,
                                   const String & name)
        /* throw (InvalidArgumentException) */
        : Event(antecedent, op, threshold, name)
      {
        // StandardEvent can only be constructed from composite random vector whose antecedent has a spherical distribution. As we cannot check it, we just check that the distribution is elliptical
        if (!antecedent.getImplementation()->getAntecedent()->getDistribution().getImplementation()->isElliptical()) throw InvalidArgumentException(HERE) << "StandardEvent can only be constructed from composite random vectors whose antecedent is elliptical, here the distribution is " << antecedent.getImplementation()->getAntecedent()->getDistribution().getImplementation()->__repr__();
      }

      /* Constructor from an Event */
      StandardEvent::StandardEvent(const Event & event,
                                   const String & name)
        /* throw (InvalidArgumentException) */
        : Event(event)
      {
        // Get the isoprobabilistic inverse transformation from the antecedent distribution
        const Distribution distribution(event.getImplementation()->getAntecedent()->getDistribution());
        const InverseIsoProbabilisticTransformation inverse(distribution.getInverseIsoProbabilisticTransformation());
        // Get the function
        const NumericalMathFunction function(event.getImplementation()->getFunction());
        // Get the standard distribution from the antecedent distribution
        const Distribution standard(distribution.getStandardDistribution());
        // Build the antecedent
        const RandomVector antecedent(standard);
        const NumericalMathFunction composed(function, inverse);
        const RandomVector vect(composed, antecedent);
        // Set the random vector implementation
        (*this) = StandardEvent(vect, event.getOperator(), event.getThreshold(), event.getName());
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
